/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include "CDlgCalibrationWizard.h"
#include "mono_slamMain.h"

//(*InternalHeaders(CDlgCalibrationWizard)
#include <wx/string.h>
#include <wx/intl.h>
//*)

extern mono_slamDialog *theMainWin;


using namespace std;
using namespace mrpt;
using namespace mrpt::utils;
using namespace mrpt::vision;
using namespace mrpt::math;
using namespace mrpt::gui;



//(*IdInit(CDlgCalibrationWizard)
const long CDlgCalibrationWizard::ID_BUTTON1 = wxNewId();
const long CDlgCalibrationWizard::ID_BUTTON2 = wxNewId();
const long CDlgCalibrationWizard::ID_STATICTEXT3 = wxNewId();
const long CDlgCalibrationWizard::ID_SPINCTRL1 = wxNewId();
const long CDlgCalibrationWizard::ID_STATICTEXT4 = wxNewId();
const long CDlgCalibrationWizard::ID_SPINCTRL2 = wxNewId();
const long CDlgCalibrationWizard::ID_STATICTEXT5 = wxNewId();
const long CDlgCalibrationWizard::ID_TEXTCTRL1 = wxNewId();
const long CDlgCalibrationWizard::ID_STATICTEXT6 = wxNewId();
const long CDlgCalibrationWizard::ID_TEXTCTRL3 = wxNewId();
const long CDlgCalibrationWizard::ID_CHECKBOX1 = wxNewId();
const long CDlgCalibrationWizard::ID_STATICTEXT1 = wxNewId();
const long CDlgCalibrationWizard::ID_TEXTCTRL2 = wxNewId();
const long CDlgCalibrationWizard::ID_STATICTEXT2 = wxNewId();
const long CDlgCalibrationWizard::ID_STATICTEXT7 = wxNewId();
const long CDlgCalibrationWizard::ID_BUTTON3 = wxNewId();
const long CDlgCalibrationWizard::ID_CUSTOM1 = wxNewId();
const long CDlgCalibrationWizard::ID_TIMER1 = wxNewId();
//*)

BEGIN_EVENT_TABLE(CDlgCalibrationWizard,wxDialog)
	//(*EventTable(CDlgCalibrationWizard)
	//*)
END_EVENT_TABLE()

CDlgCalibrationWizard::CDlgCalibrationWizard(wxWindow* parent,wxWindowID id)
{
	m_threadCorners.clear();
	m_threadImgToProcess.clear();
	m_threadMustClose = false;
	m_threadResults.clear();
	m_threadResultsComputed=false;
	m_threadIsClosed = true;

	//(*Initialize(CDlgCalibrationWizard)
	wxStaticBoxSizer* StaticBoxSizer2;
	wxFlexGridSizer* FlexGridSizer2;
	wxFlexGridSizer* FlexGridSizer7;
	wxFlexGridSizer* FlexGridSizer4;
	wxStaticBoxSizer* StaticBoxSizer3;
	wxFlexGridSizer* FlexGridSizer3;
	wxFlexGridSizer* FlexGridSizer5;
	wxStaticBoxSizer* StaticBoxSizer1;

	Create(parent, id, _("Camera calibration GUI - Part of the MRPT project"), wxDefaultPosition, wxDefaultSize, wxDEFAULT_DIALOG_STYLE, _T("id"));
	FlexGridSizer1 = new wxFlexGridSizer(0, 2, 0, 0);
	FlexGridSizer1->AddGrowableCol(1);
	FlexGridSizer1->AddGrowableRow(0);
	FlexGridSizer2 = new wxFlexGridSizer(0, 1, 0, 0);
	FlexGridSizer2->AddGrowableRow(3);
	StaticBoxSizer1 = new wxStaticBoxSizer(wxHORIZONTAL, this, _("Calibration"));
	FlexGridSizer3 = new wxFlexGridSizer(0, 2, 0, 0);
	btnStart = new wxButton(this, ID_BUTTON1, _("Start"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON1"));
	FlexGridSizer3->Add(btnStart, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	btnStop = new wxButton(this, ID_BUTTON2, _("Stop"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON2"));
	btnStop->Disable();
	FlexGridSizer3->Add(btnStop, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	StaticBoxSizer1->Add(FlexGridSizer3, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
	FlexGridSizer2->Add(StaticBoxSizer1, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	StaticBoxSizer3 = new wxStaticBoxSizer(wxHORIZONTAL, this, _("Checkerboard detection parameters"));
	FlexGridSizer7 = new wxFlexGridSizer(0, 2, 0, 0);
	FlexGridSizer7->AddGrowableCol(1);
	StaticText2 = new wxStaticText(this, ID_STATICTEXT3, _("Size in X (squares):"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT3"));
	FlexGridSizer7->Add(StaticText2, 1, wxALL|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
	edSizeX = new wxSpinCtrl(this, ID_SPINCTRL1, _T("5"), wxDefaultPosition, wxDefaultSize, 0, 1, 200, 5, _T("ID_SPINCTRL1"));
	edSizeX->SetValue(_T("5"));
	FlexGridSizer7->Add(edSizeX, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	StaticText3 = new wxStaticText(this, ID_STATICTEXT4, _("Size in Y (squares):"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT4"));
	FlexGridSizer7->Add(StaticText3, 1, wxALL|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
	edSizeY = new wxSpinCtrl(this, ID_SPINCTRL2, _T("8"), wxDefaultPosition, wxDefaultSize, 0, 1, 200, 8, _T("ID_SPINCTRL2"));
	edSizeY->SetValue(_T("8"));
	FlexGridSizer7->Add(edSizeY, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	StaticText4 = new wxStaticText(this, ID_STATICTEXT5, _("Square length in X (mm):"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT5"));
	FlexGridSizer7->Add(StaticText4, 1, wxALL|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
	edLengthX = new wxTextCtrl(this, ID_TEXTCTRL1, _("40.0"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_TEXTCTRL1"));
	FlexGridSizer7->Add(edLengthX, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	StaticText5 = new wxStaticText(this, ID_STATICTEXT6, _("Square length in Y (mm):"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT6"));
	FlexGridSizer7->Add(StaticText5, 1, wxALL|wxEXPAND|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
	edLengthY = new wxTextCtrl(this, ID_TEXTCTRL3, _("40.0"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_TEXTCTRL3"));
	FlexGridSizer7->Add(edLengthY, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	FlexGridSizer7->Add(-1,-1,1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	cbNormalize = new wxCheckBox(this, ID_CHECKBOX1, _("Normalize image"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_CHECKBOX1"));
	cbNormalize->SetValue(true);
	FlexGridSizer7->Add(cbNormalize, 1, wxALL|wxEXPAND|wxALIGN_LEFT|wxALIGN_CENTER_VERTICAL, 5);
	StaticBoxSizer3->Add(FlexGridSizer7, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
	FlexGridSizer2->Add(StaticBoxSizer3, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	StaticBoxSizer2 = new wxStaticBoxSizer(wxHORIZONTAL, this, _("Frames to capture"));
	FlexGridSizer4 = new wxFlexGridSizer(0, 2, 0, 0);
	FlexGridSizer4->AddGrowableCol(0);
	StaticText1 = new wxStaticText(this, ID_STATICTEXT1, _("Desired number of frames:"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT1"));
	FlexGridSizer4->Add(StaticText1, 1, wxALL|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
	edNumFramesToGrab = new wxTextCtrl(this, ID_TEXTCTRL2, _("15"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_TEXTCTRL2"));
	FlexGridSizer4->Add(edNumFramesToGrab, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	StaticText6 = new wxStaticText(this, ID_STATICTEXT2, _("Progress:"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT2"));
	FlexGridSizer4->Add(StaticText6, 1, wxALL|wxALIGN_RIGHT|wxALIGN_CENTER_VERTICAL, 5);
	lbProgress = new wxStaticText(this, ID_STATICTEXT7, _("0"), wxDefaultPosition, wxDefaultSize, 0, _T("ID_STATICTEXT7"));
	FlexGridSizer4->Add(lbProgress, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	StaticBoxSizer2->Add(FlexGridSizer4, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
	FlexGridSizer2->Add(StaticBoxSizer2, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	FlexGridSizer5 = new wxFlexGridSizer(1, 1, 0, 0);
	FlexGridSizer5->AddGrowableRow(0);
	FlexGridSizer5->Add(-1,-1,1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	btnClose = new wxButton(this, ID_BUTTON3, _("Close"), wxDefaultPosition, wxDefaultSize, 0, wxDefaultValidator, _T("ID_BUTTON3"));
	FlexGridSizer5->Add(btnClose, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 5);
	FlexGridSizer2->Add(FlexGridSizer5, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
	FlexGridSizer1->Add(FlexGridSizer2, 1, wxALL|wxEXPAND|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
	m_realtimeCalibView = new mrpt::gui::wxMRPTImageControl(this,ID_CUSTOM1,wxDefaultPosition.x,wxDefaultPosition.y,wxSize(400,400).GetWidth(), wxSize(400,400).GetHeight() );
	FlexGridSizer1->Add(m_realtimeCalibView, 1, wxALL|wxALIGN_CENTER_HORIZONTAL|wxALIGN_CENTER_VERTICAL, 0);
	SetSizer(FlexGridSizer1);
	timCapture.SetOwner(this, ID_TIMER1);
	FlexGridSizer1->Fit(this);
	FlexGridSizer1->SetSizeHints(this);

	Connect(ID_BUTTON1,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&CDlgCalibrationWizard::OnbtnStartClick);
	Connect(ID_BUTTON2,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&CDlgCalibrationWizard::OnbtnStopClick);
	Connect(ID_BUTTON3,wxEVT_COMMAND_BUTTON_CLICKED,(wxObjectEventFunction)&CDlgCalibrationWizard::OnbtnCloseClick);
	Connect(ID_TIMER1,wxEVT_TIMER,(wxObjectEventFunction)&CDlgCalibrationWizard::OntimCaptureTrigger);
	//*)
}

CDlgCalibrationWizard::~CDlgCalibrationWizard()
{
	//(*Destroy(CDlgCalibrationWizard)
	//*)
}

void CDlgCalibrationWizard::OnbtnStartClick(wxCommandEvent& event)
{
	// Try to open the camera:
	if (! theMainWin->prepareVideoSource())
		return;

	// Launch thread:
	// --------------------------
	m_threadImgToProcess.clear();
	m_threadMustClose = false;
	m_threadResults.clear();
	m_threadResultsComputed=true;  // To start a new detection
	m_threadIsClosed = false;

	m_calibFrames.clear();

	m_threadCorners = mrpt::system::createThreadFromObjectMethod( this, &CDlgCalibrationWizard::threadProcessCorners );
	if (m_threadCorners.isClear())
	{
		wxMessageBox(_("Cannot spawn processing thread!"),_("Error"),wxICON_INFORMATION,this);
		return;
	}

	lbProgress->SetLabel(_("0"));

	btnStart->Disable();
	btnStop->Enable();

	// start processing:
	timCapture.Start(2,true); // One shot
}

void CDlgCalibrationWizard::OnbtnStopClick(wxCommandEvent& event)
{
	if (theMainWin->m_monoslam.m_camera)
		theMainWin->m_monoslam.m_camera.clear();

	btnStart->Enable();
	btnStop->Disable();

	m_threadMustClose = true;
	if (!this->m_threadCorners.isClear())
		while (!m_threadIsClosed) mrpt::system::sleep(2);
}

void CDlgCalibrationWizard::OnbtnCloseClick(wxCommandEvent& event)
{
	wxCommandEvent dumm;
	theMainWin->OnbtnStopClick(dumm);

	EndModal(wxID_OK);
}

// ---------------------------------------------------------------------------------------------------
// The thread for parallel detection of edges. This is needed since from OpenCV 1.1.0,
//  chessboard corner detection runs very slow, so it cannot be done in real-time properly:
// ---------------------------------------------------------------------------------------------------
void CDlgCalibrationWizard::threadProcessCorners()
{
	CDlgCalibrationWizard *obj = this;
	try
	{
		while (!obj->m_threadMustClose)
		{
			if (!obj->m_threadResultsComputed)
			{
				try
				{
					// Detect corners:
					bool foundCorners = obj->m_threadImgToProcess->image.findChessboardCorners( obj->m_threadResults, obj->m_check_size_x, obj->m_check_size_y,	obj->m_normalize_image );

					//cout << "corners: " << obj->m_threadResults.size() << endl;

					if (!foundCorners)
						obj->m_threadResults.clear();

					obj->m_threadResultsComputed = true;
				}
				catch(std::exception &e)
				{
					cerr << e.what() << endl;
				}
				catch(...)
				{
				}
			}
			else
			{
				// Nothing to do:
				mrpt::system::sleep(5);
			}
		}

		// Signal we're done
		obj->m_threadIsClosed = true;
	}
	catch(std::exception &e)
	{
		cerr << e.what() << endl;
		// Signal we're done
		obj->m_threadIsClosed = true;
	}
	catch(...)
	{
		// Signal we're done
		obj->m_threadIsClosed = true;
	}
}


void CDlgCalibrationWizard::OntimCaptureTrigger(wxTimerEvent& event)
{
	static mrpt::system::TTimeStamp  last_valid = INVALID_TIMESTAMP;

	try
	{
		if (!btnStop->IsEnabled())
		{
			timCapture.Stop();
			return;
		}

		ASSERT_(theMainWin->m_monoslam.m_camera)

		m_check_size_x = this->edSizeX->GetValue();
		m_check_size_y = this->edSizeY->GetValue();
		m_normalize_image = this->cbNormalize->GetValue();


		CObservationPtr obs = theMainWin->m_monoslam.m_camera->getNextFrame();
		ASSERT_(obs)
		ASSERT_(IS_CLASS(obs,CObservationImage))

		CImage  img_to_show;

		// get the observation:
		CObservationImagePtr  obs_img = CObservationImagePtr(obs);

		// Is there a detection??
		bool blankTime = (last_valid != INVALID_TIMESTAMP) && mrpt::system::timeDifference(last_valid,mrpt::system::now())<2.0;

		if (!blankTime && m_threadResultsComputed && !m_threadResults.empty() )
		{
			// Save image to the list:
			string  newImgName = format("frame_%03u",(unsigned int) m_calibFrames.size() );
			m_calibFrames[newImgName].img_original = m_threadImgToProcess->image;

			// Counter:
			lbProgress->SetLabel( _U( format("%u",(unsigned int)m_calibFrames.size()).c_str() ) );

			last_valid=mrpt::system::now();

			// Display now:
			m_threadImgToProcess->image.colorImage(img_to_show);

			// Draw the corners:
			img_to_show.drawChessboardCorners(m_threadResults,m_check_size_x, m_check_size_y);
		}
		else
		{
			// Show current image:
			img_to_show = obs_img->image;
		}

		// Process a new image to detect checkerboard:
		if (m_threadResultsComputed)
		{
			m_threadImgToProcess = obs_img;
			m_threadResultsComputed = false;
			//cout << "new image sent"<<endl;
		}

		// Progress:
		const unsigned int nFramesToGrab = (unsigned)atoi(this->edNumFramesToGrab->GetValue().mb_str()) ;
		img_to_show.textOut( 10,10, format("%u out of %u grabbed",(unsigned int)m_calibFrames.size(),nFramesToGrab), TColor::white );

		m_realtimeCalibView->AssignImage(img_to_show);
		m_realtimeCalibView->Refresh(false);
		wxTheApp->Yield();


		// Resize the display area, if needed:
		if (std::abs( (int)(this->m_realtimeCalibView->GetClientSize().GetWidth()) - int(img_to_show.getWidth()) )>30 )
		{
			this->m_realtimeCalibView->SetSize( img_to_show.getWidth(), img_to_show.getHeight() );
			this->m_realtimeCalibView->SetMinSize( wxSize(img_to_show.getWidth(), img_to_show.getHeight()) );
			this->FlexGridSizer1->RecalcSizes();
			this->Fit();
		}

		// Are we done?
		// --------------------
		if (m_calibFrames.size()>=  nFramesToGrab )
		{
			wxCommandEvent  dum;
			this->OnbtnStopClick(dum);
			wxTheApp->Yield();

			// Run calibration itself:
			if (wxOK==wxMessageBox(_("The desired number of frames have been grabbed.\nOkay to compute camera parameres with these frames?"),_("Done"), wxCANCEL | wxOK,this ))
			{
				double len_x = atof(this->edLengthX->GetValue().mb_str());
				double len_y = atof(this->edLengthY->GetValue().mb_str());

				CMatrixDouble  matrixParams;
				vector_double  distParams;

				bool sucess;
				double sqrErr;

				{
					wxBusyCursor	info;

					sucess=mrpt::vision::checkerBoardCameraCalibration(
						m_calibFrames,
						m_check_size_x,
						m_check_size_y,
						len_x,len_y,
						matrixParams,
						distParams,
						m_normalize_image,
						&sqrErr,
						false );
				}

				if (!sucess)
				{
					wxMessageBox(_("Couldn't calibrate the camera with these images"),_("Error"),wxICON_INFORMATION,this);
				}
				else
				{
					string str = format("Calibration done: MSE = %f\ncx=%f\ncy=%f\nfx=%f\nfy=%f\n",
						sqrErr,
						matrixParams(0,2),
						matrixParams(1,2),
						matrixParams(0,0),
						matrixParams(1,1)
						);
					wxMessageBox(_U(str.c_str()),_("Done!"),wxICON_INFORMATION,this);
				}
			}
			else
			{
				try { wxCommandEvent  dum; this->OnbtnStopClick(dum); } catch(...) {}
			}
		}
		else
		{
			// Get on...
			timCapture.Start(5,true);
		}
	}
	catch(std::exception &e)
	{
		try { wxCommandEvent  dum; this->OnbtnStopClick(dum); } catch(...) {}
		cerr << endl << e.what() << endl;
		wxMessageBox(_U(e.what()),_("Error"),wxICON_INFORMATION,this);
		return;
	}
}
