/* +---------------------------------------------------------------------------+
   |          The Mobile Robot Programming Toolkit (MRPT) C++ library          |
   |                                                                           |
   |                   http://mrpt.sourceforge.net/                            |
   |                                                                           |
   |   Copyright (C) 2005-2010  University of Malaga                           |
   |                                                                           |
   |    This software was written by the Machine Perception and Intelligent    |
   |      Robotics Lab, University of Malaga (Spain).                          |
   |    Contact: Jose-Luis Blanco  <jlblanco@ctima.uma.es>                     |
   |                                                                           |
   |  This file is part of the MRPT project.                                   |
   |                                                                           |
   |     MRPT is free software: you can redistribute it and/or modify          |
   |     it under the terms of the GNU General Public License as published by  |
   |     the Free Software Foundation, either version 3 of the License, or     |
   |     (at your option) any later version.                                   |
   |                                                                           |
   |   MRPT is distributed in the hope that it will be useful,                 |
   |     but WITHOUT ANY WARRANTY; without even the implied warranty of        |
   |     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         |
   |     GNU General Public License for more details.                          |
   |                                                                           |
   |     You should have received a copy of the GNU General Public License     |
   |     along with MRPT.  If not, see <http://www.gnu.org/licenses/>.         |
   |                                                                           |
   +---------------------------------------------------------------------------+ */

#include <mrpt/core.h>

using namespace mrpt;
using namespace mrpt::utils;
using namespace mrpt::poses;
using namespace mrpt::gui;
using namespace mrpt::slam;
using namespace mrpt::random;
using namespace std;


void simulateObsPredicts(
	const vector_double &GTx,
	const vector_double &GTy,
	vector<CPointPDFGaussian> &obs,
	vector<CPointPDFGaussian> &pred)
{
	const double max_R = 10;
	const double max_angle = DEG2RAD(90);
	const double SIGMA_EST = 0.1;

	const double SIGMA_R   = 0.05;
	const double SIGMA_ANG = DEG2RAD(3);

	const CPose3D robotPose( 1,0,DEG2RAD(0) );
	const CPose3D robotPose_pred( 1.05,-0.04,DEG2RAD(-4) );

	pred.clear();
	obs.clear();

	for (size_t i=0;i<GTx.size();i++)
	{
		CPoint3D  real_p( GTx[i], GTy[i] );
		CPoint3D  est_p = real_p;

		est_p.x_incr( randomGenerator.drawGaussian1D(0,SIGMA_EST) );
		est_p.y_incr( randomGenerator.drawGaussian1D(0,SIGMA_EST) );

		// Only add to obs / pred if it falls in range:
		double r,y,p;
		robotPose.sphericalCoordinates( est_p, r,y,p);
		if (r<max_R && fabs(y)<max_angle)
		{
			CPointPDFGaussian pred_i;
			pred_i.mean = est_p; // - robotPose;
			pred_i.cov.unit();
			pred_i.cov*= square(SIGMA_EST);

			pred.push_back(pred_i);
		}

		robotPose.sphericalCoordinates( real_p, r,y,p);
		if (r<max_R && fabs(y)<max_angle)
		{
			CPointPDFGaussian obs_i;
			obs_i.mean  = robotPose_pred + (real_p - robotPose);

			double r = obs_i.mean.norm();
			double a = atan2( obs_i.mean.y(), obs_i.mean.x() );

			CMatrixDouble22 H;
			H(0,0) =  cos(a);  H(0,1) = -r*sin(a);
			H(1,0) =  sin(a);  H(1,1) =  r*cos(a);

			CMatrixDouble22 S;
			S(0,0) = square( SIGMA_R );
			S(1,1) = square( SIGMA_ANG );

			r+=randomGenerator.drawGaussian1D(0,SIGMA_R);
			a+=randomGenerator.drawGaussian1D(0,SIGMA_ANG);

			obs_i.mean.x( cos(a)*r );
			obs_i.mean.y( sin(a)*r );

			CMatrixDouble22 cov22(false,false);
			H.multiply_HCHt(S, cov22);
			obs_i.cov = CMatrixDouble33(cov22);

			obs.push_back(obs_i);
		}
	}
}

/* ------------------------------------------------------------------------
					Test: data-association-demo
   ------------------------------------------------------------------------ */
void Test_dataAssociation( const size_t  nLandmarks )
{
	randomGenerator.randomize( 1234 );

	// Generate the ground truth:
	//---------------------------------------------
	vector_double GT_x(nLandmarks),GT_y(nLandmarks);

	randomGenerator.drawUniformVector( GT_x, 1.0,5.0 );
	randomGenerator.drawUniformVector( GT_y, -6.0,6.0 );


	// Simulate observations & predictions:
	//---------------------------------------------
	vector<CPointPDFGaussian>  observations;
	vector<CPointPDFGaussian>  predictions;

	simulateObsPredicts(GT_x,GT_y,observations, predictions);


	// Do data-association:
	//---------------------------------------------
	TDataAssociationResults		results;

	TDataAssociationMethod method = assocNN;
	//TDataAssociationMethod method = assocJCBB;

	TDataAssociationMetric metric = metricMaha;
	//TDataAssociationMetric metric = metricML;

	CTicTac  tictac;
	tictac.Tic();

	const size_t N = 200;

	for (size_t i=0;i<N;i++)
		mrpt::slam::data_association_independent_3d_points(
			predictions, 
			observations, 
			results, method, metric, 0.99, true );

	cout << "Time for data association: " << 1e3*tictac.Tac()/N << " ms" << endl;


	// Draw all:
	// --------------------------------------------------------
	CDisplayWindowPlots	win2("Predictions (red) & observations (blue)");

	for (size_t i=0;i<observations.size();i++)
	{
		CMatrixDouble COV22 = CMatrixDouble(observations[i].cov);
		COV22.resize(2,2);

		win2.plotEllipse( observations[i].mean.x(), observations[i].mean.y(), COV22, 3, "b", 
			format("obs_%u",(unsigned)i));
		win2.plot( 
			make_vector<2,double>(observations[i].mean.x(), GT_x[i]), 
			make_vector<2,double>(observations[i].mean.y(), GT_y[i]), 
			"k", format("gt_assoc_%u",(unsigned)i));
	}

	for (size_t i=0;i<predictions.size();i++)
	{
		CMatrixDouble COV22 = CMatrixDouble(predictions[i].cov);
		COV22.resize(2,2);
		win2.plotEllipse( 
			predictions[i].mean.x(), predictions[i].mean.y(), COV22, 3, "r", 
			format("pred_%u",(unsigned)i));
	}
	win2.plot(GT_x,GT_y,".k5","landmarks");

	win2.axis_fit(true);
	win2.setPos(10,10);
	win2.resize(400,400);


	// Draw individual compatibility
	// --------------------------------------------------------
	CDisplayWindowPlots	win1("Individual compatibility");
	for (size_t i=0;i<observations.size();i++)
	{
		CMatrixDouble COV22 = CMatrixDouble(observations[i].cov);
		COV22.resize(2,2);
		win1.plotEllipse( observations[i].mean.x(), observations[i].mean.y(), COV22, 3, "b", format("obs_%u",(unsigned)i));
	}
	for (size_t i=0;i<predictions.size();i++)
	{
		CMatrixDouble COV22 = CMatrixDouble(predictions[i].cov);
		COV22.resize(2,2);
		win1.plotEllipse( predictions[i].mean.x(), predictions[i].mean.y(), COV22, 3, "r", format("pred_%u",(unsigned)i));
	}
	for (size_t p=0;p<results.indiv_compatibility.getRowCount();p++)
		for (size_t o=0;o<results.indiv_compatibility.getColCount();o++)
			if (results.indiv_compatibility(p,o))
			{
				win1.plot(
					make_vector<2,double>( observations[o].mean.x(), predictions[p].mean.x() ),
					make_vector<2,double>( observations[o].mean.y(), predictions[p].mean.y() ),
					"k", format("ic_%u_%u",(unsigned)p,(unsigned)o));
			}

	win1.axis_fit(true);
	win1.setPos(430,10);
	win1.resize(400,400);

	// Draw final associations
	// --------------------------------------------------------
	CDisplayWindowPlots	win3("Final data association");
	for (size_t i=0;i<observations.size();i++)
	{
		CMatrixDouble COV22 = CMatrixDouble(observations[i].cov);
		COV22.resize(2,2);
		win3.plotEllipse( observations[i].mean.x(), observations[i].mean.y(), COV22, 3, "b", 
			format("obs_%u",(unsigned)i));
	}
	for (size_t i=0;i<predictions.size();i++)
	{
		CMatrixDouble COV22 = CMatrixDouble(predictions[i].cov);
		COV22.resize(2,2);
		win3.plotEllipse( predictions[i].mean.x(), predictions[i].mean.y(), COV22, 3, "r", 
			format("pred_%u",(unsigned)i));
	}
	cout << results.associations.size() <<  " associations: " << endl;
	for (map<size_t,size_t>::const_iterator assoc = results.associations.begin();assoc!=results.associations.end();++assoc)
	{
		const observation_index_t o = assoc->first;
		const prediction_index_t  p = assoc->second;
		cout << " obs #" << o << " -> pred #" << p << endl;

		win3.plot(
			make_vector<2,double>( observations[o].mean.x(), predictions[p].mean.x() ),
			make_vector<2,double>( observations[o].mean.y(), predictions[p].mean.y() ),
			"k", format("ic_%u_%u",(unsigned)p,(unsigned)o));
	}

	win3.axis_fit(true);
	win3.setPos(10,430);
	win3.resize(400,400);



	mrpt::system::pause();
}

// ------------------------------------------------------
//						MAIN
// ------------------------------------------------------
int main(int argc, char **argv)
{
	try
	{
		Test_dataAssociation( 10 );

		return 0;
	} catch (std::exception &e)
	{
		std::cout << "Exception: " << e.what() << std::endl;
		return -1;
	}
	catch (...)
	{
		printf("Untyped exception!");
		return -1;
	}
}
