/*
 * conffile.c
 *
 * This file is part of msmtp, an SMTP client.
 *
 * Copyright (C) 2000, 2003, 2004
 * Martin Lambers <marlam@users.sourceforge.net>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   msmtp is released under the GPL with the additional exemption that
 *   compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
extern int errno;

#include "paths.h"
#include "list.h"
#include "merror.h"
#include "conffile.h"

/* buffer size for configuration file lines */
#define LINEBUFSIZE 501


/* 
 * account_new()
 * 
 * Create a new account_t. Must be freed with account_free().
 * Returns NULL if out of memory.
 */

account_t *account_new()
{
    account_t *a;
    if (!(a = malloc(sizeof(account_t))))
    {
	return NULL;
    }
    a->id = NULL;
    a->host = NULL;
    a->port = -1;
    a->domain = NULL;
    a->from = NULL;
    a->auth_mech = NULL;
    a->username = NULL;
    a->password = NULL;
    a->tls = -1;
    a->tls_key_file = NULL;
    a->tls_cert_file = NULL;
    a->tls_trust_file = NULL;
    a->tls_nostarttls = -1;
    a->tls_nocertcheck = -1;
    a->dsn_return = NULL;
    a->dsn_notify = NULL;
    a->keepbcc = -1;
    a->logfile = NULL;
    return a;
}


/* 
 * account_free()
 *
 * see conffile.h
 */

void account_free(void *a)
{
    account_t *p = a;
    if (p)
    {
	free(p->id);
	free(p->host);
	free(p->domain);
	free(p->from);
	free(p->auth_mech);
	free(p->username);
	free(p->password);
	free(p->tls_key_file);
	free(p->tls_cert_file);
	free(p->tls_trust_file);
	free(p->dsn_return);
	free(p->dsn_notify);
	free(p->logfile);
    }
}


/*
 * find_account()
 *
 * see conffile.h
 */

account_t *find_account(list_t *acc_list, char *id)
{
    list_t *p;
    account_t *a;
    
    p = acc_list->next;
    a = NULL;
    
    while (p->next != p)
    {
	if (strcmp(id, ((account_t *)(p->data))->id) == 0)
	{
	    a = p->data;
	    break;
	}
	p = p->next;
    }

    return a;
}


/* 
 * some small helper functions
 */

int skip_blanks(char *s, int i)
{
    while (s[i] == ' ' || s[i] == '\t')
    {
	i++;
    }
    return i;
}

int get_length(char *s)
{
    int i = 0;
    
    while (s[i] != '\0' && s[i] != ' ' && s[i] != '\t')
    {
	i++;
    }

    return i;
}

/* get index of last non-blank character. -1 means there is none. */
int get_last_nonblank(char *s)
{
    int i;

    i = (int)strlen(s) - 1;
    while (i >= 0 && (s[i] == ' ' || s[i] == '\t'))
    {
	i--;
    }
    return i;
}


/*
 * get_next_cmd()
 * 
 * Read a line from 'f'. Split it in a command part (first word after
 * whitespace) and an argument part (the word after the command).
 * Whitespace is ignored.
 * Sets the flag 'empty_line' if the line is empty.
 * Sets the flag 'eof' if EOF occured.
 * Used error codes: CONFFILE_EIO, CONFFILE_ENOMEM, CONFFILE_EPARSE
 */

merror_t get_next_cmd(FILE *f, char **cmd, char **arg, int *empty_line, int *eof)
{
    char line[LINEBUFSIZE];
    char *p;
    int i;
    int l;
    
    *eof = 0;
    *empty_line = 0;
    if (!fgets(line, (int)sizeof(line), f))
    {
	if (ferror(f))
	{
	    return merror(CONFFILE_EIO, "input error");
	}
	else /* EOF */
	{
	    *eof = 1;
	    return merror(EOK, NULL);
	}
    }
    
    /* Kill '\n'. Beware: sometimes the last line of a file has no '\n' */
    if ((p = strchr(line, '\n')))
    {
	*p = '\0';
    }
    else if (strlen(line) == LINEBUFSIZE - 1)
    {
	return merror(CONFFILE_EPARSE, 
		"line longer than %d characters", LINEBUFSIZE - 1);
    }

    i = skip_blanks(line, 0);
    
    if (line[i] == '#' || line[i] == '\0')
    {
	*empty_line = 1;
	return merror(EOK, NULL);
    }

    l = get_length(line + i);
    if (!(*cmd = malloc((l + 1) * sizeof(char))))
    {
	return merror(CONFFILE_ENOMEM, "%s", strerror(errno));
    }
    strncpy(*cmd, line + i, l);
    (*cmd)[l] = '\0';
    
    i = skip_blanks(line, i + l);
    l = get_last_nonblank(line + i);
    if (l >= 1 && line[i] == '"' && line[i + l] == '"')
    {
	if (!(*arg = malloc(l * sizeof(char))))
	{
	    free(*cmd);
	    return merror(CONFFILE_ENOMEM, "%s", strerror(errno));
	}
	strncpy(*arg, line + i + 1, l - 1);
	(*arg)[l - 1] = '\0';
    }
    else
    {
	if (!(*arg = malloc((l + 2) * sizeof(char))))
	{
	    free(*cmd);
	    return merror(CONFFILE_ENOMEM, "%s", strerror(errno));
	}
	strncpy(*arg, line + i, l + 1);
	(*arg)[l + 1] = '\0';
    }
    
    return merror(EOK, NULL);
}


/* 
 * read_conffile()
 * 
 * Read configuration data from 'f' and store it in 'acc_list'.
 * Unless an error code is returned, 'acc_list' will always be a new list;
 * it may be empty if no accounts were found.
 * Used error codes: CONFFILE_EIO, CONFFILE_ENOMEM, CONFFILE_EPARSE, 
 * 		     CONFFILE_ESYNTAX
 */

merror_t read_conffile(FILE *f, list_t **acc_list)
{
    merror_t e;
    list_t *p;
    account_t *acc;
    int line;
    char *cmd;
    char *arg;
    int empty_line;
    int eof;
    /* for the 'dsn_notify' command: */
    int count;
    size_t l;


    if (!(*acc_list = list_new()))
    {
	return merror(CONFFILE_ENOMEM, "%s", strerror(errno));
    }
    p = *acc_list;
    acc = NULL;
    e = merror(EOK, NULL);
    
    for (line = 1; ; line++)
    {
	if (!merror_ok(e = get_next_cmd(f, &cmd, &arg, &empty_line, &eof)))
	{
	    break;
	}
	if (empty_line)
	{
	    continue;
	}
	if (eof)
	{
	    break;
	}

	if (!acc && strcmp(cmd, "account") != 0)
	{
	    if (!(acc = account_new()) || !(acc->id = strdup("default")) 
		    || (list_insert(p, acc) != 0))
	    {
		e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
		break;
	    }
	    p = p->next;
	}
	if (strcmp(cmd, "account") == 0)
	{
	    if (*arg == '\0')
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s needs an argument", line, cmd);
		break;
	    }
	    else if (find_account(*acc_list, arg))
	    {
		e = merror(CONFFILE_ESYNTAX,
			"line %d: account %s was already defined", line, arg);
		break;
	    }
	    else
	    {
		if (!(acc = account_new()) || !(acc->id = strdup(arg))
			|| (list_insert(p, acc) != 0))
		{
		    e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
		    break;
		}
		p = p->next;
	    }
	}
	else if (strcmp(cmd, "host") == 0)
	{
	    if (*arg == '\0')
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s needs an argument", line, cmd);
	    }
	    else
	    {
		free(acc->host);
		if (!(acc->host = strdup(arg)))
		{
		    e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
		    break;
		}
	    }
	}
	else if (strcmp(cmd, "port") == 0)
	{
	    if (*arg == '\0')
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s needs an argument", line, cmd);
		break;
	    }
	    else
	    {
		acc->port = atoi(arg);
		if (acc->port < 1 || acc->port > 65535)
		{
		    e = merror(CONFFILE_ESYNTAX,
			    "line %d: invalid port number %d", line, acc->port);
		    break;
		}
	    }
	}
	else if (strcmp(cmd, "domain") == 0)
	{
	    if (*arg == '\0')
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s needs an argument", line, cmd);
		break;
	    }
	    else
	    {
		free(acc->domain);
		if (!(acc->domain = strdup(arg)))
		{
		    e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
		    break;
		}
	    }
	}
	else if (strcmp(cmd, "from") == 0)
	{
	    if (*arg == '\0')
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s needs an argument", line, cmd);
		break;
	    }
	    else
	    {
		free(acc->from);
		if (!(acc->from = strdup(arg)))
		{
		    e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
		    break;
		}
	    }
	}
	else if (strcmp(cmd, "auth") == 0)
	{
	    free(acc->auth_mech);
	    if (*arg == '\0')
	    {
		/* choose later */
	    }
	    else if (strcmp(arg, "plain") == 0)
	    {
		acc->auth_mech = strdup("PLAIN");
	    }
	    else if (strcmp(arg, "login") == 0)
	    {
		acc->auth_mech = strdup("LOGIN");
	    }
	    else if (strcmp(arg, "cram-md5") == 0)
	    {
		acc->auth_mech = strdup("CRAM-MD5");
	    }
	    else if (strcmp(arg, "digest-md5") == 0)
	    {
		acc->auth_mech = strdup("DIGEST-MD5");
	    }
	    else if (strcmp(arg, "ntlm") == 0)
	    {
		acc->auth_mech = strdup("NTLM");
	    }
	    else
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: unknown authentication method %s", line, arg);
		break;
	    }
	    if (!acc->auth_mech && strcmp(arg, "") != 0)
	    {
		e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
		break;
	    }
	}
	else if (strcmp(cmd, "user") == 0)
	{
	    if (*arg == '\0')
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s needs an argument", line, cmd);
		break;
	    }
	    else
	    {
		free(acc->username);
		if (!(acc->username = strdup(arg)))
		{
		    e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
		    break;
		}
	    }
	}
	else if (strcmp(cmd, "password") == 0)
	{
	    if (*arg == '\0')
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s needs an argument", line, cmd);
		break;
	    }
	    else
	    {
		free(acc->password);
		if (!(acc->password = strdup(arg)))
		{
		    e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
		    break;
		}
	    }
	}
	else if (strcmp(cmd, "tls") == 0)
	{
	    if (*arg == '\0')
	    {
		acc->tls = 1;
	    }
	    else
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s does not take an argument", line, cmd);
		break;
	    }
	}
	else if (strcmp(cmd, "tls_key_file") == 0)
	{
	    if (*arg == '\0')
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s needs an argument", line, cmd);
		break;
	    }
	    else
	    {
		free(acc->tls_key_file);
		if (!(acc->tls_key_file = strdup(arg)))
		{
		    e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
		    break;
		}
		acc->tls_key_file = expand_tilde(acc->tls_key_file);
	    }
	}
	else if (strcmp(cmd, "tls_cert_file") == 0)
	{
	    if (*arg == '\0')
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s needs an argument", line, cmd);
		break;
	    }
	    else
	    {
		free(acc->tls_cert_file);
		if (!(acc->tls_cert_file = strdup(arg)))
		{
		    e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
		    break;
		}
		acc->tls_cert_file = expand_tilde(acc->tls_cert_file);
	    }
	}
	else if (strcmp(cmd, "tls_trust_file") == 0)
	{
	    if (*arg == '\0')
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s needs an argument", line, cmd);
		break;
	    }
	    else
	    {
		free(acc->tls_trust_file);
		if (!(acc->tls_trust_file = strdup(arg)))
		{
		    e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
		    break;
		}
		acc->tls_trust_file = expand_tilde(acc->tls_trust_file);
	    }
	}
	else if (strcmp(cmd, "tls_nocertcheck") == 0)
	{
	    if (*arg == '\0')
	    {
		acc->tls_nocertcheck = 1;
	    }
	    else
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s does not take an argument", line, cmd);
		break;
	    }
	}
	else if (strcmp(cmd, "tls_nostarttls") == 0)
	{
	    if (*arg == '\0')
	    {
		acc->tls_nostarttls = 1;
	    }
	    else
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s does not take an argument", line, cmd);
		break;
	    }
	}
	else if (strcmp(cmd, "dsn_return") == 0)
	{
	    if (*arg == '\0')
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s needs an argument", line, cmd);
		break;
	    }
	    else
	    {
		free(acc->dsn_return);
		if (strcmp(arg, "headers") == 0)
		{
		    acc->dsn_return = strdup("HDRS");
		}
		else if (strcmp(arg, "full") == 0)
		{
		    acc->dsn_return = strdup("FULL");
		}
		else
		{
		    e = merror(CONFFILE_ESYNTAX,
			    "line %d: invalid argument to command %s: %s", 
			    line, cmd, arg);
		    break;
		}
		if (!acc->dsn_return)
		{
		    e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
		    break;
		}
	    }
	}
	else if (strcmp(cmd, "dsn_notify") == 0)
	{
	    if (*arg == '\0')
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s needs an argument", line, cmd);
		break;
	    }
	    else
	    {
		free(acc->dsn_notify);
		if (strcmp(arg, "never") == 0)
		{
		    acc->dsn_notify = strdup("NEVER");
		}
		else
		{
		    l = 0;
		    count = 0;
		    if (strstr(arg, "failure"))
		    {
			count++;
			l += 7;
		    }
		    if (strstr(arg, "delay"))
		    {
			count++;
			l += 5;
		    }
		    if (strstr(arg, "success"))
		    {
			count++;
			l += 7;
		    }
		    if (count == 0 
			    || (strlen(arg) != l + count - 1)
			    || (count == 2 && !strchr(arg, ','))
			    || (count == 3 && !(strchr(arg, ',') 
				    && strchr(strchr(arg, ',') + 1, ','))))
		    {
			e = merror(CONFFILE_ESYNTAX,
			    	"line %d: invalid argument to command %s: %s", 
				line, cmd, arg);
			break;
		    }
		    else
		    {
			for (l = 0; l < strlen(arg); l++)
			{
			    arg[l] = toupper((unsigned char)arg[l]);
			}
			acc->dsn_notify = strdup(arg);
		    }
		    if (!acc->dsn_notify)
		    {
			e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
			break;
		    }
		}
	    }
	}
	else if (strcmp(cmd, "keepbcc") == 0)
	{
	    if (*arg == '\0')
	    {
		acc->keepbcc = 1;
	    }
	    else
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s does not take an argument", line, cmd);
		break;
	    }
	}
	else if (strcmp(cmd, "logfile") == 0)
	{
	    if (*arg == '\0')
	    {
		e = merror(CONFFILE_ESYNTAX, 
			"line %d: command %s needs an argument", line, cmd);
		break;
	    }
	    else
	    {
		free(acc->logfile);
		if (!(acc->logfile = strdup(arg)))
		{
		    e = merror(CONFFILE_ENOMEM, "%s", strerror(errno));
		    break;
		}
		acc->logfile = expand_tilde(acc->logfile);
	    }
	}
	else
	{
	    e = merror(CONFFILE_ESYNTAX, 
		    "line %d: unknown command %s", line, cmd);
	    break;
	}    
    }
    
    if (!merror_ok(e))
    {
	list_xfree(*acc_list, account_free);
    }
    return e;
}


/* 
 * complete_account()
 * 
 * Complete an account_t (fill in default values for fields not specified in the
 * configuration file).
 * Used error codes: CONFFILE_ENOMEM
 */

merror_t complete_account(account_t *acc)
{
    if (!acc->domain)
    {
	if (!(acc->domain = strdup("localhost")))
	{
	    return merror(CONFFILE_ENOMEM, "%s", strerror(errno));
	}
    }
    if (acc->tls == -1)
    {
	acc->tls = 0;
    }
    if (acc->tls_nocertcheck == -1)
    {
	acc->tls_nocertcheck = 0;
    }
    if (acc->tls_nostarttls == -1)
    {
	acc->tls_nostarttls = 0;
    }
    if (!acc->auth_mech)
    {
	if (acc->username || acc->password)
	{
	    /* choose later */
	    if (!(acc->auth_mech = strdup("")))
	    {
		return merror(CONFFILE_ENOMEM, "%s", strerror(errno));
	    }
	}
	else
	{
	    /* NULL means no authentication */
	}
    }
    if (acc->keepbcc == -1)
    {
	acc->keepbcc = 0;
    }	
    if (acc->port == -1)
    {
	if (acc->tls && acc->tls_nostarttls)
	{
	    acc->port = 465;
	}
	else
	{
	    acc->port = 25;
	}
    }

    return merror(EOK, NULL);
}


/*
 * check_account()
 * 
 * Check an account_t.
 * Used error codes: CONFFILE_ESYNTAX
 */

merror_t check_account(account_t *acc)
{
    if (!acc->id)
    {
	return merror(CONFFILE_ESYNTAX, "account has no name");
    }
    if (!acc->host)
    {
	return merror(CONFFILE_ESYNTAX, 
		"account %s has no host information", acc->id);
    }
    if (acc->auth_mech && (!acc->username || !acc->password))
    {
	return merror(CONFFILE_ESYNTAX,
		"account %s needs both username and password for authentication", 
		acc->id);
    }
    if (acc->tls_key_file && !acc->tls)
    {
	return merror(CONFFILE_ESYNTAX,
		"account %s uses the tls_key_file command but not the tls command",
		acc->id);
    }
    if (acc->tls_cert_file && !acc->tls)
    {
	return merror(CONFFILE_ESYNTAX,
		"account %s uses the tls_cert_file command but not the tls command",
		acc->id);
    }
    if (acc->tls_key_file && !acc->tls_cert_file) 
    {
	return merror(CONFFILE_ESYNTAX,
		"account %s uses the tls_key_file command but not the "
		"tls_cert_file command", acc->id);
    }
    if (!acc->tls_key_file && acc->tls_cert_file)
    {
	return merror(CONFFILE_ESYNTAX,
		"account %s uses the tls_cert_file command but not the "
		"tls_key_file command", acc->id);
    }
    if (acc->tls_trust_file && !acc->tls)
    {
	return merror(CONFFILE_ESYNTAX, 
		"account %s uses the tls_trust_file command but not the tls command",
		acc->id);
    }
    if (acc->tls_nocertcheck && !acc->tls)
    {
	return merror(CONFFILE_ESYNTAX,
		"account %s uses the tls_nocertcheck command but not the tls command",
		acc->id);
    }
    if (acc->tls_nocertcheck && acc->tls_trust_file)
    {
	return merror(CONFFILE_ESYNTAX,
		"account %s uses both tls_nocertcheck and tls_trust_file",
		acc->id);
    }
    if (acc->tls_nostarttls && !acc->tls)
    {
	return merror(CONFFILE_ESYNTAX,
		"account %s uses the tls_nostarttls command but not the tls command",
		acc->id);
    }

    return merror(EOK, NULL);
}

	
/*
 * get_conf()
 * 
 * see conffile.h
 */

merror_t get_conf(char *conffile, list_t **acc_list)
{
    FILE *f;
    merror_t e;
    list_t *p;

    if (!(f = fopen(conffile, "r")))
    {
	return merror(CONFFILE_EIO, "%s", strerror(errno));
    }

    if (!merror_ok(e = read_conffile(f, acc_list)))
    {
	fclose(f);
	return e;
    }
    fclose(f);
    
    if (list_is_empty(*acc_list))
    {
	list_free(*acc_list);
	return merror(CONFFILE_ESYNTAX, "no valid configuration data");
    }

    p = (*acc_list)->next;
    while (p->next != p)
    {
	if (!(merror_ok(e = complete_account(p->data))))
	{
	    list_xfree(*acc_list, account_free);
	    return e;
	}
	if (!(merror_ok(e = check_account(p->data))))
	{
	    list_xfree(*acc_list, account_free);
	    return e;
	}
	p = p->next;
    }

    return merror(EOK, NULL);
}
