/*
 * md5_apps.c
 *
 * This file is part of msmtp, an SMTP client.
 *
 * This code was adapted from GNU Anubis, version 3.6.2
 * Copyright (C) 2001, 2002 The Anubis Team.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   msmtp is released under the GPL with the additional exemption that
 *   compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#ifdef HAVE_GNUTLS
#include <gcrypt.h>
#elif defined (HAVE_OPENSSL)
#include <openssl/md5.h>
#elif defined (HAVE_MD5)
#include <sys/types.h>
#include <md5.h>
void MD5_Init(MD5_CTX *context) 
{ MD5Init(context); }
void MD5_Update(MD5_CTX *context, unsigned char *data, unsigned int len)
{ MD5Update(context, data, len); }
void MD5_Final(unsigned char digest[16], MD5_CTX *context)
{ MD5Final(digest, context); }
#else
#include "md5_algo.h"
#endif
 
#include "md5_apps.h"
#include "merror.h"


/*
 * md5_hmac()
 *
 * see md5_apps.h
 */

merror_t md5_hmac(char *secret, size_t secret_len, 
	char *challenge, size_t challenge_len,
	unsigned char *digest)
{
#ifdef HAVE_GNUTLS
    
    gcry_md_hd_t context;
    gcry_error_t error_code;
    unsigned char ipad[64];
    unsigned char opad[64];
    int i;

    memset(ipad, 0, sizeof(ipad));
    memset(opad, 0, sizeof(opad));
    
    if (secret_len > 64) 
    {
	if ((error_code = gcry_md_open(&context, GCRY_MD_MD5, 0)) != 0)
	{
	    return merror(MD5_ELIBFAILED, 
		    "gcry_md_open() failed (source: %s; error: %s)",
		    gcry_strsource(error_code), gcry_strerror(error_code));
	}
	gcry_md_write(context, (unsigned char *)secret, secret_len);
	gcry_md_final(context);
	memcpy(ipad, gcry_md_read(context, 0), 64);
	memcpy(opad, gcry_md_read(context, 0), 64);
	gcry_md_close(context);
    }
    else 
    {
	memcpy(ipad, secret, secret_len);
	memcpy(opad, secret, secret_len);
    }
    
    for (i = 0; i < 64; i++)
    {
	ipad[i] ^= 0x36;
	opad[i] ^= 0x5c;
    }
    
    if ((error_code = gcry_md_open(&context, GCRY_MD_MD5, 0)) != 0)
    {
	return merror(MD5_ELIBFAILED, 
		"gcry_md_open() failed (source: %s; error: %s)",
    		gcry_strsource(error_code), gcry_strerror(error_code));
    }
    gcry_md_write(context, ipad, 64);
    gcry_md_write(context, (unsigned char *)challenge, challenge_len);
    gcry_md_final(context);
    memcpy(digest, gcry_md_read(context, 0), 16);
    gcry_md_close(context);
    
    if ((error_code = gcry_md_open(&context, GCRY_MD_MD5, 0)) != 0)
    {
	return merror(MD5_ELIBFAILED, 
		"gcry_md_open() failed (source: %s; error: %s)",
    		gcry_strsource(error_code), gcry_strerror(error_code));
    }
    gcry_md_write(context, opad, 64);
    gcry_md_write(context, digest, 16);
    gcry_md_final(context);
    memcpy(digest, gcry_md_read(context, 0), 16);
    gcry_md_close(context);
    
    return merror(EOK, NULL);
	
#else
    
    MD5_CTX context;
    unsigned char ipad[64];
    unsigned char opad[64];
    int i;
    
    memset(digest, 0, (size_t)16);
    memset(ipad, 0, sizeof(ipad));
    memset(opad, 0, sizeof(opad));
    
    if (secret_len > 64) 
    {
	MD5_Init(&context);
	MD5_Update(&context, (unsigned char *)secret, secret_len);
	MD5_Final(ipad, &context);
	MD5_Final(opad, &context);
    }
    else 
    {
	memcpy(ipad, secret, secret_len);
	memcpy(opad, secret, secret_len);
    }
    
    for (i = 0; i < 64; i++)
    {
	ipad[i] ^= 0x36;
	opad[i] ^= 0x5c;
    }
    
    MD5_Init(&context);
    MD5_Update(&context, ipad, (size_t)64);
    MD5_Update(&context, (unsigned char *)challenge, challenge_len);
    MD5_Final(digest, &context);
    
    MD5_Init(&context);
    MD5_Update(&context, opad, (size_t)64);
    MD5_Update(&context, digest, (size_t)16);
    MD5_Final(digest, &context);
    
    return merror(EOK, NULL);
    
#endif
}
