/*
 * paths.c
 *
 * This file is part of msmtp, an SMTP client.
 *
 * Copyright (C) 2004
 * Martin Lambers <marlam@users.sourceforge.net>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   msmtp is released under the GPL with the additional exemption that
 *   compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <errno.h>
extern int errno;

#ifdef WINDOWS
#include <windows.h>
#endif
#ifndef WINDOWS
#ifndef DJGPP
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <pwd.h>
#endif
#endif

#include "merror.h"
#include "paths.h"


#ifdef WINDOWS
#define PATH_SEP '\\'
#else /* UNIX or DJGPP */
#define PATH_SEP '/'
#endif


/*
 * get_prgname()
 *
 * see paths.h
 */

char *get_prgname(char *argv0)
{
    char *prgname;
    
    prgname = strrchr(argv0, PATH_SEP);
    if (!prgname)
    {
	prgname = argv0;
    }
    else
    {
	prgname++;
    }
    
    return prgname;
}


/*
 * get_homedir()
 *
 * Get the users home directory (or something similar, depending on the OS).
 * Returns a pointer to an allocated string in 'h'.
 * Used error codes: PATHS_ENOHOME
 */

merror_t get_homedir(char **h)
{
#ifdef WINDOWS

    BYTE home[MAX_PATH + 1];
    HKEY hkey;
    DWORD len;
    DWORD type;
    long l;
    
    l = RegOpenKeyEx(HKEY_CURRENT_USER,
	    "Software\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders", 
	    0, KEY_READ, &hkey);
    if (l != ERROR_SUCCESS)
    {
	return merror(PATHS_ENOHOME, "cannot read registry");
    }
    len = MAX_PATH;
    l = RegQueryValueEx(hkey, "AppData", NULL, &type, home, &len);
    if (l != ERROR_SUCCESS || len >= MAX_PATH)
    {
	RegCloseKey(hkey);
	return merror(PATHS_ENOHOME, "cannot read registry");
    }
    RegCloseKey(hkey);
    if (!(*h = strdup((char *)home)))
    {
	return merror(PATHS_ENOMEM, "%s", strerror(errno));
    }    
    return merror(EOK, NULL);

#elif defined DJGPP
    
    char *home;
    
    if (!(home = getenv("HOME")))
    {
	return merror(PATHS_ENOHOME, "no environment variable HOME");
    }
    if (!(*h = strdup(home)))
    {
	return merror(PATHS_ENOMEM, "%s", strerror(errno));
    }    
    return merror(EOK, NULL);

#else /* UNIX */

    char *home;
    struct passwd *pw;
    
    if (!(home = getenv("HOME")))
    {
	pw = getpwuid(getuid());
	if (!pw || !(pw->pw_dir))
	{
	    return merror(PATHS_ENOHOME, "cannot get user's home directory");
	}
	home = pw->pw_dir;
    }
    if (!(*h = strdup(home)))
    {
	return merror(PATHS_ENOMEM, "%s", strerror(errno));
    }    
    return merror(EOK, NULL);

#endif
}


/*
 * get_default_conffile()
 *
 * see paths.h
 */

merror_t get_default_conffile(char *base, char **dst)
{
    merror_t e;
    char *conffile;
    size_t len;
    
    if (!merror_ok(e = get_homedir(&conffile)))
    {
	return e;
    }
    len = strlen(conffile);
    conffile = realloc(conffile, (len + strlen(base) + 2) * sizeof(char));
    if (!conffile)
    {
	return merror(PATHS_ENOMEM, "%s", strerror(errno));
    }
    if (len > 0 && conffile[len - 1] != PATH_SEP)
    {
	conffile[len++] = PATH_SEP;
    }
    strcpy(conffile + len, base);

    *dst = conffile;
    return merror(EOK, NULL);
}


/*
 * expand_tilde()
 *
 * see paths.h
 */

char *expand_tilde(char *filename)
{
    size_t homedirlen;
    char *new_filename;
    
    if (filename[0] == '~' && merror_ok(get_homedir(&new_filename)))
    {
	homedirlen = strlen(new_filename);
	if (!(new_filename = realloc(new_filename, 
		(homedirlen + strlen(filename)) * sizeof(char))))
	{
	    return filename;
	}
	strcpy(new_filename + homedirlen, filename + 1);
	free(filename);
	return new_filename;
    }
    else
    {
	return filename;
    }	
}
