/*
 * smtp.c
 *
 * This file is part of msmtp, an SMTP client.
 *
 * Copyright (C) 2000, 2003, 2004
 * Martin Lambers <marlam@users.sourceforge.net>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   msmtp is released under the GPL with the additional exemption that
 *   compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
extern int errno;

#ifdef USE_GSASL
#include <gsasl.h>
#else
#include "base64.h"
#include "md5_apps.h"
#endif

#include "merror.h"
#include "list.h"
#include "net.h"
#include "smtp.h"
#ifdef HAVE_SSL
#include "tls.h"
#endif /* HAVE_SSL */


/* This defines the length of the *input* buffer for SMTP messages.
 * According to RFC 2821, SMTP commands and SMTP messages may contain
 * at most 512 characters including CRLF, thus the *minimum* size is
 * 512 characters.
 */
#define SMTP_BUFSIZE 1024

/* This defines the maximum number of lines in a multiline server reply.
 * This limit exists to prevent an extremely long reply from eating
 * all the memory.
 * The longest server reply we have to expect is the repsonse to the EHLO
 * command. We should have enough lines for every SMTP extension known today
 * plus more lines for future extensions.
 */
#define SMTP_MAXLINES 50

/* This is the buffer length for copying the mail to the SMTP server.
 * According to RFC 2822, a line in a mail can contain at most 998 
 * characters + CRLF. Plus one character for '\0' = 1001 characters.
 * All lines should fit in a buffer of this size.
 * However, this length is not a limit; smtp_copy_mail() will accept 
 * arbitrary long lines. It is the responsibility of your MUA to construct
 * RFC compliant mails!
 */
#define MAIL_BUFSIZE 1024


/*
 * smtp_new()
 *
 * see smtp.h
 */

smtp_server_t smtp_new(FILE *debug)
{
    smtp_server_t srv;

    srv.fd = -1;
#ifdef HAVE_SSL
    tls_clear(&srv.tls);
#endif /* HAVE_SSL */
    srv.cap.flags = 0;
    srv.cap.size = 0;
    srv.debug = debug;
    return srv;
}


/*
 * smtp_connect()
 *
 * see smtp.h
 */

merror_t smtp_connect(smtp_server_t *srv, char *host, int port)
{
    return net_open_socket(host, port, &srv->fd);
}


/*
 * smtp_get_msg()
 * 
 * This function gets a message from the SMTP server 'srv'.
 * In case of success, 'msg' will contain a pointer to a newly created list, 
 * and each member of this list will contain one line of the message as an 
 * allocated string. The list will contain at least one line. Each line will 
 * be at least 4 characters long: The three digit status code plus a ' ' or '-'.
 * The return code will be EOK.
 * In case of failure, 'msg' will be NULL, and one of the following error codes
 * will be returned: SMTP_EIO, SMTP_ENOMEM, SMTP_EPROTO
 */

merror_t smtp_get_msg(smtp_server_t *srv, list_t **msg)
{
    merror_t e;
    list_t *l;
    list_t *lp;
    char line[SMTP_BUFSIZE];
    char *s;
    int counter;
    size_t len;

    *msg = NULL;
    if (!(l = list_new()))
    {
	return merror(SMTP_ENOMEM, "%s", strerror(errno));
    }
    lp = l;

    counter = 0;
    do
    {
#ifdef HAVE_SSL
	if (tls_is_active(&srv->tls))
	{
	    e = tls_gets(&srv->tls, line, SMTP_BUFSIZE);
	}
	else
	{
#endif /* HAVE_SSL */
	    e = net_gets(srv->fd, line, SMTP_BUFSIZE);
#ifdef HAVE_SSL
	}
#endif /* HAVE_SSL */
	if (!merror_ok(e))
	{
	    list_xfree(l, free);
	    return merror(SMTP_EIO, "%s", e.errstr);
	}
	len = strlen(line);
	if (len < 4 
		|| !(isdigit((unsigned char)line[0]) 
		    && isdigit((unsigned char)line[1]) 
		    && isdigit((unsigned char)line[2]) 
		    && (line[3] == ' ' || line[3] == '-'))
		|| line[len - 1] != '\n')
	{
	    list_xfree(l, free);
	    return merror(SMTP_EPROTO, "SMTP server sent an invalid response");
	}
	/* kill CRLF */
	line[len - 1] = '\0';
	if (line[len - 2] == '\r')
	{
	    line[len - 2] = '\0';
	}
	if (!(s = strdup(line)) || (list_insert(lp, s) != 0))
	{
	    list_xfree(l, free);
	    return merror(SMTP_ENOMEM, "%s", strerror(errno));
	}
	counter++;
	lp = lp->next;
	if (srv->debug)
	{
	    fprintf(srv->debug, "<-- %s\n", line);
	}
    }
    while (line[3] == '-' && counter <= SMTP_MAXLINES);

    if (counter > SMTP_MAXLINES)
    {
	list_xfree(l, free);
	return merror(SMTP_EPROTO, 
		"SMTP server reply is longer than %d lines", SMTP_MAXLINES);
    }

    *msg = l;
    return merror(EOK, NULL);
}


/*
 * smtp_msg_status()
 *
 * see smtp.h
 */

int smtp_msg_status(list_t *msg)
{
    return atoi(msg->next->data);
}


/*
 * smtp_put()
 *
 * This function writes a string to the SMTP server 'srv'. If 'newline' is set,
 * it will write TCP CRLF ('\r\n') after the string.
 * Used error codes: SMTP_EIO
 */

merror_t smtp_put(smtp_server_t *srv, char *s, int newline)
{
    merror_t e;

#ifdef HAVE_SSL
    if (tls_is_active(&srv->tls))
    {
	if (merror_ok(e = tls_puts(&srv->tls, s)) && newline)
	{
	    e = tls_puts(&srv->tls, "\r\n");
	}
    }
    else
    {
#endif /* HAVE_SSL */
	if (merror_ok(e = net_puts(srv->fd, s)) && newline)
	{
	    e = net_puts(srv->fd, "\r\n");
	}
#ifdef HAVE_SSL
    }
#endif /* HAVE_SSL */
    if (!merror_ok(e))
    {
	return merror(SMTP_EIO, "%s", e.errstr);
    }
    if (srv->debug)
    {
	fprintf(srv->debug, "--> %s\n", s);
    }
    
    return merror(EOK, NULL);
}


/*
 * smtp_send_cmd()
 *
 * This function writes a string to the SMTP server 'srv'. The string may not 
 * be longer than 510 characters (see RFC 2821). TCP CRLF ('\r\n') will be 
 * appended to the string. Use this function to send SMTP commands (as opposed
 * to mail data) to the SMTP server.
 * Used error codes: SMTP_EIO, SMTP_EINVAL
 */

merror_t smtp_send_cmd(smtp_server_t *srv, char *format, ...)
{
    /* max length of SMTP commands is 510 characters according to RFC 2821 */
    char line[510];
    int count;
    va_list args;

    va_start(args, format);
    count = vsnprintf(line, 510, format, args);	
    va_end(args);
    if (count >= 510)
    {
	return merror(SMTP_EINVAL, "SMTP command longer than 510 characters; "
		"this violates RFC 2821");
    }
    return smtp_put(srv, line, 1);
}


/*
 * smtp_get_greeting()
 *
 * see smtp.h
 */

merror_t smtp_get_greeting(smtp_server_t *srv, list_t **errmsg, char **buf)
{
    merror_t e;
    list_t *msg;
    
    *errmsg = NULL;
    if (!merror_ok(e = smtp_get_msg(srv, &msg)))
    {
	return e;
    }
    if (smtp_msg_status(msg) != 220)
    {
	*errmsg = msg;
	return merror(SMTP_EPROTO, "Cannot get initial OK message from SMTP server");	
    }
    if (buf)
    {
	if (!(*buf = malloc((strlen((char *)msg->next->data + 4) + 1) * sizeof(char))))
	{
	    list_xfree(msg, free);
	    return merror(SMTP_ENOMEM, "%s", strerror(errno));
	}
	strcpy(*buf, (char *)(msg->next->data) + 4);
    }
    list_xfree(msg, free);
    
    return merror(EOK, NULL);
}


/*
 * smtp_init()
 *
 * see smtp.h
 */

merror_t smtp_init(smtp_server_t *srv, char *ehlo_domain, list_t **errmsg)
{
    merror_t e;
    list_t *ehlo_response;
    list_t *lp;
    char *s;
    size_t len;
    int i;
    
    srv->cap.flags = 0;
    srv->cap.size = 0;

    *errmsg = NULL;
    if (!merror_ok(e = smtp_send_cmd(srv, "EHLO %s", ehlo_domain)))
    {
	return e;
    }
    if (!merror_ok(e = smtp_get_msg(srv, &ehlo_response)))
    {
	return e;
    }
    if (smtp_msg_status(ehlo_response) != 250)
    {
	/* fall back to HELO */
	list_xfree(ehlo_response, free);
	if (!merror_ok(e = smtp_send_cmd(srv, "HELO %s", ehlo_domain)))
	{
	    return e;
	}
	if (!merror_ok(e = smtp_get_msg(srv, &ehlo_response)))
	{
	    return e;
	}
	if (smtp_msg_status(ehlo_response) != 250)
	{
	    *errmsg = ehlo_response;
	    return merror(SMTP_EPROTO, 
		    "SMTP server does not accept EHLO or HELO commands");
	}
	list_xfree(ehlo_response, free);
	/* srv->cap.flags is 0 */
	return merror(EOK, NULL);
    }

    lp = ehlo_response;
    while (!list_is_empty(lp))
    {
	lp = lp->next;
	s = lp->data;
	len = strlen(s);
	/* we know that len is >= 4 */
	/* make line uppercase */
    	for (i = 4; (size_t)i < len; i++)
	{
	    s[i] = toupper((unsigned char)s[i]);
	}
	/* search capabilities */
	if (strncmp(s + 4, "STARTTLS", 8) == 0)
	{
    	    srv->cap.flags |= SMTP_CAP_STARTTLS;
	}
	else if (strncmp(s + 4, "DSN", 3) == 0)
	{
	    srv->cap.flags |= SMTP_CAP_DSN;
	}
	else if (strncmp(s + 4, "PIPELINING", 10) == 0)
	{
	    srv->cap.flags |= SMTP_CAP_PIPELINING;
	}
	else if (strncmp(s + 4, "SIZE", 4) == 0)
	{
	    srv->cap.flags |= SMTP_CAP_SIZE;
	    srv->cap.size = atol(s + 8);
	}
     	else if (strncmp(s + 4, "AUTH ", 5) == 0)
	{
    	    srv->cap.flags |= SMTP_CAP_AUTH;
	    if (strstr(s + 9, "PLAIN"))
	    {
    		srv->cap.flags |= SMTP_CAP_AUTH_PLAIN;
	    }
	    if (strstr(s + 9, "LOGIN"))
	    {
		srv->cap.flags |= SMTP_CAP_AUTH_LOGIN;
	    }
	    if (strstr(s + 9, "CRAM-MD5"))
	    {
		srv->cap.flags |= SMTP_CAP_AUTH_CRAM_MD5;
	    }
	    if (strstr(s + 9, "DIGEST-MD5"))
	    {
		srv->cap.flags |= SMTP_CAP_AUTH_DIGEST_MD5;
	    }
	    if (strstr(s + 9, "NTLM"))
	    {
		srv->cap.flags |= SMTP_CAP_AUTH_NTLM;
	    }
	}
    }

    list_xfree(ehlo_response, free);
    return merror(EOK, NULL);
}


/*
 * smtp_tls_init()
 *
 * see smtp.h
 */

#ifdef HAVE_SSL
merror_t smtp_tls_init(smtp_server_t *srv, 
	char *tls_key_file, char *tls_ca_file, char *tls_trust_file)
{
    return tls_init(&srv->tls, tls_key_file, tls_ca_file, tls_trust_file);
}
#endif /* HAVE_SSL */


/*
 * smtp_tls_starttls()
 *
 * see smtp.h
 */

#ifdef HAVE_SSL
merror_t smtp_tls_starttls(smtp_server_t *srv, list_t **error_msg)
{
    merror_t e;
    list_t *msg;

    *error_msg = NULL;
    if (!merror_ok(e = smtp_send_cmd(srv, "STARTTLS")))
    {
	return e;
    }
    if (!merror_ok(e = smtp_get_msg(srv, &msg)))
    {
	return e;
    }
    if (smtp_msg_status(msg) != 220)
    {
	*error_msg = msg;
	return merror(SMTP_EPROTO, 
		"SMTP server does not accept the STARTTLS command");
    }
    return merror(EOK, NULL);
}
#endif /* HAVE_SSL */


/*
 * smtp_tls()
 *
 * see smtp.h
 */

#ifdef HAVE_SSL
merror_t smtp_tls(smtp_server_t *srv, char *hostname, int tls_nocertcheck)
{
    return tls_start(&srv->tls, srv->fd, hostname, tls_nocertcheck);
}
#endif /* HAVE_SSL */


/*
 * smtp_auth_plain()
 * 
 * Do SMTP authentication via AUTH PLAIN.
 * The SMTP server must support SMTP_CAP_AUTH_PLAIN
 * Used error codes: SMTP_EIO, SMTP_EPROTO, SMTP_ENOMEM,
 * 		     SMTP_EAUTHFAIL, SMTP_EINVAL
 */

#ifndef USE_GSASL
merror_t smtp_auth_plain(
	smtp_server_t *srv, char *user, char *password, list_t **error_msg)
{
    char *s;
    char *b64;
    size_t u_len;
    size_t p_len;
    list_t *msg;
    merror_t e;
    int status;

    *error_msg = NULL;
    u_len = strlen(user);
    p_len = strlen(password);
    if (!(s = malloc((u_len + p_len + 3) * sizeof(char))))
    {
	return merror(SMTP_ENOMEM, "%s", strerror(errno));
    }
    s[0] = '\0';
    strcpy(s + 1, user);
    strcpy(s + u_len + 2, password);
    if (!merror_ok(e = base64enc(s, 1 + u_len + 1 + p_len, &b64)))
    {
	free(s);
	return merror(SMTP_ENOMEM, "%s", e.errstr);
    }
    free(s);
    
    if (!merror_ok(e = smtp_send_cmd(srv, "AUTH PLAIN %s", b64)))
    {
	free(b64);
	return e;
    }
    free(b64);
    if (!merror_ok(e = smtp_get_msg(srv, &msg)))
    {
	return e;
    }
    if ((status = smtp_msg_status(msg)) != 235)
    {
	*error_msg = msg;
	if (status == 504)
	{
	    return merror(SMTP_EPROTO, "AUTH PLAIN not accepted by SMTP server");
	}
	else
	{
	    return merror(SMTP_EAUTHFAIL, "SMTP server does not "
		"accept username/password (AUTH PLAIN)");
	}
    }
    list_xfree(msg, free);

    return merror(EOK, NULL);
}
#endif /* !USE_GSASL */


/*
 * smtp_auth_login()
 * 
 * Do SMTP authentication via AUTH LOGIN.
 * The SMTP server must support SMTP_CAP_AUTH_LOGIN
 * Used error codes: SMTP_EIO, SMTP_EPROTO, SMTP_ENOMEM,
 * 		     SMTP_EAUTHFAIL, SMTP_EINVAL
 */

#ifndef USE_GSASL
merror_t smtp_auth_login(
	smtp_server_t *srv, char *user, char *password, list_t **error_msg)
{
    merror_t e;
    list_t *msg;
    char *b64;
    
    *error_msg = NULL;
    if (!merror_ok(e = smtp_send_cmd(srv, "AUTH LOGIN")))
    {
	return e;
    }
    if (!merror_ok(e = smtp_get_msg(srv, &msg)))
    {
	return e;
    }
    if (smtp_msg_status(msg) != 334)
    {
	*error_msg = msg;
	return merror(SMTP_EPROTO, "AUTH LOGIN not accepted by SMTP server");
    }
    list_xfree(msg, free);
    if (!merror_ok(e = base64enc(user, strlen(user), &b64)))
    {
	return merror(SMTP_ENOMEM, "%s", e.errstr);
    }
    if (!merror_ok(e = smtp_send_cmd(srv, "%s", b64)))
    {
	free(b64);
	return e;
    }
    free(b64);
    if (!merror_ok(e = smtp_get_msg(srv, &msg)))
    {
	return e;
    }
    if (smtp_msg_status(msg) != 334)
    {
	*error_msg = msg;
	return merror(SMTP_EAUTHFAIL, 
		"SMTP server does not accept username (AUTH LOGIN)");
    }
    list_xfree(msg, free);
    if (!merror_ok(e = base64enc(password, strlen(password), &b64)))
    {
	return merror(SMTP_ENOMEM, "%s", e.errstr);
    }
    if (!merror_ok(e = smtp_send_cmd(srv, "%s", b64)))
    {
	free(b64);
	return e;
    }
    free(b64);
    if (!merror_ok(e = smtp_get_msg(srv, &msg)))
    {
	return e;
    }
    if (smtp_msg_status(msg) != 235)
    {
	*error_msg = msg;
	return merror(SMTP_EAUTHFAIL, 
		"SMTP server does not accept username/password (AUTH LOGIN)");
    }
    list_xfree(msg, free);

    return merror(EOK, NULL);
}
#endif /* !USE_GSASL */


/*
 * smtp_auth_cram_md5()
 *
 * Do SMTP authentication via AUTH CRAM-MD5.
 * The SMTP server must support SMTP_CAP_AUTH_CRAM_MD5
 * Used error codes: SMTP_EIO, SMTP_EPROTO, SMTP_ENOMEM,
 * 		     SMTP_EAUTHFAIL, SMTP_ELIBFAILED, SMTP_EINVAL
 */

#ifndef USE_GSASL
merror_t smtp_auth_cram_md5(
	smtp_server_t *srv, char *user, char *password, list_t **error_msg)
{
    unsigned char digest[16];
    char hex[16] = 
    	{ '0','1','2','3','4','5','6','7','8','9','a','b','c','d','e','f' };
    char *b64;
    char *buf;
    char *p;
    int len;
    int i;
    list_t *msg;
    merror_t e;
    
    *error_msg = NULL;
    if (!merror_ok(e = smtp_send_cmd(srv, "AUTH CRAM-MD5")))
    {
	return e;
    }
    if (!merror_ok(e = smtp_get_msg(srv, &msg)))
    {
	return e;
    }
    if (smtp_msg_status(msg) != 334)
    {
	*error_msg = msg;
	return merror(SMTP_EPROTO, "AUTH CRAM-MD5 not accepted by SMTP server");
    }
    /* we know the line is at least 4 characters long */
    if (!merror_ok(e = base64dec((char *)(msg->next->data) + 4, &b64, &len)))
    {
	list_xfree(msg, free);
	if (e.number == BASE64_ENOMEM)
	{
	    return merror(SMTP_ENOMEM, "%s", e.errstr);
	}
	else
	{
	    return merror(SMTP_EPROTO, 
		    "AUTH CRAM-MD5: server sent invalid challenge");
	}
    }
    list_xfree(msg, free);
    if (!merror_ok(e = md5_hmac(password, strlen(password), b64, len, digest)))
    {
	free(b64);
	return merror(SMTP_ELIBFAILED, "%s", e.errstr);
    }
    free(b64);
    
    /* construct username + ' ' + digest_in_hex */
    len = (int)strlen(user);
    if (!(buf = malloc((len + 1 + 32 + 1) * sizeof(char))))
    {
	return merror(SMTP_ENOMEM, "%s", strerror(errno));
    }
    strcpy(buf, user);
    p = buf + len;
    *p++ = ' ';
    for (i = 0; i < 16; i++)
    {
	p[2 * i] = hex[digest[i] >> 4];
	p[2 * i + 1] = hex[digest[i] & 0x0f];
    }
    p[32] = '\0';
    
    if (!merror_ok(e = base64enc(buf, len + 33, &b64)))
    {
	free(buf);
	return merror(SMTP_ENOMEM, "%s", e.errstr);
    }
    free(buf);
    if (!merror_ok(e = smtp_send_cmd(srv, "%s", b64)))
    {
	free(b64);
	return e;
    }
    free(b64);
    if (!merror_ok(e = smtp_get_msg(srv, &msg)))
    {
	return e;
    }
    if (smtp_msg_status(msg) != 235)
    {
	*error_msg = msg;
	return merror(SMTP_EAUTHFAIL, "SMTP server does not "
	    	    "accept username/password (AUTH CRAM-MD5)");
    }
    list_xfree(msg, free);

    return merror(EOK, NULL);
}
#endif /* !USE_GSASL */


/*
 * GSASL callbacks for authentication
 */

#ifdef USE_GSASL

char *auth_username;
char *auth_password;
char *auth_hostname;
char *auth_servicename;

int callback_username(Gsasl_session_ctx *ctx, char *out, size_t *outlen)
{
    *outlen = strlen(auth_username);
    if (out)
    {
	strcpy(out, auth_username);
    }
    return GSASL_OK;
}

int callback_password(Gsasl_session_ctx *ctx, char *out, size_t *outlen)
{
    *outlen = strlen(auth_password);
    if (out)
    {
	strcpy(out, auth_password);
    }
    return GSASL_OK;
}

int callback_service(Gsasl_session_ctx *ctx, 
	char *service, size_t *servicelen, 
	char *hostname, size_t *hostnamelen, 
	char *servicename, size_t *servicenamelen)
{
    *servicelen = 4;
    *hostnamelen = strlen(auth_hostname);
    *servicenamelen = 0;
    if (service)
    {
	strcpy(service, "smtp");
    }
    if (hostname)
    {
	strcpy(hostname, auth_hostname);
    }
    if (servicename)
    {
	strcpy(servicename, "");
    }
    return GSASL_OK;
}
#endif /* USE_GSASL */


/*
 * smtp_authmech_is_supported()
 *
 * see smtp.h
 */

int smtp_authmech_is_supported(char *mech)
{
#ifdef USE_GSASL

    int supported = 0;
    Gsasl_ctx *ctx;
    
    if (gsasl_init(&ctx) != GSASL_OK)
    {
	return 0;
    }
    supported = gsasl_client_support_p(ctx, mech);
    gsasl_done(ctx);
    return supported;
    
#else /* not USE_GSASL */
    
    if (strcmp(mech, "CRAM-MD5") == 0
	    || strcmp(mech, "PLAIN") == 0
	    || strcmp(mech, "LOGIN") == 0)
    {
	return 1;
    }
    else /* DIGEST-MD5 or NTLM */
    {
	return 0;
    }
    
#endif /* not USE_GSASL */
}


/*
 * smtp_auth()
 *
 * see smtp.h
 */

merror_t smtp_auth(smtp_server_t *srv,
	char *hostname,
	char *user, 
	char *password,
	char *auth_mech,
	list_t **error_msg)
{
#ifdef USE_GSASL
    merror_t e;
    list_t *msg;
    Gsasl_ctx *ctx;
    Gsasl_session_ctx *cctx;
    char *input;
    char inbuf[SMTP_BUFSIZE];
    char *outbuf;
    int error_code;
    int auth_plain_special;

    *error_msg = NULL;
    if ((error_code = gsasl_init(&ctx)) != GSASL_OK)
    {
	return merror(SMTP_ELIBFAILED, "GNU SASL: %s", 
		gsasl_strerror(error_code));
    }
    if (strcmp(auth_mech, "") != 0 && !gsasl_client_support_p(ctx, auth_mech))
    {
	return merror(SMTP_ELIBFAILED, "GNU SASL: authentication mechanism %s "
		"not supported", auth_mech);
    }
    if (strcmp(auth_mech, "") == 0)
    {
	/* Choose "best" authentication mechanism. */
	/* TODO: use gsasl_client_suggest_mechanism()? */
	if (gsasl_client_support_p(ctx, "DIGEST-MD5") 
		&& (srv->cap.flags & SMTP_CAP_AUTH_DIGEST_MD5))
	{
	    auth_mech = "DIGEST-MD5";
	}
	else if (gsasl_client_support_p(ctx, "CRAM-MD5") 
		&& (srv->cap.flags & SMTP_CAP_AUTH_CRAM_MD5))
	{
	    auth_mech = "CRAM-MD5";
	}
	else if (gsasl_client_support_p(ctx, "NTLM") 
		&& (srv->cap.flags & SMTP_CAP_AUTH_NTLM))
	{
	    auth_mech = "NTLM";
	}
#ifdef HAVE_SSL
	else if (tls_is_active(&srv->tls))
	{
	    if (gsasl_client_support_p(ctx, "PLAIN") 
		    && (srv->cap.flags & SMTP_CAP_AUTH_PLAIN))
	    {
		auth_mech = "PLAIN";
	    }
	    else if (gsasl_client_support_p(ctx, "LOGIN") 
		    && (srv->cap.flags & SMTP_CAP_AUTH_LOGIN))
	    {
		auth_mech = "LOGIN";
	    }
	}
#endif /* HAVE_SSL */
    }
    if (strcmp(auth_mech, "") == 0)
    {
	return merror(SMTP_EUNAVAIL, "GNU SASL: cannot agree on "
		"authentication mechanism");
    }
    
    auth_username = user;
    auth_password = password;
    auth_hostname = hostname;
    gsasl_client_callback_authentication_id_set(ctx, callback_username);
    gsasl_client_callback_authorization_id_set(ctx, callback_username);
    gsasl_client_callback_password_set(ctx, callback_password);
    gsasl_client_callback_service_set(ctx, callback_service);
    if ((error_code = gsasl_client_start(ctx, auth_mech, &cctx)) != GSASL_OK)
    {
	gsasl_done(ctx);
	return merror(SMTP_ELIBFAILED, "GNU SASL: %s", gsasl_strerror(error_code));
    }    
    input = NULL;
    do
    {
	error_code = gsasl_step64(cctx, input, &outbuf);
	if (error_code != GSASL_OK && error_code != GSASL_NEEDS_MORE)
	{
	    gsasl_done(ctx);
	    return merror(SMTP_ELIBFAILED, "GNU SASL: %s", gsasl_strerror(error_code));
	}
	if (!input)
	{
	    if (strcmp(auth_mech, "PLAIN") == 0 && outbuf[0])
	    {
		/* AUTH PLAIN needs special treatment because it needs to send the
		 * authentication data together with the AUTH PLAIN command */
		auth_plain_special = 1;
		if (!merror_ok(e = smtp_send_cmd(srv, "AUTH PLAIN %s", outbuf)))
		{
		    gsasl_done(ctx);
		    free(outbuf);
		    return e;
		}
	    }
	    else
	    {	    
		auth_plain_special = 0;
		if (!merror_ok(e = smtp_send_cmd(srv, "AUTH %s", auth_mech)))
		{
		    gsasl_done(ctx);
		    free(outbuf);
		    return e;
		}
	    }
	    if (!merror_ok(e = smtp_get_msg(srv, &msg)))
	    {
		gsasl_done(ctx);
	    	free(outbuf);
		return e;
	    }
	    if (smtp_msg_status(msg) != 334 && smtp_msg_status(msg) != 235)
	    {
		*error_msg = msg;
		gsasl_done(ctx);
		free(outbuf);
		return merror(SMTP_EPROTO, 
			"GNU SASL authentication error in method %s", auth_mech);
	    }	    
	    strncpy(inbuf, msg->next->data, SMTP_BUFSIZE);
	    list_xfree(msg, free);
	    input = inbuf + 4;
	    if (auth_plain_special)
	    {
		free(outbuf);
		continue;
	    }
	}
	if (outbuf[0])
	{
	    if (!merror_ok(e = smtp_send_cmd(srv, "%s", outbuf)))
	    {
		gsasl_done(ctx);
		free(outbuf);
		return e;
	    }
	    if (!merror_ok(e = smtp_get_msg(srv, &msg)))
	    {
		gsasl_done(ctx);
		free(outbuf);
		return e;
	    }
	    if (smtp_msg_status(msg) != 334 && smtp_msg_status(msg) != 235)
	    {
		*error_msg = msg;
		gsasl_done(ctx);
		free(outbuf);
		return merror(SMTP_EPROTO, 
			"GNU SASL authentication error in method %s", auth_mech);
	    }	    
	    strncpy(inbuf, msg->next->data, SMTP_BUFSIZE);
	    list_xfree(msg, free);
	    input = inbuf + 4;
	}
	free(outbuf);
    }
    while (error_code == GSASL_NEEDS_MORE);
    if (error_code != GSASL_OK)
    {
	gsasl_done(ctx);
	return merror(SMTP_EAUTHFAIL, "GNU SASL authentication mechanism %s failed: %s", 
		auth_mech, gsasl_strerror(error_code));
    }
    gsasl_done(ctx);
    /* For DIGEST-MD5, we need to send an empty answer to the last 334 
     * response before we get 235. */
    if (strncmp(inbuf, "235 ", 4) != 0)
    {
	if (!merror_ok(e = smtp_send_cmd(srv, "")))
	{
	    return e;
	}
	if (!merror_ok(e = smtp_get_msg(srv, &msg)))
	{
	    return e;
	}
	if (smtp_msg_status(msg) != 235)
	{
	    *error_msg = msg;
	    return merror(SMTP_EAUTHFAIL, 
		    "GNU SASL authentication mechanism %s failed", auth_mech);
	}
	list_xfree(msg, free);
    }
    return merror(EOK, NULL);

#else /* not USE_GSASL */
    
    *error_msg = NULL;
    if (strcmp(auth_mech, "") == 0)
    {
	/* Choose "best" authentication mechanism. */
	if (srv->cap.flags & SMTP_CAP_AUTH_CRAM_MD5)
	{
	    auth_mech = "CRAM-MD5";
	}
#ifdef HAVE_SSL
	else if (tls_is_active(&srv->tls))
	{
	    if (srv->cap.flags & SMTP_CAP_AUTH_PLAIN)
	    {
		auth_mech = "PLAIN";
	    }
	    else if (srv->cap.flags & SMTP_CAP_AUTH_LOGIN)
	    {
		auth_mech = "LOGIN";
	    }
	}
#endif /* HAVE_SSL */
    }
    if (strcmp(auth_mech, "") == 0)
    {
	return merror(SMTP_EUNAVAIL, "cannot agree on authentication mechanism");
    }

    if (strcmp(auth_mech, "CRAM-MD5") == 0)
    {
	return smtp_auth_cram_md5(srv, user, password, error_msg);
    }
    else if (strcmp(auth_mech, "PLAIN") == 0)
    {
	return smtp_auth_plain(srv, user, password, error_msg);
    }
    else if (strcmp(auth_mech, "LOGIN") == 0)
    {
	return smtp_auth_login(srv, user, password, error_msg);
    }
    else
    {
	return merror(SMTP_ELIBFAILED, "authentication mechanism %s "
		"not supported", auth_mech);
    }

#endif /* not USE_GSASL */
}


/*
 * smtp_copy_mail()
 *
 * Copies the mail from 'mailf' to 'srv'.
 * Converts '\n' to '\r\n' and a leading dot on a line to two leading dots; 
 * the SMTP server deletes a leading dot (RFC 2821, section 4.5.2).
 * Strips the Bcc header unless the keep_bcc flag is set. 
 * Stores the number of bytes transferred in 'mailsize'. In case of failure, the
 * contents of 'mailsize' are undefined. Note that only transferred bytes are
 * counted; the number of bytes in 'mailf' may be larger because of a Bcc
 * header.
 * Used error codes: SMTP_EIO
 */

merror_t smtp_copy_mail(smtp_server_t *srv, FILE *mailf, int keep_bcc, long *mailsize)
{
    char buffer[MAIL_BUFSIZE];
    size_t len;
    int in_header;
    int in_bcc;
    int line_starts;
    int line_continues;
    merror_t e;
    
    *mailsize = 0;
    in_header = 1;
    in_bcc = 0;
    line_continues = 0;
    e = merror(EOK, NULL);
    while (fgets(buffer, (int)sizeof(buffer), mailf) != NULL)
    {
	line_starts = !line_continues;
	len = strlen(buffer);
	if (len > 0 && buffer[len - 1] == '\n')
	{
	    buffer[--len] = '\0';
	    line_continues = 0;
	}
	else if (len == MAIL_BUFSIZE - 1)
	{
	    line_continues = 1;
	}
	else
	{
	    /* last line lacks a newline character */
	    line_continues = 0;
	}
	if (!keep_bcc)
	{
	    if (line_starts && in_header && buffer[0] == '\0')
	    {
		in_header = 0;
	    }
	    if (in_header)
	    {
		if (line_starts)
		{
		    if (!in_bcc)
		    {
			if (strncmp(buffer, "Bcc: ", 5) == 0)
			{
			    in_bcc = 1;
			    /* remove Bcc header by ignoring this line */
			    continue;
			}
		    }
		    else
		    {
			/* continued header lines begin with "horizontal whitespace"
			 * (RFC 2822, section 2.2.3) */
			if (buffer[0] == '\t' || buffer[0] == ' ')
			{
			    /* remove Bcc header by ignoring this line */
			    continue;
			}
			else
			{
			    in_bcc = 0;
			}
		    }
		}
		else
		{
		    if (in_bcc)
		    {
			/* remove Bcc header by ignoring this line */
		    	continue;
		    }
		}
	    }
	}
	if (line_starts && buffer[0] == '.')
	{
	    e = smtp_put(srv, ".", 0);
	    /* do not count the quoting dot in 'mailsize' */
	}
	if (merror_ok(e))
	{
    	    e = smtp_put(srv, buffer, !line_continues);
	    /* count a newline as one character */
	    *mailsize += (long)len + (line_continues ? 0 : 1);
	}
    	if (!merror_ok(e))
	{
	    return e;
	}
    }
    if (ferror(mailf))
    {
	return merror(SMTP_EIO, "input error while reading the mail");
    }

    return merror(EOK, NULL);
}


/*
 * smtp_send_mail()
 *
 * see smtp.h
 */

merror_t smtp_send_mail(smtp_server_t *srv,
	char *envelope_from, 
	int rcptc, char *rcptv[],
	char *dsn_notify,
	char *dsn_return,
	int keep_bcc,
	FILE *mailf,
	long *mailsize,
	list_t **error_msg)
{
    merror_t e;
    merror_t e_tmp;
    list_t *msg;
    int i;
    
    *error_msg = NULL;
    /* tell the server who wants to send mail */
    if (dsn_return)
    {
	e = smtp_send_cmd(srv, "MAIL FROM:<%s> RET=%s", envelope_from, dsn_return);
    }
    else
    {
	e = smtp_send_cmd(srv, "MAIL FROM:<%s>", envelope_from);
    }
    if (!merror_ok(e))
    {
	return e;
    }
    if (!(srv->cap.flags & SMTP_CAP_PIPELINING))
    {
	if (!merror_ok(e = smtp_get_msg(srv, &msg)))
	{
	    return e;
	}
	if (smtp_msg_status(msg) != 250)
	{
	    *error_msg = msg;
	    return merror(SMTP_EINVAL, 
		    "envelope from address %s not accepted by SMTP server", 
		    envelope_from);
	}
	list_xfree(msg, free);
    }
    /* tell the server who to send the mail to */
    for (i = 0; i < rcptc; i++)
    {
	if (dsn_notify)
	{
	    e = smtp_send_cmd(srv, "RCPT TO:<%s> NOTIFY=%s", rcptv[i], dsn_notify);
	}
	else
	{
	    e = smtp_send_cmd(srv, "RCPT TO:<%s>", rcptv[i]);
	}
	if (!merror_ok(e))
	{
	    return e;
	}
	if (!(srv->cap.flags & SMTP_CAP_PIPELINING))
	{
	    if (!merror_ok(e = smtp_get_msg(srv, &msg)))
	    {
		return e;
	    }
	    if (smtp_msg_status(msg) != 250)
	    {
		*error_msg = msg;
		return merror(SMTP_EINVAL, 
			"recipient address %s not accepted by SMTP server", 
			rcptv[i]);
	    }
	    list_xfree(msg, free);
	}
    }
    /* get all pipelined replies if pipelining is used */
    if (srv->cap.flags & SMTP_CAP_PIPELINING)
    {
	if (!merror_ok(e = smtp_get_msg(srv, &msg)))
	{
	    return e;
	}
     	if (smtp_msg_status(msg) != 250)
	{
	    *error_msg = msg;
	    e = merror(SMTP_EINVAL, 
		    "envelope from address %s not accepted by SMTP server", 
		    envelope_from);
	}
	else
	{
	    list_xfree(msg, free);
	}
	for (i = 0; i < rcptc; i++)
	{
	    if (!merror_ok(e_tmp = smtp_get_msg(srv, &msg)))
	    {
		return e_tmp;
	    }
    	    if (smtp_msg_status(msg) != 250)
	    {
		if (merror_ok(e))
		{
		    *error_msg = msg;
		    e = merror(SMTP_EINVAL, 
			    "recipient address %s not accepted by SMTP server", 
			    rcptv[i]);
		}
		else
		{
		    list_xfree(msg, free);
		}
	    }
	    else
	    {
		list_xfree(msg, free);
	    }
	}
	if (!merror_ok(e))
	{
	    return e;
	}
    }
    /* announce the mail */
    if (!merror_ok(e = smtp_send_cmd(srv, "DATA")))
    {
	return e;
    }
    if (!merror_ok(e = smtp_get_msg(srv, &msg)))
    {
	return e;
    }
    if (smtp_msg_status(msg) != 354)
    {
	*error_msg = msg;
	return merror(SMTP_EUNAVAIL, "SMTP server does not accept the mail");
    }
    list_xfree(msg, free);
    /* send the mail */
    if (!merror_ok(e = smtp_copy_mail(srv, mailf, keep_bcc, mailsize)))
    {
	return e;
    }
    /* end the mail */
    if (!merror_ok(e = smtp_put(srv, ".", 1)))
    {
	return e;
    }
    if (!merror_ok(e = smtp_get_msg(srv, &msg)))
    {
	return e;
    }
    if (smtp_msg_status(msg) != 250)
    {
	*error_msg = msg;
	return merror(SMTP_EUNAVAIL, "SMTP server did not accept the mail");
    }
    list_xfree(msg, free);
    
    return merror(EOK, NULL);
}


/*
 * smtp_quit()
 *
 * see smtp.h
 */

merror_t smtp_quit(smtp_server_t *srv)
{
    merror_t e;
    list_t *msg;

    if (merror_ok(e = smtp_send_cmd(srv, "QUIT")))
    {
	e = smtp_get_msg(srv, &msg);
    }
    if (msg)
    {
	list_xfree(msg, free);
    }
    return e;
}


/*
 * smtp_close()
 *
 * see smtp.h
 */

void smtp_close(smtp_server_t *srv)
{
#ifdef HAVE_SSL
    if (tls_is_active(&srv->tls))
    {
	tls_close(&srv->tls);
    }
#endif /* HAVE_SSL */
    net_close_socket(srv->fd);
}
