/*
 * tls.h
 * 
 * This file is part of msmtp, an SMTP client.
 *
 * Copyright (C) 2000, 2003, 2004
 * Martin Lambers <marlam@users.sourceforge.net>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   msmtp is released under the GPL with the additional exemption that
 *   compiling, linking, and/or using OpenSSL is allowed.
 */

#ifndef TLS_H
#define TLS_H

#ifdef HAVE_GNUTLS
#include <gnutls/gnutls.h>
#endif /* HAVE_GNUTLS */
#ifdef HAVE_OPENSSL
#include <openssl/ssl.h>
#endif /* HAVE_OPENSSL */

/*
 * Always use tls_clear() before using a tls_t!
 * Never call a tls_*() function with tls_t NULL!
 */

typedef struct
{
    int have_trust_file;
    int is_active;
#ifdef HAVE_GNUTLS
    gnutls_session session;
    gnutls_certificate_credentials cred;
#endif /* HAVE_GNUTLS */
#ifdef HAVE_OPENSSL
    SSL_CTX *ssl_ctx;
    SSL *ssl;
#endif /* HAVE_OPENSSL */
} tls_t;

#define TLS_ELIBFAILED	1	/* The underlying library failed */
#define TLS_ESEED	2	/* Cannot seed pseudo random number generator */
#define TLS_ECERT	3	/* Certificate check or verification failed */
#define TLS_EIO		4	/* Input/output error */
#define TLS_EFILE	5	/* A file does not exist/cannot be read */
#define TLS_EHANDSHAKE	6	/* TLS handshake failed */

/*
 * tls_lib_init()
 *
 * Initialize underlying TLS library.
 * Used error codes: TLS_ELIBFAILED
 */
merror_t tls_lib_init(void);

/*
 * tls_clear()
 *
 * Clears a tls_t type (marks it inactive).
 */
void tls_clear(tls_t *tls);

/*
 * tls_init()
 *
 * Initializes a tls_t. If both 'key_file' and 'ca_file' are not NULL, they are
 * set to be used when the peer request a certificate. If 'trust_file' is not
 * NULL, it will be used to verify the peer certificate.
 * All files must be in PEM format.
 * Used error codes: TLS_ELIBFAILED, TLS_EFILE
 */
merror_t tls_init(tls_t *tls, char *key_file, char *ca_file, char *trust_file);

/*
 * tls_start()
 *
 * Starts TLS encryption on a socket.
 * 'tls' must be initialized using tls_init().
 * If 'no_certcheck' is true, then no checks will be performed on the peer
 * certificate. If it is false and no trust file was set with tls_init(), 
 * only sanity checks are performed on the peer certificate. If it is false 
 * and a trust file was set, real verification of the peer certificate is performed.
 * 'hostname' is the host to start TLS with. It is needed for sanity checks/
 * verification.
 * Used error codes: TLS_ELIBFAILED, TLS_ECERT, TLS_EHANDSHAKE
 */
merror_t tls_start(tls_t *tls, int fd, char *hostname, int no_certcheck);

/*
 * tls_is_active()
 *
 * Returns whether 'tls' is an active TLS connection.
 */
int tls_is_active(tls_t *tls);

/*
 * tls_getchar()
 *
 * Reads a character using TLS and stores it in 'c'.
 * Sets the 'eof' flag to 1 if EOF occurs and to 0 otherwise.
 * Used error codes: TLS_EIO
 */
merror_t tls_getchar(tls_t *tls, char *c, int *eof);

/*
 * tls_gets()
 *
 * Reads in at most one less than 'size' characters from 'tls' and stores them
 * into the buffer pointed 'line'. Reading stops after an EOF or a newline.  
 * If a newline is read, it is stored into the buffer. A '\0' is stored after 
 * the last character in the buffer.
 * Used error codes: TLS_EIO
 */
merror_t tls_gets(tls_t *tls, char *line, int size);

/*
 * tls_puts()
 *
 * Writes a string using TLS.
 * Used error codes: TLS_EIO
 */
merror_t tls_puts(tls_t *tls, char *s);

/*
 * tls_close()
 *
 * Close a TLS connection and mark it inactive
 */
void tls_close(tls_t *tls);

/*
 * tls_lib_deinit()
 *
 * Deinit underlying TLS library.
 */
void tls_lib_deinit(void);

#endif
