/*
 * Copyright (C) 2005 William J. Poser.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"
#include "compdefs.h"

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <stddef.h>
#include <wchar.h>
#include "unicode.h"
#include "input.h"
#include "limits.h"
#include "utf8error.h"

#define MAXUTF8LEN 6

/*
 * Read a block of UTF-8 text terminated by an extra newline into a buffer,
 * reallocating storage as necessary. The variable BufferSize is updated
 * if the size of the buffer is increased.
 * Returns the block of text.
 * The number of characters put into the buffer, not including
 * the terminating null is returned in the variable status if everything goes well.
 * Otherwise, the error code is placed in status:  BUFOVERFLOW if storage cannot be allocated.
 * On end of input a count of zero characters is returned in status.
 */

UTF8 *
GetNNBlockRAUTF8(FILE *fp,
	 UTF8 *buf,
	 int *status,
	 int *BufferSize,
	 wchar_t t)
{
  int cnt = 0;
  int c;
  int state = 0;
  int eol;
  static int done = 0;

  if(done){
    done = 0;
    *status =	ENDOFINPUT;
    return(buf);
  }

  eol = (int) t;
  while( (c = getc(fp)) != EOF){
    if(cnt == *BufferSize){
      *BufferSize = 2 * *BufferSize;
      buf = (unsigned char *) realloc( (void *) buf, (size_t) (*BufferSize * sizeof(unsigned char)));
      if(buf == NULL){
	*status=BUFOVERFLOW;
	return(buf);
      }
    }
    if(c == eol) {
      if(state==0){
	buf[cnt++]=eol;
	state=1;
      }
      else if(state == 1){
	buf[cnt]='\0';
	*status=cnt;
	return(buf);
      }
    }
    else {
      if(state==1) state=0;
      buf[cnt++]=c;
    }
  } /* End of while */
	
  buf[cnt] = '\0';
  done = 1;
  *status=cnt;
  return(buf);
}

void
ucstrappend(unsigned char *tgt, unsigned char *src,int slen) {
  int i;
  for(i = 0; i <slen; i++){
    *tgt++ = *src++;
  }
  *tgt = '\0';
}

/*
 * Read a block of UTF-8 text terminated by a specified character
 * from a stream into a buffer, checking to make
 * sure that the size of the buffer is not exceeded.
 * Returns the block of text.
 * The number of characters put into the buffer, not including
 * the terminating null is returned in the variable status if everything goes well.
 * Otherwise, the error code is placed in status:  BUFOVERFLOW if storage cannot be allocated.
 * On end of input a count of zero characters is returned in status.
 */

UTF8 *
GetBlockSepCharRAUTF8(FILE *stream, UTF8 *buf, int *status, int *BufferSize, wchar_t t)
{
  UTF32 c;
  int cnt = 0;
  int UCBytes;
  unsigned char *rawptr;
  int infd;
  static int done = 0;
  extern UTF32 Get_UTF32_From_UTF8 (int,int *,unsigned char **);

  infd = fileno(stream);
  if(done){
    done = 0;
    *status=ENDOFINPUT;
    return(buf);
  }

  while ( (c = Get_UTF32_From_UTF8(infd,&UCBytes,&rawptr)) <= UNI_MAX_UTF32){
    if(cnt >= (*BufferSize -MAXUTF8LEN)){
      *BufferSize = 2 * *BufferSize;
      buf = (UTF8 *) realloc( (void *) buf, (size_t) (*BufferSize * sizeof(UTF8)));
      if(buf == NULL){
	fprintf(stderr,"null buf ptr on request for %u bytes\n",*BufferSize);fflush(stderr);
	*status=BUFOVERFLOW;
	return(buf);
      }
    }
    if(c == t){
      *status=cnt;
      return(buf);
    }
    else {
      ucstrappend(buf+cnt,rawptr,UCBytes);
      cnt += UCBytes;
    }
  }
  done = 1;
  *status=cnt;
  return(buf);
}

/*
 * Read a line into buffer from a file of UTF-8 characters, converting
 * en passant to UTF32.
 * Returns the number of characters in the line, not counting the
 * terminating null. Returns:
 *	 
 *     ENDOFINPUT     if there is no more input
 *     BUFOVERFLOW    if the length of the line exceeds the buffer size 
 *     	 
 */

#define DEFAULT 0
#define OVERFLOW 1
#define MSGSIZE 128

int
ugetline(FILE *fp,wchar_t *buf, int size)
{

  wchar_t c;
  int cnt = 0;
  int state = DEFAULT;
  static short GetLineDone = 0;
  char msg[MSGSIZE];
  /* These two are not used here but I don't want to include two versions of Get_UTF32...*/
  int UCBytes;		
  unsigned char *rawptr;

  extern UTF32 Get_UTF32_From_UTF8 (int,int *,unsigned char **);

  if(GetLineDone){
    GetLineDone = 0;
    return(ENDOFINPUT);
  }
   
  while( (c = Get_UTF32_From_UTF8(fileno(fp),&UCBytes,&rawptr)) <= UNI_MAX_UTF32){
    if(cnt == size){
      buf[cnt] = '\0';
      state = OVERFLOW;
    }
    if(c == L'\n'){
      if(state == OVERFLOW) return(BUFOVERFLOW);
      else{
	buf[cnt] = L'\0';
	return(cnt);
      }
    }
    else if(state == DEFAULT) buf[cnt++] = c;
  }
  GetLineDone = 1;
  switch (c){ 
     case NOTENOUGHBYTES:
       fprintf(stderr,"Truncated UTF-8 sequence encountered.\n");
       exit(1);
       break;
     case BADINCODE:
       fprintf(stderr,"Invalid UTF-8 code encountered.\n");
       exit(1);
       break;
     case BADOUTCODE:
       fprintf(stderr,"Encountered invalid Unicode.\n");
       exit(1);
       break;
     case IOERROR:
       snprintf(msg,MSGSIZE-1,"Error reading input.\n");
       perror(msg);
       exit(1);
       break;
     default:			/* Normal EOF */
       break;
  }
  if(state == OVERFLOW) return(BUFOVERFLOW);
  else{
    buf[cnt] = '\0';
    GetLineDone = 1;
    return(cnt);
  }
}


UTF8 *
GetFixedLengthRecord(
	     FILE *fp,	/* Stream from which to read */
	     UTF8 *buf,  /* Address of buffer */
	     int *bread, /* Pointer to number of bytes actually read - return parameter */
	     int *lenptr, /* Pointer to intended length of record, in bytes */
	     wchar_t dummy2 /* Unused argument needed for consistency with other functions  */
) {
  int BytesRead;
  BytesRead = read(fileno(fp),buf,*lenptr);
  if(BytesRead == 0) {
    *bread = ENDOFINPUT;
  }
  else if (BytesRead < *lenptr) {
    *bread = SHORTRECORD;
  } else {
    *bread = BytesRead;
    buf[*lenptr] = '\0';
  }
  return buf;
}

