/* Time-stamp: <2006-05-21 13:26:57 poser> */

/*
 * Copyright (C) 1993-2006 William J. Poser.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

static char rcsid[] = "$Id: misc.c,v 1.10 2003/11/20 02:51:49 poser Exp poser $";

#include "config.h"
#include "compdefs.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif
#ifdef LOCALE_GETTEXT
#include <libintl.h>
#define _(x) gettext(x)
#else
#define _(x) (x)
#endif
#include <wchar.h>
#include <errno.h>
#include "regex.h"
#include "key.h"
#include "unicode.h"
#include "exitcode.h"
#include "retcodes.h"
#include "ex_codes.h"

#ifdef HAVE_LONGDOUBLE
#define DOUBLE (long double)
#else
#define DOUBLE double
#endif

void
DumpExclusions(unsigned char *exct,long exctsize,FILE *fp)
{
  unsigned long i;
  unsigned char flags;
	
  fprintf(fp,_("Characters excluded:\n"));
  if(exctsize > 1) printf("\n");
  for(i = 0L;i < exctsize; i++){
    if(exct[i]){
      flags = exct[i];
      fprintf(fp,"\t0x%04lX",i);
      if(flags & EX_INITIAL) fprintf(fp,_("\tinitial"));
      if(flags & EX_MEDIAL) fprintf(fp,_("\tmedial"));
      if(flags & EX_FINAL) fprintf(fp,_("\tfinal"));
      fprintf(fp,"\n");
    }
  }
}


/*
 * Overwrite a textual key with itself, skipping characters specified
 * in the exclusion table.
 */

int
Exclude(wchar_t *str, int len, wchar_t *exct)
{
  int i;
  int cnt;

  cnt = 0;
  if(!(exct[str[0]] & EX_INITIAL)){
    str[cnt++] = str[0];
  }
  for(i = 1; i < len-1; i++){
    if(exct[str[i]] & EX_MEDIAL) continue;
    else str[cnt++] = str[i];
  }
  if(!(exct[str[len-1]] & EX_FINAL)){
    str[cnt++] = str[len-1];
  }
  str[cnt] = ((wchar_t) 0); /* Null terminate */
  return(cnt);
}

void
KeyCopy(wchar_t *t,wchar_t *s,int len)
{
  register int i;
  for(i = 0;i < len;i++) *t++ = *s++;
}

void
RevKeyCopy(wchar_t *t,wchar_t *s,int len)
{
  register int i;
	
  s += (len-2);
	
  for(i = 0;i < len-1;i++) *t++ = *s--;
  *t= (wchar_t) 0;
}

void
CopyCommandLine(FILE *fp,int ac, char **av)
{
  int i;
	
  for(i=0; i < ac-1; i++){
    fprintf(fp,"%s ",av[i]);
  }
  fprintf(fp,"%s\n",av[ac-1]);
}

/* 
 * This produces a value that is suitable for ordering dates and times.
 * It is NOT accurate for computing differences between dates and times
 * because the conversion of dates to days is not exact. For simplicity's
 * sake it treats years as containing 366 days and months as all containing
 * 31 days.
 */
#define DTITYPE unsigned long
#define SECONDSINDAY (60*60*24)
int
GetISO8601Key(wchar_t *field, DOUBLE *key)
{
  unsigned int year = 0;
  unsigned int month =0;
  unsigned int day = 0;
  unsigned int hour = 0;
  unsigned int minute = 0;
  unsigned int second = 0;

  extern unsigned int u8len (UTF8 *);
  /* This should be defined in wchar.h but for some reason we're not seeing the definition  */
  extern int swscanf (__const wchar_t *__restrict __s,
		      __const wchar_t *__restrict __format, ...);

  DTITYPE Days;
  DTITYPE DaySeconds;
  int f1, f2;

  
  f1=swscanf(field,L"%4u-%2u-%2uT%2u:%2u.%2u", &year,&month,&day,&hour,&minute,&second);
  if(f1 != 6) {
    f2=swscanf(field,L"%4u-%2u-%2uT%2u:%2u", &year,&month,&day,&hour,&minute,&second);
    if(f2 != 5) return(ERROR);
  }

  if(month < 0 || day < 0) return (ERROR);
  if(month > 12) return (ERROR);
  if(day > 31) return (ERROR);
  if(hour < 0 || minute < 0 || second < 0) return (ERROR);
  if(hour > 24) return(ERROR);
  if(minute > 60 || second > 60) return (ERROR);

  Days = (DTITYPE) ( (366 * (DTITYPE) year) + (31 * month) + day);
  DaySeconds = (DTITYPE) (((((DTITYPE) hour * 60) + (DTITYPE) minute) * 60) + (DTITYPE) second);
  *key = (DOUBLE) ( ( (DOUBLE)Days * (DOUBLE)SECONDSINDAY) + (DOUBLE)DaySeconds);
  return(SUCCESS);
}

/* Given a date string, write the equivalent double into key. */

int
GetDateKey(wchar_t *field, DOUBLE *key, struct ymdinfo *ymd)
{
  long int num[3];
  DOUBLE month;
  DOUBLE day;
  wchar_t *sepptr1;
  wchar_t *sepptr2;

  extern int errno;

  sepptr1 = wcschr(field,ymd->sep1);
  sepptr2 = wcsrchr(field,ymd->sep2);
  if( (sepptr1 == NULL) || (sepptr2 == NULL) ) {
    return(ERROR);
  }
  *sepptr1 = L'\0';
  *sepptr2 = L'\0';
  num[0] = wcstol(field,NULL,10);
  if((errno == EINVAL)||(errno == ERANGE)) return (ERROR);
  num[1] = wcstol(sepptr1+1,NULL,10);
  if((errno == EINVAL)||(errno == ERANGE)) return (ERROR);
  num[2] = wcstol(sepptr2+1,NULL,10);
  if((errno == EINVAL)||(errno == ERANGE)) return (ERROR);
  month = (DOUBLE) num[ymd->m];
  day = (DOUBLE) num[ymd->d];
  if(month < 0 || day < 0) return (ERROR);
  if(month > 12) return (ERROR);
  if(day > 31) return (ERROR);
  *key = (DOUBLE) ((366 * (DOUBLE) num[ymd->y]) + (31 * month) + day);
  return(SUCCESS);
}

/* Given a time string, write the equivalent double into key. */

int
GetTimeKey(wchar_t *field, DOUBLE *key)
{
  int hours;
  int minutes;
  int iseconds;
  double seconds;
  UTF8 *cfield;
  int FieldsFound;

  extern UTF8 * ws2u8(wchar_t *);
  cfield = ws2u8(field);
  seconds = 0;
  /* HH:MM:SS(.SSS) */
  FieldsFound = sscanf((char *)cfield,"%d:%d:%lf",&hours,&minutes,&seconds);
  if(FieldsFound == 3) goto validate;

  /* HH:MM.SS */
  FieldsFound = sscanf((char *)cfield,"%d:%d.%d",&hours,&minutes,&iseconds);
  if(FieldsFound == 3) {
    seconds = (double) iseconds;
    goto validate;
  }

  /* HH:MM */
  FieldsFound = sscanf((char *)cfield,"%d:%d",&hours,&minutes);
  if(FieldsFound == 2) {
    seconds = 0.0;
    goto validate;
  }

  free(cfield);  
  return(ERROR); 

validate:
  free(cfield);
  if(hours < 0 || minutes < 0 || seconds < 0) return (ERROR);
  if(hours > 24 || minutes > 59 || seconds >= 60) return (ERROR);
  *key = (DOUBLE) (((((DOUBLE) hours * 60) + (DOUBLE) minutes) * 60) + (DOUBLE) seconds);
  return(SUCCESS);
}

/* Given an angle string, write the equivalent double into key. */

int
GetAngleKey(wchar_t *field, DOUBLE *key)
{
  int degrees;
  int minutes;
  double seconds;
  UTF8 *cfield;
  int FieldsFound;
  short NegativeP = 0;
  DOUBLE Value;

  extern UTF8 * ws2u8(wchar_t *);

  cfield = ws2u8(field);
  FieldsFound = sscanf((char *)cfield,"%4d:%2d:%lf",&degrees,&minutes,&seconds);
  if(FieldsFound!=3) {
    FieldsFound = sscanf((char *)cfield,"%4d %2d %lf",&degrees,&minutes,&seconds);
  }
  free(cfield);
  if(FieldsFound != 3)return(ERROR);
  if(minutes < 0 || seconds < 0) return (ERROR);
  if( (minutes > 59) || (seconds >= 60)) return (ERROR);

  if(degrees < 0) {
    degrees *= (-1);
    NegativeP = 1;
  }
  degrees %= 360;
  if(NegativeP) degrees = 360 -degrees;
  Value = ((((DOUBLE) degrees * 60) + (DOUBLE) minutes) * 60) + seconds;
  *key = Value;
  return(SUCCESS);
}

void
IdentifySelf(FILE *fp)   
{
  extern char progname[];
  extern char version[];
  extern char* compdate;
	
  fprintf(fp,"%s %s\n",progname,version);
  fprintf(fp,_("Compiled %s\n"),compdate);
}

FILE *
OpenFile(char *file,char *mode,char *pgname)
{
  FILE *fp;
  extern FILE *fopen();
   
  if((fp = fopen(file,mode)) != NULL) return fp;
  else{
    switch(*mode){
    case 'r': 
      fprintf(stderr,_("%s: can't open file %s for reading.\n"),pgname,file);
      break;
    case 'w':
      fprintf(stderr,_("%s: can't open file %s for writing.\n"),pgname,file);
      break;
    default:
      fprintf(stderr,_("%s: can't open file %s for appending.\n"),pgname,file);
    }
    return(NULL);
  }
  /* NOTREACHED */
}


/*
 * Rewrite a string, replacing backslash escapes.
 *
 */

#define DEFAULT        0
#define READBACKSLASH  1
#define READONEDIGIT   2
#define READTWODIGITS  3

int
EvalEscapes(wchar_t *s)
{
  int state;
  wchar_t c;
  wchar_t odigit1 = 0L;
  wchar_t odigit2 = 0L;
  wchar_t sum = 0L;
  wchar_t *SStart;
  wchar_t *TStart;
  wchar_t *t;

  if(s == NULL) return(SUCCESS);

  /* Allocate storage for de-escaped copy of string */
	
  t = (wchar_t *) malloc(sizeof(wchar_t) * (wcslen(s)+1));
  if(t == NULL) return(ERROR);

  SStart = s;
  TStart = t;

  /* Work through string */
	
  state = 0;
  while((c = *s++) != L'\0'){
    switch(c){
    case '\\':
      if(state == READBACKSLASH){
	*t++ = L'\\'; /* Literal backslash */
	state = DEFAULT;
      }
      else if(state  == READONEDIGIT){
	*t++ = L'\\';
	*t++ = odigit1;
	state = DEFAULT;
      }
      else if(state  == READTWODIGITS){
	*t++ = L'\\';
	*t++ = odigit1;
	*t++ = odigit2;
	state = DEFAULT;
      }
      else{				/* state == DEFAULT */
	state = READBACKSLASH;
      }
      break;
    case 'n':
      if(state == READBACKSLASH){
	*t++ = L'\n';
	state = DEFAULT;
      }
      else{
	if(state  == READONEDIGIT){
	  *t++ = L'\\';
	  *t++ = odigit1;
	}
	else if(state  == READTWODIGITS){
	  *t++ = L'\\';
	  *t++ = odigit1;
	  *t++ = odigit2;
	}
	*t++ = c;
	state = DEFAULT;
      }
      break;
    case ' ':
      if(state == READBACKSLASH){
	*t++ = L' ';
	state = DEFAULT;
      }
      else{
	if(state  == READONEDIGIT){
	  *t++ = L'\\';
	  *t++ = odigit1;
	}
	else if(state  == READTWODIGITS){
	  *t++ = L'\\';
	  *t++ = odigit1;
	  *t++ = odigit2;
	}
	*t++ = c;
	state = DEFAULT;
      }
      break;
    case 't':
      if(state == READBACKSLASH){
	*t++ = L'\t';
	state = DEFAULT;
      }
      else{
	if(state  == READONEDIGIT){
	  *t++ = L'\\';
	  *t++ = odigit1;
	}
	else if(state  == READTWODIGITS){
	  *t++ = L'\\';
	  *t++ = odigit1;
	  *t++ = odigit2;
	}
	*t++ = c;
	state = DEFAULT;
      }
      break;
    case '0':
    case '1':
    case '2':
    case '3':
      if(state == READBACKSLASH){
	odigit1 = c;
	sum = ( (c - L'0') * 64);
	state = READONEDIGIT;
      }
      else if(state == READONEDIGIT){
	odigit2 = c;
	sum+=( (c - L'0') * 8);
	state = READTWODIGITS;
      }
      else if(state == READTWODIGITS){
	sum+= (c - L'0');
	*t++ = sum;
	state = DEFAULT;
      }
      else *t++ = c;
      break;
    case '4':
    case '5':
    case '6':
    case '7':
      /* These are not possible leading digits */
      if(state == READBACKSLASH){
	*t++ = L'\\';
	*t++ = c;
	state = DEFAULT;
      }
      else if(state == READONEDIGIT){
	odigit2 = c;
	sum+=( (c - L'0') * 8);
	state = READTWODIGITS;
      }
      else if(state == READTWODIGITS){
	sum+= (c - L'0');
	*t++ = sum;
	state = DEFAULT;
      }
      else *t++ = c;
      break;
    default:
      if(state == READBACKSLASH){
	*t++ = L'\\';
      }
      else if(state == READONEDIGIT){
	*t++ = L'\\';
	*t++ = odigit1;
      }
      else if(state == READTWODIGITS){
	*t++ = L'\\';
	*t++ = odigit1;
	*t++ = odigit2;
      }
      *t++ = c;
      state = DEFAULT;
      break;
    }
  }
  
  /* Null terminate copy and overwrite original */
  
  *t = L'\0';
  wcscpy(SStart,TStart);
  free((void *) TStart);
  return(SUCCESS);
}

/*
 * Overwrite a textual key with itself, skipping characters specified
 * in the exclusion table. This version differs from the main Exclude
 * function in operating on chars and in not overwriting its input.
 */

int
ExcludeChar(wchar_t *str, wchar_t **out, int len, wchar_t *exct)
{
  int i;
  int cnt;
  static wchar_t *OutStr;

  cnt = 0;
  if(OutStr != NULL) free((void *) OutStr);
  OutStr = (wchar_t *) malloc ( (size_t) (len+1) * sizeof(wchar_t));
  if (OutStr == NULL) {
    fprintf(stderr,_("ExcludeChar: out of memory\n"));
    exit(OUTOFMEMORY);
  }
  if(!(exct[str[0]] & EX_INITIAL) ){
    OutStr[cnt++] = str[0];
  }
  for(i = 1; i < len-1; i++) {
    if(exct[str[i]] & EX_MEDIAL) continue;
    else OutStr[cnt++] = str[i];
  }
  if(!(exct[str[len-1]] & EX_FINAL)) {
    OutStr[cnt++] = str[len-1];
  }
  OutStr[cnt] = ((wchar_t) 0); /* Null terminate */
  *out = OutStr;
  return(cnt);
}

