/* -*-Asm-*- */
/*
 *  DOCBoot -- A very simple Linux bootloader for DiskOnChip
 *
 *  Author: Dan Brown <dan_brown@ieee.org>
 *
 *  Diskonchip Millennium Plus support by Kalev Lember <kalev@smartlink.ee>
 *
 *  Portions taken from the DOC Grub bootloader by
 *          David Woodhouse <dwmw2@infradead.org>
 *
 *  $Id: doc_bootstub.S,v 1.3 2004/08/02 18:41:29 kalev Exp $
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "doc_bootstub.h"
	
	.file	"doc_bootstub.S"

	.text

	/* Tell GAS to generate 16-bit instructions so that this code works
	   in real mode. */
	.code16

.globl _start; _start:
	/*
	 * _start is loaded by the DiskOnChip IPL.  0x3000 bytes are loaded
	 * at the first empty (all-zeros) 64K region in the range 0x20000 to
	 * 0x90000 (with 1k alignment).  If the 64k region is at 0x20000, then
	 * CS:IP at this point will be 0x2000:0
	 * DS will be the segment of the DOC itself.
	 */

#ifndef DOC_ADDRESS
		/* Store the DiskOnChip segment */
	movw	%ds, %cs:doc_seg
#endif
#ifdef BIOS_EXTENSION
       .byte	0x55, 0xAA	/* BIOS extension signature */
       .byte	0x02		/* BIOS extension size in 512 byte blocks */

	/* BIOS will call far _start + 3 */
#endif
	cld
		/* Test for multiple loads due to aliased ROM addresses.  If
		   the code pointed to by the interrupt vector is identical to
		   our code, just return. */
	xorw	%dx, %dx
	movw	%dx, %ds
	movw	(DOC_BIOS_HOOK * 4), %es
	pushw	%cs
	popw	%ds
	xorw	%di, %di
	xorw	%si, %si
	movw	$code_end, %cx
	repe
	cmpsb
	je	skip_install

	/* What we need to do is:
	   1. Check the current end-of-memory in the BIOS
	   2. Reduce it by the amount of memory we need for our INT 18h/19h 
	   3. Return, and wait for our INT 18h/19h to be called.
	*/

	MSG(spl_string)

		/* Store the setup segment */
	movw	%cs, setup_seg

		/* Find top of memory */
	movw	%dx, %ds
	movw	0x0413, %ax

		/* Steal 1k from the top */
	decw	%ax
	movw	%ax,0x0413

		/* Generate segment address */
	shlw	$6,%ax
	movw	%ax,%es

		/* Set up our shiny new INT 18h/19h handler */
	movw	%ax,(DOC_BIOS_HOOK * 4 + 2)
	movw	$handler,(DOC_BIOS_HOOK * 4)

		/* Copy ourself into the new segment we've just reserved */
	movw	$0x200, %cx
	pushw	%cs
	popw	%ds
	xorw	%si,%si
	xorw	%di,%di
	rep
	movsw

skip_install:
	lret	

/* The actual int 18h/19h handler does three things:
   1) Copy the real-mode kernel from the DOC into the same 64k memory segment
      originally used by the MSYS IPL to store the 0x3000 bytes of SPL (on the
      theory that we already know it's safe to use).
   2) Copy the rest of the kernel (the protected-mode part) from the DOC into
      high memory starting at 0x100000.  BIOS int15h function 87h is used to
      perform the low-to-high copy.
   3) Patch some variables in the real-mode kernel, that a Linux bootloader is
      supposed to set.  Then jump to the real-mode kernel.
 */

handler:
	pushw	%cs
	popw	%ds
	MSG(loading_string)

/*
	mov	$0x88, %ah
	int	$0x15
	call	phword
*/

	cld

	movw	setup_seg, %ax
	movw	%ax, %es

		/* gdt_src is currently set to 0x8000.  Add the setup base
		   address to it, yielding a 32k offset from the start of the
		   setup segment.  The math is simplified by the fact that
		   setup_seg must be 1Kbyte aligned. */
	shrw	$4, %ax
	addw	%ax, gdt_src_mid

	movw	doc_seg, %ds
	movw	$BXREG, %bx
	movw	$SIREG, %si

		/* Enable the DiskOnChip ASIC */
#ifdef MILPLUS
	movb	$(DOC_MODE_NORMAL | DOC_MODE_MDWREN | DOC_MODE_RST_LAT | DOC_MODE_BDECT), BX_Mplus_DOCControl
	movb	$~(DOC_MODE_NORMAL | DOC_MODE_MDWREN | DOC_MODE_RST_LAT | DOC_MODE_BDECT), BX_Mplus_CtrlConfirm

		/* Assert ChipEnable and WriteProtect */
	movb	$(DOC_FLASH_CE | DOC_FLASH_WP), BX_Mplus_FlashSelect
#else
	movb	$DOC_MODE_CLR_ERR + DOC_MODE_MDWREN + DOC_MODE_NORMAL, BX_DOCControl
	movb	$DOC_MODE_CLR_ERR + DOC_MODE_MDWREN + DOC_MODE_NORMAL, BX_DOCControl

		/* Flash command:	Reset */
	movb	$NAND_CMD_RESET, %al
	call	doc_cmd
#endif

	xorw	%di, %di
	movw	$0x20, %dx	/* scan from second eraseblock */
read_setup_sects:
	call	doc_readpage
	decw	%cs:low_sects
	jnz	read_setup_sects

#if 0
		/* Print the kernel version string.  Partially for fun, mostly
		   to show that we've loaded the low part and are about to
		   load the high part. */
	pushw	%ds
	pushw	%es
	popw	%ds
	pushw	%si
	pushw	%bx
	movw	kernel_version, %si
	addw	$0x200, %si
	call	message
	popw	%bx
	popw	%si
	popw	%ds
#endif

	movw	$0x8000, %di	/* Go to 32k from setup base */
read_kernel_sects:
	call	doc_readpage
	movw	%cx, %di	/* re-use the same memory buffer */
	pushw	%es
	pushw	%cs
	popw	%es
	pushw	%si
	movw	$gdt, %si
	movw	$0x200, %cx
	movb	$0x87, %ah
	int	$0x15
	popw	%si
	popw	%es
	addw	$2, %cs:gdt_dst_mid
	decw	%cs:high_sects
	jnz	read_kernel_sects

/* We're done with the DOC.  Set up some things a kernel loader is supposed
   to set, then try to boot! */

	pushw	%cs
	popw	%ds
	MSG(done_string)
	MSG(cmdline)
	MSG(boot_string)

	pushw	%es
	popw	%ds	/* Now write parameters into setup segment */

		/* The commandline is at offset 512 from CS.  Compute the
		   physical address and store into cmd_line_ptr.  Simplified
		   by the fact that CS must be 1k-aligned. */
	pushw	%cs
	popw	%ax
	shrw	$4, %ax
	incw	%ax
	incw	%ax
	movw	%ax, cmd_line_ptr + 1

	movb	$0x81, loadflags
	movw	$0xffff, %ax
	movb	%al, type_of_loader
	movw	%ax, heap_end_ptr	/* Actually should be 0xfdff, but it
					   should never get that far */

	movw	%ds, %bx
	movw	%bx, %fs	/* Not strictly needed */
	movw	%bx, %gs	/* Not strictly needed */
	movw	%bx, %ss
	movw	%ax, %sp	/* We know we have 64k of space */

	incw	%ax		/* Now ax=0 */
	addw	$0x20, %bx
	pushw	%bx
	pushw	%ax
	cli
	lret		/* GO! */

/***************************************************************************/

		/* doc_readpage: Read a page from DOC to es:di
		   Then read the OOB.  If the magic tag (0xdbb1) isn't found
		   at OOB locations 6 and 7, try the next page, etc.  This
		   allows us not only to scan until we find the kernel image,
		   but also to skip over holes in the image (which are
		   presumably the result of bad blocks). */
nextpage:
	popw	%di
doc_readpage:
#ifdef MILPLUS
		/* Flash command:	Reset */
	movb	$NAND_CMD_RESET, %al
	call	doc_cmd
#endif
		/* Flash command:	Read0 */
	movb	$NAND_CMD_READ0, %al
	call	doc_cmd
#ifdef MILPLUS
	movb	$0, BX_Mplus_FlashAddress
	movb	%dl, BX_Mplus_FlashAddress
	movb	%dh, BX_Mplus_FlashAddress
		/* Terminate the write pipeline */
	movb	$0, BX_Mplus_WritePipeTerm
	movb	$0, BX_Mplus_WritePipeTerm
		/* Deassert ALE */
	movb	$0, BX_Mplus_FlashControl
	call	doc_wait
	testb	$0, BX_Mplus_ReadPipeInit
	testb	$0, BX_Mplus_ReadPipeInit
#else
	movb	$CDSN_CTRL_WP + CDSN_CTRL_ALE + CDSN_CTRL_CE, BX_CDSNControl
	movb	$0, SI_CDSN_IO
	movb	%dl, SI_CDSN_IO
	movb	%dh, SI_CDSN_IO
		/* This test is used in the MSYS IPL.  It appears to check
		   an undocumented bit in the ConfigurationInput register,
		   presumably to determine if the NAND chip is large enough to
		   require a 3-byte page number. */
	testb	$0x20, BX_ConfigurationInput
	jz	notbigchip
	movb	$0, SI_CDSN_IO
notbigchip:
	call	doc_wait
	testb	$0, BX_ReadPipeInit
#endif
	movw	$0x208, %cx	/* Read page + 8 bytes OOB */
	pushw	%di
rploop:
	lodsb
	stosb
	decw	%si
	loop	rploop
	incw	%dx
	cmpw	$0xb1db, %es:-2(%di)
	jne	nextpage
	subw	$8, %di
	popw	%cx	/* di = old di + 512. Return original di in cx */
	ret

                /* doc_cmd:      Send a command to the flash chip */
doc_cmd:
#ifdef MILPLUS
		/* Send the command to the flash */
	movb	%al, BX_Mplus_FlashCmd
		/* Terminate the write pipeline */
	movb	$0, BX_Mplus_WritePipeTerm
	movb	$0, BX_Mplus_WritePipeTerm
	/* fall through... */

                /* doc_wait:     Wait for the DiskOnChip to be ready */
doc_wait:
	/* FIXME: probably only three NOP's are needed */
	testb	$0xc0, BX_Mplus_NOP
	testb	$0xc0, BX_Mplus_NOP
	testb	$0xc0, BX_Mplus_NOP
	testb	$0xc0, BX_Mplus_NOP
doc_waitloop:    
	movb	BX_Mplus_FlashControl, %al
	andw	$0xc0, %ax
	cmpw	$0xc0, %ax
	jne	doc_waitloop
	ret
#else
                /* Enable CLE line to flash */
        movb    $CDSN_CTRL_WP + CDSN_CTRL_CLE + CDSN_CTRL_CE, BX_CDSNControl
                /* Write the actual command */
        movb	%al, SI_CDSN_IO
	/* fall through... */

                /* doc_wait:     Wait for the DiskOnChip to be ready */
doc_wait:
        movb	$0, BX_WritePipeTerm
	movb	$CDSN_CTRL_WP + CDSN_CTRL_CE, BX_CDSNControl
	testb	$0x80, BX_NOP
	testb	$0x80, BX_NOP
	testb	$0x80, BX_NOP
	testb	$0x80, BX_NOP
doc_waitloop:    
	testb	$0x80, BX_CDSNControl
        jz      doc_waitloop
	testb	$0x80, BX_NOP
	testb	$0x80, BX_NOP
        ret
#endif
                

/*
 * message: write the string pointed to by %si
 *
 *   WARNING: trashes %si, %ax, and %bx
 */

	/*
	 * Use BIOS "int 10H Function 0Eh" to write character in teletype mode
	 *	%ah = 0xe	%al = character
	 *	%bh = page	%bl = foreground color (graphics modes)
	 */
1:
	movw	$0x0001, %bx
	movb	$0xe, %ah
	int	$0x10		/* display a byte */

message:
	lodsb
	cmpb	$0, %al
	jne	1b	/* if not end of string, jmp to display */
	ret

/***************************************************************************/
#if 0

phword:
	pushw   %ax
	xchgb   %al,%ah
	call    phbyte
	movb    %ah,%al
	call    phbyte
	popw    %ax
	ret

phbyte:
	pushw   %ax
	movb    %al, %ah
	shrb    $4,%al
	call    phnibble
	movb    %ah, %al
	call    phnibble
	popw    %ax
	ret

phnibble:
	pushw   %ax
	andb    $0xf,%al
	addb    $48,%al
	cmpb    $57,%al
	jna     ph1
	add     $7,%al
	ph1:    mov     $0xe,%ah
	int     $0x10
	popw    %ax
	ret

#endif
/***************************************************************************/
code_end:

#ifdef DOC_ADDRESS
doc_seg:	.word DOC_ADDRESS
#else
doc_seg:	.word 0
#endif

/* gdt structure for high loading using int15/87 */
gdt:
		.skip   0x10
gdt_src_limit:  .word   0xffff
gdt_src_lo:     .byte   0
gdt_src_mid:    .byte   0x80
gdt_src_hi:     .byte   0
gdt_src_perm:   .byte   0x93
        .word   0
gdt_dst_limit:  .word   0xffff
gdt_dst_lo:     .byte   0
gdt_dst_mid:    .byte   0
gdt_dst_hi:     .byte   0x10
gdt_dst_perm:   .byte   0x93
                .word   0
                .skip   0x10

spl_string:	.string "Installing DiskOnChip bootloader.\n\r"
loading_string:	.string "Loading kernel... "
done_string:	.string "done.\n\rCommandline: "
boot_string:	.string "\n\rBooting!\n\r"

	.org CHECKSUM_LOCATION
checksum:
setup_seg:	.word 0
low_sects:	.word 0
high_sects:	.word 0

	.org 512
cmdline:

/******************************************************/
/* This section adapted from arch/i386/boot/boot.S    */
/******************************************************/
/* Although the data defined here will be placed in the object file by the
   compiler, only the first 512 bytes (above) are copied into the SPL.  This
   is just here as a convenient way of encoding the offsets for use in the
   bootloader.
 */
kernel_start:	.word 0
        # This is the setup header, and it must start at %cs:2 (old 0x9020:2)
header_sig:	.ascii  "...."          # header signature
header_version: .word   0               # header version number (>= 0x0105)
realmode_swtch: .word   0, 0            # default_switch, SETUPSEG
start_sys_seg:  .word   0
kernel_version: .word   0		# pointing to kernel version string
type_of_loader: .byte   0               # = 0, old one (LILO, Loadlin,
loadflags:	.byte   0		# If set, the kernel is loaded high
setup_move_size: .word  0               # size to move, when setup is not
code32_start:	.long   0               # 0x100000 = default for big kernel
ramdisk_image:  .long   0               # address of loaded ramdisk image
ramdisk_size:   .long   0               # its size in bytes
bootsect_kludge: .word  0, 0
heap_end_ptr:   .word   0
pad1:           .word   0
cmd_line_ptr:   .long   0               # If nonzero, a 32-bit pointer
ramdisk_max:    .long   0               # The highest safe address
/******************************************************/
