#include <curses.h>
#include <errno.h>
#include <signal.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <sys/types.h>
#include <regex.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include "version.h"
#include "error.h"

extern int path_max;

int myrand(int max)
{
	return (int)(((double)max * (double)rand()) / (double)RAND_MAX);
}

void * mymalloc(int size, char *what)
{
	void *dummy = malloc(size);
	if (!dummy)
		error_exit("failed to allocate %d bytes for %s\n", size, what);

	return dummy;
}

void * myrealloc(void *oldp, int newsize, char *what)
{
	void *dummy = realloc(oldp, newsize);
	if (!dummy)
		error_exit("failed to reallocate to %d bytes for %s\n", newsize, what);

	return dummy;
}

char * mystrdup(char *in)
{
	char *dummy = strdup(in);
	if (!dummy)
		error_exit("failed to copy string '%s': out of memory?\n", in);

	return dummy;
}

int resize(void **pnt, int n, int *len, int size)
{
        if (n == *len)
        {
                *len = (*len) ? (*len) * 2 : 4096;
                *pnt = myrealloc(*pnt, (*len) * size, "resize()");
        }
	else if (n > *len || n<0 || *len<0)
		error_exit("resize: fatal memory corruption problem: n > len || n<0 || len<0!\n");

	return 0;
}

ssize_t READ(int fd, char *whereto, size_t len)
{
	ssize_t cnt=0;

	while(len>0)
	{
		ssize_t rc;

		rc = read(fd, whereto, len);

		if (rc == -1)
		{
			if (errno != EINTR)
				error_exit("Problem reading from file\n");
		}
		else if (rc == 0)
		{
			break;
		}
		else
		{
			whereto += rc;
			len -= rc;
			cnt += rc;
		}
	}

	return cnt;
}

ssize_t WRITE(int fd, char *whereto, size_t len)
{
	ssize_t cnt=0;

	while(len>0)
	{
		ssize_t rc;

		rc = write(fd, whereto, len);

		if (rc == -1)
		{
			if (errno != EINTR)
				error_exit("Problem writing to filedescriptor\n");
		}
		else if (rc == 0)
		{
			break;
		}
		else
		{
			whereto += rc;
			len -= rc;
			cnt += rc;
		}
	}

	return cnt;
}

char * read_line_fd(int fd)
{
	char *str = mymalloc(80, "read_line_fd buffer");
	int n_in=0, size=80;

	for(;;)
	{
		char c;
		int rc;

		/* read one character */
		if ((rc = READ(fd, &c, 1)) != 1)
		{
			if (n_in == 0)
			{
				free(str);
				str = NULL;
			}
			return str;
		}

		/* resize input-buffer */
		resize((void **)&str, n_in, &size, sizeof(char));

		/* EOF or \n == end of line */
		if (c == 10 || c == EOF)
			break;

		/* ignore CR */
		if (c == 13)
			continue;

		/* add to string */
		str[n_in++] = c;
	}

	/* terminate string */
	resize((void **)&str, n_in, &size, sizeof(char));
	str[n_in] = 0x00;

	return str;
}

/** get_load
 * - in:      nothing
 * - returns: char *
 * this function returns a string containing the current load of the system
 * multitail is running on. format: %1.2f %1.2f %1.2f
 * the string is malloc()ed so the caller should free() things
 */
char *get_load(void)
{
#if defined(__FreeBSD__) || defined(linux) || defined(__OpenBSD__) || defined(__NetBSD__) || defined(__APPLE__) || defined(sun)
	double loadavg[3];
	char *str = (char *)mymalloc(20, "loadavg string");

#if defined(__GLIBC__) && ( __GLIBC__ < 2 || (__GLIBC__ == 2 && __GLIBC_MINOR__ < 2))
	/* Older glibc doesn't have getloadavg() - use sysinfo() */
	/* thanks to Ville Herva for this code! */
	double scale = 1 << SI_LOAD_SHIFT;
	struct sysinfo si;

	if (sysinfo(&si) == -1)
	{
		/* let's exit: if these kind of system-
		 * calls start to fail, something must be
		 * really wrong
		 */
		error_exit("sysinfo() failed");
	}

	loadavg[0] = (double)si.loads[0] / scale;
	loadavg[1] = (double)si.loads[1] / scale;
	loadavg[2] = (double)si.loads[2] / scale;
#else
	if (getloadavg(loadavg, 3) == -1)
	{
		/* see comment on sysinfo() */
		error_exit("getloadavg() failed");
	}
#endif
	sprintf(str, "%1.2f %1.2f %1.2f", loadavg[0], loadavg[1], loadavg[2]);

	return str;
#else
	return mystrdup("");
#endif
}

int get_vmsize(pid_t pid)
{
	FILE *fh;
	char *path = mymalloc(path_max, "path");
	int vmsize = -1;

#if defined(linux)
	sprintf(path, "/proc/%d/stat", pid);

	fh = fopen(path, "r");
	if (fh)
	{
		char *dummystr = mymalloc(path_max, "temp string");
		char dummychar;
		int dummy;

		fscanf(fh, "%d %s %c %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d", &dummy, dummystr, &dummychar, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &vmsize);

		fclose(fh);

		free(dummystr);
	}
#endif

	free(path);

	return vmsize;
}

/** stop_process
 * - in:      int pid  pid of process
 * - returns: nothing
 * this function sends a TERM-signal to the given process, sleeps for 1009 microseconds
 * and then sends a KILL-signal to the given process if it still exists. the TERM signal
 * is send so the process gets the possibility to gracefully exit. if it doesn't do that
 * in 100 microseconds, it is terminated
 */
void stop_process(int pid)
{
	if (kill(pid, SIGTERM) == -1)
	{
		if (errno != ESRCH)
			error_exit("problem stopping child-process (%d)!\n", pid);
	}

	usleep(1000);

	/* process still exists? */
	if (kill(pid, SIGTERM) == 0)
	{
		/* sleep for a millisecond... */
		usleep(1000);

		/* ...and then really terminate the process */
		if (kill(pid, SIGKILL) == -1)
		{
			if (errno != ESRCH)
				error_exit("problem killing child-process (%d)!\n", pid);
		}
	}
	else if (errno != ESRCH)
		error_exit("problem stopping child-process (%d)!\n", pid);
}

/** delete_array
 * - in:      char **list array of strings to free
 *            int n       number of strings in this array
 * - returns: nothing
 * this function frees an array of strings: all strings are freed and
 * also the pointer-list itself is freed
 */
void delete_array(char **list, int n)
{
	int loop;

	for(loop=n-1; loop>=0; loop--)
		free(list[loop]);

	free(list);
}

int find_char_offset(char *str, char what)
{
	int loop, len = strlen(str);

	for(loop=0; loop<len; loop++)
	{
		if (str[loop] == what)
			return loop;
	}

	return -1;
}

int file_info(char *filename, off_t *file_size, time_t *ts)
{
	struct stat buf;

	if (stat(filename, &buf) == -1)
	{
		if (errno != ENOENT)
			error_exit("error while doing stat() on file %s\n", filename);

		return -1;
	}

	*file_size = buf.st_size;

	*ts = buf.st_mtime;

	return 0;
}

char * convert_regexp_error(int error, const regex_t *preg)
{
	/* errors are specified not to be longer then 256 characters */
	char reerror[256];
	char *multitail_string = "MultiTail warning: regular expression failed, reason: %s\n";
	char *error_out;

	if (error == 0 || error == REG_NOMATCH)
		return NULL;

	/* convert string */
	regerror(error, preg, reerror, sizeof(reerror));

	error_out = (char *)mymalloc(strlen(reerror) + strlen(multitail_string) + 1, "regexp error string");

	sprintf(error_out, multitail_string, reerror);

	return error_out;
}

/* I'm sorry for this: */
char check_date(void)
{
	time_t now;
	struct tm *ptm;

	time(&now);
	ptm = localtime(&now);

	if (ptm -> tm_mon == 3 && ptm -> tm_mday == 2) /* April the 2nd? */
	{
		return 1;
	}

	return 0;
}

#define M_KB (1024)
#define M_MB (M_KB * 1024)
#define M_GB (M_MB * 1024)

char * amount_to_str(int amount)
{
	char *out = mymalloc(3 + 2 + 1, "converted amount string");	/* ...XB\0 */

	if (amount > M_GB)	/* GB */
		sprintf(out, "%dGB", (amount + M_GB - 1) / M_GB);
	else if (amount > M_MB)	/* MB */
		sprintf(out, "%dMB", (amount + M_MB - 1) / M_MB);
	else if (amount > M_KB)	/* KB */
		sprintf(out, "%dKB", (amount + M_KB - 1) / M_KB);
	else
		sprintf(out, "%d", amount);

	return out;
}

char * replace_string(char *in, int pos_start, int pos_end, char *with_what)
{
	int str_len_in = strlen(in);
	int str_len_ww = strlen(with_what);
	int n_remove = (pos_end - pos_start) + 1; /* +1 => including pos_end! */
	int new_len = str_len_in + str_len_ww - n_remove;
	char *out = mymalloc(new_len + 1, "replace_string new string"); /* +1 => 0x00 */

	memcpy(out, in, pos_start);
	memcpy(&out[pos_start], with_what, str_len_ww);
	memcpy(&out[pos_start + str_len_ww], &in[pos_end + 1], str_len_in - (pos_end + 1));
	out[new_len] = 0x00;

	return out;
}
