#define _LARGEFILE64_SOURCE	/* required for GLIBC to enable stat64 and friends */
#include <ctype.h>
#include <sys/types.h>
#include <regex.h>
#include <string.h>
#include <stdlib.h>
#include <pwd.h>
#include <errno.h>

#include "mt.h"
#include "error.h"
#include "my_pty.h"
#include "utils.h"
#include "term.h"
#include "help.h"
#include "mem.h"
#include "ui.h"
#include "misc.h"
#include "globals.h"


int find_string(int window, char *find, int offset)
{
	int loop, index = -1, rc;
	regex_t regex;

	/* compile the searchstring (which can be a regular expression) */
	if ((rc = regcomp(&regex, find, REG_EXTENDED)))
	{
		regexp_error_popup(rc, &regex);

		return -1;	/* failed -> not found */
	}

	for(loop=offset; loop<lb[window].curpos; loop++)
	{
		if (regexec(&regex, lb[window].Blines[loop], 0, NULL, 0) == 0)
		{
			index = loop;
			break;
		}
	}

	regfree(&regex);

	return index;
}

void scrollback_savefile(int window)
{
	char *file = NULL;
	NEWWIN *mywin = create_popup(8, 40);

	win_header(mywin, "Save buffer to file");

	mvwprintw(mywin -> win, 4, 2, "Select file");
	file = edit_string(mywin, 5, 2, 40, find_path_max(), 0, NULL, HELP_SCROLLBACK_SAVEFILE_ENTER_FILENAME, -1);
	if (file)
	{
		FILE *fh = fopen(file, "w");
		if (fh)
		{
			int loop;

			for(loop=0; loop<lb[window].curpos; loop++)
			{
				if (lb[window].Blines[loop])
				{
					char display;
					char *error;
					int dummy = -1;
					regmatch_t *pmatch = NULL;

					/* check filter */
					(void)check_filter(lb[window].pi[loop], lb[window].Blines[loop], &pmatch, &error, &dummy, 0, &display);
					if (error)
					{
						fprintf(fh, "%s\n", error);
						myfree(error);
					}
					if (display)
					{
						fprintf(fh, "%s\n", lb[window].Blines[loop]?lb[window].Blines[loop]:"");
					}

					if (pmatch) myfree(pmatch);
				}
			}

			fclose(fh);
		}
		else
		{
			error_popup("Save scrollback buffer", -1, "Cannot write to file, reason: %s", strerror(errno));
		}
	}

	delete_popup(mywin);
}

int get_lines_needed(char *string, int terminal_width)
{
	if (string)
		return (strlen(string) + terminal_width - 1) / terminal_width;
	else
		return 1;
}

void scrollback_displayline(NEWWIN *win, int window, int buffer_offset, int terminal_offset, int offset_in_line, mybool_t force_to_winwidth)
{
	proginfo *cur_line_meta = lb[window].pi[buffer_offset];
	char *cur_line = lb[window].Blines[buffer_offset];
	regmatch_t *pmatch = NULL;
	int matching_regex = -1;
	char *error = NULL;

	wmove(win -> win, terminal_offset, 0);

	if (cur_line)
	{
		if (IS_MARKERLINE(cur_line_meta))
		{
			color_print(win, cur_line_meta, cur_line, NULL, -1, MY_FALSE);
		}
		else /* just a buffered line */
		{
			char display;
			(void)check_filter(cur_line_meta, cur_line, &pmatch, &error, &matching_regex, 0, &display);
			if (error)
			{
				color_print(win, cur_line_meta, error, NULL, -1, MY_FALSE);
				myfree(error);
			}
			if (display)
			{
				if (offset_in_line)
				{
					char *buffer = mymalloc(win -> width + 1, "scrollback_displayline: scrollback output buffer");
					int line_len = strlen(cur_line);
					int new_size = 0;

					if (offset_in_line < line_len)
					{
						new_size = min(win -> width, line_len - offset_in_line);

						memcpy(buffer, &cur_line[offset_in_line], new_size);
					}

					buffer[new_size] = 0x00;

					mvwprintw(win -> win, terminal_offset, 0, buffer);

					myfree(buffer);
				}
				else
				{
					color_print(win, cur_line_meta, cur_line, pmatch, matching_regex, force_to_winwidth);
				}
			}
		}
	}
	else /* an empty line */
	{
		/* do nothing */
	}

	myfree(pmatch);
}

void scrollback(void)
{
	int window = 0;

	if (nfd > 1)
	{
		window = select_window(HELP_SCROLLBACK_SELECT_WINDOW, NULL);
	}

	if (window != -1)
	{
		if (lb[window].bufferwhat == 0)
			error_popup("Scrollback", HELP_SCROLLBACK_NO_MARK, "Cannot scrollback: buffering is disabled.");
	}

	if (window != -1 && lb[window].bufferwhat != 0)
	{
		char *find = NULL;
		int find_ok = 0;
		NEWWIN *mywin1, *mywin2;
		int nlines = max_y - 6, ncols = max_x - 6;
		int offset = max(0, lb[window].curpos - nlines);
		char redraw = 2;
		int line_offset = 0;

		mywin1 = create_popup(max_y - 4, max_x - 4);

		mywin2 = create_popup(nlines, ncols);
		scrollok(mywin2 -> win, FALSE); /* supposed to always return OK, according to the manpage */

		for(;;)
		{
			int c, uc;

			if (redraw == 2)
			{
				int index = 0;
				int lines_used = 0;

				inverse_on(mywin1);
				mvwprintw(mywin1 -> win, nlines + 1, 1, "%02d] %s - %d buffered lines", window, pi[window].filename, lb[window].curpos);
				inverse_off(mywin1);

				if (!no_linewrap) inverse_on(mywin1);
				mvwprintw(mywin1 -> win, nlines + 1, ncols - 8, "LINEWRAP");
				if (!no_linewrap) inverse_off(mywin1);
				
				werase(mywin2 -> win);

				for(;(offset + index) < lb[window].curpos;)
				{
					int prev_lines_used = lines_used;

					if (no_linewrap)
						lines_used++;
					else
						lines_used += get_lines_needed(lb[window].Blines[offset + index], ncols);
					if (lines_used > nlines)
						break;

					scrollback_displayline(mywin2, window, offset + index, prev_lines_used, line_offset, no_linewrap);
					index++;
				}

				redraw = 1;
			}

			if (redraw == 1)
			{
				mydoupdate();

				redraw = 0;
			}

			c = wait_for_keypress(HELP_SCROLLBACK_HELP, 0, NULL, 1);
			uc = toupper(c);

			if (uc == 'Q' || uc == 'X' || c == 7)
			{
				break;
			}
			else if (uc == 'Y')
			{
				no_linewrap = !no_linewrap;
				redraw = 2;
				line_offset = 0;
			}
			else if (c == KEY_LEFT && no_linewrap)
			{
				if (line_offset > 0)
					line_offset--;

				redraw = 2;
			}
			else if (c == KEY_RIGHT &&  no_linewrap)
			{
				line_offset++;
				redraw = 2;
			}
			else if (c == KEY_UP && offset > 0)
			{
				int n_to_move = 0;

				wmove(mywin2 -> win, 0, 0);

				do
				{
					offset--;

					if (no_linewrap)
					{
						if (lb[window].Blines[offset] != NULL && strlen(lb[window].Blines[offset]) > 0)
							n_to_move = 1;
					}
					else
						n_to_move = get_lines_needed(lb[window].Blines[offset], ncols);
				} while(n_to_move == 0 && offset > 0);

				winsdelln(mywin2 -> win, n_to_move);

				scrollback_displayline(mywin2, window, offset, 0, line_offset, no_linewrap);

				redraw = 1;
			}
			else if (c == KEY_DOWN && offset < (lb[window].curpos - 1))
			{
				int n_to_move = 0;

				while(n_to_move == 0 && offset < (lb[window].curpos - 1))
				{
					offset++;
					n_to_move = get_lines_needed(lb[window].Blines[offset], ncols);
				}

				redraw = 2;
			}
			else if (c == KEY_NPAGE && offset < (lb[window].curpos - 1))
			{
				int dummy = 0;
				while(offset < (lb[window].curpos - 1))
				{
					offset++;
					dummy += get_lines_needed(lb[window].Blines[offset], ncols);
					if ((dummy > nlines && c == KEY_NPAGE) ||
							(dummy > (nlines/4) && c == KEY_DOWN))
					{
						offset--;
						break;
					}
				}
				redraw = 2;
			}
			else if (c == KEY_PPAGE && offset > 0)
			{
				int dummy = 0;
				while(offset > 0)
				{
					offset--;
					dummy += get_lines_needed(lb[window].Blines[offset], ncols);
					if (dummy > nlines)
					{
						offset++;
						break;
					}
				}
				redraw = 2;
			}
			else if (c == KEY_HOME && offset > 0)
			{
				offset = 0;
			}
			else if (c == KEY_END && offset < (lb[window].curpos - 1))
			{
				offset = lb[window].curpos - 1;
			}
			else if (uc == 'F' || c == '/')
			{
				char *dummy;
				int new_f_index;
				NEWWIN *mywin = create_popup(5, 40);

				win_header(mywin, "Find");

				dummy = edit_string(mywin, 3, 2, 40, 80, 0, find, HELP_SCROLLBACK_EDIT_SEARCH_STRING, -1);
				myfree(find);
				find = dummy;

				find_ok = 0;

				if (find)
				{
					redraw = 2; /* force redraw */

					new_f_index = find_string(window, find, 0);
					if (new_f_index == -1)
					{
						wrong_key();
					}
					else
					{
						find_ok = 1;
						offset = new_f_index;
					}
				}

				delete_popup(mywin);
			}
			else if (uc == 'N')
			{
				if (find != NULL && find_ok != 0)
				{
					int new_f_index = find_string(window, find, offset + 1);
					if (new_f_index == -1)
						wrong_key();
					else
					{
						redraw = 2; /* force redraw */
						offset = new_f_index;
					}
				}
				else
					wrong_key();
			}
			else if (uc == 'S')
			{
				scrollback_savefile(window);
				redraw = 2;	/* force redraw */
			}
			else if (uc == 'H')
			{
				show_help(HELP_SCROLLBACK_HELP);
			}
			else if (uc == 'C')
			{
				toggle_colors();
				redraw = 2;	/* force redraw */
			}
			else if (uc == 'I')
			{
				info();
			}
			else if (uc == 'T')
			{
				statistics_menu();
			}
			else
			{
				wrong_key();
			}
		}

		delete_popup(mywin2);
		delete_popup(mywin1);

		myfree(find);
	}
}
