#define _LARGEFILE64_SOURCE	/* required for GLIBC to enable stat64 and friends */
#include <errno.h>
#include <signal.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <sys/types.h>
#include <regex.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include <glob.h>
#ifdef _POSIX_PRIORITY_SCHEDULING
#include <sched.h>
#endif
#include <pwd.h>
#include <sys/wait.h>
#include <fcntl.h>
#if defined(sun) || defined(__sun)
#include <sys/loadavg.h>
#endif
#include <dirent.h>

#include "version.h"
#include "error.h"
#include "mem.h"
#include "mt.h"
#include "term.h"
#include "utils.h"


int find_path_max(void)
{
#ifdef PATH_MAX
	int path_max = PATH_MAX;
#else
	int path_max = pathconf("/", _PC_PATH_MAX);
	if (path_max <= 0)
	{
		if (errno) error_exit("find_path_max: pathconf(_PC_PATH_MAX) failed\n");

		path_max = 4096;
	}
	else
		path_max++; /* since its relative to root */
#endif
	if (path_max > 4096)
		path_max = 4096;

	return path_max;
}

int myrand(int max)
{
	return (int)(((double)max * (double)rand()) / (double)RAND_MAX);
}

ssize_t WRITE(int fd, char *whereto, size_t len)
{
	ssize_t cnt=0;

	while(len>0)
	{
		ssize_t rc;

		rc = write(fd, whereto, len);

		if (rc == -1)
		{
			if (errno != EINTR && errno != EAGAIN)
				error_exit("WRITE: Problem writing to filedescriptor\n");
		}
		else if (rc == 0)
		{
			break;
		}
		else
		{
			whereto += rc;
			len -= rc;
			cnt += rc;
		}
	}

	return cnt;
}

/** get_load
 * - in:      nothing
 * - returns: char *
 * this function returns a string containing the current load of the system
 * multitail is running on. format: %1.2f %1.2f %1.2f
 * the string is malloc()ed so the caller should free() things
 */
char *get_load(void)
{
#if !defined(__UCLIBC__) && (defined(__FreeBSD__) || defined(linux) || defined(__OpenBSD__) || defined(__NetBSD__) || defined(__APPLE__) || defined(__GNU__) || defined(__sun) || defined(sun))
	double loadavg[3];
	char *str = (char *)mymalloc(LOADAVG_STR_LEN, "get_load: loadavg string");

#if defined(__GLIBC__) && ( __GLIBC__ < 2 || (__GLIBC__ == 2 && __GLIBC_MINOR__ < 2))
	/* Older glibc doesn't have getloadavg() - use sysinfo() */
	/* thanks to Ville Herva for this code! */
	double scale = 1 << SI_LOAD_SHIFT;
	struct sysinfo si;

	if (sysinfo(&si) == -1)
	{
		/* let's exit: if these kind of system-
		 * calls start to fail, something must be
		 * really wrong
		 */
		error_exit("get_load: sysinfo() failed\n");
	}

	loadavg[0] = (double)si.loads[0] / scale;
	loadavg[1] = (double)si.loads[1] / scale;
	loadavg[2] = (double)si.loads[2] / scale;
#else
	if (getloadavg(loadavg, 3) == -1)
	{
		/* see comment on sysinfo() */
		error_exit("get_load: getloadavg() failed\n");
	}
#endif
	snprintf(str, LOADAVG_STR_LEN, "%1.2f %1.2f %1.2f", loadavg[0], loadavg[1], loadavg[2]);

	return str;
#else
	return mystrdup("", "get_load:");
#endif
}

int get_vmsize(pid_t pid)
{
	int vmsize = -1;
#if defined(linux)
	FILE *fh;
	int path_max = find_path_max();
	char *path = mymalloc(path_max, "path");

	snprintf(path, path_max, "/proc/%d/stat", pid);

	fh = fopen(path, "r");
	if (fh)
	{
		char *dummystr = mymalloc(path_max, "get_vmsize: temp string");
		char dummychar;
		int dummy;

		if (fscanf(fh, "%d %s %c %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d", &dummy, dummystr, &dummychar, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &dummy, &vmsize) != 23)
			vmsize = -1;

		fclose(fh);

		myfree(dummystr);
	}

	myfree(path);
#endif

	return vmsize;
}

/** stop_process
 * - in:      int pid  pid of process
 * - returns: nothing
 * this function sends a TERM-signal to the given process, sleeps for 1009 microseconds
 * and then sends a KILL-signal to the given process if it still exists. the TERM signal
 * is send so the process gets the possibility to gracefully exit. if it doesn't do that
 * in 100 microseconds, it is terminated
 */
void stop_process(int pid)
{
	if (pid < 2)
		return;

	if (killpg(pid, SIGTERM) == -1)
	{
		if (errno != ESRCH)
			error_exit("stop_process: problem stopping child-process (%d)!\n", pid);
	}

	usleep(1000);

	/* process still exists? */
	if (killpg(pid, SIGTERM) == 0)
	{
		/* sleep for a millisecond... */
		usleep(1000);

		/* ...and then really terminate the process */
		if (killpg(pid, SIGKILL) == -1)
		{
			if (errno != ESRCH)
				error_exit("stop_process: problem killing child-process (%d)!\n", pid);
		}
	}
	else if (errno != ESRCH)
		error_exit("stop_process: problem stopping child-process (%d)!\n", pid);

	/* wait for the last remainder of the exited process to go away,
	 * otherwhise we'll find zombies on our way
	 */
	if (waitpid(pid, NULL, WNOHANG | WUNTRACED) == -1)
	{
		if (errno != ECHILD)
			error_exit("stop_process: waitpid failed\n");
	}
}

/** delete_array
 * - in:      char **list array of strings to free
 *            int n       number of strings in this array
 * - returns: nothing
 * this function frees an array of strings: all strings are freed and
 * also the pointer-list itself is freed
 */
void delete_array(char **list, int n)
{
	int loop;

	for(loop=n-1; loop>=0; loop--)
		myfree(list[loop]);

	myfree(list);
}

int find_char_offset(char *str, char what)
{
	char *pnt = strchr(str, what);
	if (!pnt)
		return -1;

	return (int)(pnt - str);
}

int file_info(char *filename, off64_t *file_size, time_field_t tft, time_t *ts, mode_t *mode)
{
	struct stat64 buf;

	if (stat64(filename, &buf) == -1)
	{
		if (errno != ENOENT)
			error_exit("file_info: error while doing stat() on file %s\n", filename);

		return -1;
	}

	if (file_size) *file_size = buf.st_size;

	if (ts)
	{
		if (tft == TT_ATIME)
			*ts = buf.st_atime;
		else if (tft == TT_MTIME)
			*ts = buf.st_mtime;
		else if (tft == TT_CTIME)
			*ts = buf.st_ctime;
		else if (tft != 0)
			error_exit("file_info: invalid timefield (%d)\n", (int)tft);
	}

	if (mode) *mode = buf.st_mode;

	return 0;
}

int file_exist(char *filename)
{
	struct stat64 buf;

	return stat64(filename, &buf);
}

char * convert_regexp_error(int error, const regex_t *preg)
{
	/* errors are specified not to be longer then 256 characters */
	char *multitail_string = "MultiTail warning: regular expression failed, reason: ";
	int len = strlen(multitail_string);
	char *error_out = NULL;
	const int max_err_len = 256;

	if (error != 0 && error != REG_NOMATCH)
	{
		error_out = (char *)mymalloc(max_err_len + len + 1, "convert_regexp_error: regexp error string");

		memcpy(error_out, multitail_string, len);

		/* convert regexp error */
		regerror(error, preg, &error_out[len], max_err_len);
	}

	return error_out;
}

/* I'm sorry for this: */
char check_date(void)
{
	time_t now = time(NULL);
	struct tm *ptm = localtime(&now);

	if (ptm -> tm_mon == (4 - 1) && ptm -> tm_mday == 2) /* April the 2nd? */
	{
		return 1;
	}

	return 0;
}

char * amount_to_str(long long int amount)
{
	char *out = mymalloc(AMOUNT_STR_LEN, "amount_to_str: converted amount string");	/* ...XB\0 */

	if (amount >= M_GB)	/* GB */
		snprintf(out, AMOUNT_STR_LEN, "%dGB", (int)((amount + M_GB - 1) / M_GB));
	else if (amount >= M_MB)	/* MB */
		snprintf(out, AMOUNT_STR_LEN, "%dMB", (int)((amount + M_MB - 1) / M_MB));
	else if (amount >= M_KB)	/* KB */
		snprintf(out, AMOUNT_STR_LEN, "%dKB", (int)((amount + M_KB - 1) / M_KB));
	else
		snprintf(out, AMOUNT_STR_LEN, "%d", (int)(amount));

	return out;
}

char * replace_string(char *in, int pos_start, int pos_end, char *with_what)
{
	int str_len_in = strlen(in);
	int str_len_ww = strlen(with_what);
	int n_remove = (pos_end - pos_start) + 1; /* +1 => including pos_end! */
	int new_len = str_len_in + str_len_ww - n_remove;
	char *out = mymalloc(new_len + 1, "replace_string: new string"); /* +1 => 0x00 */

	memcpy(out, in, pos_start);
	memcpy(&out[pos_start], with_what, str_len_ww);
	memcpy(&out[pos_start + str_len_ww], &in[pos_end + 1], str_len_in - (pos_end + 1));
	out[new_len] = 0x00;

	return out;
}

struct passwd *getuserinfo(void)
{
	struct passwd *pp = getpwuid(geteuid());
	if (!pp)
		error_exit("getuserinfo: failed to get passwdstructure for effective user id %d\n", geteuid());

	return pp;
}

/* these are there because AIX/IRIX can return EINTR for those */
int myopen(char *path, int mode)
{
	int fd;

	for(;;)
	{
		fd = open64(path, mode);
		if (fd == -1)
		{
			if (errno == EINTR || errno == EAGAIN) /* for AIX */
				continue;

			return fd;
		}

		break;
	}

	return fd;
}

int myclose(int fd)
{
	for(;;)
	{
		if (close(fd) == -1)
		{
			if (errno == EINTR || errno == EAGAIN) /* for AIX */
				continue;

			return -1;
		}

		return 0;
	}
}

char * shorten_filename(char *in, int max_len)
{
	static char buffer[4096];
	int len = strlen(in);
	int cutlen, dummy;

	if (len <= max_len)
		return in;

	cutlen = (max_len - 3) / 2;

	memcpy(buffer, in, cutlen);
	memcpy(&buffer[cutlen], "...", 3);

	dummy = max_len - (cutlen + 3);
	memcpy(&buffer[cutlen + 3], &in[len - dummy], dummy + 1);

	return buffer;
}

double get_ts(void)
{
	struct timeval ts;
	struct timezone tz;

	if (gettimeofday(&ts, &tz) == -1)
		error_exit("get_ts: gettimeofday failed");

	return (((double)ts.tv_sec) + ((double)ts.tv_usec)/1000000.0);
}

int match_files(char *search_for, char **path, char ***found, char **isdir)
{
	DIR *dir;
	struct dirent *entry;
	char *cur_dir = mymalloc(find_path_max() + 1, "match_files: current file");
	char *fname;
	char **list = NULL;
	int nfound = 0;
	size_t fname_size;
	int path_len;
	int s1, s2;
	char *slash = strrchr(search_for, '/');
	if (slash)
	{
		fname = mystrdup(slash + 1, "match_files: filename");
		*(slash + 1) = 0x00;
		*path = mystrdup(search_for, "match_files: current path");
	}
	else
	{
		*path = mystrdup("./", "match_files: current path");
		fname = mystrdup(search_for, "match_files: filename");
	}
	fname_size = strlen(fname);
	path_len = strlen(*path);

	dir = opendir(*path);
	if (!dir)
	{
		free(cur_dir);
		return 0;
	}

	memcpy(cur_dir, *path, path_len + 1);

	while((entry = readdir(dir)) != NULL)
	{
		if ((fname_size == 0 || strncmp(entry -> d_name, fname, fname_size) == 0) && strcmp(entry -> d_name, ".") != 0 &&
				strcmp(entry -> d_name, "..") != 0)
		{
			struct stat finfo;

			/* get filename */
			list = (char **)myrealloc(list, (nfound + 1) * sizeof(char *), "match_files: directory list");
			list[nfound] = mystrdup(entry -> d_name, "match_files dir entry");

			/* check if the file is a directory */
			*isdir = (char *)myrealloc(*isdir, (nfound + 1) * sizeof(char), "ismatch_files: -dir list");
			strncpy(&cur_dir[path_len], entry -> d_name, max(0,find_path_max() - path_len));
			if (stat(cur_dir, &finfo) == -1)
			{
				if (errno != ENOENT)	/* file did not disappear? then something is very wrong */
					error_exit("match_files: stat error on %s\n", cur_dir);
			}
			(*isdir)[nfound] = S_ISDIR(finfo.st_mode)?1:0;

			nfound++;
		}
	}

	if (closedir(dir) == -1)
		error_exit("match_files: closedir failed\n");

	/*	qsort( (void *)list, (size_t)nfound, sizeof(char *), compare_filenames); */
	for(s1=0; s1<(nfound - 1); s1++)
	{
		for(s2=s1+1; s2<nfound; s2++)
		{
			if (strcasecmp(list[s2], list[s1]) < 0)
			{
				char *fdummy = list[s1], ddummy = (*isdir)[s1];

				list[s1] = list[s2];
				(*isdir)[s1] = (*isdir)[s2];
				list[s2] = fdummy;
				(*isdir)[s2] = ddummy;
			}
		}
	}

	*found = list;

	myfree(fname);
	myfree(cur_dir);

	return nfound;
}

void setup_for_childproc(int fd, char close_fd_0, char *term)
{
	char *dummy = (char *)mymalloc(strlen(term) + 6, "setup_for_childproc: TERM env var");

	if (close_fd_0) if (-1 == myclose(0)) error_exit("setup_for_childproc: close failed\n");
	if (-1 == myclose(1)) error_exit("setup_for_childproc: close failed\n");
	if (-1 == myclose(2)) error_exit("setup_for_childproc: close failed\n");
	if (close_fd_0) if (-1 == dup(fd)) error_exit("setup_for_childproc: dup failed\n");
	if (-1 == dup(fd)) error_exit("setup_for_childproc: dup failed\n");
	if (-1 == dup(fd)) error_exit("setup_for_childproc: dup failed\n");

	/*
	 * not doing this: it also clears the 'PATH'
	 * I could make first do a getenv for PATH and then put it
	 * back after the clearenv, but what would be the point of
	 * doing the clearenv in the first place?
	 *
	 *	if (clearenv() == -1)
	 *	{
	 *		fprintf(stderr, "WARNING: could not clear environment variables: %s (%d)\n", strerror(errno), errno);
	 *		exit(1);
	 *	}
	 */

	/* set terminal */
	sprintf(dummy, "TERM=%s", term);
	if (putenv(dummy) == -1)
		fprintf(stderr, "setup_for_childproc: Could not set TERM environment-variable (%s): %s (%d)\n", dummy, strerror(errno), errno);

	(void)umask(007);
}

int open_null(void)
{
	int fd = myopen("/dev/null", O_RDWR);
	if (fd == -1)
		error_exit("open_null: Failed to open /dev/null\n");

	return fd;
}

void free_re(re *cur_re)
{
	if (cur_re)
	{	
		myfree(cur_re -> regex_str);
		if (cur_re -> use_regex)
			regfree(&cur_re -> regex);

		myfree(cur_re -> cmd);
	}
}

char * find_most_recent_file(char *filespec, char *cur_file)
{
	glob_t files;
	char *selected_file = NULL;
	unsigned int loop;
	time_t prev_ts = (time_t)0;

	LOG("find file for %s\n", filespec);

	/* get timestamp of previous file */
	if (cur_file)
	{
		if (file_info(cur_file, NULL, TT_MTIME, &prev_ts, NULL) == -1)
		{
			LOG("could not get fileinfo for %s\n", cur_file);
			prev_ts = (time_t)0;
		}
	}

	/* get list of files that match */
#if defined(__APPLE__) || defined(__CYGWIN__)
	if (glob(filespec, GLOB_ERR | GLOB_NOSORT, NULL, &files) != 0)
#else
		if (glob(filespec, GLOB_ERR | GLOB_NOSORT | GLOB_NOESCAPE, NULL, &files) != 0)
#endif
		{
			LOG("glob failed %d\n", errno);
			return NULL;
		}

	LOG("number of files found: %d\n", files.gl_pathc);

	/* see if any of them is more recent than the current one */
	for(loop=0; loop<files.gl_pathc; loop++)
	{
		time_t new_ts;

		/* current file? then skip it */
		if (cur_file != NULL && strcmp(cur_file, files.gl_pathv[loop]) == 0)
			continue;

		/* get the modificationtime of this found file */
		if (file_info(files.gl_pathv[loop], NULL, TT_MTIME, &new_ts, NULL) == -1)
		{
			LOG("loop: failed fetching info on file %s: %d\n", files.gl_pathv[loop], errno);
			new_ts = (time_t)0;
		}

		/* more recent? */
		if (new_ts > prev_ts)
		{
			selected_file = files.gl_pathv[loop];
			prev_ts = new_ts;
		}
	}

	/* found a file? then remember the filename */
	if (selected_file != NULL)
	{
		selected_file = mystrdup(selected_file, "find_most_recent_file: new file filename");
	}

	globfree(&files);

	return selected_file;
}

char zerotomin(char c)
{
	if (c == 0)
		return 'n';
	if (c == 1)
		return 'y';

	return c;
}

char *find_next_par(char *start)
{
	char *dummy = strchr(start, ':');
	if (dummy)
	{
		*dummy = 0x00;
		dummy++;
	}

	return dummy;
}

int mydup(int old_fd)
{
	int new_fd = -1;

	for(;;)
	{
		new_fd = dup(old_fd);

		if (new_fd == -1)
		{
			if (errno == EINTR)
				continue;

			error_exit("mydup: dup() failed\n");
		}

		break;
	}

	return new_fd;
}

char * term_t_to_string(term_t term)
{
	switch(term)
	{
		case TERM_ANSI:
			return "ansi";

		case TERM_XTERM:
			return "xterm";
	}

	return "dumb";
}

void get_now_ts(char *format_str, char *dest, int dest_size)
{
	time_t now = time(NULL);
	struct tm *ptm = localtime(&now);
	if (!ptm) error_exit("get_now_ts: localtime() failed\n");

	(void)strftime(dest, dest_size, format_str, ptm);
}
