!
!  Flat     Average of a scaled set of flat-fields.
!  Copyright (C) 1997 - 2017  Filip Hroch, Masaryk University, Brno, CZ
!
!  This file is part of Munipack.
!
!  Munipack is free software: you can redistribute it and/or modify
!  it under the terms of the GNU General Public License as published by
!  the Free Software Foundation, either version 3 of the License, or
!  (at your option) any later version.
!
!  Munipack is distributed in the hope that it will be useful,
!  but WITHOUT ANY WARRANTY; without even the implied warranty of
!  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!  GNU General Public License for more details.
!
!  You should have received a copy of the GNU General Public License
!  along with Munipack.  If not, see <http://www.gnu.org/licenses/>.
!
!
!  TODO:
!   * add running statistical tests of Normal distribution of residuals (**)
!

Program aFlat

  use LibList
  use robustmean
  use weightedmean
  use robustline
  use robratio
  use medians
  use fitsio
  use xfitsio
  use iso_fortran_env

  implicit none

  ! debuging
  logical, parameter :: debug = .false.

  logical :: verbose = .false.

  integer, parameter :: dbl = selected_real_kind(15)

  character(len=*), parameter :: afid = 'FLAT'

  ! Default Output image name:
  character(len=FLEN_FILENAME) :: nameflat='flat.fits', backup

  ! No. of image dimensions
  integer, parameter :: DIM = 2

  ! Default output mean level:
  real :: level = -1

  ! saturation
  real :: saturate_set = -1

  integer :: i,j,eq,nobs, istat, naxis,n,l, iter
  integer, dimension(DIM) :: naxes
  integer :: bitpix = -32
  integer :: maxiter = 3

  ! image
  real :: avg, sig, t, savg, dt, g, d, egain, sgain, fmean, time, saturate, mindt, maxd
  real :: darketime = -1
  real :: etime = -1
  real :: xdark = -1.0
  real :: gain = 1
  logical :: gain_set = .false.
  logical :: gain_estimate = .false.
  real, allocatable, dimension(:,:) :: flat,dflat,res,des,bias,dark,ebias,edark
  real, allocatable, dimension(:) :: fbuf, dbuf, egains
  real, dimension(:,:), pointer :: ccd, err
  logical, allocatable, dimension(:,:) :: mask
  Character(len=4*FLEN_FILENAME) :: sb,key,val,record,buf,biasname='',darkname=''
  Character(len=FLEN_CARD) :: dateobs,f,dateobs1,filter1,imagetyp1,imagetyp,precession
  character(len=FLEN_KEYWORD) :: FILTER=FITS_KEY_FILTER,DATE_OBS=FITS_KEY_DATEOBS, &
       EXPTIME=FITS_KEY_EXPTIME, KEY_IMAGETYP=FITS_KEY_IMAGETYP, KEY_GAIN=FITS_KEY_GAIN,&
       KEY_SATURATE=FITS_KEY_SATURATE
  type(Imagetype), pointer :: list,curr
  type(xFITS) :: fits, biasfits, darkfits
  logical :: reliable, terminate
  real(dbl) :: gzero,ggain,dzero,dgain
  real(dbl), dimension(:), allocatable :: fmeans, sig2s

  buf = ''
  filter1 = ''
  imagetyp1 = ''

  call xfits_init(biasfits)
  call xfits_init(darkfits)

  call InitList(list)
  curr => list

  do
     read(*,'(a)',end=20) record

     eq = index(record,'=')
     if( eq == 0 ) stop 'Improper input.'
     key = record(:eq-1)
     val = record(eq+1:)

     if( key == 'OUTPUT' ) then
        read(val,*) nameflat, backup
     endif

     if( key == 'BITPIX' ) then
        read(val,*) bitpix
     endif

     if( key == 'LEVEL' ) then
        read(val,*) level
     endif

     if( key == 'SATURATE' ) then
        read(val,*) saturate_set
     endif

     if( key == 'VERBOSE' ) then
        read(val,*) verbose
     endif

     if( key == 'PRECISION' ) then
        read(val,*) precession
        if( precession == 'LOW' ) then
           maxiter = 2
        else if( precession == 'ZERO' ) then
           maxiter = 1
        else if( precession == 'EXTRA' ) then
           maxiter = 6
        else
           maxiter = 3
        end if
     endif

     if( key == 'FITS_KEY_FILTER' ) then
        read(val,*) FILTER
     endif

     if( key == 'FITS_KEY_DATEOBS' ) then
        read(val,*) DATE_OBS
     endif

     if( key == 'FITS_KEY_EXPTIME' ) then
        read(val,*) EXPTIME
     endif

     if( key == 'FITS_KEY_IMAGETYP' ) then
        read(val,*) KEY_IMAGETYP
     endif

     if( key == 'FITS_KEY_GAIN' ) then
        read(val,*) KEY_GAIN
     endif

     if( key == 'FITS_KEY_SATURATE' ) then
        read(val,*) KEY_SATURATE
     endif

     if( key == 'GAIN_ESTIMATE' ) then
        read(val,*) gain_estimate
     endif

     if( key == 'BIAS' ) then

        read(val,*) biasname
        if( verbose ) write(error_unit,*) "BIAS=",trim(biasname)
        call xfits_read(biasname,biasfits)
        if( .not. biasfits%status ) stop 'Failed to load the bias frame.'

     end if

     if( key == 'XDARK' ) then
        read(val,*) xdark
     end if

     if( key == 'DARK' ) then

        read(val,*) darkname
        if( verbose ) write(error_unit,*) "DARK=",trim(darkname)
        call xfits_read(darkname,darkfits)
        if( .not. darkfits%status ) stop 'Failed to load the dark frame.'

        istat = 0
        call xfits_kye(darkfits,EXPTIME,darketime,istat)
        if( istat /= 0 ) then
           istat = 0
           darketime = -1
           if( verbose ) &
                write(error_unit,*) "An exposure time for dark frame unknown."
        end if

     end if

     if( key == 'GAIN' ) then
        read(val,*) gain
        gain_set = .true.
     endif

     if( key == 'FILE' ) then

        if( verbose .and. getno(curr) == 0 ) &
             write(error_unit,*) "Filename, exptime[s], gain, saturate[ct], mean, std.err.:"

        read(val,*) sb
        if( verbose ) write(error_unit,'(a)',advance="no") trim(sb)//":"

        call xfits_init(fits)
        call xfits_read(sb,fits)
        if( .not. fits%status ) then
           call xfits_deallocate(fits)
           goto 665
        end if

        istat = 0
        call xfits_kys(fits,DATE_OBS,dateobs,istat)
        if( istat /= 0 ) then
           istat = 0
           dateobs = ''
        end if

        call xfits_kys(fits,FILTER,f,istat)
        if( istat /= 0 ) then
           istat = 0
           f = ''
        end if
        if( f == '' .and. verbose ) &
             write(error_unit,'(a)',advance="no") "Warning: empty filter .. "

        call xfits_kys(fits,KEY_IMAGETYP,imagetyp,istat)
        if( istat /= 0 ) then
           istat = 0
           imagetyp = ''
        end if

        call xfits_kye(fits,EXPTIME,etime,istat)
        if( istat /= 0 ) then
           istat = 0
           etime = -1
        end if

        if( .not. gain_set ) then
           call xfits_kye(fits,KEY_GAIN,gain,istat)
           if( istat /= 0 ) then
              write(error_unit,*) &
                   "Warning: Gain keyword not found in FITS header of `", &
                   trim(sb),"' (default is 1)."
              istat = 0
              gain = 1
           end if
        end if

        ! first image => initialisation
        if( getno(curr) == 0 )then

           naxis = fits%naxis
           naxes = fits%naxes

           dateobs1 = dateobs
           filter1 = f
           imagetyp1 = imagetyp

           if( biasfits%status ) then
              if( .not. all(biasfits%naxes == naxes) ) then
                 write(error_unit,*) 'bias:',biasfits%naxes(1),'x',biasfits%naxes(2), &
                      '     current frame:',naxes(1),'x',naxes(2)
                 stop "Bias has incompatible dimensions."
              end if
           end if

           if( darkfits%status ) then
              if( .not. all(darkfits%naxes == naxes) ) then
                 write(error_unit,*) 'dark:',darkfits%naxes(1),'x',darkfits%naxes(2), &
                      '     current frame:',naxes(1),'x',naxes(2)
                 stop "Dark has incompatible dimensions."
              end if
           end if

        else

           if( .not. all(naxes == fits%naxes) ) then
              write(error_unit,*) &
                   "Dimensions of images are mutually incompatible. Skipping."
              goto 665
           endif

           if( imagetyp1 /= imagetyp ) then
              write(error_unit,*) &
                   "Image-types does not corresponds. Skipping."
              goto 665
           endif

           if( f /= filter1 ) write(error_unit,'(a)',advance="no") &
                "Warning: incompatible filters .. current `",trim(f),"' (?)"

        endif

        ! dark frame multiplicator
        if( xdark > 0 )then
           time = xdark
        else if( etime > 0 .and. darketime > 0 )then
           time = etime / darketime
        else
           time = 1
        end if

        ! saturation
        if( saturate_set > 0 ) then
           saturate = saturate_set
        else
           saturate = -1
           call xfits_kye(fits,KEY_SATURATE,saturate,istat)
           if( istat /= 0 ) then
              istat = 0
              if( fits%bitpix > 0 ) &
                   saturate = 2.0**fits%bitpix
           end if
           if( saturate < 0 ) &
                saturate = maxval(fits%image) + epsilon(saturate)
        end if

        allocate(ccd(naxes(1),naxes(2)),err(naxes(1),naxes(2)), &
             mask(naxes(1),naxes(2)),bias(naxes(1),naxes(2)),dark(naxes(1),naxes(2)),&
             ebias(naxes(1),naxes(2)),edark(naxes(1),naxes(2)))

        if( biasfits%status ) then
           bias = biasfits%image
           ebias = biasfits%stderr
        else
           bias = 0
           ebias = 0
        end if
        if( darkfits%status ) then
           dark = darkfits%image
           edark = darkfits%stderr
        else
           dark = 0
           edark = 0
        end if

        ! Pre-correct the input flat.
        ! Standard deviation is set with assumption of Poisson distribution
        ! of data. It requires large light fluxes, around half of full range.
        mask =  0 < fits%image .and. fits%image < saturate
        where( mask )
           ccd = gain*(fits%image - (bias + time*dark))
        end where
        saturate = gain * saturate
        mask =  0 < ccd .and. ccd < saturate
        where( mask )
           err = sqrt(ccd + gain**2*(ebias**2 + time**2*edark**2))
        end where
        ! Important. Pixels out of the mask are undefined...

        ! the average is computed from less of one million elements
        n = (naxes(1)*naxes(2)) / 1000000
        if( n > 0 ) then
           l = 0
           do i = 1,naxes(1)
              do j = 1,naxes(2)
                 l = l + 1
                 if( mod(l,n) /= 0 ) mask(i,j) = .false.
              end do
           end do
        end if

        ! determine mean level of this frame
        call rwmean(pack(ccd,mask),pack(err,mask),avg,savg,reliable=reliable)

        if( verbose ) &
             write(error_unit,'(2x,1pg0.3,2x,0pf0.2,2x,1pg0.2,2x,1pg0.5,3x,1pg0.3,l2)') &
             etime,gain,saturate,avg,savg,reliable

        if( avg <= epsilon(avg) ) then
           write(error_unit,*) 'This frame has zero or negative mean level. Skipping.'
           deallocate(ccd,err)
        else
           ! add image to the list
           Call AddItem (curr,image=ccd,noise=err,filename=sb,mean=avg,stderr=savg, &
                dateobs=dateobs,filter=f,satur=saturate)
        end if

        deallocate(mask,bias,dark,ebias,edark)
665     continue
        call xfits_deallocate(fits)

     end if

  enddo

20 continue
  nobs = getno(curr)

  if( nobs == 0 ) stop 'No input image(s).'

  if( verbose ) then
     write(error_unit,*)
     write(error_unit,*) 'Number of input images:',nobs
     write(error_unit,*) 'Dimension:',naxes(1),'x',naxes(2)
     write(error_unit,*) 'Filter: ',trim(filter1)
  end if

  ! compute averadged flat-field
  allocate(flat(naxes(1),naxes(2)),dflat(naxes(1),naxes(2)),&
       res(naxes(1),naxes(2)),des(naxes(1),naxes(2)),mask(naxes(1),naxes(2)), &
       fbuf(nobs),dbuf(nobs),egains(nobs),fmeans(nobs),sig2s(nobs))

  terminate = .false.
  do iter = 1,5

     if( verbose ) write(error_unit,'(a,i0,a)') &
          'Calculating a flat-field frame (iteration #',iter-1,') ...'

     ! mean flat
     do j = 1,naxes(2)
        do i = 1,naxes(1)

           curr => list
           n = 0
           do
              curr => GetNext(curr)
              if( .not. associated(curr) ) exit
              call GetItem(curr,image=ccd,noise=err,mean=avg,satur=saturate)
              if( 0 < ccd(i,j) .and. ccd(i,j) < saturate ) then
                 n = n + 1
                 fbuf(n) = ccd(i,j) / avg
                 dbuf(n) = err(i,j) / avg
              end if
           enddo
           if( n > 0 ) then
              call rwmean(fbuf(1:n),dbuf(1:n),flat(i,j),dflat(i,j),reliable=reliable)

!!$              if( .not. reliable .and. .false.  ) then
!!$!                 if( all( abs(fbuf(1:n) - 1) < 0.1 ) ) &
!!$                 if( dflat(i,j) < 0 ) &
!!$                      write(*,*) i,j,fbuf(1:n),dbuf(1:n),flat(i,j),dflat(i,j),n
!!$                 call rmean(fbuf(1:n),flat(i,j),dflat(i,j),reliable=reliable)
!!$                 if( .not. reliable .and. debug ) then
!!$                    write(*,*) 'rmean:',i,j,flat(i,j),dflat(i,j),fbuf(1:n),n
!!$                 end if
!!$              end if

!           else if( n == 1 ) then
!              flat(i,j) = fbuf(1)
!              dflat(i,j) = dbuf(1) ! or 1?
           else
              flat(i,j) = 1
              dflat(i,j) = 1
           end if

        enddo
     enddo

     ! Improving precision of approximation. Thanks to very good initial estimate
     ! (and considering slow computations), we are plan to proceed a few -- only one
     ! -- iteration(s). The convergence of iterations is fast, the limit
     ! of std. deviation of mean is reached immediately after one cycle.
     ! The number of iteration is controled by terminate variable which
     ! tests whatever maximum of residual is small than expected error or not.

     if( iter == maxiter .or. terminate ) exit

     if( verbose ) then
        write(error_unit,'(a)') 'Scaling individual frames by the flat ...'
        write(error_unit,'(a)',advance="no") &
             "Filename,      mean ratio, std.err.,    correction, reliable"
        if( gain_estimate ) then
           write(error_unit,'(a)') ",  std.dev.,  gain:"
        else
           write(error_unit,'(a)') ":"
        end if
     end if

     ! mindt is minimum of standard errors, while maxd is the maximal residual
     mindt = huge(mindt)
     maxd = 0

     ! Update mean of every frame
     curr => list
     do ! over frames
        curr => GetNext(curr)
        if( .not. associated(curr) ) exit
        call GetItem(curr,filename=sb,image=ccd,noise=err,mean=avg,stderr=savg,&
             satur=saturate)

        mask = 0 < ccd .and. ccd < saturate .and. flat > 0 .and. dflat > 0
        call rcal(pack(ccd,mask),pack(err,mask),pack(flat,mask),pack(dflat,mask),t,dt,&
             xreliable=reliable,xverb=.false.)

        if( reliable ) then
           d = t - avg
           avg = t
           if( dt < mindt ) mindt = dt
           if( abs(d) > maxd ) maxd = abs(d)
        end if

        if( verbose ) then

           write(error_unit,'(a,2x,1pg13.7,1pg11.2,2x,1pg10.2,l3)',advance="no") &
                trim(sb)//": ", avg,dt,d,reliable
           !Xi2

           ! gain estimate
           if( gain_estimate ) then
              call rmean(pack(ccd/flat, mask),fmean,savg,sig)
              g = fmean / sig**2
              write(error_unit,'(1x,f11.3,2x,f6.3)') sig,g
!              write(error_unit,'(1x,f11.3,2x,g11.3)') sig,fmean
              egains(curr%i) = g
              fmeans(curr%i) = fmean
              sig2s(curr%i) = sig**2
           else
              write(error_unit,*)
           end if

           if( debug ) then
              ! (**)
              ! Diagnostics. The second column of the files
              ! are residuals intended for Normality testing.

              where( mask )
                 des = sqrt(err**2 + t**2*dflat**2)
                 res = (ccd - t*flat) / des
              end where

              open(1,file='/tmp/'//trim(sb)//'.dat')
              write(1,'(2a)') '# ',sb
              do i = 1,size(ccd,1),2
                 do j = 1,size(ccd,2),2
                    if( mask(i,j) .and. abs(res(i,j)) < 5 ) then
                       write(1,*) ccd(i,j)-t,res(i,j), flat(i,j)
                    end if
                 end do
              end do
              close(1)
           end if
        end if

        call SetItem(curr,mean=avg)

     enddo ! over frames

     terminate = maxd < mindt / sqrt(real(nobs))

  end do ! iter

  ! final mean over the whole area
  call rmean(pack(flat,.true.),avg,savg,sig)

  ! scale to required level
  if( level > 0 ) then
     flat = level*flat
     dflat = level*dflat
     avg = level
     savg = level*savg
     sig = level*sig
  end if

  if( gain_estimate ) then
     call rmean(egains,egain,sgain,t)
     call rline(sig2s,fmeans,gzero,ggain,dzero,dgain)
!     egain = real(ggain)
!     sgain = real(dgain)
!     write(*,*) a,b,da,db
  end if

  if( verbose ) then
     write(error_unit,'(2a)') ' Output image: ',trim(nameflat)
     write(error_unit,'(a,3x,1pg0.7)') ' Final mean:',avg
     write(error_unit,'(a,3x,1pg0.2)') &
          ' Expected photometry standard error per pixel:',median(pack(dflat,.true.))
     if( gain_estimate ) then
        write(error_unit,'(a,2(2x,f0.3))') ' Estimated gain, std.dev:',egain,sgain
        if( abs(gain - 1) > epsilon(gain) ) &
             write(error_unit,'(a,f0.1,a)') '   Warning! Gain pre-set on: ',gain, &
             ' (Consider re-run with -gain 1).'
     end if
  end if

  ! Output image
  istat = 0
  call fitsbackup(nameflat,backup,istat)
  call ftinit(26,nameflat,1,istat)
  call ftphps(26,bitpix,naxis,naxes,istat)
  call ftpkye(26,'MEAN',avg,6,'Mean level',istat)
  call ftpkye(26,KEY_GAIN,gain,6,'[e-/ADU] gain',istat)
  if( gain_estimate ) then
     call ftpkye(26,'GAIN_AVG',egain,6,'[e-/ADU] estimated gain',istat)
     call ftpkye(26,'GAIN_STD',sgain,2,'[e-/ADU] std.dev of estimated gain',istat)
  end if
  if( filter1 /= '' ) &
       call ftpkys(26,FILTER,filter1,'filter of the first on input',istat)
  if( imagetyp1 /= '' ) &
     call ftpkys(26,KEY_IMAGETYP,imagetyp,'image type',istat)
  call ftpkys(26,DATE_OBS,dateobs1,'UTC of the first on input',istat)
  if( bitpix > 0 ) then
     call ftpkye(26,'BSCALE',1.0,10,'',istat)
     call ftpkye(26,'BZERO',2.0**(bitpix-1),10,'',istat)
  endif

  if( nobs > 0 ) then
     write(buf,'(a,i0,a)') 'Result of robust average and scaling of ',nobs,' exposure(s).'
     call ftpcom(26,buf,istat)
     call ftpcom(26,'File name, time of start, average, std.err. for each image used:',istat)
     curr => list
     do
        curr => GetNext(curr)
        if( .not. associated(curr) ) exit
        call GetItem(curr,filename=sb,dateobs=dateobs,mean=avg,stderr=savg)
        write(f,'(1pg0.3,1x,1pg0.3)') avg,savg
        call ftpcom(26,"'"//trim(sb)//"' '"//trim(dateobs)//"' "//trim(f),istat)
     enddo
     call ftpcom(26,'All data are derived from '//trim(DATE_OBS)//' and '//trim(FILTER)//' keywords.',istat)
  endif

  if( darkname /= '' ) then
     write(buf,'(f0.5)') time
     call ftphis(26,afid//" dark: '"//trim(darkname)//"' *"//trim(buf),istat)
  end if

  if( biasname /= '' ) then
     call ftphis(26,afid//" bias: '"//trim(biasname)//"'",istat)
  end if

  call ftukys(26,FITS_KEY_CREATOR,FITS_VALUE_CREATOR,FITS_COM_CREATOR,istat)
  call ftpcom(26,MUNIPACK_VERSION,istat)

  ! flat
  call ftp2de(26,1,naxes(1),naxes(1),naxes(2),flat,istat)
  if( istat == NUMERICAL_OVERFLOW ) then
     write(error_unit,*) 'Warning: Numerical overflow occurred while writing flat.'
     istat = 0
  end if

  ! standard error of mean
  call ftiimg(26,bitpix,naxis,naxes,istat)
  call ftukys(26,'EXTNAME',EXT_STDERR,'',istat)
  call ftpcom(26,'The estimation of standard error of mean of pixels of flat-field.',istat)
  call ftp2de(26,1,naxes(1),naxes(1),naxes(2),dflat,istat)
  if( istat == NUMERICAL_OVERFLOW ) then
     write(error_unit,*) 'Warning: Numerical overflow occurred while writing std.err.'
     istat = 0
  end if

  call ftclos(26,istat)
  call ftrprt('STDERR',istat)

  if( allocated(flat) ) deallocate(flat,dflat,res,des,mask,fbuf,dbuf,egains,fmeans,sig2s)
  call DestroyList(list)
  call xfits_deallocate(biasfits)
  call xfits_deallocate(darkfits)

  if( istat == 0 ) then
     stop 0
  else
     stop 'Failed on FITS I/O error.'
  end if

end Program AFlat
