/*
 *  Copyright (c) by Shuu Yamaguchi <shuu@dotaster.com>
 *
 *  $Id: init.c,v 1.12 2004/09/12 08:03:10 shuu Exp shuu $
 *
 *  Can be freely distributed and used under the terms of the GNU GPL.
 */
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<sys/types.h>
#include	<unistd.h>
#include	<wait.h>

#include	"murasaki.h"
#include	"murasaki_ieee1394.h"

#ifdef	DEBUG
#define	PROC_FILE	"./devices"
#else
#define	PROC_FILE	"/proc/bus/ieee1394/devices"
#endif

#define	STR_VENDOR	"Vendor/Model ID:"
#define	STR_SPEC	"Software Specifier ID:"
#define	STR_VER		"Software Version:"

#define	MATCH(A,B)	strncmp((A),(B),sizeof((A))-1) == 0

#define	ST_NEWLINE	1
#define	ST_DUMB		2

/* global */
static int msg_level = MU_MSG_STD;

static char *
get_vendor_model(MU_ieee1394_config_t *config,char *line)
{
	char *ptr,*nextp;

	ptr = strchr(line,'[');
	if (ptr == NULL)
		return NULL;
	config->vendor = (unsigned int)strtoul(ptr+1,&nextp,16);
	ptr = strchr(nextp,'[');
	if (ptr == NULL)
		return NULL;
	config->model = (unsigned int)strtoul(ptr+1,&nextp,16);

	return nextp;
}

static void
execute_hotplug(MU_ieee1394_config_t *config)
{
	char buf[256],*scratch;
	char *argv[3],*envp[16];
	int i;

	switch(fork()) {
	case -1:
		LOG(MU_MSG_QUIET,msg_level,"fork error\n");
		exit(1);
	case 0:
		i = init_argument(argv,envp,MU_ARG_IEEE1394);

		/*
		 * ignore "GUID_ID"
		 */
		scratch = buf;
		envp[i++] = scratch;
		scratch += sprintf(scratch,"VENDOR_ID=%06x",config->vendor);
		LOG(MU_MSG_DESC,msg_level,"VENDOR_ID=%06x",config->vendor);

		scratch++;
		envp[i++] = scratch;
		scratch += sprintf(scratch,"SPECIFIER_ID=%06x",config->specifier);
		LOG(MU_MSG_DESC,msg_level,"SPECIFIER_ID=%06x",config->specifier);

		scratch++;
		envp[i++] = scratch;
		sprintf(scratch,"VERSION=%06x",config->version);
		envp[i] = 0;
		LOG(MU_MSG_TRACE,msg_level,"execve [0]:%s [1]:%s\n",argv[0],argv[1]);

		execve(argv[0], argv, envp);
		LOG(MU_MSG_QUIET,msg_level,"Exec error");
		break;
	default:
		if (msg_level >= MU_MSG_TRACE) {
			wait(NULL);
		}
		break;
	}
}

static int
parse_devices(char *buf,int sum)
{
	char *ptr,*nextp;
	int status = ST_NEWLINE;
	MU_ieee1394_config_t config;

	for(ptr = buf; ptr <= buf+sum;ptr++) {
		if (status == ST_NEWLINE) {
			if (*ptr == ' ')
				continue;
			if (MATCH("Node",ptr)) {
				;
			} else if (MATCH(STR_VENDOR,ptr)) {
				ptr = get_vendor_model(&config,ptr);
				if (ptr == NULL)
					return -1;
			} else if (MATCH(STR_SPEC,ptr)) {
				config.specifier = (unsigned int)strtoul(ptr+sizeof(STR_SPEC)-1,&nextp,16);
				ptr = nextp;
			} else if (MATCH(STR_VER,ptr)) {
				config.version = (unsigned int)strtoul(ptr+sizeof(STR_VER)-1,&nextp,16);
				ptr = nextp;
				LOG(MU_MSG_TRACE,msg_level, "vendor:0x%x model:0x%x specifier:0x%x version:0x%x\n",config.vendor, config.model, config.specifier, config.version);
				/* call hotplug */
				execute_hotplug(&config);
			}
			status = ST_DUMB;
		}
		if (*ptr == '\n') {
			status = ST_NEWLINE;
		}
	}

	return 0;
}

int
main(int argc,char **argv)
{
	char *buf;
	int sum;
	char *sysfs_dir;

	sysfs_dir = coldplug_init(MU_INIT_IEEE1394,MU_ARG_IEEE1394,&msg_level);
	if (msg_level == -1)
		exit(0);

	if (access(PROC_FILE,R_OK) != 0)	/* no IEEE1394 devices */
		return 0;

	buf = alloc_read(PROC_FILE,&sum);
	if (buf == NULL)
		return -1;

	parse_devices(buf,sum);
	
#ifdef	KERNEL_JOB
	free(buf);
#endif

	return 0;
}

