/*
 *  Copyright (c) by Shuu Yamaguchi <shuu@dotaster.com>
 *
 *  $Id: loader.c,v 3.27 2004/09/12 07:41:53 shuu Exp shuu $
 *
 *  Can be freely distributed and used under the terms of the GNU GPL.
 */
#include	<unistd.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<fcntl.h>
#include	<limits.h>
#include	<string.h>
#include	<stdio.h>
#include	<sys/mman.h>
#include	<sys/utsname.h>
#include	<stdlib.h>

#include	"murasaki.h"

#define	STICKY_PATH	MU_CONF_DIR "/" STICKY_FILE
#define	BLACKLIST_PATH	MU_CONF_DIR "/" BLACKLIST_FILE

#define	MODULE_BASEDIR	"/lib/modules/"
#define	CHAR_COMMENT	'#'
#define ENABLE_NONE_MODULE

/*
 * called: load_module
 */
static int
get_module(mu_op_t *opp,char *fname)
{
	char *mp;
	int count;
	unsigned int size;

	if ((mp = map_file(fname,&size,opp->msg_level)) == NULL)
		return 0;

	if ((count = opp->match_map(opp,mp,mp+size)) == 0) {
		LOG_OPP_TRACE("Not found in %s", fname);
	}
	munmap(mp,size);

	return count;
}


/*
 * called: load_module
 */
static char *
make_defaultpath(char *fname)
{
	struct utsname uts;
	static char fullpath[PATH_MAX];

	if (uname(&uts) == -1)
		return NULL;
	if (sizeof(MODULE_BASEDIR)-1 + 1 + strlen(fname) + strlen(uts.release) > PATH_MAX)
		return NULL;
	strcpy(fullpath,MODULE_BASEDIR);
	strcpy(fullpath+sizeof(MODULE_BASEDIR)-1,uts.release);
	strcat(fullpath+sizeof(MODULE_BASEDIR),"/");
	strcat(fullpath+sizeof(MODULE_BASEDIR),fname);

	return fullpath;
}

/*
 * called: load_module
 *
 * If module name begins with "alias-", it should be expand name and
 * be not registered in module_list
 * And it is put into alias_list to be called the specified script in
 * murasaki.call.
 */
static void
delete_alias(mu_op_t *opp)
{
	int i;

	for (i = 0;i < opp->module_list.used;i++) {
		if (strncmp(opp->module_list.list[i],ALIAS_STR,sizeof(ALIAS_STR)-1) == 0) {
			add_to_list(&(opp->alias_list),opp->module_list.list[i],opp->msg_level);
			remove_from_list(&(opp->module_list),i);
			i--;		/* look the new module_list[i] */
		}
	}
}

/*
 * called: load_module
 *
 * When action is "remove", module arguments(AAA=xxx) need not only to
 * be given to 'modprobe' but also are worthless.
 * So, they should be removed.
 */
static void
delete_option(mu_op_t *opp)
{
	int i;

	for (i = 0;i < opp->module_list.used;i++) {
		if (strchr(opp->module_list.list[i],'=') != NULL) {
			remove_from_list(&(opp->module_list),i);
			i--;	/* look the new module_list.list[i] */
		}
	}
}

/*
 * called: main
 *
 * Shun-ichi TAHARA fixed free()
 */
int
create_modulelist(mu_op_t *opp)
{
	char *mappath;
	int i;
	char *except_path;

	if (opp->match_map == NULL)
		return INVALID;
	inform_config(opp);	/* print device configuration */

	/*
	 * create unexpected module list for both adding and removing
	 */
	if (opp->action == ACTION_ADD) { 
		except_path = BLACKLIST_PATH;
	} else {	/* ACTION_REMOVE */
		except_path = STICKY_PATH;
	}
	/* 
	 * When list is empty, list[0] == NULL
	 */
	if (list_from_file(except_path,&(opp->except_list)) == INVALID) {
		LOG_OPP_QUIET("Memory allocation failed");
	}
	LOG_ARGS(MU_MSG_TRACE,opp->msg_level,except_path,opp->except_list.list);

	/*
	 * 1st: search murasaki.XXXmap (XXX:usb,pci,ieee1394...)
	 *	:
	 * last: search modules.XXXmap
	 */
	for(i = 0;opp->mappath[i] != NULL;i++) {
		if (opp->mappath[i][0] != '/') {
			mappath = make_defaultpath(opp->mappath[i]);
			if (mappath == NULL)
				return INVALID;
		} else {
			mappath = opp->mappath[i];
		}
		if (get_module(opp,mappath) == INVALID)
			return INVALID;
	}
	LOG_OPP_TRACE("module sum = %d",opp->module_list.used);
	/*
	 * Consider the correct mappath
	 * mappath[0] is correct?
	 */
	if (opp->module_list.used == 0) {	/* not found */
		beep(INVALID,opp->flag);
		LOG_OPP_WHISPER("The device match nothing in mapfile");
		LOG_OPP_WHISPER("Please change MODULE in following line to the appropriate module name, add it to %s",opp->mappath[0]);
		print_config(opp);

		return GOOD;	/* it may be static link */
	}
	beep(GOOD,opp->flag);

	LOG_ARGS(MU_MSG_TRACE,opp->msg_level,"module_list(Pass 1)",opp->module_list.list);

	/*
	 * adding depended modules & expanding alias modules 
	 */
	if (get_depend(opp) == INVALID) {
		LOG_OPP_TRACE("get_depend return INVALID");
		return INVALID;
	}
	LOG_ARGS(MU_MSG_TRACE,opp->msg_level,"module_list(Pass 2)",opp->module_list.list);

	/*
	 *  clean up module_list[]
	 */
	delete_alias(opp);

	if (opp->action == ACTION_REMOVE) {
		delete_option(opp);
	}

	LOG_ARGS(MU_MSG_TRACE,opp->msg_level,"module_list",opp->module_list.list);
	for (i = 0;i < opp->module_list.used;i++) {
		if (strchr(opp->module_list.list[i],'=') != NULL)
			continue;
		LOG_OPP_WHISPER("%s %s",opp->action == ACTION_ADD ? "Loading" : "Unloading" , opp->module_list.list[i]);
	}

	return GOOD;
}

/*
 * The number of modules is not 0, because create_modulelist() counts
 *  the sum of modules.
 */
int
load_module(mu_op_t *opp)
{
	int i,j;
	char *loader;
	char *command_line[LIST_MAX+2];	/* 2 for modprobe + options */

	if (opp->module_list.used == 0)
		return GOOD;
#ifdef	ENABLE_NONE_MODULE
	/*
	 * if module is "none", do nothing.
	 */
	if (opp->module_list.used == 1 && strcmp(opp->module_list.list[0],"none") == 0) {
		LOG_OPP_TRACE("\"none\" -> do nothing");
		return GOOD;
	}
#endif	/* ENABLE_NONE_MODULE */
	/* 
	 * command_line[0]:"modprobe"
	 * command_line[1]:"-as" or "-asr"
	 * command_line[2-]: module names are copied from module_list
	 * command_line[END]: NULL names
	 */
	if ((loader = get_loader()) == NULL) {
		LOG_OPP_QUIET("fork error");
		return INVALID;
	}
	/*
	 * command_line[0](loader) must be specified as a full path.
	 * loader need to be checked permissions.
	 */
	command_line[0] = loader;
	LOG_OPP_TRACE("loader = \"%s\"",loader);
	if (FLAG_CHECK(opp->flag,FLAG_KERNEL25)) {	/* 2.5.x > version */
		LOG_OPP_TRACE("Version >= 2.5");
		if (opp->action == ACTION_ADD)
			command_line[1] = "-qs";
		else 	/* ACTION_REMOVE */
			command_line[1] = "-qsr";
		for(i = 0,j = 2;i < opp->module_list.used;) {
			command_line[j++] = opp->module_list.list[i++];
			if (i < opp->module_list.used && strchr(opp->module_list.list[i],'=') != NULL) {
				continue;
			}
			command_line[j] = NULL;	/* verbose */
			j = 2;

			executing(opp,command_line,FORK_ON);
		}
	} else {
		LOG_OPP_TRACE("Version <= 2.4");
		if (opp->action == ACTION_ADD)
			command_line[1] = "-as";
		else 	/* ACTION_REMOVE */
			command_line[1] = "-asr";
		for (i = 0,j = 2;i < opp->module_list.used;i++) {
			command_line[j++] = opp->module_list.list[i];
		}
		command_line[j] = NULL;	/* verbose */

		executing(opp,command_line,FORK_ON);
	}

	return GOOD;
}
