//muroarstream.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2009-2011
 *
 *  This file is part of µRoar,
 *  a minimalist library to access a RoarAudio Sound Server.
 *
 *  µRoar is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  µRoar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with µRoar.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "muroar.h"
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#define NAME "muroarstream"

void usage (char * progname) {
 printf("Usage: %s [OPTIONS]\n", progname);
 printf("\n");
 printf("Options:\n");
 printf("    --help              - Display this help.\n"
        "    --server   SERVER   - Sets the server. (default: autodetect)\n"
        " -R --rate     RATE     - Sets the sample rate (default: 44100Hz)\n"
        " -C --channels CHANS    - Sets the number of channels (default: 2 channels)\n"
        " -B --bits     BITS     - Sets the number of bits per sample (default: 16bit)\n"
        "    --wave              - Use Waveform subsystem (default)\n"
        "    --midi              - Use MIDI subsystem\n"
        "    --light             - Use Light Control subsystem\n"
       );
}

void run_stream(int stream, int tx) {
 char buf[1024];
 ssize_t len;

 if ( tx ) {
  while ((len = read(STDIN_FILENO, buf, sizeof(buf))) > 0)
   if ( muroar_write(stream, buf, len) != len )
    return;
 } else {
  while ((len = muroar_read(stream, buf, sizeof(buf))) > 0)
   if ( write(STDOUT_FILENO, buf, len) != len )
    return;
 }
}

int main (int argc, char *argv[]) {
 int fh;
 int dir   = MUROAR_PLAY_WAVE;
 int codec = -1;     /* use default depending in dir */
 int rate  =  MUROAR_AID_RATE;
 int chans =  MUROAR_AID_CHANNELS;
 int bits  =  MUROAR_AID_BITS;
 int tx    =  1;
 int i;
 char * k;
 char * server = NULL;

 for (i = 1; i < argc; i++) {
  k = argv[i];

  if ( !strcmp(k, "--help") ) {
   usage(argv[0]);
   return 0;
  } else if ( !strcmp(k, "--server") ) {
   server = argv[++i];
  } else if ( !strcmp(k, "--rate") || !strcmp(k, "-R") ) {
   rate = atoi(argv[++i]);
  } else if ( !strcmp(k, "--channels") || !strcmp(k, "-C") ) {
   chans = atoi(argv[++i]);
  } else if ( !strcmp(k, "--bits") || !strcmp(k, "-B") ) {
   bits = atoi(argv[++i]);
  } else if ( !strcmp(k, "--play") ) {
   tx = 1;
  } else if ( !strcmp(k, "--monitor") ) {
   tx = 0;
  } else if ( !strcmp(k, "--wave") ) {
   dir = MUROAR_PLAY_WAVE;
  } else if ( !strcmp(k, "--midi") ) {
   dir = MUROAR_PLAY_MIDI;
  } else if ( !strcmp(k, "--light") ) {
   dir = MUROAR_PLAY_LIGHT;
  } else {
   fprintf(stderr, "%s: Unknown argument: %s\n", argv[0], k);
   return 4;
  }
 }

 // Set codec depending on dir in case codec is default (-1):
 if ( codec == -1 ) {
  switch (dir) {
   case MUROAR_PLAY_WAVE:  codec = MUROAR_CODEC_PCM;    break;
   case MUROAR_PLAY_MIDI:  codec = MUROAR_CODEC_MIDI;   break;
   case MUROAR_PLAY_LIGHT: codec = MUROAR_CODEC_DMX512; break;
  }
 }

 // Correct dir in case we do not transmit to MONITOR dirs:
 if ( !tx ) {
  switch (dir) {
   case MUROAR_PLAY_WAVE:  dir = MUROAR_MONITOR_WAVE;  break;
   case MUROAR_PLAY_MIDI:  dir = MUROAR_MONITOR_MIDI;  break;
   case MUROAR_PLAY_LIGHT: dir = MUROAR_MONITOR_LIGHT; break;
  }
 }

 // Connect to server:
 if ( (fh = muroar_connect(server, NAME)) == -1 ) {
  fprintf(stderr, "Error: Can not connect to server.\n");
  return 1;
 }

 if ( muroar_stream(fh, dir, NULL, codec, rate, chans, bits) == -1 ) {
  fprintf(stderr, "Error: Can not connect stream to server.\n");
  // We do not need to close fh here as it is already closed by muroar_stream().
  return 2;
 }

 run_stream(fh, tx);

 muroar_close(fh);

 return 0;
}

//ll
