/*
 * Implementation of the generic proxy for Rhythmbox.
 *
 * Music Applet
 * Copyright (C) 2006 Paul Kuliniewicz <paul.kuliniewicz@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.
 */

/*
 * Why isn't MaAnyProxy enough?  There *are* some common aspects of working
 * with Rhythmbox regardless of the IPC mechanism being used, so this
 * separate class is used to collect them.  It also gives an easy way for
 * MaAnyProxy to only see one Rhythmbox proxy, simplifying its code.  (We
 * don't want the user to worry about D-Bus versus Bonobo.)
 */

#include <config.h>

#include "ma-rhythmbox-proxy.h"
#include "ma-rhythmbox-bonobo-proxy.h"

#ifdef WITH_DBUS
#  include "ma-rhythmbox-dbus-proxy.h"
#endif

#include <string.h>


#define GET_PRIVATE(o) 			(G_TYPE_INSTANCE_GET_PRIVATE ((o), MA_TYPE_RHYTHMBOX_PROXY, MaRhythmboxProxyPrivate))


typedef struct _MaRhythmboxProxyPrivate		MaRhythmboxProxyPrivate;


struct _MaRhythmboxProxyPrivate
{
	MaProxy *bonobo;
	MaProxy *dbus;

	MaProxy *active;
};

static MaProxyClass *parent_class;


/*********************************************************************
 *
 * Function declarations
 *
 *********************************************************************/

static void ma_rhythmbox_proxy_class_init (MaRhythmboxProxyClass *klass);
static void ma_rhythmbox_proxy_init (MaRhythmboxProxy *rproxy);

static void ma_rhythmbox_proxy_dispose (GObject *object);
static void ma_rhythmbox_proxy_finalize (GObject *object);

static void ma_rhythmbox_proxy_toggle_playback (MaProxy *proxy);
static void ma_rhythmbox_proxy_previous (MaProxy *proxy);
static void ma_rhythmbox_proxy_next (MaProxy *proxy);

static void ma_rhythmbox_proxy_rate_song (MaProxy *proxy, gdouble rating);

static void ma_rhythmbox_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml);

static void ma_rhythmbox_proxy_enable (MaProxy *proxy);
static void ma_rhythmbox_proxy_disable (MaProxy *proxy);

static void ma_rhythmbox_proxy_launch (MaProxy *proxy);

static void choose_active_proxy (MaRhythmboxProxy *rproxy);
static void set_active (MaRhythmboxProxy *rproxy, MaProxy *active);

static void notify_cb (MaProxy *proxy, GParamSpec *pspec, MaRhythmboxProxy *rproxy);

static void error_reported_cb (MaProxy *proxy,
			       const gchar *primary,
			       const gchar *secondary,
			       gboolean important,
			       MaRhythmboxProxy *rproxy);

static void command_toggled_cb (GtkToggleButton *button, GtkWidget *widget);

/*********************************************************************
 *
 * GType stuff
 *
 *********************************************************************/

GType
ma_rhythmbox_proxy_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (MaRhythmboxProxyClass),			/* class_size */
			NULL,						/* base_init */
			NULL,						/* base_finalize */
			(GClassInitFunc) ma_rhythmbox_proxy_class_init,	/* class_init */
			NULL,						/* class_finalize */
			NULL,						/* class_data */
			sizeof (MaRhythmboxProxy),			/* instance_size */
			0,						/* n_preallocs */
			(GInstanceInitFunc) ma_rhythmbox_proxy_init,	/* instance_init */
			NULL
		};

		type = g_type_register_static (MA_TYPE_PROXY, "MaRhythmboxProxy", &info, 0);
	}

	return type;
}

static void
ma_rhythmbox_proxy_class_init (MaRhythmboxProxyClass *klass)
{
	GObjectClass *object_class = (GObjectClass *) klass;
	MaProxyClass *proxy_class = (MaProxyClass *) klass;
	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = ma_rhythmbox_proxy_dispose;
	object_class->finalize = ma_rhythmbox_proxy_finalize;

	proxy_class->toggle_playback = ma_rhythmbox_proxy_toggle_playback;
	proxy_class->previous = ma_rhythmbox_proxy_previous;
	proxy_class->next = ma_rhythmbox_proxy_next;

	proxy_class->rate_song = ma_rhythmbox_proxy_rate_song;

	proxy_class->prepare_prefs = ma_rhythmbox_proxy_prepare_prefs;

	proxy_class->enable = ma_rhythmbox_proxy_enable;
	proxy_class->disable = ma_rhythmbox_proxy_disable;

	proxy_class->launch = ma_rhythmbox_proxy_launch;

	g_type_class_add_private (klass, sizeof (MaRhythmboxProxyPrivate));
}

static void
ma_rhythmbox_proxy_init (MaRhythmboxProxy *rproxy)
{
	MaRhythmboxProxyPrivate *priv = GET_PRIVATE (rproxy);

	priv->bonobo = NULL;
	priv->dbus = NULL;

	priv->active = NULL;
}


/*********************************************************************
 *
 * Public interface
 *
 *********************************************************************/

MaProxy *
ma_rhythmbox_proxy_new (MaConf *conf)
{
	MaRhythmboxProxy *rproxy;
	MaRhythmboxProxyPrivate *priv;

	rproxy = g_object_new (MA_TYPE_RHYTHMBOX_PROXY,
			       "name", "Rhythmbox",
			       "icon-name", "rhythmbox",
			       "conf", conf,
			       NULL);

	priv = GET_PRIVATE (rproxy);

	priv->bonobo = ma_rhythmbox_bonobo_proxy_new (conf);
	g_signal_connect (priv->bonobo, "error-reported", G_CALLBACK (error_reported_cb), rproxy);

#ifdef WITH_DBUS
	priv->dbus = ma_rhythmbox_dbus_proxy_new (conf);
	g_signal_connect (priv->dbus, "error-reported", G_CALLBACK (error_reported_cb), rproxy);
#endif

	return MA_PROXY (rproxy);
}


/*********************************************************************
 *
 * GObject overrides
 *
 *********************************************************************/

static void
ma_rhythmbox_proxy_dispose (GObject *object)
{
	MaRhythmboxProxy *rproxy = MA_RHYTHMBOX_PROXY (object);
	MaRhythmboxProxyPrivate *priv = GET_PRIVATE (rproxy);

	ma_rhythmbox_proxy_disable (MA_PROXY (rproxy));

	priv->active = NULL;

	if (priv->bonobo != NULL)
	{
		g_object_unref (priv->bonobo);
		priv->bonobo = NULL;
	}

	if (priv->dbus != NULL)
	{
		g_object_unref (priv->dbus);
		priv->dbus = NULL;
	}

	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
ma_rhythmbox_proxy_finalize (GObject *object)
{
	G_OBJECT_CLASS (parent_class)->finalize (object);
}


/*********************************************************************
 *
 * MaProxy overrides
 *
 *********************************************************************/

static void
ma_rhythmbox_proxy_toggle_playback (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_PROXY (proxy));

	ma_proxy_toggle_playback (GET_PRIVATE (proxy)->active);
}

static void
ma_rhythmbox_proxy_previous (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_PROXY (proxy));

	ma_proxy_previous (GET_PRIVATE (proxy)->active);
}

static void
ma_rhythmbox_proxy_next (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_PROXY (proxy));

	ma_proxy_next (GET_PRIVATE (proxy)->active);
}

static void
ma_rhythmbox_proxy_rate_song (MaProxy *proxy, gdouble rating)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_PROXY (proxy));

	ma_proxy_set_rating (GET_PRIVATE (proxy)->active, rating);
}

static void
ma_rhythmbox_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml)
{
	MaConf *conf;

	GtkWidget *page;
	GtkWidget *launch_dbus;
	GtkWidget *launch_command;
	GtkWidget *command;
	GtkWidget *browse;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_PROXY (proxy));

	conf = _ma_proxy_get_conf (proxy);

	page = glade_xml_get_widget (xml, "rhythmbox");
	launch_dbus = glade_xml_get_widget (xml, "rhythmbox-launch-dbus");
	launch_command = glade_xml_get_widget (xml, "rhythmbox-launch-command");
	command = glade_xml_get_widget (xml, "rhythmbox-command");
	browse = glade_xml_get_widget (xml, "rhythmbox-browse");

	ma_conf_bind_string_radio (conf, "rhythmbox_launch", "D-Bus", GTK_RADIO_BUTTON (launch_dbus));
	ma_conf_bind_string_radio (conf, "rhythmbox_launch", "Command", GTK_RADIO_BUTTON (launch_command));
	ma_conf_bind_string_entry (conf, "rhythmbox_command", GTK_ENTRY (command));
	ma_conf_bind_string_browse (conf, "rhythmbox_command", GTK_BUTTON (browse));

#ifndef WITH_DBUS
	gtk_widget_set_sensitive (launch_dbus, FALSE);
#endif

	gtk_widget_set_sensitive (command, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (launch_command)));
	gtk_widget_set_sensitive (browse, gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (launch_command)));

	g_signal_connect (launch_command, "toggled", G_CALLBACK (command_toggled_cb), command);
	g_signal_connect (launch_command, "toggled", G_CALLBACK (command_toggled_cb), browse);

	gtk_widget_show (page);
}

static void
ma_rhythmbox_proxy_enable (MaProxy *proxy)
{
	MaRhythmboxProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	g_signal_connect (priv->bonobo, "notify", G_CALLBACK (notify_cb), proxy);
	ma_proxy_enable (priv->bonobo);

#ifdef WITH_DBUS
	g_signal_connect (priv->dbus, "notify", G_CALLBACK (notify_cb), proxy);
	ma_proxy_enable (priv->dbus);
#endif

	choose_active_proxy (MA_RHYTHMBOX_PROXY (proxy));
}

static void
ma_rhythmbox_proxy_disable (MaProxy *proxy)
{
	MaRhythmboxProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	if (priv->bonobo != NULL)
	{
		ma_proxy_disable (priv->bonobo);
		g_signal_handlers_disconnect_by_func (priv->bonobo, notify_cb, proxy);
	}

#ifdef WITH_DBUS
	if (priv->dbus != NULL)
	{
		ma_proxy_disable (priv->dbus);
		g_signal_handlers_disconnect_by_func (priv->dbus, notify_cb, proxy);
	}
#endif
}

static void
ma_rhythmbox_proxy_launch (MaProxy *proxy)
{
	MaRhythmboxProxyPrivate *priv;

	MaConf *conf;
	gchar *launch;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_RHYTHMBOX_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	conf = _ma_proxy_get_conf (proxy);
	launch = ma_conf_get_string (conf, "rhythmbox_launch");

	if (strcmp (launch, "Command") == 0 || priv->dbus == NULL)
		ma_proxy_launch (priv->bonobo);
	else
		ma_proxy_launch (priv->dbus);
}


/*********************************************************************
 *
 * Internal functions
 *
 *********************************************************************/

static void
choose_active_proxy (MaRhythmboxProxy *rproxy)
{
	MaRhythmboxProxyPrivate *priv = GET_PRIVATE (rproxy);

	MaProxyState state;

#ifdef WITH_DBUS
	state = ma_proxy_get_state (MA_PROXY (priv->dbus));
	if (state == MA_PROXY_STATE_CONNECTED)
	{
		if (priv->active != priv->dbus)
		{
			set_active (rproxy, priv->dbus);
			ma_proxy_disable (priv->bonobo);
		}
		return;
	}
	else if (state != MA_PROXY_STATE_CONNECTED && priv->active == priv->dbus)
	{
		ma_proxy_enable (priv->bonobo);
	}
#endif

	state = ma_proxy_get_state (MA_PROXY (priv->bonobo));
	if (state == MA_PROXY_STATE_CONNECTED)
	{
		if (priv->active != priv->bonobo)
			set_active (rproxy, priv->bonobo);
		return;
	}

	priv->active = NULL;
	_ma_proxy_set_connected (MA_PROXY (rproxy), FALSE);
}

static void
set_active (MaRhythmboxProxy *rproxy, MaProxy *active)
{
	MaRhythmboxProxyPrivate *priv = GET_PRIVATE (rproxy);

	priv->active = active;

	_ma_proxy_set_connected (MA_PROXY (rproxy), TRUE);

	_ma_proxy_set_title (MA_PROXY (rproxy), ma_proxy_get_title (priv->active));
	_ma_proxy_set_artist (MA_PROXY (rproxy), ma_proxy_get_artist (priv->active));
	_ma_proxy_set_album (MA_PROXY (rproxy), ma_proxy_get_album (priv->active));
	_ma_proxy_set_duration (MA_PROXY (rproxy), ma_proxy_get_duration (priv->active));
	_ma_proxy_set_rating (MA_PROXY (rproxy), ma_proxy_get_rating (priv->active));
	_ma_proxy_set_playing (MA_PROXY (rproxy), ma_proxy_get_playing (priv->active));
	_ma_proxy_set_elapsed (MA_PROXY (rproxy), ma_proxy_get_elapsed (priv->active));
}


/*********************************************************************
 *
 * Callbacks
 *
 *********************************************************************/

static void
notify_cb (MaProxy *proxy, GParamSpec *pspec, MaRhythmboxProxy *rproxy)
{
	MaRhythmboxProxyPrivate *priv = GET_PRIVATE (rproxy);

	if (strcmp(pspec->name, "state") == 0)
		choose_active_proxy (rproxy);
	else if (proxy == priv->active)
	{
		if (strcmp(pspec->name, "elapsed") == 0)
		{
			_ma_proxy_set_elapsed (MA_PROXY (rproxy), ma_proxy_get_elapsed (proxy));
		}
		else if (strcmp(pspec->name, "playing") == 0)
		{
			_ma_proxy_set_playing (MA_PROXY (rproxy), ma_proxy_get_playing (proxy));
		}
		else if (strcmp(pspec->name, "rating") == 0)
		{
			_ma_proxy_set_rating (MA_PROXY (rproxy), ma_proxy_get_rating (proxy));
		}
		else if (strcmp(pspec->name, "title") == 0)
		{
			_ma_proxy_set_title (MA_PROXY (rproxy), ma_proxy_get_title (proxy));
		}
		else if (strcmp(pspec->name, "artist") == 0)
		{
			_ma_proxy_set_artist (MA_PROXY (rproxy), ma_proxy_get_artist (proxy));
		}
		else if (strcmp(pspec->name, "album") == 0)
		{
			_ma_proxy_set_album (MA_PROXY (rproxy), ma_proxy_get_album (proxy));
		}
		else if (strcmp(pspec->name, "duration") == 0)
		{
			_ma_proxy_set_duration (MA_PROXY (rproxy), ma_proxy_get_duration (proxy));
		}
	}
}

static void
error_reported_cb (MaProxy *proxy,
		   const gchar *primary,
		   const gchar *secondary,
		   gboolean important,
		   MaRhythmboxProxy *rproxy)
{
	const gchar *name = (important) ? "error-reported::important" : "error-reported::normal";
	g_signal_emit_by_name (rproxy, name, primary, secondary, important);
}

static void
command_toggled_cb (GtkToggleButton *button, GtkWidget *widget)
{
	gtk_widget_set_sensitive (widget, gtk_toggle_button_get_active (button));
}
