/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 conversions.cpp  -  Various IP naming/numbering conversions and base64 codec

 the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "basicstruct.h"
#include "packet.h"
#include "conversions.h"
#include "mthread.h"
#include "common.h"
#include "asyncsocket.h" // for INADDR_NONE on some systems

#ifdef _DEBUG
#include <time.h>
#endif

#include <ctype.h>

CString DWrdtoStr(DWORD in)
{
	//char buff[16];
	//::sprintf (buff, "%u", in);
	//return buff;
	CString sTmp;
	return sTmp.format("%u", in);
}

CString Ip2Str(const IP& ip)
{
    char buffer[32];
    ::sprintf(buffer,"%d.%d.%d.%d",ip.S_ip.a,ip.S_ip.b,ip.S_ip.c,ip.S_ip.d);
	return buffer;
}

bool Str2Ip(const CString& sIn, IP& ip)
{
	CString s = StripWhite(sIn);
	struct in_addr addr;
	addr.s_addr == INADDR_NONE;
	#warning "we also use g_LibcMutex here"
	MLock lock(g_LibcMutex);
	addr.s_addr = inet_addr(s.c_str());
	if (addr.s_addr != INADDR_NONE)
	{
		ip.S_addr = addr.s_addr;
		return true;
	}
	return false;
}


bool Str2Ip_lookup(const CString& sIn, IP& ip)
{
	CString s = StripWhite(sIn);
	struct in_addr addr;
	#warning "we temporarily place g_LibcMutex where I dont want to have it"
	MLock lock(g_LibcMutex);
	addr.s_addr = inet_addr(s.c_str());
	if (addr.s_addr != INADDR_NONE)
	{
		ip.S_addr = addr.s_addr;
		return true;
	}
#ifdef _DEBUG
	time_t t;
#endif
	IP out;
	//MLock lock(g_GetHostByNameMutex);
	out.S_addr = 0;
#ifdef _DEBUG
	time(&t);
#endif
	hostent * phe = ::GetHostByName(s.c_str());
	if (NULL != phe)
	{
		// unpack the address -- quick and ...
		// TODO: check if it is correct
		memcpy(&ip, phe->h_addr_list[0], 4);
	}
#ifdef _DEBUG
	if (xtime()-t>1)
	{
		TRACE2("StrtoIP: blocking in gethostbyname when resolving ", s);
	}
#endif
	return phe;
}

CString IpNet2Str(const IPNET& ipn)
{
	char buffer[48];
	::sprintf(buffer,"%d.%d.%d.%d/%d",ipn.S_ip.a,ipn.S_ip.b,ipn.S_ip.c,ipn.S_ip.d,ipn.net_bits);
	return buffer;
}

bool Str2IpNet(const CString& sNet, IPNET& ipn)
{
	int nSlashPos = sNet.find('/');
	if (nSlashPos==0) //==0 would mena the string starts with /
		return false;
	if (nSlashPos > 0)
		ipn.net_bits = atol(sNet.substr(nSlashPos+1).c_str());
	else
	{
		ipn.net_bits = 32; // no slash means just an IP
		nSlashPos = sNet.length();
	}
	if (ipn.net_bits < 0 || ipn.net_bits > 32)
		return false;
	MLock lock(g_LibcMutex);
	ipn.S_addr = ::inet_addr(sNet.substr(0,nSlashPos).c_str());
	if (ipn.S_addr == INADDR_NONE)
		return false;
	return true;
}

DWORD GetSpeedinBytes(CString Speed)
// The protocol is messed, bytes are bits 
{
	if(Speed == "Cell Modem")
		return 0;
	if(Speed == "14.4 Modem")
		return 14;
	if(Speed == "28.8 Modem")
		return 28;
	if(Speed == "56K Modem")
		return 53;
	if(Speed == "ISDN")
		return 128;
	if(Speed == "Cable")
		return 384;
	if(Speed == "DSL")
		return 768;
	if(Speed == "T1")
		return 1500;
	if(Speed == "T3 or Better")
		return 45000;
	
	return atol(Speed.c_str());
}

CString GetSpeedString(DWORD dwSpeed)
// The protocol is messed, bytes are bits 
{
	if(dwSpeed >= 45000)
		return "T3 or Better";
	if(dwSpeed >= 1500)
		return "T1";
	if(dwSpeed >= 768)
		return "DSL";
	if(dwSpeed >= 384)
		return "Cable";
	if(dwSpeed >= 128)
		return "ISDN";
	if(dwSpeed >= 53)
		return "56K Modem";
	if(dwSpeed >= 28)
		return "28.8 Modem";
	if(dwSpeed >= 14)
		return "14.4 Modem";
	else
		return "Cell Modem";
}

bool ValidVendor(CString VendorID)
{
	if(GetVendor(VendorID)[0] == '(')
		return false;
	
	return true;
}

/*
Code Name
-----------------
BEAR BearShare * +
CULT Cultiv8r *
FISH PEERanha *
GNEW gnewtellium *
GNOT Gnotella
GNUC Gnucleus *
GNUT Gnut
GTKG Gtk-Gnutella *
HSLG Hagelslag
LIME LimeWire * ++
MACT Mactella
MRPH Morpheus *
MUTE Mutella *
NAPS NapShare *
OCFG OpenCola
OPRA Opera
PHEX Phex *
QTEL Qtella
SALM Salmonella *
SNUT SwapNut +
SWAP Swapper
TOAD ToadNode
XOLO XoloX **
ZIGA Ziga
*/

CString GetVendor(CString VendorID)
{
	MakeUpper(VendorID);
	//
	if(VendorID == "ACQX") return "Acqusition";
	if(VendorID == "BEAR") return "BearShare";
	if(VendorID == "CULT") return "Cultiv8r";
	if(VendorID == "FISH") return "PEERanha";
	if(VendorID == "GNEW") return "Gnewtellium";
	if(VendorID == "GNOT") return "Gnotella";
	if(VendorID == "GNUC") return "Gnucleus";
	if(VendorID == "GNUT") return "Gnut";
	if(VendorID == "GTKG") return "Gtk-Gnutella";
	if(VendorID == "HSLG") return "Hagelslag";
	if(VendorID == "LIME") return "LimeWire";
	if(VendorID == "MACT") return "Mactella";
	if(VendorID == "MRPH") return "Morpheus";
	if(VendorID == "MUTE") return "Mutella";
	if(VendorID == "NAPS") return "NapShare";
	if(VendorID == "OCFG") return "OpenCola";
	if(VendorID == "OPRA") return "Opera";
	if(VendorID == "PHEX") return "Phex";
	if(VendorID == "QTEL") return "Qtella";
	if(VendorID == "RAZA") return "Shareaza";
	if(VendorID == "SALM") return "Salmonella";
	if(VendorID == "SNUT") return "SwapNut";
	if(VendorID == "SWAP") return "Swapper";
	if(VendorID == "TOAD") return "ToadNode";
	if(VendorID == "XOLO") return "XoloX";
	if(VendorID == "ZIGA") return "Ziga";
	// if the code is alfa-numeric
	if (VendorID.size()==4   &&
		isalnum(VendorID[0]) &&
		isalnum(VendorID[1]) &&
		isalnum(VendorID[2]) &&
		isalnum(VendorID[3]) )
		return "(" + VendorID + ")";
	return "(Unknown)";
}

static hostent *internal_fakehostent;
// MZ: would not compile
/*MZ: __private_extern__*/ struct hostent *
fake_hostent(const char *name, struct in_addr addr)
{
	int addr_len;

	if (name == NULL) return NULL;

	// If not created, calloc our internal hostent.
	if (internal_fakehostent==NULL)
	{
		internal_fakehostent = (struct hostent *)calloc(1, sizeof(struct hostent));
		ASSERT(internal_fakehostent);
		internal_fakehostent->h_aliases = (char**)calloc(1, sizeof(char *));
		ASSERT(internal_fakehostent->h_aliases);
		internal_fakehostent->h_addrtype=AF_INET;
		internal_fakehostent->h_length = sizeof(long);
		internal_fakehostent->h_addr_list = (char **)calloc(2, sizeof(u_long *));
		ASSERT(internal_fakehostent->h_addr_list);
		internal_fakehostent->h_addr_list[0] = (char *)calloc(1, sizeof(long));
		ASSERT(internal_fakehostent->h_addr_list[0]);
	}

	// Now, clear any volatile data.
	if (internal_fakehostent->h_name != NULL)
	{
		free( (char*)internal_fakehostent->h_name );
		internal_fakehostent->h_name = NULL;
	}

	internal_fakehostent->h_name = strdup(name);
	ASSERT(internal_fakehostent->h_name);
	
	memmove(internal_fakehostent->h_addr_list[0],
		&(addr.s_addr), sizeof(long));

	return internal_fakehostent;
}

static int is_ip(const char *host)
{
	while ((*host == '.') || isdigit(*host))
	host++;
	return (*host == '\0');
}


hostent *GetHostByName(const char *host)
{
	struct hostent *res = NULL;

	if ((host==NULL)||(host[0]=='\0'))
	{
		h_errno = NO_DATA;
		return NULL;
	}	

	if (is_ip(host))
	{
		// MZ: inet_aton does not link on Solaris
		struct in_addr addr;
		#warning "we also use g_LibcMutex here"
		MLock lock(g_LibcMutex);
		addr.s_addr = inet_addr(host);
		if (addr.s_addr == INADDR_NONE)
		{
			h_errno = NO_DATA;
			res = NULL;
		}
		else res = fake_hostent(host, addr);
	}
	else
	{
		res = gethostbyname(host);
	}

	return res;
}

//*********************************************************************
//* Base64 - a simple base64 encoder and decoder.
//*
//*     Copyright (c) 1999, Bob Withers - bwit@pobox.com
//*
//* This code may be freely used for any purpose, either personal
//* or commercial, provided the authors copyright notice remains
//* intact.
//*********************************************************************

const char    Base64::fillchar = '=';

                          // 00000000001111111111222222
                          // 01234567890123456789012345
const CString Base64::cvt = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"

                          // 22223333333333444444444455
                          // 67890123456789012345678901
                            "abcdefghijklmnopqrstuvwxyz"

                          // 555555556666
                          // 234567890123
                            "0123456789+/";

CString Base64::encode(CString data)
{
    auto     int       i;
    auto     char      c;
    auto     int       len = data.length();
    auto     CString   ret;

    for (i = 0; i < len; ++i)
    {
        c = (data[i] >> 2) & 0x3f;
        ret += cvt[c];
        c = (data[i] << 4) & 0x3f;
        if (++i < len)
            c |= (data[i] >> 4) & 0x0f;

        ret += cvt[c];
        if (i < len)
        {
            c = (data[i] << 2) & 0x3f;
            if (++i < len)
                c |= (data[i] >> 6) & 0x03;

            ret += cvt[c];
        }
        else
        {
            ++i;
            ret += fillchar;
        }

        if (i < len)
        {
            c = data[i] & 0x3f;
            ret += cvt[c];
        }
        else
        {
            ret += fillchar;
        }
    }

    return(ret);
}

CString Base64::decode(CString data)
{
    auto     int     i;
    auto     char    c;
    auto     char    c1;
    auto     int     len = data.length();
    auto     CString ret;

    for (i = 0; i < len; ++i)
    {
        c = (char) cvt.find(data[i]);
        ++i;
        c1 = (char) cvt.find(data[i]);
        c = (c << 2) | ((c1 >> 4) & 0x3);
        ret += c;
        if (++i < len)
        {
            c = data[i];
            if (fillchar == c)
                break;

            c = (char) cvt.find(c);
            c1 = ((c1 << 4) & 0xf0) | ((c >> 2) & 0xf);
            ret += c1;
        }

        if (++i < len)
        {
            c1 = data[i];
            if (fillchar == c1)
                break;

            c1 = (char) cvt.find(c1);
            c = ((c << 6) & 0xc0) | c1;
            ret += c;
        }
    }

    return(ret);
}

///////////////////////////////////////////////////////////////////
// I've borrowed Base32 code from gtk-gnutella, but it appears that
// it was also borrowed from somewhere else

/*
 *                       The Base 32 Alphabet
 *
 *     Value Encoding  Value Encoding  Value Encoding  Value Encoding
 *         0 A             9 J            18 S            27 3
 *         1 B            10 K            19 T            28 4
 *         2 C            11 L            20 U            29 5
 *         3 D            12 M            21 V            30 6
 *         4 E            13 N            22 W            31 7
 *         5 F            14 O            23 X
 *         6 G            15 P            24 Y         (pad) =
 *         7 H            16 Q            25 Z
 *         8 I            17 R            26 2
 */
static char values[256] = {
/*  0  1  2  3  4  5  6  7  8  9  */	/* 0123456789              */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            -  00 ->  09 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            -  10 ->  19 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            -  20 ->  29 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            -  30 ->  39 */
    -1,-1,-1,-1,-1,-1,-1,-1,			/*            -  40 ->  47 */

    -1,-1,26,27,28,29,30,31,-1,-1,		/* 0123456789 -  48 ->  57 */
    -1,-1,-1,-1,-1,-1,-1, 0, 1, 2,		/* :;<=>?@ABC -  58 ->  67 */
     3, 4, 5, 6, 7, 8, 9,10,11,12,		/* DEFGHIJKLM -  68 ->  77 */
    13,14,15,16,17,18,19,20,21,22,		/* NOPQRSTUVW -  78 ->  87 */
    23,24,25,							/* XYZ        -  88 ->  90 */

    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            -  91 -> 100 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 101 -> 110 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 111 -> 120 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 121 -> 130 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 131 -> 140 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 141 -> 150 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 151 -> 160 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 161 -> 170 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 171 -> 180 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 181 -> 190 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 191 -> 200 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 201 -> 210 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 211 -> 220 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 221 -> 230 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 231 -> 240 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 241 -> 250 */
    -1,-1,-1,-1,-1,						/*            - 251 -> 255 */
};

static char *b32_alphabet = "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567";

/*
 * Older base32 alphabet: "ABCDEFGHIJK MN PQRSTUVWXYZ  23456789"
 * We decode it only.
 */
static char old_values[256] = {
/*  0  1  2  3  4  5  6  7  8  9  */	/* 0123456789              */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            -  00 ->  09 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            -  10 ->  19 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            -  20 ->  29 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            -  30 ->  39 */
    -1,-1,-1,-1,-1,-1,-1,-1,			/*            -  40 ->  47 */

    -1,-1,24,25,26,27,28,29,30,31,		/* 0123456789 -  48 ->  57 */
    -1,-1,-1,-1,-1,-1,-1, 0, 1, 2,		/* :;<=>?@ABC -  58 ->  67 */
     3, 4, 5, 6, 7, 8, 9,10,-1,11,		/* DEFGHIJKLM -  68 ->  77 */
    12,-1,13,14,15,16,17,18,19,20,		/* NOPQRSTUVW -  78 ->  87 */
    21,22,23,							/* XYZ        -  88 ->  90 */

    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            -  91 -> 100 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 101 -> 110 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 111 -> 120 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 121 -> 130 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 131 -> 140 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 141 -> 150 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 151 -> 160 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 161 -> 170 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 171 -> 180 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 181 -> 190 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 191 -> 200 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 201 -> 210 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 211 -> 220 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 221 -> 230 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 231 -> 240 */
    -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,		/*            - 241 -> 250 */
    -1,-1,-1,-1,-1,						/*            - 251 -> 255 */
};

/*
 * encode_pad_length
 *
 * Compute the number of base32 digits and amount of padding necessary
 * to encode `len' bytes.
 *
 * Returns the number of base32 digits necessary.
 * Furthermore, if `pad' is a non-NULL pointer, it is filled with the amount
 * of padding chars that would be necessary.
 */
static int encode_pad_length(int len, int *pad)
{
	int ndigits;			/* Base32 digits necessary */
	int npad = 0;			/* Final padding chars necessary */
	int qcount;			/* Amount of full quintets */
	int remainder;			/* Amount of input bytes in final quintet */

	ASSERT(len > 0);

	qcount = len / 5;
	remainder = len - (qcount * 5);

	ASSERT(remainder >= 0);

	switch (remainder) {
	case 0: npad = 0; break;
	case 1: npad = 6; break;
	case 2: npad = 4; break;
	case 3: npad = 3; break;
	case 4: npad = 1; break;
	default: ASSERT(0);		/* Not possible */
	}

	ndigits = qcount * 8;		/* Each full quintet encoded on 8 bytes */
	if (npad != 0)
		ndigits += (8 - npad);

	if (pad)
		*pad = npad;

	return ndigits;
}

/*
 * base32_encode_exactly
 *
 * Encode `len' bytes from `buf' into `enclen' bytes starting from `encbuf'.
 * Caller must have ensured that there was EXACTLY the needed room in encbuf.
 */
static void base32_encode_exactly(const BYTE *buf, int len,
	char *encbuf, int enclen)
{
	long i = 0;					/* Input accumulator, 0 for trailing pad */
	BYTE const *ip = buf + len;	/* Input pointer, one byte off end */
	char *op = encbuf + enclen;	/* Output pointer, one byte off end */

	ASSERT(buf);
	ASSERT(encbuf);
	ASSERT(len > 0);
	ASSERT(enclen >= len * 8 / 5);

	/*
	 * In the following picture, we represent how the 5 bytes of input
	 * are split into groups of 5 bits, each group being encoded as a
	 * single base32 digit.
	 *
	 * input byte       0        1        2        3        4
	 *              +--------+--------+--------+--------+--------+
	 *              |01234012|34012340|12340123|40123401|23401234|
	 *              +--------+--------+--------+--------+--------+
	 *               <---><----><---><----><----><---><----><--->
	 * output digit    0     1    2     3     4    5     6    7
	 *
	 *
	 * Because of possible padding, which must be done as if the input
	 * was 0, and because the fractional part is at the end, we'll
	 * start encoding from the end.  The encoding loop is unrolled for
	 * greater performance (using the infamous Duff's device to directly
	 * switch at the proper stage within the do {} while loop).
	 */

	switch (len % 5) {
	case 0:
		do {
			ASSERT(op - encbuf >= 8);
			i = *--ip;							/* Input #4 */
			*--op = b32_alphabet[i & 0x1f];		/* Ouput #7 */
			i >>= 5;							/* upper <234>, input #4 */
			/* FALLTHROUGH */
	case 4:
			i |= ((long) *--ip) << 3;		/* had 3 bits in `i' */
			*--op = b32_alphabet[i & 0x1f];		/* Output #6 */
			i >>= 5;							/* upper <401234>, input #3 */
			*--op = b32_alphabet[i & 0x1f];		/* Output #5 */
			i >>= 5;							/* upper <4>, input #3 */
			/* FALLTHROUGH */
	case 3:
			i |= ((long) *--ip) << 1;		/* had 1 bits in `i' */
			*--op = b32_alphabet[i & 0x1f];		/* Output #4 */
			i >>= 5;							/* upper <1234>, input #2 */
			/* FALLTHROUGH */
	case 2:
			i |= ((long) *--ip) << 4;		/* had 4 bits in `i' */
			*--op = b32_alphabet[i & 0x1f];		/* Output #3 */
			i >>= 5;							/* upper <3401234>, input #1 */
			*--op = b32_alphabet[i & 0x1f];		/* Output #2 */
			i >>= 5;							/* upper <34>, input #1 */
			/* FALLTHROUGH */
	case 1:
			i |= ((long) *--ip) << 2;		/* had 2 bits in `i' */
			*--op = b32_alphabet[i & 0x1f];		/* Output #1 */
			i >>= 5;							/* upper <01234>, input #0 */
			*--op = b32_alphabet[i & 0x1f];		/* Output #0 */
			i >>= 5;							/* Holds nothing, MBZ */
			ASSERT(i == 0);
			ASSERT(op >= encbuf);
		} while (op > encbuf);
	}
}

/*
 * base32_encode_into
 *
 * Encode `len' bytes from `buf' into `enclen' bytes starting from `encbuf'.
 * Caller must have ensured that there was enough room in encbuf.
 */
void base32_encode_into(const BYTE *buf, int len,
	char *encbuf, int enclen)
{
	int pad;
	int exactlen = encode_pad_length(len, &pad);

	ASSERT(enclen >= (exactlen + pad));

	base32_encode_exactly(buf, len, encbuf, exactlen);
	if (pad)
		memset(encbuf + exactlen, '=', pad);
}

/*
 * base32_encode
 *
 * Encode `len' bytes starting at `buf' into new allocated buffer.
 * No trailing padding chars are emitted.
 *
 * Returns the new encoded buffer, NUL-terminated, and the added amount
 * of padding chars in `retpad' if it is a non-NULL pointer.
 */
char *base32_encode(const BYTE *buf, int len, int *retpad)
{
	int pad;
	int enclen = encode_pad_length(len, &pad);
	char *encbuf = (char*) malloc(enclen + pad + 1);	/* Allow for trailing NUL */

	base32_encode_exactly(buf, len, encbuf, enclen);
	if (pad)
		memset(encbuf + enclen, '=', pad);
	encbuf[enclen + pad] = '\0';

	if (retpad)
		*retpad = pad;

	return encbuf;
}

/*
 * base32_decode_alphabet
 *
 * Decode `len' bytes from `buf' into `declen' bytes starting from `decbuf'.
 * Caller must have ensured that there was sufficient room in decbuf.
 * Uses the specified decoding alphabet.
 *
 * Return TRUE if successful, FALSE if the input was not valid base32.
 */
static bool base32_decode_alphabet(char valmap[256],
	const char *buf, int len, char *decbuf, int declen)
{
	long i = 0;					/* Input accumulator, 0 for trailing pad */
	char const *ip = buf + len;	/* Input pointer, one byte off end */
	int dlen = (len >> 3) * 5;		/* Exact decoded lenth */
	char *op;						/* Output pointer, one byte off end */
	char v;

	ASSERT(buf);
	ASSERT(decbuf);
	ASSERT(len > 0);
	ASSERT((len & 0x7) == 0);			/* `len' is a multiple of 8 bytes */
	ASSERT(declen >= dlen);

	/*
	 * If the last byte of input is '=', there is padding and we need to
	 * zero the tail of the decoding buffer.
	 */

	if (buf[len-1] == '=') {
		int pad = 0;
		int n = 0;							/* Amount of bytes to zero */

		/*
		 * Remove and count trailing input padding bytes.
		 */

		while (*--ip == '=')
			pad++;

		ip++;			/* Points one byte after real non-padding input */

		switch (pad) {
		case 1: n = 1; break;
		case 3: n = 2; break;
		case 4: n = 3; break;
		case 6: n = 4; break;
		default:
			return FALSE;			/* Cannot be valid base32 */
		}

		memset(decbuf + (dlen - n), 0, n);
		op = decbuf + (dlen - n);
	} else
		op = decbuf + dlen;

	/*
	 * In the following picture, we represent how the 8 bytes of input,
	 * each consisting of only 5 bits of information forming a base32 digit,
	 * are concatenated back into 5 bytes of binary information.
	 *
	 * input digit     0     1    2     3     4    5     6    7
	 *               <---><----><---><----><----><---><----><--->
	 *              +--------+--------+--------+--------+--------+
	 *              |01234012|34012340|12340123|40123401|23401234|
	 *              +--------+--------+--------+--------+--------+
	 * output byte      0        1        2        3        4
	 *
	 *
	 * Because of possible padding, which must be done as if the input
	 * was 0, and because the fractional part is at the end, we'll
	 * start decoding from the end.  The decoding loop is unrolled for
	 * greater performance (using the infamous Duff's device to directly
	 * switch at the proper stage within the do {} while loop).
	 */

	switch ((ip - buf) % 8) {
	case 0:
		do {
			i = valmap[*--ip];				/* Input #7 */
			if (i < 0) return FALSE;
			/* FALLTHROUGH */
	case 7:
			v = valmap[*--ip];				/* Input #6 */
			if (v < 0) return FALSE;
			i |= v << 5;					/* had 5 bits */
			*--op = i & 0xff;				/* Output #4 */
			i >>= 8;						/* lower <01> of output #3 */
			/* FALLTHROUGH */
	case 6:
			v = valmap[*--ip];				/* Input #5 */
			if (v < 0) return FALSE;
			i |= v << 2;					/* had 2 bits */
			/* FALLTHROUGH */
	case 5:
			v = valmap[*--ip];				/* Input #4 */
			if (v < 0) return FALSE;
			i |= v << 7;					/* had 7 bits */
			*--op = i & 0xff;				/* Output #3 */
			i >>= 8;						/* lower <0123> of output #2 */
			/* FALLTHROUGH */
	case 4:
			v = valmap[*--ip];				/* Input #3 */
			if (v < 0) return FALSE;
			i |= v << 4;					/* had 4 bits */
			*--op = i & 0xff;				/* Output #2 */
			i >>= 8;						/* lower <0> of output #1 */
			/* FALLTHROUGH */
	case 3:
			v = valmap[*--ip];				/* Input #2 */
			if (v < 0) return FALSE;
			i |= v << 1;					/* had 1 bit */
			/* FALLTHROUGH */
	case 2:
			v = valmap[*--ip];				/* Input #1 */
			if (v < 0) return FALSE;
			i |= v << 6;					/* had 6 bits */
			*--op = i & 0xff;				/* Output #1 */
			i >>= 8;						/* lower <012> of output #0 */
			/* FALLTHROUGH */
	case 1:
			v = valmap[*--ip];				/* Input #0 */
			if (v < 0) return FALSE;
			i |= v << 3;					/* had 3 bits */
			*--op = i & 0xff;				/* Output #0 */
			i >>= 8;						/* Holds nothing, MBZ */
			ASSERT(i == 0);
			ASSERT(op >= decbuf);
		} while (op > decbuf);
	}

	return TRUE;
}

/*
 * base32_decode_into
 *
 * Decode `len' bytes from `buf' into `declen' bytes starting from `decbuf'.
 * Caller must have ensured that there was sufficient room in decbuf.
 *
 * Return TRUE if successful, FALSE if the input was not valid base32.
 */
bool base32_decode_into(const char *buf, int len,
	char *decbuf, int declen)
{
	return base32_decode_alphabet(values, buf, len, decbuf, declen);
}

/*
 * base32_decode_old_into
 *
 * Decode `len' bytes from `buf' into `declen' bytes starting from `decbuf'.
 * Caller must have ensured that there was sufficient room in decbuf.
 * The "old" base32 alphabet is used for decoding.
 *
 * Return TRUE if successful, FALSE if the input was not valid base32.
 */
bool base32_decode_old_into(const char *buf, int len,
	char *decbuf, int declen)
{
	return base32_decode_alphabet(old_values, buf, len, decbuf, declen);
}

/*
 * base32_decode
 *
 * Decode `len' bytes starting at `buf' into new allocated buffer.
 *
 * Returns the new decoded buffer, or NULL if the input was not valid base32
 * encoding.  The caller knows the length of the returned buffer: it's the
 * size of the input divided by 8 and multiplied by 5.
 */
char *base32_decode(const char *buf, int len)
{
	int declen;
	char *decbuf;

	if (len & 0x7)					/* Padding bytes missing */
		return NULL;

	declen = (len >> 3) * 5;
	decbuf = (char*) malloc(declen);

	if (!base32_decode_into(buf, len, decbuf, declen)) {
		free(decbuf);
		return NULL;
	}

	return decbuf;
}




