/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 gnusearch.cpp  -  Representation of a search in progress

    begin                : Wed May 30 2001
    copyright            : (C) 2001 by 
    email                : maksik@gmx.co.uk
 ***************************************************************************/
 
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <time.h>

#include "mutella.h"
#include "structures.h"

#include "asyncsocket.h"
#include "packet.h"
#include "gnusearch.h"
#include "gnudirector.h"
#include "property.h"
#include "preferences.h"
#include "packet.h"
#include "conversions.h"
#include "common.h"

MGnuSearch::MGnuSearch(MGnuDirector* pControl, const CString& search, const CString& sha1, int type, int size/*=0*/, int sizeMode/*=LIMIT_NONE*/)
{
	m_pDirector = pControl;
	m_pPrefs = m_pDirector->GetPrefs();
	m_nType = type;
	m_MinSpeed = 0;

	if (sha1.length()==32)
		m_setSha1.insert(sha1);
	
	m_szSearch = NULL;

	m_Packet = new BYTE[256];
	ASSERT(m_Packet);
	m_nPacketLength = 0;

	m_SizeFilterMode   = sizeMode;
	m_SizeFilterValue  = size;

	m_SpeedFilterMode  = LIMIT_NONE;
	m_SpeedFilterValue = 0;
	m_nHits = 0;
	m_nGroups = 0;
	m_bUpdated = false;	
	m_bAutoget = false;
	
	m_dwID = 0;
	//
	SetSearchString(search, false);
}

MGnuSearch::~MGnuSearch()
{
	delete [] m_Packet;
	delete [] m_szSearch;
}

void MGnuSearch::SetSearchString(const CString& s, bool bLockMutex /*=true*/)
{
	if (bLockMutex)
		m_mutex.lock();
	m_Search   = StripWhite(s);
	if (m_szSearch)
		delete [] m_szSearch;
	m_szSearch = new char[m_Search.length()+1];
	ASSERT(m_szSearch);
	strcpy(m_szSearch, m_Search.c_str());
	MakeLower(m_szSearch);
	m_PlusWords.clear();
	m_MinusWords.clear();
	MakeWordList(m_szSearch, m_PlusWords, m_MinusWords);
	if (bLockMutex)
		m_mutex.unlock();
}

void MGnuSearch::Clear()
{
	MLock lock(m_mutex);
	m_nHits = 0;
	m_nGroups = 0;
	m_listGroups.clear();
	m_mapResults.clear();
}

void MGnuSearch::SendQuery()
{
	// Send query through network
	// but dont send queries smaller than 4 chars
	CString sQuery;
	int nWords = m_PlusWords.size();
	for (int i = 0; i<nWords; ++i)
	{
		sQuery += m_PlusWords[i];
		if (i+1<nWords)
			sQuery += " ";
	}
	if (sQuery.length() >= 4)
	{
		SendSpecificQuery(sQuery, "");
	}
	m_mutex.lock();
	for (set<CString>::iterator it = m_setSha1.begin(); it != m_setSha1.end(); ++it)
		SendSpecificQuery("", *it);
	m_mutex.unlock();
}

void MGnuSearch::SendSpecificQuery(const CString& sText, const CString& sSha1)
{
	int nSearchTextLen = 0;
	m_nPacketLength = 25;
	if (!sText.empty())
	{
		nSearchTextLen = min(sText.length(), 256-25-1-5); // -packet_header -trailing0 -urn:0
		if (sSha1.length()==32)
			nSearchTextLen -= 32 + 5; // +"sha1:"
		memcpy(m_Packet + m_nPacketLength, sText.c_str(), nSearchTextLen);
		m_nPacketLength += nSearchTextLen;
	}
	m_Packet[m_nPacketLength] = '\0';
	++m_nPacketLength;
	if (sSha1.length()==32)
	{
		memcpy(m_Packet + m_nPacketLength, "urn:sha1:", 9);
		m_nPacketLength += 9;
		ASSERT(m_nPacketLength <=256 );
		memcpy(m_Packet + m_nPacketLength, sSha1.c_str(), 33);
		m_nPacketLength += 33;
	}
	else
	{
		memcpy(m_Packet + m_nPacketLength, "urn:", 5);
		m_nPacketLength += 5;
	}
	ASSERT(m_nPacketLength <=256 );
	//
	m_pDirector->Broadcast_LocalQuery(m_Packet, m_MinSpeed, m_nPacketLength);
}

bool MGnuSearch::IsFull()
{
	MLock lock(m_mutex);
	return m_mapResults.size() >= m_pPrefs->m_nMaxPerSearchResults;
}

bool MGnuSearch::UpdateExisting(const Result & result)
{
	m_mutex.lock();
	for (ResultMap::iterator it = m_mapResults.begin(); it != m_mapResults.end(); ++it)
	{
		if (result.Port        == it->second.Port        &&
		    result.FileIndex   == it->second.FileIndex   &&
		    result.Host.S_addr == it->second.Host.S_addr &&
		    result.NameLower   == it->second.NameLower   )
		{
			// we have a hit, but we are not going to add it again
			// we not only update its change time
			it->second.ChangeTime = xtime();
			// but also a GUID and Origin
			it->second.PushID   = result.PushID;
			it->second.OriginID = result.OriginID;
			m_bUpdated = true;
			m_mutex.unlock();
			return true;
		}
	}
	m_mutex.unlock();
	return false;
}

bool MGnuSearch::CheckAgainstResult(Result & result)
{
	// first check the size filter: it cheap and does not get changed by user
	// than check if we have found this result already. It's not cheap
	// and then continue with casual checking
	// TODO: optimize what we really should do first
	//
	// Size Filter
	if(m_SizeFilterMode != LIMIT_NONE)
	{
		if(!CheckLimit(m_SizeFilterMode, m_SizeFilterValue, result.Size))
			return false;
	}
	// if the search is full
	if (m_nHits >= m_pPrefs->m_nMaxPerSearchResults)
	{
		// the search is full but we might still update the results
		UpdateExisting(result);
		return false;
	}
	// Busy Filter
	if(m_pPrefs->m_bScreenBusy)
		if(result.Busy)
			return false;
	// Speed Filter
	if(m_SpeedFilterMode != LIMIT_NONE)
		if(!CheckLimit(m_SpeedFilterMode, m_SpeedFilterValue, result.Speed / 8))
			return false;
	// Word Filter
	if (m_pPrefs->m_bWordFilterActive)
	{
		set<CString> BadWords;
		m_pPrefs->m_setWordFilter.CopyTo(BadWords); // TODO: make it more efficient to eliminate copying of the set
		// POSSIBLE BUG: we assume that BadWords are all lower case
		for (set<CString>::iterator itWord = BadWords.begin(); itWord != BadWords.end(); ++itWord)
			if(result.NameLower.find(*itWord) != -1)
				return false;
	}
	
	// Sha1 Check:
	// important: we consider sha1 as a way to broaden the search, rather to narrow it!
	m_mutex.lock();
	bool bSha1Match = false;
	if (result.Sha1.length()==32)
		bSha1Match = (m_setSha1.find(result.Sha1) != m_setSha1.end());
	m_mutex.unlock();
	// Name Check
	// match against exclusive word list than against incluseive word list
	if (!bSha1Match)
	{
		if (MatchWordList(result.NameLower, m_MinusWords, false) ||
			!MatchWordList(result.NameLower, m_PlusWords)        )
			return false;
	}
	// now we have a definete hit
	// if it's an old hit -- just update the respective result
	if (UpdateExisting(result))
		return true;
	// nope, it's new -- add it
	m_mutex.lock();
	m_nHits++;
	result.dwID = GenID();
	result.ChangeTime = xtime();
	ASSERT(m_mapResults.end()==m_mapResults.find(result.dwID));
	m_mapResults[result.dwID]=(result);
	m_mutex.unlock();
	AddtoGroup(result);
	m_bUpdated = true;
	return true;
}

void MGnuSearch::AddtoGroup(const Result& result)
{
	ASSERT(result.dwID);
	MLock lock(m_mutex);
	MGnuPreferences* pPrefs = m_pPrefs;

	bool NewGroup = true;
	ResultGroup* ReturnGroup = NULL;
	//
	set<CString> setAutoWords;
	if (pPrefs->m_bGroupbyFuzzyRules)
	{
		CString sAutoSearch = MakeSearchOfFilename(result.NameLower);
		char * szAutoSearch = (char*) alloca(sAutoSearch.length()+1);
		strcpy(szAutoSearch,sAutoSearch.c_str());
	    MakeWordList(szAutoSearch, setAutoWords);
	}
	// Check if item is part of any group
	GroupList::iterator itGroup;
	for(itGroup = m_listGroups.begin(); itGroup != m_listGroups.end(); itGroup++)
		if(itGroup->Size == result.Size)
		{
			if( (pPrefs->m_bGroupbyFuzzyRules && itGroup->MatchWords == setAutoWords) ||
				itGroup->NameLower == result.NameLower ||
				(result.Sha1.length()==32 && itGroup->m_setSha1.find(result.Sha1) != itGroup->m_setSha1.end()))
			{
				ASSERT(itGroup->ResultSet.end() == itGroup->ResultSet.find(result.dwID));
				if (result.Sha1.length()==32)
					itGroup->m_setSha1.insert(result.Sha1);
				itGroup->ResultSet.insert(result.dwID);
				NewGroup = false;
				break;
			}
		}  
	// If not create a new group
	if(NewGroup)
	{
		ResultGroup InsertGroup;

		InsertGroup.Name		= result.Name;
		InsertGroup.NameLower	= result.NameLower;
		InsertGroup.MatchWords 	= setAutoWords;
		InsertGroup.Size		= result.Size;
		InsertGroup.AvgSpeed	= result.Speed;
		InsertGroup.dwID = GenID();
		InsertGroup.ResultSet.insert(result.dwID);
		if (result.Sha1.length()==32)
			InsertGroup.m_setSha1.insert(result.Sha1);

		m_listGroups.push_back(InsertGroup);
		//
		m_nGroups = m_listGroups.size();
	}
	else
	{
		// may be rename the group to mininum string length
		if ((*itGroup).Name.length()>result.Name.length())
		{
			(*itGroup).Name      = result.Name;
			(*itGroup).NameLower = result.NameLower;
		}
		// calc. avg. speed
		int Hosts = 0, SpeedSum = 0;
		set<DWORD>::iterator itResult;
		for(itResult = (*itGroup).ResultSet.begin(); itResult != (*itGroup).ResultSet.end(); itResult++)
		{
			SpeedSum += m_mapResults[(*itResult)].Speed;
			Hosts++;
		}
		if(Hosts)
			(*itGroup).AvgSpeed = SpeedSum / Hosts;
	}
}

bool MGnuSearch::CheckLimit(int Limit, DWORD Value, DWORD Compare)
{
	switch (Limit)
	{
		case LIMIT_MORE:    return (Compare >= Value);
		case LIMIT_EXACTLY: return (Compare == Value);
		case LIMIT_LESS:    return (Compare <= Value);
		case LIMIT_APPROX:  return (Compare >= 0.9 * Value && Compare <= 1.1 * Value); // +- 10%
	}
	return true; // unknown Limit parameter: be tolerant
}





