/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 structures.h  -  Universal structures used by gnutella backend.

    begin                : Mon Jan 28 2002
    copyright            : (C) 2002 by Max Zaitsev
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifndef __STRUCTURES_H_INCLUDED__
#define __STRUCTURES_H_INCLUDED__

#include "basicstruct.h"

enum SocketStatus {
	SOCK_UNDEFINED,
	SOCK_CONNECTING,
	SOCK_NEGOTIATING,
	SOCK_CONNECTED,
	SOCK_CLOSED
};

class MGnuNode;

struct Node
{
	IP   Host;
	UINT Port;

	UINT Ping;
	UINT Speed;
	BYTE Distance;

	DWORD ShareSize;
	DWORD ShareCount;

	const Node& operator=(const Node& orig){
		Host = orig.Host;
		Port = orig.Port;
		Ping = orig.Ping;
		Speed = orig.Speed;
		Distance = orig.Distance;
		ShareSize = orig.ShareSize;
		ShareCount = orig.ShareCount;
		//Friends = orig.Friends;
		return *this;
	}
	Node() :
		Port(0),
		Ping(0),
		Speed(0),
		Distance(0),
		ShareSize(0),
		ShareCount(0)
	{
		Host.S_addr = 0;
	}
	Node(const Node& orig) :
		Host(orig.Host),
		Port(orig.Port),
		Ping(orig.Ping),
		Speed(orig.Speed),
		Distance(orig.Distance),
		ShareSize(orig.ShareSize),
		ShareCount(orig.ShareCount)//,
		//Friends(orig.Friends)
	{}
};

enum ClientMode {
	CM_NORMAL,     // Normal 0.6 or 0.4 client
	CM_LEAF,       // Leaf
	CM_ULTRAPEER,  // Ultrapeer
	CM_ULTRALEAF   // Ultrapeer-capable peer, but currently in leaf mode
};

enum NodeRates {
	NR_PACKET_IN,
	NR_PACKET_OUT,
	NR_BYTE_IN,
	NR_BYTE_OUT,
	NR_QREPLY_IN,
	NR_DROP_OUT,
	NR_LAST
};

struct SGnuNode {
	int     m_nStatus;
	CString m_sHost; /* MZ: is it really needed here ? */
	CString m_sRemoteClient;
	IP      m_ipHost;
	u_int   m_nPort;
	bool    m_bInbound;
	long    m_nUptime;
	long    m_nPeerUptime;
	DWORD   m_dwVersion;
	long    m_nPeerMode;

	DWORD   m_dwFriendsTotal;
	long long m_llLibraryTotal;
	DWORD	m_dwFilesTotal;
	DWORD	m_dwSharingTotal;

	DWORD   m_dwFriends[MAX_TTL_SELF];
	DWORD   m_dwLibrary[MAX_TTL_SELF];
	DWORD   m_dwFiles[MAX_TTL_SELF];

	// Bandwidth, [0] is Received, [1] is Sent, [2] is Dropped
	//u_long  m_nTotalPackets[3];     // Packets received/sent
	//long long m_nllTotalBytes[3];   // Bytes received/sent
	//double m_dPacketRate[3];        // receive/send rate in packets/sec
	//double m_dByteRate[3];          // receive/send rate in bytes/sec
	//double m_dReplyReceiveRate;     // number of reply packets per minute
	long long m_llStatisticsTotal[NR_LAST];
	double    m_dStatisticsRate  [NR_LAST];

	long m_nEfficiency;
	
	u_long m_nSendQueueSize;
	u_long m_nAvgDroppedPackets;
	u_long m_nDroppedPackets;
	u_long m_nBadPackets;
	u_long m_nRoutingErrors;
	u_long m_nLoopBackErrors;
	u_long m_nDuplicatePackets;
	//
	DWORD m_dwID;
};

// transfer status sorted preferably by the measure of success
enum TransferStatus {
	TRANSFER_COMPLETED,
	TRANSFER_SEGM_COMPLETED,
	TRANSFER_RECEIVING,
	TRANSFER_SENDING,
	TRANSFER_NEGOTIATING,
	TRANSFER_CONNECTED,
	TRANSFER_PUSH_CONNECTED,  // used by ulpoad only
	TRANSFER_PUSH_CONNECTING, // used by ulpoad only
	TRANSFER_CONNECTING,
	TRANSFER_PUSH,
	TRANSFER_QUEUED,
	TRANSFER_NEW,
	TRANSFER_COOLDOWN,
	TRANSFER_FAILED,
	TRANSFER_OPEN,           // used by MGnuDownload only
	TRANSFER_OPENING,        // used by MGnuDownload only
	TRANSFER_INTERNAL_ERROR,
	TRANSFER_FILE_ERROR,	
	TRANSFER_CLOSING,        // used by MGnuDownload only
	TRANSFER_CLOSED
};

enum DisconnectReason {
	REASON_UNDEFINED,
	REASON_EMPTY,
	REASON_STOPPED,
	REASON_NO_DATA,
	REASON_SOCKET_ERROR,
	REASON_NO_RESUME_SUPPORT,
	REASON_WRONG_FILE_SIZE,
	REASON_FILE_NOT_FOUND,
	REASON_SERVER_BUSY,
	REASON_UNKNOWN_SERVER_ERROR,
	REASON_WRONG_STATE,
	REASON_CONNECTION_ERROR,
	REASON_REMOTELY_CANCELED,
	REASON_FAILED_TO_OPEN_FILE,
	REASON_PARTIAL_FILESIZE_DIFFERENT,
	REASON_HOST_OCCUPIED,
	REASON_FILES_ARE_DIFFERENT,
	REASON_WRITE_ERROR,
	REASON_FILE_ERROR,
	REASON_COMPLETED,
	REASON_CORRUPT,
	REASON_NO_RESPONSE,
	REASON_BELOW_MINIMUM_SPEED,
	REASON_INTERNAL,
	REASON_BAD_HEADER,
	REASON_STORAGE_DENIED,
	REASON_TRANSFER_FAILED
};

struct Result
{
	// File info
	CString   Name;      // ???
	CString   NameLower; // ???
	DWORD     FileIndex;
	DWORD     Size;
	// extra info
	CString   Extra;     // ???
	CString   Sha1;
	
	// Node info
	IP		  Host;
	WORD      Port;
	int       Speed;
	CString	  Vendor;    // ???

	// Flags
	bool Firewall;
	bool OpenSlots;
	bool Busy;
	bool Stable;
	bool ActualSpeed;


	// Push info
	DWORD     OriginID;
	GUID      PushID;
	int		  Distance;

	// Download info
	DWORD     ChangeTime;
	// Numerical ID
	DWORD     dwID;
	// consructor
	Result(){
		dwID = 0;
		ChangeTime = 0;//ChangeTime = xtime();
	}
};
typedef vector<Result> ResultVec;

struct ResultGroup
{
	CString  Name;           // Name of the group
	CString  NameLower;      // Lower-case name
	set<CString> MatchWords; // auto-generated search for fuzzy matching
	set<CString> m_setSha1;  // all unique hashes
	DWORD    Size;
	DWORD    AvgSpeed;

	set<DWORD> ResultSet;  // There is a picularity with this set. Internally it contains
	                       // IDs of the results. When exported these ID's are replaced by
	                       // indeces in the accompanying result array

	DWORD    dwID;

	ResultGroup() : Size(0), AvgSpeed(0), dwID(0) {}
	/*ResultGroup(const ResultGroup& orig) :
		Name(orig.Name),
		NameLower(orig.NameLower),
		MatchWords(orig.MatchWords),
		Size(orig.Size),
		AvgSpeed(orig.AvgSpeed),
		ResultList(orig.ResultList),
		dwID(orig.dwID)
	{}
	const ResultGroup& operator=(const ResultGroup& orig){
		Name = orig.Name;
		NameLower = orig.NameLower;
		MatchWords = orig.MatchWords;
		Size = orig.Size;
		AvgSpeed = orig.AvgSpeed;
		ResultList = orig.ResultList;
		dwID = orig.dwID;
		return *this;
	}*/
};
typedef vector<ResultGroup> ResultGroupVec;

struct SGnuDownloadItem
{
	// Node info
	IP		  ipHost;
	WORD      nPort;
	CString	  sVendor;
	// Download info
	int       nStatus;
	int       nLastStatusChange;
	int       nLastActive;
	int       nDisconnectReason;
	DWORD     dwRate;
	//
	bool      bServerIsUp;   // true if we recieved anything from it ever
	int       nConErrors;    // counter of consequitive connection errors
	int       nPushTimeouts; // counter of consequitive push timeouts
};
typedef vector<SGnuDownloadItem> DownloadItemVec;

struct SGnuDownload
{
	// Download Properties
	CString m_sName; // ???
	DWORD   m_dwSearchID;
	// File info
	DWORD   m_dwFileLength;
	DWORD   m_dwBytesCompleted;
	// source addresses, etc
	double m_dRate;          // real download rate in bytes per second
	//
	int    m_nActiveConnections;
	int    m_nCreationTime;
	// status
	int    m_nLastDisconnectReason;
	int    m_nStatus;
	// ID for UI
	DWORD  m_dwID;
	// Download items -- individual hosts
	DownloadItemVec m_vecItems; //???
	int    m_nBadResults;   // size of the "bad results" array
	// few static methods
	static LPCSTR GetErrorString(int nCode);  // ???
	static LPCSTR GetStatusString(int nStatus); // ???
};

struct SGnuUpload
{
	CString m_sFileName; // ???
	int     m_nFileIndex;
	int     m_nFileLength;
	int     m_nBytesCompleted;
	int     m_nChangeTime;
	int     m_nStatus;
	int     m_nError;
	// remote peer info
	IP      m_ipHost;
	UINT    m_nPort;
	CString m_sUserAgent;
	// bandwidth
	double m_dRate;
	// numeric ID
	DWORD  m_dwID;
	// few static methods
	static LPCSTR GetErrorString(int nError);
};

enum SearchType{
	ST_UNKNOWN     = 0,
	ST_USER        = 1, // added by user
	ST_ALTERNATIVE = 2, // added by download
	ST_PARTIAL     = 3  // added on mutella start
};

enum SearchLimitType{
	LIMIT_NONE,
	LIMIT_LESS,
	LIMIT_EXACTLY,
	LIMIT_MORE,
	LIMIT_APPROX
};

struct SGnuSearch
{
	CString m_Search; // ???
	// sha1 support: we'll enable multiple hashes
	set <CString> m_setSha1;
	int     m_nType; // see enum SearchType
	// autoget support
	bool    m_bAutoget;
	CString m_Filename; // ??? if we really need it here ?
	// filters
	DWORD   m_MinSpeed; // ??? what is it ?
	int     m_SpeedFilterMode;
	int     m_SpeedFilterValue;
	// size filters
	int     m_SizeFilterMode;
	int     m_SizeFilterValue;
	// numeric ID
	DWORD m_dwID;
	// results
	int m_nHits;
	int m_nGroups;
	// member finctions
	bool IsAutomatic(){return m_nType==ST_ALTERNATIVE || m_nType==ST_PARTIAL;} // ??? what is it doing here ?
};

struct SharedFile
{
	CString Name;      //
	CString NameLower; //
	CString Path;      //
	CString Sha1Hash;

	DWORD Size;
	DWORD Matches;
	DWORD Uploads;
	bool  Hidden;
	
	SharedFile() : Size(0), Matches(0), Uploads(0), Hidden(false) {}
};

struct SharedDirectory
{
	CString 	Name;      //
	bool		Recursive;
	long long	Size;
	DWORD		FileCount;
	//
	SharedDirectory() : Recursive(false), Size(0), FileCount(0) {}
	SharedDirectory(const SharedDirectory& orig) : Name(orig.Name), Recursive(orig.Recursive), Size(orig.Size), FileCount(orig.FileCount) {}
	const SharedDirectory& operator=(const SharedDirectory& orig){
		Name = orig.Name;
		Recursive = orig.Recursive;
		Size = orig.Size;
		FileCount = orig.FileCount;
		return *this;
	}
};

#define MAX_QUERY_LEN 255
struct QueryComp
{
	//CString
	char		Text[MAX_QUERY_LEN+1];
	//vector<CString> Extended;
	CString		ExtendedPart;
	GUID		QueryGuid;
	DWORD		OriginID;
	int			nHops;
};

// it is no good place for it but I could not find better one
///////////////////////////////////////////////////////////////////////////
// callbacks for various "for-each" functions in MController & MGnuDirector
typedef bool (*tEachSearch)(void* pData, SGnuSearch* pSearch);
typedef bool (*tEachConnection)(void* pData, SGnuNode* pNode);
typedef bool (*tEachUpload)(void* pData, SGnuUpload* pUpload);
typedef bool (*tEachDownload)(void* pData, SGnuDownload* pDownload);

#endif //__STRUCTURES_H_INCLUDED__

