#import "MAHealth.h"
#import "MAHealthGraph.h"
#import <MTextImageCell.h>
#import <MMySQLDispatcher.h>

#include <sys/time.h>
#include <sys/select.h>

#include "myx_admin_public_interface.h"

typedef struct 
{
  MYX_HEALTH_PAGE *page;
  NSTextField *description;
  NSView *box;
  NSMutableArray *groups;
} GraphPage;

typedef struct 
{
  GraphPage *page;
  MYX_HEALTH_GROUP *group;
  NSBox *frame;
  NSMutableArray *graphs;
} GraphGroup;



@interface Item : NSObject
{
  @public
  void *ptr;
  int type;
  NSMutableArray *children;
}
+ (Item*)itemWithPtr:(void*)ptr type:(int)type;
@end
@implementation Item

+ (Item*)itemWithPtr:(void*)aptr type:(int)atype
{
  Item *item= [[[Item alloc] init] autorelease];
  item->ptr= aptr;
  item->type= atype;
  return item;
}

- (void)dealloc
{
  [children release];
  [super dealloc];
}
@end

@interface ListingDataSource : NSObject
{
  MYX_TRANS *_trans;
  MYX_VARIABLES_LISTING *_listing;
  
  NSMutableArray *_items;
}
- initWithListing:(MYX_VARIABLES_LISTING*)listing translation:(MYX_TRANS*)trans;

- (id)outlineView:(NSOutlineView *)outlineView 
            child:(int)index 
           ofItem:(id)item;

- (BOOL)outlineView:(NSOutlineView *)outlineView 
   isItemExpandable:(id)item;

- (int)outlineView:(NSOutlineView *)outlineView 
numberOfChildrenOfItem:(id)item;

- (id)outlineView:(NSOutlineView *)outlineView 
objectValueForTableColumn:(NSTableColumn *)tableColumn 
           byItem:(id)item;

@end

@implementation ListingDataSource
- initWithListing:(MYX_VARIABLES_LISTING*)listing translation:(MYX_TRANS*)trans
{
  self= [super init];
  if (self)
  {
    unsigned int i, j;
    _listing= listing;
    _trans= trans;
    
    _items= [[NSMutableArray alloc] initWithCapacity:_listing->groups_num];
    for (i= 0; i < _listing->groups_num; i++)
    {
      Item *item= [Item itemWithPtr:_listing->groups+i type:'G'];
      [_items addObject:item];
      
      item->children= [[NSMutableArray alloc] init];
      for (j= 0; j < _listing->groups[i].subgroups_num; j++)
      {
        Item *sitem= [Item itemWithPtr:_listing->groups[i].subgroups+j type:'S'];
        sitem->children= [[NSMutableArray alloc] init];
        [item->children addObject:sitem];
      }
    }
  }
  return self;
}

- (void)dealloc
{
  [_items release];
  [super dealloc];
}

- (id)outlineView:(NSOutlineView *)outlineView 
            child:(int)index 
           ofItem:(id)item
{
  if (item == nil)
    return [_items objectAtIndex:index];
  else
  {
    Item *it= (Item*)item;
    return [it->children objectAtIndex:index];
  }
}

- (BOOL)outlineView:(NSOutlineView *)outlineView 
   isItemExpandable:(id)item
{
  if (item == nil)
    return YES;
  else if (((Item*)item)->type=='G')
    return YES;
  else
    return NO;
}

- (int)outlineView:(NSOutlineView *)outlineView 
numberOfChildrenOfItem:(id)item
{
  if (item == nil)
    return [_items count];
  else if (((Item*)item)->type=='G')
    return [((Item*)item)->children count];
  else
    return 0;
}

- (id)outlineView:(NSOutlineView *)outlineView 
objectValueForTableColumn:(NSTableColumn *)tableColumn 
           byItem:(id)item
{
  Item *it= (Item*)item;

  switch (it->type)
  {
    case 'G':
      return [NSString stringWithUTF8String:((MYX_VARIABLES_GROUP*)it->ptr)->name];
    case 'S':
      return [NSString stringWithUTF8String:((MYX_VARIABLES_SUBGROUP*)it->ptr)->name];
    default:
      return @"?";
  }
}

@end


@interface VariableDataSource : NSObject
{
  NSMutableArray *_items;
  MYX_VARIABLES *_values;
  MYX_TRANS *_trans;
  
  MAHealth *_owner;
}
- initWithTranslations:(MYX_TRANS*)trans owner:(MAHealth*)owner;
- (void)setFromSubgroup:(MYX_VARIABLES_SUBGROUP*)sg;
- (void)setFromGroup:(MYX_VARIABLES_GROUP*)g;
- (NSArray*)items;
- (MYX_VARIABLES*)values;

- (int)numberOfRowsInTableView:(NSTableView *)aTableView;
- (id)tableView:(NSTableView *)aTableView 
objectValueForTableColumn:(NSTableColumn *)aTableColumn
            row:(int)rowIndex;
- (void)tableView:(NSTableView *)aTableView
   setObjectValue:(id)anObject
   forTableColumn:(NSTableColumn *)aTableColumn
              row:(int)rowIndex;
@end

@implementation VariableDataSource
- initWithTranslations:(MYX_TRANS*)trans owner:(MAHealth*)owner
{
  self= [super init];
  if (self)
  {
    _trans= trans;
    _owner= owner;
  }
  return self;
}

- (void)dealloc
{
  [_items release];
  [super dealloc];
}

- (void)setFromSubgroup:(MYX_VARIABLES_SUBGROUP*)sg
{
  unsigned int j;
  [_items release];
  _items= [[NSMutableArray alloc] initWithCapacity:sg->variables_num];
  for (j= 0; j < sg->variables_num; j++)
  {
    Item *vitem= [Item itemWithPtr:sg->variables+j type:'V'];
    [_items addObject:vitem];
  }
}

- (void)setFromGroup:(MYX_VARIABLES_GROUP*)g
{
  unsigned int j;
  [_items release];
  _items= [[NSMutableArray alloc] initWithCapacity:g->variables_num];
  for (j= 0; j < g->variables_num; j++)
  {
    Item *vitem= [Item itemWithPtr:g->variables+j type:'V'];
    [_items addObject:vitem];
  }  
}

- (NSArray*)items
{
  return _items;
}

- (MYX_VARIABLES*)values
{
  return _values;
}

- (void)setValues:(MYX_VARIABLES*)values
{
  _values= values;
}

- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
  return [_items count];
}

- (id)tableView:(NSTableView *)aTableView 
objectValueForTableColumn:(NSTableColumn *)aTableColumn
            row:(int)rowIndex
{
  Item *it= (Item*)[_items objectAtIndex:rowIndex];
  MYX_VARIABLE_ELEMENT *var= (MYX_VARIABLE_ELEMENT*)it->ptr;
  
  if ([[aTableColumn identifier] isEqualToString:@"variable"])
  {
    return [NSString stringWithUTF8String:var->mysql_id];
  }
  else if ([[aTableColumn identifier] isEqualToString:@"value"])
  {
    unsigned int i;
    if (_values)
    {
      for (i= 0; i < _values->variables_num; i++)
      {
        if (strcmp(_values->variables[i].name, var->mysql_id)==0)
          return [NSString stringWithUTF8String:_values->variables[i].value?:""];
      }
    }
    return @"";
  }
  else
  {    
    const char *v= var->desc_id;
    v= myx_t(_trans, "MySQLVariables", v, v);
    return [NSString stringWithUTF8String:v];
  }
}

- (void)tableView:(NSTableView *)aTableView
   setObjectValue:(id)anObject
   forTableColumn:(NSTableColumn *)aTableColumn
              row:(int)rowIndex
{
  Item *it= (Item*)[_items objectAtIndex:rowIndex];
  MYX_VARIABLE_ELEMENT *var= (MYX_VARIABLE_ELEMENT*)it->ptr;
  unsigned int i;

  // need to set this here too, even if it'll be reloaded
  // later, because otherwise when you finish editing with Return,
  // it will display the old value and somehow we'll get another
  // message telling our value changed to the old value.
  if (_values)
  {
    for (i= 0; i < _values->variables_num; i++)
    {
      if (strcmp(_values->variables[i].name, var->mysql_id)==0)
      {
        g_free(_values->variables[i].value);
        _values->variables[i].value= g_strdup([anObject UTF8String]);
        break;
      }
    }
  }
  
  [_owner setValue:anObject forVariable:var->mysql_id];
}
@end



@interface MAHealth(Private)
- (void)buildPage:(MYX_HEALTH_PAGE*)page;
- (MAHealthGraph*)buildGraph:(MYX_HEALTH_GRAPH*)graph;
- (void)updateThread:(id)arg;
- (void)update:(id)arg;
- (void)outlineViewSelectionDidChange:(NSNotification *)notification;
- (BOOL)tableView:(NSTableView *)aTableView
 shouldEditTableColumn:(NSTableColumn *)aTableColumn
              row:(int)rowIndex;
- (void)tableView:(NSTableView *)aTableView 
  willDisplayCell:(id)aCell
   forTableColumn:(NSTableColumn *)aTableColumn 
              row:(int)rowIndex;
- (void)setValue:(NSString*)value forVariable:(const char*)var;
- (void)valueSaved:(id)argument result:(void*)result;
@end

@implementation MAHealth(Private)

- (void)updateThread:(id)arg
{
  MYX_VARIABLES *vars;
  MYX_USER_CONNECTION *conn= (MYX_USER_CONNECTION*)[arg pointerValue];
  MYSQL *mysql;
  struct timeval tv1, tv2;
  
  mysql= myx_mysql_init();

  if (myx_connect_to_instance(conn,mysql) < 0)
  {
    NSLog(@"Could not connect to MySQL instance");
    myx_mysql_close(mysql);
    return;
  }
  
  [self retain];
  
  while (!_cancelThread)
  {
    struct timeval elapsed, wait;
    id value;
    
    gettimeofday(&tv1, NULL);

    vars= myx_get_status_variables(mysql);
    if (!vars)
    {
      NSLog(@"error fetching status variables");
      break;
    }

    value= [[NSValue alloc] initWithBytes:&vars objCType:@encode(void *)];
    [self performSelectorOnMainThread:@selector(update:) 
                           withObject:value
                        waitUntilDone:YES];
    [value release];
    gettimeofday(&tv2, NULL);
    
    // wait to complete 1s
    elapsed.tv_usec= (tv2.tv_usec - tv1.tv_usec) % 1000000;
    elapsed.tv_sec= (tv2.tv_sec - tv1.tv_sec) + (tv2.tv_usec - tv1.tv_usec)/1000000;
    if (elapsed.tv_sec < 1)
    {
      wait.tv_usec= 1000000 - elapsed.tv_usec;
      wait.tv_sec= 0;
      select(0, NULL, NULL, NULL, &wait);
    }
  }
  myx_mysql_close(mysql);
  
  [self release];
}


- (void)update:(id)arg
{
  MYX_VARIABLES *vars= [arg pointerValue];
  int i, c= [_graphs count];

  for (i= 0; i < c; i++)
  {
    MAHealthGraph *graph= [_graphs objectAtIndex:i];
    [graph updateValues:vars];
  }
}


- (MAHealthGraph*)buildGraph:(MYX_HEALTH_GRAPH*)graph
{
  switch (graph->graphtype)
  {
    case MYX_LINE_GRAPH:
    {
      MALineHealthGraph *hg;
      hg= [[MALineHealthGraph alloc] initWithGraph:graph];
      [hg setMeterImage:[self loadImage:@"health_level_fg"]];
      return hg;
    }

    case MYX_BAR_GRAPH:
      return nil;
  }
  return nil;
}

- (void)buildPage:(MYX_HEALTH_PAGE*)page
{
  NSTabViewItem *item= [[NSTabViewItem alloc] initWithIdentifier:[NSValue valueWithPointer:page]];
  unsigned int i;
  NSView *view;
  float y;
  
  [tabView insertTabViewItem:item atIndex:0];
  [tabView selectTabViewItem:item];
    
  [item setLabel:page->caption?[NSString stringWithUTF8String:page->caption]:@""];
  [item release];
  view= [item view];
  
  y= [view frame].size.height - 10.0;
  for (i= 0; i < page->groups_num; i++)
  {
    MYX_HEALTH_GROUP *group= page->groups+i;
    NSBox *b= [[NSBox alloc] init];
    NSView *gbox= [b contentView];
    int j;
    float yy;
    
    [b setBoxType:NSBoxPrimary];
    [b setTitle:group->caption?[NSString stringWithUTF8String:group->caption]:@""];
    [b setContentView:gbox];
    
    yy= 0.0;
    for (j= (int)group->graphs_num-1; j >= 0; j--)
    {
      MAHealthGraph *graph= [self buildGraph:group->graphs+j];
      if (graph)
      {
        NSRect grect;
        
        [[graph view] setAutoresizingMask:NSViewWidthSizable|NSViewMaxYMargin];
        [_graphs addObject:graph];
        
        grect= [[graph view] frame];
        [[graph view] setFrameOrigin:NSMakePoint(10.0, yy+10.0)];
        
        [gbox setFrameSize:NSMakeSize(grect.size.width+20.0, grect.size.height+[gbox frame].size.height+30.0)];
        
        [gbox addSubview:[graph view]];
        yy+= grect.size.height+ 20;
      }
    }    
    [b setAutoresizingMask:NSViewWidthSizable|NSViewMinYMargin];
    y-= (yy+25);
    [b setFrame:NSMakeRect(17.0, y, [view frame].size.width-17.0*2, yy+25)];
    [view addSubview:b];
  }
}

- (void)outlineViewSelectionDidChange:(NSNotification *)notification
{
  id sender= [notification object];
  Item *item= (Item*)[sender itemAtRow:[sender selectedRow]];
  if (sender == statusIndex)
  {
    if (item->type == 'S')
      [[statusTable dataSource] setFromSubgroup:item->ptr];
    else
      [[statusTable dataSource] setFromGroup:item->ptr];
    [statusTable reloadData];
  }
  else
  {
    if (item->type == 'S')
      [[serverTable dataSource] setFromSubgroup:item->ptr];
    else
      [[serverTable dataSource] setFromGroup:item->ptr];    
    [serverTable reloadData];
  }
}


- (BOOL)tableView:(NSTableView *)aTableView 
shouldEditTableColumn:(NSTableColumn *)aTableColumn
              row:(int)rowIndex
{
  if ([(VariableDataSource*)[aTableView dataSource] values])
  {
    Item *it= (Item*)[[[aTableView dataSource] items] objectAtIndex:rowIndex];
    MYX_VARIABLE_ELEMENT *var= (MYX_VARIABLE_ELEMENT*)it->ptr;
    return var->editable ? YES : NO;
  }
  return NO;
}


- (void)tableView:(NSTableView *)aTableView 
  willDisplayCell:(id)aCell
   forTableColumn:(NSTableColumn *)aTableColumn 
              row:(int)rowIndex
{
  if ([[aTableColumn identifier] isEqualToString:@"variable"])
  {
    if ([self tableView:aTableView shouldEditTableColumn:aTableColumn row:rowIndex])
      [aCell setImage:_editableIcon];
    else
      [aCell setImage:_nonEditableIcon];
  }
}

- (void)variablesArrived:(id)arg result:(void*)result
{
  if ([arg isEqualToString:@"status"])
  {
    if (_statusVariables)
      myx_free_variables(_statusVariables);
    _statusVariables= result;
    [[statusTable dataSource] setValues:_statusVariables];
    [statusTable reloadData];
  }
  else
  {
    int i, c;
    if (_serverVariables)
      myx_free_variables(_serverVariables);
    _serverVariables= result;
    [[serverTable dataSource] setValues:_serverVariables];
    [serverTable reloadData];
    
    // update values in graphs
    c= [_graphs count];
    for (i= 0; i < c; i++)
      [[_graphs objectAtIndex:i] updateServerVariables:_serverVariables];
      
  }
}

- (void)valueSaved:(id)argument result:(void*)result
{
  g_free((char*)argument);
  [self refreshServerVariables:self];
}

- (void)setValue:(NSString*)value 
     forVariable:(const char*)var
{
  char *tmp= g_strdup([value UTF8String]);
  
  [[_owner dispatcher] performCallback:(void*(*)(MYSQL*,void*,void*))myx_set_variable
                              argument:(char*)var
                              argument:tmp
                      finishedSelector:@selector(valueSaved:result:)
                              argument:(id)tmp
                                target:self];
}

@end


@implementation MAHealth

+ (NSImage*)icon
{
  return [[[NSImage alloc] initWithContentsOfFile: 
    [[NSBundle mainBundle] pathForResource:@"health"
                                    ofType:@"png"]] autorelease];
}

+ (NSString*)label
{
  return @"Health";
}

+ (NSString*)toolTip
{
  return @"Server Health and Statistics.";
}

- (id)initWithOwner: (id<MAdministratorProtocol>)owner
{
  self= [super initWithNibFile: @"Health" panelOwner: owner];
  if (self)
  {
    MYX_LIB_ERROR xerr;
    MYX_ADMIN_LIB_ERROR err;
    NSString *path;
    unsigned int i;
    id ds;
    
    _defaultFrame= [[self topView] frame];
        
    _graphs= [[NSMutableArray alloc] init];
    
    _editableIcon= [[self loadImage:@"variable_editable"] retain];
    _nonEditableIcon= [[self loadImage:@"variable"] retain];
    
    path= [[NSBundle mainBundle] pathForResource:@"mysqladmin_health"
                                          ofType:@"xml"];
    
    _pages= myx_read_in_health_pages((char*)[path UTF8String], &err);
    
    [[_owner dispatcher] performCallback:(void*(*)(MYSQL*))myx_get_server_variables
                          finishedSelector:@selector(variablesArrived:result:)
                            argument:@"server"
                                  target:self];
    [[_owner dispatcher] performCallback:(void*(*)(MYSQL*))myx_get_status_variables
                          finishedSelector:@selector(variablesArrived:result:)
                            argument:@"status"
                                  target:self];
    
    for (i= 0; i < _pages->pages_num; i++)
      [self buildPage:_pages->pages+i];

    
    path= [[NSBundle mainBundle] pathForResource:@"mysqlx_translations_administrator"
                                          ofType:@"xml"];
    {
      NSString *path0= [[NSBundle mainBundle] pathForResource:@"mysqlx_translations_general"
                                                       ofType:@"xml"];
      _trans= myx_init_trans([path0 UTF8String], [path UTF8String], "en", &xerr);
    }
    
    path= [[NSBundle mainBundle] pathForResource:@"mysqladmin_status_variables"
                                          ofType:@"xml"];
    _statusVariableListing= myx_get_variables_listing((char*)[path UTF8String], &err);

    path= [[NSBundle mainBundle] pathForResource:@"mysqladmin_system_variables"
                                          ofType:@"xml"];
    _serverVariableListing= myx_get_variables_listing((char*)[path UTF8String], &err);
    
    ds= [[ListingDataSource alloc] initWithListing:_statusVariableListing
                                       translation:_trans];
    [statusIndex setDataSource:ds];
    
    ds= [[ListingDataSource alloc] initWithListing:_serverVariableListing
                                       translation:_trans];
    [serverIndex setDataSource:ds];
    
    ds= [[VariableDataSource alloc] initWithTranslations:_trans owner:self];
    [ds setValues:_statusVariables];
    [statusTable setDataSource:ds];
    {
      MTextImageCell *cell = [[[MTextImageCell alloc] init] autorelease];
      [cell setEditable: YES];
      [[statusTable tableColumnWithIdentifier:@"variable"] setDataCell: cell];
    }
    
    ds= [[VariableDataSource alloc] initWithTranslations:_trans owner:self];
    [ds setValues:_serverVariables];
    [serverTable setDataSource:ds];
    {
      MTextImageCell *cell = [[[MTextImageCell alloc] init] autorelease];
      [cell setEditable: YES];
      [[serverTable tableColumnWithIdentifier:@"variable"] setDataCell: cell];
    }

    [self update:nil];
    
    _cancelThread= NO;
    [NSThread detachNewThreadSelector:@selector(updateThread:)
                             toTarget:self
                           withObject:[NSValue valueWithPointer:[_owner serverInfo]]];
  }
  return self;
}

- (IBAction)refreshServerVariables:(id)sender
{
  [[_owner dispatcher] performCallback:(void*(*)(MYSQL*))myx_get_server_variables
                      finishedSelector:@selector(variablesArrived:result:)
                              argument:@"server"
                                target:self];
}

- (IBAction)refreshStatusVariables:(id)sender
{
  [[_owner dispatcher] performCallback:(void*(*)(MYSQL*))myx_get_status_variables
                      finishedSelector:@selector(variablesArrived:result:)
                              argument:@"status"
                                target:self];
}


- (void)dealloc
{
  if (_statusVariables)
    myx_free_variables(_statusVariables);
  if (_serverVariables)
    myx_free_variables(_serverVariables);      
  myx_free_variables_listing(_statusVariableListing);
  myx_free_variables_listing(_serverVariableListing);
  myx_free_trans(_trans);
  [_editableIcon release];
  [_nonEditableIcon release];
  [_graphs release];
  [super dealloc];
}

- (BOOL)willClose
{
  _cancelThread= YES;
  return YES;
}
@end
