//
//  MAHealthGraph.m
//  MySQL Administrator
//
//  Created by Alfredo Kojima on Thu Jul 22 2004.
//  Copyright (c) 2004 MySQL AB. All rights reserved.
//

#import "MAHealthGraph.h"

#import <GSTable.h>

#import "MATimeGraph.h"
 
#define GRAPH_HEIGHT 80
#define METER_GRAPH_WIDTH 65
#define GRAPH_SPACING 5
#define STATUS_HEIGHT 16

//XXX bar graph
//XXX graph editor

/*
class BarGraph : public MAHealthPanel::BaseGraph {
    Gtk::HBox *_hbox;
    Gtk::Label *_label;
    MGHMeterGraphPlotter *_graph;
    Glib::ustring _expr, _max_expr;
    MYX_COMPILED_EXPRESSION *_cexpr, *_max_cexpr;
  public:
    BarGraph(MYX_HEALTH_GRAPH *graph)
      : _cexpr(0), _max_cexpr(0)
    {
      _hbox= Gtk::manage(new Gtk::HBox(false, 8));
      _hbox->show();

      _label= Gtk::manage(new Gtk::Label((char*)graph->graph_caption, 0.0, 0.5));
      _label->show();
      _hbox->pack_start(*_label, false, true);

      _graph= Gtk::manage(new MGHMeterGraphPlotter());
      _graph->show();
      _hbox->pack_start(*_graph, true, true);
      _graph->set_size_request(-1, BAR_GRAPH_HEIGHT);

      modify(graph);

      _hbox->show();
    }
    virtual void modify(MYX_HEALTH_GRAPH *graph)
    {
      if (graph->value_caption)
        _graph->set_current_format(Glib::ustring((char*)graph->value_caption));
      else
        _graph->set_current_format("");

      _graph->set_value_unit((MGHMeterGraphPlotter::Unit)graph->value_unit);
      if (graph->max_caption)
        _graph->set_total_format(Glib::ustring((char*)graph->max_caption));
      else
        _graph->set_total_format("");

      _expr= graph->value_formula?:"";
      _max_expr= graph->max_formula?:"";

      if (_cexpr)
        myx_free_expression(_cexpr);
      _cexpr= 0;
    }
    virtual ~BarGraph()
    {
      delete _hbox;
      if (_cexpr)
        myx_free_expression(_cexpr);
    }
    virtual Gtk::Widget *get_widget() { return _hbox; };
    virtual Gtk::Widget *get_aligned_widget() { return _label; };
    virtual void set_style(const Glib::RefPtr<Gtk::Style> &style)
    {
      _graph->set_style(style);
    }
    virtual void set_images(const Glib::RefPtr<Gdk::Pixbuf> &back_image,
                            const Glib::RefPtr<Gdk::Pixbuf> &fore_image)
    {
      _graph->set_images(back_image, fore_image);
    }
    virtual void update_values(MYX_VARIABLES *vars)
    {
      MYX_EXPRESSION_ERROR err;

      _old_values= _values;
      _values= vars;

      if (_server_values_changed)
      {
        Glib::ustring expr= replace_variables(_expr, false);
        if (_cexpr)
          myx_free_expression(_cexpr);
        _cexpr= myx_compile_expression(expr.c_str(), &err);
        if (!_cexpr)
          g_warning("could not compile expression %s", expr.c_str());

        expr= replace_variables(_max_expr, false);
        if (_max_cexpr)
          myx_free_expression(_max_cexpr);
        _max_cexpr= myx_compile_expression(expr.c_str(), &err);
        if (!_max_cexpr)
          g_warning("could not compile expression %s", expr.c_str());

        _server_values_changed= false;
      }

      if (_cexpr && _old_values)
      {
        double value;
        value= myx_eval_expression(_cexpr, _old_values, _values, &err);
        _graph->set_current_value(value);

        value= myx_eval_expression(_max_cexpr, _old_values, _values, &err);
        _graph->set_total_value(value);
      }
    }
};

*/


@implementation MAHealthGraph 

- (id)initWithGraph:(MYX_HEALTH_GRAPH*)graph
{
  self= [super init];
  if (self)
  {
    _graphSpec= graph;
  }
  return self;
}

- (void)dealloc
{
  [super dealloc];
}

- (void)updateServerVariables:(MYX_VARIABLES*)vars
{
  _serverValues= vars;
  _serverValuesChanged= YES;
}

- (void)updateValues:(MYX_VARIABLES*)vars
{
}

- (double)evaluate:(NSString*)s
{
  MYX_COMPILED_EXPRESSION *expr;
  MYX_EXPRESSION_ERROR err;
  double res;
  
  expr= myx_compile_expression([[self prepareExpression:s serverVariablesOnly:NO] UTF8String], &err);
  if (!expr)
    return 0.0;
  res= myx_eval_expression(expr, _oldValues, _values, &err);
  myx_free_expression(expr);
  
  return res;
}

- (NSString*)prepareExpression:(NSString*)expr
           serverVariablesOnly:(BOOL)serverOnly
{
  NSMutableString *s= [[[NSMutableString alloc] init] autorelease];
  unsigned int i;
  
  [s setString:expr];
  
  if (_values && !serverOnly)
  {
    // replace status variables with their indices
    for (i= 0; i < _values->variables_num; i++)
    {
      [s replaceOccurrencesOfString:[NSString stringWithCString:(char*)_values->variables[i].name]
                         withString:[NSString stringWithFormat:@"%i",i]
                            options:NSCaseInsensitiveSearch
                              range:NSMakeRange(0,[s length])];
    }
  }
  
  // replace server variable static values
  for (i= 0; i < _serverValues->variables_num; i++)
  {
    [s replaceOccurrencesOfString:[NSString stringWithFormat:@"[%s]",(char*)_serverValues->variables[i].name]
                       withString:[NSString stringWithCString:(char*)_serverValues->variables[i].value?:"0"]
                          options:NSCaseInsensitiveSearch
                            range:NSMakeRange(0,[s length])];
  }
  return s;
}

- (NSView*)view
{
  return nil;
}

@end

//======================================================================


@interface MAMeterGraph : NSView
{
  NSImage *_image;
  float _value;
  NSString *_format;
}
- (void)setImage:(NSImage*)image;
- (void)setValue:(float)value;
- (void)setCaptionFormat:(NSString*)format;
@end

@implementation MAMeterGraph
- (void)dealloc
{
  [_format release];
  [_image release];
  [super dealloc];
}

- (void)setImage:(NSImage*)image
{
  [_image release];
  _image= [image retain];
}

- (void)setCaptionFormat:(NSString*)format
{
  [_format release];
  _format= [format retain];
}

- (void)setValue:(float)value
{
  _value= value;
  [self setNeedsDisplay:YES];
}

- (void)drawRect:(NSRect)aRect
{
  NSRect rect;
  NSRect frame= [self frame];
  NSSize isize;
  NSPoint pos;
  NSString *cap;
  NSFont *font;
  float y;
  int i;
  
  [[NSColor blackColor] set];
  NSRectFill(aRect);
  
  rect.origin= NSMakePoint(0,0);
  isize= rect.size= [_image size];
  
  pos= NSMakePoint((frame.size.width - rect.size.width)/2,
                   10 + (frame.size.height - rect.size.height)/2);
  y= pos.y;

  [_image dissolveToPoint:pos
                 fromRect:rect
                 fraction:0.2];

  rect.origin= NSMakePoint(0,0);
  rect.size= [_image size];
  rect.size.height = ceil(((ceil(_value)*rect.size.height)/10)*10/100.0);
  [_image dissolveToPoint:pos
                 fromRect:rect
                 fraction:1.0];

  font= [NSFont systemFontOfSize:8];
  cap= [NSString stringWithFormat:_format,_value];
  pos.x= (frame.size.width - [font widthOfString:cap])/2;
  pos.y= 5;
  
  [cap drawAtPoint:pos 
    withAttributes:[NSDictionary dictionaryWithObjectsAndKeys:[NSColor whiteColor],NSForegroundColorAttributeName,
      font, NSFontAttributeName,nil]];
  
  [[NSColor blackColor] set];
  for (i= 0; i < 10; i++)
  {
    [NSBezierPath strokeLineFromPoint:NSMakePoint(0,y+isize.height/10*i)
                               toPoint:NSMakePoint(frame.size.width,y+isize.height/10*i)];
  }
}

@end

@implementation MALineHealthGraph 

- (void)dealloc
{
  int i;
  for (i= 0; i < 4; i++)
    [_cells[i] release];
  [_base release];
  [_graph release];
  [_meter release];
  [_infoMatrix release];
  
  [super dealloc];
}

- (id)initWithGraph:(MYX_HEALTH_GRAPH*)graph
{
  self= [super initWithGraph:graph];
  if (self)
  {
    NSTextFieldCell *cell;
    int i;

    _base= [[NSView alloc] init];
    
    _meter= [[MAMeterGraph alloc] init];
    [_meter setFrame:NSMakeRect(0,STATUS_HEIGHT,METER_GRAPH_WIDTH,GRAPH_HEIGHT)];
    [_meter setAutoresizingMask:NSViewMaxXMargin];
    
    _graph= [[MATimeGraph alloc] init];
    [_graph setFrame:NSMakeRect(METER_GRAPH_WIDTH+GRAPH_SPACING,STATUS_HEIGHT,100,GRAPH_HEIGHT)];
    [_graph setAutoresizingMask:NSViewWidthSizable];
    
    cell= [[[NSTextFieldCell alloc] initTextCell:@""] autorelease];
    [cell setFont:[NSFont systemFontOfSize:[NSFont smallSystemFontSize]]];
    [cell setBordered:YES];
    [cell setDrawsBackground:YES];
    
    _infoMatrix= [[NSMatrix alloc] initWithFrame:NSMakeRect(METER_GRAPH_WIDTH+GRAPH_SPACING, 0, 100, STATUS_HEIGHT)
                                            mode:NSTrackModeMatrix
                                       prototype:cell
                                    numberOfRows:1
                                 numberOfColumns:4];
    [_infoMatrix setAutosizesCells:YES];
    for (i= 0; i < 4; i++)
      _cells[i]= [_infoMatrix cellAtRow:0 column:i];

    [_infoMatrix setAutoresizingMask:NSViewWidthSizable];
    [_infoMatrix setIntercellSpacing:NSMakeSize(0,0)];

    [_base setFrame:NSMakeRect(0, 0, 100+METER_GRAPH_WIDTH+GRAPH_SPACING, GRAPH_HEIGHT+STATUS_HEIGHT)];
    [_base addSubview:_meter];
    [_base addSubview:_graph];
    [_base addSubview:_infoMatrix];
    
    [self setGraphAttributes: graph];
  }
  return self;
}

- (void)setMeterImage:(NSImage*)image
{
  [_meter setImage:[image retain]];
}

- (void)setGraphAttributes:(MYX_HEALTH_GRAPH*)graph
{
  NSRect base= [_base frame];
  if (graph->value_caption)
  {
    [_meter setHidden:NO];
    [_graph setFrame:NSMakeRect(METER_GRAPH_WIDTH+GRAPH_SPACING, [_graph frame].origin.y, base.size.width-(METER_GRAPH_WIDTH+GRAPH_SPACING), GRAPH_HEIGHT)];
    [_infoMatrix setFrame:NSMakeRect(METER_GRAPH_WIDTH+GRAPH_SPACING, [_infoMatrix frame].origin.y, base.size.width-(METER_GRAPH_WIDTH+GRAPH_SPACING), STATUS_HEIGHT)];
  }
  else
  {
    [_meter setHidden:YES];
    [_graph setFrame:NSMakeRect(0, [_graph frame].origin.y, base.size.width, GRAPH_HEIGHT)];
    [_infoMatrix setFrame:NSMakeRect(0, [_infoMatrix frame].origin.y, base.size.width, STATUS_HEIGHT)]; 
  }

  if (_meter)
    [_meter setCaptionFormat:
      [NSString stringWithFormat:@"%s %%.01f%%", (char*)graph->value_caption?:""]];

  if (graph->graph_caption && graph->display_graph_caption)
    [_graph setCaption:[NSString stringWithUTF8String:graph->graph_caption]];
  else
    [_graph setCaption:nil];

  [_graph setRangeMin:graph->min max:graph->max];

  if (_exprSrc)
    [_exprSrc release];
  _exprSrc= [[NSString alloc] initWithUTF8String:graph->value_formula];
  
  _min= graph->min;
  _max= graph->max;
  
 // if (graph->max_formula)
 //   _max_formula= graph->max_formula;

  if (_expr)
    myx_free_expression(_expr);
  _expr= NULL;
  
  _graphSpec= graph;
}


- (NSView*)view
{
  return _base;
}


- (void)updateValues:(MYX_VARIABLES*)vars
{
  MYX_EXPRESSION_ERROR err;
        
  _oldValues= _values;
  _values= vars;
      
  if (_serverValuesChanged)
  {
    NSString *expr= [self prepareExpression:_exprSrc
                        serverVariablesOnly:NO];
    if (_expr)
      myx_free_expression(_expr);
    _expr= myx_compile_expression([expr UTF8String], &err);
    if (!_expr)
    {
      NSLog(@"could not compile expression %@", _exprSrc);
    }
    _serverValuesChanged= NO;
    
    if (_graphSpec->max_formula)
    {
      _max= [self evaluate:[NSString stringWithCString:_graphSpec->max_formula]];
      [_graph setRangeMin:_min max:_max];
    }
  }
  
  if (_expr && _oldValues)
  {
    double value;
    value= myx_eval_expression(_expr, _oldValues, _values, &err);
    if (err != MYX_EXPRESSION_NO_ERROR)
      value= 0.0;
        
    [_graph addValue:value timestamp:time(NULL)];
    if (value > _max)
      _max= value;
    if (_meter)
    {
      [_meter setValue:((value-_min)*100/_max)];
    }

    {
      double max, min, avg;
      [_graph getStatsMin:&min max:&max average:&avg];
      [_cells[0] setStringValue:[NSString stringWithFormat:@"Current: %.0f", value]];
      [_cells[1] setStringValue:[NSString stringWithFormat:@"Maximal: %.0f", max]];
      [_cells[2] setStringValue:[NSString stringWithFormat:@"Minimal: %.0f", min]];
      [_cells[3] setStringValue:[NSString stringWithFormat:@"Average: %.0f", avg]];
    }
  }
}

@end


@implementation MABarHealthGraph


@end
