//
//  MAdministrator.m
//  MySQL Administrator
//
//  Created by Alfredo Kojima on Thu Jun 24 2004.
//  Copyright (c) 2004 MySQL AB. All rights reserved.
//

#import "MAdministrator.h"
#import "MAPanel.h"
#import "MSchemaDataSource.h"

#include "myx_admin_public_interface.h"

static NSMutableArray *Panels= nil;
static NSMutableDictionary *PanelsById= nil;

NSString *MASchemaDataSourceChanged=@"MASchemaDataSourceChanged";

BOOL MARegisterPanel(id panelClass)
{
  if (!Panels)
    Panels= [[NSMutableArray alloc] init];
  if (!PanelsById)
    PanelsById= [[NSMutableDictionary alloc] init];

  if (![panelClass isSubclassOfClass: [MAPanel class]])
  {
    NSLog(@"ERROR: Not registering invalid panel (%@)", [panelClass className]);
    return NO;
  }

  NSLog(@"registering panel %@", [panelClass className]);
  [Panels addObject: panelClass];
 
  [PanelsById setObject: panelClass forKey: [panelClass className]];
  
  return YES;
}



@interface MAdministrator(Private)
- (NSMutableArray*)fetchSchemas;
- (void)tablesArrived:(NSArray*)args result:(MYX_SCHEMA_TABLES*)tables;
- (void)requestTablesForSchema:(MSchemaDataSource*)sender item:(MSchemaItem*)item;
@end

@implementation MAdministrator(Private)

- (NSMutableArray*)fetchSchemas
{
  NSMutableArray *array;
  unsigned int i;
  MYX_CATALOGS *cats;
  
  cats= [_dispatcher performCallback:(void*(*)(MYSQL*))myx_get_catalogs
                       waitForWindow:[self window]
                             message:@"Retrieving catalog list..."];
  if (!cats)
  {
    NSLog(@"ERROR: Could not retrieve catalog list");
    return nil;
  }
  array= [NSMutableArray arrayWithCapacity: cats->catalogs_num];
  for (i= 0; i < cats->catalogs_num; i++)
  {
    unsigned int j;
    NSMutableArray *schemas= [NSMutableArray arrayWithCapacity: cats->catalogs[i].schemata_num];
    MSchemaItem *citem= [[MSchemaItem alloc] initWithCatalog: cats->catalogs+i
                                                        icon: _catalogIcon];
    
    for (j= 0; j < cats->catalogs[i].schemata_num; j++)
    {
      MSchemaItem *item= [[MSchemaItem alloc] initWithSchema: cats->catalogs[i].schemata+j
                                                        icon: _schemaIcon];
      [schemas addObject: item];
      [item release];
    }
    [citem setChildren: schemas];
    [array addObject: citem];
    [citem release];
  }
  
  _catalogs= cats;
  
  return array;
}


- (void)requestTablesForSchema:(MSchemaDataSource*)sender item:(MSchemaItem*)item
{
  MYX_SCHEMA *schema= [item schema];

  [_dispatcher performCallback:(void*(*)(MYSQL*,void*,void*))myx_get_schema_tables
                      argument:schema->catalog_name
                      argument:schema->schema_name
              finishedSelector:@selector(tablesArrived:result:)
                      argument:[[NSArray arrayWithObjects:sender,item,nil] retain]
                        target:self];
}


- (void)tablesArrived:(NSArray*)args result:(MYX_SCHEMA_TABLES*)tables
{
  MSchemaDataSource *ds= [args objectAtIndex:0];
  MSchemaItem *sitem= [args objectAtIndex:1];
  MYX_SCHEMA *schema= [sitem schema];
  NSMutableArray *array;
  unsigned int i;
  [args release];

  if (tables)
  {
    if (schema->schema_tables)
      myx_free_schema_tables(schema->schema_tables);
    schema->schema_tables= tables;
  }
  if (!tables)
  {
    NSLog(@"could not retrieve tables list");
    return;
  }
  array= [NSMutableArray arrayWithCapacity: tables->schema_tables_num];
  for (i= 0; i < tables->schema_tables_num; i++)
  {
    unsigned int j;
    NSMutableArray *columns= [NSMutableArray arrayWithCapacity: tables->schema_tables[i].columns_num];
    MSchemaItem *citem= [[MSchemaItem alloc] initWithTable: tables->schema_tables+i
                                                      icon: _tableIcon];
    
    for (j= 0; j < tables->schema_tables[i].columns_num; j++)
    {
      MYX_SCHEMA_TABLE_COLUMN *column= tables->schema_tables[i].columns+j;
      MSchemaItem *item= [[MSchemaItem alloc] initWithColumn: column
                                                        icon: column->primary_key ? _keyIcon : _columnIcon];
      [columns addObject: item];
      [item release];
    }
    [citem setChildren: columns];
    [array addObject: citem];
    [citem release];
  }
  [ds setRootList:array];
}
@end


@implementation MAdministrator

- (id)init
{  
  self = [super initWithWindowNibName:@"Administrator" owner:self];
  if (!self)
    return nil;

  [self window];
  if (![self isWindowLoaded])
    NSLog(@"ERROR: Could not load document window");
  
  _panels= [[NSMutableDictionary alloc] init];
  _mysqlLock= [[NSLock alloc] init];

  _catalogIcon= [[NSImage alloc] initWithContentsOfFile:
    [[NSBundle mainBundle] pathForResource:@"16x16_Catalog"
                                    ofType:@"png"]];
  _schemaIcon= [[NSImage alloc] initWithContentsOfFile:
    [[NSBundle mainBundle] pathForResource:@"16x16_Database"
                                    ofType:@"png"]];
  _tableIcon= [[NSImage alloc] initByReferencingFile:
    [[NSBundle mainBundle] pathForResource:@"16x16_Table"
                                    ofType:@"png"]];
  _columnIcon= [[NSImage alloc] initByReferencingFile:
    [[NSBundle mainBundle] pathForResource:@"16x16_Field"
                                    ofType:@"png"]];
  _keyIcon= [[NSImage alloc] initByReferencingFile:
    [[NSBundle mainBundle] pathForResource:@"16x16_KeyColumn"
                                    ofType:@"png"]];

  return self;
}


- (id)initWithConnection: (MYSQL*)mysql
                    info: (MYX_USER_CONNECTION*)info
{
  self = [self init];
  if (!self)
    return nil;
  
  if (self)
  {
    _mysql=mysql;
    _info= info;
    
    _dispatcher= [[MMySQLDispatcher alloc] initWithMySQL:mysql];
    
    if (mysql->unix_socket)
      _connectedInstance= [[NSString alloc] initWithFormat:@"%s via socket", info->hostname];
    else
      _connectedInstance= [[NSString alloc] initWithFormat:@"%s:%i", info->hostname, info->port];
    
    [self setTitleForPanel: nil];
  }
  return self;
}


- (void)dealloc
{
  [_dispatcher release];
  [_panels release];
  [_connectedInstance release];
  [_mysqlLock release];
  [_catalogIcon release];
  [_schemaIcon release];
  [_tableIcon release];
  [_columnIcon release];
  [_schemaDS release];
  [_columnDS release];

  if (_catalogs)
    myx_free_catalogs(_catalogs);
  
  NSLog(@"DEALLOC MAdministrator");
  if (_mysql)
    myx_mysql_close(_mysql);
  if (_info)
    myx_free_user_connection_content(_info);
  
  [super dealloc];
}

- (void)awakeFromNib
{
  NSToolbar *toolbar= [[NSToolbar alloc] initWithIdentifier:@"toolbar"];
  
  [toolbar setDelegate: self];
  [[self window] setToolbar:toolbar];
  
  [toolbar release];
}


- (MMySQLDispatcher*)dispatcher
{
  return _dispatcher;
}


- (void)setCanSwitchPanel:(BOOL)flag
{
  
}

- (NSToolbarItem *)toolbar:(NSToolbar *)toolbar 
     itemForItemIdentifier:(NSString *)itemIdentifier 
 willBeInsertedIntoToolbar:(BOOL)flag
{
  NSToolbarItem *newItem = [[[NSToolbarItem alloc] initWithItemIdentifier:itemIdentifier] autorelease];
  id itemClass = [PanelsById objectForKey:itemIdentifier];
  
  [newItem setLabel:[itemClass label]];
  [newItem setPaletteLabel:[itemClass label]];
  [newItem setImage:[itemClass icon]];
  [newItem setToolTip:[itemClass toolTip]];
  [newItem setTarget:self];
  [newItem setAction:@selector(toolbarItemClicked:)];
//  [newItem setMenuFormRepresentation:[item menuFormRepresentation]];

  return newItem;
}

- (NSArray *)toolbarDefaultItemIdentifiers:(NSToolbar*)toolbar
{
  NSMutableArray *array= [NSMutableArray array];
  int i;
  int count= [Panels count];
  [array addObject: [[Panels objectAtIndex: 0] className]];
  [array addObject: NSToolbarSeparatorItemIdentifier];  
  for (i= 1; i < count; i++)
    [array addObject: [[Panels objectAtIndex: i] className]];
  return array;
}

- (NSArray *)toolbarAllowedItemIdentifiers:(NSToolbar*)toolbar
{
  NSMutableArray *array= [NSMutableArray array];
  int i;
  int count= [Panels count];
  for (i= 0; i < count; i++)
    [array addObject: [[Panels objectAtIndex: i] className]];
  return array;
}

- (NSArray *)toolbarSelectableItemIdentifiers:(NSToolbar *)toolbar
{
  return [self toolbarAllowedItemIdentifiers:toolbar];
}

- (void)toolbarItemClicked:(id)sender
{
  MAPanel *panel;
  NSString *panelName= [sender itemIdentifier];
  
  panel= [_panels objectForKey: panelName];
  if (!panel)
  {
    panel= [[[PanelsById objectForKey: panelName] alloc] initWithOwner: self];
    [_panels setObject: panel forKey: panelName];
    [panel release];
  }
  
  [self switchToPanel: panel];
}

- (void)setTitleForPanel: (NSString*)name
{
  if (name)
    [[self window] setTitle: [NSString stringWithFormat:@"%@  %@", 
      name, _connectedInstance]];
  else
    [[self window] setTitle: _connectedInstance];
}

- (void)show
{
  MYX_MACHINE_INFO *mi= myx_get_server_info(_info, _mysql);
  [[self window] makeKeyAndOrderFront:nil];
  mi= myx_get_server_info(_info, _mysql);
}

- (void)switchToPanel: (MAPanel*)panel
{
  NSRect oldFrame;
  NSRect newFrame;
  NSWindow *window= [self window];
  
  if (panel == _currentPanel)
    return;
  
  if (_currentPanel)
  {
    if (![_currentPanel willHide])
      return;
  }
  if (![panel willShow])
    return;

  oldFrame= [window frame];
  
  newFrame= [window frameRectForContentRect: [[panel topView] frame]];
  newFrame.origin= oldFrame.origin;
  newFrame.origin.y+= (oldFrame.size.height - newFrame.size.height);

  if (_currentPanel)
    [[_currentPanel topView] retain];
  [window setContentView:nil];
  
  [window setFrame:newFrame display:YES animate:YES];
  [window setMinSize: [panel defaultFrame].size];
  
  [window setContentView:[panel topView]];
  [[panel topView] setAutoresizingMask:NSViewWidthSizable|NSViewHeightSizable|NSViewMinYMargin];

  [self setTitleForPanel: [[panel class] label]];

  {
    NSView *sideBox= [panel sideView];
    [drawer setContentView: sideBox];
    if (sideBox)
    {
      [sideBox setFrameSize: [drawer contentSize]];
      [drawer open];
    }
    else
    {
      [drawer close];
    }
  }
  
  if (_currentPanel)
    [_currentPanel didHide];
  [panel didShow];
}


- (BOOL)isConnected
{
  if (_mysql)
    return YES;
  return NO;
}

- (BOOL)isLocal
{
  if (_mysql)
    return myx_is_localhost(_info->hostname)!=0;
  else
    return NO;
}

- (int)serverStatus
{
  gchar *output= NULL;
  gint exst;
  
  if ([self isConnected])
    return 1;
  
  if (![self isLocal])
    return -1;
  
  if (g_spawn_command_line_sync("/bin/ps ax", &output, NULL, &exst, NULL)
      && exst == 0)
  {
    char *ptr= strtok(output, "\r");
    while (ptr != NULL)
    {
      char *a;
      if ((a=strstr(ptr, "mysqld")))
      {
        g_free(output);
        return 1;
      }
      ptr= strtok(NULL, "\r");
    }
  }
  if (output)
    g_free(output);
  return 0;
}

- (void)unlockMySQL
{
  [_mysqlLock unlock];
}

- (MYSQL*)mysql
{
  return _mysql;
}

- (MYSQL*)mysqlLock
{
  [_mysqlLock lock];
  return _mysql;
}

- (MYX_USER_CONNECTION*)serverInfo
{
  return _info;
}

- (MSchemaDataSource*)sharedSchemaDS
{
  if (!_schemaDS)
  {
    _schemaDS= [[MSchemaDataSource alloc] initWithRoot:MCatalogItemType leaf:MSchemaItemType];
    if (_columnDS)
    {
      [_schemaDS setRootList: [_columnDS rootList]];
    }
    else
    {
      id result= [self fetchSchemas];
      [_schemaDS setRootList: result];
    }
  }
  return _schemaDS;
}

- (MSchemaDataSource*)sharedColumnDS
{
  if (!_columnDS)
  {
    _columnDS= [[MSchemaDataSource alloc] initWithRoot:MCatalogItemType leaf:MColumnItemType];
    
    if (_schemaDS)
    {
      [_columnDS setRootList: [_schemaDS rootList]];
    }
    else
    {
      NSMutableArray *result= [self fetchSchemas];
      [_columnDS setRootList: result];
      [_columnDS setTableFetcher:self selector:@selector(requestTablesForSchema:item:)];
    }
  }
  return _columnDS;
}

@end
