/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#include "myx_grt_private.h"

/**
 * @file  myx_grt_value.c
 * @brief GRT value handling 
 * 
 * See also: <a href="../grt.html#GRTValue">GRT Value</a>
 */


/*
 ****************************************************************************
 * @brief Frees a list struct
 *
 *  Frees a list value and all of its contents.
 *****************************************************************************/
int myx_grt_free_list(MYX_GRT_LIST *list)
{
  unsigned int i;
  for (i= 0; i < list->items_num; i++)
  {
    myx_grt_value_release(list->items[i]);
  }
  g_free(list->content_struct_name);
  g_free(list->items);
  g_free(list);
  
  return 0;
}


/*
 ****************************************************************************
 * @brief Frees a dictionary struct
 *
 * Frees a dictionary value and all of its contents.
 *****************************************************************************/
int myx_grt_free_dict(MYX_GRT_DICT *dict)
{
  unsigned int i;
  for (i= 0; i < dict->items_num; i++)
  {
    g_free(dict->items[i].key);
    myx_grt_value_release(dict->items[i].value);
  }
  g_free(dict->struct_name);
  g_free(dict->items);
  g_free(dict);
  
  return 0;
}


/**
 ****************************************************************************
 * @brief Assigns a MYX_GRT_STRUCT to a dict value
 *
 *  This will assign a MYX_GRT_STRUCT to a list value. 
 *
 * @param dict a dict value
 * @param gstruct the name struct to assign or NULL
 *
 * @return 0 if ok, -1 if the parameters are invalid.
 *****************************************************************************/
int myx_grt_dict_assign_struct(MYX_GRT_VALUE *dict, const char *struct_name)
{
  g_return_val_if_fail(dict != NULL, -1);
  g_return_val_if_fail(dict->type == MYX_DICT_VALUE, -1);

  if (dict->value.d->struct_name)
    g_free(dict->value.d->struct_name);
  dict->value.d->struct_name= g_strdup(struct_name);
  
  return 0;
}


/**
 ****************************************************************************
 * @brief Assigns a MYX_GRT_STRUCT to a list value
 *
 *  This will assign a MYX_GRT_STRUCT to a list value. The list should then
 * only accept values that match the given struct.
 * 
 * @param list a list value containing dictionaries
 * @param struct_name the name struct to assign or NULL. 
 *
 * @return 0 if ok, -1 if the parameters are invalid.
 *****************************************************************************/
int myx_grt_list_assign_struct(MYX_GRT_VALUE *list, const char *struct_name)
{
  g_return_val_if_fail(list != NULL, -1);
  g_return_val_if_fail(list->type == MYX_LIST_VALUE, -1);

  if (list->value.l->content_struct_name)
    g_free(list->value.l->content_struct_name);
  list->value.l->content_struct_name= g_strdup(struct_name);

  return 0;
}


/**
 ****************************************************************************
 * @brief Returns the struct name associated with the value
 * 
 *   This will return the name of the MYX_GRT_STRUCT associated with the 
 *  dictionary or list which was previously set with 
 *  \c myx_grt_dict_assign_struct or \c myx_grt_list_assign_struct.
 *
 * @param value a dict or list GRT value
 *
 * @return A reference to the struct name or NULL if there's nothing associated.
 *****************************************************************************/
const char *myx_grt_value_struct_name(MYX_GRT_VALUE *value)
{
  g_return_val_if_fail(value != NULL, NULL);
  g_return_val_if_fail(value->type == MYX_DICT_VALUE || value->type == MYX_LIST_VALUE, NULL);
  
  if (value->type == MYX_DICT_VALUE)
    return value->value.d->struct_name;
  else if (value->type == MYX_LIST_VALUE)
  {
    if (value->value.l->content_struct_name)
      return value->value.l->content_struct_name;
    else
      return NULL;
  }
  else
    return NULL;
}


static void print_value(MYX_GRT *grt, MYX_GRT_VALUE *value, int depth)
{
  unsigned int i;
  char spaces[1024];
  char buffer[200];

  g_return_if_fail(depth < (int)sizeof(spaces));

  switch (value->type)
  {
  case MYX_INT_VALUE:
    g_snprintf(buffer, sizeof(buffer), "%i"NL, value->value.i);
    MYX_PRINT(grt, buffer);
    break;
  case MYX_REAL_VALUE:
    g_snprintf(buffer, sizeof(buffer), "%f"NL, value->value.r);
    MYX_PRINT(grt, buffer);
    break;
  case MYX_STRING_VALUE:
    MYX_PRINT(grt, value->value.s);
    MYX_PRINT(grt, NL);
    break;
  case MYX_LIST_VALUE:
    MYX_PRINT(grt, "["NL);
    for (i= 0; i < value->value.l->items_num; i++)
    {
      g_snprintf(spaces, sizeof(spaces), "%*s", depth, "");
      MYX_PRINT(grt, spaces);
      print_value(grt, value->value.l->items[i], depth+1);
    }
    g_snprintf(spaces, sizeof(spaces), "%*s", depth, "");
    MYX_PRINT(grt, spaces);
    MYX_PRINT(grt, "]"NL);
    break;
  case MYX_DICT_VALUE:
    MYX_PRINT(grt, "{"NL);
    for (i= 0; i < value->value.d->items_num; i++)
    {
      g_snprintf(spaces, sizeof(spaces), "%*s", depth+1, "");
      MYX_PRINT(grt, spaces);
      myx_grt_printf(grt, "%s = ", value->value.d->items[i].key);
      print_value(grt, value->value.d->items[i].value, depth+1);
    }
    g_snprintf(spaces, sizeof(spaces), "%*s", depth, "");
    MYX_PRINT(grt, spaces);
    MYX_PRINT(grt, "}"NL);
    break;
  default:
    MYX_PRINT(grt, "Invalid value"NL);
    break;
  }
}

/**
 ****************************************************************************
 * @brief Prints the contents of a value to stdout
 *
 * @param grt the GRT environment
 * @param value the value to be printed
 *
 * @return 0
 *****************************************************************************/
int myx_grt_value_print(MYX_GRT *grt, MYX_GRT_VALUE *value)
{
  print_value(grt, value, 0);
  return 0;
}


/**
 ****************************************************************************
 * @brief Decrements the reference count of the value
 * 
 *   Will decrease the reference count of the value by one. When it reaches 0
 *   it will recursively release (and free) the whole contents of the value.
 *
 * @param value the value to be released
 *
 * @return 0
 * 
 * @see myx_grt_value_retain
 *****************************************************************************/
int myx_grt_value_release(MYX_GRT_VALUE *value)
{
  g_return_val_if_fail(value != NULL, -1);
  g_return_val_if_fail(value->refcount > 0, -1);

  value->refcount--;
  if (value->refcount == 0)
  {
    switch (value->type)
    {
    case MYX_INT_VALUE:
    case MYX_REAL_VALUE:
      break;
    case MYX_STRING_VALUE:
      g_free(value->value.s);
      break;
    case MYX_LIST_VALUE:
      myx_grt_free_list(value->value.l);
      break;
    case MYX_DICT_VALUE:
      myx_grt_free_dict(value->value.d);
      break;
    }
    g_free(value);
  }
  return 0;
}


/** 
 ****************************************************************************
 * @brief Increments the reference count of the value.
 *
 *  Increments the reference count of the value by one. Newly created values
 * start with a reference count of 1, so you normally don't need to retain them.
 *
 * @param value - GRT value
 *
 * @return the GRT value passed as argument
 *
 * @see myx_grt_value_release
 ****************************************************************************
 */
MYX_GRT_VALUE *myx_grt_value_retain(MYX_GRT_VALUE *value)
{
  g_return_val_if_fail(value != NULL, value);

  value->refcount++;

  return value;
}


/**
 ****************************************************************************
 * @brief create a dict value from a list of key/type/value tuples.
 *
 *   If value is a list of dict, you must pass a MYX_GRT_LIST or
 *   MYX_GRT_DICT. These must not be freed. But usually you will give this
 *   function a MYX_GRT_VALUE containing a list or dict. In that case 
 *   the value will be just retained, so you can release it later.
 *
 * @param  key the dict key for the value or NULL, followed by vtype and value\n
 *    vtype type of the value may be 0 for a MYX_GRT_VALUE\n
 *    value the value\n
 * Repeat these as many times as needed. End list of values with NULL
 *
 * @return  A GRT value of type dict.
 *****************************************************************************/
MYX_GRT_VALUE *myx_grt_dict_create(const char *struct_name, const char *key, ...)
{
  va_list args;
  MYX_GRT_VALUE *dict;
  MYX_GRT_VALUE *gvalue;
  MYX_GRT_VALUE_TYPE vtype;
  
  g_return_val_if_fail(key != NULL, NULL);

  dict= myx_grt_dict_new(struct_name);

  va_start(args, key);

  while (key != NULL)
  {
    vtype= va_arg(args, MYX_GRT_VALUE_TYPE);
    
    switch (vtype)
    {
    case MYX_INT_VALUE:
      gvalue= myx_grt_value_from_int(va_arg(args, int));
      break;
    case MYX_REAL_VALUE:
      gvalue= myx_grt_value_from_double(va_arg(args, double));
      break;
    case MYX_STRING_VALUE:
      gvalue= myx_grt_value_from_string(va_arg(args, char*));
      break;
    case MYX_LIST_VALUE:
      gvalue= g_new0(MYX_GRT_VALUE, 1);
      gvalue->type= MYX_LIST_VALUE;
      gvalue->value.l= va_arg(args, MYX_GRT_LIST*);
      gvalue->refcount= 1;
      break;
    case MYX_DICT_VALUE:
      gvalue= g_new0(MYX_GRT_VALUE, 1);
      gvalue->type= MYX_DICT_VALUE;
      gvalue->value.d= va_arg(args, MYX_GRT_DICT*);
      gvalue->refcount= 1;
      break;
    default:
      gvalue= va_arg(args, MYX_GRT_VALUE*);
      myx_grt_value_retain(gvalue);
    }

    myx_grt_dict_set_item(dict, key, gvalue);
    
    myx_grt_value_release(gvalue);
    
    key= va_arg(args, const char*);
  }
  
  va_end(args);
  
  return dict;
}


/**
 ****************************************************************************
 * @brief Creates a GRT value of MYX_INT_VALUE type
 *
 *   Creates a GRT value of type MYX_INT_VALUE and initialized it with the 
 * passed argument.
 * 
 * @param i the integer value that it will be initialized to.
 *
 * @return A newly created value struct.
 *****************************************************************************/
MYX_GRT_VALUE *myx_grt_value_from_int(int i)
{
  MYX_GRT_VALUE *value= g_new0(MYX_GRT_VALUE, 1);
  value->type= MYX_INT_VALUE;
  value->value.i= i;
  value->refcount= 1;
  return value;
}


/**
 ****************************************************************************
 * @brief creates a GRT value of MYX_REAL_VALUE type
 *
 *   Creates a GRT value of type MYX_REAL_VALUE and initialized it with the 
 * passed argument.
 *
 * @param d the value that it will be initialized to.
 *
 * @return A newly created value struct.
 *****************************************************************************/
MYX_GRT_VALUE *myx_grt_value_from_double(double d)
{
  MYX_GRT_VALUE *value= g_new0(MYX_GRT_VALUE, 1);
  value->type= MYX_REAL_VALUE;
  value->value.r= d;
  value->refcount= 1;
  return value;
}


/**
 ****************************************************************************
 * @brief Creates a GRT value of MYX_STRING_VALUE type
 * 
 *   Creates a GRT value of type MYX_STRING_VALUE and initialized it with the 
 * passed argument. The string will be copied.
 *
 * @param s the value that it will be initialized to.
 * 
 * @return A newly created value struct.
 *****************************************************************************/
MYX_GRT_VALUE *myx_grt_value_from_string(const char *s)
{
  MYX_GRT_VALUE *value= g_new0(MYX_GRT_VALUE, 1);
  value->type= MYX_STRING_VALUE;
  value->value.s= g_strdup(s);
  value->refcount= 1;
  return value;
}


/**
 ****************************************************************************
 * @brief Creates a new empty GRT value of MYX_GRT_DICT type
 *
 *   Creates a GRT value of type MYX_GRT_DICT, which is a dictionary or
 * mapping of keys strings to values. You can add, query and remove values
 * to it with the other dict functions.
 * 
 * \b NOTE
 *   The values are inserted in alphabetical order.
 * 
 * @return  A newly created dict value struct.
 *****************************************************************************/
MYX_GRT_VALUE *myx_grt_dict_new(const char *struct_name)
{
  MYX_GRT_VALUE *value= g_new0(MYX_GRT_VALUE, 1);
  value->type= MYX_DICT_VALUE;
  value->value.d= g_new0(MYX_GRT_DICT, 1);
  value->refcount= 1;

  if (struct_name)
    myx_grt_dict_assign_struct(value, struct_name);

  return value;
}


static int bisect(MYX_GRT_DICT *dict, const char *key, int *index)
{
  int i;
  int j, k;
  
  j= 0;
  k= dict->items_num-1;
  while (j <= k)
  {
    int r;
    i= (j+k)/2;
    r= strcmp(key, dict->items[i].key);
    
    if (r == 0)
    {
      *index= i;
      return 1;
    }
    else if (r < 0)
      k= i-1;
    else if (r > 0)
      j= i+1;
  }
  
  if (j > k)
    *index= j;
  else
    *index= k;

  return 0;
}


/**
 ****************************************************************************
 * @brief Sets a value for a key in the dict
 * 
 *   Inserts a key value pair into the dict. The value will have its
 * reference count increased, therefore can be released afterwards.
 * If the key already exists in the dictionary, the old one will be released
 * before it's replaced.
 * 
 * \b NOTE
 *   The values are inserted in already order so that a binary search can
 * be used to lookup a specific value.
 * 
 * @param dict  a GRT value of type MYX_GRT_DICT
 * @param key   the key name where the value should be inserted
 * @param value the value to be inserted. The value will have its reference
 *     count increased.
 *
 * @return  0 on success, -1 on error
 *****************************************************************************/
int myx_grt_dict_set_item(MYX_GRT_VALUE *dict, const char *key, MYX_GRT_VALUE *value)
{
  int i;
  g_return_val_if_fail(dict!=NULL, -1);
  g_return_val_if_fail(dict->type == MYX_DICT_VALUE, -1);
  g_return_val_if_fail(key!=NULL, -1);
  g_return_val_if_fail(*key, -1);
  g_return_val_if_fail(value!=NULL, -1);
  
  if (dict->value.d->items_num == 0)
  {
    /* dict is empty */
    dict->value.d->items_num++;
    dict->value.d->items= g_new0(MYX_GRT_DICT_ITEM, dict->value.d->items_num);

    dict->value.d->items[0].key= g_strdup(key);
    dict->value.d->items[0].value= myx_grt_value_retain(value);
  }
  else
  {
    if (!bisect(dict->value.d, key, &i))
    {
      // the key was not in the tree already
      dict->value.d->items_num++;
      dict->value.d->items= g_realloc(dict->value.d->items,sizeof(MYX_GRT_DICT_ITEM)*dict->value.d->items_num);
      
      if (i < (int)dict->value.d->items_num-1)
      {
        memmove(dict->value.d->items + i + 1,
                dict->value.d->items + i,
                sizeof(MYX_GRT_DICT_ITEM)*(dict->value.d->items_num-i-1));
      }
      dict->value.d->items[i].key= g_strdup(key);
      dict->value.d->items[i].value= myx_grt_value_retain(value);
    }
    else
    {
      // replace value
      if (dict->value.d->items[i].value != value)
      {
        myx_grt_value_release(dict->value.d->items[i].value);
        dict->value.d->items[i].value= myx_grt_value_retain(value);
      }
    }
  }

  return 0;
}

/**
 ****************************************************************************
 * @brief Looks up for the key in the dict and returns the value associated to it.
 *
 *   This will return the value for the specified key.
 * 
 * @param dict a GRT value of type MYX_GRT_DICT
 * @param key the key name to search
 *
 * @return  NULL if the value does not exist in the dict or the value if it does.
 *****************************************************************************/
MYX_GRT_VALUE *myx_grt_dict_get_item_value(MYX_GRT_VALUE *dict, const char *key)
{
  int i;
  g_return_val_if_fail(dict!=NULL, NULL);
  g_return_val_if_fail(dict->type == MYX_DICT_VALUE, NULL);
  g_return_val_if_fail(key!=NULL, NULL);
  g_return_val_if_fail(*key, NULL);

  if (!bisect(dict->value.d, key, &i))
    return NULL;
  else
    return dict->value.d->items[i].value;
}

/**
 ****************************************************************************
 * @brief Removes the key and it's value from the dict
 *
 *   Remove the value associated with the key in the dict. The value will be 
 *   released.
 *
 * @param dict a GRT value of type MYX_GRT_DICT
 * @param key the key to remove
 * 
 * @return 0 if the value was removed, -1 if the key was not in the dict.
 *****************************************************************************/
int myx_grt_dict_del_item(MYX_GRT_VALUE *dict, const char *key)
{
  int i;
  g_return_val_if_fail(dict!=NULL, -1);
  g_return_val_if_fail(dict->type == MYX_DICT_VALUE, -1);
  g_return_val_if_fail(key!=NULL, -1);
  g_return_val_if_fail(*key, -1);

  if (!bisect(dict->value.d, key, &i))
    return -1;
  else
  {
    g_free(dict->value.d->items[i].key);
    myx_grt_value_release(dict->value.d->items[i].value);
    
    memmove(dict->value.d->items + i, dict->value.d->items + i + 1,
            sizeof(MYX_GRT_DICT_ITEM)*(dict->value.d->items_num-i-1));
    dict->value.d->items_num--;
    return 0;
  }
}

/**
 ****************************************************************************
 * @brief Returns the items of elements in the dict value
 *
 *   Returns the number of items in the dict. Use in conjunction with
 *  myx_grt_dict_item_by_index() to traverse full contents of the dictionary.
 *
 * @param dict the dict
 *
 * @return Number of items in the dict.
 *****************************************************************************/
unsigned int myx_grt_dict_item_num(MYX_GRT_VALUE *dict)
{
  g_return_val_if_fail(dict!=NULL, 0);
  g_return_val_if_fail(dict->type == MYX_DICT_VALUE, 0);
  
  return dict->value.d->items_num;
}


/**
 ****************************************************************************
 * @brief Get the item at index in the dict
 *
 *  Retrieves the item at the specified index in the dict. If you are
 * traversing the dictionary, you should not modify its contents during
 * that.
 * 
 * @param dict the dict
 * @param index index in the dict to fetch
 * @param retkey pointer to where the key string will be returned
 * @param retvalue pointer to where the return GRT value will be returned
 *
 * @return -1 if there's some error (bad parameters or invalid index), 0 otherwise.
 *****************************************************************************/
int myx_grt_dict_item_by_index(MYX_GRT_VALUE *dict, unsigned int index, 
                          const char **retkey, MYX_GRT_VALUE **retvalue)
{
  g_return_val_if_fail(dict!=NULL, -1);
  g_return_val_if_fail(dict->type == MYX_DICT_VALUE, -1);

  if (index >= dict->value.d->items_num)
    return -1;

  *retkey= dict->value.d->items[index].key;
  *retvalue= dict->value.d->items[index].value;
  
  return 0;
}

/**
 ****************************************************************************
 * @brief Get the item's key name at index in the dict
 *
 *  Retrieves the item's key name at the specified index in the dict. 
 * 
 * @param dict the dict
 * @param index index in the dict to fetch
 *
 * @return NULL if there's some error (bad parameters or invalid index), the string otherwise.
 *****************************************************************************/
const char * myx_grt_dict_item_key_by_index(MYX_GRT_VALUE *dict, unsigned int index)
{
  g_return_val_if_fail(dict!=NULL, NULL);
  g_return_val_if_fail(dict->type == MYX_DICT_VALUE, NULL);

  if (index >= dict->value.d->items_num)
    return NULL;

  return dict->value.d->items[index].key;
}

/**
 ****************************************************************************
 * @brief Get the item's value at index in the dict
 *
 *  Retrieves the item's value at the specified index in the dict. If you are
 * traversing the dictionary, you should not modify its contents during
 * that.
 * 
 * @param dict the dict
 * @param index index in the dict to fetch
 *
 * @return NULL if there's some error (bad parameters or invalid index), the value otherwise.
 *****************************************************************************/
MYX_GRT_VALUE * myx_grt_dict_item_value_by_index(MYX_GRT_VALUE *dict, unsigned int index)
{
  g_return_val_if_fail(dict!=NULL, NULL);
  g_return_val_if_fail(dict->type == MYX_DICT_VALUE, NULL);

  if (index >= dict->value.d->items_num)
    return NULL;
  
  return dict->value.d->items[index].value;
}

/**
 ****************************************************************************
 * @brief Convenience function to create a list value from a stringlist.
 *
 *   Will create a MYX_GRT_VALUE of type list containing the strings in sl.
 *   All the strings will be copied.
 *
 * @param  sl the stringlist
 *
 * @return  The newly created value or NULL on error.
 *
 *****************************************************************************/
MYX_GRT_VALUE *myx_grt_value_from_stringlist(MYX_STRINGLIST *sl)
{
  MYX_GRT_VALUE *list;
  unsigned int i;

  g_return_val_if_fail(sl != NULL, NULL);

  list= myx_grt_list_new(MYX_STRING_VALUE, NULL);
  list->value.l->items_num= sl->strings_num;
  list->value.l->items= g_malloc(sizeof(MYX_GRT_VALUE*)*sl->strings_num);
  for (i= 0; i < sl->strings_num; i++)
    list->value.l->items[i]= myx_grt_value_from_string(sl->strings[i]);

  return list;
}

/**
 ****************************************************************************
 * @brief Creates a new empty list GRT value 
 * 
 *   Creates a GRT value of type list. All elements inserted to the list
 * must be of content_type.
 *
 * @param  content_type the type that the contents of the list will have
 *
 * @return  The new list value or NULL if there's an error.
 *****************************************************************************/
MYX_GRT_VALUE *myx_grt_list_new(MYX_GRT_VALUE_TYPE content_type, const char *struct_name)
{
  MYX_GRT_VALUE *value;
  g_return_val_if_fail(content_type!=0, NULL);

  value= g_new0(MYX_GRT_VALUE, 1);
  value->type= MYX_LIST_VALUE;
  value->value.l= g_new0(MYX_GRT_LIST, 1);
  value->value.l->content_type= content_type;
  value->refcount= 1;

  if (struct_name)
    myx_grt_list_assign_struct(value, struct_name);

  return value;
}

/**
 ****************************************************************************
 * @brief Returns the content type of the list
 * 
 *   Every list can only have values of one type. This function returns this 
 * content type.
 *
 * @param list a GRT value of type list
 *
 * @return  The content type as MYX_GRT_VALUE_TYPE
 *****************************************************************************/
MYX_GRT_VALUE_TYPE myx_grt_list_content_type(MYX_GRT_VALUE *list)
{
  return list->value.l->content_type;
}

/**
 ****************************************************************************
 * @brief Returns the content of the list as MYX_STRINGLIST
 * 
 *
 * @param list a GRT value of type list
 *
 * @return  A new allocated MYX_STRINGLIST
 *****************************************************************************/
MYX_STRINGLIST * myx_grt_list_as_stringlist(MYX_GRT_VALUE *list)
{
  MYX_STRINGLIST *strlist;
  unsigned int i;

  g_return_val_if_fail(list->type == MYX_LIST_VALUE, NULL);
  g_return_val_if_fail(list->value.l->content_type == MYX_STRING_VALUE, NULL);

  strlist= g_malloc0(sizeof(MYX_STRINGLIST));
  strlist->strings_num= list->value.l->items_num;
  strlist->strings= g_malloc(sizeof(char *)*strlist->strings_num);

  for (i= 0; i<strlist->strings_num; i++)
  {
    strlist->strings[i]= g_strdup(myx_grt_list_get_item(list, i)->value.s);
  }

  return strlist;
}

/**
 ****************************************************************************
 * @brief Inserts an element to the list
 *
 *   Inserts a value to the list. The value to be inserted will not be copied
 * but will be have its reference count increased, therefore can be released.
 *
 * @param list a GRT value of type list
 * @param index the index in the list where the item should be inserted. -1 means append
 * @param value the value to insert.
 * 
 * @return  -1 on error, 0 on success.
 *****************************************************************************/
int myx_grt_list_insert_item(MYX_GRT_VALUE *list, int index, MYX_GRT_VALUE *value)
{
  g_return_val_if_fail(list != NULL, -1);
  g_return_val_if_fail(list->type == MYX_LIST_VALUE, -1);
  g_return_val_if_fail(index <= (int)list->value.l->items_num, -1);
  g_return_val_if_fail(list->value.l->content_type == value->type, -1);

  myx_grt_value_retain(value);
  
  list->value.l->items_num++;
  list->value.l->items= g_realloc(list->value.l->items, 
                                  sizeof(MYX_GRT_VALUE)*list->value.l->items_num);
  if (index < 0)
    list->value.l->items[list->value.l->items_num-1]= value;
  else
  {
    memmove(list->value.l->items + index + 1, list->value.l->items + index,
            sizeof(MYX_GRT_VALUE*)*(list->value.d->items_num-index));
    list->value.l->items[index]= value;
  }
  return 0;
}

/**
 ****************************************************************************
 * @brief Removes an element from the list
 *
 *   Removes the element at the given index from the list. The element will
 * be released.
 *
 * @param list GRT value of type list
 * @param index index in the list of the element to remove
 *
 * @return 0 on success, -1 if the element does not exist.
 *****************************************************************************/
int myx_grt_list_del_item(MYX_GRT_VALUE *list, int index)
{
  g_return_val_if_fail(list != NULL, -1);
  g_return_val_if_fail(list->type == MYX_LIST_VALUE, -1);
  g_return_val_if_fail(index >= 0 && index < (int)list->value.l->items_num, -1);

  myx_grt_value_release(list->value.l->items[index]);
  
  memmove(list->value.l->items + index, list->value.l->items + index + 1,
          sizeof(MYX_GRT_VALUE*)*(list->value.d->items_num-index-1));

  return 0;
}

/**
 ****************************************************************************
 * @brief Returns the number of elements in the list
 *
 *   Returns element count in the list.
 *
 * @param list GRT value of type list
 *
 * @return Element count in the list.
 *****************************************************************************/
unsigned int myx_grt_list_item_num(MYX_GRT_VALUE *list)
{
  g_return_val_if_fail(list != NULL, 0);
  g_return_val_if_fail(list->type == MYX_LIST_VALUE, 0);
  
  return list->value.l->items_num;
}


/**
 ****************************************************************************
 * @brief Returns the number of elements in the list
 *
 *   Returns the element at the given index.
 *
 * @param list GRT value of type list
 * @param index in the list to return
 *
 * @return The item or NULL if the index is invalid.
 *****************************************************************************/
MYX_GRT_VALUE *myx_grt_list_get_item(MYX_GRT_VALUE *list, unsigned int index)
{
  g_return_val_if_fail(list != NULL, NULL);
  g_return_val_if_fail(list->type == MYX_LIST_VALUE, NULL);
  g_return_val_if_fail(index < list->value.l->items_num, NULL);

  return list->value.l->items[index];
}

/**
 ****************************************************************************
 * @brief Replaces a value in the list
 *
 *  This will replace the value at the specified index with the new one.
 *  The old value will be released and the new one retained.
 *
 * @param list the list
 * @param index index in the list to replace the value. Should be < list_size
 * @param new_value the new value to put in the list
 *
 * @return 0 if ok, -1 on error (such as invalid arguments or bad index)
 *****************************************************************************/
int myx_grt_list_set_item(MYX_GRT_VALUE *list, unsigned int index, MYX_GRT_VALUE *new_value)
{  
  g_return_val_if_fail(list!=NULL, -1);
  g_return_val_if_fail(list->type == MYX_LIST_VALUE, -1);
  g_return_val_if_fail(index < myx_grt_list_item_num(list), -1);
  g_return_val_if_fail(new_value!=NULL, -1);

  if (list->value.l->items[index] != new_value)
  {
    myx_grt_value_release(list->value.l->items[index]);
    list->value.l->items[index]= myx_grt_value_retain(new_value);
  }
  return 0;
}



/**
 ****************************************************************************
 * @brief Converts a value of type list to dict
 *
 *  The passed value withh be converted to a dict. The keys in the dict
 * will be strings containing the index of each item. The original list is
 * modified and becomes a dict.
 *
 * @param value a GRT value of type list that will be converted to dict
 *
 * @return 0 on success, -1 on error.
 *****************************************************************************/
int myx_grt_convert_list_to_dict(MYX_GRT_VALUE *value)
{
  unsigned int i;
  MYX_GRT_LIST *list;
  char key[128];

  g_return_val_if_fail(value != NULL, -1);
  g_return_val_if_fail(value->type == MYX_LIST_VALUE, -1);

  list= value->value.l;
  value->value.d= g_new0(MYX_GRT_DICT, 1);
  for (i= 0; i < list->items_num; i++)
  {
    g_snprintf(key, sizeof(key), "%i", i);
    myx_grt_dict_set_item(value, key, list->items[i]);
    myx_grt_value_release(list->items[i]);
  }
  g_free(list);
  
  return 0;
}


/**
 ****************************************************************************
 * @brief Traverses a dict and return the requested value
 *
 *   This will treat the given value as a tree, where each node is named by 
 * the key in its parent.
 *   For dictionaries, the path component is used as the key name. For lists,
 * the given component is used to match the name attribute of each list item.
 *
 * @param dict a dict type value to be traversed which nested dicts
 * @param path a / separated list of key names to the value to be returned.
 *
 * @return A pointer to the value if it's found or NULL otherwise.
 *****************************************************************************/
MYX_GRT_VALUE *myx_grt_dict_set_value_by_path(MYX_GRT_VALUE *dict, const char *path)
{
  MYX_GRT_VALUE *value= dict;
  char *p;
  char *part;

  g_return_val_if_fail(dict != NULL, NULL);
  g_return_val_if_fail(path != NULL, NULL);
  g_return_val_if_fail(*path == '/', NULL);

  p= g_strdup(path);

  part= strtok(p, "/");

  while (part && *part && value)
  {
    if (value->type == MYX_DICT_VALUE)
      value= myx_grt_dict_get_item_value(value, part);
    else if (value->type == MYX_LIST_VALUE)
    {
      unsigned int i;
      MYX_GRT_LIST *list= value->value.l;
      
      if (list->content_type == MYX_DICT_VALUE)
      {
        for (i= 0; i < list->items_num; i++)
        {
          MYX_GRT_VALUE *name;
          name= myx_grt_dict_get_item_value(list->items[i], "name");
          if (name && strcmp(myx_grt_value_as_string(name), part)==0)
          {
            value= list->items[i];
            break;
          }
        }
      }
    }
    else
    {
      value= NULL;
      break;
    }
    
    part= strtok(NULL, "/");
  }

  g_free(p);

  return value;
}


/**
 ****************************************************************************
 * @brief Traverses a dict and changes the value in the path
 *
 *   This will treat the given value as a tree, where each node is named by 
 * the key in its parent.
 *   For dictionaries, the path component is used as the key name. For lists,
 * the given component is used to match the name attribute of each list item.
 *
 * @param dict a dict type value to be traversed which nested dicts
 * @param path a / separated list of key names to the value to be changed.
 * @param new_value the value to assign in the dict
 *
 * @return 0 on success
 * @return -1 if some component of the path doesn't exist or the
 * referenced object is not a dictionary or list.
 *
 *****************************************************************************/
int myx_grt_dict_get_value_by_path(MYX_GRT_VALUE *dict, const char *path, MYX_GRT_VALUE *new_value)
{
  MYX_GRT_VALUE *value= dict, *parent= NULL;
  char *p;
  char *part;
  char *name;
  unsigned int index;
  int index_ok= 0;
  int rc= -1;

  g_return_val_if_fail(dict != NULL, -1);
  g_return_val_if_fail(path != NULL, -1);
  g_return_val_if_fail(*path == '/', -1);
  g_return_val_if_fail(new_value != NULL, -1);

  p= g_strdup(path);
  name= strrchr(p, '/');
  if (name > p && *(name+1) == 0) /* check if the path ends with a / */
  {
    *name= 0;
    name= strrchr(p, '/')+1;
  }
  else
    name++;
  name= g_strdup(name);

  part= strtok(p, "/");

  while (part && *part && value)
  { 
    index_ok= 0;

    parent= value;

    if (value->type == MYX_DICT_VALUE)
      value= myx_grt_dict_get_item_value(value, part);
    else if (value->type == MYX_LIST_VALUE)
    {
      unsigned int i;
      MYX_GRT_LIST *list= value->value.l;
      
      if (list->content_type == MYX_DICT_VALUE)
      {
        for (i= 0; i < list->items_num; i++)
        {
          MYX_GRT_VALUE *name;
          name= myx_grt_dict_get_item_value(list->items[i], "name");
          if (name && strcmp(myx_grt_value_as_string(name), part)==0)
          {
            index= i;
            index_ok= 1;
            value= list->items[i];
            break;
          }
        }
      }
    }
    else
    {
      value= NULL;
      break;
    }
    
    part= strtok(NULL, "/");
  }

  g_free(p);

  if (parent)
  {    
    if (parent->type == MYX_DICT_VALUE)
    {
      myx_grt_dict_set_item(parent, name, new_value);
      rc= 0;
    }
    else if (parent->type == MYX_LIST_VALUE)
    {
      myx_grt_list_set_item(parent, index, new_value);
      rc= 0;
    }
  }

  g_free(name);
  
  return rc;
} 

/**
 ****************************************************************************
 * @brief Returns the name of a dict
 *
 *   This returns the name of a dict. The name of a dict is defined by
 * one of its item with the key "name"
 *
 * @param dict a dict type value
 *
 * @return the name as a string which must not be freed
 * @return NULL if no item with the key "name" exists
 *
 *****************************************************************************/
const char * myx_grt_dict_name_item_as_string(MYX_GRT_VALUE *dict)
{
  MYX_GRT_VALUE *name_item;
  
  g_return_val_if_fail(dict != NULL, NULL);
  g_return_val_if_fail(dict->type == MYX_DICT_VALUE, NULL);

  name_item= myx_grt_dict_get_item_value(dict, "name");

  if (name_item)
    return myx_grt_value_as_string(name_item);
  else
    return NULL;
}


/**
 ****************************************************************************
 * @brief Returns the _id member of a dict
 *
 *   This returns the id of a dict. The id of a dict is defined by
 * one of its item with the key "_id"
 *
 * @param dict a dict type value
 *
 * @return the name as a string which must not be freed
 * @return NULL if no item with the key "name" exists
 *
 *****************************************************************************/
const char * myx_grt_dict_id_item_as_string(MYX_GRT_VALUE *dict)
{
  MYX_GRT_VALUE *id_item;

  g_return_val_if_fail(dict != NULL, NULL);
  g_return_val_if_fail(dict->type == MYX_DICT_VALUE, NULL);

  id_item= myx_grt_dict_get_item_value(dict, "_id");

  if (id_item)
    return myx_grt_value_as_string(id_item);
  else
    return NULL;
}



MYX_GRT_VALUE_TYPE myx_grt_value_get_type(MYX_GRT_VALUE *value)
{
  g_return_val_if_fail(value != NULL, 0);

  return value->type;
}

const char * myx_get_value_type_as_string(MYX_GRT_VALUE_TYPE value_type)
{
  switch (value_type)
  {
    case MYX_INT_VALUE:
      return "int";
    case MYX_REAL_VALUE:
      return "real";
    case MYX_STRING_VALUE:
      return "string";
    case MYX_LIST_VALUE:
      return "list";
    case MYX_DICT_VALUE:
      return "dict";
  }
  return NULL;
}

MYX_GRT_VALUE_TYPE myx_get_value_type_from_string(const char *value_type_name, MYX_GRT_ERROR *error)
{
  if (!value_type_name)
  {
    *error= MYX_GRT_BAD_DATA;
    return MYX_INT_VALUE;
  }

  *error= MYX_GRT_NO_ERROR;

  if (strcmp2(value_type_name, "int") == 0)
    return MYX_INT_VALUE;
  else if (strcmp2(value_type_name, "real") == 0)
    return MYX_REAL_VALUE;
  else if (strcmp2(value_type_name, "string") == 0)
    return MYX_STRING_VALUE;
  else if (strcmp2(value_type_name, "list") == 0)
    return MYX_LIST_VALUE;
  else if (strcmp2(value_type_name, "dict") == 0)
    return MYX_DICT_VALUE;

  *error= MYX_GRT_BAD_DATA;
  return MYX_INT_VALUE;
}


/* Get value contents */

int myx_grt_value_as_int(MYX_GRT_VALUE *value)
{
  g_return_val_if_fail(value != NULL, 0);
  g_return_val_if_fail(value->type == MYX_INT_VALUE, 0);

  return value->value.i;
}


double myx_grt_value_as_double(MYX_GRT_VALUE *value)
{
  g_return_val_if_fail(value != NULL, 0.0);
  g_return_val_if_fail(value->type == MYX_REAL_VALUE
                       || value->type == MYX_INT_VALUE, 0.0);

  if (value->type == MYX_REAL_VALUE)
    return value->value.r;
  else
    return value->value.i;
}


const char *myx_grt_value_as_string(MYX_GRT_VALUE *value)
{
  g_return_val_if_fail(value != NULL, NULL);
  g_return_val_if_fail((value->type == MYX_STRING_VALUE), NULL);

  return value->value.s;
}

char * myx_grt_value_format_as_string(MYX_GRT_VALUE *value)
{
  const char *name;

  switch (value->type)
  {
    case MYX_INT_VALUE:
      return g_strdup_printf("%d", value->value.i);
    case MYX_STRING_VALUE:
      return g_strdup(value->value.s);
    case MYX_REAL_VALUE:
      return g_strdup_printf("%16.8d", value->value.i);
    case MYX_LIST_VALUE:
      return g_strdup("LIST");
    case MYX_DICT_VALUE:
      name= myx_grt_dict_name_item_as_string(value);
      if (name)
        return g_strdup(name);
      else
        return g_strdup("DICT");
  }
  return g_strdup("");
}


MYX_GRT_LIST *myx_grt_value_as_list(MYX_GRT_VALUE *value)
{
  g_return_val_if_fail(value != NULL, NULL);
  g_return_val_if_fail(value->type == MYX_LIST_VALUE, NULL);
  
  return value->value.l;
}


MYX_GRT_DICT *myx_grt_value_as_dict(MYX_GRT_VALUE *value)
{
  g_return_val_if_fail(value != NULL, NULL);
  g_return_val_if_fail(value->type == MYX_DICT_VALUE, NULL);
  
  return value->value.d;
}


/* Utility stuff to get the contents of a dict directly */

const char *myx_grt_dict_item_as_string(MYX_GRT_VALUE *dict, const char *key)
{
  MYX_GRT_VALUE *value= myx_grt_dict_get_item_value(dict, key);
  
  if (!value)
    return NULL;

  return myx_grt_value_as_string(value);
}


int myx_grt_dict_item_as_int(MYX_GRT_VALUE *dict, const char *key)
{
  MYX_GRT_VALUE *value= myx_grt_dict_get_item_value(dict, key);
  
  if (!value)
    return 0;

  return myx_grt_value_as_int(value);
}


double myx_grt_dict_item_as_double(MYX_GRT_VALUE *dict, const char *key)
{
  MYX_GRT_VALUE *value= myx_grt_dict_get_item_value(dict, key);
  
  if (!value)
    return 0.0;

  return myx_grt_value_as_double(value);
}


int _myx_grt_get_refcount(MYX_GRT_VALUE *value)
{
  return value->refcount;
}
