/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_canvas.h 
 * @brief Generic canvas main class.
 * 
 */

#ifndef __GC_CANVAS_H__
#define __GC_CANVAS_H__

#include "myx_gc_base.h"

#include "myx_gc_utilities.h"
#include "myx_gc_layer.h"
#include "myx_gc_view.h"

//----------------------------------------------------------------------------------------------------------------------

class CFigure;
class CFigureInstance;
class CLayer;
class CGenericCanvas;
class CGCView;

//----------------- CFigureInstanceEnumerator --------------------------------------------------------------------------

/**
 * The CFigureInstanceEnumerator class is for quick access to all figure instances on all (common) layers.
 * Enumeration happens depth-first. That means for each layer first all instances are enumerated before the next layer
 * is taken.
 */
class GENERIC_CANVAS_API CFigureInstanceEnumerator
{
private:
  CGenericCanvas* FCanvas;
  CLayers::iterator FLayerIterator;
  CFigureInstances::iterator FFigureInstanceIterator;
public:
  CFigureInstanceEnumerator(CGenericCanvas* Canvas);

  virtual bool __cdecl hasNext(void);
  virtual CFigureInstance* __cdecl next(void);
  virtual void __cdecl release(void);
  virtual void __cdecl reset(void);
};

//----------------- CGenericCanvas -------------------------------------------------------------------------------------

#define GC_STATE_PENDING_ACTIVATION 0x0001

class CCanvasListener: public CGCListener
{
  friend class CGenericCanvas;
protected:
  CGenericCanvas* canvas;
public:
  virtual void __cdecl onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason);
  virtual void __cdecl onDestroy(CGCBase* object);
  virtual void __cdecl onError(CGCBase* sender, CGCBase* origin, const char* message);
};

/**
 * CGenericCanvas is the main class of the library and is the base for all further functionality (e.g. it creates and 
 * maintains the model). Instances are created via the exported CreateGenericCanvas function (if called from non C++ 
 * languages). CGenericCanvas serves as the controller in the model-view-controller pattern, which is used here and 
 * communicates with the viewer via callbacks.
 * The viewer is platform specific and must be implemented individually. It is responsible to create a canvas controller class.
 *
 * @see CreateGenericCanvas
 */
class GENERIC_CANVAS_API CGenericCanvas: public CGCBase 
{
  friend class CGCModel;
  friend class CFeedbackLayer;
  friend class CFigureInstanceEnumerator;
private:
  wstring FName;
  GCContext FContext;                  // The always active OpenGL rendering cntext.
  CGCModel* FModel;                    // The model this canvas is controlling.
  CLayers FLayers;                     // A list of layers currently in this canvas.
  CFeedbackLayer* FFeedbackLayer;      // The selection layer is special.
  int FUpdateCount;                    // If > 0 then the canvas is currently updating internal structures. Display will not update.
  CViews FViews;                       // A list of views that can be used by this canvas.
  CGCView* FCurrentView;               // The currently active set of layers.
  unsigned int FStates;               // Persistent storage for certain states in the canvas.
  CCanvasListener FListener;
protected:
  void clearBuffers();
public:
  CGenericCanvas(GCContext Context, wstring Name);
  virtual ~CGenericCanvas(void);

  virtual void __cdecl addLayer(CLayer* Layer);
  virtual TGCError __cdecl addLayoutsFromFile(const char* FileName);
  virtual TGCError __cdecl addStylesFromFile(const char* FileName);
  virtual void __cdecl addToSelection(CFigureInstance* Instance);
  virtual void __cdecl beginUpdate(void);
  virtual void __cdecl checkError(void);
  virtual void __cdecl clearContent(void);
  virtual void __cdecl clearLayouts(void);
  virtual void __cdecl clearSelection(void);
  virtual void __cdecl clearStyles(void);
  virtual CFigure* __cdecl createFigure(const char* type, const char* class_);
  virtual CLayer* __cdecl createLayer(const char* Name, bool AddToCurrentView);
  virtual CGCView* __cdecl createView(const char* Name);
  virtual void __cdecl currentView(CGCView* View);
  virtual CGCView* __cdecl currentView(void);
  virtual bool __cdecl doAction(CFigureInstance* Instance, const float X, const float Y);
  virtual void __cdecl endUpdate(void);
  virtual CFigureInstanceEnumerator* __cdecl getFigureInstanceEnumerator(void);
  CGCModel* getModel(void) { return FModel; }; // For special use only. Speaking MVC pattern, the view should not access the model!
  virtual TGCDirection __cdecl getSelectionInfo(const float X, const float Y);
  virtual void __cdecl invalidateSelectionBounds(CFigureInstance* Instance);
  virtual bool __cdecl isUpdating(void);
  virtual CLayer* __cdecl layerByName(const char* Name);
  virtual TGCVariant __cdecl property(const char* name, unsigned int index);
  virtual void __cdecl property(const char* name, unsigned int index, const TGCVariant& value);
  virtual void __cdecl removeFromSelection(CFigureInstance* Instance);
  virtual void __cdecl removeLayer(CLayer* Layer);
  virtual void __cdecl removeView(CGCView* View);
  virtual void __cdecl render(void);
  virtual void __cdecl resizeFiguresStart(int X, int Y, TGCDirection Direction);
  virtual void __cdecl resizeFiguresStop(void);
  virtual void __cdecl resizeFiguresTo(int X, int Y);
  virtual void __cdecl rubberbandResize(int X, int Y, TRBSelectionAction Action);
  virtual void __cdecl rubberbandStart(TRubberbandStyle Style, int X, int Y, bool ClearSelection);
  virtual void __cdecl rubberbandStop(void);
  virtual void __cdecl showSelection(bool Visible);
  virtual CGCView* __cdecl viewByName(const char* Name);
};

//----------------------------------------------------------------------------------------------------------------------

// Factory function to create a generic canvas. This function is exported and must be used by the viewer implementations
// to actually create a canvas instance. This is the only way to get hold of a generic canvas instance for non-C++ languages.
extern "C" GENERIC_CANVAS_API CGenericCanvas* CreateGenericCanvas(GCContext Context, char* Name);

#endif // __GC_CANVAS_H__
