/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_datatypes.h 
 * @brief Some commonly used data types.
 * 
 */

#ifndef __GC_DATATYPES_H__
#define __GC_DATATYPES_H__

#include <assert.h>

#include "myx_gc.h"

//----------------- Macros ---------------------------------------------------------------------------------------------

// Assertion macro with custom message.
#ifdef _WINDOWS
  #define ASSERT(exp, message) (void)( (exp) || (_assert(message, __FILE__, __LINE__), 0))
#endif // #fidef _WINDOWS

// There is no ANSI rounding function for float numbers, so define our own.
#define ROUND(X) (int)((X) < 0 ? (X) - 0.5 : (X) + 0.5)

#define XML_IS(Node, Name) (xmlStrcmp(Node->name, (const xmlChar *) Name) == 0)

//----------------- Data types -----------------------------------------------------------------------------------------

#ifdef _WINDOWS
  // An opaque handle to a rendering context. Must be provided by the viewer.
  typedef HDC GCContext;
#elif defined(__APPLE__)
  typedef CGLContextObj GCContext;
#else
  typedef GLXContext GCContext;
#endif // ifdef _WINDOWS

typedef struct tagViewport
{
  int left, top, width, height;
  tagViewport(int iLeft, int iTop, int iWidth, int iHeight)
  {
    left = iLeft;
    top = iTop;
    width = iWidth;
    height = iHeight;
  }
  tagViewport()
  {
    left = 0;
    top = 0;
    width = 0;
    height = 0;
  };
} TGCViewport;

// Some geometric data types.
typedef struct tagVertex
{
  float x;
  float y;
  float z;
  float w;

  tagVertex(float aX, float aY, float aZ, float aW)
  {
    x = aX;
    y = aY;
    z = aZ;
    w = aW;
  };
  tagVertex()
  {
    x = 0;
    y = 0;
    z = 0;
    w = 1;
  };
} TVertex;

typedef vector<TVertex> CVertexVector;

typedef GLfloat TMatrix[16];

typedef struct tagBoundingBox
{
  TVertex upper;
  TVertex lower;

  tagBoundingBox()
  {
  };

    tagBoundingBox(TVertex aUpper, TVertex aLower)
  {
    upper = aUpper;
    lower = aLower;
  };
} TBoundingBox;

typedef struct tagConstraints
{
  float maxHeight;
  float maxWidth;
  float minHeight;
  float minWidth;

  tagConstraints()
  {
    maxHeight = -1;
    maxWidth = -1;
    minHeight = -1;
    minWidth = -1;
  };
} TConstraints;

typedef enum tagGCError
{
  GC_NO_ERROR = 0, 
  GC_CANT_OPEN_FILE, 
  GC_XML_PARSE_ERROR,
  GC_XML_INVALID_DOCUMENT, 
  GC_XML_EMPTY_DOCUMENT, 
  GC_OBJECT_NOT_FOUND, 
  GC_CANT_READ_FROM_FILE, 
  GC_CHARSET_CONVERSION_ERROR,
  GC_CHARSET_WRONG_CHARSET_SPECIFIED
} TGCError;

typedef enum tagChangeReason
{
  // Selection related changes.
  GC_CHANGE_SELECTION_ADD,             // One or more figure instances were added to the current selection.
  GC_CHANGE_SELECTION_CLEAR,           // The current selection was cleared.
  GC_CHANGE_SELECTION_REMOVE,          // One or more figure instances were removed from the current selection.
  GC_CHANGE_SELECTION_CHANGE,          // One or more figure instances were added to or removed from the current selection.

  // Canvas related changes.
  GC_CHANGE_CANVAS_REFRESH,            // Used to indicate that the view must update the visual representation.
  GC_CHANGE_CANVAS_PROPERTY,           // The value of a property has been changed.
  GC_CHANGE_CANVAS_ADD_VIEW,           // A new view was added.
  GC_CHANGE_CANVAS_ADD_LAYER,          // A new layer was added.
  GC_CHANGE_CANVAS_SWITCH_VIEW,        // Another view was activated.
  GC_CHANGE_CANVAS_REMOVE_VIEW,        // A view was removed.
  GC_CHANGE_CANVAS_REMOVE_LAYER,       // A layer was removed.
  GC_CHANGE_CANVAS_CLEAR_CONTENT,      // All figures have been removed.
  GC_CHANGE_CANVAS_CLEAR_LAYOUTS,      // All layout definitions have been removed.
  GC_CHANGE_CANVAS_CLEAR_STYLES,       // All styles have been removed.

  // Model related changes.
  GC_CHANGE_MODEL_PROPERTY,            // The value of a property has been changed.
  GC_CHANGE_MODEL_ADD_FIGURE,          // A new figure was added.
  GC_CHANGE_MODEL_REMOVE_FIGURE,       // A figure was removed.
  GC_CHANGE_MODEL_ADD_STYLE,           // A new style was added.
  GC_CHANGE_MODEL_REMOVE_STYLE,        // A style was removed.

  // Caption element related changes.
  GC_CHANGE_CAPTION_PROPERTY,          // The value of a property has been changed.

  // Figure element related changes.
  GC_CHANGE_ELEMENT_PROPERTY,         // The value of a property has been changed.

  // Figure related changes.
  GC_CHANGE_FIGURE_PROPERTY,           // The value of a property has been changed.

  // Figure instance related changes.
  GC_CHANGE_FINSTANCE_PROPERTY,        // The value of a property has been changed.

  // View related changes.
  GC_CHANGE_VIEW_PROPERTY,             // The value of a property has been changed.
  GC_CHANGE_VIEW_ADD_LAYER,            // A new layer was added to a view.
  GC_CHANGE_VIEW_REMOVE_LAYER,         // A layer was removed.

  // Layer related changes.
  GC_CHANGE_LAYER_CLEAR,               // All figure instances on the layer are removed.
  GC_CHANGE_LAYER_VISIBILITY,          // The visibility of a layer has been changed.
  GC_CHANGE_LAYER_PROPERTY,            // The value of a property has been changed.
  GC_CHANGE_LAYER_ADD_INSTANCE,        // A new figure instance was added.
  GC_CHANGE_LAYER_REMOVE_INSTANCE,     // A figure instance was removed.
  GC_CHANGE_LAYER_ADD_GROUP,           // A new group was added to a view.
  GC_CHANGE_LAYER_REMOVE_GROUP         // A group was removed.
} TGCChangeReason;

typedef enum tagGCDirection
{
  GC_SI_NONE,
  GC_SI_ON_OBJECT,
  GC_SI_NORTH,
  GC_SI_NORTH_EAST,
  GC_SI_EAST,
  GC_SI_SOUTH_EAST,
  GC_SI_SOUTH,
  GC_SI_SOUTH_WEST,
  GC_SI_WEST,
  GC_SI_NORTH_WEST
} TGCDirection;

/**
 * TRubberbandStyle describes the look of the rubberband in the selection layer.
 */
typedef enum tagRubberbandStyle
{
  GC_RBSTYLE_SOLID_THIN,               // A simple black rectangle with a one pixel wide border.
  GC_RBSTYLE_SOLID_THICK,              // A simple black rectangle with a 3 pixel wide border.
  GC_RBSTYLE_DOTTED_THIN,              // A simple black rectangle with a one pixel wide dotted border.
  GC_RBSTYLE_DOTTED_THICK,             // A simple black rectangle with a 3 pixel wide dotted border.
  GC_RBSTYLE_BLENDED_CHECKERBOARD,     // A filled rectangle with a one pixel border and a translucent interior.
                                       // The system's selection color is used. The interior is a checker board.
  GC_RBSTYLE_BLENDED_DIAGONALS         // A filled rectangle with a one pixel border and a translucent interior.
                                       // The system's selection color is used. The interior consists of diagonal bands.
} TRubberbandStyle;

/**
 * TRBSelectionAction (rubber band selection action) determines how to manipulate the selection state of figure
 * instances with regard to their bounding box intersecting with the rubber band.
 */
typedef enum tagRBSelectionAction
{
  GC_RBACTION_NONE,               // Don't touch the selection state of any figure instance.
                                  // Usually used for non-selecting rubber bands (e.g. for figure creation).
  GC_RBACTION_SELECT,             // Always select figure instances if their bounding box intersects. Keep selected 
                                  // instances as their are if the do not intersect anymore.
                                  // Usually used for rubber bands with pressed shift key modifier.
  GC_RBACTION_SELECT_REMOVE,      // Select figure instances if they intersect, unselect those, which do not intersect.
                                  // Most common rubber band selection mode.
  GC_RBACTION_TOGGLE              // Revert the selection state of figure instances, which intersect. Don't touch the others.
                                  // Usually used for rubber bands with pressed control key modifier.
} TRBSelectionAction;

/** Layout variants for a figure element. */
typedef enum tagFigureElementLayout
{
  GC_LAYOUT_ROW,     
  GC_LAYOUT_COLUMN
} TFigureElementLayout;

/** Bidirectional mode */
typedef enum tagBidiMode
{
  GC_BIDI_LEFT_TO_RIGHT,          // Standard directionality (most languages).
  GC_BIDI_RIGHT_TO_LEFT           // Used for arabic and hebrew text.
} TBidiMode;

/** A struct to transport certain base data that has no previously known type. */
typedef enum tagGCVariantType
{
  GC_VAR_UNKNOWN,       // The value type is unknown (e.g. because a property does not exist).
  GC_VAR_BOOL,          // The value is a boolean.
  GC_VAR_INT,           // The value is an integer number.
  GC_VAR_FLOAT,         // The value is a floating point number.
  GC_VAR_STRING,        // The value is a sequence of characters.
  GC_VAR_LIST,          // The value is a collection of objects (e.g. layers).
  GC_VAR_OBJECT         // The value is an object with subproperties (=> dict in GRT).
} TGCVariantType;

class CGCBase;

typedef struct tagGCVariant
{
  TGCVariantType type;
  bool b;
  int i;                          // Values are not in a union because of the string entry.
  float f;
  string s;
  CGCBase* reference;
  tagGCVariant()
  {
    type = GC_VAR_UNKNOWN;
    b = false;
    i = 0;
    f = 0;
    reference = NULL;
  };
  tagGCVariant& operator=(const tagGCVariant& other)
  {
    type = other.type;
    b = other.b;
    i = other.i;
    f = other.f;
    s = other.s;
    reference = other.reference;

    return (*this);
  };
} TGCVariant;

/** Identifier for containers. Used when parsing pathes for properties. */
typedef enum tagContainerID
{
  GC_CONTAINER_UNKNOWN,
  GC_CONTAINER_LAYERS,
  GC_CONTAINER_FEEDBACK,
  GC_CONTAINER_FIGURE_INSTANCES,
  GC_CONTAINER_FIGURE_CONTENT,
  GC_CONTAINER_VIEWS,
  GC_CONTAINER_MODEL,
  GC_CONTAINER_STYLE,
  GC_CONTAINER_STYLES,
  GC_CONTAINER_LAYOUTS,
  GC_CONTAINER_FIGURE,
  GC_CONTAINER_FIGURES,
  GC_CONTAINER_SCALING,
  GC_CONTAINER_TRANSLATION,
  GC_CONTAINER_ROTATION,
  GC_CONTAINER_GROUPS,
  GC_CONTAINER_CHILDREN,
  GC_CONTAINER_CAPTION,
  GC_CONTAINER_CONTENT
} TContainerID;

/** Identifier for properties. Used when parsing property specifications. */
typedef enum tagPropertyID
{
  // Simple properties.
  GC_PROPERTY_UNKNOWN,
  GC_PROPERTY_NAME,
  GC_PROPERTY_ID,
  GC_PROPERTY_WIDTH,
  GC_PROPERTY_HEIGHT,
  GC_PROPERTY_X,
  GC_PROPERTY_Y,
  GC_PROPERTY_Z,
  GC_PROPERTY_DESCRIPTION,
  GC_PROPERTY_ZOOMX,
  GC_PROPERTY_ZOOMY,
  GC_PROPERTY_COLOR,
  GC_PROPERTY_JITTER,
  GC_PROPERTY_ANGLE,
  GC_PROPERTY_VISIBLE,
  GC_PROPERTY_ENABLED,
  GC_PROPERTY_SELECTED,
  GC_PROPERTY_LAYOUT,
  GC_PROPERTY_RESIZABLE,
  GC_PROPERTY_EXPANDED,
  GC_PROPERTY_MIN_WIDTH,
  GC_PROPERTY_MIN_HEIGHT,
  GC_PROPERTY_MAX_WIDTH,
  GC_PROPERTY_MAX_HEIGHT,
  GC_PROPERTY_TEXT,
  GC_PROPERTY_FONT_FAMILY,
  GC_PROPERTY_FONT_SIZE,
  GC_PROPERTY_FONT_WEIGHT,
  GC_PROPERTY_FONT_STYLE,
  GC_PROPERTY_ALIGNMENT_VERTICAL,
  GC_PROPERTY_ALIGNMENT_HORIZONTAL,
  GC_PROPERTY_BIDI_MODE,
  GC_PROPERTY_OWNER
} TPropertyID;

/** Determines how often a figure element is allowed to appear. */
typedef enum tagOccurence
{
  GC_OCC_ONCE,               // A single instance element (default).
  GC_OCC_ZERO_OR_MORE,       // An element in a list that can appear in any number.
  GC_OCC_ONE_OR_MORE         // An element in a list that must exist at least once.
} TOccurence;

typedef map<string, unsigned char*> TColorMap;
typedef map<string, unsigned char*>::const_iterator TColorMapIterator;
typedef pair<string, unsigned char*> TColorMapPair;

/** Lists of styles, figure templates, figures and figure instances. */
class CGCStyle;
#ifdef __GNUC__
typedef map<wstring, CGCStyle*> CStyleList;
#else
typedef hash_map<wstring, CGCStyle*> CStyleList;
#endif

class CFigureElementTemplate;
typedef vector<CFigureElementTemplate*> CElementTemplateList;

class CFigureElement;
typedef vector<CFigureElement*> CElementList;

class CFigureTemplate;
typedef multimap<wstring, CFigureTemplate*> CLayoutList;
typedef pair<wstring, CFigureTemplate*> CLayoutPair;

class CFigure;
typedef vector<CFigure*> CFigureList;

class CFigureInstance;
typedef vector<CFigureInstance*> CFigureInstances;

// Lists of listeners etc.

// A list of layers.
class CLayer;
typedef vector<CLayer*> CLayers;

// A list of views.
class CGCView;
typedef vector<CGCView*> CViews;

//----------------------------------------------------------------------------------------------------------------------

#endif // #ifndef __GC_DATATYPES_H__

