/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_canvas.cpp 
 * @brief Generic canvas main class and entry point.
 * 
 */

#include "myx_gc.h"
#include "myx_gc_figure.h"
#include "myx_gc_layer.h"
#include "myx_gc_model.h"
#include "myx_gc_gl_helper.h"
#include "myx_gc_font_manager.h"
#include "myx_gc_texture.h"
#include "myx_gc_svgparser.h"
#include "myx_gc_figure_parser.h"
#include "myx_gc_canvas.h"


//----------------------------------------------------------------------------------------------------------------------

// Factory function for a canvas.
GENERIC_CANVAS_API CGenericCanvas* CreateGenericCanvas(GCContext Context, char* name)
{
	return new CGenericCanvas(Context, utf8ToUtf16(name));
}

//----------------- CFigureInstanceEnumerator --------------------------------------------------------------------------

/**
 * Constructor of the enumerator class.
 *
 * @param canvas The canvas which contains the layers which are to be enumerated.
 */
CFigureInstanceEnumerator::CFigureInstanceEnumerator(CGenericCanvas* canvas)
{
  FCanvas = canvas;
  FLayerIterator = FCanvas->FLayers.end();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines if there is a next figure instance to enumerate.
 *
 * @return True if there is still a figure instance otherwise false.
 */
bool CFigureInstanceEnumerator::hasNext(void)
{
  return FLayerIterator != FCanvas->FLayers.end();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the next figure instance in the sequence.
 *
 * @return The next figure instance.
 */
CFigureInstance* CFigureInstanceEnumerator::next(void)
{
  CFigureInstance* result = *FFigureInstanceIterator;

  // Advance to next instance.
  ++FFigureInstanceIterator;
  while (true)
  {
    CLayer* layer = *FLayerIterator;
    if (FFigureInstanceIterator != layer->FInstances.end())
      break;

    // This layer is exhausted. Get the next one.
    ++FLayerIterator;
    if (FLayerIterator == FCanvas->FLayers.end())
    {
      // No more layers.
      break;
    };

    if (!(*FLayerIterator)->FEnabled)
      FFigureInstanceIterator = (*FLayerIterator)->FInstances.end();
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Frees this enumerator instance. Usually called by non-C++ languages as memory is managed by the C++ runtime.
 */
void CFigureInstanceEnumerator::release(void)
{
  delete this;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Resets the enumerator to the first figure instance in the canvas.
 */
void CFigureInstanceEnumerator::reset(void)
{
  FLayerIterator = FCanvas->FLayers.begin();
  while (FLayerIterator != FCanvas->FLayers.end())
  {
    if ((*FLayerIterator)->FEnabled)
    {
      CLayer* layer = *FLayerIterator;
      FFigureInstanceIterator = layer->FInstances.begin();
      if (FFigureInstanceIterator == layer->FInstances.end())
      {
        ++FLayerIterator;
      }
      else
        break; // Found the first instance.
    }
    else
      ++FLayerIterator;
  };
}

//----------------- CCanvasListener ------------------------------------------------------------------------------------

void CCanvasListener::onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason)
{
  if (canvas != NULL)
    canvas->change(origin, reason);
}

//----------------------------------------------------------------------------------------------------------------------

void CCanvasListener::onDestroy(CGCBase* object)
{
}

//----------------------------------------------------------------------------------------------------------------------

void CCanvasListener::onError(CGCBase* sender, CGCBase* origin, const char* message)
{
  if (canvas != NULL)
    canvas->error(origin, message);
}

//----------------- CGenericCanvas -------------------------------------------------------------------------------------

CGenericCanvas::CGenericCanvas(GCContext Context, wstring name)
{ 
  _className = "CGenericCanvas";
  FName = name;
  FContext = Context;
  FUpdateCount = 0;
  FCurrentView = NULL;
  FStates = 0;

  FListener.canvas = this;
  FModel = new CGCModel(this);
  FFeedbackLayer = new CFeedbackLayer("Feedback");
  FFeedbackLayer->canvas(this);
  FFeedbackLayer->addListener(&FListener);

  // Set a new lock on the font manager (and create it by the way if not yet done).
  lockFontManager();
}

//----------------------------------------------------------------------------------------------------------------------

CGenericCanvas::~CGenericCanvas(void)
{
  // Free any dynamically allocated memory. Explicitely increase update count. We don't want any recursive
  // notifcations anymore.
  ++FUpdateCount;
  FFeedbackLayer->beginUpdate();
  for (CLayers::iterator iterator = FLayers.begin(); iterator != FLayers.end(); ++iterator)
    delete *iterator;
  for (CViews::iterator iterator = FViews.begin(); iterator != FViews.end(); ++iterator)
    delete *iterator;

  delete FModel;
  delete FFeedbackLayer;

  // release the lock we placed in the constructor.
  // If this is the last GC instance then also the font manager can be released.
  unlockFontManager();
}

//----------------------------------------------------------------------------------------------------------------------

void CGenericCanvas::clearBuffers(void)
{
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @brief Adds the given layer at the end of the layer list. The new layer becomes the top layer in the view then.
 * @param layer The layer to add.
 */
void CGenericCanvas::addLayer(CLayer* layer)
{
  FLayers.push_back(layer);
  layer->canvas(this);
  change(this, GC_CHANGE_CANVAS_ADD_LAYER);
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Reads the layout info stored in the given (XML) file and creates figure templates.
 * Existing templates remain in where they are but are replaced if a new definition with an existing name is found.
 *
 * @param Filename The name of the file to load.
 * @return Returns GC_NO_ERROR if everything was ok, otherwise an error code.
 */
TGCError CGenericCanvas::addLayoutsFromFile(const char* Filename)
{
  TGCError result = GC_NO_ERROR;
  xmlNodePtr Root, current;
  CFigureParser Parser;

  string currentDir = getCurrentDir();
  xmlDocPtr Document = xmlParseFile(utf8ToANSI(string(Filename)).c_str());

  if (Document == NULL)
    return GC_XML_PARSE_ERROR;

  Root = xmlDocGetRootElement(Document);

  if (Root == NULL)
  {
    xmlFreeDoc(Document);
    error(this, "XML Error: Template file is empty.");
    return GC_XML_EMPTY_DOCUMENT;
  }
  
  if (!XML_IS(Root, "gc-layouts"))
  {
    xmlFreeDoc(Document);
    error(this, "XML Error: Template file invalid.");
    return GC_XML_INVALID_DOCUMENT;
  }

  // Switch to the directory of the given file. This is necessary to make relative file names working.
  string path = extractFilePath(Filename);
  setCurrentDir(path);

  // Parse description elements.
  glMatrixMode(GL_MODELVIEW);
  current = Root->children;
  while (current != NULL)
  {
    // Be flexible, ignore any unknown layout entries.
    if (XML_IS(current, "layout-definition"))
      Parser.parseLayoutDefinition(current, FModel);
    current = current->next;
  }

  setCurrentDir(currentDir);
  xmlFreeDoc(Document);

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Reads the style template info stored in the given (XML) file and creates style templates (OpenGL display lists).
 * Existing templates remain where they are but are replaced if a new definition with an existing name is found.
 *
 * @param Filename The name of the file to load.
 * @return Returns GC_NO_ERROR if everything was ok, otherwise an error code.
 */
TGCError CGenericCanvas::addStylesFromFile(const char* Filename)
{
  TGCError result = GC_NO_ERROR;
  xmlNodePtr Root, current;
  CSVGParser Parser;

  string currentDir = getCurrentDir();
  xmlDocPtr Document = xmlParseFile(utf8ToANSI(string(Filename)).c_str());

  if (Document == NULL)
    return GC_XML_PARSE_ERROR;

  Root = xmlDocGetRootElement(Document);

  if (Root == NULL)
  {
    xmlFreeDoc(Document);
    error(this, "XML Error: Template file is empty.");
    return GC_XML_EMPTY_DOCUMENT;
  }
  
  if (!XML_IS(Root, "gc-styles"))
  {
    xmlFreeDoc(Document);
    error(this, "XML Error: Template file invalid.");
    return GC_XML_INVALID_DOCUMENT;
  }

  // Switch to the directory of the given file. This is necessary to make relative file names working.
  string path = extractFilePath(Filename);
  setCurrentDir(path);

  // Parse description elements.
  glMatrixMode(GL_MODELVIEW);
  current = Root->children;
  while (current != NULL)
  {
    // Be flexible, ignore any unknown layout entries.
    if (XML_IS(current, "style-definition"))
    {
      Parser.parseDefinition(current, FModel);
    }
    else
      if (XML_IS(current, "texture"))
      {
        // Adds a new texture to the texture list.
        parseTextureEntry(current);
        checkError();
      };
    current = current->next;
  }

  setCurrentDir(currentDir);
  xmlFreeDoc(Document);

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds the given figure instance to the current selection. This call is simply forwared to the selection layer.
 * 
 * @param Instance The instance to be added to the selection. It is taken care that an instance is only added once.
 */
void CGenericCanvas::addToSelection(CFigureInstance* Instance)
{
  FFeedbackLayer->addToSelection(Instance);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Increases the update count by 1 to stop any recursive update until (@see endUpdate()) was called.
 */
void CGenericCanvas::beginUpdate(void)
{
  ++FUpdateCount;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks if there is an OpenGL error registered and triggers the error method if so.
 */
void CGenericCanvas::checkError(void)
{
  GLenum OGLError = glGetError();
  if (OGLError != GL_NO_ERROR)
  {
    char buffer[1000];
    #ifdef _WINDOWS
      // Microsoft has confirmed that there is no guaranteed NULL termination (see MSDN for details).
      // Hence we better add one.
      _snprintf(buffer, sizeof(buffer), "OpenGL error encountered (0x%x).\0", (int) OGLError);
    #else
      snprintf(buffer, sizeof(buffer), "OpenGL error encountered (0x%x).", (int) OGLError);
    #endif // #ifdef _WINDOWS
    error(this, buffer);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all GC content.
 */
void CGenericCanvas::clearContent(void)
{
  FModel->clearFigures();
  change(this, GC_CHANGE_CANVAS_CLEAR_CONTENT);
  change(this, GC_CHANGE_CANVAS_REFRESH);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all layout info.
 */
void CGenericCanvas::clearLayouts(void)
{
  FModel->clearLayouts();
  change(this, GC_CHANGE_CANVAS_CLEAR_LAYOUTS);
  change(this, GC_CHANGE_CANVAS_REFRESH);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all currently selected figure instances from the selection set. This call is simply forwareded to the
 * selection layer.
 */
void CGenericCanvas::clearSelection(void)
{
  FFeedbackLayer->clearSelection();
  change(this, GC_CHANGE_SELECTION_CLEAR);
  change(this, GC_CHANGE_CANVAS_REFRESH);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Causes the model to clear its style definitions.
 */
void CGenericCanvas::clearStyles(void)
{
  FModel->clearStyles();
  change(this, GC_CHANGE_CANVAS_CLEAR_STYLES);
  change(this, GC_CHANGE_CANVAS_REFRESH);
}

//----------------------------------------------------------------------------------------------------------------------

CFigure* CGenericCanvas::createFigure(const char* type, const char* class_)
{
  return FModel->createFigure(utf8ToUtf16(type), utf8ToUtf16(class_));
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a new layer with the given name and returns it to the caller. The new layer is added to this canvas.
 *
 * @param name The layer identification, encoded in UTF-8.
 * @param AddToCurrentView If true then the new layer will be added to the current view (if there is any).
 * @return The new layer.
 */
CLayer* CGenericCanvas::createLayer(const char* name, bool AddToCurrentView)
{
  CLayer* layer = new CLayer(name);
  addLayer(layer);
  if (FCurrentView != NULL && AddToCurrentView)
    FCurrentView->addLayer(layer);

  return layer;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a new view in this canvas, adds it to the internal list and returns it to the caller. If there is 
 * currently no current view set then the new view becomes the current one.
 *
 * @param name The identification of the view (encoded in UTF-8). Should be unique.
 * @return The new view.
 */
CGCView* CGenericCanvas::createView(const char* name)
{
  CGCView* View = new CGCView(this, name);
  FViews.push_back(View);
  View->addListener(&FListener);
  if (FCurrentView == NULL)
    currentView(View);

  return View;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the currently active view.
 *
 * @param View The new view to activate.
 */
void CGenericCanvas::currentView(CGCView* View)
{
  if (FCurrentView != View)
  {
    FCurrentView = View;
    if (FCurrentView != NULL)
      FStates |= GC_STATE_PENDING_ACTIVATION;
    else
      FStates &= ~GC_STATE_PENDING_ACTIVATION;
    change(FCurrentView, GC_CHANGE_CANVAS_SWITCH_VIEW);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the currently active view.
 *
 * @return The currently active view.
 */
CGCView* CGenericCanvas::currentView(void)
{
  return FCurrentView;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the given figure instance to execute its default action. This depends on which of the underlying figure
 * elements is hit by the given coordinates.
 *
 * @param Instance The figure instance whose figure is to be called.
 * @param X The horizontal window coordinate.
 * @param Y The vertical window coordinate.
 * @return True if the an action was executed, otherwise false.
 */
bool CGenericCanvas::doAction(CFigureInstance* Instance, const float X, const float Y)
{
  bool result = false;

  if (Instance != NULL && FCurrentView != NULL)
  {
    FCurrentView->applyTransformations(true);
    result = Instance->doAction(X, Y);
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * The counterpart to (@see beginUpdate). It releases one update lock and invalidates the canvas if the count drops to 0.
 */
void CGenericCanvas::endUpdate(void)
{
  if (FUpdateCount > 0)
    FUpdateCount--;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates and returns a new figure instance enumerator instance.
 *
 * @return The new enumerator.
 */
CFigureInstanceEnumerator* CGenericCanvas::getFigureInstanceEnumerator(void)
{
  return new CFigureInstanceEnumerator(this);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines whether the given position corresponds to any of the parts (body, handles) of a selection decoration.
 * This test is quite fast and can be used for cursor feedback and such. This method is simply forwarded to
 * CFeedbackLayer::getSelectionInfo after modelview and projection matrix are constructed with the current settings.
 *
 * @param X The horizontal mouse position in window coordinates.
 * @param Y The vertical mouse position in window coordinate.
 * @return One of the direction flags.
 */
TGCDirection CGenericCanvas::getSelectionInfo(const float X, const float Y)
{
  float Zoom = 1;

  if (FCurrentView != NULL)
  {
    FCurrentView->applyTransformations(true);
    Zoom = FCurrentView->feedbackZoom();
  };

  return FFeedbackLayer->getSelectionInfo(X, Y, Zoom);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the selection layer to recompute the bounds of the selection decoration for the given figure instance.
 *
 * @param Instance The figure instance for which to recompute the selection decoration.
 */
void CGenericCanvas::invalidateSelectionBounds(CFigureInstance* Instance)
{
  FFeedbackLayer->invalidateBounds(Instance);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the caller whether the canvas is currently being updated.
 */
bool CGenericCanvas::isUpdating(void)
{
  return FUpdateCount != 0;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the first layer with the given name from the layers collection.
 *
 * @param name The name of the layer to return.
 * @return The first found layer with the given name or NULL if no layer could be found.
 */
CLayer* CGenericCanvas::layerByName(const char* name)
{
  CLayer* result = NULL;

  wstring S = utf8ToUtf16(name);
  for (CLayers::const_iterator iterator = FLayers.begin(); iterator != FLayers.end(); ++iterator)
  {
    CLayer* layer = *iterator;
    if (layer->name() == S)
    {
      result = layer;
      break;
    };
  };
  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the property given by path. The path syntax is must be something like (here expressed as regex)
 * (container)*(property), where container is a slash and the name of a container class (e.g. layers, figures) and
 * property is the name of a simple property of that container.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this is the index into that list.
 * @return A description of the property value and, if the property is simple, the actual value.
 */
TGCVariant CGenericCanvas::property(const char* name, unsigned int index)
{
  TGCVariant result;

  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_NAME:
            {
              result.type = GC_VAR_STRING;
              result.s = utf16ToUtf8(FName);
              
              break;
            };
          case GC_PROPERTY_DESCRIPTION:
            {
              result.type = GC_VAR_STRING;
              result.s = "A generic 2D canvas based on OpenGL.";
              
              break;
            };
          case GC_PROPERTY_OWNER:
            {
              // There is no owner for the canvas in the canvas library itself.
              // However we recognize the request for it by returning NULL
              // instead of "unknown property".
              result.type = GC_VAR_OBJECT;
              
              break;
            };
        };
        break;
      };
    case GC_CONTAINER_LAYERS:
      {
        result.type = GC_VAR_LIST;
        if (index < FLayers.size())
          result.reference = (CGCBase*) FLayers[index];
        
        break;
      };
    case GC_CONTAINER_VIEWS:
      {
        result.type = GC_VAR_LIST;
        if (index < FViews.size())
          result.reference = (CGCBase*) FViews[index];
        
        break;
      };
    case GC_CONTAINER_MODEL:
      {
        result.type = GC_VAR_OBJECT;
        result.reference = (CGCBase*) FModel;

        break;
      };
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Set the value of the given property, which must be a simple property.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this is the index into that list.
 * @param value The new value of the property. Automatic conversion is performed where possible.
 */
void CGenericCanvas::property(const char* name, unsigned int index, const TGCVariant& value)
{
  switch (getPropertyID(name))
  {
    // There are no simple properties in the canvas that can be changed.
    // GC_PROPERTY_NAME is only a placeholder to easy future enhancements.
    case GC_PROPERTY_NAME:
      {
        FName = utf8ToUtf16(variantToString(value));
        break;
      };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given figure instance from the current selection. This call is simply forwared to the selection layer.
 *
 * @param Instance The figure instance to be removed from the selection. Nothing happens if it isn't selected.
 */
void CGenericCanvas::removeFromSelection(CFigureInstance* Instance)
{
  FFeedbackLayer->removeFromSelection(Instance);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given layer from the internal layer list. The layer itself will not be destroyed, just removed.
 *
 * @param layer The layer to be removed.
 */
void CGenericCanvas::removeLayer(CLayer* layer)
{
  for (CViews::const_iterator iterator = FViews.begin(); iterator != FViews.end(); ++iterator)
    (*iterator)->removeLayer(layer);

  layer->canvas(NULL);
  for (CLayers::iterator iterator = FLayers.begin(); iterator != FLayers.end(); ++iterator)
    if (*iterator == layer)
    {
      FLayers.erase(iterator);
      break;
    };
  change(this, GC_CHANGE_CANVAS_REMOVE_LAYER);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given view from the internal list.
 *
 * @param View The view to be removed.
 */
void CGenericCanvas::removeView(CGCView* View)
{
  if (FCurrentView == View)
    FCurrentView = NULL;
  for (CViews::iterator iterator = FViews.begin(); iterator != FViews.end(); ++iterator)
    if (*iterator == View)
    {
      (*iterator)->removeListener(&FListener);
      FViews.erase(iterator);
      break;
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * This is the main paint routine. It must be called by the viewer holding the reference to this canvas (e.g. when a 
 * window must be redrawn).
 */
void CGenericCanvas::render(void)
{
  // No display if the canvas is currently being updated.
  if (FUpdateCount == 0)
  {
    if ((FStates & GC_STATE_PENDING_ACTIVATION) != 0)
    {
      // A pending activation state always means there is a valid current view.
      FStates &= ~GC_STATE_PENDING_ACTIVATION;
      FCurrentView->activate();
    };

    clearBuffers();
    if (FCurrentView != NULL)
      FCurrentView->render();

    // The feedback layer uses the transformation of the current view!
    TBoundingBox dummy;
    FFeedbackLayer->render(1, dummy);

    checkError();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Preparation function for CFeedbackLayer::resizeFiguresStart.
 */
void CGenericCanvas::resizeFiguresStart(int X, int Y, TGCDirection Direction)
{
  // Initialize our transformation queue for convertion of the given screen coordinates into layer
  // coordinates (which happens in the feedback layer).
  if (FCurrentView != NULL)
    FCurrentView->applyTransformations(true);

  FFeedbackLayer->resizeFiguresStart(X, Y, Direction);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Preparation function for CFeedbackLayer::resizeFiguresStop.
 */
void CGenericCanvas::resizeFiguresStop(void)
{
  FFeedbackLayer->resizeFiguresStop();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Preparation function for CFeedbackLayer::resizeFiguresTo.
 */
void CGenericCanvas::resizeFiguresTo(int X, int Y)
{
  if (FCurrentView != NULL)
    FCurrentView->applyTransformations(false);

  FFeedbackLayer->resizeFiguresTo(X, Y);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Preparation function for CFeedbackLayer::rubberbandResize.
 */
void CGenericCanvas::rubberbandResize(int X, int Y, TRBSelectionAction Action)
{
  if (FCurrentView != NULL)
    FCurrentView->applyTransformations(true);

  FFeedbackLayer->rubberbandResize(X, Y, Action);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Preparation function for CFeedbackLayer::rubberbandStart.
 */
void CGenericCanvas::rubberbandStart(TRubberbandStyle Style, int X, int Y, bool clearSelection)
{
  // Initialize our transformation queue for conversion of the given screen coordinates into layer
  // coordinates (which happens in the feedback layer).
  if (FCurrentView != NULL)
    FCurrentView->applyTransformations(true);

  FFeedbackLayer->rubberbandStart(Style, X, Y, clearSelection);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Preparation function for CFeedbackLayer::rubberbandStop.
 */
void CGenericCanvas::rubberbandStop(void)
{
  FFeedbackLayer->rubberbandStop();
}

//----------------------------------------------------------------------------------------------------------------------

void CGenericCanvas::showSelection(bool visible)
{
  FFeedbackLayer->visible(visible);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the first view with the given name from the views collection.
 *
 * @param name The name of the view to return.
 * @return The first found view with the given name or NULL if no view could be found.
 */
CGCView* CGenericCanvas::viewByName(const char* name)
{
  CGCView* result = NULL;

  for (CViews::const_iterator iterator = FViews.begin(); iterator != FViews.end(); ++iterator)
  {
    CGCView* View = *iterator;
    if (View->FName == name)
    {
      result = View;
      break;
    };
  };
  return result;
}

//----------------------------------------------------------------------------------------------------------------------

