/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_svgparser.cpp
 * @brief Parser for SVG elements, which are converted to OpenGL calls.
 * 
 */

#include "myx_gc.h"

#include "myx_gc_svgparser.h"
#include "myx_gc_gl_helper.h"
#include "myx_gc_texture.h"
#include "myx_gc_font_manager.h"
#include "myx_gc_const.h"

//----------------------------------------------------------------------------------------------------------------------

typedef enum
{
  GC_PRIMITIVE_UNKNOWN = -1,
  GC_PRIMITIVE_RECT,
  GC_PRIMITIVE_LINE,
  GC_PRIMITIVE_POLYLINE,
  GC_PRIMITIVE_POLYGON,
  GC_PRIMITIVE_CIRCLE,
  GC_PRIMITIVE_TEXT,
  GC_PRIMITIVE_TSPAN,
  GC_PRIMITIVE_GROUP,
  GC_PRIMITIVE_IMAGE
} GC_PRIMITIVE;

//----------------------------------------------------------------------------------------------------------------------

// convert a primitive name to a number.
GC_PRIMITIVE FindPrimitive(string Name)
{
  static map<string, GC_PRIMITIVE> Primitives;

  if (Primitives.size() == 0)
  {
    // Initialize the map first if necessary.
    Primitives["rect"] = GC_PRIMITIVE_RECT;
    Primitives["line"] = GC_PRIMITIVE_LINE;
    Primitives["polyline"] = GC_PRIMITIVE_POLYLINE;
    Primitives["polygon"] = GC_PRIMITIVE_POLYGON;
    Primitives["circle"] = GC_PRIMITIVE_CIRCLE;
    Primitives["text"] = GC_PRIMITIVE_TEXT;
    Primitives["tspan"] = GC_PRIMITIVE_TSPAN;
    Primitives["g"] = GC_PRIMITIVE_GROUP;
    Primitives["image"] = GC_PRIMITIVE_IMAGE;
  };

  map<string, GC_PRIMITIVE>::iterator iterator = Primitives.find(Name);
  if (iterator == Primitives.end())
    return GC_PRIMITIVE_UNKNOWN;
  else
    return iterator->second;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scans the given string for coordinate values and fills the Points vector with them.
 */
void parsePoints(xmlChar* Raw, CVertexVector* Points)
{
  Points->clear();

  StringTokenizer tokenizer((char*) Raw, " ,\t\n");

  while (tokenizer.hasMoreTokens())
  {
    // Invalid input results in a value of 0, so we don't need to take special
    // care for such cases.
    TVertex v;
    v.x = tokenizer.nextTokenAsFloat();
    if (tokenizer.hasMoreTokens())
    {
      v.y = tokenizer.nextTokenAsFloat();
      Points->push_back(v);
    };
  };
}

//----------------- CSVGParser -----------------------------------------------------------------------------------------

CSVGParser::CSVGParser(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

CSVGParser::~CSVGParser(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the content of a circle definition.
 *
 * @see parseElement for a description of the parameters.
 */
void CSVGParser::parseCircle(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                             float StrokeWidth, CBoundingBoxComputer* BB)
{
  GLUquadricObj *quadric = gluNewQuadric();
  float innerRadius = getFloatAttributeDef(SVG, "inner-radius", 0);
  float outerRadius = getFloatAttributeDef(SVG, "outer-radius", 1);
  float startAngle = getFloatAttributeDef(SVG, "start-angle", 0);
  float sweepAngle = getFloatAttributeDef(SVG, "sweep-angle", 360);
  GLint Slices = getIntAttributeDef(SVG, "slices", 10);
  GLint Loops = getIntAttributeDef(SVG, "rings", 3);

  // Update the accumulated bounding box.
  GLfloat matrix[16];
  glGetFloatv(GL_MODELVIEW_MATRIX, matrix);
  BB->include(matrix, -outerRadius, -outerRadius);
  BB->include(matrix, outerRadius, outerRadius);

  // Disable polygon smoothing. We cannot have both here, alpha blended figures and polygon smoothing.
  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_POLYGON_SMOOTH);

  gluQuadricOrientation(quadric, GLU_INSIDE);

  if (FillColor != NULL)
    glColor4ubv(FillColor);

  if (DoFill)
  {
    gluQuadricDrawStyle(quadric, GLU_FILL);
    gluQuadricNormals(quadric, GLU_SMOOTH);
    gluQuadricTexture(quadric, GL_TRUE);

    gluPartialDisk(quadric, innerRadius, outerRadius, Slices, Loops, startAngle, sweepAngle);
  };

  if (DoStroke)
  {
    if (StrokeColor != NULL)
      glColor4ubv(StrokeColor);

    gluQuadricDrawStyle(quadric, GLU_SILHOUETTE);
    gluPartialDisk(quadric, innerRadius, outerRadius, Slices, Loops, startAngle, sweepAngle);
  };

  gluDeleteQuadric(quadric);
  glPopAttrib();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Recursively called function that parses the given SVG xml element and converts it to OpenGL calls.
 *
 * @param SVG Any of the supported primitives.
 * @param DoFill The parent's fill flag. If this is true then also this element is filled.
 * @param FillColor The parent's fill color. Only used if the we have a local opacity.
 * @param DoStroke The parent's outline flag. If this true then also this element is outlined.
 * @param StrokeColor The parent's stroke color. Only used if the we have a local opacity.
 * @param StrokeWidth The parent's stroke width. Used only if we draw an outline at all and no local width is given.
 * @param MasterAlpha The accumulated alpha value, which is currently active. Used in conjunction with a local opacity.
 * @param BB This calculator is used to determine the overall bounding box of the element being parsed.
 * @return Returns a new display list for the content of this element.
 */
GLuint CSVGParser::parseElement(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                                float StrokeWidth, float MasterAlpha, CBoundingBoxComputer* BB)
{
  GC_PRIMITIVE Primitive = FindPrimitive((char*) SVG->name);
  CGCTexture* Texture = NULL;
  GLuint ResultList = 0;
                                    
  if (Primitive != GC_PRIMITIVE_UNKNOWN)
  {
    bool Display = true;

    xmlChar* Attribute = xmlGetProp(SVG, (xmlChar*) "display");
    if (Attribute != NULL)
    {
      Display = xmlStrcmp(Attribute, (const xmlChar *) "none") != 0;
      xmlFree(Attribute);
    };
    
    if (Display)
    {
      Attribute = xmlGetProp(SVG, (xmlChar*) "texture");
      if (Attribute != NULL)
      {
        Texture = textureManager()->FindTexture(string((char*) Attribute));
        xmlFree(Attribute);
      };

      // See if we have a local color and/or a local opacity.
      float Opacity = MasterAlpha;
      bool hasOpacity = getFloatAttribute(SVG, "opacity", Opacity);
      if (hasOpacity)
        Opacity *= MasterAlpha;

      GLubyte localFillColor[4];
      GLubyte* fillColorPtr = NULL;

      int Conversion = convertColor(SVG, "fill", localFillColor);
      // Fill this element if either filling is not completely disabled (fill ="none")
      // and either a local color is given or the parent element used filling already.
      bool localFill = (Conversion != 3) && (DoFill || FillColor != NULL || Conversion == 0);
      if (localFill)
      {
        // If there is no valid local color then use that of the parent.
        if (Conversion != 0)
        {
          // Combine the parent color with this opacity to create a new color.
          localFillColor[0] = FillColor[0];
          localFillColor[1] = FillColor[1];
          localFillColor[2] = FillColor[2];
        };
        localFillColor[3] = GLubyte(255 * Opacity);
        fillColorPtr = localFillColor;
      };

      GLubyte localStrokeColor[4];
      GLubyte* strokeColorPtr = NULL;

      Conversion = convertColor(SVG, "stroke", localStrokeColor);
      bool LocalStroke = (Conversion != 3) && (DoStroke || StrokeColor != NULL || Conversion == 0);
      if (LocalStroke)
      {
        // If there is no valid local color then use that of the parent.
        if (Conversion != 0)
        {
          // Combine the parent color with this opacity to create a new color.
          localStrokeColor[0] = StrokeColor[0];
          localStrokeColor[1] = StrokeColor[1];
          localStrokeColor[2] = StrokeColor[2];
        };
        localStrokeColor[3] = GLubyte(255 * Opacity);
        strokeColorPtr = localStrokeColor;
      };

      float LocalStrokeWidth = getFloatAttributeDef(SVG, "stroke-width", StrokeWidth);

      // Prepare child display lists before starting with the one for this element.
      GLuint LocalList = 0;
      switch (Primitive)
      {
        case GC_PRIMITIVE_TEXT:
        case GC_PRIMITIVE_TSPAN:
          {
            LocalList = parseText(SVG, localFill, fillColorPtr, LocalStroke, strokeColorPtr, LocalStrokeWidth, Opacity, BB);
            break;
          };
        case GC_PRIMITIVE_GROUP:
          {
            LocalList = parseGroup(SVG, localFill, fillColorPtr, LocalStroke, strokeColorPtr, LocalStrokeWidth, Opacity, BB);
            break;
          };
        case GC_PRIMITIVE_IMAGE:
          {
            parseImage(SVG, BB, false);
            break;
          };
      };

      ResultList = glGenLists(1);
      // We need the display listed executed while being compiled to have the modelview matrix updated correctly.
      // This is needed to compute the bounding box correctly.
      glNewList(ResultList, GL_COMPILE_AND_EXECUTE);

      glPushMatrix();

      if (Texture != NULL)
        Texture->ActivateTexture();

      Attribute = xmlGetProp(SVG, (xmlChar*) "transform");
      if (Attribute != NULL)
      {
        parseTransformation((char*) Attribute);
        xmlFree(Attribute);
      };

      switch (Primitive) 
      {
        case GC_PRIMITIVE_RECT:
          {
            parseRectangle(SVG, localFill, fillColorPtr, LocalStroke, strokeColorPtr, LocalStrokeWidth, BB);
            break;
          };
        case GC_PRIMITIVE_LINE:
          {
            parseLine(SVG, LocalStroke, strokeColorPtr, LocalStrokeWidth, BB);
            break;
          };
        case GC_PRIMITIVE_POLYLINE:
          {
            parsePolyline(SVG, localFill, fillColorPtr, LocalStroke, strokeColorPtr, LocalStrokeWidth, BB);
            break;
          };
        case GC_PRIMITIVE_POLYGON:
          {
            parsePolygon(SVG, localFill, fillColorPtr, LocalStroke, strokeColorPtr, LocalStrokeWidth, BB);
            break;
          };
        case GC_PRIMITIVE_CIRCLE:
          {
            parseCircle(SVG, localFill, fillColorPtr, LocalStroke, strokeColorPtr, LocalStrokeWidth, BB);
            break;
          };
        case GC_PRIMITIVE_TEXT:
        case GC_PRIMITIVE_TSPAN:
        case GC_PRIMITIVE_GROUP:
          {
            if (LocalList != 0)
              glCallList(LocalList);
            break;
          };
        case GC_PRIMITIVE_IMAGE:
          {
            parseImage(SVG, BB, true);
            break;
          };
      };

      if (Texture != NULL)
        Texture->DeactivateTexture();

      glPopMatrix();

      glEndList();
    };
  };

  return ResultList;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Collects all data in an <svg:g> element.
 *
 * @return A new display list comprising all subelements.
 * @see parseElement for the description of the parameters.
 */
GLuint CSVGParser::parseGroup(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                              float StrokeWidth, float MasterAlpha, CBoundingBoxComputer* BB)
{
  xmlNodePtr child = SVG->children;
  vector<GLuint> lists;
  while (child != NULL)
  {
    if (child->type == XML_ELEMENT_NODE)
    {
      GLuint List = parseElement(child, DoFill, FillColor, DoStroke, StrokeColor, StrokeWidth, MasterAlpha, BB);
      lists.push_back(List);
    };
    child = child->next;
  };

  GLuint ResultList = glGenLists(1);
  glNewList(ResultList, GL_COMPILE);
  for (vector<GLuint>::iterator iterator = lists.begin(); iterator != lists.end(); iterator++)
    glCallList(*iterator);
  glEndList();

  return ResultList;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the content of an image definition.
 *
 * @param SVG The XML SVG element containing the definition.
 * @param Render If true then the appropriate OpenGL calls for rendering are issued, otherwise setup is performed.
 */
void CSVGParser::parseImage(xmlNodePtr SVG, CBoundingBoxComputer* BB, bool Render)
{
  string Filename = getStringAttributeDef(SVG, "href", "");
  string id = getStringAttributeDef(SVG, "id", Filename);

  if (!Render)
  {
    // Create the image only if not yet done.
    CGCTexture* Texture = textureManager()->FindTexture(id);
    if (Texture == NULL)
    {
      TLODList LODs;
      LODs.push_back(Filename);

      Texture = textureManager()->CreateTextureEntry(LODs, id, "clamp-to-edge", "clamp-to-edge", "linear-mipmap-linear", 
        "linear", 2, "modulate");
    };
  }
  else
  {
    float x = getFloatAttributeDef(SVG, "x", 0);
    float y = getFloatAttributeDef(SVG, "y", 0);
    float Width = getFloatAttributeDef(SVG, "width", 0);
    float Height = getFloatAttributeDef(SVG, "height", 0);

    // Update the accumulated bounding box.
    GLfloat matrix[16];
    glGetFloatv(GL_MODELVIEW_MATRIX, matrix);
    BB->include(matrix, x, y);
    BB->include(matrix, x + Width, y + Height);

    CGCTexture* Texture = textureManager()->FindTexture(id);
    if (Texture)
      Texture->ActivateTexture();

    glColor4f(1, 1, 1, 1);
    glTranslatef(x, y, 0);
    glBegin(GL_POLYGON);
      glTexCoord2d(0, 0);
      glVertex2f(0, 0);
      glTexCoord2d(1, 0);
      glVertex2f(Width, 0);
      glTexCoord2d(1, 1);
      glVertex2f(Width, Height);
      glTexCoord2d(0, 1);
      glVertex2f(0, Height);
    glEnd();

    if (Texture)
      Texture->DeactivateTexture();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the content of a line definition.
 *
 * @param SVG The XML SVG element containing the definition.
 * @param DoStroke Flag to indicate if the line is to be drawn or not.
 * @param StrokeColor The color to be used for the line (if set).
 * @param StrokeWidth The width of the line.
 * @param BB This calculator is used to determine the overall bounding box of the element being parsed.
 */
void CSVGParser::parseLine(xmlNodePtr SVG, bool DoStroke, GLubyte* StrokeColor, float StrokeWidth, CBoundingBoxComputer* BB)
{
  float X1 = getFloatAttributeDef(SVG, "x1", 0);
  float Y1 = getFloatAttributeDef(SVG, "y1", 0);
  float X2 = getFloatAttributeDef(SVG, "x2", 0);
  float Y2 = getFloatAttributeDef(SVG, "y2", 0);

  // Update the accumulated bounding box.
  GLfloat matrix[16];
  glGetFloatv(GL_MODELVIEW_MATRIX, matrix);
  BB->include(matrix, X1, Y1);
  BB->include(matrix, X2, Y2);

  if (DoStroke)
  {
    if (StrokeColor != NULL)
      glColor4ubv(StrokeColor);
    glLineWidth(StrokeWidth);

    glBegin(GL_LINES);
      glVertex2f(X1, Y1);
      glVertex2f(X2, Y2);
    glEnd();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the content of a polygon definition.
 *
 * @see parseElement for a description of the parameters.
 */
void CSVGParser::parsePolygon(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                              float StrokeWidth, CBoundingBoxComputer* BB)
{
  xmlChar* Raw = xmlGetProp(SVG, (xmlChar*) "points");
  if (Raw != NULL)
  {
    CVertexVector Vertices;
    parsePoints(Raw, &Vertices);
    xmlFree(Raw);

    renderVertices(DoFill, FillColor, DoStroke, StrokeColor, Vertices, NULL, StrokeWidth, true, BB);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the content of a polyline definition.
 *
 * @see parseElement for a description of the parameters.
 */
void CSVGParser::parsePolyline(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                               float StrokeWidth, CBoundingBoxComputer* BB)
{
  xmlChar* Raw = xmlGetProp(SVG, (xmlChar*) "points");
  if (Raw != NULL)
  {
    CVertexVector Vertices;
    parsePoints(Raw, &Vertices);
    xmlFree(Raw);

    renderVertices(DoFill, FillColor, DoStroke, StrokeColor, Vertices, NULL, StrokeWidth, false, BB);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the content of a rectangle definition.
 *
 * @see parseElement for a description of the parameters.
 */
void CSVGParser::parseRectangle(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                                float StrokeWidth, CBoundingBoxComputer* BB)
{
  float x, y, Width, Height, RX, RY;

  if (!getFloatAttribute(SVG, "x", x))
    x = 0;
  if (!getFloatAttribute(SVG, "y", y))
    y = 0;
  if (!getFloatAttribute(SVG, "width", Width))
    Width = 1;
  if (!getFloatAttribute(SVG, "height", Height))
    Height = 1;

  // There are a few more things to check with the corner radii.
  bool RXfound = getFloatAttribute(SVG, "rx", RX);
  bool RYfound = getFloatAttribute(SVG, "ry", RY);
  if (!RXfound && !RYfound)
  {
    RX = 0;
    RY = 0;
  }
  else
  {
    if (!RXfound)
      RX = RY;
    else
      if (!RYfound)
        RY = RX;
    if (RX > Width / 2)
      RX = Width / 2;
    if (RY > Width / 2)
      RY = Width / 2;
  };

  // TODO: Consider rounded corners.
  CVertexVector Vertices;
  TVertex v;
  v.x = x;
  v.y = y;
  Vertices.push_back(v);
  v.x = x + Width;
  Vertices.push_back(v);
  v.y = y + Height;
  Vertices.push_back(v);
  v.x = x;
  Vertices.push_back(v);

  CVertexVector TextureCoordinates;
  v.x = 0;
  v.y = 0;
  TextureCoordinates.push_back(v);
  v.x = Width;
  TextureCoordinates.push_back(v);
  v.y = Height;
  TextureCoordinates.push_back(v);
  v.x = 0;
  TextureCoordinates.push_back(v);

  renderVertices(DoFill, FillColor, DoStroke, StrokeColor, Vertices, &TextureCoordinates, StrokeWidth, true, BB);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Takes a text node and gets all attributes for direct or dynamic rendering. This function is called recursively
 * and can take either a <text> or a <tspan> node.
 *
 * @param SVG The text or tspan node to parse.
 * @param FillColor The color for the text interior.
 * @param StrokeColor The color of the outline.
 * @param The width of the outlines.
 * @param MasterAlpha Only passed through because text nodes can have children.
 * @param BB This calculator is used to determine the overall bounding box of the element being parsed.
 * @return A new display list comprising all subelements.
 */
GLuint CSVGParser::parseText(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                             float StrokeWidth, float MasterAlpha, CBoundingBoxComputer* BB)
{
  // Collect font information.
  string FontFamily = getStringAttributeDef(SVG, "font-family", DefaultFontFamily);
  string FontStyle = getStringAttributeDef(SVG, "font-style", DefaultFontStyle);
  string FontWeightString = getStringAttributeDef(SVG, "font-weight", DefaultFontWeight);
  int Weight = convertFontWeight(FontWeightString);
  int FontSize = getIntAttributeDef(SVG, "font-size", DefaultFontSize);

  vector<GLuint> lists;
  xmlNodePtr child = SVG->children;
  while (child != NULL)
  {
    if (XML_IS(child, "tspan"))
    {
      GLuint LocalList = parseElement(child, DoFill, FillColor, DoStroke, StrokeColor, StrokeWidth, MasterAlpha, BB);
      lists.push_back(LocalList);
    }
    else
    {
      string Source((char*) child->content);
      wstring Output = utf8ToUtf16(Source);

      // Make the font manager create all display lists before we build ours.
      TBoundingBox box;
      fontManager()->boundingBox(Output, FontFamily, FontStyle, Weight, FontSize, &box);

      // Update the accumulated bounding box.
      GLfloat matrix[16];
      glGetFloatv(GL_MODELVIEW_MATRIX, matrix);
      BB->include(matrix, &box);

      GLuint LocalList = glGenLists(1);
      glNewList(LocalList, GL_COMPILE);
        glColor3f(1, 0, 1);
        // FTGL uses the lower left corner as origin, we however have the upper left corner for this task.
        // Adjust the vertical coordinate accordingly.
        glRasterPos2f(0, box.lower.y);
        fontManager()->textOut(Output, FontFamily, FontStyle, Weight, FontSize);
      glEndList();
      lists.push_back(LocalList);
    };

    child = child->next;
  };

  // Create final display list.
  GLuint DisplayList = glGenLists(1);
  glNewList(DisplayList, GL_COMPILE);

  float x = getFloatAttributeDef(SVG, "x", 0);
  float y = getFloatAttributeDef(SVG, "y", 0);

  glTranslatef(x, y, 0);

  if (FillColor !=  NULL)
    glColor4ubv(FillColor);

  for (vector<GLuint>::iterator iterator = lists.begin(); iterator != lists.end(); iterator++)
    glCallList(*iterator);
  glEndList();

  return DisplayList;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parsers the given string and interprets the content as one or more transformation of the form:
 *   translate(x, y, z) scale(x, y, z) etc.
 *
 * @param Transformation The textual representation of the transformation to parse and perform.
 */
void CSVGParser::parseTransformation(char* Transformation)
{
  string Delimiters(" ,()\t\n");
  StringTokenizer tokenizer(Transformation, Delimiters);

  // Each run in the outer loop is one transformation.
  while (tokenizer.hasMoreTokens())
  {
    // Skip leading white spaces.
    string Token = tokenizer.nextToken();

    int Type = 0;
    if (Token == "matrix")
      Type = 1;
    else
      if (Token == "translate")
        Type = 2;
      else
        if (Token == "scale")
          Type = 3;
        else
          if (Token == "rotate")
            Type = 4;
          else
            if (Token == "skewX")
              Type = 5;
            else
              if (Token == "skewY")
                Type = 6;
              else
                break;

    switch (Type)
    {
      case 1: // matrix
        {
          // Argument list found. Read 6 float numbers.
          // | a  d  0  0 |
          // | b  e  0  0 |
          // | 0  0  1  0 |
          // | c  f  0  1 |
          GLfloat matrix[16] = {
            1, 0, 0, 0, 
            0, 1, 0, 0, 
            0, 0, 1, 0, 
            0, 0, 0, 1
          };

          for (int J = 0; J < 2; J++)
            for (int I = 0; I < 2; I++)
              matrix[J * 4 + I] = tokenizer.nextTokenAsFloat();
          matrix[12] = tokenizer.nextTokenAsFloat();
          matrix[13] = tokenizer.nextTokenAsFloat();
          glMultMatrixf(matrix);

          break;
        };
      case 2: // Translation
        {
          // One or two floats are possible here.
          float TX = tokenizer.nextTokenAsFloat();
          float TY = 0;
          if (tokenizer.hasMoreTokens())
            TY = tokenizer.nextTokenAsFloat();
          glTranslatef(TX, TY, 0);

          break;
        };
      case 3: // Scaling
        {
          // One or two floats are possible here.
          float SX = tokenizer.nextTokenAsFloat();
          float SY = SX;
          if (tokenizer.hasMoreTokens())
            SY = tokenizer.nextTokenAsFloat();
          glScalef(SX, SY, 0);

          break;
        };
      case 4: // Rotation
        {
          // An angle (in degrees) and an optional rotation point are possible here.
          float Angle = tokenizer.nextTokenAsFloat();
          float x = 0;
          float y = 0;
          if (tokenizer.hasMoreTokens())
          {
            x = tokenizer.nextTokenAsFloat();
            y = tokenizer.nextTokenAsFloat();
          };

          glTranslatef(x, y, 0);
          glRotatef(Angle, 0, 0, 1);
          glTranslatef(-x, -y, 0);

          break;
        };
      case 5: // x Skewing
      case 6: // y Skewing
        {
          // Only one value is expected, which gives the skewing angle in degrees.
          GLdouble matrix[16] = {
            1, 0, 0, 0, 
            0, 1, 0, 0, 
            0, 0, 1, 0, 
            0, 0, 0, 1
          };

          int Index = (Type == 5) ? 4 : 1;
          matrix[Index] = tan(tokenizer.nextTokenAsFloat() * M_PI / 180);
          glMultMatrixd(matrix);
          break;
        };
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Renders the given set of vertices.
 *
 * @param DoFill Filled output is only done if this flag is true. For strokes the existance of the color as such is used
 *               as indicator.
 * @param FillColor If not NULL then it gives the local color for this call, otherwise the current color stays untouched.
 * @param StrokeColor If not nULL then the vertices are render again as lines but using that color.
 * @param Vertices The vertex data to render.
 * @param TextureCoordinates If given (can be NULL) then there must be exactly the same number of texture coordinates
 *                           as there are vertices.
 * @param StrokeWidth The width of the border if one is rendered.
 * @param CloseShape Determines whether the border line (if StrokeColor is given) is closed (that is, the last point is
 *                   connected to the first point).
 * @param BB This calculator is used to determine the overall bounding box of the element being parsed.
 */
void CSVGParser::renderVertices(bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, 
                                const CVertexVector& Vertices, CVertexVector* TextureCoordinates, float StrokeWidth, 
                                bool CloseShape, CBoundingBoxComputer* BB)
{
  GLfloat matrix[16];
  glGetFloatv(GL_MODELVIEW_MATRIX, matrix);

  if (DoStroke)
  {
    if (StrokeColor != NULL)
      glColor4ubv(StrokeColor);
    if (StrokeWidth <= 1)
    {
      glBegin(CloseShape ? GL_LINE_LOOP : GL_LINE_STRIP);
      for (int I = 0; I < (int) Vertices.size(); I++)
        glVertex2f(Vertices[I].x, Vertices[I].y);
      glEnd();
    }
    else
    {
      // Render small rectangles for each line.
      float HalfWidth = StrokeWidth / 2;
      vector<TVertex>::size_type Index = 0;
      vector<TVertex>::size_type Count = Vertices.size();
      if (!CloseShape)
        Count--;
      while (Index < Count)
      {
        vector<TVertex>::size_type NextIndex = Index + 1;
        if (Index == Vertices.size() - 1)
          NextIndex = 0;
        // We need the angle of the current line relative to the horizontal axis.
        float dX = Vertices[NextIndex].x - Vertices[Index].x;
        float dY = Vertices[NextIndex].y - Vertices[Index].y;
        float Angle = atan2(dY, dX);

        // Compute the four corners for the rectangle. We can use symmetry to speed up computation.
        dX = HalfWidth * sin(Angle);
        dY = HalfWidth * cos(Angle);
        glBegin(GL_POLYGON);
          glVertex2f(Vertices[Index].x - dX, Vertices[Index].y + dY);
          // Update the accumulated bounding box.
          BB->include(matrix, Vertices[Index]);

          glVertex2f(Vertices[NextIndex].x - dX, Vertices[NextIndex].y + dY);
          BB->include(matrix, Vertices[Index]);

          glVertex2f(Vertices[NextIndex].x + dX, Vertices[NextIndex].y - dY);
          BB->include(matrix, Vertices[Index]);

          glVertex2f(Vertices[Index].x + dX, Vertices[Index].y - dY);
          BB->include(matrix, Vertices[Index]);
        glEnd();

        Index++;
      };
    };
  };

  if (FillColor != NULL)
    glColor4ubv(FillColor);

  if (DoFill)
  {
    glBegin(GL_POLYGON);
    for (int I = 0; I < (int) Vertices.size(); I++)
    {
      if (TextureCoordinates != NULL)
        glTexCoord2d((*TextureCoordinates)[I].x, (*TextureCoordinates)[I].y);
      glVertex2f(Vertices[I].x, Vertices[I].y);

      // Update the accumulated bounding box.
      BB->include(matrix, Vertices[I]);
    };
    glEnd();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the given SVG xml description and converts it to OpenGL calls. This method can be called within an active
 * OpenGL display list compilation (but not between glBegin/glEnd).
 *
 * @param SVG The SVG top level element (<svg:svg> </svg:svg>).
 * @param DisplayList The OpenGL display list to render into.
 * @param BB The bounding box computer to use to compute the overall bounding box of the element.
 * @note In order to render correctly the svg:svg element must have width and height attributes set. Particularly the 
 *       height attribute is needed to convert from SVG's top-down to OpenGL's bottom-up coordinate system.
 *       Without height attribute the element will be drawn head-down.
 */
void CSVGParser::convert(xmlNodePtr SVG, GLuint DisplayList, CBoundingBoxComputer* BB)
{
  vector<GLuint> lists;
  xmlNodePtr child = SVG->children;
  while (child != NULL)
  {
    if (child->type == XML_ELEMENT_NODE)
    {
      GLuint LocalList = parseElement(child, false, NULL, false, NULL, 1, 1, BB);
      lists.push_back(LocalList);
    };
    child = child->next;
  };

  // Not necessary for creating the list but used to create a clean starting point to transform bounding boxes.
  glLoadIdentity();

  glNewList(DisplayList, GL_COMPILE);

  glColor4f(0, 0, 0, 1);

  for (vector<GLuint>::iterator iterator = lists.begin(); iterator != lists.end(); iterator++)
    glCallList(*iterator);
  glEndList();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the given style definition and creates a new style, which is then added to the given model.
 *
 * @param Definition The definition to parse.
 * @Model The model to which the new style is to be added.
 */
void CSVGParser::parseDefinition(xmlNodePtr Definition, CGCModel* Model)
{
  string id; 
  if (getStringAttribute(Definition, "id", id))
  {
    // Handle sub entries of the layout definition.
    xmlNodePtr Element = Definition->children;

    // This call will create an entry if there is none yet.
    CGCStyle* Style = Model->style(utf8ToUtf16(id));
    if (Style->FDisplayList == 0)
      Style->FDisplayList = glGenLists(1);
    while (Element != NULL)
    {
      if (Element->type == XML_ELEMENT_NODE)
      {
        if (XML_IS(Element, "svg"))
        {
          CBoundingBoxComputer BBComputer;
          convert(Element, Style->FDisplayList, &BBComputer);
          Style->FBoundingBox = BBComputer.boundingBox();
          Model->checkError();
        };

        // Ignore everything else.
        break;
      };
      Element = Element->next;
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

