/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#include "myx_grt_base.h"

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  #include <shlobj.h>
  #include <shellapi.h>
#endif

// --------------------------------------------------------------------------
// module registration functions

MYX_GRT_MODULE* myx_register_builtin_grt_module_base(MYX_GRT *grt)
{
  return myx_grt_module_register_builtin(grt, &grt_module_base, grt);
}

// --------------------------------------------------------------------------

/**
 ****************************************************************************
 * @brief Returns a GUID
 *
 *   Creates a globally unique indentifier
 *
 * @param param nil
 * @param data pointer to the Grt
 * 
 * @return  A GUID
 *****************************************************************************/
MYX_GRT_VALUE * get_guid(MYX_GRT_VALUE *param, void *data)
{
  char *guid= myx_grt_get_guid();
  MYX_GRT_VALUE *value= myx_grt_value_from_string(guid);

  g_free(guid);
  return make_return_value(value);
}

/**
 ****************************************************************************
 * @brief Returns the application's data directory
 *
 *   This returns c:\documents and settings\[user]\application data\mysql on
 * on Windows and ~/.mysql on Linux
 *
 * @param param nil
 * @param data pointer to the Grt
 * 
 * @return the application's data directory
 *****************************************************************************/
MYX_GRT_VALUE * get_app_data_dir(MYX_GRT_VALUE *param, void *data)
{
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  LPITEMIDLIST pidl;
  char path[MAX_PATH];
  char *data_dir;
  MYX_GRT_VALUE *value;

  SHGetFolderLocation(NULL, CSIDL_APPDATA, NULL, 0, &pidl);

  SHGetPathFromIDList(pidl, path);

  CoTaskMemFree(pidl);

  data_dir= g_strdup_printf("%s\\MySQL\\", path);

  value= myx_grt_value_from_string(data_dir);

  g_free(data_dir);

  return make_return_value(value);
#else
  return make_return_value(myx_grt_value_from_string("~/.mysql/"));
#endif
}

/**
 ****************************************************************************
 * @brief Returns the type name of the OS
 *
 *   Returns WINDOWS on Windows, OSX on a mac and UNIX on all other platforms
 *
 * @param param nil
 * @param data pointer to the Grt
 * 
 * @return type name of the OS
 *****************************************************************************/
MYX_GRT_VALUE * get_os_type_name(MYX_GRT_VALUE *param, void *data)
{
  MYX_GRT_VALUE *value;

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  value= myx_grt_value_from_string("WINDOWS");
#elif defined(__APPLE__)
  value= myx_grt_value_from_string("OSX");
#else
  value= myx_grt_value_from_string("UNIX");
#endif

  return make_return_value(value);
}