unit GrtDbConn;

interface

uses
  GnuGetText, Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, Buttons, TntButtons, StdCtrls, TntStdCtrls, VirtualTrees,
  ExtCtrls, TntExtCtrls, ComCtrls, PNGImage, PNGTools,
  AuxFuncs, TntForms, TntClasses, Grt,
  myx_grt_public_interface, Menus, TntMenus;

{$include Consts.ini}

const ParamLineHeight = 32;

type
  TGrtDbConnForm = class(TTntForm)
    HeaderPnl: TTntPanel;
    HeaderImg: TTntImage;
    ParamsMainPnl: TTntPanel;
    AdvParamsMainPnl: TTntPanel;
    ConnectGBox: TTntGroupBox;
    BottomPnl: TTntPanel;
    OKBtn: TTntButton;
    CancelBtn: TTntButton;
    ClearFieldsBtn: TTntButton;
    AdvancedBtn: TTntBitBtn;
    ConnTypeMainPnl: TTntPanel;
    ConnTypeGBox: TTntGroupBox;
    ConnTypePnl: TTntPanel;
    DriverLbl: TTntLabel;
    DriverComboBox: TTntComboBox;
    RdbmsComboBox: TTntComboBox;
    RdbmsLbl: TTntLabel;
    StoredConnPnl: TTntPanel;
    ConnectionLbl: TTntLabel;
    StoredConnCBox: TTntComboBox;
    StoredConnDelBtn: TTntSpeedButton;
    StoredConnAddBtn: TTntSpeedButton;
    ParamsPnl: TTntPanel;
    ConnectToInstanceAni: TAnimate;
    DriverNotInstalledPnl: TTntPanel;
    TntLabel2: TTntLabel;
    TntLabel9: TTntLabel;
    LocateJdbcDriverBtn: TTntButton;
    DownloadDriverPnl: TTntPanel;
    TntLabel12: TTntLabel;
    DownloadJdbcDriverBtn: TTntButton;
    LookupMenu: TTntPopupMenu;
    TntGroupBox1: TTntGroupBox;
    AdvParamsPnl: TTntPanel;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure DriverComboBoxCloseUp(Sender: TObject);
    procedure RdbmsComboBoxCloseUp(Sender: TObject);
    procedure StoredConnCBoxCloseUp(Sender: TObject);
    procedure AdvancedBtnClick(Sender: TObject);
    procedure OKBtnClick(Sender: TObject);
    procedure CancelBtnClick(Sender: TObject);
  private
    { Private declarations }
    FGrt: TGrt;
    FComponentParamMapping: TTntStringList;
    FSettingConnectionValues,
      FDisplayRdbmsSelection,
      FDisplayDescriptions,
      FDisplaySchemaSelection: Boolean;

    FConnInfoPath,
      FConnTargetPath: WideString;

    FOkBtn: TTntButton;

    procedure FreeChildControls(Control: TWinControl);
    procedure BuildDriverControls(
      Target: TWinControl; PDriver: Pointer;
      DoAdvancedParams: Boolean = False);

    procedure DriverParamValueChanged(Sender: TObject);
    procedure ParamLookupBtnClick(Sender: TObject);
    procedure ParamLookupMIClick(Sender: TObject);
    procedure BrowseDirBtnClick(Sender: TObject);
    procedure BrowseFileBtnClick(Sender: TObject);

    function BuildDriverComboBoxItems(PRdbms: Pointer): Integer;

    procedure FillDropdownWithStoredConnections(
      StoredConnComboBox: TTntComboBox; PDriver: Pointer);

    procedure SetConnection(PConnection: Pointer);

    procedure SetSelectedRdbms(SelectedRdbms: WideString);
    function GetSelectedRdbms: WideString;

    procedure SetDisplayRdbmsSelection(DisplayRdbmsSelection: Boolean);

    procedure WriteConnectionToTarget;
  public
    { Public declarations }
    property ConnInfoPath: WideString read FConnInfoPath write FConnInfoPath;
    property ConnTargetPath: WideString read FConnTargetPath write FConnTargetPath;

    property DisplayRdbmsSelection: Boolean read FDisplayRdbmsSelection write SetDisplayRdbmsSelection;
    property DisplayDescriptions: Boolean read FDisplayDescriptions write FDisplayDescriptions;
    property DisplaySchemaSelection: Boolean read FDisplaySchemaSelection write FDisplaySchemaSelection;

    property OkButton: TTntButton read FOkBtn write FOkBtn;
    property Grt: TGrt read FGrt write FGrt;
    property SelectedRdbms: WideString read GetSelectedRdbms write SetSelectedRdbms;

    procedure RefreshConnInfo;
  end;

var
  GrtDbConnForm: TGrtDbConnForm;

implementation

{$R *.dfm}

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.FormCreate(Sender: TObject);

begin
  InitForm(self);

  //Load resources
  LoadPNGImageFromResource('connection_dialog_header', HeaderImg, True);
  ConnectToInstanceAni.ResName := 'progress_indicator';

  Caption := Application.Title + ' '+
    product_version + ' ' +
    product_build_level;

  FComponentParamMapping := TTntStringList.Create;
  FSettingConnectionValues := False;
  FDisplayRdbmsSelection := False;
  FDisplayDescriptions := False;
  FDisplaySchemaSelection := True;

  FOkBtn := OKBtn;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.FormDestroy(Sender: TObject);

begin
  FComponentParamMapping.Free;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.RefreshConnInfo;

var
  I: integer;
  PRdbmsList,
    PRdbms: Pointer;

begin
  FreeChildControls(ParamsPnl);

  PRdbmsList := FGrt.Global[FConnInfoPath + '/driverInfos'];

  // Clear controls
  RDBMSComboBox.Clear;
  DriverComboBox.Items.Clear;
  DriverComboBoxCloseUp(self);

  //Set RDBMSCBox Items
  for I := 0 to myx_grt_list_item_count(PRdbmsList) - 1 do
  begin
    PRdbms := myx_grt_list_item_get(PRdbmsList, I);

    if (BuildDriverComboBoxItems(PRdbms) > 0) then
      RDBMSComboBox.AddItem(
        FGrt.DictString[PRdbms, 'caption'],
        PRdbms);
  end;

  {if (FConnStoragePath <> '') then
    SetConnection(FGrt.Global[FConnStoragePath]);}
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.FreeChildControls(Control: TWinControl);

begin
  while (Control.ControlCount > 0) do
    Control.Controls[0].Free;

  Application.ProcessMessages;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.BuildDriverControls(
  Target: TWinControl; PDriver: Pointer; DoAdvancedParams: Boolean);

var
  I: Integer;
  PParams,
    PParam: Pointer;
  Caption,
    Desc,
    ParamType: WideString;
  OffsetLeft,
    CurrentRow,
    MaxRow,
    CurrentTop: Integer;
  CaptionLbl: TTntLabel;
  ParamControl: TControl;
  BrowseBtn: TTntButton;
  CurrentLeft: Array [1 .. 30] of Integer;
  RowDescription: TTntStringList;

begin
  RowDescription := TTntStringList.Create;
  PParams := FGrt.DictItem[PDriver, 'parameters'];

  OffsetLeft := 116;
  MaxRow := 0;

  for I := 1 to 30 do
  begin
    CurrentLeft[I] := 0;
    RowDescription.Add('');
  end;

  for I := 0 to FGrt.ListCount(PParams) - 1 do
  begin
    PParam := FGrt.ListItem[PParams, I];

    if (FGrt.DictInt[PParam, 'layoutAdvanced'] <>
      Ord(DoAdvancedParams)) then
      continue;

    // if the "schema" should not be displayed
    if (Not(FDisplaySchemaSelection)) and
      (WideSameText(FGrt.DictString[PParam, 'name'], 'schema')) then
      continue;

    CurrentRow := FGrt.DictInt[PParam, 'layoutRow'];
    if (CurrentRow = -1) then
      CurrentRow := MaxRow + 1;
    MaxRow := CurrentRow;

    CurrentTop := 1 + (CurrentRow - 1) * ParamLineHeight;

    Caption := FGrt.DictString[PParam, 'caption'];
    Desc := FGrt.DictString[PParam, 'description'];
    ParamType := FGrt.DictString[PParam, 'paramType'];


    if (Not(WideSameText(ParamType, 'boolean'))) and
      (Not(WideSameText(ParamType, 'tristate'))) then
    begin
      // Create Edit with Caption
      if (Caption <> '') then
      begin
        CaptionLbl := TTntLabel.Create(self);
        CaptionLbl.Name := 'ParamLbl' + IntToStr(I);
        CaptionLbl.Caption := _(Caption);

        // if this is the first param on that row,
        // move the CaptionLbl to the left
        // so the Param edits are aligned left
        if (CurrentLeft[CurrentRow] = 0) then
          CaptionLbl.Left := OffsetLeft + CurrentLeft[CurrentRow] -
            CaptionLbl.Width - 10
        else
        begin
          CaptionLbl.Left := OffsetLeft + CurrentLeft[CurrentRow];
          CurrentLeft[CurrentRow] := CurrentLeft[CurrentRow] +
            CaptionLbl.Width + 10;
        end;

        CaptionLbl.Top := CurrentTop + 4;

        CaptionLbl.Parent := Target;
      end;


      // Create Param Edit
      ParamControl := TTntEdit.Create(self);

      ParamControl.Left := OffsetLeft + CurrentLeft[CurrentRow];

      TTntEdit(ParamControl).OnChange := DriverParamValueChanged;

      TTntEdit(ParamControl).Text :=
        FGrt.DictString[PParam, 'defaultValue'];

      // Set password char for password fields
      if (WideSameText(ParamType, 'password')) then
        TTntEdit(ParamControl).PasswordChar := '*';
    end
    else
    begin
      // Create Checkbox
      ParamControl := TTntCheckbox.Create(self);

      ParamControl.Left := OffsetLeft + CurrentLeft[CurrentRow] - 100;

      TTntCheckbox(ParamControl).OnClick := DriverParamValueChanged;

      TTntCheckbox(ParamControl).Checked :=
        (FGrt.DictString[PParam, 'defaultValue'] = '1');

      TTntCheckbox(ParamControl).Caption := _(Caption);
    end;

    // Set common options
    ParamControl.Name := 'Param' + IntToStr(I);
    ParamControl.Top := CurrentTop;

    ParamControl.Width := FGrt.DictInt[PParam, 'layoutWidth'];

    if (Desc <> '') then
    begin
      ParamControl.Hint := Desc;
      ParamControl.ShowHint := True;
    end;

    // move CurrentLeft
    CurrentLeft[CurrentRow] := CurrentLeft[CurrentRow] +
      ParamControl.Width + 20;

    FComponentParamMapping.AddObject(ParamControl.Name, PParam);

    ParamControl.Parent := Target;

    // Add lookup button
    if (FGrt.DictString[PParam, 'lookupValueMethod'] <> '') or
      WideSameText(ParamType, 'file') or
      WideSameText(ParamType, 'dir') then
    begin
      BrowseBtn := TTntButton.Create(self);
      BrowseBtn.Name := 'LookupBtn' + IntToStr(I);
      BrowseBtn.Tag := I;

      BrowseBtn.Caption := _('...');
      BrowseBtn.Width := 27;
      BrowseBtn.Height := 23;
      BrowseBtn.Left := ParamControl.Left + ParamControl.Width -
        BrowseBtn.Width;
      BrowseBtn.Top := CurrentTop;

      ParamControl.Width := ParamControl.Width -
        BrowseBtn.Width - 6;

      FComponentParamMapping.AddObject(BrowseBtn.Name, PParam);

      // Add lookup action
      if (FGrt.DictString[PParam, 'lookupValueMethod'] <> '') then
        BrowseBtn.OnClick := ParamLookupBtnClick
      else
        if (WideSameText(ParamType, 'file')) then
          BrowseBtn.OnClick := BrowseFileBtnClick
        else
          if (WideSameText(ParamType, 'dir')) then
            BrowseBtn.OnClick := BrowseDirBtnClick;

      BrowseBtn.Parent := Target;
    end;

    // build parameter description
    if (Desc <> '') and (CurrentRow - 1 >= 0) and
      (CurrentRow - 1 < RowDescription.Count) then
    begin
      if (RowDescription[CurrentRow - 1] = '') then
        RowDescription[CurrentRow - 1] := _(Desc)
      else
        RowDescription[CurrentRow - 1] :=
          RowDescription[CurrentRow - 1] + ' / ' + _(Desc);
    end;
  end;

  // Add descriptions for all lines
  if (FDisplayDescriptions) then
    for I := 0 to 9 do
    begin
      CurrentTop := 1 + I * ParamLineHeight;

      if (RowDescription[I] <> '') then
      begin
        CaptionLbl := TTntLabel.Create(self);
        CaptionLbl.Name := 'ParamDescLbl' + IntToStr(I + 1);
        CaptionLbl.Caption := RowDescription[I];

        CaptionLbl.Left := OffsetLeft + CurrentLeft[I + 1];

        CaptionLbl.Top := CurrentTop + 4;

        CaptionLbl.Parent := Target;
      end;
    end;

  RowDescription.Free;

  Target.Height := MaxRow * ParamLineHeight;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.DriverParamValueChanged(Sender: TObject);

var
  I: Integer;
  PParam: Pointer;
  RequiredParamsSet: Boolean;
  Component: TComponent;
begin
  if (Not(FSettingConnectionValues)) then
  begin
    // Check if all required fields are set
    RequiredParamsSet := True;
    for I := 0 to FComponentParamMapping.Count - 1 do
    begin
      PParam := FComponentParamMapping.Objects[I];

      if (FGrt.DictInt[PParam, 'required'] = 1) then
      begin
        Component := FindComponent(FComponentParamMapping[I]);

        if (Component is TTntEdit) then
          if (TTntEdit(Component).Text = '') then
          begin
            RequiredParamsSet := False;
            break;
          end;
      end;
    end;

    FOkBtn.Enabled := RequiredParamsSet;

    StoredConnAddBtn.Enabled := True;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.ParamLookupBtnClick(Sender: TObject);

var
  I,
    J: Integer;
  PParam,
    Res: Pointer;
  LookupValueModule,
    LookupValueMethod: WideString;
  MenuItem: TTntMenuItem;
  ScreenPoint: TPoint;

begin
  I := FComponentParamMapping.IndexOf(TControl(Sender).Name);

  if (I > -1) then
  begin
    PParam := FComponentParamMapping.Objects[I];

    LookupValueModule := FGrt.DictString[PParam, 'lookupValueModule'];
    LookupValueMethod := FGrt.DictString[PParam, 'lookupValueMethod'];

    LookupMenu.Items.Clear;

    // Store current settings
    WriteConnectionToTarget;

    try
      // Call LookupValueMethod
      Res := FGrt.ExecuteModalFunction(
        LookupValueModule, LookupValueMethod,
        [FGrt.Global[FConnTargetPath]]);

      for J := 0 to FGrt.ListCount(Res) - 1 do
      begin
        MenuItem := TTntMenuItem.Create(LookupMenu);
        MenuItem.Name := 'LookupMI' + IntToStr(J);
        MenuItem.Caption := FGrt.ListString[Res, J];

        MenuItem.OnClick := ParamLookupMIClick;
        MenuItem.Tag := I;

        LookupMenu.Items.Add(MenuItem);
      end;
    except
      MenuItem := TTntMenuItem.Create(LookupMenu);
        MenuItem.Name := 'LookupMI';
        MenuItem.Caption := _('Fetching of list failed.');
        
      LookupMenu.Items.Add(MenuItem);
    end;

    ScreenPoint := TControl(Sender).ClientToScreen(Point(0, 0));

    LookupMenu.Popup(ScreenPoint.X, ScreenPoint.Y + TControl(Sender).Height);
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.ParamLookupMIClick(Sender: TObject);

var
  I: Integer;
  ParamEdName: WideString;

begin
  ParamEdName := 'ParamEd' + IntToStr(TComponent(Sender).Tag);

  for I := 0 to ParamsPnl.ControlCount - 1 do
    if (WideSameText(ParamsPnl.Controls[I].Name, ParamEdName)) then
    begin
      TTntEdit(ParamsPnl.Controls[I]).Text := TTntMenuItem(Sender).Caption;
      break;
    end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.BrowseFileBtnClick(
  Sender: TObject);

var
  OpenDlg: TOpenDialog;
  I: Integer;
  PParam,
    PParamTypeDetails: Pointer;
  FileType,
    FileExtension,
    ParamEdName: WideString;

begin
  I := FComponentParamMapping.IndexOf(TControl(Sender).Name);

  if (I > -1) then
  begin
    PParam := FComponentParamMapping.Objects[I];

    PParamTypeDetails :=
      myx_grt_dict_item_get_value(PParam, 'paramTypeDetails');

    OpenDlg := TOpenDialog.Create(self);
    try
      if (PParamTypeDetails <> nil) then
      begin
        OpenDlg.Title :=
          _(myx_grt_dict_item_get_as_string(PParamTypeDetails,
            'fileOpenDialogCaption'));

        FileType :=
          _(myx_grt_dict_item_get_as_string(PParamTypeDetails,
            'fileType'));

        FileExtension :=
          _(myx_grt_dict_item_get_as_string(PParamTypeDetails,
            'fileExtension'));
      end;

      if (OpenDlg.Title = '') then
        OpenDlg.Title := _('Open File ...');

      if (FileType <> '') and (FileExtension <> '') then
        OpenDlg.Filter := FileType +
          ' (*.' + FileExtension + ')|*.' + FileExtension + '|' +
          _('All files') + ' (*.*)|*.*'
      else
        OpenDlg.Filter := _('All files') + ' (*.*)|*.*';

      if (OpenDlg.Execute) then
      begin
        ParamEdName := 'ParamEd' + IntToStr(TComponent(Sender).Tag);

        for I := 0 to ParamsPnl.ControlCount - 1 do
          if (WideSameText(ParamsPnl.Controls[I].Name, ParamEdName)) then
          begin
            TTntEdit(ParamsPnl.Controls[I]).Text := OpenDlg.FileName;
            break;
          end;
      end;
    finally
      OpenDlg.Free;
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.BrowseDirBtnClick(Sender: TObject);

begin
  //
end;

// -----------------------------------------------------------------------------

function TGrtDbConnForm.BuildDriverComboBoxItems(PRdbms: Pointer): Integer;

var
  I: integer;
  PDrivers,
    PDriver: Pointer;

begin
  PDrivers := FGrt.DictItem[PRdbms, 'drivers'];

  DriverComboBox.Clear;

  for I := 0 to FGrt.ListCount(PDrivers) - 1 do
  begin
    PDriver := FGrt.ListItem[PDrivers, I];

    // place default driver on top of dropdown
    if (FGrt.DictRef[PRdbms, 'defaultDriver'] = PDriver) then
      DriverComboBox.Items.InsertObject(
        0,
        FGrt.DictString[PDriver, 'caption'],
        PDriver)
    else
      DriverComboBox.AddItem(
        FGrt.DictString[PDriver, 'caption'],
        PDriver);
  end;

  Result := DriverComboBox.Items.Count;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.DriverComboBoxCloseUp(Sender: TObject);

var
  PRdbms: Pointer;
  PDriver: Pointer;

begin
  FreeChildControls(ParamsPnl);

  if (DriverComboBox.ItemIndex > -1) then
  begin
    PRdbms := RdbmsComboBox.Items.Objects[RdbmsComboBox.ItemIndex];
    PDriver :=
      DriverComboBox.Items.Objects[DriverComboBox.ItemIndex];

    if (FGrt.DictInt[PDriver, 'isInstalled'] = 1) then
    begin
      DriverNotInstalledPnl.Visible := False;

      // Clear mapping
      FComponentParamMapping.Clear;

      // Build params
      BuildDriverControls(ParamsPnl, PDriver);
      ParamsMainPnl.Height := StoredConnPnl.Height +
        ParamsPnl.Height + 38;

      // Build advanced params
      BuildDriverControls(AdvParamsPnl, PDriver, True);
      AdvParamsMainPnl.Height := AdvParamsPnl.Height + 38;

      // Get stored connections
      FillDropdownWithStoredConnections(StoredConnCBox, PDriver);

      {JdbcDriverNameLbl.Caption := _(myx_grt_dict_item_get_as_string(
        PRdbms, 'caption'));
      JdbcDriverDescLbl.Caption := _(myx_grt_dict_item_get_as_string(
        PJdbcDriver, 'description'));}

      StoredConnAddBtn.Enabled := False;

      StoredConnPnl.Visible := True;
    end
    else
    begin
      DownloadDriverPnl.Visible :=
        (FGrt.DictString[PDriver, 'downloadUrl'] <> '');

      DriverNotInstalledPnl.BringToFront;
      DriverNotInstalledPnl.Visible := True;

      StoredConnPnl.Visible := False;
    end;
  end
  else
  begin
    {JdbcDriverNameLbl.Caption := _('No RDBMS selected');
    JdbcDriverDescLbl.Caption := _('Please choose a Database System from the list above.');}

    StoredConnPnl.Visible := False;

    StoredConnAddBtn.Enabled := False;

    FreeChildControls(ParamsPnl);
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.FillDropdownWithStoredConnections(
  StoredConnComboBox: TTntComboBox; PDriver: Pointer);

var
  I: Integer;
  PConnections,
    PConnection,
    PRdbms: Pointer;

begin
  PConnections := FGrt.Global[FConnInfoPath + '/storedConns'];

  StoredConnComboBox.Items.Clear;
  StoredConnComboBox.ItemIndex := -1;

  if (RDBMSComboBox.ItemIndex <> -1) and
    (DriverComboBox.ItemIndex <> -1) then
  begin
    PRdbms := RDBMSComboBox.Items.Objects[RDBMSComboBox.ItemIndex];

    for I := 0 to FGrt.ListCount(PConnections) - 1 do
    begin
      PConnection := FGrt.ListItem[PConnections, I];

      if (FGrt.DictString[PConnection, 'rdbmsName'] =
          FGrt.DictString[PRdbms, 'name']) and
        (FGrt.DictString[PConnection, 'jdbcDriverName'] =
          FGrt.DictString[PDriver, 'name']) then
      begin
        StoredConnComboBox.Items.AddObject(
          FGrt.DictString[PConnection, 'name'],
          PConnection);
      end;
    end;
  end;

  StoredConnComboBox.Items.Add(_('<New Connection>'));
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.SetConnection(PConnection: Pointer);

var
  PParameter,
    PParameterValues: Pointer;
  I,
    Index: Integer;

begin
  StoredConnDelBtn.Enabled := False;

  // if the last entry is selected <New Connection>, clear edits
  if (PConnection = nil) then
  begin
    for I := 0 to ParamsPnl.ControlCount - 1 do
      if (ParamsPnl.Controls[I] is TTntEdit) then
        TTntEdit(ParamsPnl.Controls[I]).Text := '';

    StoredConnCBox.ItemIndex := -1;

    OKButton.Enabled := False;
  end
  else
  begin
    // Re-select RDBMS and JdbcDriver
    for I := 0 to RDBMSComboBox.Items.Count - 1 do
      if (WideSameText(FGrt.DictString[
        RDBMSComboBox.Items.Objects[I], 'name'],
        FGrt.DictString[PConnection, 'rdbmsName'])) then
      begin
        if (RDBMSComboBox.ItemIndex <> I) then
        begin
          RDBMSComboBox.ItemIndex := I;
          RDBMSComboBoxCloseUp(self);
        end;

        break;
      end;

    for I := 0 to DriverComboBox.Items.Count - 1 do
      if (WideSameText(FGrt.DictString[
        DriverComboBox.Items.Objects[I], 'name'],
        FGrt.DictString[PConnection, 'driverName'])) then
      begin
        if (DriverComboBox.ItemIndex <> I) then
        begin
          DriverComboBox.ItemIndex := I;
          DriverComboBoxCloseUp(self);
        end;

        break;
      end;

    PParameterValues := FGrt.DictItem[PConnection, 'parameterValues'];

    for I := 0 to ParamsPnl.ControlCount - 1 do
    begin
      if (ParamsPnl.Controls[I] is TTntEdit) then
      begin
        Index := FComponentParamMapping.IndexOf(
          ParamsPnl.Controls[I].Name);

        if (Index > -1) then
        begin
          PParameter := FComponentParamMapping.Objects[Index];

          TTntEdit(ParamsPnl.Controls[I]).Text :=
            FGrt.DictString[PParameterValues,
              FGrt.DictString[PParameter, 'name']];
        end;
      end;
    end;

    {JdbcStringEd.Text :=
      Grt.DictString[PConnection, 'connectionString'];}
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.RdbmsComboBoxCloseUp(Sender: TObject);

begin
  if (RDBMSComboBox.ItemIndex > -1) then
  begin
    BuildDriverComboBoxItems(
      RDBMSComboBox.Items.Objects[RDBMSComboBox.ItemIndex]);

    DriverComboBox.ItemIndex := 0;
    DriverComboBoxCloseUp(self);
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.SetDisplayRdbmsSelection(DisplayRdbmsSelection: Boolean);

begin
  ConnTypeMainPnl.Visible := DisplayRdbmsSelection;
  FDisplayRdbmsSelection := DisplayRdbmsSelection;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.SetSelectedRdbms(SelectedRdbms: WideString);

var
  I: Integer;

begin
  for I := 0 to RdbmsComboBox.Items.Count - 1 do
  begin
    if (WideSameText(
        FGrt.DictString[RdbmsComboBox.Items.Objects[I], 'name'],
        SelectedRdbms)) then
    begin
      RdbmsComboBox.ItemIndex := I;
      RdbmsComboBoxCloseUp(self);
      break;
    end;
  end;
end;

// -----------------------------------------------------------------------------

function TGrtDbConnForm.GetSelectedRdbms: WideString;

begin
  Result := '';

  if (RdbmsComboBox.ItemIndex > -1) then
  begin
    Result := FGrt.DictString[
      RdbmsComboBox.Items.Objects[RdbmsComboBox.ItemIndex],
      'name'];
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.StoredConnCBoxCloseUp(Sender: TObject);

begin
  if (StoredConnCBox.ItemIndex =
    StoredConnCBox.Items.Count - 1) or
    (StoredConnCBox.ItemIndex = -1) then
    SetConnection(nil)
  else
    SetConnection(StoredConnCBox.Items.Objects[
      StoredConnCBox.ItemIndex]);
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.AdvancedBtnClick(Sender: TObject);

begin
  AdvParamsMainPnl.Visible := Not(AdvParamsMainPnl.Visible);

  if (Not(FDisplayRdbmsSelection)) then
    ConnTypeMainPnl.Visible := AdvParamsMainPnl.Visible;

  BottomPnl.Top := 1000;

  if (AdvParamsMainPnl.Visible) then
    AdvancedBtn.Caption := _('<< Advanced')
  else
    AdvancedBtn.Caption := _('Advanced >>');
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.WriteConnectionToTarget;

var
  I: Integer;
  PConnection,
    PRdbms,
    PDriver,
    PParamValues,
    PParam: Pointer;
  Component: TComponent;

begin
  if ((RdbmsComboBox.ItemIndex = -1) or
      (DriverComboBox.ItemIndex = -1)) then
    Exit;

  PRdbms := RdbmsComboBox.Items.Objects[RdbmsComboBox.ItemIndex];
  PDriver := DriverComboBox.Items.Objects[DriverComboBox.ItemIndex];

  // create new connection
  PConnection := FGrt.DictNew('db.conn.Connection');

  // add a reference to the driver
  FGrt.DictRef[PConnection, 'driver'] := PDriver;

  // create parameter dict
  PParamValues := FGrt.DictNewTyped(MYX_STRING_VALUE, '');
  FGrt.DictItem[PConnection, 'parameterValues'] := PParamValues;

  // set values from controls
  for I := 0 to FComponentParamMapping.Count - 1 do
  begin
    PParam := FComponentParamMapping.Objects[I];

    Component := FindComponent(FComponentParamMapping[I]);

    // set the new value
    if (Component is TTntEdit) then
      FGrt.DictString[PParamValues,
        FGrt.DictString[PParam, 'name']] := TTntEdit(Component).Text
    else
      if (Component is TTntCheckbox) then
        FGrt.DictString[PParamValues,
          FGrt.DictString[PParam, 'name']] :=
          IntToStr(Ord(TTntCheckbox(Component).Checked));
  end;

  // create modules list
  FGrt.DictItem[PConnection, 'modules'] :=
    FGrt.ValueDuplicate(FGrt.DictItem[PDriver, 'defaultModules']);

  FGrt.Global[FConnTargetPath] := PConnection;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.OKBtnClick(Sender: TObject);

begin
  WriteConnectionToTarget;

  ModalResult := mrOK;
end;

// -----------------------------------------------------------------------------

procedure TGrtDbConnForm.CancelBtnClick(Sender: TObject);
begin
  ModalResult := mrCancel;
end;

end.
