/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_figure.h 
 * @brief Implementation of the model element class.
 * 
 */

#ifndef __GC_FIGURE_H__
#define __GC_FIGURE_H__

#include "myx_gc_style.h"
#include "myx_gc_layer.h"
#include "myx_gc_layout.h"

//----------------------------------------------------------------------------------------------------------------------

class CGenericCanvas;
class CGCModel;
class CLayer;
class CFigureInstance;
class CLayouter;

//----------------- Figure templates -----------------------------------------------------------------------------------

const string DefaultLayout = "column";      // The layout to be used for figure elements without any given layout.
const string DefaultResize = "false";       // By default figure elements cannot be resized.

class CFigureElementTemplate;
class CCaptionElementTemplate;
class CFigureElement;
class CCaptionElement;

class CStyleListener: public CGCListener
{
  friend class CFigureElementTemplate;
protected:
  CFigureElementTemplate* template_;
public:
  virtual void __cdecl onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason);
  virtual void __cdecl onDestroy(CGCBase* sender);
  virtual void __cdecl onError(CGCBase* sender, CGCBase* origin, const char* message);
};

/** 
 * A figure element is one detail in a figure template and so also in a figure. There can be a hierarchy of figure
 * elements to form complex figures.
 */
class CFigureElementTemplate
{
  friend class CFigureElement;
private:
  wstring FKey;                        // An identifier that can be used to search for this template.
  TFigureElementLayout FLayout;        // How are the element's children laid out.
  bool FResizable;                     // Indicates whether this element can be resized.
  TBoundingBox FBoundingBox;           // The box that includes this and all child elements.
  CElementTemplateList FChildren;      // A list of child elements attached to this element.
  CGCStyle* FStyle;                    // The style to be used for this element.
  CFigureElementTemplate* FParent;     // The parent element of this element (if at all).
  CFigure* FFigure;                    // The figure this element belongs to if not a child element of another element.
                                       // FParent and FFigure are mutual exclusive. Only one of both (or none) can be set.
  CCaptionElementTemplate* FCaption;   // An optional caption for this element.
  TConstraints FConstraints;           // Size constraints for the element. A value of -1 means no constraint.
  TOccurence FOccurence;               // Number of times this element may occur.
  CStyleListener FListener;
protected:
  void computeBoundingBox(void);
public:
  CFigureElementTemplate(wstring key);
  virtual ~CFigureElementTemplate(void);

  void addChild(CFigureElementTemplate* Child);
  void freeNotification(CGCBase* object);
  CFigureElementTemplate* getListElement(void);
  void initialize(TFigureElementLayout Layout, bool Resizable, CGCStyle* style, const TConstraints& Constraints,
    TOccurence Occurence);
  bool isList(void);
  wstring key(void) { return FKey; };
  TOccurence occurence(void) { return FOccurence; };
  void setCaption(CCaptionElementTemplate* Caption);
};

/** Text alignment constants. */
typedef enum
{
  GC_ALIGN_LEFT_TOP,
  GC_ALIGN_CENTER,
  GC_ALIGN_RIGHT_BOTTOM
} TAlignment;

/**
 * Special text element class.
 * Note: captions are directly bound to their owning parent element and not handled as a separate child element.
 */
class CCaptionElementTemplate
{
  friend class CFigureElementTemplate;
  friend class CFigureElement;
private:
  wstring FText;                       // The text of the caption.
  wstring FKey;                        // An identifier that can be used to search for this template.
  string FFontFamily;                  // The font to be used for output (e.g. Arial, Verdana)
  int FFontSize;                       // Font size in points.
  int FWeight;                         // The "boldness" of the text.
  string FFontStyle;                   // normal or italic
  TAlignment FHorizontalAlignment;     // Left, center, right
  TAlignment FVerticalAlignment;       // Top, center, bottom
  TBoundingBox FBoundingBox;           // The box that includes the text.
  CFigureElementTemplate* FParent;     // The parent element of this element (if at all).
  GLfloat FColor[4];                   // The text color.
  bool FHasColor;                      // True, if this caption has an own color.
  TConstraints FConstraints;           // Size constraints for the element. A value of -1 means no constraint.
  float FOffsetX;                      // Horizontal fine tuning offset.
  float FOffsetY;                      // Vertical fine tuning offset.
public:
  CCaptionElementTemplate(wstring key);

  void initialize(wstring Text, float X, float Y, string FontFamily, int FontSize, int Weight, string FontStyle, 
    TAlignment HorzontalAlignment, TAlignment VerticalAlignment, GLubyte* Color, const TConstraints& Constraints);
  wstring key(void) { return FKey; };
};

/**
 * CFigureTemplate is a description of how a concrete figure has to look and act. It is loaded from a description file
 * and created by the figure parser.
 */
class CFigureTemplate: public CGCBase
{
  friend class CFigureParser;
private:
  wstring FType;                       // The type for which this template can be used.
  wstring FClass;                      // A string describing the kind of layout realized in the template. 
  CFigureElementTemplate* FContent;    // The root element of the template.
  CGCModel* FModel;
public:
  CFigureTemplate(CGCModel* model, wstring type, wstring layoutClass);
  virtual ~CFigureTemplate(void);

  CFigureElementTemplate* content(void) { return FContent; };
  wstring layoutClass(void) { return FClass; };
  wstring type(void) { return FType; };
  virtual TGCVariant __cdecl property(const char* name, unsigned int index);
  virtual void __cdecl property(const char* name, unsigned int index, const TGCVariant& value);
};

//----------------------------------------------------------------------------------------------------------------------

/**
 * Instance for figure elements and captions.
 */
class CCaptionElement: public CGCBase
{
  friend class CFigureElement;
private:
  GLuint FList;                        // The display list for that text.
  wstring FText;                       // The text of the caption.
  wstring FDisplayText;                // The text that is actually displayed (might be shortend with ellipsis).
  string FFontFamily;                  // The font to be used for output (e.g. Arial, Verdana)
  int FFontSize;                       // Font size in points at 100% zoom.
  int FActualFontSize;                 // Actual font size in points, adjusted by current zoom.
  int FWeight;                         // The "boldness" of the text.
  string FFontStyle;                   // normal or italic
  TAlignment FHorizontalAlignment;     // Left, center, right
  TAlignment FVerticalAlignment;       // Top, center, bottom
  TBoundingBox FBounds;                // The box that includes the text.
  CFigureElement* FParent;             // The parent element of this element (if at all).
  bool FDirty;                         // True if anything changed in the element.
  GLfloat FColor[4];                   // The text color.
  bool FHasColor;                      // True, if this caption has an own color.
  float FAlignOffsetX;                 // Horizontal text alignment offset.
  float FAlignOffsetY;                 // Vertical text alignment offset.
  float FOffsetX;                      // Horizontal fine tuning offset given by the template.
  float FOffsetY;                      // Vertical fine tuning offset given by the template.
  TConstraints FConstraints;           // Size constraints for the element. A value of -1 means no constraint.
  TBidiMode FBidiMode;                 // Determines text directionality.
public:
  CCaptionElement(CGenericCanvas* canvas);
  virtual ~CCaptionElement(void);

  void applyAlignment(void);
  TBoundingBox bounds(void) { return FBounds; };
  void makeDirty(void);
  virtual TGCVariant __cdecl property(const char* name, unsigned int index);
  virtual void __cdecl property(const char* name, unsigned int index, const TGCVariant& value);
  void render(void);
  void text(wstring newText);
  void validate(void);
  bool zoomChanged(float ZoomFactor);
};

class CElementListener: public CGCListener
{
  friend class CFigureElement;
protected:
    CFigureElement* element;
public:
  virtual void __cdecl onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason);
  virtual void __cdecl onDestroy(CGCBase* sender);
  virtual void __cdecl onError(CGCBase* sender, CGCBase* origin, const char* message);
};

class GENERIC_CANVAS_API CFigureElement: public CGCBase
{
  friend class CCaptionElement;
private:
  float FTranslation[3];          // Determines the location of the element after layouting.
  TFigureElementLayout FLayout;   // How are the element's children laid out.
  bool FResizable;                // Indicates whether this element can be resized.
  TBoundingBox FBounds;           // The box that includes this and all child elements.
  CElementList FChildren;         // A list of child elements attached to this element.
  CGCStyle* FStyle;               // The style to be used for this element.
  CFigureElement* FParent;        // The parent element of this element (if at all).
  CFigure* FFigure;               // The figure this element belongs to if not a child element of another element.
                                  // FParent and FFigure are mutual exclusive. Only one of both (or none) can be set.
  CCaptionElement* FCaption;      // An optional caption for this element.
  CLayouter* FLayouter;           // The class that computes the layout.
  bool FDirty;                    // True if anything changed in the element.
  bool FExpanded;                 // True if child elements are visible.
  TConstraints FConstraints;      // Size constraints for the element. A value of -1 means no constraint.
  CElementListener FListener;
  CFigureElementTemplate* FTemplate; // Reference to the template part for this element.
protected:
  void computeBoundingBox(void);
public:
  CFigureElement(CFigureElementTemplate* aTemplate, CGenericCanvas* canvas);
  virtual ~CFigureElement(void);

  CFigureElement* addSubElement(void);
  TBoundingBox bounds(void) { return FBounds; };
  CElementList* children(void) { return &FChildren; };
  static CFigureElement* createFromTemplate(CGenericCanvas* canvas, CFigure* Owner, CFigureElementTemplate* Template);
  bool doAction(CFigureInstance* Instance, const float X, const float Y);
  CFigure* figure(void);
  TFigureElementLayout layout(void) { return FLayout; };
  void makeDirty(void);
  virtual TGCVariant __cdecl property(const char* name, unsigned int index);
  virtual void __cdecl property(const char* name, unsigned int index, const TGCVariant& value);
  void render(void);
  void setCaption(const char* text);
  void style(CGCStyle* NewStyle);
  CGCStyle* style(void);
  CFigureElementTemplate* template_(void) { return FTemplate; };
  void validate(void);
  bool zoomChanged(float ZoomFactor);
};

//----------------------------------------------------------------------------------------------------------------------

typedef hash_map<wstring, CFigureElement*> CFigureElementMap;

class CFigureElementListener: public CGCListener
{
  friend class CFigure;
protected:
    CFigure* figure;
public:
  virtual void __cdecl onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason);
  virtual void __cdecl onDestroy(CGCBase* sender);
  virtual void __cdecl onError(CGCBase* sender, CGCBase* origin, const char* message);
};

/**
 * CFigure is the main element in the model and is created from a figure template. It cannot itself appear in a scene but
 * is represented by one or more figure instances.
 */
class GENERIC_CANVAS_API CFigure: public CGCBase
{
  friend class CGCModel;
  friend class CFigureInstance;
private:
  CGCModel* FModel;          // The model to which this figure belongs.
  CFigureElement* FContent;  // The root element of the figure.
  bool FDirty;               // True if any of the properties changed that affect the display list.
  float FScaling[3];         // The factors to scale the figure in each direction.
  float FMinimalScale;       // The lesser of x and y scale factors.
  float FTranslation[3];     // The factors to move the figure.
  float FRotation[4];        // The factors to rotate the figure. FRotation[0] is an angle in radians. Index 1-3 form a vector to rotate around.
                             // Note: Order of application of the 3 transformations is scaling first, then rotation, finally translation.
  TBoundingBox FBounds;      // Bounding box of the figure in figure instance space (i.e. relative to its figure instances).
  float FLastZoom;           // The most recent minimal overall scale factor (including its own scale factors).
  CFigureTemplate* FTemplate; // The template from which this figure was built.
  CFigureElementListener FListener;
  CFigureElementMap FElementMap; // A map for quick lookup of figure elements given by a path.
protected:
  void applyTransformations(void);
  void buildFromTemplate(CFigureTemplate* Template);
  void validate(void);
public:
  CFigure(CGCModel* Owner, CFigureTemplate* Template);
  virtual ~CFigure(void);

  void addMapping(wstring path, CFigureElement* element);
  virtual TBoundingBox __cdecl bounds(void) { return FBounds; };
  CFigureElement* content(void) { return FContent; };
  CFigureElement* elementFromKey(const char* key);
  CFigureElement* elementFromKey(wstring key);
  void freeNotification(CFigureElement* object);
  void makeDirty(void);
  CGCModel* model(void) { return FModel; };
  virtual TGCVariant __cdecl property(const char* name, unsigned int index);
  virtual void __cdecl property(const char* name, unsigned int index, const TGCVariant& value);
  void removeMapping(CFigureElement* element);
  virtual void __cdecl render(float currentZoom);
  virtual void __cdecl rotate(float Angle, const float Axis[3]);
  virtual void __cdecl rotate(float Angle, float Rx, float Ry, float Rz);
  virtual void __cdecl scale(const float Factor[3], bool Accumulative = false);
  virtual void __cdecl scale(float Sx, float Sy, float Sz, bool Accumulative = false);
  CFigureTemplate* template_(void) { return FTemplate; };
  virtual void __cdecl translate(const float Factor[3], bool Accumulative = false);
  virtual void __cdecl translate(float Tx, float Ty, float Tz, bool Accumulative = false);
};

//----------------------------------------------------------------------------------------------------------------------

class CFigureListener: public CGCListener
{
  friend class CFigureInstance;
protected:
  CFigureInstance* instance;
public:
  virtual void __cdecl onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason);
  virtual void __cdecl onDestroy(CGCBase* sender);
  virtual void __cdecl onError(CGCBase* sender, CGCBase* origin, const char* Message);
};

/**
 * The figure instance class is a proxy for a figure on a particular layer. There can be more than one instance pointing 
 * to the same figure.
 */
class GENERIC_CANVAS_API CFigureInstance: public CGCBase
{
  friend class CGenericCanvas;
  friend class CLayer;
  friend class CFigure;
  friend class CFeedbackLayer;
  friend class CFigureListener;
private:
  CLayer* FLayer;            // The layer on which this figure is displayed.
  CFigure* FFigure;          // The figure of which this class is an instance.
  bool FDirty;               // True if any of the properties changed that affect the display list.
  float FScaling[3];         // The factors to scale the figure in each direction.
  float FMinimalScale;       // The lesser of x and y scale factors.
  float FTranslation[3];     // The factors to move the figure.
  float FRotation[4];        // The factors to rotate the figure. FRotation[0] is an angle in radians. Index 1-3 form a vector to rotate around.
                             // Note: Order of apllication of the 3 transformations is scaling first, then rotation, finally translation.
  TBoundingBox FBounds;      // Bounding box of the figure instance in layer coordinates.
  bool FSelected;            // True if this instance is currently selected.
  CFigureListener FListener;
protected:
  void applyTransformations(void);
  bool doAction(const float X, const float Y);
  void makeDirty(void);
  void onDestroy(CGCBase* Figure);
  void replaceFigure(CFigure* figure);
  void validate(void);
public:
  CFigureInstance(CLayer* Owner, CFigure* Figure);
  virtual ~CFigureInstance(void);

  virtual TBoundingBox __cdecl bounds(void) { return FBounds; };
  virtual bool __cdecl containsPoint(const float X, const float Y);
  virtual bool __cdecl overlaps(TBoundingBox& Box);
  virtual TGCVariant __cdecl property(const char* name, unsigned int index);
  virtual void __cdecl property(const char* name, unsigned int index, const TGCVariant& value);
  virtual void __cdecl render(float CurrentZoom);
  virtual void __cdecl rotate(float Angle, float Rx, float Ry, float Rz);
  virtual void __cdecl rotateV(float Angle, const float Axis[3]);
  virtual void __cdecl scale(float Sx, float Sy, float Sz, bool Accumulative = false);
  virtual void __cdecl scaleV(const float Factor[3], bool Accumulative = false);
  virtual bool __cdecl selected(void);
  virtual void __cdecl translate(float Tx, float Ty, float Tz, bool Accumulative = false);
  virtual void __cdecl translateV(const float Factor[3], bool Accumulative = false);
};

//----------------------------------------------------------------------------------------------------------------------

#endif // __GC_FIGURE_H__
