/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANy WARRANTy; without even the implied warranty of
   MERCHANTABILITy or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   you should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_figure.cpp 
 * @brief Implementation of the model element class.
 * 
 */

#include "myx_gc.h"

#include <libxml/tree.h>

#include "myx_gc_datatypes.h"
#include "myx_gc_const.h"
#include "myx_gc_figure.h"
#include "myx_gc_model.h"
#include "myx_gc_layer.h"
#include "myx_gc_canvas.h"
#include "myx_gc_font_manager.h"

//----------------------------------------------------------------------------------------------------------------------

void CStyleListener::onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason)
{
}

//----------------------------------------------------------------------------------------------------------------------

void CStyleListener::onDestroy(CGCBase* sender)
{
  if (template_ != NULL)
    template_->freeNotification(sender);
}

//----------------------------------------------------------------------------------------------------------------------

void CStyleListener::onError(CGCBase* sender, CGCBase* origin, const char* message)
{
}

//----------------- CFigureElementTemplate ------------------------------------------------------------------------------

CFigureElementTemplate::CFigureElementTemplate(wstring key)
{
  FKey = key;
  FLayout = GC_LAYOUT_ROW;
  FResizable = false; 
  FParent = NULL;
  FFigure = NULL;
  FCaption = NULL;
  FOccurence = GC_OCC_ONCE;
  FListener.template_ = this;
}

//----------------------------------------------------------------------------------------------------------------------

CFigureElementTemplate::~CFigureElementTemplate(void)
{
  if (FStyle != NULL && !FStyle->destroying())
    FStyle->removeListener(&FListener);
  for (CElementTemplateList::iterator iterator = FChildren.begin(); iterator != FChildren.end(); ++iterator)
    delete *iterator;
  delete FCaption;
}

//----------------------------------------------------------------------------------------------------------------------

void CFigureElementTemplate::addChild(CFigureElementTemplate* Child)
{
  Child->FParent = this;
  Child->FFigure = NULL;
  FChildren.push_back(Child);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Called when a style is about to be destroyed.
 *
 * @param object The object, which is about to be freed.
 */
void CFigureElementTemplate::freeNotification(CGCBase* object)
{
  if (FStyle == object)
    FStyle = NULL;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the list element of this figure element template. There is only a list element if this template is a list
 * that is, has only one child that can appear more than once. This child element is then the list element.
 *
 * @return The only child element of this template if it is a list template, otherwise NULL.
 */
CFigureElementTemplate* CFigureElementTemplate::getListElement(void)
{
  CFigureElementTemplate* result = NULL;

  if (FChildren.size() == 1)
  {
    CFigureElementTemplate* child = FChildren[0];
    if (child->occurence() != GC_OCC_ONCE)
      result = FChildren[0];
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Helper method used by the figure parser to initialize some members.
 *
 * @param Layout The the layout to be used in the element.
 * @param Resizable Resizable flag.
 * @param Constraints The resize constraints.
 */
void CFigureElementTemplate::initialize(TFigureElementLayout Layout, bool Resizable, CGCStyle* style, 
                                        const TConstraints& Constraints, TOccurence Occurence)
{
  FLayout = Layout;
  FResizable = Resizable;
  FStyle = style;
  if (FStyle != NULL)
    FStyle->addListener(&FListener);
  FConstraints = Constraints;
  FOccurence = Occurence;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines if this template is a list. A list is defined as having only one child, which is allowed
 * to appear more than once.
 *
 * @return True if this template is a list, otherwise false.
 */
bool CFigureElementTemplate::isList(void)
{
  if (FChildren.size() == 1)
  {
    CFigureElementTemplate* child = FChildren[0];
    return child->occurence() != GC_OCC_ONCE;
  };

  return false;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the (optional) caption for this element. If there is already one then it is freed.
 *
 * @param Caption The caption to be used from now on.
 */
void CFigureElementTemplate::setCaption(CCaptionElementTemplate* Caption)
{
  if (FCaption != NULL)
    delete FCaption;
  FCaption = Caption;
  if (FCaption != NULL)
    FCaption->FParent = this;
}

//----------------- CCaptionElementTemplate ------------------------------------------------------------------------------------

CCaptionElementTemplate::CCaptionElementTemplate(wstring key)
{
  FKey = key;
  FFontSize = 20;
  FWeight = 400;
  FHorizontalAlignment = GC_ALIGN_LEFT_TOP;
  FVerticalAlignment = GC_ALIGN_LEFT_TOP;
  FHasColor = false;
  FColor[0] = 0;
  FColor[1] = 0;
  FColor[2] = 0;
  FColor[3] = 255;
}

//----------------------------------------------------------------------------------------------------------------------

void CCaptionElementTemplate::initialize(wstring Text, float x, float y, string FontFamily, int FontSize, int Weight, 
                                         string FontStyle, TAlignment HorizontalAlignment, TAlignment VerticalAlignment, 
                                         GLubyte* Color, const TConstraints& Constraints)
{
  FText = Text;
  FOffsetX = x;
  FOffsetY = y;
  FFontFamily = FontFamily;
  FFontSize = FontSize;
  FWeight = Weight;
  FFontStyle = FontStyle;
  FHorizontalAlignment = HorizontalAlignment;
  FVerticalAlignment = VerticalAlignment;
  FHasColor = Color != NULL;
  if (FHasColor)
  {
    for (int I = 0; I < 4; ++I)
      FColor[I] = (float) Color[I] / 255;
  };
  FConstraints = Constraints;
}

//----------------- CFigureTemplate ------------------------------------------------------------------------------------

CFigureTemplate::CFigureTemplate(CGCModel* model, wstring type, wstring layoutClass): CGCBase(model->canvas())
{
  FModel = model;
  FType = type;
  FClass = layoutClass;
  FContent = NULL;
  _className = "CFigureTemplate";
}

//----------------------------------------------------------------------------------------------------------------------

CFigureTemplate::~CFigureTemplate(void)
{
  delete FContent;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the property given by path. The path syntax is must be something like (here expressed as regex)
 * (container)*(property), where container is a slash and the name of a container class (e.g. layers, figures) and
 * property is the name of a simple property of that container.
 *
 * @param Name The name of the property to return.
 * @param index The index of the sub property to return if it is located in a list.
 * @return A description of the property value and, if the property is simple, the actual value.
 */
TGCVariant CFigureTemplate::property(const char* name, unsigned int index)
{
  TGCVariant result;

  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_NAME:
            {
              result.type = GC_VAR_STRING;
              result.s = utf16ToUtf8(FType + L"@" + FClass);
              break;
            };
          case GC_PROPERTY_OWNER:
            {
              result.type = GC_VAR_OBJECT;
              result.reference = FModel;
              break;
            };
        };

        break;
      };
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the value of the given property, which must be a simple property.
 *
 * @param name The name of the property.
 * @param index The index of the sub property to return if it is located in a list.
 * @param value The new value of the property. Automatic conversion is performed where possible.
 */
void CFigureTemplate::property(const char* name, unsigned int index, const TGCVariant& value)
{
  // There are currently no properties that could be changed. The name is a unique identifier and must not be changed.
}

//----------------- CCaptionElement ------------------------------------------------------------------------------------

CCaptionElement::CCaptionElement(CGenericCanvas* canvas): CGCBase(canvas)
{
  _className = "CCaptionElement";
  FFontFamily = "Arial";
  FFontSize = 20;
  FWeight = 400;
  FFontStyle = "normal";
  FHorizontalAlignment = GC_ALIGN_LEFT_TOP;
  FVerticalAlignment = GC_ALIGN_LEFT_TOP;
  FParent = NULL;
  FDirty = true;
  FHasColor = false;
  FBidiMode = GC_BIDI_LEFT_TO_RIGHT;
  FActualFontSize = 1;
  FList = 0;
}

//----------------------------------------------------------------------------------------------------------------------

CCaptionElement::~CCaptionElement(void)
{
  if (FList != 0)
    glDeleteLists(FList, 1);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Recomputes the offsets for the caption to maintain the current alignments.
 */
void CCaptionElement::applyAlignment(void)
{
  // -- Horizontal alignment
  // If the text is larger than the available space then shorten it.
  TBoundingBox mainBounds = FParent->FBounds;

  // Apply constraints to the bounding box.
  if (FConstraints.maxHeight > -1)
  {
    if (mainBounds.lower.y - mainBounds.upper.y > FConstraints.maxHeight)
      mainBounds.lower.y = mainBounds.upper.y + FConstraints.maxHeight;
  };
  if (FConstraints.maxWidth > -1)
  {
    if (mainBounds.lower.x - mainBounds.upper.x > FConstraints.maxWidth)
      mainBounds.lower.x = mainBounds.upper.x + FConstraints.maxWidth;
  };
  if (FConstraints.minHeight > -1)
  {
    if (mainBounds.lower.y - mainBounds.upper.y < FConstraints.minHeight)
      mainBounds.lower.y = mainBounds.upper.y + FConstraints.minHeight;
  };
  if (FConstraints.minWidth > -1)
  {
    if (mainBounds.lower.x - mainBounds.upper.x < FConstraints.minWidth)
      mainBounds.lower.x = mainBounds.upper.x + FConstraints.minWidth;
  };

  float availableWidth = mainBounds.lower.x - mainBounds.upper.x - FOffsetX;
  float Width = FBounds.lower.x - FBounds.upper.x;
  if (Width > availableWidth)
  {
    // There is not enough space to display the full text.
    FAlignOffsetX = 0;
    int Len = FText.size();
    if ((Len == 0) || (Width <= 0))
      FDisplayText.clear();
    else
    {
      // Determine width of triple point using the current font settings.
      TBoundingBox LocalBox;
      fontManager()->boundingBox(L"...", FFontFamily, FFontStyle, FWeight, FFontSize, &LocalBox);
      float ellipsisWidth = LocalBox.lower.x - LocalBox.upper.x;

      if (Width <= ellipsisWidth)
        FDisplayText.clear();
      else
      {
        // Do a binary search for the optimal string length which fits into the given width.
        int L = 0;
        int H = Len - 1;
        if (FBidiMode == GC_BIDI_RIGHT_TO_LEFT)
        {
          while (L < H)
          {
            int N = (L + H) >> 1;
            fontManager()->boundingBox(FText.substr(N), FFontFamily, FFontStyle, FWeight, FFontSize, &LocalBox);
            float W = LocalBox.lower.x - LocalBox.upper.x + ellipsisWidth;
            if (W <= availableWidth)
              H = N;
            else
              L = N + 1;
          };
          FDisplayText = L"..." + FText.substr(L + 1);
        }
        else
        {
          while (L < H)
          {
            int N = (L + H + 1) >> 1;
            fontManager()->boundingBox(FText.substr(0, N), FFontFamily, FFontStyle, FWeight, FFontSize, &LocalBox);
            float W = LocalBox.lower.x - LocalBox.upper.x + ellipsisWidth;
            if (W <= availableWidth)
              L = N;
            else
              H = N - 1;
          };
          FDisplayText = FText.substr(0, L - 1) + L"...";
        };
      };
    };
  }
  else
  {
    FDisplayText = FText;

    TAlignment Alignment = FHorizontalAlignment;
    // In right-to-left mode alignment is reverted.
    if (FBidiMode == GC_BIDI_RIGHT_TO_LEFT)
      if (Alignment == GC_ALIGN_LEFT_TOP)
        Alignment = GC_ALIGN_RIGHT_BOTTOM;
      else
        if (Alignment == GC_ALIGN_RIGHT_BOTTOM)
          Alignment = GC_ALIGN_LEFT_TOP;

    switch (FHorizontalAlignment)
    {
      case GC_ALIGN_LEFT_TOP: // Left aligned.
        {
          FAlignOffsetX = 0;
          break;
        };
      case GC_ALIGN_CENTER: // Centered.
        {
          FAlignOffsetX = (availableWidth - Width) / 2;
          break;
        };
      case GC_ALIGN_RIGHT_BOTTOM: // Right aligned.
        {
          FAlignOffsetX = availableWidth - Width;
          break;
        };
    };
  };
  
  // -- Vertical alignment
  float availableHeight = mainBounds.lower.y - mainBounds.upper.y - FOffsetY;
  float Height = FBounds.lower.y - FBounds.upper.y;
  switch (FVerticalAlignment)
  {
    case GC_ALIGN_LEFT_TOP: // Left aligned.
      {
        FAlignOffsetY = 0;
        break;
      };
    case GC_ALIGN_CENTER: // Centered.
      {
        FAlignOffsetY = (availableHeight - Height) / 2;
        break;
      };
    case GC_ALIGN_RIGHT_BOTTOM: // Right aligned.
      {
        FAlignOffsetY = availableHeight - Height;
        break;
      };
  };

  if (FList == 0)
    FList = glGenLists(1);
  glNewList(FList, GL_COMPILE);
  if (FActualFontSize > 4)
  {
    // FTGL uses the lower left corner as origin, we however have the upper left corner for this task.
    // Adjust the vertical coordinate accordingly.
    glRasterPos2f(FOffsetX + FAlignOffsetX, FOffsetY + FAlignOffsetY + FBounds.lower.y);

    fontManager()->textOut(FDisplayText, FFontFamily, FFontStyle, FWeight, FActualFontSize);
  }
  else
  {
    glColor4fv(FColor);
    float x = FOffsetX + FAlignOffsetX;
    float y = FOffsetY + FAlignOffsetY;
    float LocalWidth = (Width < availableWidth) ? Width : availableWidth;
    glRectf(x + 1, y + 1, x + LocalWidth - 1, y + FBounds.lower.y - 1);
  };

  glEndList();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Marks the element as changed so its bounding box it is validated next time it is used.
 */
void CCaptionElement::makeDirty(void)
{
  if (!FDirty)
  {
    FDirty = true;
    if (FParent != NULL)
      FParent->makeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the property given by name. The name syntax is must be something like (here expressed as regex)
 * (container)*(property), where container is a slash and the name of a container class (e.g. layers, figures) and
 * property is the name of a simple property of that container.
 *
 * @param name The name of the property to return.
 * @param index If the property is a list then this is the index into that list.
 * @return A description of the property value and, if the property is simple, the actual value.
 */
TGCVariant CCaptionElement::property(const char* name, unsigned int index)
{
  TGCVariant result;

  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_OWNER:
            {
              result.type = GC_VAR_OBJECT;
              result.reference = FParent;
              break;
            };
          case GC_PROPERTY_DESCRIPTION:
            {
              result.type = GC_VAR_STRING;
              result.s = "The optional caption of an element";
              break;
            };
          case GC_PROPERTY_NAME:
          case GC_PROPERTY_TEXT:
            {
              result.type = GC_VAR_STRING;
              result.s = utf16ToUtf8(FText);
              break;
            };
          case GC_PROPERTY_FONT_FAMILY:
            {
              result.type = GC_VAR_STRING;
              result.s = FFontFamily;
              break;
            };
          case GC_PROPERTY_FONT_SIZE:
            {
              result.type = GC_VAR_INT;
              result.i = FFontSize;
              break;
            };
          case GC_PROPERTY_FONT_WEIGHT:
            {
              result.type = GC_VAR_INT;
              result.i = FWeight;
              break;
            };
          case GC_PROPERTY_FONT_STYLE:
            {
              result.type = GC_VAR_STRING;
              result.s = FFontStyle;
              break;
            };
          case GC_PROPERTY_COLOR:
            {
              result.type = GC_VAR_STRING;
              result.s = colorToString(FColor);
              break;
            };
          case GC_PROPERTY_ALIGNMENT_VERTICAL:
            {
              result.type = GC_VAR_STRING;
              static char* VAlignmentToString[3] = {"top", "center", "bottom"};
              result.s = VAlignmentToString[FVerticalAlignment];
              break;
            };
          case GC_PROPERTY_ALIGNMENT_HORIZONTAL:
            {
              result.type = GC_VAR_STRING;
              static char* HAlignmentToString[3] = {"left", "center", "right"};
              result.s = HAlignmentToString[FHorizontalAlignment];
              break;
            };
          case GC_PROPERTY_BIDI_MODE:
            {
              result.type = GC_VAR_STRING;
              static char* BidiModeToString[2] = {"left-to-right", "right-to-left"};
              result.s = BidiModeToString[FBidiMode];
              break;
            };
          case GC_PROPERTY_MIN_WIDTH:
            {
              result.type = GC_VAR_FLOAT;
              result.f = FConstraints.minWidth;
              
              break;
            };
          case GC_PROPERTY_MIN_HEIGHT:
            {
              result.type = GC_VAR_FLOAT;
              result.f = FConstraints.minHeight;
              
              break;
            };
          case GC_PROPERTY_MAX_WIDTH:
            {
              result.type = GC_VAR_FLOAT;
              result.f = FConstraints.maxWidth;
              
              break;
            };
          case GC_PROPERTY_MAX_HEIGHT:
            {
              result.type = GC_VAR_FLOAT;
              result.f = FConstraints.maxHeight;
              
              break;
            };
        };
        
        break;
      };
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the value of the given property, which must be a simple property.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this is the index into that list.
 * @param Value The new value of the property. Automatic conversion is performed where possible.
 */
void CCaptionElement::property(const char* name, unsigned int index, const TGCVariant& value)
{
  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_TEXT:
            {
              FText = utf8ToUtf16(variantToString(value));
              change(this, GC_CHANGE_CAPTION_PROPERTY);
              makeDirty();

              break;
            };
          case GC_PROPERTY_FONT_FAMILY:
            {
              FFontFamily = variantToString(value);
              change(this, GC_CHANGE_CAPTION_PROPERTY);
              makeDirty();

              break;
            };
          case GC_PROPERTY_FONT_SIZE:
            {
              FFontSize = variantToInt(value);
              change(this, GC_CHANGE_CAPTION_PROPERTY);
              makeDirty();

              break;
            };
          case GC_PROPERTY_FONT_WEIGHT:
            {
              FWeight = variantToInt(value);
              change(this, GC_CHANGE_CAPTION_PROPERTY);
              makeDirty();

              break;
            };
          case GC_PROPERTY_FONT_STYLE:
            {
              FFontStyle = variantToString(value);
              change(this, GC_CHANGE_CAPTION_PROPERTY);
              makeDirty();

              break;
            };
          case GC_PROPERTY_COLOR:
            {
              GLfloat newColor[4];
              if (stringToColor(variantToString(value), newColor) == 0)
              {
                memcpy(FColor, newColor, sizeof(newColor));
                FColor[3] = 1;
              };
              change(this, GC_CHANGE_CAPTION_PROPERTY);
              makeDirty();

              break;
            };
          case GC_PROPERTY_ALIGNMENT_VERTICAL:
            {
              string s = variantToString(value);
              if (s == "top")
                FVerticalAlignment = GC_ALIGN_LEFT_TOP;
              else 
                if (s == "center")
                  FVerticalAlignment = GC_ALIGN_CENTER;
                else
                  if (s == "bottom")
                    FVerticalAlignment = GC_ALIGN_RIGHT_BOTTOM;
              change(this, GC_CHANGE_CAPTION_PROPERTY);
              makeDirty();

              break;
            };
          case GC_PROPERTY_ALIGNMENT_HORIZONTAL:
            {
              string s = variantToString(value);
              if (s == "left")
                FHorizontalAlignment = GC_ALIGN_LEFT_TOP;
              else 
                if (s == "center")
                  FHorizontalAlignment = GC_ALIGN_CENTER;
                else
                  if (s == "right")
                    FHorizontalAlignment = GC_ALIGN_RIGHT_BOTTOM;
              change(this, GC_CHANGE_CAPTION_PROPERTY);
              makeDirty();

              break;
            };
          case GC_PROPERTY_BIDI_MODE:
            {
              string s = variantToString(value);
              if (s == "left-to-right")
                FBidiMode = GC_BIDI_LEFT_TO_RIGHT;
              else 
                if (s == "right-to-left")
                  FBidiMode = GC_BIDI_RIGHT_TO_LEFT;
              change(this, GC_CHANGE_CAPTION_PROPERTY);
              makeDirty();

              break;
            };
          case GC_PROPERTY_MIN_WIDTH:
            {
              FConstraints.minWidth = variantToFloat(value);
              change(this, GC_CHANGE_ELEMENT_PROPERTY);
              makeDirty();

              break;
            };
          case GC_PROPERTY_MIN_HEIGHT:
            {
              FConstraints.minHeight = variantToFloat(value);
              change(this, GC_CHANGE_ELEMENT_PROPERTY);
              makeDirty();

              break;
            };
          case GC_PROPERTY_MAX_WIDTH:
            {
              FConstraints.maxWidth = variantToFloat(value);
              change(this, GC_CHANGE_ELEMENT_PROPERTY);
              makeDirty();

              break;
            };
          case GC_PROPERTY_MAX_HEIGHT:
            {
              FConstraints.maxHeight = variantToFloat(value);
              change(this, GC_CHANGE_ELEMENT_PROPERTY);
              makeDirty();

              break;
            };
        };

        break;
      };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Renders the caption element.
 */
void CCaptionElement::render(void)
{
  // Apply color if we have been one given.
  if (FHasColor)
  {
    glPixelTransferf(GL_RED_SCALE, FColor[0]);
    glPixelTransferf(GL_GREEN_SCALE, FColor[1]);
    glPixelTransferf(GL_BLUE_SCALE, FColor[2]);
    glPixelTransferf(GL_ALPHA_SCALE, FColor[3]);
  };

  glPixelStorei( GL_UNPACK_ROW_LENGTH, 8);
  glPixelStorei( GL_UNPACK_ALIGNMENT, 1);
  glCallList(FList);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets a new text string.
 *
 * @param newText The new text to set.
 */
void CCaptionElement::text(wstring newText)
{
  FText = newText;
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Validates the bounding box.
 */
void CCaptionElement::validate(void)
{
  if (FDirty)
  {
    FDirty = false;
    fontManager()->boundingBox(FText, FFontFamily, FFontStyle, FWeight, FFontSize, &FBounds);

    // Display list recreation happens in applyAlignment, since we need the alignment offsets for this.
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Called when the current zoom (scale) factor has changed. Recompute font size.
 *
 * @param zoomFactor The new zoom factor.
 * @return True if the zoom change has an effect on this element.
 */
bool CCaptionElement::zoomChanged(float zoomFactor)
{
  int NewFontSize = ROUND(zoomFactor * FFontSize);
  FDirty = FActualFontSize != NewFontSize;
  FActualFontSize = NewFontSize;

  return FDirty;
}

//----------------- CElementListener -----------------------------------------------------------------------------------

void CElementListener::onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason)
{
  if (element != NULL)
    element->change(origin, reason);
}

//----------------------------------------------------------------------------------------------------------------------

void CElementListener::onDestroy(CGCBase* sender)
{
  if (element != NULL)
  {
    if (sender == element->style())
      element->style(NULL);
    else
    {
      // A figure or caption element is about to be freed.
      if (strcmp(sender->className(), "CFigureElement") == 0)
        element->figure()->removeMapping((CFigureElement*) sender);
      element->removeListener(this);
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

void CElementListener::onError(CGCBase* sender, CGCBase* origin, const char* message)
{
  if (element != NULL)
    element->error(origin, message);
}

//----------------- CFigureElement -------------------------------------------------------------------------------------

CFigureElement::CFigureElement(CFigureElementTemplate* aTemplate, CGenericCanvas* canvas): CGCBase(canvas)
{
  _className = "CFigureElement";

  FTemplate = aTemplate;

  FTranslation[0] = 0;
  FTranslation[1] = 0;
  FTranslation[2] = 0;
  FTranslation[3] = 0;

  FLayout = GC_LAYOUT_COLUMN;
  FResizable = false; 
  FStyle = NULL;
  FParent = NULL;
  FFigure = NULL;
  FCaption = NULL;
  FLayouter = new CColumnLayouter(this);
  FExpanded = true;
  FListener.element = this;
}

//----------------------------------------------------------------------------------------------------------------------

CFigureElement::~CFigureElement(void)
{
  if (FStyle != NULL)
  {
    CGCStyle* OldStyle = FStyle;
    // Avoid recursion.
    FStyle = NULL;
    if (OldStyle != NULL && !OldStyle->destroying())
      OldStyle->removeListener(&FListener);
  };

  for (CElementList::iterator iterator = FChildren.begin(); iterator != FChildren.end(); ++iterator)
    delete *iterator; // No need to remove the listener at this point.

  delete FCaption;
  delete FLayouter;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * (Re) computes the overall bounding box for this element. This includes the bounds of the style as well as all children.
 * It is assumed that the caller has already validated the owning figure (and so all contained elements).
 */
void CFigureElement::computeBoundingBox(void)
{
  CBoundingBoxComputer BBComputer;

  if (FStyle != NULL)
    BBComputer.include(NULL, FStyle->boundingBox());

  if (FExpanded)
  {
    TBoundingBox LocalBox;
    FLayouter->reset();
    while (FLayouter->hasNext())
    {
      FLayouter->nextBoundingBox(&LocalBox);
      BBComputer.include(NULL, &LocalBox);
    };
  };

  FBounds = BBComputer.boundingBox();

  // Apply constraints to the bounding box.
  if (FConstraints.maxHeight > -1)
  {
    if (FBounds.lower.y - FBounds.upper.y > FConstraints.maxHeight)
      FBounds.lower.y = FBounds.upper.y + FConstraints.maxHeight;
  };
  if (FConstraints.maxWidth > -1)
  {
    if (FBounds.lower.x - FBounds.upper.x > FConstraints.maxWidth)
      FBounds.lower.x = FBounds.upper.x + FConstraints.maxWidth;
  };
  if (FConstraints.minHeight > -1)
  {
    if (FBounds.lower.y - FBounds.upper.y < FConstraints.minHeight)
      FBounds.lower.y = FBounds.upper.y + FConstraints.minHeight;
  };
  if (FConstraints.minWidth > -1)
  {
    if (FBounds.lower.x - FBounds.upper.x < FConstraints.minWidth)
      FBounds.lower.x = FBounds.upper.x + FConstraints.minWidth;
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Helper method to explicitely trigger creation of a child figure element.
 * The point here is that this must only be called for figures that represent lists, that is,
 * figures that only have one child template element, which can appear any number of times.
 */
CFigureElement* CFigureElement::addSubElement(void)
{
  CFigureElement* result = NULL;
  if (FTemplate != NULL)
  {
    CFigureElementTemplate* child = FTemplate->getListElement();
    // There is only a list element if the template is actually a list.
    if (child != NULL)
    {
      result = CFigureElement::createFromTemplate(canvas(), FFigure, child);
      if (result)
      {
        FChildren.push_back(result);
        change(this, GC_CHANGE_ELEMENT_ADD_CHILD);
        makeDirty();
      };
    };
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a figure element from a layoutTemplate and returns it.
 * 
 * @param owner The controller for the new figure element. It is responsible to free the returned instance.
 * @param layoutTemplate The layoutTemplate to be used when creating the figure element.
 * @param Model The model to which the new element belongs.
 *
 * @return The new figure element instance.
 */
CFigureElement* CFigureElement::createFromTemplate(CGenericCanvas* canvas, CFigure* owner, 
                                                   CFigureElementTemplate* layoutTemplate)
{
  CFigureElement* newElement = NULL;
  
  if (layoutTemplate != NULL)
  {
    newElement = new CFigureElement(layoutTemplate, canvas);
    newElement->FLayout = layoutTemplate->FLayout;
    newElement->FLayouter = LayoutMapper::layouterForElement(newElement);
    newElement->FResizable = layoutTemplate->FResizable;
    newElement->style(layoutTemplate->FStyle);
    newElement->FConstraints = layoutTemplate->FConstraints;

    CCaptionElementTemplate* CaptionTemplate = layoutTemplate->FCaption;
    if (CaptionTemplate != NULL)
    {
      newElement->FCaption = new CCaptionElement(canvas);
      newElement->FCaption->addListener(&newElement->FListener);
      newElement->FCaption->FText = CaptionTemplate->FText;
      newElement->FCaption->FFontFamily = CaptionTemplate->FFontFamily;
      newElement->FCaption->FFontSize = CaptionTemplate->FFontSize;
      newElement->FCaption->FWeight = CaptionTemplate->FWeight;
      newElement->FCaption->FFontStyle = CaptionTemplate->FFontStyle;
      newElement->FCaption->FHorizontalAlignment = CaptionTemplate->FHorizontalAlignment;
      newElement->FCaption->FVerticalAlignment = CaptionTemplate->FVerticalAlignment;
      newElement->FCaption->FParent = newElement;
      newElement->FCaption->FHasColor = CaptionTemplate->FHasColor;
      for (int j = 0; j < 4; ++j)
        newElement->FCaption->FColor[j] = CaptionTemplate->FColor[j];
      newElement->FCaption->FConstraints = CaptionTemplate->FConstraints;
      newElement->FCaption->FOffsetX = CaptionTemplate->FOffsetX;
      newElement->FCaption->FOffsetY = CaptionTemplate->FOffsetY;
    }
    else
      newElement->FCaption = NULL;

    // Child elements.
    for (CElementTemplateList::iterator iterator = layoutTemplate->FChildren.begin(); iterator != layoutTemplate->FChildren.end(); 
      ++iterator)
    {
      CFigureElementTemplate* template_ = *iterator;
      if (template_->occurence() != GC_OCC_ZERO_OR_MORE)
      {
        CFigureElement* childElement = CFigureElement::createFromTemplate(canvas, owner, template_);
        if (childElement != NULL)
        {
          childElement->FParent = newElement;
          childElement->addListener(&newElement->FListener);
          newElement->FChildren.push_back(childElement);
          owner->addMapping(template_->key(), childElement);
        };
      };
    };

    newElement->FFigure = owner;
  };

  return newElement;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Triggers the default action of the figure element that is located at the given location. The given coordinates are
 * already in figure space, so no further conversion is needed.
 *
 * @param Instance The figure instance for which the action was originally triggered.
 * @param x The horizontal coordinate for the hit test.
 * @param y The vertical coordinate for the hit test.
 * @return True if an action was executed, otherwise false.
 */
bool CFigureElement::doAction(CFigureInstance* Instance, const float x, const float y)
{
  bool Result = false;
  bool Handled = false;

  if (boundsContainPoint(FBounds, x, y))
  {
    if (FChildren.size() > 0)
    {
      for (CElementList::const_iterator iterator = FChildren.begin(); iterator != FChildren.end(); ++iterator)
        {
          Handled = (*iterator)->doAction(Instance, x, y);
          if (Handled) 
          {
            Result = true;
            break;
          };
        };
    };

    if (!Handled)
    {
      FExpanded = !FExpanded;
      Handled = true;
      makeDirty();
    };
  };


  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the owning figure for this element.
 *
 * @return The owner of this figure element.
 */
CFigure* CFigureElement::figure(void)
{
  return FFigure;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Called when either a child element or the style changes. The event is propagated to the parent element.
 */
void CFigureElement::makeDirty()
{
  if (!updating() && !FDirty)
  {
    FDirty = true;
    if (FParent != NULL)
      FParent->makeDirty();
    else
      if (FFigure != NULL)
        FFigure->makeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the property given by name. The name syntax is must be something like (here expressed as regex)
 * (container)*(property), where container is a slash and the name of a container class (e.g. layers, figures) and
 * property is the name of a simple property of that container.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this is the index into that list.
 * @return A description of the property value and, if the property is simple, the actual value.
 */
TGCVariant CFigureElement::property(const char* name, unsigned int index)
{
  TGCVariant result;

  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_OWNER:
            {
              result.type = GC_VAR_OBJECT;
              if (FParent != NULL)
                result.reference = FParent;
              else
                result.reference = FFigure;
              break;
            };
          case GC_PROPERTY_NAME:
            {
              result.type = GC_VAR_STRING;
              if (FStyle == NULL)
                result.s = "style element";
              else
              {
                TGCVariant styleName = FStyle->property("name", 0);
                result.s = variantToString(styleName) + " element";
              };

              break;
            };
          case GC_PROPERTY_DESCRIPTION:
            {
              result.type = GC_VAR_STRING;
              result.s = "The visual atom of which figures consist";
              
              break;
            };
          case GC_PROPERTY_LAYOUT:
            {
              result.type = GC_VAR_STRING;
              switch (FLayout)
              {
                case GC_LAYOUT_ROW: 
                  {
                    result.s = "row";
                    break;
                  };
                case GC_LAYOUT_COLUMN:
                  {
                    result.s = "column";
                    break;
                  };
              };

              break;
            };
          case GC_PROPERTY_RESIZABLE:
            {
              result.type = GC_VAR_INT;
              result.i = FResizable;
              
              break;
            };
          case GC_PROPERTY_EXPANDED:
            {
              result.type = GC_VAR_INT;
              result.i = FExpanded;
              
              break;
            };
          case GC_PROPERTY_MIN_WIDTH:
            {
              result.type = GC_VAR_FLOAT;
              result.f = FConstraints.minWidth;
              
              break;
            };
          case GC_PROPERTY_MIN_HEIGHT:
            {
              result.type = GC_VAR_FLOAT;
              result.f = FConstraints.minHeight;
              
              break;
            };
          case GC_PROPERTY_MAX_WIDTH:
            {
              result.type = GC_VAR_FLOAT;
              result.f = FConstraints.maxWidth;
              
              break;
            };
          case GC_PROPERTY_MAX_HEIGHT:
            {
              result.type = GC_VAR_FLOAT;
              result.f = FConstraints.maxHeight;
              
              break;
            };
        };

        break;
      };
    case GC_CONTAINER_STYLE:
      {
        result.type = GC_VAR_OBJECT;
        result.reference = FStyle;
        
        break;
      };
    case GC_CONTAINER_TRANSLATION:
      {
        if (index < 3)
        {
          result.type = GC_VAR_FLOAT;
          result.f = FTranslation[index];
        };
        break;
      };
    case GC_CONTAINER_CHILDREN:
      {
        result.type = GC_VAR_LIST;
        if (index < FChildren.size())
          result.reference = (CGCBase*) FChildren[index];

        break;
      };
    case GC_CONTAINER_CAPTION:
      {
        result.type = GC_VAR_OBJECT;
        result.reference = FCaption;
        
        break;
      };
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the value of the given property, which must be a simple property.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this is the index into that list.
 * @param value The new value of the property. Automatic conversion is performed where possible.
 */
void CFigureElement::property(const char* name, unsigned int index, const TGCVariant& value)
{
  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_LAYOUT:
            {
              string s = variantToString(value);
              if (s == "row")
                FLayout = GC_LAYOUT_ROW;
              else 
                if (s == "column")
                  FLayout = GC_LAYOUT_COLUMN;
              change(this, GC_CHANGE_ELEMENT_PROPERTY);
              break;
            };
          case GC_PROPERTY_RESIZABLE:
            {
              FResizable = variantToBool(value);
              change(this, GC_CHANGE_ELEMENT_PROPERTY);
              break;
            };
          case GC_PROPERTY_EXPANDED:
            {
              FExpanded = variantToBool(value);
              change(this, GC_CHANGE_ELEMENT_PROPERTY);
              break;
            };
          case GC_PROPERTY_MIN_WIDTH:
            {
              FConstraints.minWidth = variantToFloat(value);
              change(this, GC_CHANGE_ELEMENT_PROPERTY);
              
              break;
            };
          case GC_PROPERTY_MIN_HEIGHT:
            {
              FConstraints.minHeight = variantToFloat(value);
              change(this, GC_CHANGE_ELEMENT_PROPERTY);
              
              break;
            };
          case GC_PROPERTY_MAX_WIDTH:
            {
              FConstraints.maxWidth = variantToFloat(value);
              change(this, GC_CHANGE_ELEMENT_PROPERTY);
              
              break;
            };
          case GC_PROPERTY_MAX_HEIGHT:
            {
              FConstraints.maxHeight = variantToFloat(value);
              change(this, GC_CHANGE_ELEMENT_PROPERTY);
              
              break;
            };
        };

        break;
      };
      case GC_CONTAINER_STYLE:
        {
          // Be flexible, accept a new style as string or as reference.
          CGCStyle* newStyle = NULL;
          if (value.type == GC_VAR_STRING)
            newStyle = FFigure->model()->style(utf8ToUtf16(variantToString(value)));
          else
            if (strcmp(value.reference->className(), "CGCStyle"))
              newStyle = (CGCStyle*) value.reference;

          if (newStyle != NULL)
            style(newStyle);

          break;
        };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * This method renders this element and triggers rendering of its (potential) caption as well as its child elements.
 */
void CFigureElement::render()
{
  if (FDirty)
    validate();

  if (FStyle != NULL)
    glCallList(FStyle->displayList());

  if (FExpanded || FStyle == NULL)
  {
    FLayouter->reset();
    while (FLayouter->hasNext())
      FLayouter->renderNext();
  };

  if (FCaption != NULL)
    FCaption->render();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Convenience method to set the text of the caption of this element (if there is a caption at all).
 *
 * @param text The new text to display. Must be UTF-8 encoded.
 */
void CFigureElement::setCaption(const char* text)
{
  if (FCaption != NULL)
  {
    wstring newText = utf8ToUtf16(text);
    FCaption->text(newText);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets a new style to be used for this element.
 *
 * @param NewStyle The new style to be used.
 */
void CFigureElement::style(CGCStyle* NewStyle)
{
  CGCStyle* OldStyle = FStyle;
  if (FStyle != NewStyle)
  {
    if (OldStyle != NULL && !OldStyle->destroying())
      OldStyle->removeListener(&FListener);
    FStyle = NewStyle;
    if (FStyle != NULL)
      FStyle->addListener(&FListener);

    makeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the currently used style.
 *
 * @return The current style for this element.
 */
CGCStyle* CFigureElement::style(void)
{
  return FStyle;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Called before the owner figure creates its display list, so it can be used to create anything necessary that
 * must not be done while a display list is being compiled.
 */
void CFigureElement::validate(void)
{
  if (FDirty)
  {
    FDirty = false;

    if (FCaption != NULL)
      FCaption->validate();

    for (CElementList::iterator iterator = FChildren.begin(); iterator != FChildren.end(); ++iterator)
      (*iterator)->validate();
    computeBoundingBox();

    if (FCaption != NULL)
      FCaption->applyAlignment();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Called when the current scale factor was changed. Recompute caption if necessary.
 *
 * @param zoomFactor The current zoom (scale) factor.
 * @return True if the zoom change has an effect on this element.
 */
bool CFigureElement::zoomChanged(float zoomFactor)
{
  if (FCaption != NULL)
    if (FCaption->zoomChanged(zoomFactor))
      FDirty = true;

  for (CElementList::iterator iterator = FChildren.begin(); iterator != FChildren.end(); ++iterator)
    if ((*iterator)->zoomChanged(zoomFactor))
      FDirty = true;

  return FDirty;
}

//----------------- CFigureELementListener -----------------------------------------------------------------------------

void CFigureElementListener::onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason)
{
  if (figure != NULL)
    figure->change(origin, reason);
}

//----------------------------------------------------------------------------------------------------------------------

void CFigureElementListener::onDestroy(CGCBase* sender)
{
  if (figure != NULL)
  {
    CFigureElement* element = (CFigureElement*) sender;
    if (element != NULL)
      figure->freeNotification(element);
  };
}

//----------------------------------------------------------------------------------------------------------------------

void CFigureElementListener::onError(CGCBase* sender, CGCBase* origin, const char* message)
{
  if (figure != NULL)
    figure->error(origin, message);
}

//----------------- CFigure --------------------------------------------------------------------------------------------

CFigure::CFigure(CGCModel* owner, CFigureTemplate* layoutTemplate): CGCBase(owner->canvas())
{
  _className = "CFigure";
  FModel = owner;
  if (owner != NULL)
    owner->addFigure(this);
  FContent = NULL;

  // Initialize with useful values.
  FRotation[0] = 0;
  FRotation[1] = 0;
  FRotation[2] = 0;
  FRotation[3] = 1;

  FScaling[0] = 1;
  FScaling[1] = 1;
  FScaling[2] = 1;

  FTranslation[0] = 0;
  FTranslation[1] = 0;
  FTranslation[2] = 0;

  FTemplate = NULL;
  buildFromTemplate(layoutTemplate);

  FDirty = true;
  FListener.figure = this;
}

//----------------------------------------------------------------------------------------------------------------------

CFigure::~CFigure(void)
{
  delete FContent;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Applies the current translation, rotation and scalie factors.
 */
void CFigure::applyTransformations(void)
{
  glTranslatef(FTranslation[0], FTranslation[1], FTranslation[2]);
  // Index 0 contains the angle, while the other three coordinates form the axis to rotate around.
  glRotatef(FRotation[0], FRotation[1], FRotation[2], FRotation[3]);
  glScalef(FScaling[0], FScaling[1], FScaling[2]);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Parses the given layoutTemplate and creates its child structure.
 */
void CFigure::buildFromTemplate(CFigureTemplate* layoutTemplate)
{
  if (layoutTemplate->content() == NULL)
  {
    wstring name = layoutTemplate->type() + L"@" + layoutTemplate->layoutClass();
    string message = "CFigure::buildFromTemplate: No content for layoutTemplate \"" + utf16ToUtf8(name) + "\" found.";
    error(this, message.c_str());
  }
  else
  {
    if (FContent != NULL)
    {
      delete FContent; // No need to remove the listener here.
      FContent = NULL;
    };

    FTemplate = layoutTemplate;
    FContent = CFigureElement::createFromTemplate(canvas(), this, layoutTemplate->content());
    if (FContent != NULL)
      FContent->addListener(&FListener);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates the display list of this figure (and all child figures) if necessary.
 */
void CFigure::validate(void)
{
  if (FDirty)
  {
    FDirty = false;
    FMinimalScale = FScaling[0] < FScaling[1] ? FScaling[0] : FScaling[1];

    // Let the content prepare itself if necessary.
    if (FContent != NULL)
    {
      FContent->validate();

      // Update bounding box.
      TMatrix M;

      memcpy(M, Identity, sizeof(TMatrix));
      matrixTranslate(M, FTranslation[0], FTranslation[1], FTranslation[2]);
      matrixRotate(M, FRotation[0], FRotation[1], FRotation[2], FRotation[3]);
      matrixScale(M, FScaling[0], FScaling[1], FScaling[2]);

      // Take the bounds of the figure this instance standes for and transform that with 
      // the local rotation, translation and scaling factors.
      TBoundingBox contentBounds = FContent->bounds();
      FBounds.lower = matrixTransform(M, contentBounds.lower);
      FBounds.upper = matrixTransform(M, contentBounds.upper);
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds a new mapping between the given key and element to the figure, for later lookup.
 *
 * @param path The key used for the mapping.
 * @param element The element to map to the given key.
 */
void CFigure::addMapping(wstring path, CFigureElement* element)
{
  if (path.size() > 0)
    FElementMap[path] = element;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines the child figure element with the given key and returns it.
 *
 * @param key The key to search for (UTF-8 encoded).
 * @return The figure element that corresponds to the given key or NULL if there is none.
 */
CFigureElement* CFigure::elementFromKey(const char* key)
{
  return elementFromKey(utf8ToUtf16(key));
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines the child figure element with the given key and returns it.
 * See also description of CFigureElement::elementFromKey.
 *
 * @param key The key to search for.
 * @return The figure element that corresponds to the given key or NULL if there is none.
 */
CFigureElement* CFigure::elementFromKey(wstring key)
{
  CFigureElement* result = NULL;
  CFigureElementMap::iterator iterator = FElementMap.find(key);
  if (iterator != FElementMap.end())
    result = FElementMap[key];

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

void CFigure::freeNotification(CFigureElement* object)
{
  if (object == FContent)
  {
    FContent = NULL;
    makeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Marks the display list for this figure as invalid, hence it will be recreated next time validate is called.
 */
void CFigure::makeDirty(void)
{
  if (!FDirty)
  {
    FDirty = true;
    change(this, GC_CHANGE_CANVAS_REFRESH);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the property given by name. The name syntax is must be something like (here expressed as regex)
 * (container)*(property), where container is a slash and the name of a container class (e.g. layers, figures) and
 * property is the name of a simple property of that container.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this is the index into that list.
 * @return The value of the property, if found.
 */
TGCVariant CFigure::property(const char* name, unsigned int index)
{
  TGCVariant result;

  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_NAME:
            {
              result.type = GC_VAR_STRING;
              if (FTemplate == NULL)
                result.s = "figure";
              else
                result.s = utf16ToUtf8(FTemplate->type() + L"@" + FTemplate->layoutClass() + L" figure");
              break;
            };
          case GC_PROPERTY_DESCRIPTION:
            {
              result.type = GC_VAR_STRING;
              result.s = "The visual base element of the canvas";
              break;
            };
          case GC_PROPERTY_OWNER:
            {
              result.type = GC_VAR_OBJECT;
              result.reference = FModel;
              break;
            };
        };
        break;
      };
    case GC_CONTAINER_SCALING:
      {
        if (index < 3)
        {
          result.type = GC_VAR_FLOAT;
          result.f = FScaling[index];
        };
        break;
      };
    case GC_CONTAINER_TRANSLATION:
      {
        if (index < 3)
        {
          result.type = GC_VAR_FLOAT;
          result.f = FTranslation[index];
        };
        break;
      };
    case GC_CONTAINER_ROTATION:
      {
        if (index < 4)
        {
          result.type = GC_VAR_FLOAT;
          result.f = FRotation[index];
        };

        break;
      };
    case GC_CONTAINER_CONTENT:
      {
        result.type = GC_VAR_OBJECT;
        result.reference = FContent;

        break;
      };
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the value of the given property, which must be a simple property.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this is the index into that list.
 * @param Value The new value of the property. Automatic conversion is performed where possible.
 */
void CFigure::property(const char* name, unsigned int index, const TGCVariant& value)
{
  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_DESCRIPTION:
            {
              // This branch only as a placeholder.
              break;
            };
        };

        break;
      };
    case GC_CONTAINER_SCALING:
      {
        if (index < 3)
        {
          FScaling[index] = variantToFloat(value);
          change(this, GC_CHANGE_FIGURE_PROPERTY);
          makeDirty();
        };

        break;
      };
    case GC_CONTAINER_TRANSLATION:
      {
        if (index < 3)
        {
          FTranslation[index] = variantToFloat(value);
          change(this, GC_CHANGE_FIGURE_PROPERTY);
          makeDirty();
        };

        break;
      };
    case GC_CONTAINER_ROTATION:
      {
        if (index < 4)
        {
          FRotation[index] = variantToFloat(value);
          change(this, GC_CHANGE_FIGURE_PROPERTY);
          makeDirty();
        };

        break;
      };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes a previously added mapping (@see addMapping).
 *
 * @param element The element to map to the given key.
 */
void CFigure::removeMapping(CFigureElement* element)
{
  wstring key = element->template_()->key();
  CFigureElementMap::iterator iterator = FElementMap.find(key);
  if (iterator != FElementMap.end())
    FElementMap.erase(iterator);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks the validity of the figure display list and executes it.
 *
 * @param The current scale factor.
 */
void CFigure::render(float currentZoom)
{
  if (FContent != NULL)
  {
    currentZoom *= FMinimalScale;
    if (FLastZoom != currentZoom)
    {
      FLastZoom = currentZoom;
      if (FContent->zoomChanged(currentZoom))
        FDirty = true;
    };

    if (FDirty)
      validate();

    glPushMatrix();

    applyTransformations();
    FContent->render();
  
    glPopMatrix();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Turns the figure around the given axis by the angle Angle (in radians). This version of Rotate uses a single float 
 * values in the parameter list.
 *
 * @param Angle The angle in radians to turn the figure.
 * @param Rx The x part of the rotation axis.
 * @param Ry The y part of the rotation axis.
 * @param Rz The z part of the rotation axis.
 * @note Currently there is no accumulative version of Rotate available (requires a quaternion lib, which we don't have yet).
 */
void CFigure::rotate(float Angle, float Rx, float Ry, float Rz)
{
  FRotation[0] = Angle;
  FRotation[1] = Rx;
  FRotation[2] = Ry;
  FRotation[3] = Rz;
  change(this, GC_CHANGE_FIGURE_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Turns the figure around the given axis by the angle Angle (in radians). This version of Rotate uses a vector for the
 * rotation axis in the parameter list.
 *
 * @param Angle The angle in radians to turn the figure.
 * @param Axis The axis around which the figure is to be rotated.
 * @note Currently there is no accumulative version of Rotate available (requires a quaternion lib, which we don't have yet).
 */
void CFigure::rotate(float Angle, const float Axis[3])
{
  FRotation[0] = Angle;
  FRotation[1] = Axis[0];
  FRotation[2] = Axis[1];
  FRotation[3] = Axis[2];
  change(this, GC_CHANGE_FIGURE_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Scales the figure by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses single float values as parameters.
 *
 * @param Sx The scale factor in x direction.
 * @param Sy The scale factor in y direction.
 * @param Sz The scale factor in z direction.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CFigure::scale(float Sx, float Sy, float Sz, bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Sx;
    FScaling[1] += Sy;
    FScaling[2] += Sz;
  }
  else
  {
    FScaling[0] = Sx;
    FScaling[1] = Sy;
    FScaling[2] = Sz;
  };
  change(this, GC_CHANGE_FIGURE_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scales the figure by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses an array of values in the parameter list.
 *
 * @param Factor An array containing the three scale values for x, y and z.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CFigure::scale(const float Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Factor[0];
    FScaling[1] += Factor[1];
    FScaling[2] += Factor[2];
  }
  else
  {
    FScaling[0] = Factor[0];
    FScaling[1] = Factor[1];
    FScaling[2] = Factor[2];
  };
  change(this, GC_CHANGE_FIGURE_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Moves the figure by the amount given in Factor. If Accumulative is true then the new translation factors are 
 * multiplied with the existing values. This version of Translate uses an array for the values in the parameter list.
 *
 * @param Tx Scale factor for the x axis.
 * @param Ty Scale factor for the y axis.
 * @param Tz Scale factor for the z axis.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CFigure::translate(float Tx, float Ty, float Tz, bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Tx;
    FTranslation[1] += Ty;
    FTranslation[2] += Tz;
  }
  else
  {
    FTranslation[0] = Tx;
    FTranslation[1] = Ty;
    FTranslation[2] = Tz;
  };
  change(this, GC_CHANGE_FIGURE_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the figure by the amount given in Factor. If Accumulative is true then the new translation factors are 
 * multiplied with the existing values. This version of Translate uses an array for the values in the parameter list.
 *
 * @param Factor An array of translation values, for each axis one.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CFigure::translate(const float Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Factor[0];
    FTranslation[1] += Factor[1];
    FTranslation[2] += Factor[2];
  }
  else
  {
    FTranslation[0] = Factor[0];
    FTranslation[1] = Factor[1];
    FTranslation[2] = Factor[2];
  };
  change(this, GC_CHANGE_FIGURE_PROPERTY);
  makeDirty();
}

//----------------- CFigureListener ------------------------------------------------------------------------------------

void CFigureListener::onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason)
{
  if (instance != NULL)
  {
    switch (reason)
    {
      case GC_CHANGE_FIGURE_PROPERTY:
        {
          instance->makeDirty();
          break;
        };      
      case GC_CHANGE_FIGURE_EXCHANGE:
        {
          instance->replaceFigure((CFigure*) origin);
          break;
        };
    };
    instance->change(origin, reason); 
  };
}

//----------------------------------------------------------------------------------------------------------------------

void CFigureListener::onDestroy(CGCBase* sender)
{ 
  if (instance != NULL)
    instance->onDestroy(sender); 
}

//----------------------------------------------------------------------------------------------------------------------

void CFigureListener::onError(CGCBase* sender, CGCBase* origin, const char* message)
{
  if (instance != NULL)
    instance->error(origin, message);
}

//----------------- CFigureInstance ------------------------------------------------------------------------------------

CFigureInstance::CFigureInstance(CLayer* owner, CFigure* figure): CGCBase(owner->canvas())
{
  _className = "CFigureInstance";
  FLayer = NULL;
  FFigure = figure;
  FSelected = false;

  FRotation[0] = 0;
  FRotation[1] = 0;
  FRotation[2] = 0;
  FRotation[3] = 1;

  FScaling[0] = 1;
  FScaling[1] = 1;
  FScaling[2] = 1;

  FTranslation[0] = 0;
  FTranslation[1] = 0;
  FTranslation[2] = 0;

  FDirty = true;
  if (owner != NULL)
    owner->addInstance(this);
  FListener.instance = this;
  if (figure != NULL)
    figure->addListener(&FListener);
}

//----------------------------------------------------------------------------------------------------------------------

CFigureInstance::~CFigureInstance(void)
{
  if (FLayer != NULL)
  {
    if (FSelected) 
      FLayer->canvas()->removeFromSelection(this);
    if (!FLayer->updating())
      FLayer->removeInstance(this);
  };

  if ((FFigure != NULL) && (!FFigure->destroying()))
    FFigure->removeListener(&FListener);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Applies the current translation, rotation and scalie factors.
 */
void CFigureInstance::applyTransformations(void)
{
  glTranslatef(FTranslation[0], FTranslation[1], FTranslation[2]);
  // Index 0 contains the angle, while the other three coordinates form the axis to rotate around.
  glRotatef(FRotation[0], FRotation[1], FRotation[2], FRotation[3]);
  glScalef(FScaling[0], FScaling[1], FScaling[2]);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Triggers the default action of the figure element that is located at the given location. The given coordinates must
 * be converted to figure space first, though. They are given in window coordinates (at subpixel accuracy).
 * On call of this function the current view's transformations are already applied.
 *
 * @param x The horizontal coordinate for the hit test.
 * @param y The vertical coordinate for the hit test.
 * @return True if an action was executed, otherwise false.
 */
bool CFigureInstance::doAction(const float x, const float y)
{
  bool Result = false;

  if (FLayer != NULL && FFigure != NULL && FFigure->FContent != NULL)
  {
    FLayer->applyTransformations();

    // Apply own transformations (both, for the instance and for the figure) and use the resulting
    // transformation matrix to convert the given window coordinates into figure space.
    applyTransformations();
    FFigure->applyTransformations();

    GLint Viewport[4];
    glGetIntegerv(GL_VIEWPORT, Viewport);

    GLdouble ModelviewMatrix[16];
    glGetDoublev(GL_MODELVIEW_MATRIX, ModelviewMatrix);

    GLdouble ProjectionMatrix[16];
    glGetDoublev(GL_PROJECTION_MATRIX, ProjectionMatrix);

    // Convert window coordinates into object (scene) coordinates.
    double Localx, Localy, Localz;
    gluUnProject(x, Viewport[3] - y, 0, ModelviewMatrix, ProjectionMatrix, Viewport, &Localx, &Localy, &Localz);

    Result = FFigure->FContent->doAction(this, (float) Localx, (float) Localy);
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Called by a class (usually CFigure) with which we registered us as notification sink and which is
 * about to be destroyed.
 */
void CFigureInstance::onDestroy(CGCBase* figure)
{
  if (figure == FFigure)
    delete this;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Replaces the figure this instance points to.
 *
 * @param figure The new figure this instance should use from now on. It must not be NULL.
 */
void CFigureInstance::replaceFigure(CFigure* figure)
{
  FFigure->removeListener(&FListener);
  FFigure = figure;
  FFigure->addListener(&FListener);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Marks the display list for this figure instance as invalid, hence it will be recreated next time validate 
 * is called. If a list already exists then it is freed. If the figure is connected to a parent figure then this parent 
 * is invalidated as well.
 */
void CFigureInstance::makeDirty(void)
{
  if (!FDirty)
  {
    FDirty = true;
    if (FLayer != NULL)
    {
      if (FSelected) 
        FLayer->canvas()->invalidateSelectionBounds(this);
      if (!FLayer->updating())
        FLayer->makeDirty();
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Validates the associated figure if it is dirty.
 */
void CFigureInstance::validate(void)
{
  if (FDirty)
  {
    FDirty = false;
    FMinimalScale = FScaling[0] < FScaling[1] ? FScaling[0] : FScaling[1];

    if (FFigure != NULL)
    {
      if (FFigure->FDirty)
        FFigure->validate();

      // Update bounding box.
      TMatrix M;

      memcpy(M, Identity, sizeof(TMatrix));
      matrixTranslate(M, FTranslation[0], FTranslation[1], FTranslation[2]);
      matrixRotate(M, FRotation[0], FRotation[1], FRotation[2], FRotation[3]);
      matrixScale(M, FScaling[0], FScaling[1], FScaling[2]);

      // Take the bounds of the figure this instance standes for and transform that with 
      // the local rotation, translation and scaling factors.
      TBoundingBox FigureBounds = FFigure->bounds();
      FBounds.lower = matrixTransform(M, FigureBounds.lower);
      FBounds.upper = matrixTransform(M, FigureBounds.upper);
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines whether the bounds of this instance overlap the given coordinates.
 *
 * @param x The horizontal hit coordinate.
 * @param y The vertical hit coordinate.
 * @return True if given coordinates are within the instance's bounds, otherwise false,
 */
bool CFigureInstance::containsPoint(const float x, const float y)
{
  return (FBounds.upper.x <= x) && (FBounds.upper.y <= y) && (FBounds.lower.x >= x) && (FBounds.lower.y >= y);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines whether this instance lies fully or partially within the given box.
 *
 * @param Box The coordinates to check against to learn if this figure instances is in.
 * @return True if the bounds of this instance at least partially overlap the given box bounds.
 */
bool CFigureInstance::overlaps(TBoundingBox& Box)
{
  return boundsIntersect(Box, FBounds);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the property given by name. The name syntax is must be something like (here expressed as regex)
 * (container)*(property), where container is a slash and the name of a container class (e.g. layers, figures) and
 * property is the name of a simple property of that container.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this is the index into that list.
 * @return A description of the property value and, if the property is simple, the actual value.
 */
TGCVariant CFigureInstance::property(const char* name, unsigned int index)
{
  TGCVariant result;

  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_NAME:
            {
              // figure instance do not have names.
              result.type = GC_VAR_STRING;
              if (FFigure == NULL)
                result.s = "figure instance";
              else
              {
                TGCVariant figureName = FFigure->property("name", 0);
                result.s = variantToString(figureName) + " instance";
              };
              break;
            };
          case GC_PROPERTY_DESCRIPTION:
            {
              result.type = GC_VAR_STRING;
              result.s = "An instance of a figure on a layer";
              break;
            };
          case GC_PROPERTY_OWNER:
            {
              result.type = GC_VAR_OBJECT;
              result.reference = FLayer;
              break;
            };
          case GC_PROPERTY_SELECTED:
            {
              result.type = GC_VAR_INT;
              result.i = FSelected;
              break;
            };
        };
        break;
      };
    case GC_CONTAINER_SCALING:
      {
        if (index < 3)
        {
          result.type = GC_VAR_FLOAT;
          result.f = FScaling[index];
        };
        break;
      };
    case GC_CONTAINER_TRANSLATION:
      {
        if (index < 3)
        {
          result.type = GC_VAR_FLOAT;
          result.f = FTranslation[index];
        };
        break;
      };
    case GC_CONTAINER_ROTATION:
      {
        if (index < 4)
        {
          result.type = GC_VAR_FLOAT;
          result.f = FRotation[index];
        };
        break;
      };
    case GC_CONTAINER_FIGURE:
      {
        result.type = GC_VAR_OBJECT;
        result.reference = FFigure;

        break;
      };
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the value of the given property, which must be a simple property.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this is the index into that list.
 * @param Value The new value of the property. Automatic conversion is performed where possible.
 */
void CFigureInstance::property(const char* name, unsigned int index, const TGCVariant& value)
{
  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_SELECTED:
            {
              FSelected = variantToBool(value);
              change(this, GC_CHANGE_FINSTANCE_PROPERTY);
              break;
            };
        };

        break;
      };
    case GC_CONTAINER_SCALING:
      {
        if (index < 3)
          FScaling[index] = variantToFloat(value);
        change(this, GC_CHANGE_FINSTANCE_PROPERTY);
        makeDirty();

        break;
      };
    case GC_CONTAINER_TRANSLATION:
      {
        if (index < 3)
          FTranslation[index] = variantToFloat(value);
        change(this, GC_CHANGE_FINSTANCE_PROPERTY);
        makeDirty();

        break;
      };
    case GC_CONTAINER_ROTATION:
      {
        if (index < 4)
          FRotation[index] = variantToFloat(value);
        change(this, GC_CHANGE_FINSTANCE_PROPERTY);
        makeDirty();

        break;
      };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks the validity of the figure instance display list and executes it.
 *
 * @param currentZoom The current scale factor.
 */
void CFigureInstance::render(float currentZoom)
{
  if (FDirty)
    validate();

  glPushMatrix();

  currentZoom *= FMinimalScale;
  applyTransformations();
  if (FFigure != NULL)
    FFigure->render(currentZoom);

  glPopMatrix();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Rotates the figure around the given axis by the angle Angle (in radians). This version of Rotate uses a vector for the
 * rotation axis in the parameter list.
 *
 * @param Angle The rotation angle in radians.
 * @param Rx The x part of the axis around which to rotate the figure instance.
 * @param Ry The y part of the axis around which to rotate the figure instance.
 * @param Rz The z part of the axis around which to rotate the figure instance.
 * @note: Currently there is no accumulative version of Rotate available (requires a quaternion lib, which we don't have yet).
 */
void CFigureInstance::rotate(float Angle, float Rx, float Ry, float Rz)
{
  FRotation[0] = Angle;
  FRotation[1] = Rx;
  FRotation[2] = Ry;
  FRotation[3] = Rz;
  change(this, GC_CHANGE_FINSTANCE_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Rotates the figure around the given axis by the angle Angle (in radians). This version of Rotate uses a vector for the
 * rotation axis in the parameter list.
 *
 * @param Angle The rotation angle in radians.
 * @param Axis The axis around which to rotate the figure instance.
 * @note: Currently there is no accumulative version of Rotate available (requires a quaternion lib, which we don't have yet).
 */
void CFigureInstance::rotateV(float Angle, const float Axis[3])
{
  FRotation[0] = Angle;
  FRotation[1] = Axis[0];
  FRotation[2] = Axis[1];
  FRotation[3] = Axis[2];
  change(this, GC_CHANGE_FINSTANCE_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scales the figure by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses an array of values in the parameter list.
 *
 * @param Sx The scale value for the x-axis
 * @param Sy The scale value for the y-axis
 * @param Sz The scale value for the z-axis
 * @param Accumulative If true then the new scale values are added to any previously assigned values.
 */
void CFigureInstance::scale(float Sx, float Sy, float Sz, bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Sx;
    FScaling[1] += Sy;
    FScaling[2] += Sz;
  }
  else
  {
    FScaling[0] = Sx;
    FScaling[1] = Sy;
    FScaling[2] = Sz;
  };
  change(this, GC_CHANGE_FINSTANCE_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scales the figure by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses an array of values in the parameter list.
 *
 * @param Factor Contains the scaling factors for all three axes. Index 0 contains the value for the x-axis, index 1
 *               that for the y-axis and index 2 for z.
 * @param Accumulative If true then the new scale values are added to any previously assigned values.
 */
void CFigureInstance::scaleV(const float Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Factor[0];
    FScaling[1] += Factor[1];
    FScaling[2] += Factor[2];
  }
  else
  {
    FScaling[0] = Factor[0];
    FScaling[1] = Factor[1];
    FScaling[2] = Factor[2];
  };
  change(this, GC_CHANGE_FINSTANCE_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Tells the caller whether this instance is currently selected.
 * 
 * @return True if this figure instance is currently selected, otherwise false.
 */
bool CFigureInstance::selected(void)
{
  return FSelected;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the figure by the amount given in Factor. If Accumulative is true then the new translation factors are multiplied
 * with the existing values.
 *
 * @param Tx The scale factor to apply on the x-axis.
 * @param Ty The scale factor to apply on the y-axis.
 * @param Tz The scale factor to apply on the z-axis.
 * @param Accumulative If true scaling factors are added to the values already set previously.
 */
void CFigureInstance::translate(float Tx, float Ty, float Tz, bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Tx;
    FTranslation[1] += Ty;
    FTranslation[2] += Tz;
  }
  else
  {
    FTranslation[0] = Tx;
    FTranslation[1] = Ty;
    FTranslation[2] = Tz;
  };
  change(this, GC_CHANGE_FINSTANCE_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the figure by the amount given in Factor. If Accumulative is true then the new translation factors are multiplied
 * with the existing values.
 *
 * @param Factor The scale factor to apply. Index 0 contains the factor for the x-axis etc.
 * @param Accumulative If true scaling factors are added to the values already set previously.
 */
void CFigureInstance::translateV(const float Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Factor[0];
    FTranslation[1] += Factor[1];
    FTranslation[2] += Factor[2];
  }
  else
  {
    FTranslation[0] = Factor[0];
    FTranslation[1] = Factor[1];
    FTranslation[2] = Factor[2];
  };
  change(this, GC_CHANGE_FINSTANCE_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

