/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_layer.cpp 
 * @brief Implementation of the GC layer class.
 * 
 */

#include "myx_gc.h"
#include "myx_gc_figure.h"
#include "myx_gc_layer.h"
#include "myx_gc_gl_helper.h"

//----------------- CInstanceListener ----------------------------------------------------------------------------------

void CInstanceListener::onChange(CGCBase* sender, CGCBase* origin, TGCChangeReason reason)
{
  if (layer != NULL)
    layer->change(origin, reason);
}

//----------------------------------------------------------------------------------------------------------------------

void CInstanceListener::onDestroy(CGCBase* object)
{
  if (layer != NULL)
    layer->removeInstance((CFigureInstance*) object);
}

//----------------------------------------------------------------------------------------------------------------------

void CInstanceListener::onError(CGCBase* sender, CGCBase* origin, const char* message)
{
  if (layer != NULL)
    layer->error(origin, message);
}

//----------------- CLayer ---------------------------------------------------------------------------------------------

CLayer::CLayer(string Name, CGenericCanvas* canvas): CGCBase(canvas)
{
  _className = "CLayer";
  FCanvas = canvas;
  FName = utf8ToUtf16(Name);

  // Initialize with useful values.
  FScaling[0] = 1;
  FScaling[1] = 1;
  FScaling[2] = 1;

  FTranslation[0] = 0;
  FTranslation[1] = 0;
  FTranslation[2] = 0;

  FDirty = true;
  FVisible = true;
  FEnabled = true;

  FListener.layer = this;
}

//----------------------------------------------------------------------------------------------------------------------

CLayer::~CLayer(void)
{
  beginUpdate();
  if ((FCanvas != NULL) && (!FCanvas->updating()))
    FCanvas->removeLayer(this);
  clear();
  endUpdate();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Applies the layer's transformations for rendering, feedback etc.
 */
void CLayer::applyTransformations()
{
  glTranslatef(FTranslation[0], FTranslation[1], FTranslation[2]);
  glScalef(FScaling[0], FScaling[1], FScaling[2]);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Triggers the error checking of the canvas.
 */
void CLayer::checkError(void)
{
  if (FCanvas != NULL)
    FCanvas->checkError();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Marks the display list for this layer as invalid, hence it will be recreated next time validate is called.
 * If a list already exists then it is freed.
 */
void CLayer::makeDirty(void)
{
  if (!FDirty)
  {
    FDirty = true;
    change(this, GC_CHANGE_CANVAS_REFRESH);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/*
 * Helper method to determine the transformed vertices of the given figure instance. The layer applies its own
 * transformations and only renders the figure instance.
 *
 * @param instance The figure instance for which feedback data is requested.
 */
void CLayer::renderFeedback(CFigureInstance* instance)
{
  applyTransformations();
  instance->render(1);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Renders layer content that is not determined by figure instances. This method might be overridden by descendants.
 */
void CLayer::renderLayerContent(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates the display list of this figure (and all child figures) if necessary.
 */
void CLayer::validate(void)
{
  if (FDirty)
  {
    FDirty = false;

    FMinimalScale = FScaling[0] < FScaling[1] ? FScaling[0] : FScaling[1];
    validateLayerContent();

    // Give the instances the opportunity to trigger validation of their associated figures.
    for (CFigureInstances::iterator iterator = FInstances.begin(); iterator != FInstances.end(); ++iterator)
      (*iterator)->validate();

    checkError();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Prepares layer content that is not determined by figure instances. This method might be overridden by descendants.
 */
void CLayer::validateLayerContent(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds the given figure instance to the end of the instance list. If instance belongs to another layer currently 
 * it is removed from the other's instance list first.
 *
 * @param instance The figure instance to add.
 */
void CLayer::addInstance(CFigureInstance* instance)
{
  if (instance->FLayer != this)
  {
    beginUpdate();
    if (instance->FLayer != NULL)
      instance->FLayer->removeInstance(instance);
    instance->addListener(&FListener);
    FInstances.push_back(instance);
    instance->FLayer = this;
    endUpdate();
    change(this, GC_CHANGE_LAYER_ADD_INSTANCE);
    makeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all figure instances from this layer.
 */
void CLayer::clear(void)
{
  beginUpdate();

  for (CFigureInstances::iterator iterator = FInstances.begin(); iterator != FInstances.end(); ++iterator)
  {
    CFigureInstance* instance = *iterator;
    instance->FLayer = NULL;
    if (instance->selected() && FCanvas != NULL)
      FCanvas->removeFromSelection(instance);
    delete instance;
  };
  FInstances.clear();
  endUpdate();

  change(this, GC_CHANGE_LAYER_CLEAR);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a new instance for the given figure and adds it to this layer.
 *
 * @param Figure The figure for which the instance is to be created.
 * @return A new figure instance.
 */
CFigureInstance* CLayer::createInstance(CFigure* Figure)
{
  beginUpdate();
  CFigureInstance* instance = new CFigureInstance(this, Figure);
  instance->addListener(&FListener);
  endUpdate();

  return instance;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Fills the hit results with all figure instances whose bounds contain the given coordinates.
 *
 * @param Hits [out] The hit collection that is updated.
 * @param x The horizontal hit point coordinated given in view space.
 * @param y The vertical coordinate.
 * @param SingleHit If true only one hit is returned.
 */
void CLayer::getHitTestInfoAt(CHitResults* Hits, const float x, const float y, bool SingleHit)
{
  // Bounding boxes of figure instances are already in layer space coordinates. We only need to convert the
  // given hit point to our local coordinate system.
  float LocalX = (x - FTranslation[0]) / FScaling[0];
  float LocalY = (y - FTranslation[1]) / FScaling[1];

  // Iterate backwards. Figures rendered later lay on top of earlier rendered figures.
  for (CFigureInstances::reverse_iterator iterator = FInstances.rbegin(); iterator != FInstances.rend(); ++iterator)
  {
    CFigureInstance* instance = *iterator;
    if (instance->containsPoint(LocalX, LocalY))
    {
      Hits->addHit(instance);
      if (SingleHit)
        break;
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the layer's enabled state.
 *
 * @param IsEnabled Set it to true if you want the layer to be visible.
 */
void CLayer::enabled(bool IsEnabled)
{
  if (FEnabled != IsEnabled)
  {
    FEnabled = IsEnabled;
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the current enabled state.
 *
 * @return The current enabled state.
 */
bool CLayer::enabled(void)
{
  return FEnabled;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the property given by path. The path syntax is must be something like (here expressed as regex)
 * (container)*(property), where container is a slash and the name of a container class (e.g. layers, figures) and
 * property is the name of a simple property of that container.
 *
 * @param name The name of the property to return.
 * @param index If the property is a list then this is the index into that list.
 * @return A description of the property value and, if the property is simple, the actual value.
 */
TGCVariant CLayer::property(const char* name, unsigned int index)
{
  TGCVariant result;

  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_NAME:
            {
              result.type = GC_VAR_STRING;
              result.s = utf16ToUtf8(FName);
              break;
            };
          case GC_PROPERTY_DESCRIPTION:
            {
              result.type = GC_VAR_STRING;
              result.s = "A collection of figure instances.";
              break;
            };
          case GC_PROPERTY_OWNER:
            {
              result.type = GC_VAR_OBJECT;
              result.reference = FCanvas;
              break;
            };
          case GC_PROPERTY_VISIBLE:
            {
              result.type = GC_VAR_INT;
              result.i = FVisible;
              break;
            };
          case GC_PROPERTY_ENABLED:
            {
              result.type = GC_VAR_INT;
              result.i = FEnabled;
              break;
            };
        };
        break;
      };
    case GC_CONTAINER_FIGURE_INSTANCES:
      {
        result.type = GC_VAR_LIST;
        if (index < FInstances.size())
          result.reference = (CGCBase*) FInstances[index];
        break;
      };
    case GC_CONTAINER_GROUPS:
      {
        result.type = GC_VAR_LIST;
        // TODO: Should actually be the groups.
        if (index < FInstances.size())
          result.reference = (CGCBase*) FInstances[index];
        break;
      };
    case GC_CONTAINER_SCALING:
      {
        if (index < 3)
        {
          result.type = GC_VAR_FLOAT;
          result.f = FScaling[index];
        };
        break;
      };
    case GC_CONTAINER_TRANSLATION:
      {
        if (index < 3)
        {
          result.type = GC_VAR_FLOAT;
          result.f = FTranslation[index];
        };
        break;
      };
  };

  return result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the value of the given property, which must be a simple property.
 *
 * @param name The name of the property.
 * @param index If the property is a list then this is the index into that list.
 * @param Value The new value of the property. Automatic conversion is performed where possible.
 */
void CLayer::property(const char* name, unsigned int index, const TGCVariant& value)
{
  switch (getContainerID(name))
  {
    case GC_CONTAINER_UNKNOWN:
      {
        switch (getPropertyID(name))
        {
          case GC_PROPERTY_NAME:
            {
              FName = utf8ToUtf16(variantToString(value));
              change(this, GC_CHANGE_LAYER_PROPERTY);

              break;
            };
          case GC_PROPERTY_VISIBLE:
            {
              FVisible = variantToBool(value);
              change(this, GC_CHANGE_LAYER_PROPERTY);
              
              break;
            };
          case GC_PROPERTY_ENABLED:
            {
              FEnabled = variantToBool(value);
              change(this, GC_CHANGE_LAYER_PROPERTY);
              
              break;
            };
        };
        break;
      };
    case GC_CONTAINER_SCALING:
      {
        if (index < 3)
        {
          FScaling[index] = variantToFloat(value);
          change(this, GC_CHANGE_LAYER_PROPERTY);
        };

        break;
      };
    case GC_CONTAINER_TRANSLATION:
      {
        if (index < 3)
        {
          FTranslation[index] = variantToFloat(value);
          change(this, GC_CHANGE_LAYER_PROPERTY);
        };

        break;
      };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given figure instance from the instance list if it is currently there.
 * No error is raised if the instance does not belong to this layer.
 *
 * @param instance The instance to be removed.
 */
void CLayer::removeInstance(CFigureInstance* instance)
{
  if (!updating())
  {
    beginUpdate();
    for (CFigureInstances::iterator iterator = FInstances.begin(); iterator != FInstances.end(); ++iterator)
      if (*iterator == instance)
      {
        (*iterator)->removeListener(&FListener);
        FInstances.erase(iterator);
        instance->FLayer = NULL;
        change(this, GC_CHANGE_LAYER_REMOVE_INSTANCE);
        makeDirty();
        break;
      };
    endUpdate();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks the validity of the figure display list and executes it.
 *
 * @param CurrentZoom The current zoom factor of the scene.
 * @param bounds The area currently visible. No need to render anything outside that area.
 */
void CLayer::render(float CurrentZoom, TBoundingBox bounds)
{
  if (FVisible && !updating()) 
  {
    if (FDirty)
      validate();

    CurrentZoom *= FMinimalScale;
    glPushMatrix();
    applyTransformations();

    // Transform viewport coordinates into view space. They are used for occlusion culling.
    TBoundingBox visibleBounds;
    visibleBounds.upper.x = (bounds.upper.x - FTranslation[0]) / FScaling[0];
    visibleBounds.upper.y = (bounds.upper.y - FTranslation[1]) / FScaling[1];
    visibleBounds.upper.z = (bounds.upper.z - FTranslation[2]) / FScaling[2];
    visibleBounds.lower.x = (bounds.lower.x - FTranslation[0]) / FScaling[0];
    visibleBounds.lower.y = (bounds.lower.y - FTranslation[1]) / FScaling[1];
    visibleBounds.lower.z = (bounds.lower.z - FTranslation[2]) / FScaling[2];

    for (CFigureInstances::iterator iterator = FInstances.begin(); iterator != FInstances.end(); ++iterator)
    {
      CFigureInstance* instance = *iterator;
      if (instance->overlaps(visibleBounds))
        instance->render(CurrentZoom);
    };

    renderLayerContent();
    glPopMatrix();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scales the layer by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of scale uses single float values as parameters.
 *
 * @param Sx scale factor for the x axis.
 * @param Sy scale factor for the y axis.
 * @param Sz scale factor for the z axis.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CLayer::scale(float Sx, float Sy, float Sz, bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Sx;
    FScaling[1] += Sy;
    FScaling[2] += Sz;
  }
  else
  {
    FScaling[0] = Sx;
    FScaling[1] = Sy;
    FScaling[2] = Sz;
  };
  change(this, GC_CHANGE_LAYER_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Scales the layer by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of scale uses an array of values in the parameter list.
 *
 * @param Factor An array of 3 scale values, one for each axis.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CLayer::scale(const float Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Factor[0];
    FScaling[1] += Factor[1];
    FScaling[2] += Factor[2];
  }
  else
  {
    FScaling[0] = Factor[0];
    FScaling[1] = Factor[1];
    FScaling[2] = Factor[2];
  };
  change(this, GC_CHANGE_LAYER_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the layer by the amount given in Tx, Ty and Tz. If Accumulative is true then the new translation factors 
 * are multiplied with the existing values. This version of translate uses an array for the values in the parameter list.
 *
 * @param Tx scale factor for the x axis.
 * @param Ty scale factor for the y axis.
 * @param Tz scale factor for the z axis.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CLayer::translate(float Tx, float Ty, float Tz, bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Tx;
    FTranslation[1] += Ty;
    FTranslation[2] += Tz;
  }
  else
  {
    FTranslation[0] = Tx;
    FTranslation[1] = Ty;
    FTranslation[2] = Tz;
  };
  change(this, GC_CHANGE_LAYER_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the layer by the amount given in Factor. If Accumulative is true then the new translation factors are multiplied
 * with the existing values. This version of translate uses an array for the values in the parameter list.
 *
 * @param Factor An array of translation values, for each axis one.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CLayer::translateV(const float Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Factor[0];
    FTranslation[1] += Factor[1];
    FTranslation[2] += Factor[2];
  }
  else
  {
    FTranslation[0] = Factor[0];
    FTranslation[1] = Factor[1];
    FTranslation[2] = Factor[2];
  };
  change(this, GC_CHANGE_LAYER_PROPERTY);
  makeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the layer's visibility state.
 *
 * @param IsVisible Set it to true if you want the layer to be visible.
 */
void CLayer::visible(bool IsVisible)
{
  if (FVisible != IsVisible)
  {
    FVisible = IsVisible;
    change(this, GC_CHANGE_LAYER_VISIBILITY);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the visibility state.
 *
 * @return The current visibility state.
 */
bool CLayer::visible(void)
{
  return FVisible;
}

//----------------- CGridLayer -----------------------------------------------------------------------------------------

CGridLayer::CGridLayer(CGCView* View, CGenericCanvas* canvas): CLayer("grid", canvas)
{
  FView = View;
  visible(false);
}

//----------------------------------------------------------------------------------------------------------------------

void CGridLayer::renderLayerContent(void)
{
  if (FView != NULL)
  {
    TGCViewport viewport = FView->viewport();
    float zoomX = FView->zoomX();
    float zoomY = FView->zoomY();

    double left = viewport.left * zoomX;
    double Right = (viewport.left + viewport.width) * zoomX;
    double top = viewport.top;
    double Bottom = (viewport.top + viewport.height) * zoomY;

    // Disable antialiasing temporarily.
    glPushAttrib(GL_ENABLE_BIT);
    glDisable(GL_LINE_SMOOTH);
    glColor4d(0, 0, 0, 0.2);

    glBegin(GL_LINES);
    for (double y = top; y < Bottom; y += 30)
    {
      glVertex2d(left, y);
      glVertex2d(Right, y);
    };

    for (double x = left; x < Right; x += 30)
    {
      glVertex2d(x, top);
      glVertex2d(x, Bottom);
    };

    glEnd();
    glPopAttrib();
  };
}

/*
void CGLView::DrawNet2(double xsize, double zsize, double xstep, double
zstep)
{

 //adjust xsize , zsize
 if (xstep<0.0000001 || zstep<0.0000001) return; file://avoid too small grid
 int countx= (int)xsize/xstep;
 int countz= (int)zsize/zstep;

 //okay
 xsize= (double)countx * xstep;
 zsize= (double)countz * zstep;



 glPushMatrix();

 double x,z;

   glEnable (GL_LINE_STIPPLE);
   glLineStipple (1, 0x1c47);  //  dotted
   glLineWidth(1); //thin line

   minor line
 glBegin(GL_LINES);
 for (x=-xsize; x<xsize+xstep; x+=xstep) {
  glVertex3d(x, 0, -zsize);
  glVertex3d(x, 0,  zsize);

 }

 for (z=-zsize; z<zsize+zstep; z+=zstep) {
  glVertex3d(-xsize, 0, z);
  glVertex3d( xsize, 0, z);
 }

 glEnd();

 main line
 glDisable(GL_LINE_STIPPLE);
 int c=0;
 glBegin(GL_LINES);
 c=0;
 for (x=0; x<=xsize; x+=xstep) {
  if ( c++%5 == 0 ) {
  glVertex3d(x, 0, -zsize);
  glVertex3d(x, 0,  zsize);
  }
 }

 c=0;
 for (x=0; x>=-xsize; x-=xstep) {
  if ( c++%5 == 0 ) {
  glVertex3d(x, 0, -zsize);
  glVertex3d(x, 0,  zsize);
  }
 }

 c=0;
 for (z=0; z<=zsize; z+=zstep) {
  if (c++%5==0) {
   glVertex3d(-xsize, 0, z);
   glVertex3d( xsize, 0, z);
  }
 }

 c=0;
 for (z=0; z>=-zsize; z-=zstep) {
  if (c++%5==0) {
   glVertex3d(-xsize, 0, z);
   glVertex3d( xsize, 0, z);
  }
 }

 glEnd();



 major line
 glLineWidth(2);
 glBegin(GL_LINES);
 glVertex3d(0, 0, -zsize);
 glVertex3d(0, 0,  zsize);
 glVertex3d(-xsize, 0, 0);
 glVertex3d( xsize, 0, 0);
 glLineWidth(1);

 glEnd();


 glPopMatrix();

}
*/
//----------------- CFeedbackLayer ------------------------------------------------------------------------------------

CFeedbackLayer::CFeedbackLayer(string Name, CGenericCanvas* canvas): CLayer(Name, canvas)
{
  FHandleSize = 5;
  FRubberband = 0;
  FStates = 0;
  FSelectionDecoration = 0;
}

//----------------------------------------------------------------------------------------------------------------------

CFeedbackLayer::~CFeedbackLayer(void)
{
  clearSelection();
  if (FSelectionDecoration != 0)
    glDeleteLists(FSelectionDecoration, 1);
  if (FRubberband != 0)
    glDeleteLists(FRubberband, 1);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates the display list for the selection decoration, which is shared among all selection entries.
 */
void CFeedbackLayer::createSelectionDecoration(void)
{
  GLubyte Color[4];
  if (!colorByName("Highlight", Color))
  {
    // If the system's highlight color could not be found then use a default one.
    Color[0] = 64;
    Color[1] = 64;
    Color[2] = 128;
  };

  FSelectionDecoration = glGenLists(1);
  glNewList(FSelectionDecoration, GL_COMPILE);

  // The interior.
  Color[3] = 30;
  glColor4ubv(Color);
  glRectd(0, 1, 1, 0);

  glPushAttrib(GL_LINE_BIT);
  glDisable(GL_LINE_SMOOTH);

  // The border.
  Color[3] = 200;
  glColor4ubv(Color);
  glBegin(GL_LINE_LOOP);
    glVertex2d(0, 1);
    glVertex2d(1, 1); 
    glVertex2d(1, 0);
    glVertex2d(0, 0); 
  glEnd();

  // The handles.
  Color[3] = 100;
  glColor4ubv(Color);
  glPointSize(FHandleSize);
  glBegin(GL_POINTS);
    glVertex2d(0, 1);
    glVertex2d(0.5, 1);
    glVertex2d(1, 1);

    glVertex2d(0, 0.5);
    glVertex2d(1, 0.5);

    glVertex2d(0, 0);
    glVertex2d(0.5, 0);
    glVertex2d(1, 0);
  glEnd();

  glPopAttrib();

  glEndList();

  checkError();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Helper method to add a figure instance to the selection list. No change event is triggered.
 *
 * @param instance The instance to add.
 * @return If the instance was added (because it wasn't already there) then true is returned, otherwise false.
 */
bool CFeedbackLayer::internalAddToSelection(CFigureInstance* instance)
{
  bool Result = !instance->FSelected;
  if (Result)
  {
    instance->FSelected = true;
    TSelectionEntry* entry = new TSelectionEntry;
    entry->instance = instance;
    entry->dirty = true;
    FSelection[instance] = entry;
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Helper method to remove a figure instance from the selection list. No change event is triggered.
 *
 * @param instance The instance to add.
 */
void CFeedbackLayer::internalRemoveFromSelection(CFigureInstance* instance)
{
  if (instance->FSelected)
  {
    instance->FSelected = false;
    CSelectionIterator iterator = FSelection.find(instance);
    if (iterator != FSelection.end())
    {
      delete iterator->second;
      FSelection.erase(iterator);
    };
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Renders the decorations for all figure instances that are currently selected.
 */
void CFeedbackLayer::renderLayerContent(void)
{
  for (CSelectionIterator iterator = FSelection.begin(); iterator != FSelection.end(); iterator++)
  {
    TSelectionEntry* Entry = iterator->second;
    glPushMatrix();
    glTranslated(Entry->bounds.upper.x, Entry->bounds.upper.y, 1);
    glScaled(Entry->bounds.lower.x - Entry->bounds.upper.x, Entry->bounds.lower.y - Entry->bounds.upper.y, 1);
    glCallList(FSelectionDecoration);
    glPopMatrix();
  };
  
  if ((FStates & GC_FBSTATE_RUBBERBAND) != 0)
  {
    // In order to avoid backface culling if the coordinates of the rubber band do not form a counter-clock-wise
    // face we simply disable face culling for the moment.
    glPushAttrib(GL_ENABLE_BIT);
    glDisable(GL_CULL_FACE);
    glTranslated(FRubberbandBounds.upper.x, FRubberbandBounds.upper.y, 0);
    glScaled(FRubberbandBounds.lower.x - FRubberbandBounds.upper.x, FRubberbandBounds.lower.y - FRubberbandBounds.upper.y, 1);
    glCallList(FRubberband);
    glPopAttrib();

  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates display lists for all invalid decorations.
 */
void CFeedbackLayer::validateLayerContent(void)
{
  if (FSelectionDecoration == 0)
    createSelectionDecoration();

  for (CSelectionIterator iterator = FSelection.begin(); iterator != FSelection.end(); iterator++)
    if (iterator->second->dirty)
    {
      iterator->second->dirty = false;

      // Update the bounding box of the figure instance.
      TBoundingBox bounds = iterator->second->instance->bounds();

      // Give a small border around the figure instance.
      bounds.upper.x -= 1;
      bounds.upper.y -= 1;
      bounds.lower.x += 1;
      bounds.lower.y += 1;
      iterator->second->bounds = bounds;
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds the given figure instance to the current selection.
 *
 * @param instance The instance to be added to the selection. If it is already in the set it won't be added again.
 */
void CFeedbackLayer::addToSelection(CFigureInstance* instance)
{
  if (internalAddToSelection(instance))
  {
    change(instance, GC_CHANGE_SELECTION_ADD);
    if (!updating())
      makeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all figure instances from the selection set, making it empty.
 */
void CFeedbackLayer::clearSelection(void)
{
  if (FSelection.size() > 0)
  {
    for (CSelectionIterator iterator = FSelection.begin(); iterator != FSelection.end(); iterator++)
    {
      iterator->second->instance->FSelected = false;
      delete iterator->second;
    };

    FSelection.clear();
    change(NULL, GC_CHANGE_SELECTION_CLEAR);
    if (!updating())
      makeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines whether the given position corresponds to any of the parts (body, handles) of a selection decoration.
 * This test is quite fast and can be used for cursor feedback and such.
 * The method needs correctly set modelview and projection matrices on enter in order to convert mouse coordinates
 * correctly.
 *
 * @param x The horizontal mouse coordinate in layer coordinates.
 * @param y The vertical mouse coordinate in layer coordinates.
 * @param Zoom The current zoom factor. Hit distances vary with zoom.
 * @return One of the direction flags.
 */
TGCDirection CFeedbackLayer::getSelectionInfo(float x, float y, float Zoom)
{
  TGCDirection Result = GC_SI_NONE;

  applyTransformations();

  GLint viewport[4];
  glGetIntegerv(GL_VIEWPORT, viewport);

  GLdouble ModelviewMatrix[16];
  glGetDoublev(GL_MODELVIEW_MATRIX, ModelviewMatrix);

  GLdouble ProjectionMatrix[16];
  glGetDoublev(GL_PROJECTION_MATRIX, ProjectionMatrix);

  // Convert window coordinates into object (scene) coordinates.
  double LocalX, LocalY, LocalZ;
  gluUnProject(x, viewport[3] - y, 0, ModelviewMatrix, ProjectionMatrix, viewport, &LocalX, &LocalY, &LocalZ);

  // Now try to find a decoration that is located at the given position.
  // We examine only those that are not dirty. dirty decorations cannot be in view.
  // Iteration happens backwards to find decorations on top earlier than others.
  float Distance = 8 / Zoom;
  for (CSelectionIteratorReverse iterator = FSelection.rbegin(); iterator != FSelection.rend(); ++iterator)
  {
    if (!iterator->second->dirty)
    {
      TBoundingBox bounds = iterator->second->bounds;
      if ((LocalX >= (bounds.upper.x - Distance)) && (LocalX <= (bounds.lower.x + Distance)) &&
        (LocalY > (bounds.upper.y - Distance)) && (LocalY <= (bounds.lower.y + Distance)))
      {                                                
        // Found a decoration. Check if a handle is hit.
        Result = GC_SI_ON_OBJECT;

        bool LeftColumn = ::fabs(bounds.upper.x - LocalX) <= Distance;
        bool MiddleColumn = ::fabs((bounds.upper.x + bounds.lower.x) / 2 - LocalX) <= Distance;
        bool RightColumn = ::fabs(bounds.lower.x - LocalX) <= Distance;
        bool TopRow = ::fabs(bounds.upper.y - LocalY) <= Distance;
        bool MiddleRow = ::fabs((bounds.upper.y + bounds.lower.y) / 2 - LocalY) <= Distance;
        bool BottomRow = ::fabs(bounds.lower.y - LocalY) <= Distance;

        if (LeftColumn)
        {
          if (TopRow)
            Result = GC_SI_NORTH_WEST;
          else
            if (MiddleRow)
              Result = GC_SI_WEST;
            else
              if (BottomRow)
                Result = GC_SI_SOUTH_WEST;
        }
        else
          if (MiddleColumn)
          {
            if (TopRow)
              Result = GC_SI_NORTH;
            else
              if (BottomRow)
                Result = GC_SI_SOUTH;
          }
          else
            if (RightColumn)
            {
              if (TopRow)
                Result = GC_SI_NORTH_EAST;
              else
                if (MiddleRow)
                  Result = GC_SI_EAST;
                else
                  if (BottomRow)
                    Result = GC_SI_SOUTH_EAST;
            };

        break;
      };
    };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Invalidates the selection decoration of the given instance (or all instances if instance is NULL) so they are recomputed
 * next time the selection layer draws them.
 *
 * @param instance The figure instance whose bounds need recomputation. If this parameter is NULL then all bounds are invalidated.
 */
void CFeedbackLayer::invalidateBounds(CFigureInstance* instance)
{
  if (instance == NULL)
  {
    for (CSelectionIterator iterator = FSelection.begin(); iterator != FSelection.end(); ++iterator)
      iterator->second->dirty = true;
  }
  else
    if (instance->FSelected)
    {
      CSelectionIterator iterator = FSelection.find(instance);
      if (iterator != FSelection.end())
        iterator->second->dirty = true;
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given figure instance from the current selection.
 *
 * @param instance The instance to be removed. If it isn't actually selected then nothing happens.
 */
void CFeedbackLayer::removeFromSelection(CFigureInstance* instance)
{
  if (instance->FSelected)
  {
    internalRemoveFromSelection(instance);
    change(instance, GC_CHANGE_SELECTION_REMOVE);
    if (!updating())
      makeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

void CFeedbackLayer::resizeFiguresStart(int x, int y, TGCDirection Direction)
{
}

//----------------------------------------------------------------------------------------------------------------------

void CFeedbackLayer::resizeFiguresStop(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

void CFeedbackLayer::resizeFiguresTo(int x, int y)
{
}

//----------------------------------------------------------------------------------------------------------------------

/**
  * When in rubber band mode then this function extends the current rubber band rectangle from the start point to
  * the given coordinates and handles selection/deselection of figure instances.
  *
  * @param x The x coordinate of the new corner.
  * @param y The y coordinate of the new corner.
  * @param Action Determines if and how figure instance selection is to be handled. See TRBSelectionAction for
  *               a description of the various modes.
 */
void CFeedbackLayer::rubberbandResize(int x, int y, TRBSelectionAction Action)
{
  if ((FStates & GC_FBSTATE_RUBBERBAND) != 0)
  {
    // Convert window coordinates into object (scene) coordinates.
    applyTransformations();

    GLint viewport[4];
    glGetIntegerv(GL_VIEWPORT, viewport);

    GLdouble ModelviewMatrix[16];
    glGetDoublev(GL_MODELVIEW_MATRIX, ModelviewMatrix);

    GLdouble ProjectionMatrix[16];
    glGetDoublev(GL_PROJECTION_MATRIX, ProjectionMatrix);

    double LocalX, LocalY, LocalZ;
    gluUnProject(x, viewport[3] - y, 0, ModelviewMatrix, ProjectionMatrix, viewport, &LocalX, &LocalY, &LocalZ);

    TBoundingBox oldBounds = sortBounds(FRubberbandBounds);
    FRubberbandBounds.lower.x = (float) LocalX;
    FRubberbandBounds.lower.y = (float) LocalY;
    TBoundingBox NewBounds = sortBounds(FRubberbandBounds);

    bool haveSelectionRemoved = false;
    bool haveSelectionAdded = false;

    if (Action != GC_RBACTION_NONE)
    {
      CFigureInstanceEnumerator* enumerator = canvas()->getFigureInstanceEnumerator();
      enumerator->reset();
      switch (Action)
      {
        case GC_RBACTION_SELECT:
          {
            // Select all figure instances if they intersect. Leave all others alone.
            while (enumerator->hasNext())
            {
              CFigureInstance* instance = enumerator->next();
              TBoundingBox InstanceBounds = instance->bounds();
              if (!boundsAreEmpty(InstanceBounds))
              {
                if (boundsIntersect(NewBounds, InstanceBounds))
                {
                  internalAddToSelection(instance);
                  haveSelectionAdded = true;
                };
              };
            };
            if (haveSelectionAdded)
              canvas()->change(NULL, GC_CHANGE_SELECTION_ADD);

            break;
          };
        case GC_RBACTION_SELECT_REMOVE:
          {
            // Select figure instances, which intersect, unselect all others.
            while (enumerator->hasNext())
            {
              CFigureInstance* instance = enumerator->next();
              TBoundingBox InstanceBounds = instance->bounds();
              if (!boundsAreEmpty(InstanceBounds))
              {
                if (instance->FSelected)
                {
                  // instance is selected. See if it is still within the rubberband bounds.
                  // Remove it from the selection list if not.
                  if (!boundsIntersect(NewBounds, InstanceBounds))
                  {
                    internalRemoveFromSelection(instance);
                    haveSelectionRemoved = true;
                  };
                }
                else
                {
                  // instance is not selected. Add it to the current selection if its bounds
                  // fall within the rubberband bounds.
                  if (boundsIntersect(NewBounds, InstanceBounds))
                  {
                    internalAddToSelection(instance);
                    haveSelectionAdded = true;
                  };
                };
              };
            };

            if (haveSelectionRemoved)
              canvas()->change(NULL, GC_CHANGE_SELECTION_REMOVE);

            if (haveSelectionAdded)
              canvas()->change(NULL, GC_CHANGE_SELECTION_ADD);

            break;
          };
        case GC_RBACTION_TOGGLE:
          {
            // Switch figure instance selection state if their bounds intersect either
            // the new rubber band bounds or the old, but not both.
            while (enumerator->hasNext())
            {
              CFigureInstance* instance = enumerator->next();
              TBoundingBox InstanceBounds = instance->bounds();

              if (!boundsAreEmpty(InstanceBounds))
              {
                if (boundsIntersect(NewBounds, InstanceBounds) != boundsIntersect(oldBounds, InstanceBounds))
                {
                  if (instance->FSelected)
                  {
                    internalRemoveFromSelection(instance);
                    haveSelectionRemoved = true;
                  }
                  else
                  {
                    internalAddToSelection(instance);
                    haveSelectionAdded = true;
                  };
                };
              };
            };

            if (haveSelectionRemoved)
              canvas()->change(NULL, GC_CHANGE_SELECTION_REMOVE);

            if (haveSelectionAdded)
              canvas()->change(NULL, GC_CHANGE_SELECTION_ADD);

            break;
          };
      };
      delete enumerator;
    };
    
    if (haveSelectionAdded || haveSelectionRemoved)
    {
      makeDirty();
      change(this, GC_CHANGE_SELECTION_CHANGE);
    }
    else
      change(this, GC_CHANGE_CANVAS_REFRESH);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Starts rubber banding if none is active currently. Otherwise it does nothing.
 *
 * @param Style Determines the visible style of the rubber band.
 * @param x The x coordinate of the start point in window coordinates.
 * @param x The y coordinate of the start point in window coordinates.
 * @param RemoveSelection If true then the current selection will be cleared.
 */
void CFeedbackLayer::rubberbandStart(TRubberbandStyle Style, int x, int y, bool RemoveSelection)
{
  if ((FStates & GC_FBSTATE_RUBBERBAND) == 0)
  {
    FStates |= GC_FBSTATE_RUBBERBAND;

    if (RemoveSelection)
      clearSelection();

    // Convert window coordinates into object (scene) coordinates.
    applyTransformations();

    GLint viewport[4];
    glGetIntegerv(GL_VIEWPORT, viewport);

    GLdouble ModelviewMatrix[16];
    glGetDoublev(GL_MODELVIEW_MATRIX, ModelviewMatrix);

    GLdouble ProjectionMatrix[16];
    glGetDoublev(GL_PROJECTION_MATRIX, ProjectionMatrix);

    double LocalX, LocalY, LocalZ;
    gluUnProject(x, viewport[3] - y, 0, ModelviewMatrix, ProjectionMatrix, viewport, &LocalX, &LocalY, &LocalZ);

    FRubberbandBounds.upper.x = (float) LocalX;
    FRubberbandBounds.upper.y = (float) LocalY;
    FRubberbandBounds.lower.x = (float) LocalX;
    FRubberbandBounds.lower.y = (float) LocalY;

    if (FRubberband == 0)
      FRubberband = glGenLists(1);
    glNewList(FRubberband, GL_COMPILE);
    glPushAttrib(GL_LINE_BIT);
    glDisable(GL_LINE_SMOOTH);

    switch (Style)
    {
      case GC_RBSTYLE_SOLID_THIN:   // A simple black rectangle with a one pixel wide border.
        {
          glColor3f(0, 0, 0);
          glLineWidth(1);
          glBegin(GL_LINE_LOOP);
            glVertex3d(0, 0, 1);
            glVertex3d(1, 0, 1);
            glVertex3d(1, 1, 1);
            glVertex3d(0, 1, 1);
          glEnd();
          break;
        };
      case GC_RBSTYLE_SOLID_THICK:  // A simple black rectangle with a 3 pixel wide border.
        {
          glColor3f(0, 0, 0);
          glLineWidth(3);
          glBegin(GL_LINE_LOOP);
            glVertex3d(0, 0, 1);
            glVertex3d(1, 0, 1);
            glVertex3d(1, 1, 1);
            glVertex3d(0, 1, 1);
          glEnd();
          break;
        };
      case GC_RBSTYLE_DOTTED_THIN:  // A simple black rectangle with a one pixel wide dotted border.
        {
          glColor3f(0, 0, 0);
          glLineWidth(1);
          glEnable(GL_LINE_STIPPLE);
          glLineStipple(1, 0xFF);
          glBegin(GL_LINE_LOOP);
            glVertex3d(0, 0, 1);
            glVertex3d(1, 0, 1);
            glVertex3d(1, 1, 1);
            glVertex3d(0, 1, 1);
          glEnd();
          break;
        };
      case GC_RBSTYLE_DOTTED_THICK: // A simple black rectangle with a 3 pixel wide dotted border.
        {
          glColor3f(0, 0, 0);
          glLineWidth(3);
          glEnable(GL_LINE_STIPPLE);
          glLineStipple(1, 0xFF);
          glBegin(GL_LINE_LOOP);
            glVertex3d(0, 0, 1);
            glVertex3d(1, 0, 1);
            glVertex3d(1, 1, 1);
            glVertex3d(0, 1, 1);
          glEnd();
          break;
        };
      case GC_RBSTYLE_BLENDED_CHECKERBOARD: // A filled rectangle with a one pixel border and a translucent interior.
        {
          GLubyte Color[4];
          if (!colorByName("Highlight", Color))
          {
            // If the system's highlight color could not be found then use a default one.
            Color[0] = 64;
            Color[1] = 64;
            Color[2] = 128;
          };

          glLineWidth(1);
          glEnable(GL_LINE_SMOOTH);
          for (int Row = 0; Row < 10; Row++)
          {
            for (int Cell = 0; Cell < 10; Cell++)
            {
              if (((Row + Cell) & 1) == 0)
                Color[3] = 60;
              else
                Color[3] = 70;
              glColor4ubv(Color);
              glBegin(GL_POLYGON);
                glVertex3d(Cell * 0.1, Row * 0.1, 1);
                glVertex3d(Cell * 0.1, (Row + 1) * 0.1, 1);
                glVertex3d((Cell + 1) * 0.1, (Row + 1) * 0.1, 1);
                glVertex3d((Cell + 1) * 0.1, Row * 0.1, 1);
              glEnd();
            };
          };

          Color[3] = 200;
          glColor4ubv(Color);
          glBegin(GL_LINE_LOOP);
            glVertex3d(0, 0, 1);
            glVertex3d(1, 0, 1);
            glVertex3d(1, 1, 1);
            glVertex3d(0, 1, 1);
          glEnd();
          break;
        };
      case GC_RBSTYLE_BLENDED_DIAGONALS: // A filled rectangle with a one pixel border and a translucent interior.
        {
          GLubyte Color[4];
          if (!colorByName("Highlight", Color))
          {
            // If the system's highlight color could not be found then use a default one.
            Color[0] = 64;
            Color[1] = 64;
            Color[2] = 128;
          };

          glLineWidth(1);

          // Interior.
          int Steps = 50;
          double X1 = 0;
          double Y1 = 0;
          double X2 = 0;
          double Y2 = 0;
          double dStep = 1.0 / Steps;

          // left lower half.
          for (int x = 0; x < Steps; x++)
          {
            if (x % 2 == 0)
              Color[3] = 60;
            else
              Color[3] = 100;

            glColor4ubv(Color);
            glBegin(GL_POLYGON);
              glVertex3d(x * dStep, 0, 1);
              glVertex3d(0, x * dStep, 1);
              glVertex3d(0, (x + 1) * dStep, 1);
              glVertex3d((x + 1) * dStep, 0, 1);
            glEnd();
          };

          // Right upper half.
          for (int y = 0; y < Steps; y++)
          {
            if (y % 2 == 0)
              Color[3] = 60;
            else
              Color[3] = 100;

            glColor4ubv(Color);
            glBegin(GL_POLYGON);
              glVertex3d(1, y * dStep, 1);
              glVertex3d(y * dStep, 1, 1);
              glVertex3d((y + 1) * dStep, 1, 1);
              glVertex3d(1, (y + 1) * dStep, 1);
            glEnd();
          };

          // Border.
          Color[3] = 200;
          glColor4ubv(Color);
          glBegin(GL_LINE_LOOP);
            glVertex3d(0, 0, 1);
            glVertex3d(1, 0, 1);
            glVertex3d(1, 1, 1);
            glVertex3d(0, 1, 1);
          glEnd();
          break;
        };
    };
    glPopAttrib();
    glEndList();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Stops rubber banding if it is active currently. Does nothing if not.
 */
void CFeedbackLayer::rubberbandStop(void)
{
  if ((FStates & GC_FBSTATE_RUBBERBAND) != 0)
  {
    FStates &= ~GC_FBSTATE_RUBBERBAND;
    makeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------
