/* Copyright (C) 2005 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#include "MGRT.h"
#include "myx_grt_builtin_module_public_interface.h"
#include <gtkmm/main.h>
/**
 * @file  MGRT.cc
 * @brief
 */



class BlockingRequest : public MGRT::Request {
    Glib::ustring _module, _function;
    MGRTValue _arguments;
    Glib::Dispatcher _finished;
  public:
    BlockingRequest(const Glib::ustring &module,
                    const Glib::ustring &function,
                    const MGRTValue &arguments)
      : _module(module), _function(function), _arguments(arguments)
    {
      _finished.connect(sigc::mem_fun(*this, &BlockingRequest::finished));
    }
    
    void finished()
    {
      Gtk::Main::instance()->quit();
    }
    
    virtual void finish()
    {
      _finished.emit();
    }
    
    virtual void wait()
    {
      Gtk::Main::instance()->run();
    }

    virtual void execute(MGRT *grt)
    {
      _result= MGRTValue(myx_grt_function_get_and_call(grt->grt(),
                                                       _module.c_str(),
                                                       _function.c_str(),
                                                       0,
                                                       _arguments.grtValue(),
                                                       &_error));
      finish();
    }
};


static Glib::ustring makeErrorMessageFromResult(const MGRTValue &result)
{
  if (result.isValid())
  {
    const char *error= result["error"].asString();
    const char *detail= result["detail"].asString();
    if (detail)
      return ufmt("%s (%s)", error, detail);
    else
      return error;
  }
  return "";
}


MGRT::MGRT()
{
  _grt= myx_grt_initialize(0);
}
  
  
void MGRT::initialize_grt_thread(const Glib::ustring &resourcePath)
{ 
  _thread_ready= false;
  
  Glib::Thread::create(sigc::bind<std::string>(sigc::mem_fun(*this, &MGRT::request_thread), resourcePath),
                       false);
  
  // busy wait until thread ready
  while (!_thread_ready); 
}


void MGRT::init_thread(const std::string &resourcePath)
{
  MYX_GRT_MODULE_LOADER *loader;
  MYX_GRT_ERROR error;
  
  myx_grt_set_output_callback(_grt, this, process_grt_output);
  myx_grt_set_message_callback(_grt, this, process_grt_message);
  
  myx_grt_shell_print_welcome(_grt);
  
  scan_structs_in_path(resourcePath+"/grt");
  
    // loaders
  myx_register_builtin_grt_module_base(_grt);
  myx_register_builtin_grt_module_reverse_engineer_mysql(_grt);
  myx_register_builtin_grt_module_transformation_mysql(_grt);

#ifdef ENABLE_JAVA_MODULES
  // java modules
  out_text("Loading Java modules...");
  loader= myx_java_init_loader(_grt, NULL, &error, NULL,
                               std::string(resourcePath+"/").c_str());
  if (loader)
  {
    if (myx_grt_register_module_loader(_grt, loader) < 0)
      out_text("Error registering Java module loader\n");
    else
    {
      int i= myx_grt_scan_for_modules(_grt, std::string(resourcePath+"/java/com/mysql/grt/modules").c_str(), &error);
      if (error != MYX_GRT_NO_ERROR)
        out_text(ufmt("Error scanning Java modules (%i)\n", error));
      else
        out_text(ufmt("Registered %i Java modules\n", i));
    }
  }
  else
    out_text(ufmt("Error initializing Java module loader (%i)\n",error));
#endif

#ifdef ENABLE_PHP_MODULES
  // php modules
  out_text("Loading PHP modules...");
  loader= myx_php_init_loader(_grt, &error);
  if (loader)
  {
    if (myx_grt_register_module_loader(_grt, loader) < 0)
      out_text("Error registering PHP module loader\n");
    else
    {
      i= myx_grt_scan_for_modules(_grt, 
                                  std::string(resourcePath+"/php/modules").c_str());
      if (error != MYX_GRT_NO_ERROR)
        out_text(ufmt("Error scanning PHP modules (%i)\n", error));
      else
        out_text("Registered %i PHP modules\n", i);
    }
  }
  else
    out_text("Error initializing PHP module loader (%i)\n",error);
#endif

  // lua modules
  out_text("Loading Lua modules...");
  loader= myx_lua_init_loader(_grt, &error, std::string(resourcePath+"/lua").c_str());
  if (loader)
  {
    if (myx_grt_register_module_loader(_grt, loader) < 0)
      out_text("Error registering Lua module loader\n");
    else
    {
      int i= myx_grt_scan_for_modules(_grt, std::string(resourcePath+"/lua").c_str(), &error);
      if (error != MYX_GRT_NO_ERROR)
        out_text(ufmt("Error scanning Lua modules (%i)\n", error));
      else
        out_text(ufmt("Registered %i Lua modules\n", i));
    }
  }
  else
    out_text(ufmt("Error initializing Lua module loader (%i)\n",error));

  _thread_ready= true;
  
  myx_grt_init_lua_shell(_grt);
}




MGRT::~MGRT()
{
  myx_grt_finalize(_grt);
}
    

void MGRT::set_console(MGRTShellView *shell)
{
  _console= shell;
  
  //_console->make_ready();
}


void MGRT::process_grt_output(const char *text, void *data)
{
  MGRT *me= (MGRT*)data;

  me->out_text(text);
}


void MGRT::process_grt_message(MYX_GRT_MSGS *msgs, void *data)
{
  MGRT *me= (MGRT*)data;
  
  if (msgs)
  {
    Glib::ustring type;
    for (unsigned int i= 0; i < msgs->msgs_num; i++)
    {
      switch (msgs->msgs[i].msg_type)
      {
      case 0: type= "ERROR"; break;
      case 1: type= "WARNING"; break;
      default: type= "MESSAGE"; break;
      }
      if (msgs->msgs[i].msg_detail)
      {
        Glib::ustring msg= msgs->msgs[i].msg;
        for (unsigned int j= 0; j < msgs->msgs[i].msg_detail->strings_num; j++)
          msg+= ufmt("    %s\n", msgs->msgs[i].msg_detail->strings[j]);
        
        me->out_message(msg, type);
      }
      else
        me->out_message(msgs->msgs[i].msg, type);
    }
  }
}


void MGRT::out_text(const Glib::ustring &text)
{
  if (_outputHandler.empty())
    _outputHandler(text);
//  else if (_console)

  g_message("%s", text.c_str());
}


void MGRT::out_message(const Glib::ustring &text,
                      const Glib::ustring &type)
{
  if (_messageHandler.empty())
    _messageHandler(text, type);
  
    
}


void MGRT::perform_shell_command(const Glib::ustring &command)
{
  MYX_GRT_SHELL_COMMAND error;
  
  error= myx_grt_lua_shell_execute(_grt, command.c_str());
  
  myx_grt_messages_stack_flush(_grt, 0);
}


Glib::ustring MGRT::shell_prompt()
{
  return myx_grt_lua_shell_get_prompt(_grt);
}



bool MGRT::save_app_dict()
{
  if (myx_grt_store_to_file(_grt,
                            global_value("/app").grtValue(),
                            _appDictPath.c_str()) == MYX_GRT_NO_ERROR)
    return true;
  return false;
}


bool MGRT::load_app_dict_from_path(const std::string &path)
{
  MYX_GRT_VALUE *appDict;
  
  _appDictPath= path;

  appDict= myx_grt_retrieve_from_file(_grt, _appDictPath.c_str());

  if (appDict)
  {
    set_global_value("/app", appDict);
    myx_grt_value_release(appDict);
    return true;
  }
  return false;

}


MGRTValue MGRT::global_app_dict()
{
  MGRTValue result(global_value("/app"));

  if (!result.isValid())
  {
    result= myx_grt_dict_new(_grt, "base.ApplicationData");
    set_global_value("/app", result.grtValue());
  }
  return result;
}



bool MGRT::scan_structs_in_path(const std::string &path)
{
  int i;
  MYX_GRT_ERROR error;

  i= myx_grt_scan_for_structs(_grt, path.c_str(), &error);
  if (error != MYX_GRT_NO_ERROR)
  {
    out_text(ufmt("Error loading struct definition files from %s (%i)\n",path.c_str(),error));
    return false;
  }
  else
  {
    if (i == 1)
      out_text("Registered one struct definition\n");
    else
      out_text(ufmt("Registered %i struct definitions\n",i));
    return true;
  }
}


bool MGRT::scan_modules_in_path(const std::string &path)
{
  MYX_GRT_ERROR error;
  int i;

  i= myx_grt_scan_for_modules(_grt, path.c_str(), &error);
  if (error != MYX_GRT_NO_ERROR)
  {
    out_text(ufmt("Error scanning for modules in %s (%i)\n", path.c_str(), error));
    return false;
  }
  else
  {
    out_text(ufmt("Registered %i modules\n", i));
    return true;
  }
}



void MGRT::request_thread(std::string resourcePath)
{
  init_thread(resourcePath);
  
  for (;;)
  {
    Request *req;
    
    _requestQueueLock.lock();
    while (_requestQueue.empty())
      _requestReady.wait(_requestQueueLock);

    req= *_requestQueue.begin();
    _requestQueue.erase(_requestQueue.begin());
    _requestQueueLock.unlock();

    req->execute(this);
  }
}


void MGRT::queue_request(Request *req)
{
  _requestQueueLock.lock();
  _requestQueue.push_front(req);
  _requestReady.signal();
  _requestQueueLock.unlock();
}


bool MGRT::call_procedure(const std::string &module,
                         const std::string &procedure,
                         const MGRTValue &arguments)
{
  MGRTValue result(call_function(module, procedure, arguments));
  
  return result.isValid();
}


MGRTValue MGRT::call_function(const std::string &module,
                              const std::string &function,
                              const MGRTValue &arguments)
{
  BlockingRequest *request;
  
  _lastError= MYX_GRT_NO_ERROR;
  _lastErrorMessage= "";
  
  request= new BlockingRequest(module, function, arguments);
  
  queue_request(request);
  
  request->wait();
  
  MGRTValue result(request->result());
  MYX_GRT_ERROR error= request->error();
  
  delete request;
  
  myx_grt_messages_stack_flush(_grt, 0);
  
  if ((!result.isValid() || !result_is_error(result)) && error == MYX_GRT_NO_ERROR)
  {
    return result;
  }
  else
  {
    _lastError= error;
    _lastErrorMessage= makeErrorMessageFromResult(result);
    
    if (result.isValid())
      out_text(ufmt("%s calling %s.%s: %s",
                    error == MYX_GRT_NO_ERROR ? "Error" : error_text(error).c_str(),
                    module.c_str(), function.c_str(),
                    myx_grt_dict_item_get_as_string(result, "error")?:""));
    else
      out_text(ufmt("%s calling %s.%s",
                    error == MYX_GRT_NO_ERROR ? "Error" : error_text(error).c_str(),
                    module.c_str(), function.c_str()));
    return MGRTValue();
  }  
}

    
int MGRT::call_int_function(const std::string &module,
                          const std::string &function,
                          const MGRTValue &arguments)
{
  MGRTValue result(call_function(module, function, arguments));
  
  if (result.isValid())
    return result.asInt();
  return 0;
}


Glib::ustring MGRT::call_string_function(const std::string &module,
                                       const std::string &function,
                                       const MGRTValue &arguments)
{
  MGRTValue result(call_function(module, function, arguments));
  
  if (result.isValid())
    return result.asString();
  return "";
}
    
void MGRT::set_output_handler(sigc::slot<void,Glib::ustring> handler)
{
  _outputHandler= handler;
}


void MGRT::reset_output_handler()
{
  _outputHandler.disconnect();
}



void MGRT::set_message_handler(sigc::slot<void,Glib::ustring,Glib::ustring> handler)
{
  _messageHandler= handler;
}


void MGRT::reset_message_handler()
{
  _messageHandler.disconnect();
}


    
void MGRT::report_error(MYX_GRT_ERROR error)
{
  out_text(error_text(error));
}


    
MYX_GRT_ERROR MGRT::last_error()
{
  return _lastError;
}


Glib::ustring MGRT::last_error_description()
{
  return _lastErrorMessage;
}



bool MGRT::result_is_error(const MGRTValue &value)
{
  if (value.isValid() && value["error"].isValid())
    return true;
  return false;
}


void MGRT::report_error_result(const MGRTValue &result)
{
  if (result.isValid() && result["error"].isValid())
    myx_grt_value_print(_grt, result["error"].grtValue());
  else
    myx_grt_value_print(_grt, MGRTValue("no error").grtValue());
}



Glib::ustring MGRT::error_text(MYX_GRT_ERROR error)
{
  switch (error)
  {
  case MYX_GRT_NO_ERROR: return "Success";
  case MYX_GRT_INTERNAL_ERROR: return "Internal error";
  case MYX_GRT_BAD_PATH: return "Invalid path";
  case MYX_GRT_CANT_OPEN_FILE: return "Cannot open file";
  case MYX_GRT_BAD_FUNCTION: return "Invalid function";
  case MYX_GRT_DUPLICATE_ENTRY: return "Duplicate entry";
  case MYX_GRT_BAD_VALUE: return "Bad value";
  case MYX_GRT_BAD_DATA: return "Bad data";
    
  case MYX_GRT_VALIDATION_ERROR: return "Validation error";
  case MYX_GRT_FUNCTION_CALL_ERROR: return "Function call error";
  case MYX_GRT_MODULE_INIT_ERROR: return "Module init error";
  case MYX_GRT_BAD_MODULE: return "Bad module";
  case MYX_GRT_UNKNOWN_MODULE_TYPE: return "Unknown module type";
    
  case MYX_GRT_JAVA_NOT_FOUND: return "Java Runtime not found";
  case MYX_GRT_JAVA_REGISTRY_CORRUPTED: return "Java registry corrupted";
  case MYX_GRT_JAVA_JRE_CANNOT_BE_LOADED: return "JRE cannot be loaded";
  }
  return "";
}



MGRTValue MGRT::global_value(const char *path)
{
  return MGRTValue::fromGlobal(_grt, path);
}


MGRTValue MGRT::global_ref_value(const char *path)
{
  return MGRTValue::refObject(_grt, MGRTValue::fromGlobal(_grt, path).asString());
}


void MGRT::set_global_value(const char *path, const MGRTValue &value)
{
  myx_grt_dict_item_set_by_path(myx_grt_get_root(_grt), path, value.grtValue());
}

