/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#ifndef __MYX_GRT_H__
#define __MYX_GRT_H__


#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
# define __LCC__
#define MYX_SOCKET SOCKET
#else
# define MYX_SOCKET int
#endif

#include <myx_public_interface.h>
#ifdef ENABLE_LUA_MODULES
#include <lua.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif


#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
#define MYX_PUBLIC_FUNC __declspec(dllexport)
#else
#define MYX_PUBLIC_FUNC
#endif

/*
 * PUBLIC INTERFACE definition for MYSQLLibInterfaceMapper
 */

/// [SCRIPT::LibInterfaceMapper] -public_interface "libmysqlgrt"
#define libmysqlgrt_PUBLIC_INTERFACE_VERSION 10000

/// [SCRIPT::LibInterfaceMapper] -add_to_uses "myx_public_interface"
/// [SCRIPT::LibInterfaceMapper] -add_datatypes_from "..\..\..\mysql-gui-common\library\include\myx_public_interface.h"

/*
 * Enums
 */

typedef enum
{
  MYX_GRT_NO_ERROR = 0,
  MYX_GRT_FIRST_ERROR = 10000,
  MYX_GRT_INTERNAL_ERROR = MYX_GRT_FIRST_ERROR,
  MYX_GRT_BAD_PATH,
  MYX_GRT_CANT_OPEN_FILE,
  MYX_GRT_BAD_FUNCTION,
  MYX_GRT_DUPLICATE_ENTRY,
  MYX_GRT_BAD_VALUE,
  MYX_GRT_BAD_DATA,

  MYX_GRT_VALIDATION_ERROR,
  MYX_GRT_FUNCTION_CALL_ERROR,
  MYX_GRT_MODULE_INIT_ERROR,
  MYX_GRT_BAD_MODULE,
  MYX_GRT_UNKNOWN_MODULE_TYPE
} MYX_GRT_ERROR;

typedef enum 
{
  MYX_INT_VALUE= 1,
  MYX_REAL_VALUE,
  MYX_STRING_VALUE,
  MYX_LIST_VALUE,
  MYX_DICT_VALUE
} MYX_GRT_VALUE_TYPE;

typedef enum
{
  MYX_GRT_FUNCTION_HAS_ARGUMENT = 1<<0,
  MYX_GRT_FUNCTION_RETURNS_VALUE = 1<<1,
  MYX_GRT_FUNCTION_IS_GUI = 1<<2,
  MYX_GRT_FUNCTION_IS_OPTIONAL = 1<<3
} MYX_GRT_FUNCTION_FLAGS;

/*
 * Structs
 */

typedef struct MYX_GRT_MODULE_LOADER MYX_GRT_MODULE_LOADER;
typedef struct MYX_GRT MYX_GRT;

typedef enum MYX_GRT_SHELL_COMMAND
{
  MYX_GRT_SHELL_COMMAND_UNKNOWN = -1,
  MYX_GRT_SHELL_COMMAND_EXIT = 0,
  MYX_GRT_SHELL_COMMAND_ALL,
  MYX_GRT_SHELL_COMMAND_ERROR,
  MYX_GRT_SHELL_COMMAND_LUA,
  MYX_GRT_SHELL_COMMAND_HELP,
  MYX_GRT_SHELL_COMMAND_LS,
  MYX_GRT_SHELL_COMMAND_CD,
  MYX_GRT_SHELL_COMMAND_RUN
} MYX_GRT_SHELL_COMMAND;

  
  
typedef struct MYX_GRT_STRUCT_MEMBER
{
  char *name;

  MYX_GRT_VALUE_TYPE value_type;
  char *value_struct_name; /* only if value_type == dict or list or string*/
  MYX_GRT_VALUE_TYPE value_content_type; /* only if value_type == list */
  unsigned int is_ref; /* only if value_type == string or list*/
} MYX_GRT_STRUCT_MEMBER;


typedef struct MYX_GRT_STRUCT
{
  char *name;
  char *parent_struct_name;

  unsigned int members_num;
  MYX_GRT_STRUCT_MEMBER *members; // list of members
} MYX_GRT_STRUCT;

typedef struct MYX_GRT_STRUCTS
{
  unsigned int structs_num;
  MYX_GRT_STRUCT *structs;
} MYX_GRT_STRUCTS;

typedef struct 
{
  MYX_GRT_VALUE_TYPE content_type;
  char *content_struct_name; /* only if content_type == dict */

  unsigned int items_num;
  struct MYX_GRT_VALUE **items;
} MYX_GRT_LIST;

typedef struct
{
  char *key;
  struct MYX_GRT_VALUE *value;
} MYX_GRT_DICT_ITEM;

typedef struct
{
  char *struct_name;

  unsigned int items_num;
  MYX_GRT_DICT_ITEM *items; // this must be in strcmp() order of name
} MYX_GRT_DICT;


typedef struct MYX_GRT_VALUE 
{
  MYX_GRT_VALUE_TYPE type;
  unsigned int refcount;
  union {
    int i;
    double r;
    char *s;
    MYX_GRT_LIST *l;
    MYX_GRT_DICT *d;
  } value;
} MYX_GRT_VALUE;
  

/// [SCRIPT::LibInterfaceMapper] -generate_classes MYX_GRT_MSG
typedef struct MYX_GRT_MSG {
  int msg_type;
  char *msg;
  MYX_STRINGLIST *msg_detail;
} MYX_GRT_MSG;

/// [SCRIPT::LibInterfaceMapper] -generate_classes MYX_GRT_MSGS
typedef struct MYX_GRT_MSGS {
  unsigned int msgs_num;
  MYX_GRT_MSG *msgs;
} MYX_GRT_MSGS;


/// [SCRIPT::LibInterfaceMapper] -generate_classes MYX_GRT_FUNCTION
typedef struct MYX_GRT_FUNCTION
{
  struct MYX_GRT_MODULE *module;
  
  char *name; // the function name must match the interface name it implements

  char *param_struct_name; // may be NULL
  char *return_struct_name; // may be NULL

  struct MYX_GRT_FUNCTION_PRIVATE *priv; // private data for each loader
} MYX_GRT_FUNCTION;

/// [SCRIPT::LibInterfaceMapper] -generate_classes MYX_GRT_MODULE
typedef struct MYX_GRT_MODULE
{
  MYX_GRT_MODULE_LOADER *loader;
  char *name;
  char *path;
  unsigned int functions_num;
  MYX_GRT_FUNCTION *functions;
  char *extends;
  struct MYX_GRT_MODULE_PRIVATE *priv; // private data for each loader
} MYX_GRT_MODULE;


typedef struct
{
  char *name;

  MYX_GRT_VALUE *(*function)(MYX_GRT_VALUE*,void*);
} MYX_GRT_BUILTIN_FUNCTION;


typedef struct
{
  char *name;
  char *extends;
  unsigned int functions_num;
  MYX_GRT_BUILTIN_FUNCTION *functions;
} MYX_GRT_BUILTIN_MODULE;
  

typedef enum
{
  MYX_BUILTIN_MODULE_TYPE,
    MYX_JAVA_MODULE_TYPE,
    MYX_LUA_MODULE_TYPE,
    MYX_PYTHON_MODULE_TYPE
//    MYX_PERL_MODULE_TYPE,
} MYX_GRT_MODULE_TYPE;

/*
 * Functions
 */

typedef void (*MYX_GRT_PRINT_CALLBACK)(const char *, void*);
typedef void (*MYX_GRT_LOG_CALLBACK)(MYX_GRT *, int, const char *, const char*);

MYX_PUBLIC_FUNC int myx_free_stringlist(MYX_STRINGLIST *stringlist);

MYX_PUBLIC_FUNC MYX_GRT * myx_grt_initialize();
MYX_PUBLIC_FUNC void myx_grt_finalize(MYX_GRT *grt);

MYX_PUBLIC_FUNC MYX_GRT_ERROR myx_grt_register_module_loader(MYX_GRT *grt, MYX_GRT_MODULE_LOADER *loader);

MYX_PUBLIC_FUNC int myx_grt_scan_for_modules(MYX_GRT *grt, const char *directory, MYX_GRT_ERROR *error);
MYX_PUBLIC_FUNC int myx_grt_scan_for_structs(MYX_GRT *grt, const char *directory, MYX_GRT_ERROR *error);

MYX_PUBLIC_FUNC void myx_grt_add_msg(MYX_GRT *grt, int msg_type, const char *message, MYX_STRINGLIST *details, int copy_details);

// Shell

MYX_PUBLIC_FUNC void myx_grt_set_output_callback(MYX_GRT *grt, void *user_data,
                                 void (*process_output_func)(const char *text, void *userdata));

// Modules

MYX_PUBLIC_FUNC void myx_grt_module_set_log_callback(MYX_GRT *grt, 
                                                    void (*log_func)(MYX_GRT*, int, const char *, const char *));

MYX_PUBLIC_FUNC MYX_GRT_ERROR myx_grt_module_init(MYX_GRT *grt, const char *filename);

MYX_PUBLIC_FUNC MYX_GRT_ERROR myx_grt_module_register_builtin(MYX_GRT *grt, MYX_GRT_BUILTIN_MODULE *module, void *function_data);

MYX_PUBLIC_FUNC int myx_grt_module_get_count(MYX_GRT *grt);
MYX_PUBLIC_FUNC MYX_STRINGLIST * myx_grt_module_get_names(MYX_GRT *grt);
MYX_PUBLIC_FUNC MYX_GRT_MODULE * myx_grt_module_get(MYX_GRT *grt, const char *name);
MYX_PUBLIC_FUNC MYX_GRT_MODULE * myx_grt_module_get_by_index(MYX_GRT *grt, unsigned int index);
MYX_PUBLIC_FUNC int myx_grt_modules_get_that_extend(MYX_GRT *grt, const char *module, MYX_GRT_MODULE **retmodules[]);
MYX_PUBLIC_FUNC MYX_GRT_MODULE_TYPE myx_grt_module_get_type(MYX_GRT_MODULE *module);

MYX_PUBLIC_FUNC int myx_grt_module_function_get_count(MYX_GRT_MODULE *module);
MYX_PUBLIC_FUNC MYX_STRINGLIST * myx_grt_module_function_get_names(MYX_GRT_MODULE *module);
MYX_PUBLIC_FUNC MYX_GRT_FUNCTION * myx_grt_module_function_get(MYX_GRT_MODULE *module, const char *name);
MYX_PUBLIC_FUNC MYX_GRT_FUNCTION * myx_grt_module_function_get_by_index(MYX_GRT_MODULE *module, unsigned int index);
MYX_PUBLIC_FUNC char * myx_grt_module_function_get_params(MYX_GRT_FUNCTION *func);
MYX_PUBLIC_FUNC char * myx_grt_module_function_get_return_type(MYX_GRT_FUNCTION *func);

// GRT Functions

MYX_PUBLIC_FUNC MYX_GRT_FUNCTION * myx_grt_function_get(MYX_GRT *grt, const char *module, const char *function_name, int search_parent);
MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_function_call(MYX_GRT *grt, MYX_GRT_FUNCTION *func,
                                    MYX_GRT_VALUE *argument, MYX_GRT_ERROR *error);

MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_function_get_and_call(MYX_GRT *grt, const char *module, const char *function_name, int search_parent,
                                    MYX_GRT_VALUE *argument, MYX_GRT_ERROR *error);

// GRT global functions

MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_get_root(MYX_GRT *grt);
MYX_PUBLIC_FUNC MYX_GRT_ERROR myx_grt_set_root(MYX_GRT *grt, MYX_GRT_VALUE *new_root);

MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_retrieve_from_file(const char *filename);
MYX_PUBLIC_FUNC MYX_GRT_ERROR myx_grt_store_to_file(MYX_GRT_VALUE *value, const char *filename);
  
// GRT Structs

MYX_PUBLIC_FUNC int myx_grt_struct_get_count(MYX_GRT *grt);
MYX_PUBLIC_FUNC int myx_grt_struct_get_child_count(MYX_GRT *grt, const char *struct_name);
MYX_PUBLIC_FUNC MYX_GRT_STRUCT * myx_grt_struct_get_by_index(MYX_GRT *grt, unsigned int index);
MYX_PUBLIC_FUNC MYX_GRT_STRUCT * myx_grt_struct_get_child_by_index(MYX_GRT *grt, const char *struct_name, unsigned int index);

MYX_PUBLIC_FUNC MYX_GRT_STRUCT * myx_grt_struct_get(MYX_GRT *grt, const char *name);

MYX_PUBLIC_FUNC MYX_GRT_STRUCTS * myx_grt_structs_get(MYX_GRT *grt);
MYX_PUBLIC_FUNC int myx_grt_structs_free(MYX_GRT_STRUCTS *gstructs);
MYX_PUBLIC_FUNC int myx_grt_struct_free(MYX_GRT_STRUCT *gstruct);

MYX_PUBLIC_FUNC MYX_GRT_ERROR myx_grt_struct_register(MYX_GRT *grt, MYX_GRT_STRUCT *gstruct);

MYX_PUBLIC_FUNC int myx_grt_struct_check_inherits_from(MYX_GRT *grt, const char *struct_name, const char *parent_name);

MYX_PUBLIC_FUNC const char * myx_grt_struct_get_name(MYX_GRT_STRUCT *gstruct);
MYX_PUBLIC_FUNC const char * myx_grt_struct_get_parent_name(MYX_GRT_STRUCT *gstruct);

MYX_PUBLIC_FUNC char * myx_grt_struct_to_xml(MYX_GRT *grt, MYX_GRT_STRUCT *gstruct, int include_children);
MYX_PUBLIC_FUNC MYX_GRT_STRUCTS * myx_grt_struct_from_xml(const char *str, size_t size);

MYX_PUBLIC_FUNC MYX_GRT_ERROR myx_grt_struct_save_list(MYX_GRT_STRUCTS *gstructs, const char *filename);
MYX_PUBLIC_FUNC MYX_GRT_STRUCTS * myx_grt_struct_load_list(const char *filename, MYX_GRT_ERROR *error);

MYX_PUBLIC_FUNC MYX_GRT_ERROR myx_grt_struct_load_and_register(MYX_GRT *grt, const char *filename);

MYX_PUBLIC_FUNC MYX_GRT_ERROR myx_grt_struct_export_java_classes(MYX_GRT_STRUCTS *gstructs, 
                                                                 const char *package_name, const char *output_path);

// GRT Structs Members

MYX_PUBLIC_FUNC unsigned int myx_grt_struct_get_member_count(MYX_GRT_STRUCT *gstruct);
MYX_PUBLIC_FUNC MYX_GRT_STRUCT_MEMBER * myx_grt_struct_get_member_by_index(MYX_GRT_STRUCT *gstruct, unsigned int index);

MYX_PUBLIC_FUNC const char * myx_grt_struct_get_member_name(MYX_GRT_STRUCT_MEMBER *member);

MYX_PUBLIC_FUNC MYX_GRT_VALUE_TYPE myx_grt_struct_get_member_type(MYX_GRT_STRUCT_MEMBER *member);
MYX_PUBLIC_FUNC const char * myx_grt_struct_get_member_value_struct_name(MYX_GRT_STRUCT_MEMBER *member);

// GRT Value handling

MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_value_retain(MYX_GRT_VALUE *value);
MYX_PUBLIC_FUNC int myx_grt_value_release(MYX_GRT_VALUE *value);
MYX_PUBLIC_FUNC int myx_grt_value_print(MYX_GRT *grt, MYX_GRT_VALUE *value);
  
MYX_PUBLIC_FUNC char * myx_grt_value_to_xml(MYX_GRT_VALUE *value);
MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_value_from_xml(const char *str, size_t size);

MYX_PUBLIC_FUNC MYX_GRT_VALUE_TYPE myx_grt_value_get_type(MYX_GRT_VALUE *value);
MYX_PUBLIC_FUNC const char * myx_get_value_type_as_string(MYX_GRT_VALUE_TYPE value_type);
MYX_PUBLIC_FUNC MYX_GRT_VALUE_TYPE myx_get_value_type_from_string(const char *value_type_name, MYX_GRT_ERROR *error);

MYX_PUBLIC_FUNC int myx_grt_value_as_int(MYX_GRT_VALUE *value);
MYX_PUBLIC_FUNC double myx_grt_value_as_double(MYX_GRT_VALUE *value);
MYX_PUBLIC_FUNC const char * myx_grt_value_as_string(MYX_GRT_VALUE *value);
MYX_PUBLIC_FUNC char * myx_grt_value_format_as_string(MYX_GRT_VALUE *value);
MYX_PUBLIC_FUNC MYX_GRT_LIST * myx_grt_value_as_list(MYX_GRT_VALUE *value);
MYX_PUBLIC_FUNC MYX_GRT_DICT * myx_grt_value_as_dict(MYX_GRT_VALUE *value);

// GRT Basic Type Values  

MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_value_from_int(int i);
MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_value_from_double(double d);
MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_value_from_string(const char *s);

// GRT Dictionary Values

MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_dict_new(const char *struct_name);
MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_dict_create(const char *struct_name, const char *key, ...);

MYX_PUBLIC_FUNC int myx_grt_dict_set_item(MYX_GRT_VALUE *dict, const char *key, MYX_GRT_VALUE *value);
MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_dict_get_item_value(MYX_GRT_VALUE *dict, const char *key);
MYX_PUBLIC_FUNC int myx_grt_dict_del_item(MYX_GRT_VALUE *dict, const char *key);

MYX_PUBLIC_FUNC const char * myx_grt_dict_item_as_string(MYX_GRT_VALUE *dict, const char *key);
MYX_PUBLIC_FUNC int myx_grt_dict_item_as_int(MYX_GRT_VALUE *dict, const char *key);
MYX_PUBLIC_FUNC double myx_grt_dict_item_as_double(MYX_GRT_VALUE *dict, const char *key);

MYX_PUBLIC_FUNC unsigned int myx_grt_dict_item_num(MYX_GRT_VALUE *dict);
MYX_PUBLIC_FUNC int myx_grt_dict_item_by_index(MYX_GRT_VALUE *dict, unsigned int index, 
                                          const char **retkey, MYX_GRT_VALUE **retvalue);
MYX_PUBLIC_FUNC const char * myx_grt_dict_item_key_by_index(MYX_GRT_VALUE *dict, unsigned int index);
MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_dict_item_value_by_index(MYX_GRT_VALUE *dict, unsigned int index);
  
MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_dict_set_value_by_path(MYX_GRT_VALUE *dict, const char *path);
MYX_PUBLIC_FUNC int myx_grt_dict_get_value_by_path(MYX_GRT_VALUE *dict, const char *path, MYX_GRT_VALUE *new_value);

MYX_PUBLIC_FUNC const char * myx_grt_dict_name_item_as_string(MYX_GRT_VALUE *dict);
MYX_PUBLIC_FUNC const char * myx_grt_dict_id_item_as_string(MYX_GRT_VALUE *dict);

// GRT List Values

MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_list_new(MYX_GRT_VALUE_TYPE content_type, const char *struct_name);

MYX_PUBLIC_FUNC MYX_GRT_VALUE_TYPE myx_grt_list_content_type(MYX_GRT_VALUE *list);
MYX_PUBLIC_FUNC MYX_STRINGLIST * myx_grt_list_as_stringlist(MYX_GRT_VALUE *list);

MYX_PUBLIC_FUNC int myx_grt_list_insert_item(MYX_GRT_VALUE *list, int index, MYX_GRT_VALUE *value);
#define myx_grt_list_add_item(list, value) myx_grt_list_insert_item(list, -1, value)
MYX_PUBLIC_FUNC int myx_grt_list_del_item(MYX_GRT_VALUE *list, int index);
MYX_PUBLIC_FUNC unsigned int myx_grt_list_item_num(MYX_GRT_VALUE *list);
MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_list_get_item(MYX_GRT_VALUE *list, unsigned int index);
MYX_PUBLIC_FUNC int myx_grt_list_set_item(MYX_GRT_VALUE *list, unsigned int index, MYX_GRT_VALUE *new_value);

// GRT Value - Struct handling

MYX_PUBLIC_FUNC int myx_grt_dict_assign_struct(MYX_GRT_VALUE *dict, const char *struct_name);
MYX_PUBLIC_FUNC int myx_grt_list_assign_struct(MYX_GRT_VALUE *list, const char *struct_name);

MYX_PUBLIC_FUNC const char * myx_grt_value_struct_name(MYX_GRT_VALUE *value);
MYX_PUBLIC_FUNC int myx_grt_value_validate_struct(MYX_GRT *grt, MYX_GRT_VALUE *value, const char *struct_name, int strict);

// GRT Utility Value Functions

MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_make_dict_from_list(MYX_GRT_VALUE *list);
MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_value_from_stringlist(MYX_STRINGLIST *sl);


// Java module
MYX_PUBLIC_FUNC MYX_GRT_MODULE_LOADER * myx_java_init_loader(MYX_GRT *grt, const char *class_path, MYX_GRT_ERROR *error);

#ifdef ENABLE_LUA_MODULES

// Lua module
MYX_PUBLIC_FUNC MYX_GRT_MODULE_LOADER * myx_lua_init_loader(MYX_GRT *grt, MYX_GRT_ERROR *error);


// Lua Shell
MYX_PUBLIC_FUNC MYX_GRT_ERROR myx_grt_init_lua_shell(MYX_GRT *grt);
MYX_PUBLIC_FUNC char * myx_grt_lua_shell_get_prompt(MYX_GRT *grt);
MYX_PUBLIC_FUNC MYX_GRT_SHELL_COMMAND myx_grt_lua_shell_execute(MYX_GRT *grt, const char *linebuf);
MYX_PUBLIC_FUNC int myx_grt_lua_shell_run_file(MYX_GRT *grt, const char *file_name);
MYX_PUBLIC_FUNC void myx_grt_shell_print_welcome(MYX_GRT *grt);
MYX_PUBLIC_FUNC lua_State * myx_grt_lua_shell_get_lua(MYX_GRT *grt);
MYX_PUBLIC_FUNC MYX_GRT_VALUE * myx_grt_lua_shell_get_global_var(MYX_GRT *grt, const char *var_name);
MYX_PUBLIC_FUNC void myx_grt_lua_shell_set_global_var(MYX_GRT *grt, const char *var_name, MYX_GRT_VALUE *value);

#endif

// Module messaging (with sockets)
MYX_PUBLIC_FUNC int myx_grt_setup_module_messaging(MYX_GRT *grt); 
MYX_PUBLIC_FUNC int myx_grt_cleanup_module_messaging(MYX_GRT *grt);
MYX_PUBLIC_FUNC MYX_SOCKET myx_grt_get_messaging_fd(MYX_GRT *grt);
MYX_PUBLIC_FUNC MYX_SOCKET myx_grt_get_messaging_module_fd(MYX_GRT *grt);
MYX_PUBLIC_FUNC int myx_grt_check_module_connected(MYX_GRT *grt);
MYX_PUBLIC_FUNC int myx_grt_check_module_message(MYX_GRT *grt, char **message, int timeout);
MYX_PUBLIC_FUNC int myx_grt_send_module_message(MYX_GRT *grt, const char *message);
  
//
MYX_PUBLIC_FUNC int _myx_grt_get_refcount(MYX_GRT_VALUE *value);

#ifdef __cplusplus
}
#endif

#endif
