unit UniCodeConsole;

interface

uses UniCodeEditor, Unicode, Classes, Contnrs, Types,
  Windows, SysUtils, Controls;

type
  TExecuteCommandProcedure = procedure(Cmd: WideString; Key: Word; Shift: TShiftState) of object;
  TIsMultilineCommandFunction = function(Cmd: WideString; Key: Word; Shift: TShiftState): Boolean of object;

  TCommandHistoryObject = class
    constructor Create(Cmd: WideString);
  private
    FCmd: WideString;
  published
    property Cmd: WideString read FCmd write FCmd;
  end;

  TCustomUniCodeConsole = class(TCustomUniCodeEdit)
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  private
    CommandHistory: TObjectList;
    CommandHistoryPos: integer;

    StoredCaretPos: TPoint;

    CurrentCommandStartLineNr: Integer;
    CurrentCommandEndLineNr: Integer;

    FOnExecuteCommand: TExecuteCommandProcedure;
    FOnIsMultilineCommand: TIsMultilineCommandFunction;

    FConsolePrompt: WideString;
    FConsolePromptLen: Integer;

    FConsoleDelimiter: WideString;

    procedure SetConsolePrompt(ConsolePrompt: WideString);
    procedure SetConsoleCommandSelStart(value: integer);

    procedure SetConsoleCommand(Cmd: WideString);
    function GetConsoleCommand: WideString;

    procedure SetConsoleCommandExcludeDelimiter(Cmd: WideString);
    function GetConsoleCommandExcludeDelimiter: WideString;
  protected
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;

    function GetConsolePromptMultiline: WideString;

    property ConsolePromptLen: Integer read FConsolePromptLen;
    property ConsolePrompt: WideString read FConsolePrompt write SetConsolePrompt;

    property ConsoleDelimiter: WideString read FConsoleDelimiter write FConsoleDelimiter;

    property OnExecuteCommand: TExecuteCommandProcedure read FOnExecuteCommand write FOnExecuteCommand;
    property OnIsMultilineCommand: TIsMultilineCommandFunction read FOnIsMultilineCommand write FOnIsMultilineCommand;
  public
    procedure PrepareNextConsoleCommand;
    procedure PrepareNextConsoleCommandMultiline;

    property ConsoleCommand: WideString read GetConsoleCommand write SetConsoleCommand;
    property ConsoleCommandExcludeDelim: WideString read GetConsoleCommandExcludeDelimiter write
      SetConsoleCommandExcludeDelimiter;
    property ConsoleCommandSelStart: Integer write SetConsoleCommandSelStart;
  end;

  TUniCodeConsole = class(TCustomUniCodeConsole)
  published
    property Align;
    property Anchors;
    property Constraints;
    property BevelEdges;
    property BevelInner;
    property BevelOuter;
    property BevelKind;
    property BevelWidth;
    property BorderWidth;
    property BorderStyle;
    property BookMarkOptions;
    property CharWidth;
    property Color;
    property Ctl3D;
    property Enabled;
    property ExtraLineSpacing;
    property Font;
    property GutterColor;
    property GutterWidth;
    property Height;
    property HighLighter;
    property IndentSize;
    property InsertCaret;
    property Keystrokes;
    property LineNumberFont;
    property MarginColor;
    property MaxRightChar;
    property MaxUndo;
    property Name;
    property Options;
    property OverwriteCaret;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property RightMargin;
    property ShowHint;
    property ScrollBars;
    property ScrollHintColor;
    property SelectedColor;
    property TabOrder;
    property TabSize;
    property TabStop default True;
    property Tag;
    property Visible;
    property Width;

    property OnBookmarkChange;
    property OnCaretChange;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnPaint;
    property OnProcessCommand;
    property OnProcessUserCommand;
    property OnReplaceText;
    property OnSettingChange;
    property OnStartDrag;

    property ConsolePromptLen;
    property ConsolePrompt;

    property ConsoleDelimiter;

    property OnExecuteCommand;
    property OnIsMultilineCommand;
  end;

implementation

constructor TCommandHistoryObject.Create(Cmd: WideString);
begin
  inherited Create;

  FCmd := Cmd;
end;

constructor TCustomUniCodeConsole.Create(AOwner: TComponent);
begin
  inherited;

  CommandHistory := TObjectList.Create;

  FOnExecuteCommand := nil;
  FOnIsMultilineCommand := nil;

  ConsolePrompt := '> ';

  FConsoleDelimiter := ';';
end;

destructor TCustomUniCodeConsole.Destroy;
begin
  CommandHistory.Free;

  inherited;
end;

procedure TCustomUniCodeConsole.PrepareNextConsoleCommand;
begin
  Content.AddLine(ConsolePrompt);
  CaretY := Content.Count;
  CaretX := ConsolePromptLen;
  Invalidate;

  CurrentCommandStartLineNr := Content.Count - 1;
  CurrentCommandEndLineNr := CurrentCommandStartLineNr;
end;

procedure TCustomUniCodeConsole.KeyDown(var Key: Word; Shift: TShiftState);
var
  Cmd: WideString;
  CmdTrimmed: WideString;
begin
  if (Length(Content[CaretY].Text) >= MaxRightChar) then
    MaxRightChar := MaxRightChar + 20;

  if (Key = VK_Return) and (Shift<>[ssShift]) then
  begin
    Cmd := GetConsoleCommand;
    CmdTrimmed := Trim(Cmd);

    if (CompareText(CmdTrimmed, 'cls') = 0) then
    begin
      Content.Clear;
      MaxRightChar := 80;

      PrepareNextConsoleCommand;
      Invalidate;
      Key := 0;
    end
    else
      if (CmdTrimmed = '') then
      begin
        PrepareNextConsoleCommand;
        Key := 0;
      end
      else
        if (CmdTrimmed <> '') then
        begin
          if (Copy(CmdTrimmed, Length(CmdTrimmed) -
            Length(FConsoleDelimiter) + 1, Length(FConsoleDelimiter)) = FConsoleDelimiter) then
          begin
            if (Assigned(OnExecuteCommand)) then
              OnExecuteCommand(Cmd, Key, Shift)
            else
              PrepareNextConsoleCommand;
          end
          else
          begin
            if (Assigned(OnIsMultilineCommand)) then
            begin
              if (not (OnIsMultilineCommand(Cmd, Key, Shift))) then
              begin
                if (Assigned(OnExecuteCommand)) then
                  OnExecuteCommand(Cmd, Key, Shift)
                else
                  PrepareNextConsoleCommand;
              end
              else
                PrepareNextConsoleCommandMultiline;
            end
            else
              PrepareNextConsoleCommandMultiline;

            //Add command to history
            if (cmd <> '') then
            begin
              CommandHistory.Add(TCommandHistoryObject.Create(cmd));
              CommandHistoryPos := CommandHistory.Count;
            end;

            cmd := '';
          end;

          Key := 0;
        end;
  end
  else if (Key = VK_Return) and (Shift=[ssShift]) then
  begin
    ConsoleCommand := ConsoleCommand + #13#10;
  end
  else
    if (Key = VK_UP) and (Shift = []) then
    begin
      if (CaretY <= CurrentCommandStartLineNr) then
        Key := 0;
    end
    else
      if (Key = VK_DOWN) and (Shift = []) then
      begin
        if (CaretY >= CurrentCommandEndLineNr) then
          Key := 0;
      end

      else
        if (Key = VK_UP) and (Shift = [ssCtrl]) then
        begin
          if (CommandHistoryPos > 0) then
          begin
            dec(CommandHistoryPos);
            ConsoleCommand := TCommandHistoryObject(CommandHistory[CommandHistoryPos]).Cmd;
          end;

          Key := 0;
        end
        else
          if (Key = VK_DOWN) and (Shift = [ssCtrl]) then
          begin
            if (CommandHistoryPos < CommandHistory.Count - 1) then
            begin
              inc(CommandHistoryPos);
              ConsoleCommand := TCommandHistoryObject(CommandHistory[CommandHistoryPos]).Cmd;
            end
            else
            begin
              ConsoleCommand := '';
      {Lines[CaretY]:=ConsolePrompt;
      CaretX:=ConsolePromptLen;}
              CommandHistoryPos := CommandHistory.Count;
            end;

            Key := 0;
          end
          else
            if (Key = VK_HOME) then
            begin
              CaretX := ConsolePromptLen;

              Key := 0;
            end
            else
              if (Key = VK_PRIOR) then
              begin
                Key := 0;
              end
              else
                if (Key = VK_NEXT) then
                begin
                  Key := 0;
                end
                else
                  if (Key = VK_BACK) or (Key = VK_LEFT) then
                  begin
                    if (CaretX < Integer(ConsolePromptLen + 1)) then
                      Key := 0;
                  end
                  else
                    if (Key = VK_RIGHT) then
                    begin
                      if (CaretX >= Length(Content[CaretY].Text)) then
                        Key := 0;
                    end
                    else
                      if (Key = VK_ESCAPE) then
                      begin
                        ConsoleCommand := '';

                        Key := 0;
                      end;

  if (Key <> 0) then
    inherited;
end;

procedure TCustomUniCodeConsole.PrepareNextConsoleCommandMultiline;
begin
  Content.AddLine(GetConsolePromptMultiline);
  CaretY := Content.Count;
  CaretX := ConsolePromptLen;
  Invalidate;

  inc(CurrentCommandEndLineNr);
end;

procedure TCustomUniCodeConsole.SetConsoleCommand(Cmd: WideString);
var
  CmdLines: TWideStringList;
  i: integer;
begin
  if (Content.Count > 0) then
  begin
    //Clear current command
    for i := 0 to CurrentCommandEndLineNr - CurrentCommandStartLineNr do
      Content.DeleteLine(Content.Count - 1);
  end;

  CurrentCommandStartLineNr := Content.Count;

  //Add new command
  CmdLines := TWideStringList.Create;
  try
    CmdLines.Text := Cmd;

    if (CmdLines.Count > 0) then
    begin
      Content.AddLine(ConsolePrompt + CmdLines[0]);

      for i := 1 to CmdLines.Count - 1 do
        Content.AddLine(GetConsolePromptMultiline + CmdLines[i]);
    end
    else
      Content.AddLine(ConsolePrompt);
  finally
    CmdLines.Free;
  end;

  CurrentCommandEndLineNr := Content.Count - 1;

  SetCaretToEditorBottom;

  Invalidate;
end;

function TCustomUniCodeConsole.GetConsoleCommand: WideString;
var
  cmd: WideString;
  i: integer;
begin
  if (Content.Count > 0) then
  begin
    cmd := '';
    for i := CurrentCommandStartLineNr to CurrentCommandEndLineNr do
      if (CurrentCommandEndLineNr < Content.Count) then
        cmd := cmd +
          Copy(Content[i].Text, ConsolePromptLen + 1, Length(Content[i].Text)) + #13#10;

    Result := cmd;
  end
  else
    Result := '';
end;

procedure TCustomUniCodeConsole.SetConsoleCommandExcludeDelimiter(Cmd: WideString);
begin
  Cmd := TrimRight(Cmd);

  if (Copy(Cmd, Length(Cmd) -
    Length(ConsoleDelimiter) + 1, Length(ConsoleDelimiter)) = ConsoleDelimiter) then
    ConsoleCommand := Cmd
  else
    ConsoleCommand := Cmd + ConsoleDelimiter;
end;

function TCustomUniCodeConsole.GetConsoleCommandExcludeDelimiter: WideString;
begin
  Result := TrimRight(ConsoleCommand);

  Result := Copy(Result, 1, Length(Result) -
    Length(ConsoleDelimiter));
end;

procedure TCustomUniCodeConsole.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  StoredCaretPos.X := CaretX;
  StoredCaretPos.Y := CaretY;

  if (StoredCaretPos.Y > Content.Count) then
    StoredCaretPos.Y := Content.Count;

  if (StoredCaretPos.X < Integer(ConsolePromptLen)) then
    StoredCaretPos.X := ConsolePromptLen;

  inherited;
end;

procedure TCustomUniCodeConsole.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  if (SelStart<>SelEnd) then
    CopyToClipboard;

  CaretY := StoredCaretPos.Y;
  CaretX := StoredCaretPos.X;

  //ClearSelection;
  //SelStart := 0;

  BlockBegin := Point(0,0);
  BlockEnd := Point(0,0);

  inherited;
end;

procedure TCustomUniCodeConsole.SetConsolePrompt(ConsolePrompt: WideString);
var
  OldConsolePrompt: WideString;
  OldConsolePromptLen: Integer;
  i: integer;
begin
  OldConsolePrompt := FConsolePrompt;
  OldConsolePromptLen := ConsolePromptLen;

  FConsolePrompt := ConsolePrompt;
  FConsolePromptLen := Length(FConsolePrompt);

  if (Content.Count > 0) then
    if (CurrentCommandStartLineNr < Content.Count) then
      if (Copy(Content[CurrentCommandStartLineNr].Text, 1, OldConsolePromptLen) =
        OldConsolePrompt) then
      begin
        for i := CurrentCommandStartLineNr to CurrentCommandEndLineNr do
        begin
          if (i = CurrentCommandStartLineNr) then
            Content[i].Text := ConsolePrompt +
              Copy(Content[i].Text, OldConsolePromptLen + 1, Length(Content[i].Text))
          else
            Content[i].Text := GetConsolePromptMultiline +
              Copy(Content[i].Text, OldConsolePromptLen + 1, Length(Content[i].Text));
        end;
      end;

  SetCaretToEditorBottom;
end;

procedure TCustomUniCodeConsole.SetConsoleCommandSelStart(value: integer);
var
  Loop: Integer;
  Count: Integer;

begin
  Loop := CurrentCommandStartLineNr;
  Count := 0;
  while ((Count + Length(Content[Loop].Text) - ConsolePromptLen + 2) < value)
    and (Loop < Content.Count) do
  begin
    Count := Count + Length(Content[Loop].Text) - ConsolePromptLen + 2;
    Inc(loop);
  end;
  CaretY := Loop;
  CaretX := ConsolePromptLen + Value - Count;
end;

function TCustomUniCodeConsole.GetConsolePromptMultiline: WideString;
begin
  if (ConsolePromptLen > 3) then
    Result := StringOfChar(' ', ConsolePromptLen - 3) + '>> '
  else
    Result := '>>';
end;

end.

