# -*- coding: utf-8 -*-
#
# «core» - Mythbuntu Control Centre base class
#
# Portions from GDebi copyright Michael Vogt
# Copyright (C) 2007, Mario Limonciello, for Mythbuntu
#
#
# Mythbuntu is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this application; if not, write to the Free Software Foundation, Inc., 51
# Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##################################################################################

import os
import sys
import pygtk
import re
import string
pygtk.require("2.0")

import gtk
import gtk.glade

#MySQL testing support
import MySQLdb

#Case for change
from MythbuntuControlCentre.changer import Applicator

#To Store Current/Old settings
import ConfigParser

#Checking package status cleanly
import warnings
warnings.filterwarnings("ignore", "apt API not stable yet", FutureWarning)
import apt_pkg

#Package reloading support
from softwareproperties.gtk.SoftwarePropertiesGtk import DialogCacheOutdated

from debconftalk import debconftalk

from aptsources import distro
from aptsources.sourceslist import SourcesList, SourceEntry
from softwareproperties.SoftwareProperties import SoftwareProperties

import commands

GLADEDIR = '/usr/share/mythbuntu-control-centre/glade'

# Define what tabs of the UI we want to load.
SUBTABS = [
    "tab_system_roles",
    "tab_mythtv_configuration",
    "tab_applications_plugins",
    "tab_proprietary_drivers",
    "tab_proprietary_codecs",
    "tab_remote_control",
    "tab_system_services",
    "tab_artwork_behavior",
    "tab_network_shares",
    "tab_advanced"
]

class ControlCentre():
    def __init__(self):
        """Initializes the control center app"""
        def add_subpage(self, tabs, name):
            """Inserts a subtab into the notebook.  This assumes the file
            shares the same base name as the page you are looking for."""
            gladefile = GLADEDIR + '/' + name + '.glade'
# See http://faq.pygtk.org/index.py?req=show&file=faq22.002.htp
# for internationalisation support
            gladexml = gtk.glade.XML(gladefile, name, 'mythbuntu-control-centre')
            widget = gladexml.get_widget(name)
            tabs.append_page(widget)
            add_widgets(self, gladexml)
            gladexml.signal_autoconnect(self)

        def add_widgets(self, glade):
            """Makes all widgets callable by the toplevel."""
            for widget in glade.get_widget_prefix(""):
                setattr(self, widget.get_name(), widget)
                # We generally want labels to be selectable so that people can
                # easily report problems in them
                # (https://launchpad.net/bugs/41618), but GTK+ likes to put
                # selectable labels in the focus chain, and I can't seem to turn
                # this off in glade and have it stick. Accordingly, make sure
                # labels are unfocusable here.
                if isinstance(widget, gtk.Label):
                    widget.set_property('can-focus', False)
        def populate_lirc(self):
            """Fills the lirc pages with the appropriate data"""
            #Note, this requires lirc >= 0.8.2
            self.remote_count=1
            hwdb = open('/usr/share/lirc/lirc.hwdb').readlines()
            hwdb.sort()
            #Filter out uncessary lines
            filter = "^\#|^\["
            #Filter out the /dev/input/eventX remote
            filter += "|http"
            pattern = re.compile(filter)
            for line in hwdb:
                if pattern.search(line) is None:
                    list = string.split(line, ";")
                    if len(list) > 1:
                        #Make sure we have a config file before including
                        if list[4] != "":
                            self.remote_list.append_text(list[0].translate(string.maketrans('',''),','))
                            self.remote_driver.append_text(list[1])
                            if list[2] != "none":
                                self.remote_modules.append_text(list[2])
                            else:
                                self.remote_modules.append_text("")
                            self.remote_rc.append_text(list[4])
                            self.remote_count = self.remote_count + 1
            self.remote_list.set_active(0)

        def populate_user_ids(self):
            """Returns all valid user id's for autostart"""
            self.user_count=0
            in_f=open("/etc/passwd")
            for line in in_f:
                fields =  string.split(line,":")
                id=int(fields[2])
                if id > 999 and id < 10000:
                    self.autologin_user.append_text(fields[0])
                    self.user_count+=1
            #double check to see if the user 'mythtv' is in usage too.
            if os.path.exists("/etc/gdm/gdm-cdd.conf"):
                in_f=open("/etc/gdm/gdm-cdd.conf")
                for line in in_f:
                    if re.compile("^AutomaticLogin=").search(line):
                        user=string.split(string.split(line,"=")[1],'\n')[0]
                        if user == "mythtv":
                            self.autologin_user.append_text(user)
                            self.user_count+=1
                        break

        # Initialize apt
        apt_pkg.init()
        self.cache = apt_pkg.GetCache()

        #Initialize configuration comparer
        self.config = ConfigParser.ConfigParser()
        self.config.add_section("mythbuntu")

        # load the main interface
        self.glade = gtk.glade.XML(GLADEDIR + '/' + 'mythbuntu_control_centre.glade')
        add_widgets(self, self.glade)

        # load subtabs of interface
        tabs = self.glade.get_widget("tabs")
        for tab in SUBTABS:
            add_subpage(self, tabs, tab)
        self.glade.signal_autoconnect(self)

        #Populate statically detected interface content
        populate_lirc(self)
        populate_user_ids(self)

        #Populate dynamically detected interface content
        self.revert_gui()


    def destroy(self, widget, data=None):
        gtk.main_quit()

    def display_error(self,message):
        """Displays an error message"""
        dialog = gtk.MessageDialog(self.main_window, gtk.DIALOG_MODAL,
                                   gtk.MESSAGE_ERROR, gtk.BUTTONS_CLOSE,
                                   message)
        dialog.run()
        dialog.hide()

    def run(self):
        self.main_window.show()
        gtk.main()


#Helper Functions
    def find_case_for_change(self):
        """Finds and marks all changes made within the interface"""

        to_install = []
        to_remove = []
        to_reconfigure = {}
        text = ""

        #Gather Information
        ##Backend Role
        old_backend=self.config.get("mythbuntu","backend")
        if self.primary_backend_radio.get_active():
            new_backend="primary"
        elif self.secondary_backend_radio.get_active():
            new_backend="secondary"
        else:
            new_backend="none"
        ##Frontend Role
        old_frontend=self.config.get("mythbuntu","frontend")
        if self.frontend_radio.get_active():
            new_frontend="frontend"
        else:
            new_frontend="none"
        #mysql.txt
        old_user=self.config.get("mythbuntu","mythtv_mysql_user")
        new_user=self.mysql_user.get_text()
        old_pass=self.config.get("mythbuntu","mythtv_mysql_password")
        new_pass=self.mysql_pass.get_text()
        old_database=self.config.get("mythbuntu","mythtv_mysql_database")
        new_database=self.mysql_database.get_text()
        old_server=self.config.get("mythbuntu","mythtv_mysql_server")
        new_server=self.mysql_server.get_text()
        # mythweb auth
        # shouldn't we get old values from debconf instead? :(
        old_mythwebauthenable=self.config.get("mythbuntu","mythweb_auth_activated")
        if old_mythwebauthenable=="False":
            old_mythwebauthenable=False
        else:
            old_mythwebauthenable=True
        new_mythwebauthenable=self.mythweb_password_checkbutton.get_active()
        old_mythwebauthuser=self.config.get("mythbuntu","mythweb_auth_username")
        new_mythwebauthuser=self.mythweb_username.get_text()
        old_mythwebauthpass=self.config.get("mythbuntu","mythweb_auth_password")
        new_mythwebauthpass=self.mythweb_password.get_text()

        #remote control
        old_remote=self.config.get("mythbuntu","remote")
        new_remote=self.remote_list.get_active_text()
        #Medibuntu
        old_medibuntu=self.config.get("mythbuntu","medibuntu")
        if old_medibuntu=="False":
            old_medibuntu=False
        else:
            old_medibuntu=True
        new_medibuntu=self.enable_medibuntu.get_active()

        #Automatic Login
        old_autologin=self.config.get("mythbuntu","autologin")
        old_autologin_user=self.config.get("mythbuntu","autologin_user")
        if old_autologin=="False":
            old_autologin=False
        else:
            old_autologin=True
        new_autologin=self.enableautologin.get_active()
        new_autologin_user=self.autologin_user.get_active_text()
        #Artwork
        old_artwork=self.config.get("mythbuntu","artwork")
        if old_artwork=="False":
            old_artwork=False
        else:
            old_artwork=True
        new_artwork=self.enable_artwork.get_active()

        #Generate text and action lists
        ##Roles
        if old_backend != new_backend or old_frontend != new_frontend:
            text+="System Roles:\n"
        ###Backend
        if old_backend != new_backend:
            if new_backend == "none":
                text+="   Remove " + old_backend + " backend role from system.\n"
                if old_backend == "primary":
                    to_remove.append("mythtv-backend-master")
                    to_remove.append("mythtv-database")
                to_remove.append("mythtv-backend")
                to_remove.append("mythweb")
            elif old_backend == "none":
                text+="   Add " + new_backend + " backend role to the system.\n"
                if new_backend == "primary":
                    to_install.append("mythtv-backend-master")
                else:
                    to_install.append("mythtv-backend")
            else:
                text+="   Switch " + old_backend + " backend role into a " + new_backend + " role.\n"
                if new_backend == "primary":
                    to_install.append("mythtv-backend-master")
                else:
                    to_remove.append("mythtv-backend-master")
                    to_remove.append("mythtv-database")
        if new_backend == "primary" or new_backend == "secondary":
            to_reconfigure["mythtv-backend"]=True
        else:
            to_reconfigure["mythtv-backend"]=False
        ###Frontend
        if old_frontend != new_frontend:
            if new_frontend == "none":
                text+="   Remove frontend role from system.\n"
                to_remove.append("mythtv-frontend")
                to_remove.append("mythtv-themes")
            else:
                text+="   Add frontend role to system.\n"
                to_install.append("mythtv-frontend")
                to_install.append("mythtv-themes")
        if new_frontend == "frontend":
            to_reconfigure["mythtv-frontend"]=True
        else:
            to_reconfigure["mythtv-frontend"]=False
        ##MythTV Configuration
        if old_pass != new_pass or old_user != new_user or old_server != new_server or old_database != new_database:
            text+="MythTV Configuration\n"
            to_reconfigure["mythtv_mysql_pass"]=new_pass
            to_reconfigure["mythtv_mysql_database"]=new_database
            to_reconfigure["mythtv_mysql_user"]=new_user
            to_reconfigure["mythtv_mysql_server"]=new_server
            if old_user != new_user:
                text +="   Change MythTV MySQL user name.\n"
            if old_pass != new_pass:
                text +="   Change MythTV MySQL user password.\n"
            if old_database != new_database:
                text +="   Change MythTV MySQL database.\n"
            if old_server != new_server:
                text +="   Change MythTV MySQL server.\n"

        #Checkbox items (Plugins, Media Players, And Proprietary Codecs, MythWeb auth)
        seen_title=False
        mythweb_do_not_configure=False
        if old_medibuntu != new_medibuntu:
            if not seen_title:
                text+="Plugins, Media Players, and Proprietary Codecs:\n"
                seen_title=True
            if new_medibuntu:
                text+="   Enabling medibuntu repository.\n"
                to_reconfigure["medibuntu"]=True
            else:
                text+="   Disabling medibuntu repository.\n"
                to_reconfigure["medibuntu"]=False

        list = { "mytharchive": self.mytharchive_checkbox,   \
                 "mythbrowser": self.mythbrowser_checkbox,   \
                 "mythcontrols": self.mythcontrols_checkbox, \
                 "mythdvd": self.mythdvd_checkbox,           \
                 "mythflix": self.mythflix_checkbox,         \
                 "mythgame": self.mythgame_checkbox,         \
                 "mythgallery": self.mythgallery_checkbox,   \
                 "mythmusic": self.mythmusic_checkbox,       \
                 "mythnews": self.mythnews_checkbox,         \
                 "mythstream":self.mythstream_checkbox,      \
                 "mythphone": self.mythphone_checkbox,       \
                 "mythvideo": self.mythvideo_checkbox,       \
                 "mythweather": self.mythweather_checkbox,   \
                 "mythweb": self.mythweb_checkbox,           \
                 "mplayer": self.mplayer_checkbox,           \
                 "ubuntu-desktop": self.ubuntu_desk_check,   \
                 "xubuntu-desktop": self.xubuntu_desk_check, \
                 "kubuntu-desktop": self.kubuntu_desk_check, \
                 "xine-ui": self.xine_checkbox,              \
                 "vlc":  self.vlc_checkbox,                  \
                 "libdvdcss2": self.enable_libdvdcss2,       \
                 "ffmpeg": self.enable_ffmpeg,               \
                 "w32codecs": self.enable_w32codecs}
        if commands.getstatusoutput('uname -m')[1] == "x86_64":
            list["w64codecs"] = self.enable_w64codecs
        for item in list:
            result=self.config.get("mythbuntu",item)
            active=list[item].get_active()
            # if the plugin was previously installed and is not checked anymore in the GUI, remove it
            if result=="True" and not active:
                if not seen_title:
                    text+="Plugins, Media Players, and Proprietary Codecs:\n"
                    seen_title=True
                to_remove.append(item)
                if item == "mplayer":
                    to_remove.append("mplayer-nogui")
                if item == "mythweb":
                   # we mustn't reconfigure mythweb auth if it's to be removed
                    mythweb_do_not_configure = True
                text+="   Remove " + item + " from system.\n"
            elif result=="False" and active:
                if not seen_title:
                    text+="Plugins, Media Players, and Proprietary Codecs:\n"
                    seen_title=True
                to_install.append(item)
                text+="   Add " + item + " to system.\n"
            if item == "mplayer" or item == "xine-ui" or item == "vlc" or\
               item == "mythvideo" or item == "mythgallery" or item == "mythmusic":
                to_reconfigure[item]=active
        # mythweb auth
        # we only want to change anything if mythweb is installed
        # maybe we should wrap other mythweb code as well, but this should be sufficient
        if self.query_installed("mythweb") and not mythweb_do_not_configure:
            if old_mythwebauthenable != new_mythwebauthenable or old_mythwebauthuser != new_mythwebauthuser  or old_mythwebauthpass != new_mythwebauthpass:
                # FIXME: the following strings are likely not translatable yet
                text+="MythWeb Authentication\n"
                to_reconfigure["mythweb_auth_activated"]=new_mythwebauthenable
                to_reconfigure["mythweb_auth_username"]=new_mythwebauthuser
                to_reconfigure["mythweb_auth_password"]=new_mythwebauthpass
                if old_mythwebauthenable != new_mythwebauthenable:
                    text +="   Enable/Disable MythWeb authentication\n"
                if old_mythwebauthuser  != new_mythwebauthuser:
                    text +="   Change user name for MythWeb access\n"
                # old_mythwebauthpas should always be an empty string since we clear that debconf question in mythweb.postinst
                if old_mythwebauthpass != new_mythwebauthpass:
                    text +="   Change password for MythWeb access\n"

        seen_tweaks_title = False
        # mysql tweaks
        # yes, that's a string comparison here
        old_mysql_tweaks = self.config.get("mythbuntu","mysql_tweaks_enabled")
        if self.enable_mysql_tweaks.get_active() == True:
             # remember, config.get doesn't like booleans, we're working with strings here
             new_mysql_tweaks = "True"
        else:
             new_mysql_tweaks = "False"
        if old_mysql_tweaks != new_mysql_tweaks:
            to_reconfigure["mysql_tweaks_enabled"]=new_mysql_tweaks
            if not seen_tweaks_title:
                text+="Various Tweaks/Optimizations:\n"
                seen_tweaks_title = True
            text +="   Enable/Disable MySQL tweaks\n"

        old_defrag = self.config.get("mythbuntu","enable_nightly_defrag")
        # again, remember: we're converting boolean to strings here
        if self.enable_nightly_defrag.get_active() == True:
            new_defrag = "True"
        else:
            new_defrag = "False"
        if new_defrag != old_defrag:
            to_reconfigure["enable_nightly_defrag"]=new_defrag
            if not seen_tweaks_title:
                text+="Various Tweaks/Optimizations:\n"
                seen_tweaks_title = True
            text +="   Enable/Disable daily XFS cron job\n"
            if new_defrag == "True":
                # xfsdump contains xfs_fsr
                # BTW: we also need ionice which is in schedutils. ubuntu-minimal depends on schedutils so we should be safe
                if not self.query_installed("xfsdump"):
                    to_install.append("xfsdump")
            # if new_defrag == "False":
                # It feels wrong to remove xfsdump because the user might need it for other things
                # if self.query_installed("xfsdump"):
                #    to_remove.append("xfsdump")


        old_mysql_repair = self.config.get("mythbuntu","enable_mysql_repair")
        if self.enable_mysql_repair.get_active() == True:
            new_mysql_repair = "True"
        else:
            new_mysql_repair = "False"

        if old_mysql_repair != new_mysql_repair:
            to_reconfigure["new_mysql_repair"]=new_mysql_repair
            if not seen_tweaks_title:
                text+="Various Tweaks/Optimizations:\n"
                seen_tweaks_title = True
            text += "   Enable/Disable mysql repair/optimization cron job\n"

        #Services
        list = { "vnc4server": self.enablevnc,          \
                 "samba": self.enablesamba,             \
                 "nfs-kernel-server": self.enablenfs,   \
                 "openssh-server": self.enablessh,      \
                 "mysql-server": self.enablemysql       \
               }
        seen_title=False
        for item in list:
            result=self.config.get("mythbuntu",item)
            active=list[item].get_active()
            if result=="True" and active == 0:
                if not seen_title:
                    text+="Services:\n"
                    seen_title=True
                to_remove.append(item)
                if item == "vnc4server":
                    to_reconfigure[item]=False
                text+="   Remove " + item + " service.\n"
            elif result=="True" and active == 2:
                if not seen_title:
                    text+="Services:\n"
                    seen_title=True
                to_reconfigure[item]=True
                text+="   Reconfigure " + item + " service.\n"
                if item == "vnc4server":
                    to_reconfigure["vnc_password"] = self.vnc_password.get_text()
            elif result=="False" and active == 1:
                if not seen_title:
                    text+="Services:\n"
                    seen_title=True
                to_install.append(item)
                to_reconfigure[item]=True
                text+="   Add " + item + " service.\n"
                if item == "vnc4server":
                    to_reconfigure["vnc_password"] = self.vnc_password.get_text()
        #remote control
        seen_title=False
        if old_remote != new_remote:
            seen_title=True
            text+="Remote Control:\n"
            if not self.remotecontrol.get_active():
                text+="   Disable " + old_remote + " remote control.\n"
                to_reconfigure["remote"]=''
                to_reconfigure["driver"]=''
                to_reconfigure["modules"]=''
                to_reconfigure["lircd"]=''
            else:
                if old_remote != "(No Remote Selected)":
                    text+="   Change to " + new_remote + " remote control.\n"
                else:
                    text+="   Add to " + new_remote + " remote control.\n"
                to_reconfigure["remote"]=new_remote
                to_reconfigure["driver"]=self.remote_driver.get_active_text()
                to_reconfigure["modules"]=self.remote_modules.get_active_text()
                to_reconfigure["lircd"]=self.remote_rc.get_active_text()

        if self.generate_lircrc_checkbox.get_active():
            if not seen_title:
                seen_title=True
                text+="Remote Control:\n"
            to_reconfigure["lircrc"]=True
            text+="   Generate application specific lirc settings.\n"
        else:
            to_reconfigure["lircrc"]=False
        #automatic login
        seen_title=False
        if old_autologin != new_autologin or old_autologin_user != new_autologin_user:
            if new_autologin_user is not None and new_autologin_user != "[Disabled]":
                if not seen_title:
                    text+="Automatic Login & Artwork\n"
                    seen_title=True
                if old_autologin != new_autologin and  new_autologin:
                    text+="   Activate automatic login for " + new_autologin_user + ".\n"
                    to_reconfigure["autologin"]=True
                    to_reconfigure["autologin_user"]=new_autologin_user
                    if not self.query_installed("mythbuntu-desktop"):
                        to_install.append("mythbuntu-desktop")
                elif old_autologin != new_autologin and old_autologin:
                    text+="   Disable automatic login.\n"
                    to_reconfigure["autologin"]=False
                    to_reconfigure["autologin_user"]=old_autologin_user
                    if self.query_installed("ubuntu-mythtv-frontend"):
                        to_remove.append("ubuntu-mythtv-frontend")
                else:
                    text+="   Change automatic login from " + old_autologin_user + " to " + new_autologin_user + ".\n"
                    to_reconfigure["autologin"]=True
                    to_reconfigure["autologin_user"]=new_autologin_user
        #artwork
        if old_artwork != new_artwork:
            if not seen_title:
                text+="Automatic Login & Artwork\n"
                seen_title=True
            if new_artwork:
                text+="   Add Mythbuntu artwork.\n"
                to_install.append("mythbuntu-desktop")
            else:
                text+="   Remove Mythbuntu artwork.\n"
                to_remove.append("mythbuntu-default-settings")
                to_remove.append("mythbuntu-gdm-theme")
                to_remove.append("mythbuntu-artwork-usplash")
                to_remove.append("mythbuntu-desktop")
                #we can't have autologin without artwork, take out autologin if we take out artwork
                to_reconfigure["autologin"]=False
                to_reconfigure["autologin_user"]=old_autologin_user
                if self.query_installed("ubuntu-mythtv-frontend"):
                    to_remove.append("ubuntu-mythtv-frontend")



        for child in self.summary_text.get_children():
            self.summary_text.remove(child)

        if text == "":
            text = "No changes detected."
            self.apply_button3.set_sensitive(False)
        else:
            self.apply_button3.set_sensitive(True)
        summary_buffer = gtk.TextBuffer()
        summary_buffer.set_text(text)
        self.summary_text.set_buffer(summary_buffer)

        return [to_install,to_remove,to_reconfigure]

    def query_installed(self,package):
        """Determines if a single package is installed"""
        return self.cache[package].CurrentVer

    def set_gui_defaults(self):
        """Loads default settings"""
        #MythTV Setup
        self.mythtv_setup_vbox.set_sensitive(False)
        #Remote frontend/backend connection info
        self.mysql_config_vbox.set_sensitive(False)
        self.mysql_user.set_text("mythtv")
        self.mysql_pass.set_text("")
        self.mysql_database.set_text("mythconverg")
        self.mysql_server.set_text("")
        self.mysql_test_hbox.hide()
        #Remote Controls
        self.remotecontrol.set_active(False)
        #Plugins
        self.selectall_checkbox.set_active(False)
        self.toggle_plugins(self.selectall_checkbox)
        plugin_list = [ self.mytharchive_checkbox, self.mythbrowser_checkbox, self.mythcontrols_checkbox, \
                self.mythdvd_checkbox, self.mythflix_checkbox, self.mythgame_checkbox, \
                self.mythgallery_checkbox, self.mythmusic_checkbox, self.mythnews_checkbox, \
                self.mythstream_checkbox, self.mythphone_checkbox, self.mythvideo_checkbox, \
                self.mythweather_checkbox, self.mythweb_checkbox ]
        for plugin in plugin_list:
            plugin.set_sensitive(False)
            plugin.set_active(False)
        #Applications
        self.mplayer_checkbox.set_active(False)
        self.xine_checkbox.set_active(False)
        self.vlc_checkbox.set_active(False)
        #Roles
        self.no_backend_radio.set_active(True)
        self.no_frontend_radio.set_active(True)
        #Services
        self.vnc_installed=False
        self.enablevnc.set_active(0)
        self.vnc_pass_hbox.set_sensitive(False)
        self.vnc_password.set_text("")
        self.enablessh.set_active(0)
        self.enablesamba.set_active(0)
        self.enablenfs.set_active(0)
        self.mysql_vbox.set_sensitive(False)
        self.enablemysql.set_active(0)
        #Proprietary Drivers
        self.amd_vbox.set_sensitive(False)
        self.nvidia_vbox.set_sensitive(False)
        self.xorg_vbox.set_sensitive(False)
        #Proprietary Codecs
        self.enable_medibuntu.set_active(False)
        self.enable_libdvdcss2.set_active(False)
        self.enable_w32codecs.set_active(False)
        self.enable_w32codecs.set_active(False)
        self.enable_ffmpeg.set_active(False)
        #Autologin user
        self.autologin_user.set_active(0)
        self.enableautologin.set_active(False)
        #Artwork
        self.enable_artwork.set_active(False)
        self.artwork_vbox.set_sensitive(False)
        # mythweb auth
        # FIXME
        # self.mythweb_password_checkbutton.set_active(True)
        # mysqld tweaks
        self.enable_mysql_tweaks.set_active(False)
        # xfs defrag
        self.enable_nightly_defrag.set_active(False)
        # mythtv database optimization/repair
        self.enable_mysql_repair.set_active(False)


    def query_system_state(self):
        """Determines the state of the system, and what settings to activate"""
        #Roles
        ##Desktop Roles
        #DO NOT USE desktop. use one of the other 3.
        self.config.set("mythbuntu","desktop","none")
        if self.query_installed("ubuntu-desktop"):
            self.config.set("mythbuntu","ubuntu-desktop","True")
            self.artwork_vbox.set_sensitive(True)
        else:
            self.config.set("mythbuntu","ubuntu-desktop","False")
        if self.query_installed("kubuntu-desktop"):
            self.config.set("mythbuntu","kubuntu-desktop","True")
            self.artwork_vbox.set_sensitive(True)
        else:
            self.config.set("mythbuntu","kubuntu-desktop","False")
        if self.query_installed("xubuntu-desktop"):
            self.config.set("mythbuntu","xubuntu-desktop","True")
            self.artwork_vbox.set_sensitive(True)
        else:
            self.config.set("mythbuntu","xubuntu-desktop","False")


        ##Backend / Frontend Roles
        master_backend=self.query_installed("mythtv-backend-master")
        slave_backend=self.query_installed("mythtv-backend")
        frontend=self.query_installed("mythtv-frontend")

        enable_experimental_options_vbox = False
        if master_backend or slave_backend:
            self.mythtv_setup_vbox.set_sensitive(True)
            self.mythweb_checkbox.set_sensitive(True)
            enable_experimental_options_vbox = True

            if master_backend:
                self.config.set("mythbuntu","backend","primary")
                self.primary_backend_radio.set_active(True)
            else:
                self.config.set("mythbuntu","backend","secondary")
                self.secondary_backend_radio.set_active(True)
                self.mysql_config_vbox.set_sensitive(True)
        else:
            self.config.set("mythbuntu","backend","none")
            enable_experimental_options_vbox = False

        if frontend:
            self.config.set("mythbuntu","frontend","frontend")
            self.frontend_radio.set_active(True)
            if not master_backend:
                self.mysql_config_vbox.set_sensitive(True)
            plugin_list = [ self.mytharchive_checkbox, self.mythbrowser_checkbox, self.mythcontrols_checkbox, \
                self.mythdvd_checkbox, self.mythflix_checkbox, self.mythgame_checkbox, \
                self.mythgallery_checkbox, self.mythmusic_checkbox, self.mythnews_checkbox, \
                self.mythstream_checkbox, self.mythphone_checkbox, self.mythvideo_checkbox, \
                self.mythweather_checkbox ]
            for plugin in plugin_list:
                plugin.set_sensitive(True)
        else:
            self.config.set("mythbuntu","frontend","none")

        #Medibuntu Items
        if self.find_medibuntu():
            self.enable_medibuntu.set_active(True)
            self.enable_w32codecs.set_sensitive(True)
            self.enable_w64codecs.set_sensitive(True)
            self.enable_ffmpeg.set_sensitive(True)
            self.enable_libdvdcss2.set_sensitive(True)
            self.config.set("mythbuntu","medibuntu","True")
        else:
            self.enable_w32codecs.set_sensitive(False)
            self.enable_w64codecs.set_sensitive(False)
            self.enable_libdvdcss2.set_sensitive(False)
            self.enable_ffmpeg.set_sensitive(True)
            self.config.set("mythbuntu","medibuntu","False")
        if commands.getstatusoutput('uname -m')[1] != "x86_64":
            self.enable_w64codecs.set_sensitive(False)
        #Checkbox items (Plugins, Media Players, And Proprietary Codecs)
        list = { "mytharchive": self.mytharchive_checkbox,   \
                 "mythbrowser": self.mythbrowser_checkbox,   \
                 "mythcontrols": self.mythcontrols_checkbox, \
                 "mythdvd": self.mythdvd_checkbox,           \
                 "mythflix": self.mythflix_checkbox,         \
                 "mythgame": self.mythgame_checkbox,         \
                 "mythgallery": self.mythgallery_checkbox,   \
                 "mythmusic": self.mythmusic_checkbox,       \
                 "mythnews": self.mythnews_checkbox,         \
                 "mythphone": self.mythphone_checkbox,       \
                 "mythstream": self.mythstream_checkbox,     \
                 "mythvideo": self.mythvideo_checkbox,       \
                 "mythweather": self.mythweather_checkbox,   \
                 "mythweb": self.mythweb_checkbox,           \
                 "mplayer": self.mplayer_checkbox,           \
                 "xine-ui": self.xine_checkbox,              \
                 "ubuntu-desktop": self.ubuntu_desk_check,   \
                 "xubuntu-desktop": self.xubuntu_desk_check, \
                 "kubuntu-desktop": self.kubuntu_desk_check, \
                 "vlc":  self.vlc_checkbox,                  \
                 "libdvdcss2": self.enable_libdvdcss2,       \
                 "ffmpeg": self.enable_ffmpeg,               \
                 "w32codecs": self.enable_w32codecs}
        if commands.getstatusoutput('uname -m')[1] == "x86_64":
            list["w64codecs"] = self.enable_w64codecs
        for item in list:
            result = self.query_installed(item)
            if item == "mplayer" and not result:
                result = self.query_installed("mplayer-nogui")
            if result:
                list[item].set_active(True)
                self.config.set("mythbuntu",item,"True")
            else:
                list[item].set_active(False)
                self.config.set("mythbuntu",item,"False")
        #Services
        self.enablesamba.remove_text(2)
        self.enablevnc.remove_text(2)
        self.enablenfs.remove_text(2)
        if self.query_installed("samba"):
            self.enablesamba.set_active(1)
            self.enablesamba.append_text("Reconfigure")
            self.config.set("mythbuntu","samba","True")
        else:
            self.config.set("mythbuntu","samba","False")
        if self.query_installed("vnc4server"):
            self.vnc_installed=True
            self.enablevnc.set_active(1)
            self.enablevnc.append_text("Reconfigure")
            self.config.set("mythbuntu","vnc4server","True")
        else:
            self.config.set("mythbuntu","vnc4server","False")
        if self.query_installed("nfs-kernel-server"):
            self.enablenfs.set_active(1)
            self.enablenfs.append_text("Reconfigure")
            self.config.set("mythbuntu","nfs-kernel-server","True")
        else:
            self.config.set("mythbuntu","nfs-kernel-server","False")
        if self.query_installed("openssh-server"):
            self.enablessh.set_active(1)
            self.config.set("mythbuntu","openssh-server","True")
        else:
            self.config.set("mythbuntu","openssh-server","False")
        if self.query_installed("mythtv-database") and self.query_installed("mysql-server"):
            try:
                in_f=open("/etc/mysql/conf.d/mythtv.cnf")
                self.mysql_vbox.set_sensitive(True)
                found=False
                for line in in_f:
                    if re.compile("^bind-address").search(line):
                        self.enablemysql.set_active(1)
                        found=True
                        self.config.set("mythbuntu","mysql-server","True")
                if not found:
                    self.config.set("mythbuntu","mysql-server","False")
                in_f.close()
            except IOError:
                self.config.set("mythbuntu","mysql-server","False")
                self.display_error("/etc/mysql/conf.d/mythtv.cnf is missing.\nDisabling modifications to MySQL Service")
        else:
            self.config.set("mythbuntu","mysql-server","False")
        #Proprietary Drivers
        driver = self.find_proprietary_driver()
        if  driver == 'amdcccle':
            self.amd_vbox.set_sensitive(True)
            self.nvidia_vbox.set_sensitive(False)
        elif driver == 'nvidia-settings':
            self.amd_vbox.set_sensitive(False)
            self.nvidia_vbox.set_sensitive(True)
        if os.path.exists('/usr/bin/displayconfig-gtk'):
            self.xorg_vbox.set_sensitive(True)
        #Advanced tab
        if os.path.exists('/usr/bin/update-manager'):
            self.update_manager_vbox.set_sensitive(True)
        if os.path.exists('/usr/sbin/synaptic'):
            self.synaptic_vbox.set_sensitive(True)
        #Medibuntu Items
        if self.find_medibuntu():
            self.enable_medibuntu.set_active(True)
            self.config.set("mythbuntu","medibuntu","True")
        else:
            #w32codec_tmp = self.config.set("mythbuntu","w32codecs")
            #libdvdcss_tmp = self.config.get("mythbuntu","libdvdcss2")
            self.enable_w32codecs.set_sensitive(False)
            self.enable_w64codecs.set_sensitive(False)
            self.enable_libdvdcss2.set_sensitive(False)
            self.enable_ffmpeg.set_sensitive(False)
            self.config.set("mythbuntu","medibuntu","False")
        #Backend connection info
        try:
            in_f=open("/etc/mythtv/mysql.txt")
            for line in in_f:
                if re.compile("^DBHostName").search(line):
                    text=string.split(string.split(line,"=")[1],'\n')[0]
                    self.mysql_server.set_text(text)
                elif re.compile("^DBUserName").search(line):
                    text=string.split(string.split(line,"=")[1],'\n')[0]
                    self.mysql_user.set_text(text)
                elif re.compile("^DBName").search(line):
                    text=string.split(string.split(line,"=")[1],'\n')[0]
                    self.mysql_database.set_text(text)
                elif re.compile("^DBPassword").search(line):
                    text=string.split(string.split(line,"=")[1],'\n')[0]
                    self.mysql_pass.set_text(text)
            in_f.close()
        except IOError:
             self.display_error("/etc/mythtv/mysql.txt is missing.\nUsing default settings.")
        self.config.set("mythbuntu","mythtv_mysql_server",self.mysql_server.get_text())
        self.config.set("mythbuntu","mythtv_mysql_user",self.mysql_user.get_text())
        self.config.set("mythbuntu","mythtv_mysql_database",self.mysql_database.get_text())
        self.config.set("mythbuntu","mythtv_mysql_password",self.mysql_pass.get_text())
        #Remote Control info
        try:
            in_f=open("/etc/lirc/hardware.conf")
            pattern=re.compile("^REMOTE")
            found=False
            for line in in_f:
                if pattern.search(line):
                    for item in range(self.remote_count):
                        self.remote_list.set_active(item)
                        if self.remote_list.get_active_text() == string.split(line,'"')[1]:
                            found = True
                            self.remotecontrol.set_active(True)
                            self.config.set("mythbuntu","remote",self.remote_list.get_active_text())
                            break
            if not found:
                self.remote_list.set_active(0)
                self.generate_lircrc_checkbox.set_active(False)
                self.config.set("mythbuntu","remote","(No Remote Selected)")
            in_f.close()
        except IOError:
            self.display_error("/etc/lirc/hardware.conf is missing.\nUsing default settings.")
            self.remote_list.set_active(0)
            self.generate_lircrc_checkbox.set_active(False)
            self.config.set("mythbuntu","remote","(No Remote Selected)")
        #Automatic Login
        if os.path.exists("/etc/gdm/gdm-cdd.conf"):
            in_f=open("/etc/gdm/gdm-cdd.conf")
            pattern=re.compile("^AutomaticLoginEnable=true")
            pattern2=re.compile("^AutomaticLogin=")
            found_autologin=False
            found_user=False
            for line in in_f:
                if pattern.search(line):
                    self.enableautologin.set_active(True)
                    self.config.set("mythbuntu","autologin","True")
                    self.artwork_vbox.set_sensitive(False)
                    found_autologin=True
                if pattern2.search(line):
                    user=string.split(string.split(line,"=")[1],'\n')[0]
                    for item in range(1,self.user_count+1):
                        self.autologin_user.set_active(item)
                        if self.autologin_user.get_active_text() == user:
                            self.config.set("mythbuntu","autologin_user",user)
                            found_user=True
                            break
                if found_autologin and found_user:
                    break
            if not found_autologin:
                self.config.set("mythbuntu","autologin","False")
            if not found_user:
                self.config.set("mythbuntu","autologin_user","[Disabled]")
                self.autologin_user.set_active(0)
        else:
            self.config.set("mythbuntu","autologin","False")
            self.config.set("mythbuntu","autologin_user","[Disabled]")
        #Artwork Package
        if self.query_installed("mythbuntu-desktop"):
            self.enable_artwork.set_active(True)
            self.config.set("mythbuntu","artwork","True")
        else:
            self.config.set("mythbuntu","artwork","False")
        #Role not toggled
        self.role_toggled=False

        # mythweb auth
        if self.query_installed("mythweb"):
            # FIXME: looks like m-c-c is not supposed to install mythweb unless this is a master or slave backend
            # if the user installs mythweb manually, this could lead to an inconsistent GUI
            #
            self.mythweb_password_checkbutton.set_sensitive(True)

            if debconftalk("get","mythweb/enable") == "true":
                self.config.set("mythbuntu","mythweb_auth_activated","True")
            elif self.mythweb_password_checkbutton.get_active() == True:
                self.config.set("mythbuntu","mythweb_auth_activated","True")
            else:
                self.config.set("mythbuntu","mythweb_auth_activated","False")

            if debconftalk("get","mythweb/username"):
                self.config.set("mythbuntu","mythweb_auth_username",debconftalk("get","mythweb/username"));
                self.mythweb_username.set_text(debconftalk("get","mythweb/username"))
            elif  self.mythweb_username.get_text():
                self.config.set("mythbuntu","mythweb_auth_username",self.mythweb_username.get_text());
            else:
                self.config.set("mythbuntu","mythweb_auth_username","")

            if debconftalk("get","mythweb/password"):
                self.config.set("mythbuntu","mythweb_auth_password",self.mythweb_password.get_text());
            elif self.mythweb_password.get_text():
               self.config.set("mythbuntu", "mythweb_auth_password",self.mythweb_password.get_text())
            else:
               self.config.set("mythbuntu", "mythweb_auth_password","")

        else:
            self.mythweb_password_checkbutton.set_sensitive(False)
            self.config.set("mythbuntu","mythweb_auth_activated","False")
            self.config.set("mythbuntu","mythweb_auth_username","")
            self.config.set("mythbuntu", "mythweb_auth_password","")

        if self.config.get("mythbuntu","mythweb_auth_activated") == "True":
            self.mythweb_password_checkbutton.set_active(True)

        # mysql tweaks
        # maybe I should check if there is a master backend instead?
        # setting mysql_tweaks_enabled to false here so we don't break if mysql-server is not installed
        self.config.set("mythbuntu","mysql_tweaks_enabled","False")
        if self.query_installed("mysql-server"):
            self.enable_mysql_tweaks.set_sensitive(True)
            # I really should be doing this via debconf, but I can't yet.
            if os.path.exists("/etc/mysql/conf.d/mythtv-tweaks.cnf"):
                self.enable_mysql_tweaks.set_active(True)
                self.config.set("mythbuntu","mysql_tweaks_enabled","True")
            else:
                self.enable_mysql_tweaks.set_active(False)
                self.config.set("mythbuntu","mysql_tweaks_enabled","False")
        else:
            self.enable_mysql_tweaks.set_sensitive(False)

        # xfs defrag
        # check if xfsdump and util-linux are installed as well?
        # util-linux is part of ubuntu-minimal
        if os.path.exists("/etc/cron.daily/mythtv-xfs-defrag"):
            self.enable_nightly_defrag.set_active(True)
            self.config.set("mythbuntu","enable_nightly_defrag","True")
        else:
            self.enable_nightly_defrag.set_active(False)
            self.config.set("mythbuntu","enable_nightly_defrag","False")

        # mythtv database optimization/repair
        # optimize_mythdb.pl is in package mythtv-backend
        # maybe it'd be more future-proof if we checked for the existence of the file as well
        # FIXME: what happens if the user removes mythtv-backend? Maybe we should remove the cron job from the system then
        if os.path.exists("/etc/cron.daily/optimize_mythdb.pl"):
            self.enable_mysql_repair.set_active(True)
            self.mysql_tweak_button.set_sensitive(True)
            self.config.set("mythbuntu","enable_mysql_repair","True")
        else:
            self.enable_mysql_repair.set_active(False)
            self.mysql_tweak_button.set_sensitive(False)
            self.config.set("mythbuntu","enable_mysql_repair","False")

        if enable_experimental_options_vbox:
            self.experimental_options_vbox.set_sensitive(True)
        else:
            self.experimental_options_vbox.set_sensitive(False)
            self.enable_nightly_defrag.set_active(False)
            self.enable_mysql_tweaks.set_active(False)
            self.enable_mysql_repair.set_active(False)



    def find_medibuntu(self):
        """Determines if the medibuntu repo is active"""
        in_f=open("/etc/apt/sources.list")
        pattern=re.compile("^deb http://packages.medibuntu.org/ gutsy free non-free")
        found=False
        for line in in_f:
            if pattern.search(line):
                found=True
        if not found and os.path.exists("/etc/apt/sources.list.d/medibuntu.list"):
            found=True
        return found

    def find_proprietary_driver(self):
        """Determines if amdcccle or nvidia-settings are avail"""
        if os.path.exists('/usr/bin/amdcccle'):
            return 'amdcccle'
        elif os.path.exists('/usr/bin/nvidia-settings'):
            return 'nvidia-settings'

    def find_lircrc(self):
        """Determines if there is current a lircrc for the current user"""
        return os.path.exists(os.getenv('HOME') + '/.lircrc')

#Signal call back
    def refresh_pressed(self,widget):
        """Asks the user if they want to reload package lists"""
        refresh = DialogCacheOutdated(self.main_window,"/usr/share/software-properties/")
        refresh.run()

    def apply_pressed(self,widget):
        """Asks the user if they want to continue"""

        def call_applicator(self,to_install,to_remove,to_reconfigure):
            """Calls the application routine"""
            self.main_window.hide()
            apply = Applicator(to_install,to_remove,to_reconfigure)
            if not apply.get_init_error():
                apply.commit_changes()
            self.cache = apt_pkg.GetCache()
            self.main_window.show()

        (to_install,to_remove,to_reconfigure) = self.find_case_for_change()
        if (widget is not None):
            #apply button from main window
            if (widget.get_name() == "apply_button"):
                self.apply_dialog.run()
                self.apply_dialog.hide()
            #apply button from summary window
            elif (widget.get_name() == "apply_button3"):
                self.apply_dialog.hide()
                call_applicator(self,to_install,to_remove,to_reconfigure)
                self.revert_gui()
            #apply button from forced apply window
            elif (widget.get_name() == "apply_button2"):
                self.quick_apply_dialog.hide()
                call_applicator(self,to_install,to_remove,to_reconfigure)
                self.revert_gui()

    def revert_gui(self,widget=None):
        """Resets the GUI to old settings"""
        self.set_gui_defaults()
        self.query_system_state()

    def tab_change(self,widget):
        """Called whenever a request to change tabs is made"""
        if widget is not None:
            if self.role_toggled:
                self.quick_apply_dialog.run()
                self.quick_apply_dialog.hide()
                self.role_toggled=False
            if widget.get_name() == 'system_roles_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_system_roles))
            elif widget.get_name() == 'mythtv_configuration_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_mythtv_configuration))
            elif widget.get_name() == 'remote_control_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_remote_control))
            elif widget.get_name() == 'applications_plugins_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_applications_plugins))
            elif widget.get_name() == 'proprietary_drivers_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_proprietary_drivers))
            elif widget.get_name() == 'system_services_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_system_services))
            elif widget.get_name() == 'artwork_behavior_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_artwork_behavior))
            elif widget.get_name() == 'network_shares_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_network_shares))
            elif widget.get_name() == 'proprietary_codecs_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_proprietary_codecs))
            elif widget.get_name() == 'advanced_button':
                self.tabs.set_current_page(self.tabs.page_num(self.tab_advanced))
            self.buttons_area.set_sensitive(True)

    def toggle_autologin(self,widget):
        """Called whenever autologin is toggled"""
        if widget is not None:
            if widget.get_name() == "enableautologin":
                active = widget.get_active()
                no_desktop = not (self.ubuntu_desk_check.get_active() or \
                self.kubuntu_desk_check.get_active() or \
                self.xubuntu_desk_check.get_active())
                self.autologin_users_hbox.set_sensitive(active)
                if not active:
                    self.autologin_user.set_active(0)
                    if no_desktop:
                        self.artwork_vbox.set_sensitive(False)
                    else:
                        self.artwork_vbox.set_sensitive(True)
                else:
                    self.autologin_user.set_active(1)
                    self.artwork_vbox.set_sensitive(False)

            elif widget.get_name() == "autologin_user":
                if widget.get_active() == 0:
                    self.enableautologin.set_active(False)
                if widget.get_active_text() == "mythtv":
                    self.mythtv_user_warning.show()
                else:
                    self.mythtv_user_warning.hide()

    def toggle_plugins(self,widget):
        """Called whenever a request to enable / disable all plugins"""
        if (widget is not None and widget.get_name() == 'selectall_checkbox'):
            plugin_list = [ self.mytharchive_checkbox, self.mythbrowser_checkbox, self.mythcontrols_checkbox, \
                self.mythdvd_checkbox, self.mythflix_checkbox, self.mythgame_checkbox, \
                self.mythgallery_checkbox, self.mythmusic_checkbox, self.mythnews_checkbox, \
                self.mythstream_checkbox, self.mythphone_checkbox, self.mythvideo_checkbox, \
                self.mythweather_checkbox, self.mythweb_checkbox ]

            toggle = widget.get_active()
            for item in plugin_list:
                if item.flags() & gtk.SENSITIVE:
                    item.set_active(toggle)


    def on_mythweb_password_checkbutton_toggled(self,widget):
        """Called whenever the advanced configuration button for plugins is pressed"""
        # show the username/password fields
        self.password_table.set_sensitive(self.mythweb_password_checkbutton.get_active())

    def toggle_medibuntu(self,widget):
        """Called whenever medibuntu is toggled"""
        if (widget is not None and widget.get_name() == "enable_medibuntu"):
            self.medibuntu_vbox.set_sensitive(widget.get_active())
            #Revert old package settings


    def toggle_vnc(self,widget):
        """Called whenever VNC service is toggled"""
        if (widget is not None and widget.get_name() == 'enablevnc'):
            iteration=1
            if self.vnc_installed:
                iteration = 2
            if widget.get_active() == iteration:
                self.vnc_pass_hbox.set_sensitive(True)
                self.vnc_error_image.show()
                self.apply_button.set_sensitive(False)
            else:
                self.vnc_pass_hbox.set_sensitive(False)
                self.vnc_error_image.hide()
                self.apply_button.set_sensitive(True)
                self.vnc_password.set_text("")
                self.vnc_error_image.hide()
                self.apply_button.set_sensitive(True)

    def toggle_vnc_password(self,widget):
        if (widget is not None and widget.get_name() == 'vnc_password'):
            password= widget.get_text().split(' ')[0]
            if len(password) >= 6:
                self.vnc_error_image.hide()
                self.apply_button.set_sensitive(True)
            else:
                self.vnc_error_image.show()
                self.apply_button.set_sensitive(False)

    def toggle_remotecontrol(self,widget):
        """Called whenever a request to enable/disable remote is called"""
        if widget is not None:
            if widget.get_name() == 'remotecontrol':
                self.remote_hbox.set_sensitive(widget.get_active())
                self.generate_lircrc_checkbox.set_sensitive(widget.get_active())
                if widget.get_active():
                    self.generate_lircrc_checkbox.set_active(not self.find_lircrc())
                    if self.remote_list.get_active() == 0:
                        self.remote_list.set_active(1)
                else:
                    self.remote_list.set_active(0)
                    self.generate_lircrc_checkbox.set_active(False)
            elif widget.get_name() == 'remote_list':
                if self.remote_list.get_active() == 0:
                    self.remotecontrol.set_active(False)
                self.remote_driver.set_active(widget.get_active())
                self.remote_modules.set_active(widget.get_active())
                self.remote_rc.set_active(widget.get_active())
                self.generate_lircrc_checkbox.set_active(True)

    def do_connection_test(self,widget):
        """Tests to make sure that the backend is accessible"""
        self.mysql_test_hbox.show()
        host = self.mysql_server.get_text()
        database = self.mysql_database.get_text()
        user = self.mysql_user.get_text()
        password = self.mysql_pass.get_text()
        try:
            db = MySQLdb.connect(host=host, user=user, passwd=password,db=database)
            cursor = db.cursor()
            cursor.execute("SELECT NULL")
            result = cursor.fetchone()
            cursor.close()
            db.close()
            result = "Successful"
            self.fail_mysql.hide()
            self.pass_mysql.show()
        except:
            result = "Failure"
            self.fail_mysql.show()
            self.pass_mysql.hide()
        self.connection_results_label.show()
        self.connection_results.set_text(result)

    def toggle_role(self,widget):
        """Called when a role change occurs"""
        if widget is not None:
            name=widget.get_name()
            activate_artwork=False
            if name == "no_backend_radio" or name == "primary_backend_radio" or name == "secondary_backend_radio":
                old_active=self.config.get("mythbuntu","backend")
                if (self.no_backend_radio.get_active() and old_active != "none"):
                    self.role_toggled=True
                elif (self.primary_backend_radio.get_active() and old_active != "primary") or \
                   (self.secondary_backend_radio.get_active() and old_active != "secondary"):
                    activate_artwork=True
                    self.role_toggled=True
            elif name == "no_frontend_radio" or name == " frontend_radio":
                old_active=self.config.get("mythbuntu","frontend")
                if (self.no_frontend_radio.get_active() and old_active != "none"):
                    self.role_toggled=True
                elif (self.frontend_radio.get_active() and old_active != "frontend" ):
                    activate_artwork=True
                    self.role_toggled=True
            elif name == "ubuntu_desk_check" or name == "kubuntu_desk_check" or name == "xubuntu_desk_check":
                old_u_active=self.config.get("mythbuntu","ubuntu-desktop")
                old_x_active=self.config.get("mythbuntu","xubuntu-desktop")
                old_k_active=self.config.get("mythbuntu","kubuntu-desktop")
                if (self.ubuntu_desk_check.get_active() and old_u_active == "False") or \
                   (self.kubuntu_desk_check.get_active() and old_k_active == "False") or \
                   (self.xubuntu_desk_check.get_active() and old_x_active == "False"):
                    self.role_toggled=True
            if activate_artwork:
                self.enable_artwork.set_active(True)

    def launch_app(self,widget):
        """Launches an external application"""
        if widget is not None:
            self.main_window.hide()
            while gtk.events_pending():
                gtk.main_iteration()
            if widget.get_name() == 'restricted_drivers_button':
                os.system('/usr/bin/restricted-manager')
            elif widget.get_name() == 'amdcccle_button':
                os.system('/usr/bin/amdcccle')
            elif widget.get_name() == 'nvidia_settings_button':
                os.system('/usr/bin/nvidia-settings')
            elif widget.get_name() == 'mythtv_setup_button':
                os.system("/usr/bin/mythtv-setup")
            elif widget.get_name() == 'displayconfig_button':
                os.system("/usr/bin/displayconfig-gtk")
            elif widget.get_name() == 'synaptic_button':
                os.system("/usr/sbin/synaptic")
            elif widget.get_name() == 'updatemanager_button':
                os.system("/usr/bin/update-manager")
            elif widget.get_name() == 'terminal_button':
                os.system(self.getUserTerm())
            elif widget.get_name() == 'mysql_tweak_button':
                os.system('/usr/bin/x-terminal-emulator -e perl /usr/share/doc/mythtv-backend/contrib/optimize_mythdb.pl')
            self.main_window.show()
    def getUserTerm(self):
        uid = os.getenv('SUDO_UID')
        if uid == None:
            return "/usr/bin/x-terminal-emulator"
        else:
            in_f=open("/etc/passwd")
            for line in in_f:
                fields =  string.split(line,":")
                if fields[2] == uid:
                    return "su " + fields[0] + ' -c "x-terminal-emulator"'

    def on_mythweb_checkbox_toggled(self,widget):
        if self.mythweb_checkbox.get_active() == False:
            self.mythweb_password_checkbutton.set_active(False)
            self.mythweb_password_checkbutton.set_sensitive(False)
        else:
            # only make mythweb auth UI available if mythweb is installed!
            # for some reason, config.get does not work here. query_installed might be slower but.. oh well.
            if self.query_installed("mythweb"):
                self.mythweb_password_checkbutton.set_sensitive(True)
        # Note: it's not a good idea to enable the mythweb auth UI unless mythweb is completely installed
        # The UI will be activated once m-c-c reloads its config
