#! /usr/bin/perl -w

# collect2_data.pl - The collector of nagios_grapher
# Copyright (C) 2004, NETWAYS GmbH, Gerd Mueller, Marius Hein
# $Id: collect2.pl 1557 2007-03-07 12:18:05Z gmueller $
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

use strict;
use RRDs;
use POSIX;
use Data::Dumper;
use Digest::MD5 qw(md5_hex);
use Getopt::Long;
use File::Basename;
use NagiosGrapher;
use Time::HiRes qw( usleep );
use IO::Socket::INET;

use vars qw (
  $progname
  $rrdpath
  $rrdtool
  $configfile
  $url
  $notes_url
  $step
  $default_heartbeat
  $heartbeat
  $host
  $service
  $perf
  $perfdata
  $orig_service
  @lines
  @blocks
  %tmp
  @tmp
  $path
  $file
  @system
  $update
  $template
  %values
  $key
  $rrd_err
  @srvext_lines
  $serviceextinfo
  $serviceext_path
  $serviceext_host
  $serviceext_type
  $check_host
  $check
  $srvext
  $xml_hash
  $xml_file
  $xml_tmp
  $tmppath
  $pipe
  $fifo_string
  $opt_verbose
  $opt_help
  $i
  @sigs
  $image_tag
  $image_src
  $image_static
  $image_script
  $new_image_tag
  $new_image_script
  $graph_url
  $log_file
  $log_level
  $dummy
  $socket
  $net_port
  $net_buffer
  $interface
  $nagios_config
  $ng
);

# LogLevel
my $LOG_VALUES_OK=1;
my $LOG_VALUES_NOK=2;
my $LOG_PRG_STATUS=4;
my $LOG_SERVICEEXT=8;
my $LOG_RRD_STATUS=16;
my $LOG_PIPE=32;
my $LOG_DEBUG_REGEX=64;
my $LOG_PRINT_CONFIG=128;
my $LOG_HOOK_MESSAGES=256;

# signals
@sigs = keys(%SIG);
for (@sigs) {
    $SIG{$_} = \&sig_handler;
}

# subs
sub print_help;
sub sig_handler;
sub load_config;
sub daemonize;
sub delete_pipe;
sub create_pipe;

# Configvalues
$progname = basename($0);

# Object
$ng = new NagiosGrapher();

# Prepare the daemon environment
$ng->daemon_prepare();


# Fetching some options...
Getopt::Long::Configure('bundling');
GetOptions(
    "v"    => \$opt_verbose,
    "h"    => \$opt_help,
    "help" => \$opt_help,
);

# Daemonize if needed
unless ($opt_verbose) {
    &daemonize;
}
# Load default config.
load_config;

# Restart marker
RESTART:

# Deamon settings
$pipe            = $ng->get_ngraphercfg_value('pipe');
$rrdpath         = $ng->get_ngraphercfg_value('rrdpath');
$tmppath         = $ng->get_ngraphercfg_value('tmppath');
$configfile      = $ng->get_ngraphercfg_value('ngraphcfg');
$serviceext_type = $ng->get_ngraphercfg_value('serviceext_type');
$serviceextinfo  = $ng->get_ngraphercfg_value('serviceextinfo');
$serviceext_path = $ng->get_ngraphercfg_value('serviceext_path');

# Nagios icons
$image_tag       = $ng->get_ngraphercfg_value('icon_image_tag');
$image_src		 = $ng->get_ngraphercfg_value('icon_image_src');
$image_static	 = $ng->get_ngraphercfg_value('icon_image_static');
$image_script	 = $ng->get_ngraphercfg_value('icon_image_script');

$default_heartbeat       = $ng->get_ngraphercfg_value('heartbeat');
$url = $ng->get_ngraphercfg_value('url');
$notes_url = $ng->get_ngraphercfg_value('notes_url');

$interface = 'pipe';
$interface = $ng->get_ngraphercfg_value('interface');

$net_port = 5667;
$net_port = $ng->get_ngraphercfg_value('port');

$net_buffer = 1024;
$net_buffer = $ng->get_ngraphercfg_value('buffer');


## write a pidfile and change the user/group
#$ng->daemon_write_pidfile();
#
#if (!$ng->daemon_change_user()) {
#	$ng->print_log ("PRG: Can not change user/group. EXIT.");
#	exit (1);
#}

# Setting up XML Files
$xml_file = $rrdpath . 'index.xml';
unless (-e $tmppath) {
    mkdir $tmppath;
}
$xml_tmp  = $tmppath. 'nagios_grapher.xml.tmp';

if($opt_verbose) {
	$ng->set_log_level(511);
} else {
	$opt_verbose=$ng->get_log_level();
}

# Print help if wanted
print_help if ($opt_help);

$ng->print_log ("PRG: Starting up $progname (PID: $$) ...") if ($opt_verbose & $LOG_PRG_STATUS);

# Setting up main rrd path
unless ( -e $rrdpath ) {
    mkdir($rrdpath);
    $ng->print_log ("RRD: Creating path $rrdpath ...") if ($opt_verbose & $LOG_RRD_STATUS);
}

# Getting XML Data
$xml_hash = $ng->get_ngrapherxml();

if ($interface =~ /network/i && defined($net_port)) {
	# Creating a UDP socket
	$ng->print_log ("PRG: using UDP socket (port: $net_port)") if ($opt_verbose & $LOG_PRG_STATUS);
	$socket = IO::Socket::INET->new(
		LocalPort => $net_port,
		Proto => 'udp'
	) || die "Creating socket faild: $@";
}
else {
	# Setting Up Named Pipe
	$ng->print_log ("PRG: using FIFO") if ($opt_verbose & $LOG_PRG_STATUS);
	create_pipe();
}

# LOOP MAIN
while (1) {
	# No buffered streams
	$| = 1;

	# Reading from FIFO or from UDP socket.
	if ($interface =~ /network/i && defined($net_port) && defined($socket)) {
		# Read from UDP socket
		$fifo_string = '';

		$socket->recv($fifo_string, $net_buffer);

		my ($s_port, $s_ipaddr) = sockaddr_in($socket->peername);
		$s_ipaddr = gethostbyaddr($s_ipaddr, AF_INET);

		$ng->print_log("NET: got udp message from $s_ipaddr:$s_port") if ($opt_verbose & $LOG_PIPE);
		$fifo_string =~ s/\\t/\t/g;
		$fifo_string =~ s/\\n/\n/g;

		$socket->send("NagiosGrapher: thanks $s_ipaddr:$s_port.");
		process_data($fifo_string);
	}
	else {
	    # Open the PIPE and read
	    open( FIFO, $pipe )
	      || die "Can't open pipe ($pipe)";
	    process_data($_) while(<FIFO>);
	    close(FIFO);
	}

}
# End LOOP MAIN

# Default exit
exit(0);


# Subs:
sub process_data {
	my $fifo_string = shift();
    chomp($fifo_string);
    $ng->print_log ("PIPE: $fifo_string") if ( defined($fifo_string) && ($opt_verbose & $LOG_PIPE));

    # Splitting pipe values
    ( $host, $service, $perf, $perfdata ) = split( /\t/, $fifo_string );

    if (   defined($host)
        && defined($service)
        && defined($perf)
        && defined($fifo_string) )
    {

	my $log_header="[".$host."][".$service."]:";

        # Change defaults...
        $orig_service = $service;
        $perf =~ tr/[A-Z]/[a-z]/;
        $perf =~ s/\\/\\/g;

        @blocks = $ng->get_ngrapherdef($service, 'graph');
        if ( @blocks > 0 ) {

			$ng->print_log("REGEX: ". scalar @blocks. " blocks for '$service' found.") if ($opt_verbose & $LOG_DEBUG_REGEX);

            # Getting the values from RegEX and perf data
            $check = "FALSE";
            foreach (@blocks) {
            	$ng->print_log("REGEX: graph_value=". $_->{graph_value}) if ($opt_verbose & $LOG_DEBUG_REGEX);
                $_->{service_name} =~ s/\s+$//g;
                if ( defined($perfdata)
                    && exists( $_->{graph_perf_regex} ) )
                {
                	$ng->print_log("REGEX: output=perfdata") if ($opt_verbose & $LOG_DEBUG_REGEX);
                    $ng->print_log("REGEX: ". 'regex=m/'. $_->{graph_perf_regex}. '/i') if ($opt_verbose & $LOG_DEBUG_REGEX);
                    $ng->print_log("REGEX: ". 'perfdata='. $perfdata) if ($opt_verbose & $LOG_DEBUG_REGEX);

                    # Taking PerfData
                    if ( $perfdata =~ m/$_->{graph_perf_regex}/i ) {
                    	my $value = $1;
                    	$value =~ s/,/\./;
                        $values{ $_->{graph_value} } = $value;
                        $check = "TRUE";
                        $ng->print_log("REGEX: ". 'match='. $value) if ($opt_verbose & $LOG_DEBUG_REGEX);
                    }
                    else {
                    	$ng->print_log("REGEX: NO MATCH.") if ($opt_verbose & $LOG_DEBUG_REGEX);
                        $ng->print_log ("VALUES: ".$log_header."No matching perfdata values found...")
                          if ($opt_verbose & $LOG_VALUES_NOK);
                        goto OUT;
                    }
                }
                else {
                	$ng->print_log("REGEX: output=plugin.") if ($opt_verbose & $LOG_DEBUG_REGEX);
                    $ng->print_log("REGEX: ". 'regex=m/'. $_->{graph_log_regex}. '/i') if ($opt_verbose & $LOG_DEBUG_REGEX);
                    $ng->print_log("REGEX: ". 'perfdata='. $perf) if ($opt_verbose & $LOG_DEBUG_REGEX);

                    # Taking Plugin Output
                    if ( $perf =~ m/$_->{graph_log_regex}/i ) {
                    	my $value = $1;
                    	$value =~ s/,/\./;
                        $values{ $_->{graph_value} } = $value;
                        $check = "TRUE";
                        $ng->print_log("REGEX: ". 'match='. $value) if ($opt_verbose & $LOG_DEBUG_REGEX);
                    }
                    else {
                    	$ng->print_log("REGEX: NO MATCH.") if ($opt_verbose & $LOG_DEBUG_REGEX);
                        $ng->print_log ("VALUES: ".$log_header."No matching output values found...")
                          if ($opt_verbose & $LOG_VALUES_NOK);
                        goto OUT;
                    }
                }
            }

            # If no regex match the service ... next round
            if ( $check eq "FALSE" ) {
                $ng->print_log ("VALUES: ".$log_header."No Action taken...") if ($opt_verbose & $LOG_VALUES_NOK);
                goto OUT;
            }

            # Checking XML Entry
            if ( $xml_hash->{host}->{$host}->{service}->{$service}->{service_name} )
            {
                # print "Exists\n";
            }
            else {

                # new default values
                $xml_hash->{host}->{$host}->{service}->{$service} = {
                    rrd_file => md5_hex( $host . "_" . $service . "_" . $orig_service ). ".rrd",
                    service_title => $orig_service,
                    service_name  => $service,
                };

                # TODO: Not the right position ...
                # $ng->write_ngrapherxml($xml_tmp, $xml_file, $xml_hash);
                # $ng->save_object();
            }

            # Clearing some var's
            undef @system;
            undef $update;
            undef $template;
            undef $rrd_err;

            # Adding path if not exists
            $path = $rrdpath . $host;
            mkdir($path) if ( !( -e $path ) );

            # Setting up the RRD Filename
            $file =
                $path . "/"
              . $xml_hash->{host}->{$host}->{service}->{$service}->{rrd_file};

            # If the RRD not exists:
            if ( !( -e $file ) ) {

                # Adding Graph Value Key's
                foreach (@blocks) {

                			my $min='U';
                			my $max='U';

                        # change the default rrdtype if a different is defined
                        my $rrd_type = "GAUGE";
                        $rrd_type = $_->{rrd_type}
                          if ( exists( $_->{rrd_type} ) );

                        # Setting the step
                        if (exists($_->{step})) {
                            $step = $ng->StepFromCache($host,$service, $_->{step});
                        }
                        else {
                            $step = $ng->StepFromCache($host,$service);
                        }

                        # setting the heartbeat
                        if (!defined($heartbeat)) {
                        	$heartbeat = $default_heartbeat;
                        	$heartbeat = $_->{heartbeat} if ( exists( $_->{heartbeat} ));
                        }

                 		# Setting the Heartbeat to auto value
						if (defined($step) && defined($default_heartbeat) && $default_heartbeat =~ m/auto/i) {
							$heartbeat = $step*2;
						}

                        if (exists($_->{rrd_min})) {
                            $min=$_->{rrd_min};
                        }
                        if (exists($_->{rrd_max})) {
                            $max=$_->{rrd_max};
                        }

                        # Adding the datasources ...
                        push @system,
                          "DS:"
                          . $_->{graph_value} . ":"
                          . $rrd_type
                          . ":". $heartbeat. ":".$min.":".$max;
                }

                # Adding some default average frames
                # OLD: 5, 30, 120, 1440;

                for (1, 6, 24, 288) {
	                push @system, "RRA:AVERAGE:0.5:$_:$heartbeat";
	                push @system, "RRA:MAX:0.5:$_:$heartbeat";
	                push @system, "RRA:MIN:0.5:$_:$heartbeat";
                }

                $heartbeat = undef;

                # Create the RRD
                # $step = $default_step;

                RRDs::create( $file, "--step", $step, @system );

                if($opt_verbose & $LOG_RRD_STATUS) {
                	$dummy="";
                	foreach (@system) {
                		$dummy.=$_. ' ';
                	}
                	$ng->print_log("RRD: rrdtool create $file --step=$step $dummy");
                }

                # Setting the step to undef for next loop
                $step = undef;

                $rrd_err = RRDs::error;

                # Debugging for create RRD's
                if ($rrd_err) {
                    $ng->print_log ("RRD: ".$log_header."$rrd_err") if ($opt_verbose & $LOG_RRD_STATUS);
                }
                else {
                	$ng->write_ngrapherxml($xml_tmp, $xml_file, $xml_hash);
					$ng->print_log ("PRG: writing index file.") if ($opt_verbose & $LOG_PRG_STATUS);
                	$ng->save_object();
                    $ng->print_log ("RRD: ".$log_header."$file - successfully created!") if ($opt_verbose & $LOG_RRD_STATUS);
                }
            }

            # Check if an serviceext entry exists
            if ( $serviceext_type eq "MULTIPLE" ) {
                mkdir $serviceext_path unless ( -e $serviceext_path );
                $serviceext_host = $host;
                $serviceext_host =~ s/\W/_/gs;
                $serviceextinfo =
                  $serviceext_path . "/" . $serviceext_host . ".cfg";
            }

            # don't know if this is very cool ...
            #elsif ( $serviceext_type eq "SINGLE" && !( -e $serviceextinfo ) ) {
            #    die("No serviceextinfo file defined, can't write!");
            #}

            system( "touch", $serviceextinfo ) if ( !-e $serviceextinfo );
			if ( -w $serviceextinfo ) {
	            open( SRVEXT, "<$serviceextinfo" )
	              || die("Can open '$serviceextinfo' for reading\n");
	            @srvext_lines = <SRVEXT>;
	            close(SRVEXT);

	            $check_host = $host;
	            $check_host =~ s/\./\./g;

	            $check = "0";
	            foreach (@srvext_lines) {
	                if($_ =~ /define\s+serviceextinfo\s*\{/../\}/) {

	                    $check = "1" if ( $_ eq "\thost_name\t\t$host\n" && $check eq "0" );

	                    if ( $_ eq "\tservice_description\t$orig_service\n" && $check eq "1" ) {
	                        $check = "2";
	                        last;
	                    }

	                } else {
	                    $check = "0";
	                }
	            }

	            unless ( $check eq "2" ) {
	                print "Creating New !!!\n\n";
	                $ng->print_log ("SRVEXT: ".$log_header."Entry in serviceextinfo not exists, create NEW!")
	                  if ($opt_verbose & $LOG_SERVICEEXT);

	                # Writing the serviceextinfo for the NAGIOS FE
	                $new_image_tag = $image_tag;
	                $graph_url     = "host=$host&service=$service";

	                # Replace wrappers misc
	                $new_image_tag =~ s/###HOST###/$host/gi;
	                $new_image_tag =~ s/###SERVICE###/$service/gi;
	                $new_image_tag =~ s/###URL###/$graph_url/gi;

					# Replace wrappers, src specific
					if ($image_static =~ m/yes/i) {
						$new_image_tag =~ s/###IMAGESRC###/$image_src/gi;
		            }
		            elsif ($image_static =~ m/no/i && $image_script) {
		            	$new_image_script = $image_script;
		            	$new_image_script =~ s/###URL###/$graph_url/gi;
		            	$new_image_tag =~ s/###IMAGESRC###/$new_image_script/gi;
		            }

					my %args = (
						'host_name' => \$host,
						'service_description' => \$service,
						'file' => \$serviceextinfo,
						'icon_image' => \$new_image_tag,
						'notes_url' => \$notes_url,
					);

					$ng->call_hook (
						ARGSRef => \%args,
						Host => $host,
						Service => $service,
						Type => 'before_serviceext',
					);

	                open( OUT, ">>$serviceextinfo" )
	                  || die "Can not open file '$serviceextinfo'\n";
	                print OUT "\n# ExtInfo for $host, $service\n";
	                print OUT "define serviceextinfo{\n";
	                print OUT "\thost_name\t\t$host\n";
	                print OUT "\ticon_image\t\t$new_image_tag\n";
	                print OUT "\tservice_description\t$orig_service\n";

	                print OUT "\tnotes_url\t\t$notes_url\n" if($notes_url ne '');
	                print OUT "}\n\n";
	                close(OUT);
	            }

			}
			else {
				$ng->print_log("SRVEXT: $serviceextinfo not writable!") if ($opt_verbose & $LOG_PRG_STATUS);
			}

            # Updating the RRD_FILE...

			# Calling the before_rrdupdate hooks
            $ng->call_hook(
            	ARGSRef => \%values,
            	Host => $host,
            	Service => $service,

            	Type => 'before_rrdupdate',
            );

            $dummy="";
            foreach $key ( keys(%values) ) {
                $update = "N" if ( !defined($update) );
                $update .= ":" . $values{$key};
                $template.= ":" if(defined($template));
                $template.=$key;
                $dummy.=$key."=".$values{$key}." ";
            }
            $ng->print_log ("VALUES: ".$log_header." ".$dummy) if($opt_verbose & $LOG_VALUES_OK);

            # Updating... really

            RRDs::update( $file, "--template",$template, $update );
            $ng->print_log("RRD: rrdtool update $file --template=$template $update") if($opt_verbose & $LOG_RRD_STATUS);

            $rrd_err = RRDs::error;

            # Debugging for update RRD's
            if ($rrd_err) {
                $ng->print_log ("RRD: ".$log_header."$rrd_err") if ($opt_verbose & $LOG_RRD_STATUS);
            }
        } else {
    		$ng->print_log("REGEX: No blocks for '$service' found.") if ($opt_verbose & $LOG_DEBUG_REGEX);
        }

    }

  OUT:

    # Clearing another values...
    %values = ();
}


# Subs:
# Helpscreen
sub print_help {
    print "$progname: Helpscreen\n";
    print "\t-v\t\tverbose output with loglevel, e.g. '-v 255'\n";
    print "\t-h\t\thelp screen\n";
    print "\n";
    exit(0);
}

# Daemonize the Script...
sub daemonize {
    chdir '/' or die "Can't chdir to /: $!";

    open STDIN,  '/dev/null'   or die "Can't read /dev/null: $!";
    open STDOUT, '>>/dev/null' or die "Can't write to /dev/null: $!";
    open STDERR, '>>/dev/null' or die "Can't write to /dev/null: $!";

    defined( my $pid = fork )  or die "Can't fork: $!";

    exit if $pid;

    setsid or die "Can't start a new session: $!";

    umask 022;
}

# Signalhandler
sub sig_handler {
    my $signal = shift;
    %SIG = ();

    # Writing the errors into the log.
    $ng->print_log ("PRG: syserror, $?") if ($? && $opt_verbose & $LOG_PRG_STATUS);
    $ng->print_log ("PRG: error, $!") if ($! && $opt_verbose & $LOG_PRG_STATUS);
    $ng->print_log ("PRG: evalerror, $@") if ($@ && $opt_verbose & $LOG_PRG_STATUS);

    if ( $signal eq 'TERM' ) {
    	$ng->daemon_cleanup();
        delete_pipe();
        $ng->print_log ("PRG: $progname Terminated ...") if ($opt_verbose & $LOG_PRG_STATUS);
        exit(0);
    }
    elsif ( $signal eq 'HUP' ) {
        $ng->print_log ("PRG: $progname reload Configuration next loop ...")
          if ($opt_verbose & $LOG_PRG_STATUS);

        load_config;

        $ng->print_log ("PRG: Restarting $progname ...") if ($opt_verbose & $LOG_PRG_STATUS);

        $SIG{$signal} = \&sig_handler;
    }
    elsif ( $signal eq 'INT' ) {
    	$ng->daemon_cleanup();
        delete_pipe();
        $ng->print_log ("PRG: $progname interupted ..") if ($opt_verbose & $LOG_PRG_STATUS);
        exit(0);
    }
    elsif ( $signal ne "__WARN__" ) {
        $ng->print_log ("$progname: Got signal: $signal ...") if ($opt_verbose & $LOG_PRG_STATUS);
        $SIG{$signal} = \&sig_handler;
    }
}

# Load the complete config
sub load_config {
 	# Log settings

	$ng->set_log_file($ng->get_ngraphercfg_value('log_file'));

	if($opt_verbose) {
		$ng->set_log_level(511);
		$opt_verbose=511;
	} else {
		$opt_verbose=$ng->get_log_level();
		$ng->set_log_level($ng->get_ngraphercfg_value('log_level'));
	}

    $ng->print_log ("PRG: Loading main config ...") if ($opt_verbose & $LOG_PRG_STATUS);
    $ng->read_ngrapher_cfg();
    $ng->log_ngraphercfg();

	# write a pidfile and change the user/group
	$ng->daemon_write_pidfile();

	if (!$ng->daemon_change_user()) {
		$ng->print_log ("PRG: Can not change user/group. EXIT.");
		exit (1);
	}

    $ng->print_log ("PRG: Loading ngraph/mngraph configs ...") if ($opt_verbose & $LOG_PRG_STATUS);
    $ng->read_ngrapher_def();
    $ng->read_ngrapher_xml();
    $nagios_config	 = $ng->get_ngraphercfg_value('nagios_config');
    $ng->read_nagios_cfg($nagios_config);
    $ng->save_object();
    return;
}

sub delete_pipe {
	if ( -e $pipe ) {
		unlink($pipe)
			|| die("Can't unlink $pipe: $!");

		$ng->print_log ("PIPE: deleted") if ($opt_verbose & $LOG_PIPE);
	}
}

sub create_pipe {
	unless ( -p $pipe ) {
	    unlink $pipe;
	    POSIX::mkfifo( $pipe, 0666 )
	      || die("Can't mknod $pipe: $!");

	    $ng->print_log ("PIPE: created") if ($opt_verbose & $LOG_PIPE);
	}
}

