/*
 * NASPRO - NASPRO Architecture for Sound Processing
 * DSSI bridge
 *
 * Copyright (C) 2007-2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stddef.h>
#include <string.h>

#include <math.h>

#include <dssi.h>
#include <ladspa.h>

#include <NASPRO/core/lib.h>

#include "descriptor.h"
#include "lv2api.h"

void
_nadssi_descriptor_fill(struct nacore_descriptor *desc,
			const DSSI_Descriptor *ddesc)
{
	const LADSPA_Descriptor *ldesc;
	size_t i;
	char is_synth;

	ldesc = ddesc->LADSPA_Plugin;

	desc->data = (void *)ddesc;

	desc->name = (char *)ldesc->Name;
	desc->creator = (*ldesc->Maker != '\0') ? NULL : (char *)ldesc->Maker;
	desc->rights = strcmp(ldesc->Copyright, "None")
		       ? (char *)ldesc->Copyright : NULL;

	desc->instantiate = _nadssi_lv2api_instantiate;
	desc->connect_port = _nadssi_lv2api_connect_port;
	desc->activate = ldesc->activate != NULL ? _nadssi_lv2api_activate
			 : NULL;
	desc->run = _nadssi_lv2api_run;
	desc->deactivate = ldesc->deactivate != NULL
			   ? _nadssi_lv2api_deactivate : NULL;
	desc->cleanup = _nadssi_lv2api_cleanup;
	desc->extension_data = NULL;

	desc->properties = 0;
	if (LADSPA_IS_REALTIME(ldesc->Properties))
		desc->properties |= NACORE_DESCRIPTOR_HAS_LIVE_DEP;
	if (LADSPA_IS_INPLACE_BROKEN(ldesc->Properties))
		desc->properties |= NACORE_DESCRIPTOR_INPLACE_BROKEN;
	if (LADSPA_IS_HARD_RT_CAPABLE(ldesc->Properties))
		desc->properties |= NACORE_DESCRIPTOR_HARD_RT_CAPABLE;

	desc->classes_1 = 0;
	desc->classes_2 = 0;

	is_synth = (ddesc->run_synth != NULL) ? 1 : 0;
	for (i = 0; i < desc->port_descs_count - is_synth; i++)
	  {
		/* This check is due to Josh Green's broken vocoder plugin */
		if (ldesc->PortNames[i] == NULL)
			desc->port_descs[i].name = "(null)";
		else
			desc->port_descs[i].name = (char *)ldesc->PortNames[i];

		desc->port_descs[i].data = NULL;
		desc->port_descs[i].scale.unit = nacore_scale_unit_none;
		desc->port_descs[i].scale.points = NULL;
		desc->port_descs[i].scale.points_count = 0;

		desc->port_descs[i].properties = 0;
		if (LADSPA_IS_PORT_OUTPUT(ldesc->PortDescriptors[i]))
			desc->port_descs[i].properties |= NACORE_PORT_IS_OUTPUT;
		if (LADSPA_IS_PORT_AUDIO(ldesc->PortDescriptors[i]))
			desc->port_descs[i].properties |= NACORE_PORT_IS_AUDIO;

		if (desc->port_descs[i].properties & NACORE_PORT_IS_AUDIO)
			continue;

		if (!strcmp(ldesc->PortNames[i], "latency")
		    || !strcmp(ldesc->PortNames[i], "_latency"))
			desc->port_descs[i].properties |=
				NACORE_PORT_REPORTS_LATENCY;
		if (LADSPA_IS_HINT_TOGGLED(
				ldesc->PortRangeHints[i].HintDescriptor))
			desc->port_descs[i].properties |= NACORE_PORT_TOGGLED;
		if (LADSPA_IS_HINT_SAMPLE_RATE(
				ldesc->PortRangeHints[i].HintDescriptor))
			desc->port_descs[i].properties |=
				NACORE_PORT_SAMPLE_RATE;
		if (LADSPA_IS_HINT_INTEGER(
				ldesc->PortRangeHints[i].HintDescriptor))
			desc->port_descs[i].properties |= NACORE_PORT_INTEGER;
		if (LADSPA_IS_HINT_LOGARITHMIC(
				ldesc->PortRangeHints[i].HintDescriptor))
			desc->port_descs[i].properties |=
				NACORE_PORT_LOGARITHMIC;

		desc->port_descs[i].scale.properties = 0;
		if (LADSPA_IS_HINT_BOUNDED_BELOW(
				ldesc->PortRangeHints[i].HintDescriptor))
		  {
			desc->port_descs[i].scale.properties |=
				NACORE_SCALE_HAS_MIN;
			desc->port_descs[i].scale.min =
				ldesc->PortRangeHints[i].LowerBound;
		  }
		if (LADSPA_IS_HINT_BOUNDED_ABOVE(
				ldesc->PortRangeHints[i].HintDescriptor))
		  {
			desc->port_descs[i].scale.properties |=
				NACORE_SCALE_HAS_MAX;
			desc->port_descs[i].scale.max =
				ldesc->PortRangeHints[i].UpperBound;
		  }
		if (LADSPA_IS_HINT_HAS_DEFAULT(
				ldesc->PortRangeHints[i].HintDescriptor))
		  {
			desc->port_descs[i].scale.properties |=
				NACORE_SCALE_HAS_DEFAULT;
			if (LADSPA_IS_HINT_DEFAULT_MINIMUM(
				ldesc->PortRangeHints[i].HintDescriptor))
				desc->port_descs[i].scale.defaultv =
					desc->port_descs[i].scale.min;
			else if (LADSPA_IS_HINT_DEFAULT_LOW(
				ldesc->PortRangeHints[i].HintDescriptor))
				desc->port_descs[i].scale.defaultv =
					(desc->port_descs[i].properties & NACORE_PORT_LOGARITHMIC)
					? exp(log(desc->port_descs[i].scale.min) * 0.75 + log(desc->port_descs[i].scale.max) * 0.25)
					: desc->port_descs[i].scale.min * 0.75 + desc->port_descs[i].scale.max * 0.25;
			else if (LADSPA_IS_HINT_DEFAULT_MIDDLE(
				ldesc->PortRangeHints[i].HintDescriptor))
				desc->port_descs[i].scale.defaultv =
					(desc->port_descs[i].properties & NACORE_PORT_LOGARITHMIC)
					? exp(log(desc->port_descs[i].scale.min) * 0.5 + log(desc->port_descs[i].scale.max) * 0.5)
					: desc->port_descs[i].scale.min * 0.5 + desc->port_descs[i].scale.max * 0.5;
			else if (LADSPA_IS_HINT_DEFAULT_HIGH(
				ldesc->PortRangeHints[i].HintDescriptor))
				desc->port_descs[i].scale.defaultv =
					(desc->port_descs[i].properties & NACORE_PORT_LOGARITHMIC)
					? exp(log(desc->port_descs[i].scale.min) * 0.25 + log(desc->port_descs[i].scale.max) * 0.75)
					: desc->port_descs[i].scale.min * 0.25 + desc->port_descs[i].scale.max * 0.75;
			else if (LADSPA_IS_HINT_DEFAULT_MAXIMUM(
				ldesc->PortRangeHints[i].HintDescriptor))
				desc->port_descs[i].scale.defaultv =
					desc->port_descs[i].scale.max;
			else if (LADSPA_IS_HINT_DEFAULT_0(
				ldesc->PortRangeHints[i].HintDescriptor))
				desc->port_descs[i].scale.defaultv = 0.0;
			else if (LADSPA_IS_HINT_DEFAULT_1(
				ldesc->PortRangeHints[i].HintDescriptor))
				desc->port_descs[i].scale.defaultv = 1.0;
			else if (LADSPA_IS_HINT_DEFAULT_100(
				ldesc->PortRangeHints[i].HintDescriptor))
				desc->port_descs[i].scale.defaultv = 100.0;
			else /* LADSPA_IS_HINT_DEFAULT_440 */
			  {
				desc->port_descs[i].scale.defaultv = 440.0;

				if ((desc->port_descs[i].scale.properties &
				     NACORE_SCALE_HAS_MIN) &&
				    (desc->port_descs[i].scale.properties &
				     NACORE_SCALE_HAS_MAX))
					if ((desc->port_descs[i].scale.min
					     > 440.0)
					    || (desc->port_descs[i].scale.max
						< 440.0))
						desc->port_descs[i].scale.properties &= ~NACORE_SCALE_HAS_DEFAULT;
			  }
		  }
	  }

	if (ddesc->run_synth != NULL)
	  {
		desc->port_descs[i].name = "MIDI in";

		desc->port_descs[i].data = NULL;
		desc->port_descs[i].scale.unit = nacore_scale_unit_none;
		desc->port_descs[i].scale.points = NULL;
		desc->port_descs[i].scale.points_count = 0;

		desc->port_descs[i].properties = NACORE_PORT_IS_MIDI;
	  }
}
