/*
 * NASPRO - NASPRO Architecture for Sound Processing
 * LADSPA bridge
 *
 * Copyright (C) 2007-2010 Stefano D'Angelo <zanga.mail@gmail.com>
 *
 * See the COPYING file for license conditions.
 */

#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>

#include <locale.h>

#include <ladspa.h>

#include <librdf.h>

#include <NASPRO/core/lib.h>

#include "lrdf.h"

#define LADSPA_PREFIX "http://ladspa.org/ontology#"
#define RDF_PREFIX "http://www.w3.org/1999/02/22-rdf-syntax-ns#"

static librdf_world *world = NULL;
static librdf_parser *parser = NULL;
static librdf_storage *storage = NULL;
static librdf_model *model = NULL;

static void
parse_file(const char *file, const char *basename, void *data)
{
	librdf_uri *uri;

	uri = librdf_new_uri_from_filename(world, file);
	if (uri == NULL)
		return;

	librdf_parser_parse_into_model(parser, uri, NULL, model);

	librdf_free_uri(uri);
}

static char
rdf_filter(const char *file)
{
	size_t file_len;

	file_len = strlen(file);

	if (file_len <= 4)
		return 0;
	if (!strcmp(file + file_len - 4, ".rdf"))
		return 1;

	if (file_len <= 5)
		return 0;
	
	return !strcmp(file + file_len - 5, ".rdfs");
}

void
_naladspa_lrdf_load_all()
{
	char *lrdf_path;
	char *prev_locale;

	world = librdf_new_world();
	if (world == NULL)
		goto world_err;

	parser = librdf_new_parser(world, NULL, NULL, NULL);
	if (parser == NULL)
		goto parser_err;

	storage = librdf_new_storage(world, "hashes", NULL,
				     "hash-type='memory'");
	if (storage == NULL)
		goto storage_err;

	model = librdf_new_model(world, storage, NULL);
	if (model == NULL)
		goto model_err;

	prev_locale = setlocale(LC_NUMERIC, NULL);
	setlocale(LC_NUMERIC, "C");

	lrdf_path = nacore_env_get_var("LADSPA_RDF_PATH");
	if (NACORE_STRING_IS_NULL_OR_EMPTY(lrdf_path))
	  {
#ifdef __APPLE__
		nacore_path_home_for_each("Library/Audio/Plug-Ins/LADSPA/rdf"
					  ":.ladspa/rdf", parse_file,
					  rdf_filter, NULL);
		nacore_path_for_each("/Library/Audio/Plug-Ins/LADSPA/rdf"
				     ":/usr/local/share/ladspa/rdf"
				     ":/usr/share/ladspa/rdf", parse_file,
				     rdf_filter, NULL);
#elif defined (__HAIKU__)
		/* FIXME: find_directory() should be used */
		nacore_path_home_for_each("config/add-ons/ladspa/rdf",
					  parse_file, rdf_filter, NULL);
		nacore_path_for_each("/boot/common/add-ons/ladspa/rdf",
				     parse_file, rdf_filter, NULL);
#elif defined (__SYLLABLE__)
		nacore_path_home_for_each("extensions/ladspa/rdf",
					  parse_file, rdf_filter, NULL);
		nacore_path_for_each("/system/extensions/ladspa/rdf",
				     parse_file, rdf_filter, NULL);
#else
		nacore_path_home_for_each(".ladspa/rdf", parse_file, rdf_filter,
					  NULL);
		nacore_path_for_each("/usr/local/share/ladspa/rdf"
				     ":/usr/share/ladspa/rdf",
				     parse_file, rdf_filter, NULL);
#endif
	  }
	else
		nacore_path_for_each(lrdf_path, parse_file, rdf_filter, NULL);

	if (lrdf_path != NULL)
		nacore_env_free_var_value(lrdf_path);

	setlocale(LC_NUMERIC, prev_locale);

	return;

model_err:
	librdf_free_storage(storage);
storage_err:
	librdf_free_parser(parser);
parser_err:
	librdf_free_world(world);
world_err:
	return;
}

void
_naladspa_lrdf_unload_all()
{
	if (model == NULL)
		return;

	librdf_free_model(model);
	model = NULL;
	librdf_free_storage(storage);
	storage = NULL;
	librdf_free_parser(parser);
	parser = NULL;
	librdf_free_world(world);
	world = NULL;
}

static char
model_contains_statement_with_subject_node(librdf_node *subject_node,
					   const char *predicate,
					   const char *object)
{
	librdf_node *predicate_node, *object_node;
	librdf_statement *statement;
	char ret;

	predicate_node = librdf_new_node_from_uri_string(world,
				(const unsigned char *)predicate);
	if (predicate_node == NULL)
	  {
		librdf_free_node(subject_node);
		return 0;
	  }

	object_node = librdf_new_node_from_uri_string(world,
				(const unsigned char *)object);
	if (object_node == NULL)
	  {
		librdf_free_node(predicate_node);
		librdf_free_node(subject_node);
		return 0;
	  }

	statement = librdf_new_statement_from_nodes(world, subject_node,
			predicate_node, object_node);
	if (statement == NULL)
	  {
		librdf_free_node(object_node);
		librdf_free_node(predicate_node);
		librdf_free_node(subject_node);
		return 0;
	  }

	ret = librdf_model_contains_statement(model, statement) != 0;

	librdf_free_statement(statement);

	return ret;	
}

static char
model_contains_statement(const char *subject, const char *predicate,
			 const char *object)
{
	librdf_node *subject_node;

	subject_node = librdf_new_node_from_uri_string(world,
				(const unsigned char*)subject);
	if (subject_node == NULL)
		return 0;

	return model_contains_statement_with_subject_node(subject_node,
			predicate, object);
}

static librdf_stream *
model_find_statements_with_subject_node(librdf_node *subject_node,
					const char *predicate,
					const char *object)
{
	librdf_node *predicate_node, *object_node;
	librdf_statement *statement;
	librdf_stream *ret;
	  
	predicate_node = NULL;
	if (predicate != NULL)
	  {
		predicate_node = librdf_new_node_from_uri_string(world,
					(const unsigned char*)predicate);
		if (predicate_node == NULL)
		  {
			if (subject_node != NULL)
				librdf_free_node(subject_node);
			return NULL;
		  }
	  }
	  
	object_node = NULL;
	if (object_node != NULL)
	  {
		object_node = librdf_new_node_from_uri_string(world,
					(const unsigned char*)object);
		if (object_node == NULL)
		  {
			if (predicate_node != NULL)
				librdf_free_node(predicate_node);
			if (subject_node != NULL)
				librdf_free_node(subject_node);
			return NULL;
		  }
	  }

	statement = librdf_new_statement(world);
	if (statement == NULL)
	  {
		if (object_node != NULL)
			librdf_free_node(object_node);
		if (predicate_node != NULL)
			librdf_free_node(predicate_node);
		if (subject_node != NULL)
			librdf_free_node(subject_node);
		return NULL;
	  }

	if (subject_node != NULL)
		librdf_statement_set_subject(statement, subject_node);
	if (predicate_node != NULL)
		librdf_statement_set_predicate(statement, predicate_node);
	if (object_node != NULL)
		librdf_statement_set_object(statement, object_node);

	ret = librdf_model_find_statements(model, statement);

	librdf_free_statement(statement);

	return ret;
}

static librdf_stream *
model_find_statements(const char *subject, const char *predicate,
		      const char *object)
{
	librdf_node *subject_node;

	subject_node = NULL;
	if (subject != NULL)
	  {
		subject_node = librdf_new_node_from_uri_string(world,
					(const unsigned char*)subject);
		if (subject_node == NULL)
			return NULL;
	  }		

	return model_find_statements_with_subject_node(subject_node,
			predicate, object);
}

#define X_MATCH(string, code) \
	if (model_contains_statement(subject, RDF_PREFIX "type", \
				     LADSPA_PREFIX string)) \
	  { \
		code \
	  }

#define S_MATCH_1(string, na_class) \
	X_MATCH(string, classes_1 |= na_class;)

#define C_MATCH_1(string, na_class, na_out_class) \
	X_MATCH(string, \
		classes_1 |= na_class; \
		classes_1 &= ~(na_out_class);)

#define S_MATCH_2(string, na_class) \
	X_MATCH(string, classes_2 |= na_class;)

#define C_MATCH_2(string, na_class, na_out_class) \
	X_MATCH(string, \
		classes_2 |= na_class; \
		classes_2 &= ~(na_out_class);)

void
_naladspa_lrdf_get_classes(struct nacore_descriptor *desc)
{
	LADSPA_Descriptor *ldesc;
	char subject[strlen(LADSPA_PREFIX) + 9];
	uint32_t classes_1 = 0, classes_2 = 0;

	if (model == NULL)
		return;

	ldesc = (LADSPA_Descriptor *)desc->data;

	sprintf(subject, "%s%lu", LADSPA_PREFIX, ldesc->UniqueID);

	S_MATCH_1("UtilityPlugin",	NACORE_DESCRIPTOR_CLASS_UTILITY);
	S_MATCH_1("GeneratorPlugin",	NACORE_DESCRIPTOR_CLASS_GENERATOR);
	S_MATCH_1("SimulatorPlugin",	NACORE_DESCRIPTOR_CLASS_SIMULATOR);
	S_MATCH_1("DelayPlugin",	NACORE_DESCRIPTOR_CLASS_DELAY);
	S_MATCH_1("ModulatorPlugin",	NACORE_DESCRIPTOR_CLASS_MODULATOR);
	S_MATCH_1("FrequencyPlugin",	NACORE_DESCRIPTOR_CLASS_SPECTRAL);
	S_MATCH_1("FilterPlugin",	NACORE_DESCRIPTOR_CLASS_FILTER);
	S_MATCH_2("AmplitudePlugin",	NACORE_DESCRIPTOR_CLASS_DYNAMICS);
	S_MATCH_1("AmplifierPlugin",	NACORE_DESCRIPTOR_CLASS_AMPLIFIER);
	S_MATCH_1("DistortionPlugin",	NACORE_DESCRIPTOR_CLASS_DISTORTION);
	S_MATCH_1("ModulatorPlugin",	NACORE_DESCRIPTOR_CLASS_MODULATOR);
	S_MATCH_2("DynamicsPlugin",	NACORE_DESCRIPTOR_CLASS_DYNAMICS);
	C_MATCH_1("OscillatorPlugin",	NACORE_DESCRIPTOR_CLASS_OSCILLATOR,
					NACORE_DESCRIPTOR_CLASS_GENERATOR);
	C_MATCH_1("PhaserPlugin",	NACORE_DESCRIPTOR_CLASS_PHASER,
					NACORE_DESCRIPTOR_CLASS_MODULATOR);
	C_MATCH_1("FlangerPlugin",	NACORE_DESCRIPTOR_CLASS_FLANGER,
					NACORE_DESCRIPTOR_CLASS_MODULATOR);
	C_MATCH_1("ChorusPlugin",	NACORE_DESCRIPTOR_CLASS_CHORUS,
					NACORE_DESCRIPTOR_CLASS_MODULATOR);
	C_MATCH_1("ReverbPlugin",	NACORE_DESCRIPTOR_CLASS_REVERB,
					NACORE_DESCRIPTOR_CLASS_SIMULATOR
					| NACORE_DESCRIPTOR_CLASS_DELAY);
	C_MATCH_1("FrequencyMeterPlugin", NACORE_DESCRIPTOR_CLASS_SPECTRAL
					  | NACORE_DESCRIPTOR_CLASS_ANALYSER,
					  NACORE_DESCRIPTOR_CLASS_UTILITY);
	C_MATCH_1("LowpassPlugin",	NACORE_DESCRIPTOR_CLASS_LOWPASS,
					NACORE_DESCRIPTOR_CLASS_FILTER);
	C_MATCH_1("HighpassPlugin",	NACORE_DESCRIPTOR_CLASS_HIGHPASS,
					NACORE_DESCRIPTOR_CLASS_FILTER);
	C_MATCH_1("BandpassPlugin",	NACORE_DESCRIPTOR_CLASS_BANDPASS,
					NACORE_DESCRIPTOR_CLASS_FILTER);
	C_MATCH_1("CombPlugin",		NACORE_DESCRIPTOR_CLASS_COMB,
					NACORE_DESCRIPTOR_CLASS_FILTER);
	C_MATCH_1("AllpassPlugin",	NACORE_DESCRIPTOR_CLASS_ALLPASS,
					NACORE_DESCRIPTOR_CLASS_FILTER);
	C_MATCH_1("EQPlugin",		NACORE_DESCRIPTOR_CLASS_EQ,
					NACORE_DESCRIPTOR_CLASS_FILTER);
	C_MATCH_1("ParaEQPlugin",	NACORE_DESCRIPTOR_CLASS_PARAEQ,
					NACORE_DESCRIPTOR_CLASS_FILTER
					| NACORE_DESCRIPTOR_CLASS_EQ);
	C_MATCH_1("MultiEQPlugin",	NACORE_DESCRIPTOR_CLASS_MULTIEQ,
					NACORE_DESCRIPTOR_CLASS_FILTER
					| NACORE_DESCRIPTOR_CLASS_EQ);
	C_MATCH_1("PitchPlugin",	NACORE_DESCRIPTOR_CLASS_PITCH,
					NACORE_DESCRIPTOR_CLASS_SPECTRAL);
	C_MATCH_1("WaveshaperPlugin",	NACORE_DESCRIPTOR_CLASS_WAVESHAPER,
					NACORE_DESCRIPTOR_CLASS_DISTORTION);
	C_MATCH_2("CompressorPlugin",	NACORE_DESCRIPTOR_CLASS_COMPRESSOR,
					NACORE_DESCRIPTOR_CLASS_DYNAMICS);
	C_MATCH_2("ExpanderPlugin",	NACORE_DESCRIPTOR_CLASS_EXPANDER,
					NACORE_DESCRIPTOR_CLASS_DYNAMICS);
	C_MATCH_2("LimiterPlugin",	NACORE_DESCRIPTOR_CLASS_LIMITER,
					NACORE_DESCRIPTOR_CLASS_DYNAMICS);
	C_MATCH_2("GatePlugin",		NACORE_DESCRIPTOR_CLASS_GATE,
					NACORE_DESCRIPTOR_CLASS_DYNAMICS);

	desc->classes_1 = classes_1;
	desc->classes_2 = classes_2;
}

void
_naladspa_lrdf_get_scale_defaults(struct nacore_descriptor *desc,
				  struct nacore_port_descriptor *port_descs)
{
	LADSPA_Descriptor *ldesc;
	librdf_node *setting_node;
	librdf_node *setting_node2;
	librdf_node *default_node;
	librdf_node *port_value_node;
	librdf_node *port_value_node2;
	librdf_node *port_value_node3;
	librdf_stream *stream;
	librdf_stream *stream2;
	char subject[strlen(LADSPA_PREFIX) + 9];
	float value;
	unsigned long index;
	char *prev_locale;

	if (model == NULL)
		return;

	prev_locale = setlocale(LC_NUMERIC, NULL);
	setlocale(LC_NUMERIC, "C");

	ldesc = (LADSPA_Descriptor *)desc->data;

	sprintf(subject, "%s%lu", LADSPA_PREFIX, ldesc->UniqueID);

	stream = model_find_statements(subject, LADSPA_PREFIX "hasSetting",
				       NULL);
	if (stream == NULL)
		goto end;

	default_node = NULL;
	while (!librdf_stream_end(stream))
	  {
		setting_node = librdf_statement_get_object(
					librdf_stream_get_object(stream));
		setting_node2 = librdf_new_node_from_node(setting_node);
			      
		if (model_contains_statement_with_subject_node(setting_node2,
			RDF_PREFIX "type", LADSPA_PREFIX "Default"))
		  {
			default_node = librdf_new_node_from_node(setting_node);
			break;
		  }

		librdf_stream_next(stream);
	  }
	librdf_free_stream(stream);

	if (default_node == NULL)
		goto end;

	stream = model_find_statements_with_subject_node(default_node,
			LADSPA_PREFIX "hasPortValue", NULL);
	if (stream == NULL)
		goto end;

	while (!librdf_stream_end(stream))
	  {
		port_value_node = librdf_statement_get_object(
					librdf_stream_get_object(stream));
		port_value_node2 = librdf_new_node_from_node(port_value_node);
		port_value_node3 = librdf_new_node_from_node(port_value_node);

		stream2 = model_find_statements_with_subject_node(
				port_value_node2, RDF_PREFIX "value", NULL);
		if (stream2 == NULL)
			break;

		if (librdf_stream_end(stream2))
		  {
			librdf_free_stream(stream2);
			break;
		  }
		value = atof((char *)
				librdf_node_get_literal_value(
				  librdf_statement_get_object(
				    librdf_stream_get_object(stream2))));

		librdf_free_stream(stream2);

		stream2 = model_find_statements_with_subject_node(
				port_value_node3, LADSPA_PREFIX "forPort",
				NULL);
		if (stream2 == NULL)
			break;

		if (librdf_stream_end(stream2))
		  {
			librdf_free_stream(stream2);
			break;
		  }

		index = strtoul(
			  strrchr((char *)
			    librdf_uri_as_string(
			      librdf_node_get_uri(
				librdf_statement_get_object(
				  librdf_stream_get_object(stream2)))),
			    '.') + 1,
			  NULL, 10);

		librdf_free_stream(stream2);

		if (index < desc->port_descs_count)
		  {
			port_descs[index].scale.properties |=
				NACORE_SCALE_HAS_DEFAULT;
			port_descs[index].scale.defaultv = value;
		  }

		librdf_stream_next(stream);
	  }
	librdf_free_stream(stream);

end:
	setlocale(LC_NUMERIC, prev_locale);
}

void
_naladspa_lrdf_get_scale_points(struct nacore_descriptor *desc,
				struct nacore_port_descriptor *port,
				size_t index)
{
	LADSPA_Descriptor *ldesc;
	librdf_stream *stream;
	librdf_stream *stream2;
	librdf_stream *stream3;
	librdf_node *scale_node;
	librdf_node *scale_node2;
	librdf_node *point_node;
	librdf_node *point_node2;
	char subject[strlen(LADSPA_PREFIX) + 20];
	char *label, *p;
	size_t len, count, i;
	char *prev_locale;

	if (model == NULL)
		return;

	prev_locale = setlocale(LC_NUMERIC, NULL);
	setlocale(LC_NUMERIC, "C");

	ldesc = (LADSPA_Descriptor *)desc->data;

	sprintf(subject, "%s%lu.%lu", LADSPA_PREFIX, ldesc->UniqueID,
		(unsigned long)index);

	stream = model_find_statements(subject, LADSPA_PREFIX "hasScale", NULL);
	if (stream == NULL)
		goto stream_err;

	if (librdf_stream_end(stream))
		goto stream_end;
	
	scale_node = librdf_statement_get_object(
			librdf_stream_get_object(stream));
	scale_node2 = librdf_new_node_from_node(scale_node);
	if (scale_node2 == NULL)
		goto stream_end;

	stream2 = model_find_statements_with_subject_node(scale_node2,
			LADSPA_PREFIX "hasPoint", NULL);
	if (stream2 == NULL)
	  {
		librdf_free_node(scale_node2);
		goto stream_end;
	  }

	count = 0;
	len = 0;
	while (!librdf_stream_end(stream2))
	  {
		point_node = librdf_statement_get_object(
				librdf_stream_get_object(stream2));
		point_node2 = librdf_new_node_from_node(point_node);

		stream3 = model_find_statements_with_subject_node(
				point_node2, LADSPA_PREFIX "hasLabel", NULL);
		if (stream3 == NULL)
		  {
			librdf_free_stream(stream2);
			goto stream_end;
		  }

		if (librdf_stream_end(stream3))
		  {
			librdf_free_stream(stream3);
			librdf_free_stream(stream2);
			goto stream_end;
		  }

		label = (char *)
			librdf_node_get_literal_value(
			  librdf_statement_get_object(
			    librdf_stream_get_object(stream3)));

		len += strlen(label) + 1;
		count++;
					    
		librdf_free_stream(stream3);

		librdf_stream_next(stream2);
	  }
	librdf_free_stream(stream2);

	if (count == 0)
		goto stream_end;

	port->data = malloc(len);
	if (port->data == NULL)
		goto stream_end;

	port->scale.points = malloc(count * sizeof(struct nacore_scale_point));
	if (port->scale.points == NULL)
	  {
		free(port->data);
		port->data = NULL;
		goto stream_end;
	  }

	scale_node2 = librdf_new_node_from_node(scale_node);
	if (scale_node2 == NULL)
	  {
		free(port->data);
		port->data = NULL;
		free(port->scale.points);
		port->scale.points = NULL;
		goto stream_end;
	  }

	stream2 = model_find_statements_with_subject_node(scale_node2,
			LADSPA_PREFIX "hasPoint", NULL);
	if (stream2 == NULL)
	  {
		free(port->data);
		port->data = NULL;
		free(port->scale.points);
		port->scale.points = NULL;
		goto stream_end;
	  }

	i = 0;
	p = (char *)port->data;
	while (!librdf_stream_end(stream2))
	  {
		point_node = librdf_statement_get_object(
				librdf_stream_get_object(stream2));
		point_node2 = librdf_new_node_from_node(point_node);

		stream3 = model_find_statements_with_subject_node(
				point_node2, LADSPA_PREFIX "hasLabel", NULL);
		if (stream3 == NULL)
		  {
			librdf_free_stream(stream2);
			free(port->data);
			port->data = NULL;
			free(port->scale.points);
			port->scale.points = NULL;
			goto stream_end;
		  }

		if (librdf_stream_end(stream3))
		  {
			librdf_free_stream(stream3);
			librdf_free_stream(stream2);
			free(port->data);
			port->data = NULL;
			free(port->scale.points);
			port->scale.points = NULL;
			goto stream_end;
		  }

		label = (char *)
			librdf_node_get_literal_value(
			  librdf_statement_get_object(
			    librdf_stream_get_object(stream3)));

		port->scale.points[i].label = p;
				
		strcpy(p, label);
		p += strlen(p) + 1;

		librdf_free_stream(stream3);

		point_node2 = librdf_new_node_from_node(point_node);

		stream3 = model_find_statements_with_subject_node(
				point_node2, RDF_PREFIX "value", NULL);
		if (stream3 == NULL)
		  {
			librdf_free_stream(stream2);
			free(port->data);
			port->data = NULL;
			free(port->scale.points);
			port->scale.points = NULL;
			goto stream_end;
		  }

		if (librdf_stream_end(stream3))
		  {
			librdf_free_stream(stream3);
			librdf_free_stream(stream2);
			free(port->data);
			port->data = NULL;
			free(port->scale.points);
			port->scale.points = NULL;
			goto stream_end;
		  }

		port->scale.points[i].value = atof((char *)
			librdf_node_get_literal_value(
			  librdf_statement_get_object(
			    librdf_stream_get_object(stream3))));

		librdf_free_stream(stream3);

		i++;

		librdf_stream_next(stream2);
	  }
	librdf_free_stream(stream2);

	port->scale.points_count = count;

stream_end:
	librdf_free_stream(stream);
stream_err:
	setlocale(LC_NUMERIC, prev_locale);
}

#define MATCH_UNIT(s, u) \
	if (!strcmp(value, LADSPA_PREFIX s)) \
	  { \
		port->scale.unit = u; \
		librdf_free_stream(stream); \
		return; \
	  }

void
_naladspa_lrdf_get_scale_units(struct nacore_descriptor *desc,
			       struct nacore_port_descriptor *port,
			       size_t index)
{
	LADSPA_Descriptor *ldesc;
	librdf_stream *stream;
	char subject[strlen(LADSPA_PREFIX) + 20];
	char *value;

	port->scale.unit = nacore_scale_unit_none;

	if (model == NULL)
		return;

	ldesc = (LADSPA_Descriptor *)desc->data;

	sprintf(subject, "%s%lu.%lu", LADSPA_PREFIX, ldesc->UniqueID,
		(unsigned long)index);

	stream = model_find_statements(subject, LADSPA_PREFIX "hasUnit", NULL);
	if (stream == NULL)
		return;

	if (librdf_stream_end(stream))
	  {
		librdf_free_stream(stream);
		return;
	  }

	value = (char *)librdf_node_get_literal_value(
			  librdf_statement_get_object(
			    librdf_stream_get_object(stream)));

	MATCH_UNIT("dB",		nacore_scale_unit_db);
	MATCH_UNIT("coef",		nacore_scale_unit_coeff);
	MATCH_UNIT("Hz",		nacore_scale_unit_hz);
	MATCH_UNIT("seconds",		nacore_scale_unit_s);
	MATCH_UNIT("milliseconds",	nacore_scale_unit_ms);
	MATCH_UNIT("minutes",		nacore_scale_unit_m);

	librdf_free_stream(stream);
}

void
_naladspa_lrdf_free_data(struct nacore_port_descriptor *port)
{
	if (port->scale.points != NULL)
		free(port->scale.points);
	if (port->data != NULL)
		free(port->data);
}
