/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * Copyright 2007 Sun Microsystems, Inc. All rights reserved.
 */

package org.netbeans.server.uihandler.statistics;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.servlet.jsp.PageContext;
import org.netbeans.server.uihandler.*;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.logging.LogRecord;

/** Computes how much physical memory is used.
 *
 * @author Radim Kubacki
 */
public final class Memory extends Statistics<Map<Long,Set<String>>> {
    
    private final static String TEMP_ID = "temporaryID";
    
    public static List<MemBean> convertToDataset(Map<Long, Set<String>> logs) {
        List<MemBean> result = new ArrayList<MemBean>();
        for (Map.Entry<Long, Set<String>> entry: logs.entrySet()) {
            MemBean mb = new MemBean();
            long mem = entry.getKey();
            String memLabel = "";
            if (mem > 1024 * 1024 * 1024) {
                memLabel = MessageFormat.format("{0,number,###0.###} GB", mem / 1024.0 / 1024.0 / 1024.0);
            } else if (mem > 1024 * 1024) {
                memLabel = MessageFormat.format("{0,number,###0.###} MB", mem / 1024.0 / 1024.0);
            } else {
                memLabel = MessageFormat.format("{0,number,###0} kB", mem / 1024.0);
            }
            mb.setMemory(memLabel);
            entry.getValue().remove(TEMP_ID);
            mb.setCount(entry.getValue().size());
            result.add(mb);
        }
        return result;
    }
    
    public Memory() {
        super("Memory");
    }

    protected Map<Long, Set<String>> newData() {
        return Collections.emptyMap();
    }

    protected Map<Long, Set<String>> process(LogRecord rec) {
        if ("MEMORY".equals(rec.getMessage())) {
            Set<String> s = new HashSet<String>();
            s.add(TEMP_ID);
            return Collections.singletonMap(
                    Long.valueOf((String)rec.getParameters()[0]), s);
        }
        else {
            return Collections.emptyMap();
        }
    }

    protected Map<Long, Set<String>> join(
            Map<Long, Set<String>> one, Map<Long, Set<String>> two) {
        Map<Long, Set<String>> merge = new TreeMap<Long, Set<String>>(one);
        for (Map.Entry<Long, Set<String>> entry: two.entrySet()) {
            if (merge.containsKey(entry.getKey())) {
                merge.get(entry.getKey()).addAll(entry.getValue());
            }
            else {
                merge.put(entry.getKey(), entry.getValue());
            }
        }
        return merge;
    }

    protected Map<Long, Set<String>> 
            finishSessionUpload(String userId, int sessionNumber, boolean initialParse, Map<Long, Set<String>> d) {
        // replace all TEMP_ID occurences with userId
        for (Map.Entry<Long,Set<String>> entry: d.entrySet()) {
            if (entry.getValue().remove(TEMP_ID)) {
                entry.getValue().add(userId);
            }
        }
        return d;
    }

    @Override
    protected void write(Preferences pref, Map<Long, Set<String>> d) throws BackingStoreException {
        for (Map.Entry<Long, Set<String>> l : d.entrySet()) {
            for (String name : l.getValue()) {
                if (name.length() > Preferences.MAX_KEY_LENGTH) {
                    name = name.substring(0, Preferences.MAX_KEY_LENGTH);
                }
                
                pref.putLong(name, l.getKey());
            }
        }
    }

    @Override
    protected Map<Long, Set<String>> read(Preferences pref) throws BackingStoreException {
        Map<Long, Set<String>> map = new HashMap<Long, Set<String>>();
        Set<String> set;
        for (String n : pref.keys()) {
            Long value = pref.getLong(n, -1);
            
            set = map.get(n);
            if (set == null) {
                set = new HashSet<String>();
                map.put(value, set);
            }
        }
        return map;
    }

    @Override
    protected void registerPageContext(PageContext page, String name, Map<Long, Set<String>> data) {
        page.setAttribute(name, convertToDataset(data));
    }

    public static class MemBean {

        private int count;
        private String memory;
        
        public MemBean() {}
        
        public String getMemory() { return memory; }
        public void setMemory(String mem) { memory = mem; }
        
        public int getCount() { return count; }
        public void setCount(int count) { this.count = count; }
        
        public String getSerie() { return ""; }

        @Override
        public String toString() {
            return "Memory: "+memory+" # of users: "+count;
        }
    }
}
    
