/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.cnd.modelimpl.test;

import java.io.File;
import java.io.PrintWriter;
import java.util.*;
import static junit.framework.Assert.fail;
import org.netbeans.api.editor.mimelookup.MimePath;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.classpath.GlobalPathRegistry;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectManager;
import org.netbeans.api.project.ui.OpenProjects;
import org.netbeans.editor.BaseDocument;
import org.netbeans.modules.cnd.api.model.CsmModel;
import org.netbeans.modules.cnd.api.model.CsmProject;
import org.netbeans.modules.cnd.api.model.services.CsmCacheManager;
import org.netbeans.modules.cnd.api.model.util.CsmTracer;
import org.netbeans.modules.cnd.debug.CndTraceFlags;
import org.netbeans.modules.cnd.modelimpl.csm.core.LibraryManager;
import org.netbeans.modules.cnd.modelimpl.csm.core.ProjectBase;
import org.netbeans.modules.cnd.modelimpl.platform.CndIndexer;
import org.netbeans.modules.cnd.modelimpl.trace.TestModelHelper;
import org.netbeans.modules.cnd.modelimpl.trace.TraceModelBase;
import org.netbeans.modules.cnd.test.CndCoreTestUtils;
import org.netbeans.modules.cnd.utils.CndUtils;
import org.netbeans.modules.cnd.utils.MIMENames;
import org.netbeans.modules.parsing.impl.indexing.RepositoryUpdater;
import org.netbeans.spi.editor.mimelookup.MimeDataProvider;
import org.netbeans.spi.java.classpath.support.ClassPathSupport;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.util.Exceptions;
import org.openide.util.Lookup;
import org.openide.util.lookup.Lookups;
import org.openide.util.lookup.ServiceProvider;

/**
 * IMPORTANT NOTE:
 * If This class is not compiled with the notification about not resolved
 * BaseTestCase class => cnd/core tests are not compiled
 *
 * To solve this problem compile or run tests for cnd/core
 */

/**
 * test case for working with projects
 * test has possibility to copy project data into working test dir to prevent changes
 * in source folders when test makes any changes in content of files
 * 
 * @author Vladimir Voskresensky
 */
public abstract class ProjectBasedTestCase extends ModelBasedTestCase {

    private final Map<String, TestModelHelper> projectHelpers = new HashMap<>();
    private final Map<String, List<String>>    sysIncludes = new HashMap<>();
    private final Map<String, List<String>>    usrIncludes = new HashMap<>();
    private final Map<String, List<String>> projectDependencies = new HashMap<>();

    protected PrintWriter outputWriter  = null;
    
    protected PrintWriter logWriter = null;
    
    private final boolean performInWorkDir;
    private File workDirBasedProject = null;
    
    /**
     * Creates a new instance of CompletionBaseTestCase
     */
    public ProjectBasedTestCase(String testName) {
        this(testName, false);
    }
    
    /**
     * if test performs any modifications in data files or create new files
     * => pass performInWorkDir as 'true' to create local copy of project in work dir
     */
    public ProjectBasedTestCase(String testName, boolean performInWorkDir) {
        super(testName);
        this.performInWorkDir = performInWorkDir;
    }

    protected final List<String> getSysIncludes(String prjPath) {
        return getProjectPaths(this.sysIncludes, prjPath);
    }

    protected void setSysIncludes(String prjPath, List<String> sysIncludes) {
        this.sysIncludes.put(prjPath, sysIncludes);
    }

    protected final List<String> getUsrIncludes(String prjPath) {
        return getProjectPaths(this.usrIncludes, prjPath);
    }

    protected void setLibProjectsPaths(String prjPath, List<String> dependentProjects) {
        this.projectDependencies.put(prjPath, dependentProjects);
    }

    protected List<String> getLibProjectsPaths(String prjPath) {
        return getProjectPaths(this.projectDependencies, prjPath);
    }

    protected void setUsrIncludes(String prjPath, List<String> usrIncludes) {
        this.usrIncludes.put(prjPath, usrIncludes);
    }
    
//    protected final void initDocumentSettings() {
//        String methodName = ProjectBasedTestCase.class.getName() + ".getIdentifierAcceptor";
//        Preferences prefs;
//        prefs = MimeLookup.getLookup(MIMENames.CPLUSPLUS_MIME_TYPE).lookup(Preferences.class);
//        prefs.put(EditorPreferencesKeys.IDENTIFIER_ACCEPTOR, methodName);
//        prefs = MimeLookup.getLookup(MIMENames.HEADER_MIME_TYPE).lookup(Preferences.class);
//        prefs.put(EditorPreferencesKeys.IDENTIFIER_ACCEPTOR, methodName);
//        prefs = MimeLookup.getLookup(MIMENames.C_MIME_TYPE).lookup(Preferences.class);
//        prefs.put(EditorPreferencesKeys.IDENTIFIER_ACCEPTOR, methodName);
//    }

//    public static Acceptor getIdentifierAcceptor() {
//        return AcceptorFactory.JAVA_IDENTIFIER;
//    }

    protected boolean needRepository() {
        return false;
    }
    
    @Override
    protected void setUp() throws Exception {
        CndUtils.clearLastAssertion();
        super.setUp();
        System.setProperty("cnd.modelimpl.persistent", needRepository() ? "true" : "false");
        //initDocumentSettings();
        super.clearWorkDir();
        
        outputWriter  = new PrintWriter(getRef());
        logWriter = new PrintWriter(getLog());
        
        log("setUp preparing project.");
        File projectDir;
        if (performInWorkDir) {
            workDirBasedProject = new File(getWorkDir(), "project"); // NOI18N
            // copy data dir
            CndCoreTestUtils.copyDirToWorkDir(getTestCaseDataDir(), workDirBasedProject);
            projectDir = workDirBasedProject; 
        } else {
            projectDir = getTestCaseDataDir();
        }
        File[] changedDirs = changeDefProjectDirBeforeParsingProjectIfNeeded(projectDir);
        FileObject fobjs[] = new FileObject[changedDirs.length];
        for (int i = 0; i < changedDirs.length; i++) {
            File file = changedDirs[i];
            TestModelHelper projectHelper = new TestModelHelper(i==0);
            String prjPath = file.getAbsolutePath();
            projectHelper.initParsedProject(prjPath, getSysIncludes(prjPath), getUsrIncludes(prjPath), getLibProjectsPaths(prjPath));
            projectHelpers.put(prjPath, projectHelper);
            fobjs[i] = FileUtil.toFileObject(file);
        }
        
        if (CndTraceFlags.USE_INDEXING_API) {
            System.setProperty("org.netbeans.modules.parsing.impl.indexing.LogContext$EventType.PATH.minutes", "1");
            System.setProperty("org.netbeans.modules.parsing.impl.indexing.LogContext$EventType.PATH.treshold", "32000");
            RepositoryUpdater.getDefault().start(false);
            Project prj = ProjectManager.getDefault().findProject(FileUtil.toFileObject(projectDir));
            if (prj != null) {
                OpenProjects.getDefault().open(new Project[] {prj}, false);
            }

            ClassPath classPath = ClassPathSupport.createClassPath(fobjs);
            GlobalPathRegistry.getDefault().register("org.netbeans.modules.cnd.makeproject/SOURCES", new ClassPath[]{classPath});
        }

        log("setUp finished preparing project.");
        log("Test "+getName()+  "started");
    }
    
    @ServiceProvider(service=MimeDataProvider.class)
    public static final class MimeDataProviderImpl implements MimeDataProvider {

        private static final Lookup L = Lookups.singleton(new CndIndexer.Factory());

        @Override
        public Lookup getLookup(MimePath mimePath) {
            if (MIMENames.isHeaderOrCppOrC(mimePath.getPath())) {
                return L;
            }
            return null;
        }
    }
    
    /**
     * change the folder if needed from test folder to subfolder
     * i.e. if test folder has several folders: for project and libs =>
     * change dir to subfolders corresponding to projects dirs
     * @param projectDir current project dir
     * @return folders that should be used as project directories
     */
    protected File[] changeDefProjectDirBeforeParsingProjectIfNeeded(File projectDir) {
        return new File[] {projectDir};
    }

    protected void checkDir(File srcDir) {
        assertTrue("Not existing directory" + srcDir, srcDir.exists());
        assertTrue("Not directory" + srcDir, srcDir.isDirectory());
    }

    @Override
    protected void tearDown() throws Exception {
        super.tearDown();
        Iterator<TestModelHelper> iterator = projectHelpers.values().iterator();
        while (iterator.hasNext()) {
            TestModelHelper testModelHelper = iterator.next();
            testModelHelper.shutdown(!iterator.hasNext());
        }
        if (outputWriter != null) {
            outputWriter.flush();
            outputWriter.close();
        }
        if (logWriter != null) {
            logWriter.flush();
            logWriter.close();
        }
        sysIncludes.clear();
        usrIncludes.clear();
        projectDependencies.clear();
        assertTrue("unexpected exception " + CndUtils.getLastAssertion(), CndUtils.getLastAssertion() == null);
    }

    @Override
    protected File getDataFile(String filename) {
        if (performInWorkDir) {
            return new File(workDirBasedProject, filename);
        } else {
            return super.getDataFile(filename);
        }
    }     
    
    protected CsmProject getProject() {
        for (TestModelHelper testModelHelper : projectHelpers.values()) {
            return testModelHelper.getProject();
        }
        assert false : "no initialized projects";
        return null;
    }

    protected CsmProject getProject(String name) {
        CsmProject out = null;
        for (TestModelHelper testModelHelper : projectHelpers.values()) {
            if (name.contentEquals(testModelHelper.getProjectName())) {
                CsmProject project = testModelHelper.getProject();
                assertTrue("two projects with the same name " + name + " " + out + " and " + project, out == null);
                out = project;
                // do not break to allow initialization of all names in TestModelHelpers
            }
        }
        return out;
    }

    protected CsmModel getModel() {
        for (TestModelHelper testModelHelper : projectHelpers.values()) {
            return testModelHelper.getModel();
        }
        assert false : "no initialized projects";
        return null;
    }

    protected void reopenProject(String name, boolean waitParse) {
        for (TestModelHelper testModelHelper : projectHelpers.values()) {
            if (name.contentEquals(testModelHelper.getProjectName())) {
                testModelHelper.reopenProject();
                if (waitParse) {
                    waitAllProjectsParsed();
                }
                return;
            }
        }
    }

    protected void reparseAllProjects() {
        Collection<CsmProject> projects = getModel().projects();
        int expectedNrProjects = projects.size();
        getModel().scheduleReparse(projects);
        try {
            Thread.sleep(2000);
        } catch (InterruptedException ex) {
            Exceptions.printStackTrace(ex);
        }
        projects = getModel().projects();
        assertEquals("projects " + projects, expectedNrProjects, projects.size());
        waitAllProjectsParsed();
    }

    protected void closeProject(String name) {
        for (TestModelHelper testModelHelper : projectHelpers.values()) {
            if (name.contentEquals(testModelHelper.getProjectName())) {
                testModelHelper.resetProject();
                return;
            }
        }
        assertFalse("Project not found or getProject was not called for this name before: " + name, true);
    }
    
    protected int getOffset(File testSourceFile, int lineIndex, int colIndex) throws Exception {
        BaseDocument doc = getBaseDocument(testSourceFile);
        assert doc != null;
        int offset = CndCoreTestUtils.getDocumentOffset(doc, lineIndex, colIndex);  
        return offset;
    }

    private List<String> getProjectPaths(Map<String, List<String>> map, String prjPath) {
        List<String> dependentProjects = map.get(prjPath);
        if (dependentProjects == null) {
            return Collections.emptyList();
        }
        return dependentProjects;
    }

    protected void dumpModel() {
        CsmCacheManager.enter();
        try {        
            for (CsmProject prj : getModel().projects()) {
                new CsmTracer(System.err).dumpModel(prj);
                for (CsmProject lib : prj.getLibraries()) {
                    new CsmTracer(System.err).dumpModel(lib);
                }
            }
            LibraryManager.dumpInfo(new PrintWriter(System.err), true);
        } finally {
            CsmCacheManager.leave();
        }
    }

    protected void waitAllProjectsParsed() {
        sleep(1000);
        Collection<CsmProject> projects;
        projects = getModel().projects();
        for (CsmProject csmProject : projects) {
            TraceModelBase.waitProjectParsed(((ProjectBase) csmProject), true);
        }
    }
    
    protected void checkDifference(File workDir, File goldenDataFile, File output) throws Exception {
        if (!goldenDataFile.exists()) {
            fail("No golden file " + goldenDataFile.getAbsolutePath() + "\n to check with output file " + output.getAbsolutePath());
        }
        if (CndCoreTestUtils.diff(output, goldenDataFile, null)) {
            // copy golden
            File goldenCopyFile = new File(workDir, goldenDataFile.getName() + ".golden");
            CndCoreTestUtils.copyToWorkDir(goldenDataFile, goldenCopyFile); // NOI18N
            StringBuilder buf = new StringBuilder("OUTPUT Difference between diff " + output + " " + goldenCopyFile);
            File diffErrorFile = new File(output.getAbsolutePath() + ".diff");
            CndCoreTestUtils.diff(output, goldenDataFile, diffErrorFile);
            showDiff(diffErrorFile, buf);
            fail(buf.toString());
        }         
    }
}
