/* netToe Version 1.4.2
 *
 * Copyright 2000,2001 Gabriele Giorgetti <ggdev@users.sourceforge.net>
 *           2009-2013 Mats Erik Andersson <meand@users.sourceforge.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <signal.h>

#include "terminal.h"
/* #include "matrices.h" New Code for the nettoe board matrix, not used yet */
#include "misc.h"
#include "network.h"
#include "game.h"

/* Has network partner reported EOF? */
int has_given_eof = 0;


char c11, c12, c13;
char c21, c22, c23;
char c31, c32, c33;

char *C11, *C12, *C13;
char *C21, *C22, *C23;
char *C31, *C32, *C33;

/* Board vector for harder AI-mode. */
char * board[9];

char player1_name[MAX_PNAME_LEN];
char player2_name[MAX_PNAME_LEN];

int game_mode = 0;
int who_starts = 0;
int computer_AI = 0;
int winner = 0;

int player1_status  = 0;
int player2_status  = 0;
int computer_status = 0;

int server_sock;
int client_sock;
int client_sock;

void main_menu (void);
void print_header (int type);
void init_matrix (void);
void init_1_player_game (void);
void init_2_player_game (void);
void show_game (void);
void get_player1_move (void);
void get_player2_move (void);
void get_cpu_AI (void);
void get_cpu_normal_move (void);
void get_cpu_hard_move (void);
void get_cpu_harder_move (void);
int game_check (void);
void network_options (void);
int server_start (void);
void client_start (void);
void init_server_network_game (void);
void init_client_network_game (void);

void net_game_send_server_matrix ( void );
void net_game_recv_server_matrix ( void );
void net_game_send_client_matrix ( void );
void net_game_recv_client_matrix ( void );

void quit_game (void);

/* Reset colour at interruption.
 * Use ECMA-48 character code. */
void reset_color (int sig)
{
  printf ("%c[0m\n", 0x1B);
  exit (EXIT_SUCCESS);
}

int main (int argc, char *argv[])
{

  NO_BEEP   = 0; /* beeps are enabled by default */
                 /* --no-beep disable beeps      */
  NO_COLORS = 0; /* colors are enabled by default */
                 /* --no-colors disable colors    */

  parse_cmd_args(argc, argv);

  /* Remove any elevated access. */
  setegid(getgid());
  setgid(getgid());
  seteuid(getuid());
  setuid(getuid());

#if USE_TERMINFO || USE_TERMCAP
  setup_terminfo();
#endif

  /* Reset colour at normal exit. */
  atexit(nettoe_term_set_default_color);
  /* And at abnormal exits. */
  signal(SIGINT, reset_color);
  signal(SIGKILL, reset_color);
  signal(SIGTERM, reset_color);

  main_menu ();

  quit_game ();

  return EXIT_SUCCESS;
}


void print_header (int type)
{
  nettoe_term_clear ();

  nettoe_term_reset_color();
  printf ("+-----------------------------------------------------------+\n");
  printf ("|");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" netToe %s ", PACKAGE_VERSION);
  nettoe_term_reset_color();
  printf ("|-XOX-OXO-OOO-XXX-|");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" The Net Tic Tac Toe Game ");
  nettoe_term_reset_color();
  printf ("|\n");

  if (type == 0)
     printf ("+-----------------------------------------------------------+\n");
  else
     printf ("+--------------------------+--------------------------------+\n");
     
  nettoe_term_set_default_color();
}

void main_menu (void)
{
  int selection = 0;
  char answer[20];

  print_header (0);
  nettoe_term_reset_color();
  printf ("\n       [ ");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("Main Menu");
  nettoe_term_reset_color();
  printf (" ]\n\n");
  /* Player vs CPU */
  printf ("   (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("1");
  nettoe_term_reset_color();
  printf (") Player vs ");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("C");
  nettoe_term_reset_color();
  printf ("PU\n");
  /* Player vs Player */
  printf ("   (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("2");
  nettoe_term_reset_color();
  printf (") Player vs ");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("P");
  nettoe_term_reset_color();
  printf ("layer\n");
  /* Two players over network */
  printf ("   (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("3");
  nettoe_term_reset_color();
  printf (") Two players over ");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  putchar ('n');
  nettoe_term_reset_color();
  printf ("etwork\n");
  /* Info */
  printf ("   (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("4");
  nettoe_term_reset_color();
  printf (") ");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  putchar ('I');
  nettoe_term_reset_color();
  printf ("nfo\n");
  /* Quit */
  printf ("   (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("5");
  nettoe_term_reset_color();
  printf (") ");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  putchar ('Q');
  nettoe_term_reset_color();
  printf ("uit\n");

  while ( selection == 0 )
  {
    nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
    printf ("\n Selection: ");
    nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);

    fflush (stdin);
    fgets (answer, sizeof(answer), stdin);

    if ( sscanf (answer, " %d", &selection) == 0 )
    {
      char *p = answer;

      while (*p == ' ')
	++p;
      switch (*p)
	{
	  case 'c':
	  case 'C': selection = 1;
		    break;
	  case 'p':
	  case 'P': selection = 2;
		    break;
	  case 'n':
	  case 'N': selection = 3;
		    break;
	  case 'i':
	  case 'I': selection = 4;
		    break;
	  case 'q':
	  case 'Q': selection = 5;
		    break;
	  default:  selection = 0;
	}
    }

    if ( (selection < 1) || (selection > 5) )
    {
      selection = 0;
    }
  }
  nettoe_term_set_default_color ();

  /* Reset scores every time a new main mode has been selected. */
  player1_status  = 0;
  player2_status  = 0;
  computer_status = 0;

  if (selection == 1)
    {
      get_cpu_AI ();

      who_starts = who_start_first ();
      nettoe_term_reset_color();
      printf ("\n Player 1 please type your name: ");
      nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
      get_player_pname(player1_name, MAX_PNAME_LEN);
      nettoe_term_set_default_color ();

      while ((check_pname(player1_name, MAX_PNAME_LEN)) != 0) {
	printf(" Error: name cannot be more than %d chars long.\n", MAX_PNAME_LEN);
	printf("\n Player 1 please type your name: ");
	get_player_pname(player1_name, MAX_PNAME_LEN);
      }
	  
      if (who_starts == 1)
	{
      nettoe_term_reset_color();
	  printf ("\n Who will start first ? (random selection)\n");
	  printf ("\n %s will start first !\n", player1_name);
	  nettoe_term_set_default_color ();
	  sleep ( 2 );
	}
      else if (who_starts == 2)
	{
      nettoe_term_reset_color();
	  printf ("\n Who will start first ? (random selection)\n");
	  printf ("\n Computer will start first !\n");
	  nettoe_term_set_default_color ();
	  sleep ( 2 );
	}
      init_1_player_game ();
    }
  else if (selection == 2)
    {
      print_header (0);
      who_starts = who_start_first ();
      nettoe_term_reset_color();
      printf ("\n Player 1 please type your name: ");
      nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
      get_player_pname(player1_name, MAX_PNAME_LEN);
      nettoe_term_reset_color ();
       
      while ((check_pname(player1_name, MAX_PNAME_LEN)) != 0) {
	 printf(" Error: name cannot be more than %d chars long.\n", MAX_PNAME_LEN);
	 printf("\n Player 1 please type your name: ");
	 get_player_pname(player1_name, MAX_PNAME_LEN);
      }
       
      printf ("\n Player 2 please type your name: ");
      nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
      get_player_pname(player2_name, MAX_PNAME_LEN);
      nettoe_term_reset_color ();

      while ((check_pname(player2_name, MAX_PNAME_LEN)) != 0) {
	 printf(" Error: name cannot be more than %d chars long.\n", MAX_PNAME_LEN);
	 printf("\n Player 2 please type your name: ");
	 get_player_pname(player2_name, MAX_PNAME_LEN);
      }       
       
      if (who_starts == 1)
	{
      nettoe_term_reset_color();
	  printf ("\n Who will start first ? (random selection)\n");
	  printf ("\n %s will start first !\n", player1_name);
	  nettoe_term_set_default_color ();
	  sleep ( 2 );
	}
      else if (who_starts == 2)
	{
      nettoe_term_reset_color();
	  printf ("\n Who will start first ? (random selection)\n");
	  printf ("\n %s will start first !\n", player2_name);
	  nettoe_term_set_default_color ();
	  sleep ( 2 );
	}

      init_2_player_game ();
    }
  else if (selection == 3)
    {
      network_options ();
    }
  else if (selection == 4)
    {
      print_header (0);
      print_infos_screen ();
      main_menu();
    }

  else if (selection == 5)
    {
      quit_game ();
    }
  else
    {
      nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
      printf ("\nError:");
      nettoe_term_reset_color();
      printf (" Incorrect choice.\n");
      nettoe_term_set_default_color ();
      quit_game ();
    }
}

void switch_char_pointers(char ** a, char ** b) {
	char * tmp = *a;

	*a = *b;
	*b = tmp;
}

/* Transform board position to array index.
 * Here 1 <= row,col <= 3. */
int pos_to_ind(char row, char col)
{
  return 3 * (row - 1) + (col - 1);
}

void init_matrix (void)
{
  c11 = ' ', c12 = ' ', c13 = ' ';
  c21 = ' ', c22 = ' ', c23 = ' ';
  c31 = ' ', c32 = ' ', c33 = ' ';
  C11 = &c11, C12 = &c12, C13 = &c13;
  C21 = &c21, C22 = &c22, C23 = &c23;
  C31 = &c31, C32 = &c32, C33 = &c33;

  /* Randomize internal board view point. */
  /* An initial srandom(time(0)) was applied in misc.c. */
  if ( random() > RAND_MAX / 2 ) {
	  /* Vertical mirroring. */
	switch_char_pointers(&C11, &C31);
	switch_char_pointers(&C12, &C32);
	switch_char_pointers(&C13, &C33);
  }
  if ( random() > RAND_MAX / 2 ) {
	  /* Horizontal mirroring. */
	switch_char_pointers(&C11, &C13);
	switch_char_pointers(&C21, &C23);
	switch_char_pointers(&C31, &C33);
  }
  if ( random() > RAND_MAX / 2 ) {
	  /* Diagonal mirroring. */
	switch_char_pointers(&C11, &C33);
	switch_char_pointers(&C12, &C23);
	switch_char_pointers(&C21, &C32);
  }

  /* Set up the board array. Used in AI-mode 3. */
  board[pos_to_ind(1,1)] = C11;
  board[pos_to_ind(1,2)] = C12;
  board[pos_to_ind(1,3)] = C13;
  board[pos_to_ind(2,1)] = C21;
  board[pos_to_ind(2,2)] = C22;
  board[pos_to_ind(2,3)] = C23;
  board[pos_to_ind(3,1)] = C31;
  board[pos_to_ind(3,2)] = C32;
  board[pos_to_ind(3,3)] = C33;
}

void init_1_player_game (void)
{
  char y_n[12];
  winner = 0;
  game_mode = 1;
  init_matrix ();

  if (who_starts == 1)
    {
      do
	{
	  who_starts = 2;
	  show_game ();
	  get_player1_move ();
	  show_game ();
	  winner = game_check ();
	  if (winner == 1)
	    break;
	  else if (winner == 3)
	    break;
	  if (computer_AI == 1)
	    {
	      get_cpu_normal_move ();
	    }
	  else if (computer_AI == 2)
	    {
	      get_cpu_hard_move ();
	    }
	  else if (computer_AI == 3)
	    {
	      get_cpu_harder_move ();
	    }


	  show_game ();
	  winner = game_check ();
	  if (winner == 2)
	    break;
	  else if (winner == 3)
	    break;
	}
      while (winner == 0);
    }
  else if (who_starts == 2)
    {
      do
	{
	  who_starts = 1;
	  show_game ();
	  if (computer_AI == 1)
	    {
	      get_cpu_normal_move ();
	    }
	  else if (computer_AI == 2)
	    {
	      get_cpu_hard_move ();
	    }
	  else if (computer_AI == 3)
	    {
	      get_cpu_harder_move ();
	    }

	  show_game ();
	  winner = game_check ();
	  if (winner == 2)
	    break;
	  else if (winner == 3)
	    break;
	  get_player1_move ();
	  show_game ();
	  winner = game_check ();
	  if (winner == 1)
	    break;
	  else if (winner == 3)
	    break;
	}
      while (winner == 0);
    }
  if (winner == 1)
    {
      nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
      printf ("\n%s wins !\n\n", player1_name);
      nettoe_term_set_default_color ();
      player1_status++;
      nettoe_term_reset_color();
      printf ("Do you want to play again ? [y/n]: ");
      scanf ("%11s", y_n);
      getchar();
      nettoe_term_set_default_color ();
      if ( *y_n == 'y' || *y_n == 'Y' )
	{
	  init_1_player_game ();
	}
      else if ( *y_n == 'n' || *y_n == 'N' )
	{
	  main_menu ();
	}
      else
	{
      nettoe_term_reset_color();
	  printf ("Incorrect answer. I suppose you want to play again.\n");
	  nettoe_term_set_default_color ();
	  sleep ( 2 );
	  init_1_player_game ();
	}
    }
  else if (winner == 2)
    {
      nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
      printf ("\nComputer wins !\n\n");
      nettoe_term_set_default_color ();
      computer_status++;
      nettoe_term_reset_color();
      printf ("Do you want to play again ? [y/n]: ");
      nettoe_term_set_default_color ();
      scanf ("%11s", y_n);
      getchar();
      nettoe_term_set_default_color ();
      if ( *y_n == 'y' || *y_n == 'Y' )
	{
	  init_1_player_game ();
	}
      else if ( *y_n == 'n' || *y_n == 'N' )
	{
	  main_menu ();
	}
      else
	{
      nettoe_term_reset_color();
	  printf ("Incorrect answer. I suppose you want to play again.\n");
	  nettoe_term_set_default_color ();
	  sleep ( 2 );
	  init_1_player_game ();
	}
    }
  else if (winner == 3)
    {
      printf ("\nThere is no winner !\n\n");
      printf ("Do you want to play again ? [y/n]: ");
      scanf ("%11s", y_n);
      getchar();
      if ( *y_n == 'y' || *y_n == 'Y' )
	{
	  init_1_player_game ();
	}
	  else if ( *y_n == 'n' || *y_n == 'N' )
	{
	  main_menu ();
	}
      else
	{
	  printf ("Incorrect answer. I suppose you want to play again.\n");
	  sleep ( 2 );
	  init_1_player_game ();
	}
    }
}

void init_2_player_game (void)
{
  char y_n[12];
  winner = 0;
  game_mode = 2;
  init_matrix ();

  if (who_starts == 1)
    {
      do
	{
	  who_starts = 2;
	  show_game ();
	  get_player1_move ();
	  show_game ();
	  winner = game_check ();
	  if (winner == 1)
	    break;
	  else if (winner == 3)
	    break;
	  get_player2_move ();
	  show_game ();
	  winner = game_check ();
	  if (winner == 2)
	    break;
	  else if (winner == 3)
	    break;
	}
      while (winner == 0);
    }
  else if (who_starts == 2)
    {
      do
	{
	  who_starts = 1;
	  show_game ();
	  get_player2_move ();
	  show_game ();
	  winner = game_check ();
	  if (winner == 2)
	    break;
	  if (winner == 3)
	    break;
	  get_player1_move ();
	  show_game ();
	  winner = game_check ();
	  if (winner == 1)
	    break;
	  if (winner == 3)
	    break;
	}
      while (winner == 0);
    }

  if (winner == 1)
    {
      printf ("\n%s wins !\n\n", player1_name);
      player1_status++;
      printf ("Do you want to play again ? [y/n]: ");
      scanf ("%11s", y_n);
      getchar();
      if ( *y_n == 'y' || *y_n == 'Y' )
	{
	  init_2_player_game ();
	}
      else if ( *y_n == 'n' || *y_n == 'N' )
	{
	  main_menu ();
	}
      else
	{
	  printf ("Incorrect answer. I suppose you want to play again.\n");
	  sleep ( 2 );
	  init_2_player_game ();
	}
    }

  if (winner == 2)
    {
      printf ("\n%s wins !\n\n", player2_name);
      player2_status++;
      printf ("Do you want to play again ? [y/n]: ");
      scanf ("%11s", y_n);
      getchar();
      if ( *y_n == 'y' || *y_n == 'Y' )
	{
	  init_2_player_game ();
	}
      else if ( *y_n == 'n' || *y_n == 'N' )
	{
	  main_menu ();
	}
      else
	{
	  printf ("Incorrect answer. I suppose you want to play again.\n");
	  sleep ( 2 );
	  init_2_player_game ();
	}
    }
  else if (winner == 3)
    {
      printf ("\nThere is no winner !\n\n");
      printf ("Do you want to play again ? [y/n]: ");
      scanf ("%11s", y_n);
      getchar();
      if ( *y_n == 'y' || *y_n == 'Y' )
	{
	  init_2_player_game ();
	}
      else if ( *y_n == 'n' || *y_n == 'N' )
	{
	  main_menu ();
	}
      else
	{
	  printf ("Incorrect answer. I suppose you want to play again.\n");
	  sleep ( 2 );
	  init_1_player_game ();
	}
    }
}

void show_game (void)
{
  print_header (1);

  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("    Current Game");
  nettoe_term_reset_color();
  printf ("           |      ");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("Coordinates scheme\n");
  nettoe_term_reset_color();
  printf ("                           |                        \n");
  printf ("       |   |               |            |    |    \n");
  printf ("     %c | %c | %c             |", c11, c12, c13);
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("         a1");
  nettoe_term_reset_color();
  printf (" |");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" a2");
  nettoe_term_reset_color();
  printf (" |");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" a3 \n");
  nettoe_term_reset_color();
  printf ("    ---|---|---            |        ----|----|---- \n");
  printf ("     %c | %c | %c             |",c21, c22, c23);
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("         b1");
  nettoe_term_reset_color();
  printf (" |");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" b2");
  nettoe_term_reset_color();
  printf (" |");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" b3 \n");
  nettoe_term_reset_color();
  printf ("    ---|---|---            |        ----|----|---- \n");
  printf ("     %c | %c | %c             |", c31, c32, c33);
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("         c1");
  nettoe_term_reset_color();
  printf (" |");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" c2");
  nettoe_term_reset_color();
  printf (" |");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" c3 \n");
  nettoe_term_reset_color();
  printf ("       |   |               |            |    |    \n");
  nettoe_term_set_default_color();

  nettoe_term_reset_color();
  printf ("+--------------------------+--------------------------------+\n");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" Score:");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" %s", player1_name);
  nettoe_term_reset_color();
  printf (" %d", player1_status);
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" %s", (game_mode <= 1) ? "Computer" : player2_name);
  nettoe_term_reset_color();
  printf (" %d\n", (game_mode <= 1) ? computer_status : player2_status);
  printf ("+-----------------------------------------------------------+\n");
  nettoe_term_set_default_color();
}


void get_player1_move (void)
{
  char p1move[4];
  
  nettoe_beep();

  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf("\n%s", player1_name);
  nettoe_term_reset_color();
  printf(", playing X, it is your turn: ");

  scanf ("%3s", p1move);
  if (!strncmp (p1move, "a1", 2) || !strncmp (p1move, "A1", 2))
    {
      if (c11 == ' ')
	{
	  c11 = 'X';
	}
      else if (c11 != ' ')
	{
	  get_player1_move ();
	}
    }
  else if (!strncmp (p1move, "a2", 2) || !strncmp (p1move, "A2", 2))
    {
      if (c12 == ' ')
	{
	  c12 = 'X';
	}
      else if (c12 != ' ')
	{
	  get_player1_move ();
	}
    }
  else if (!strncmp (p1move, "a3", 2) || !strncmp (p1move, "A3", 2))
    {
      if (c13 == ' ')
	{
	  c13 = 'X';
	}
      else if (c13 != ' ')
	{
	  get_player1_move ();
	}
    }
  else if (!strncmp (p1move, "b1", 2) || !strncmp (p1move, "B1", 2))
    {
      if (c21 == ' ')
	{
	  c21 = 'X';
	}
      else if (c21 != ' ')
	{
	  get_player1_move ();
	}
    }
  else if (!strncmp (p1move, "b2", 2) || !strncmp (p1move, "B2", 2))
    {
      if (c22 == ' ')
	{
	  c22 = 'X';
	}
      else if (c22 != ' ')
	{
	  get_player1_move ();
	}
    }
  else if (!strncmp (p1move, "b3", 2) || !strncmp (p1move, "B3", 2))
    {
      if (c23 == ' ')
	{
	  c23 = 'X';
	}
      else if (c23 != ' ')
	{
	  get_player1_move ();
	}
    }
  else if (!strncmp (p1move, "c1", 2) || !strncmp (p1move, "C1", 2))
    {
      if (c31 == ' ')
	{
	  c31 = 'X';
	}
      else if (c31 != ' ')
	{
	  get_player1_move ();
	}
    }
  else if (!strncmp (p1move, "c2", 2) || !strncmp (p1move, "C2", 2))
    {
      if (c32 == ' ')
	{
	  c32 = 'X';
	}
      else if (c32 != ' ')
	{
	  get_player1_move ();
	}
    }
  else if (!strncmp (p1move, "c3", 2) || !strncmp (p1move, "C3", 2))
    {
      if (c33 == ' ')
	{
	  c33 = 'X';
	}
      else if (c33 != ' ')
	{
	  get_player1_move ();
	}
    }
  else
    {
      get_player1_move ();
    }
}

void get_player2_move (void)
{
  char p2move[4];

  nettoe_beep();

  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf("\n%s", player2_name);
  nettoe_term_reset_color();
  printf(", playing O, it is your turn: ");

  scanf ("%3s", p2move);

  if (!strncmp (p2move, "a1", 2) || !strncmp (p2move, "A1", 2))
    {
      if (c11 == ' ')
	{
	  c11 = 'O';
	}
      else if (c11 != ' ')
	{
	  get_player2_move ();
	}
    }
  else if (!strncmp (p2move, "a2", 2) || !strncmp (p2move, "A2", 2))
    {
      if (c12 == ' ')
	{
	  c12 = 'O';
	}
      else if (c12 != ' ')
	{
	  get_player2_move ();
	}
    }
  else if (!strncmp (p2move, "a3", 2) || !strncmp (p2move, "A3", 2))
    {
      if (c13 == ' ')
	{
	  c13 = 'O';
	}
      else if (c13 != ' ')
	{
	  get_player2_move ();
	}
    }
  else if (!strncmp (p2move, "b1", 2) || !strncmp (p2move, "B1", 2))
    {
      if (c21 == ' ')
	{
	  c21 = 'O';
	}
      else if (c21 != ' ')
	{
	  get_player2_move ();
	}
    }
  else if (!strncmp (p2move, "b2", 2) || !strncmp (p2move, "B2", 2))
    {
      if (c22 == ' ')
	{
	  c22 = 'O';
	}
      else if (c22 != ' ')
	{
	  get_player2_move ();
	}
    }
  else if (!strncmp (p2move, "b3", 2) || !strncmp (p2move, "B3", 2))
    {
      if (c23 == ' ')
	{
	  c23 = 'O';
	}
      else if (c23 != ' ')
	{
	  get_player2_move ();
	}
    }
  else if (!strncmp (p2move, "c1", 2) || !strncmp (p2move, "C1", 2))
    {
      if (c31 == ' ')
	{
	  c31 = 'O';
	}
      else if (c31 != ' ')
	{
	  get_player2_move ();
	}
    }
  else if (!strncmp (p2move, "c2", 2) || !strncmp (p2move, "C2", 2))
    {
      if (c32 == ' ')
	{
	  c32 = 'O';
	}
      else if (c32 != ' ')
	{
	  get_player2_move ();
	}
    }
  else if (!strncmp (p2move, "c3", 2) || !strncmp (p2move, "C2", 2))
    {
      if (c33 == ' ')
	{
	  c33 = 'O';
	}
      else if (c33 != ' ')
	{
	  get_player2_move ();
	}
    }
  else
    {
      get_player2_move ();
    }
}

void get_cpu_AI (void)
{
  int selection = 0;
  char answer[20];

  print_header (0);

  nettoe_term_reset_color();
  printf ("\n   [ ");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("Choose a level of difficulty:");
  nettoe_term_reset_color();
  printf (" ]\n\n");
  printf ("    (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("1");
  nettoe_term_reset_color();
  printf (")");
  printf (" Normal \n");
  printf ("    (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("2");
  nettoe_term_reset_color();
  printf (")");
  printf (" Hard   \n");
  printf ("    (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("3");
  nettoe_term_reset_color();
  printf (")");
  printf (" Harder \n");

  while ( selection == 0 )
  {
    nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
    printf ("\n Selection: ");
    nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);

    fgets (answer, sizeof(answer), stdin);

    if ( sscanf (answer, "%d", &selection) == 0 )
    {
      continue;
    }

    if ( (selection < 1) || (selection > 3) )
    {
      selection = 0;
    }
  }

  nettoe_term_set_default_color ();

  if (selection == 1)
    {
      computer_AI = 1;
    }
  else if (selection == 2)
    {
      computer_AI = 2;
    }
  else if (selection == 3)
    {
      computer_AI = 3;
    }
}

void get_cpu_normal_move (void)
{
  printf ("\nComputer is at turn ...\n");
  sleep ( 1 );

  if ((*C11 == 'X') & (*C12 == 'X') & (*C13 == ' '))
    {
      *C13 = 'O';
      return;
    }
  else if ((*C31 == ' ') & (*C32 == 'X') & (*C33 == 'X'))
    {
      *C31 = 'O';
      return;
    }
  else if ((*C11 == 'X') & (*C21 == 'X') & (*C31 == ' '))
    {
      *C31 = 'O';
      return;
    }
  else if ((*C13 == 'X') & (*C23 == 'X') & (*C33 == ' '))
    {
      *C33 = 'O';
      return;
    }
  else if ((*C13 == 'X') & (*C22 == 'X') & (*C31 == ' '))
    {
      *C31 = 'O';
      return;
    }
  else if ((*C21 == 'X') & (*C22 == 'X') & (*C23 == ' '))
    {
      *C23 = 'O';
      return;
    }
  else if ((*C21 == ' ') & (*C22 == 'X') & (*C23 == 'X'))
    {
      *C21 = 'O';
      return;
    }
  else if ((*C11 == 'X') & (*C12 == ' ') & (*C13 == 'X'))
    {
      *C12 = 'O';
      return;
    }
  else if ((*C13 == ' ') & (*C22 == 'O') & (*C31 == 'O'))
    {
      *C13 = 'O';
      return;
    }

  else if ((*C11 == ' ') & (*C22 == 'O') & (*C33 == 'O'))
    {
      *C11 = 'O';
      return;
    }
  else if ((*C12 == ' ') & (*C22 == 'O') & (*C32 == 'O'))
    {
      *C12 = 'O';
      return;
    }
  else if ((*C12 == 'O') & (*C22 == 'O') & (*C32 == ' '))
    {
      *C32 = 'O';
      return;
    }
  else if (*C22 == ' ')
    {
      *C22 = 'O';
      return;
    }
  else if (*C31 == ' ')
    {
      *C31 = 'O';
      return;
    }
  else if (*C12 == ' ')
    {
      *C12 = 'O';
      return;
    }
  else if (*C33 == ' ')
    {
      *C33 = 'O';
      return;
    }
  else if (*C23 == ' ')
    {
      *C23 = 'O';
      return;
    }
  else if (*C21 == ' ')
    {
      *C21 = 'O';
      return;
    }
  else if (*C13 == ' ')
    {
      *C13 = 'O';
      return;
    }
  else if (*C11 == ' ')
    {
      *C11 = 'O';
      return;
    }
  else if (*C32 == ' ')
    {
      *C32 = 'O';
      return;
    }
}

void get_cpu_hard_move (void)
{
  printf ("\nComputer is at turn ...\n");
  sleep ( 1 );

  if ((*C11 == 'X') & (*C12 == 'X') & (*C13 == ' '))
    {
      *C13 = 'O';
      return;
    }
  else if ((*C31 == ' ') & (*C32 == 'X') & (*C33 == 'X'))
    {
      *C31 = 'O';
      return;
    }
  else if ((*C11 == 'X') & (*C21 == 'X') & (*C31 == ' '))
    {
      *C31 = 'O';
      return;
    }
  else if ((*C13 == 'X') & (*C23 == 'X') & (*C33 == ' '))
    {
      *C33 = 'O';
      return;
    }
  else if ((*C13 == 'X') & (*C22 == 'X') & (*C31 == ' '))
    {
      *C31 = 'O';
      return;
    }
  else if ((*C21 == 'X') & (*C22 == 'X') & (*C23 == ' '))
    {
      *C23 = 'O';
      return;
    }
  else if ((*C21 == ' ') & (*C22 == 'X') & (*C23 == 'X'))
    {
      *C21 = 'O';
      return;
    }
  else if ((*C11 == 'X') & (*C12 == ' ') & (*C13 == 'X'))
    {
      *C12 = 'O';
      return;
    }
  else if ((*C12 == 'X') & (*C22 == 'X') & (*C32 == ' '))
    {
      *C32 = 'O';
      return;
    }
  else if ((*C12 == ' ') & (*C22 == 'X') & (*C32 == 'X'))
    {
      *C32 = 'O';
      return;
    }
  else if ((*C11 == 'O') & (*C22 == 'O') & (*C33 == ' '))
    {
      *C33 = 'O';
    }
  else if ((*C31 == 'O') & (*C32 == ' ') & (*C33 == 'O'))
    {
      *C32 = 'O';
    }
  else if ((*C21 == ' ') & (*C22 == 'O') & (*C23 == 'O'))
    {
      *C21 = 'O';
    }
  else if ((*C12 == 'O') & (*C22 == 'O') & (*C32 == ' '))
    {
      *C32 = 'O';
    }
  else if ((*C11 == 'O') & (*C22 == ' ') & (*C33 == 'O'))
    {
    	*C22 = 'O';
    }		
  else if ((*C13 == ' ') & (*C22 == 'O') & (*C31 == 'O'))
    {
      *C13 = 'O';
      return;
    }

  else if ((*C11 == ' ') & (*C22 == 'O') & (*C33 == 'O'))
    {
      *C11 = 'O';
      return;
    }
  else if ((*C12 == ' ') & (*C22 == 'O') & (*C32 == 'O'))
    {
      *C12 = 'O';
      return;
    }
  else if (*C22 == ' ')
    {
      *C22 = 'O';
	  return;
    }
  else if (*C31 == ' ')
    {
      *C31 = 'O';
      return;
    }
  else if (*C12 == ' ')
    {
      *C12 = 'O';
      return;
    }
  else if (*C33 == ' ')
    {
      *C33 = 'O';
      return;
    }
  else if (*C23 == ' ')
    {
      *C23 = 'O';
      return;
    }
  else if (*C21 == ' ')
    {
      *C21 = 'O';
      return;
    }
  else if (*C13 == ' ')
    {
      *C13 = 'O';
      return;
    }
  else if (*C11 == ' ')
    {
      *C11 = 'O';
      return;
    }
  else if (*C32 == ' ')
    {
      *C32 = 'O';
      return;
    }
}

/* The eight possible winning lines expressed as indices.
 *
 * These give the first and last positions, the middle
 * position is directly retreived as the mean of those.
 *
 * A graphical translation is easy:
 *
 *   0 | 1 | 2
 *   ---------
 *   3 | 4 | 5   <====>   board[ 0 1 2 3 4 5 6 7 8 ]
 *   ---------
 *   6 | 7 | 8
 *
 * */
struct { int first, last; }
  winning_line[8] = {
    { 0, 2}, { 3, 5}, { 6, 8}, /* horizontals */
    { 0, 6}, { 1, 7}, { 2, 8}, /* verticals */
	{ 0, 8}, { 2, 6} /* diagonals */
  };

int play_block_or_win (char marker, int first, int last)
{
  int n;
  int marker_count = 0, blank_count = 0;

  for ( n=first; n<=last; n += (last - first)/2 )
  {
    if ( *board[n] == ' ')
    {
      blank_count++;
    }
    else if ( *board[n] == marker )
    {
      marker_count++;
    }
  }

  if ( blank_count == 1 && marker_count == 2 )
  {
    int play_this_pos;

    /* Is winning line or a serious threat. */
    if ( *board[first] == ' ' )
    {
      play_this_pos = first;
	}
	else if ( *board[last] == ' ' )
    {
      play_this_pos = last;
	}
	else
    {
      play_this_pos = (first + last)/2;
    }

	*board[play_this_pos] = 'O';
	/* Report a performed play. */
    return 1;
  }

  /* Nothing done. */
  return 0;
}

void get_cpu_harder_move (void)
{
  int n;

  printf ("\nComputer is at turn ...\n");
  sleep ( 1 );

  /* Locate any winning move. */
  for (n = 0; n < 8; n++)
  {
    if ( play_block_or_win('O',
                  winning_line[n].first, winning_line[n].last) )
      return;
  }

  /* Locate any threat. */
  for (n = 0; n < 8; n++)
  {
    if ( play_block_or_win('X',
                  winning_line[n].first, winning_line[n].last) )
      return;
  }

  /* Find any unoccupied position. */
  if (*C22 == ' ')
    {
      *C22 = 'O';
	  return;
    }
  else if (*C31 == ' ')
    {
      *C31 = 'O';
      return;
    }
  else if (*C12 == ' ')
    {
      *C12 = 'O';
      return;
    }
  else if (*C33 == ' ')
    {
      *C33 = 'O';
      return;
    }
  else if (*C23 == ' ')
    {
      *C23 = 'O';
      return;
    }
  else if (*C21 == ' ')
    {
      *C21 = 'O';
      return;
    }
  else if (*C13 == ' ')
    {
      *C13 = 'O';
      return;
    }
  else if (*C11 == ' ')
    {
      *C11 = 'O';
      return;
    }
  else if (*C32 == ' ')
    {
      *C32 = 'O';
      return;
    }
}

int game_check (void)
{

  if ((c11 == 'X') & (c12 == 'X') & (c13 == 'X'))
    {
      return 1;
    }
  if ((c21 == 'X') & (c22 == 'X') & (c23 == 'X'))
    {
      return 1;
    }
  if ((c31 == 'X') & (c32 == 'X') & (c33 == 'X'))
    {
      return 1;
    }

  if ((c11 == 'X') & (c21 == 'X') & (c31 == 'X'))
    {
      return 1;
    }
  if ((c12 == 'X') & (c22 == 'X') & (c32 == 'X'))
    {
      return 1;
    }
  if ((c13 == 'X') & (c23 == 'X') & (c33 == 'X'))
    {
      return 1;
    }

  if ((c11 == 'X') & (c22 == 'X') & (c33 == 'X'))
    {
      return 1;
    }
  if ((c13 == 'X') & (c22 == 'X') & (c31 == 'X'))
    {
      return 1;
    }

  if ((c11 == 'O') & (c12 == 'O') & (c13 == 'O'))
    {
      return 2;
    }
  if ((c21 == 'O') & (c22 == 'O') & (c23 == 'O'))
    {
      return 2;
    }
  if ((c31 == 'O') & (c32 == 'O') & (c33 == 'O'))
    {
      return 2;
    }

  if ((c11 == 'O') & (c21 == 'O') & (c31 == 'O'))
    {
      return 2;
    }
  if ((c12 == 'O') & (c22 == 'O') & (c32 == 'O'))
    {
      return 2;
    }
  if ((c13 == 'O') & (c23 == 'O') & (c33 == 'O'))
    {
      return 2;
    }

  if ((c11 == 'O') & (c22 == 'O') & (c33 == 'O'))
    {
      return 2;
    }
  if ((c13 == 'O') & (c22 == 'O') & (c31 == 'O'))
    {
      return 2;
    }

  if ((c11 != ' ') && (c12 != ' ') && (c13 != ' ') && (c21 != ' ') && (c22 != ' ') && (c23 != ' ') && (c31 != ' ') && (c32 != ' ') & (c33 != ' '))
    {
      return 3;
    }

  return 0;
}

void network_options (void)
{
  int selection = 0;
  char answer[MAX_PNAME_LEN];

  print_header (0);
  nettoe_term_reset_color();
  printf ("\n   [ ");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("Network game options");
  nettoe_term_reset_color();
  printf (" ]\n\n");
  printf ("    (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("1");
  nettoe_term_reset_color();
  printf (")");
  printf (" Host the game\n");
  printf ("    (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("2");
  nettoe_term_reset_color();
  printf (")");
  printf (" Connect to host\n");
  printf ("    (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("3");
  nettoe_term_reset_color();
  printf (")");
  printf (" Back to main menu\n");

  while ( selection == 0 )
  {
    nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
    printf ("\n Selection: ");
    nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);

    fgets (answer, sizeof (answer), stdin);

    if ( sscanf (answer, "%d", &selection) == 0 )
    {
      continue;
    }

    if ( (selection < 1) || (selection > 3) )
    {
      selection = 0;
    }
  }

  nettoe_term_reset_color ();

  if (selection == 1)
    {
      signal(SIGPIPE, SIG_IGN);
      server_start ();
    }
  else if (selection == 2)
    {
      signal(SIGPIPE, SIG_IGN);
      client_start ();
    }
  else if (selection == 3)
    {
      main_menu ();
    }
}

int server_start (void)
{
  int transf_bytes;
  char local_ip_address[INET6_ADDRSTRLEN];
  char peer_ip_address[INET6_ADDRSTRLEN];

  print_header (0);

  nettoe_term_reset_color();
  printf ("\n   [ ");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("Host the game");
  nettoe_term_reset_color();
  printf (" ]\n\n");
  printf ("    Player name: ");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  get_player_pname(player1_name, MAX_PNAME_LEN);
  local_ip_address[0] = '\0';
  give_local_IP (local_ip_address, sizeof(local_ip_address));
  nettoe_term_reset_color();
  printf ("\n   You should now tell your IP address to any interested\n");
  printf ("   competitor. There might be more than one address, but\n");
  printf ("   \"%s\" should do.\n", local_ip_address);
  printf ("\n   Connected players:\n");
  printf ("   ------------------\n");
  printf ("   -");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" %s", player1_name);
  nettoe_term_reset_color();
  printf (" (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("you");
  nettoe_term_reset_color();
  printf (")\n");

  server_sock = establish_listening_socket (SERVER_PORT_NUMBER, peer_ip_address,
		  			    sizeof(peer_ip_address));
  transf_bytes = write_to_socket (server_sock, player1_name,
			sizeof(player1_name));

  sleep ( 2 );

  transf_bytes = read (server_sock, player2_name, sizeof(player2_name));
  nettoe_term_reset_color();
  printf ("   -");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" %s", player2_name);
  nettoe_term_reset_color();
  printf (" (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("player 2");
  nettoe_term_reset_color();
  printf (")\n");

  printf ("\n   Found 2 players, starting game ...\n");
  sleep ( 3 );

  print_header (0);
  printf ("\n   Who will start first ? (random selection)\n\n");

  who_starts = who_start_first ();

  if (who_starts == 1)
    {
      printf ("   %s will start first !\n", player1_name);
      transf_bytes = write_to_socket (server_sock, player1_name, sizeof(player1_name));
    }

  else if (who_starts == 2)
    {
      printf ("   %s will start first !\n", player2_name);
      transf_bytes = write_to_socket (server_sock, player2_name, sizeof(player2_name));
    }

  printf ("\n   Starting game ...\n");

  nettoe_term_set_default_color ();

  sleep ( 2 );

  if ( ! has_given_eof )
    init_server_network_game ();

  return 0;
}

void client_start (void)
{
  int transf_bytes;
  char buf[MAXDATASIZE];
  char host_ip_number[INET6_ADDRSTRLEN + 1];

  print_header (0);

  nettoe_term_reset_color();
  printf ("\n   [ ");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("Connect to host");
  nettoe_term_reset_color();
  printf (" ]\n\n");
  printf ("    Player name: ");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  get_player_pname(player2_name, MAX_PNAME_LEN);
  nettoe_term_reset_color();
  printf ("    Host to connect to (IP or hostname): ");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  scanf (" %46s", host_ip_number);
  nettoe_term_reset_color();
  printf ("\n   Connecting ...\n");
  printf ("   -------------------------\n");

  client_sock = connect_to_socket (host_ip_number, SERVER_PORT_NUMBER);

  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("   connected !");
  nettoe_term_reset_color();
  printf (" Players are:\n\n");
  transf_bytes = read_from_socket (client_sock, player1_name, sizeof(player1_name));

  printf ("   -");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" %s", player1_name);
  nettoe_term_reset_color();
  printf (" (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("player 1");
  nettoe_term_reset_color();
  printf (")\n");

  printf ("   -");
  nettoe_term_set_color (COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
  printf (" %s", player2_name);
  nettoe_term_reset_color();
  printf (" (");
  nettoe_term_set_color (COLOR_RED, COLOR_BLACK, ATTRIB_BRIGHT);
  printf ("you");
  nettoe_term_reset_color();
  printf (")\n");

  transf_bytes = write_to_socket (client_sock, player2_name, sizeof(player2_name));

  printf ("\n   Found 2 players, starting game ...\n");
  sleep ( 3 );

  print_header (0);
  printf ("\n   Who will start first ? (random selection)\n\n");

  if ( ! has_given_eof )
  {
    transf_bytes = read_from_socket (client_sock, buf, MAX_PNAME_LEN);

	if ( has_given_eof )
	  return;

    printf ("   %s will start first !\n", buf);
    printf ("\n   Starting game ...\n");

    nettoe_term_set_default_color ();

    sleep ( 2 );

    init_client_network_game ();
  }
}

void init_server_network_game (void)
{
  char y_n[12];
  char yes_no[2];
  int transf_bytes;
  char buf[MAXDATASIZE];

  game_mode = 3;

  init_matrix ();

  if (who_starts == 1)
    {	
      transf_bytes = write_to_socket (server_sock, "1", 1);
    }
  else if (who_starts == 2)
    {	
      transf_bytes = write_to_socket (server_sock, "2", 1);
    }

  if (who_starts == 1)
    /* mark a */
    {
      do
	{
	  who_starts = 2;
	  show_game ();
	  get_player1_move ();
	  show_game ();
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  net_game_send_server_matrix ();
	  transf_bytes = write_to_socket (server_sock, "S", 2);

	  winner = game_check ();
	  if ( winner )
	    {
	      show_game ();
	      break;
	    }

	  printf("\nWaiting for ");
	  nettoe_term_set_color(COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
	  printf("%s", player2_name);
	  nettoe_term_reset_color();
	  printf(" to move ...\n");

	  buf[0] = 'n';
	  while ( strncmp (buf, "y", 2) && ! has_given_eof )
	    transf_bytes = read_from_socket (server_sock, buf, 2);

	  net_game_recv_server_matrix ();

	  buf[0] = 'n';
	  while ( strncmp (buf, "C", 2) && ! has_given_eof )
	    transf_bytes = read_from_socket (server_sock, buf, 2);

	  if ( has_given_eof )
		  break;

	  winner = game_check ();
	  if ( winner)
	    {
	      show_game ();
	      break;
	    }
	}
      while (winner == 0 && ! has_given_eof);
    }

  else if (who_starts == 2)
    /* mark b */
    {
      do
	{
	  who_starts = 1;
	  show_game ();

	  printf("\nWaiting for ");
	  nettoe_term_set_color(COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
	  printf("%s", player2_name);
	  nettoe_term_reset_color();
	  printf(" to move ...\n");

	  buf[0] = 'n';
	  while ( strncmp (buf, "y", 2) && ! has_given_eof )
	    transf_bytes = read_from_socket (server_sock, buf, 2);

	  net_game_recv_server_matrix ();

	  buf[0] = 'n';
	  while ( strncmp (buf, "C", 2) && ! has_given_eof )
	    transf_bytes = read_from_socket (server_sock, buf, 2);

	  if ( has_given_eof )
		  break;

	  show_game ();
	  winner = game_check ();

	  if ( winner )
	    {
	      show_game ();
	      break;
	    }

	  get_player1_move ();
	  show_game ();
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  net_game_send_server_matrix ();
	  transf_bytes = write_to_socket (server_sock, "S", 2);

	  winner = game_check ();
	  if ( winner )
	    {
	      show_game ();
	      break;
	    }
	}
      while (winner == 0 && ! has_given_eof);
    }

  if (winner == 1)
    {
      printf ("\n%s wins !\n\n", player1_name);
      player1_status++;
      nettoe_beep();
      printf ("Do you want to play again ? [y/n]: ");
      scanf ("%11s", y_n);
      getchar();
      if ( *y_n == 'y' || *y_n == 'Y' )
	{
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  printf ("Waiting for %s to choose if play again or not ...\n",
		  player2_name);

	  buf[0] = 'n';
	  while ( strncmp (buf, "y", 2) && ! has_given_eof )
	    transf_bytes = read_from_socket (server_sock, buf, 2);

	  transf_bytes = read_from_socket (server_sock, yes_no, 2);

	  if (!strncmp (yes_no, "y", 2) || !strncmp (yes_no, "Y", 2))
	    {
	      printf ("\n%s wants to play again.\n", player2_name);
	      printf ("Starting ...\n");
	      sleep ( 4 );
	      init_server_network_game ();
	    }
	}
      else if ( *y_n == 'n' || *y_n == 'N' )
	{
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  transf_bytes = write_to_socket (server_sock, "n", 2);
	  sleep ( 2 );
	  quit_game ();
	}
      else
	{
	  printf ("Incorrect answer. I suppose you want to play again.\n");
	  sleep ( 2 );
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  printf ("Waiting for %s to choose if to play again or not ...\n",
                 player2_name);

	  buf[0] = 'n';
	  while ( strncmp (buf, "y", 2) && ! has_given_eof)
	    transf_bytes = read_from_socket (server_sock, buf, 2);

	  transf_bytes = read_from_socket (server_sock, yes_no, 2);

	  if (!strncmp (yes_no, "y", 2) || !strncmp (yes_no, "Y", 2))
	    {
	      printf ("\n%s wants to play again.\n", player2_name);
	      printf ("Starting ...\n");
	      sleep ( 4 );
	      init_server_network_game ();
	    }	
	}
    }

  if (winner == 2)
    {
      printf ("\n%s wins !\n\n", player2_name);
      player2_status++;
      nettoe_beep();
      printf ("Do you want to play again ? [y/n]: ");
      scanf ("%11s", y_n);
      getchar();
      if ( *y_n == 'y' || *y_n == 'Y' )
	{
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  printf ("Waiting for %s to choose if to play again or not ...\n",
		  player2_name);

	  buf[0] = 'n';
	  while ( strncmp (buf, "y", 2) && ! has_given_eof )
	    transf_bytes = read_from_socket (server_sock, buf, 2);

	  transf_bytes = read_from_socket (server_sock, yes_no, 2);

	  if (!strncmp (yes_no, "y", 2) || !strncmp (yes_no, "Y", 2))
	    {
	      printf ("\n%s wants to play again.\n", player2_name);
	      printf ("Starting ...\n");
	      sleep ( 4 );
	      init_server_network_game ();
	    }
	}
      else if ( *y_n == 'n' || *y_n == 'N' )
	{
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  transf_bytes = write_to_socket (server_sock, "n", 2);
	  sleep ( 3 );
	  quit_game ();
	}
      else
	{
	  printf ("Incorrect answer. I suppose you want to play again.\n");
	  sleep ( 2 );
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  printf ("Waiting for %s to choose if to play again or not ...\n",
                 player2_name);

	  buf[0] = 'n';
	  while ( strncmp (buf, "y", 2) && ! has_given_eof )
	    transf_bytes = read_from_socket (server_sock, buf, 2);

	  transf_bytes = read_from_socket (server_sock, yes_no, 2);

	  if (!strncmp (yes_no, "y", 2) || !strncmp(yes_no, "Y", 2))
	    {
	      printf ("\n%s wants to play again.\n", player2_name);
	      printf ("Starting ...\n");
	      sleep ( 4 );
	      init_server_network_game ();
	    }	
	}
    }
  else if (winner == 3)
    {
      printf ("\nThere is no winner !\n\n");
      nettoe_beep();
      printf ("Do you want to play again ? [y/n]: ");
      scanf ("%11s", y_n);
      getchar();
      if ( *y_n == 'y' || *y_n == 'Y' )
	{
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  printf ("Waiting for %s to choose if to play again or not ...\n",
                 player2_name);

	  buf[0] = 'n';
	  while ( strncmp (buf, "y", 2) && ! has_given_eof )
	    transf_bytes = read_from_socket (server_sock, buf, 2);

	  transf_bytes = read_from_socket (server_sock, yes_no, 2);

	  if (!strncmp (yes_no, "y", 2) || !strncmp (yes_no, "Y", 2))
	    {
	      printf ("\n%s wants to play again.\n", player2_name);
	      printf ("Starting ...\n");
	      sleep ( 4 );
	      init_server_network_game ();
	    }
	}
      else if ( *y_n == 'n' || *y_n == 'N' )
	{
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  transf_bytes = write_to_socket (server_sock, "n", 2);
	  sleep ( 2 );
	  quit_game ();
	}
      else
	{
	  printf ("Incorrect answer. I suppose you want to play again.\n");
	  sleep ( 2 );
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  transf_bytes = write_to_socket (server_sock, "y", 2);
	  printf ("Waiting for %s to choose if to play again or not ...\n",
                 player2_name);

	  buf[0] = 'n';
	  while ( strncmp (buf, "y", 2) && ! has_given_eof )
	    transf_bytes = read_from_socket (server_sock, buf, 2);

	  transf_bytes = read_from_socket (server_sock, yes_no, 2);
	  if (!strncmp (yes_no, "y", 2) || !strncmp (yes_no, "Y", 2))
	    {
	      printf ("\n%s wants to play again.\n", player2_name);
	      printf ("Starting ...\n");
	      sleep ( 4 );
	      init_server_network_game ();
	    }	
	}
    }

  quit_game ();
}

void init_client_network_game (void)
{
  char y_n[12];
  char yes_no[2];
  int transf_bytes;
  char buf[MAXDATASIZE];

  game_mode = 3;
  init_matrix ();

  transf_bytes = read_from_socket (client_sock, buf, 1);

  if (!strncmp (buf, "1", 1))
    {				
      who_starts = 1;
    }
  else if (!strncmp (buf, "2", 1))
    {				
      who_starts = 2;
    }

  if (who_starts == 1)
    /* mark c */
    {
      do
	{
	  who_starts = 2;
	  show_game ();

	  printf("\nWaiting for ");
	  nettoe_term_set_color(COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
	  printf("%s", player1_name);
	  nettoe_term_reset_color();
	  printf(" to move ...\n");

	  buf[0] = 'n';
	  while ( strncmp (buf, "y", 2) && ! has_given_eof )
	    transf_bytes = read_from_socket (client_sock, buf, 2);

	  net_game_recv_client_matrix ();

	  buf[0] = 'n';
	  while ( strncmp (buf, "S", 2) && ! has_given_eof )
	    transf_bytes = read_from_socket (client_sock, buf, 2);

	  if ( has_given_eof )
		  break;

	  show_game ();
	  winner = game_check ();

	  if ( winner )
	    {
	      show_game ();
	      break;
	    }

	  get_player2_move ();
	  show_game ();
	  transf_bytes = write_to_socket (client_sock, "y", 2);
	  net_game_send_client_matrix ();
	  transf_bytes = write_to_socket (client_sock, "C", 2);

	  winner = game_check ();
	  if ( winner )
	    {
	      show_game ();
	      break;
	    }
	}
      while (winner == 0 && ! has_given_eof);
    }

  else if (who_starts == 2)
    /* mark d */
    {
      do
	{
	  who_starts = 1;
	  show_game ();
	  get_player2_move ();
	  show_game ();

	  transf_bytes = write_to_socket (client_sock, "y", 2);
	  net_game_send_client_matrix ();
	  transf_bytes = write_to_socket (client_sock, "C", 2);

	  winner = game_check ();
	  if ( winner )
	    {
	      show_game ();
	      break;
	    }

	  printf("\nWaiting for ");
	  nettoe_term_set_color(COLOR_BLUE, COLOR_BLACK, ATTRIB_BRIGHT);
	  printf("%s", player1_name);
	  nettoe_term_reset_color();
	  printf(" to move ...\n");

	  buf[0] = 'n';
	  while ( strncmp (buf, "y", 2) && ! has_given_eof )
	    transf_bytes = read_from_socket (client_sock, buf, 2);

	  net_game_recv_client_matrix ();

	  buf[0] = 'n';
	  while ( strncmp (buf, "S", 2) && ! has_given_eof )
	    transf_bytes = read_from_socket (client_sock, buf, 2);

	  if ( has_given_eof )
		  break;

	  winner = game_check ();
	  if ( winner )
	    {
	      show_game ();
	      break;
	    }
	}
      while (winner == 0 && ! has_given_eof);
    }

  if (winner == 1)
    {
      printf ("\n%s wins !\n\n", player1_name);
      player1_status++;
      printf ("Waiting for %s to choose if to play again or not ...\n",
             player1_name);

      buf[0] = 'n';
      while ( strncmp (buf, "y", 2) && ! has_given_eof )
        transf_bytes = read_from_socket (client_sock, buf, 2);

      transf_bytes = read_from_socket (client_sock, yes_no, 2);

      if (!strncmp (yes_no, "y", 2) || !strncmp (yes_no, "Y", 2))
	{
	  printf ("\n%s wants to play again. What about you ?\n",
                 player1_name);
          nettoe_beep();
	  printf ("Do you want to play again ? [y/n]: ");
	  scanf ("%11s", y_n);
	  getchar();

	  if ( *y_n == 'y' || *y_n == 'Y' )
	    {
	      transf_bytes = write_to_socket (client_sock, "y", 2);
	      transf_bytes = write_to_socket (client_sock, "y", 2);
	      printf ("Starting ...\n");
	      sleep ( 4 );
	      init_client_network_game ();
	    }
	  else
	    {
	      transf_bytes = write_to_socket (client_sock, "y", 2);
	      transf_bytes = write_to_socket (client_sock, "n", 2);
	      printf ("Ending session ...\n");
	    }
	}
      else if (!strncmp (yes_no, "n", 2) || !strncmp (yes_no, "N", 2))
	{
	  printf ("\n%s doesn't want to play again. Sorry.\n",
                 player1_name);
	  sleep ( 3 );
	  close (client_sock);
	  close (client_sock);
	  exit (EXIT_SUCCESS);
	}
      else
        {
	  printf ("Incorrect answer. I suppose you want to play again.\n");
	  sleep ( 2 );
	  transf_bytes = write_to_socket (client_sock, "y", 2);
	  transf_bytes = write_to_socket (client_sock, "y", 2);
	  printf ("Starting ...\n");
	  sleep ( 4 );
	  init_client_network_game ();
        }
    }

  if (winner == 2)
    {
      printf ("\n%s wins !\n\n", player2_name);
      player2_status++;
      printf ("Waiting for %s to choose if to play again or not ...\n",
             player1_name);

      buf[0] = 'n';
      while ( strncmp (buf, "y", 2) && ! has_given_eof )
        transf_bytes = read_from_socket (client_sock, buf, 2);

      transf_bytes = read_from_socket (client_sock, yes_no, 2);
      if (!strncmp (yes_no, "y", 2) || ! strncmp (yes_no, "Y", 2))
	{
	  printf ("\n%s wants to play again. What about you ?\n",
                 player1_name);
          nettoe_beep();
	  printf ("Do you want to play again ? [y/n]: ");
	  scanf ("%11s", y_n);
	  getchar();

	  if ( *y_n == 'y' || *y_n == 'Y' )
	    {
	      transf_bytes = write_to_socket (client_sock, "y", 2);
	      transf_bytes = write_to_socket (client_sock, "y", 2);
	      printf ("Starting ...\n");
	      sleep ( 4 );
	      init_client_network_game ();
	    }
	  else
	    {
	      transf_bytes = write_to_socket (client_sock, "y", 2);
	      transf_bytes = write_to_socket (client_sock, "n", 2);
	    }
	}
      else if (!strncmp (yes_no, "n", 2) || !strncmp (yes_no, "N", 2))
	{
	  printf ("\n%s doesn't want to play again. Sorry.\n",
                 player1_name);
	  sleep ( 3 );
	  quit_game ();
	}
      else
	{
	  printf ("Incorrect answer. I suppose you want to play again.\n");
          sleep ( 2 );
	  transf_bytes = write_to_socket (client_sock, "y", 2);
	  transf_bytes = write_to_socket (client_sock, "y", 2);
	  printf ("Starting ...\n");
	  sleep ( 4 );
	  init_client_network_game ();
	}
    }
  else if (winner == 3)
    {
      printf ("\nThere is no winner !\n\n");
      printf ("Waiting for %s to choose if to play again or not ...\n",
             player1_name);

      buf[0] = 'n';
      while ( strncmp (buf, "y", 2) && ! has_given_eof )
        transf_bytes = read_from_socket (client_sock, buf, 2);

      transf_bytes = read_from_socket (client_sock, yes_no, 2);
      if (!strncmp (yes_no, "y", 2) || !strncmp (yes_no, "Y", 2))
	{
	  printf ("\n%s wants to play again. What about you ?\n",
                 player1_name);
          nettoe_beep();
	  printf ("Do you want to play again ? [y/n]: ");
	  scanf ("%11s", y_n);
	  getchar();

	  if ( *y_n == 'y' || *y_n == 'Y' )
	    {
	      transf_bytes = write_to_socket (client_sock, "y", 2);
	      transf_bytes = write_to_socket (client_sock, "y", 2);
	      printf ("Starting ...\n");
	      sleep ( 4 );
	      init_client_network_game ();
	    }
	  else
	    {
	      transf_bytes = write_to_socket (client_sock, "y", 2);
	      transf_bytes = write_to_socket (client_sock, "n", 2);
	    }
	}
      else if (!strncmp (yes_no, "n", 2) || !strncmp (yes_no, "N", 2))
	{
	  printf ("\n%s doesn't want to play again. Sorry.\n", player1_name);
	  sleep ( 3 );
          quit_game ();
	}
      else
	{
	  printf ("Incorrect answer. I suppose you want to play again.\n");
	  sleep ( 2 );
	  transf_bytes = write_to_socket (client_sock, "y", 2);
	  transf_bytes = write_to_socket (client_sock, "y", 2);
	  printf ("Starting ...\n");
	  sleep ( 4 );
	  init_client_network_game ();
	}
      quit_game ();
    }
  if ( has_given_eof )
	  quit_game();
}

void send_matrix (int fd )
{
  int transf_bytes;
  transf_bytes = write_to_socket (fd, &c11, 1);
  transf_bytes = write_to_socket (fd, &c12, 1);
  transf_bytes = write_to_socket (fd, &c13, 1);
  transf_bytes = write_to_socket (fd, &c21, 1);
  transf_bytes = write_to_socket (fd, &c22, 1);
  transf_bytes = write_to_socket (fd, &c23, 1);
  transf_bytes = write_to_socket (fd, &c31, 1);
  transf_bytes = write_to_socket (fd, &c32, 1);
  transf_bytes = write_to_socket (fd, &c33, 1);
}

void net_game_send_server_matrix ( void ) {
	send_matrix(server_sock);
}

void recv_matrix ( int fd )
{
  int transf_bytes;
  transf_bytes = read_from_socket (fd, &c11, 1);
  transf_bytes += read_from_socket (fd, &c12, 1);
  transf_bytes += read_from_socket (fd, &c13, 1);
  transf_bytes += read_from_socket (fd, &c21, 1);
  transf_bytes += read_from_socket (fd, &c22, 1);
  transf_bytes += read_from_socket (fd, &c23, 1);
  transf_bytes += read_from_socket (fd, &c31, 1);
  transf_bytes += read_from_socket (fd, &c32, 1);
  transf_bytes += read_from_socket (fd, &c33, 1);
}

void net_game_recv_server_matrix ( void ) {
	recv_matrix(server_sock);
}

void net_game_send_client_matrix ( void )
{
	send_matrix(client_sock);
}

void net_game_recv_client_matrix ( void )
{
	recv_matrix(client_sock);
}

void quit_game (void)
{

  nettoe_term_reset_color();

  if ( has_given_eof )
	  printf("\n Your opponent has ended the game by cutting the connection !\n");

  printf("\n Goodbye !\n\n");

  printf("netToe Copyright 2000,2001 Gabriele Giorgetti\n");
  printf("       Copyright 2009-2013 Mats Erik Andersson\n");
  printf("netToe project homepage: %s\n\n", HOMEPAGE);

  close (server_sock);
  close (client_sock);

  exit (EXIT_SUCCESS);
}


