/*  Neutrino:  neutrino-wav.c
 *
 *  Copyright (C) 2003 David A Knight <david@ritter.demon.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#ifdef HAVE_GNOME_VFS
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#endif

#include <glib/gstrfuncs.h>

#include <libnjb.h>

#include "neutrino-wav.h"
#include "neutrino-filename.h"

#include "nomad-util.h"

typedef struct {
	gchar chunk_id[ 4 ];
	glong chunk_size;
} __attribute((packed)) ChunkHeader;

typedef struct {
	ChunkHeader head;

	gshort format_tag;
	gushort channels;
	gulong samples_per_sec;
	gulong avg_bytes_per_sec;
	gushort block_align;
	gushort bits_per_sample;
} __attribute((packed)) FormatChunk;


struct NeutrinoWAVDetails {
	gchar *uri;

	FormatChunk *fmt;

	guint64 data_size;

	gchar *artist;
	gchar *album;
	gchar *title;
};

static gboolean neutrino_wav_scan( NeutrinoWAV *wav );
static ChunkHeader *neutrino_wav_get_chunk( void *handle,
					gsize pos,
					gsize offset,
					const gchar *id,
					gboolean read_chunk );

static void neutrino_wav_class_init( NeutrinoWAVClass *klass );
static void neutrino_wav_init( NeutrinoWAV *wav );
static void neutrino_wav_finalize( GObject *object );
static void neutrino_wav_set_prop( GObject *object, guint prop_id, 
				   const GValue *value, GParamSpec *spec );
static void neutrino_wav_get_prop( GObject *object, guint prop_id, 
				   GValue *value, GParamSpec *spec );


NomadTrack *neutrino_wav_track( const gchar *uri )
{
	NeutrinoWAV *wav;
	NomadTrack *track;

	gchar *artist;
	gchar *title;
	gchar *album;
	gchar *genre;
	gchar *length;
	gchar *year;
	gchar *size;
	gchar *tracknum;
	gchar *name;

	wav = neutrino_wav_new( uri );

       	if( ! wav ) {
		/* not really a wav file! */
		return NULL;
	}

	artist = title = album = genre = length = 
		year = size = tracknum = NULL;


	length = neutrino_wav_get_length( wav );

	neutrino_get_info_from_uri( uri,
				    &artist, &album, &title );
	name = NULL;
#ifdef HAVE_GNOME_VFS
	name = gnome_vfs_get_local_path_from_uri( uri );
#endif
	if( ! name ) {
		name = g_strdup( uri );
	}
       
	track = nomad_track_new( artist, title, album, genre,
				    length, year, size, NJB_CODEC_WAV,
				    tracknum, name );

	if( artist ) {
		g_free( artist );
	}
	if( title ) {
		g_free( title );
	}
	if( album ) {
		g_free( album );
	}
	if( genre ) {
		g_free( genre );
	}

	if( length ) {
		g_free( length );
	}
	if( year ) {
		g_free( year );
	} 
	if( size ) {
		g_free( size );
	}
	if( tracknum ) {
		g_free( tracknum );
	} 
	if( name ) {
		g_free( name );
	}

	return track;
}

void neutrino_wav_tag_file( const gchar *uri, NomadTrack *track )
{

}

void *neutrino_wav_write_header( const gchar *uri )
{
#ifndef HAVE_GNOME_VFS
	gboolean ok;
	FILE *handle;

	gchar *dirname;

	dirname = g_path_get_dirname( uri );
	nomad_mkdir_recursive( dirname );
	g_free( dirname );

	handle = fopen( uri, "a+" );
	ok = ( handle != NULL );
	if( ok ) {
		ok = ( fseek( handle, 0, SEEK_SET ) == 0 );
	}
	if( ok ) {
		FormatChunk chunk;
		size_t wrote;

		strncpy( chunk.head.chunk_id, "RIFF", 4 );
		chunk.head.chunk_size = 0;

		/* only want to write out the header */
		wrote = fwrite( &chunk, 1, sizeof( ChunkHeader ), 
				handle );
		ok = ( wrote == sizeof( ChunkHeader ) );
		if( ok ) {
			wrote = fwrite( "WAVE", 1, 4, handle );
			ok = ( wrote == 4 );
		}
		if( ok ) {
			strncpy( chunk.head.chunk_id, "fmt ", 4 );
			chunk.head.chunk_size = sizeof( FormatChunk ) -
				sizeof( ChunkHeader );

			chunk.format_tag = 1; /* PCM */
			chunk.channels = 2;
			chunk.samples_per_sec = 44100;
			chunk.avg_bytes_per_sec = 44100 * 2 * 2;
			chunk.block_align = 4;
			chunk.bits_per_sample = 16;

#if G_BYTE_ORDER == G_BIG_ENDIAN
			chunk.head.chunk_size = GUINT32_TO_LE( chunk.head.chunk_size );
			chunk.format_tag = GINT16_TO_LE( chunk.format_tag );
			chunk.channels = GUINT16_TO_LE( chunk.channels );
			chunk.samples_per_sec = GUINT32_TO_LE( chunk.samples_per_sec );
			chunk.avg_bytes_per_sec = GUINT32_TO_LE( chunk.avg_bytes_per_sec );
			chunk.block_align = GUINT16_TO_LE( chunk.block_align );
			chunk.bits_per_sample = GUINT16_TO_LE( chunk.bits_per_sample );
#endif
			wrote = fwrite( &chunk, 1, 
					sizeof( FormatChunk ),
					handle );
			ok = ( wrote == sizeof( FormatChunk ) );
		}
		if( ok ) {
			strncpy( chunk.head.chunk_id, "data", 4 );
			chunk.head.chunk_size = 0;

			/* only want to write the header again */
			wrote = fwrite( &chunk, 1,
					sizeof( ChunkHeader ),
					handle );
			ok = ( wrote == sizeof( ChunkHeader ) );
		}
	}
	if( ! ok ) {
		if( handle ) {
			fclose( handle );
		}
		handle = NULL;
		unlink( uri );
	}

	return handle;
#else
	GnomeVFSResult result;
	GnomeVFSHandle *handle;

	gchar *dirname;

	dirname = g_path_get_dirname( uri );
	nomad_mkdir_recursive( dirname );
	g_free( dirname );

	handle = NULL;
	result = gnome_vfs_create( &handle, uri,
				   GNOME_VFS_OPEN_WRITE |
				   GNOME_VFS_OPEN_RANDOM,
				   FALSE, 
				   GNOME_VFS_PERM_USER_READ |
				   GNOME_VFS_PERM_USER_WRITE |
				   GNOME_VFS_PERM_GROUP_READ |
				   GNOME_VFS_PERM_OTHER_READ );
	if( result == GNOME_VFS_OK ) {
		FormatChunk chunk;
		GnomeVFSFileSize wrote;

		strncpy( chunk.head.chunk_id, "RIFF", 4 );
		chunk.head.chunk_size = 0;

		/* only want to write out the header */
		result = gnome_vfs_write( handle, &chunk,
					  sizeof( ChunkHeader ),
					  &wrote );
		if( result == GNOME_VFS_OK ) {
			result = gnome_vfs_write( handle, "WAVE",
						  4, &wrote );
		}
		if( result == GNOME_VFS_OK ) {
			strncpy( chunk.head.chunk_id, "fmt ", 4 );
			chunk.head.chunk_size = sizeof( FormatChunk ) -
				sizeof( ChunkHeader );

			chunk.format_tag = 1; /* PCM */
			chunk.channels = 2;
			chunk.samples_per_sec = 44100;
			chunk.avg_bytes_per_sec = 44100 * 2 * 2;
			chunk.block_align = 4;
			chunk.bits_per_sample = 16;

#if G_BYTE_ORDER == G_BIG_ENDIAN
			chunk.head.chunk_size = GUINT32_TO_LE( chunk.head.chunk_size );
			chunk.format_tag = GINT16_TO_LE( chunk.format_tag );
			chunk.channels = GUINT16_TO_LE( chunk.channels );
			chunk.samples_per_sec = GUINT32_TO_LE( chunk.samples_per_sec );
			chunk.avg_bytes_per_sec = GUINT32_TO_LE( chunk.avg_bytes_per_sec );
			chunk.block_align = GUINT16_TO_LE( chunk.block_align );
			chunk.bits_per_sample = GUINT16_TO_LE( chunk.bits_per_sample );
#endif
			result = gnome_vfs_write( handle, &chunk,
						  sizeof( FormatChunk ),
						  &wrote );
		}
		if( result == GNOME_VFS_OK ) {
			strncpy( chunk.head.chunk_id, "data", 4 );
			chunk.head.chunk_size = 0;

			/* only want to write the header again */
			result = gnome_vfs_write( handle, &chunk,
						  sizeof( ChunkHeader ),
						  &wrote );
		}
		if( result != GNOME_VFS_OK ) {
			gnome_vfs_close( handle );
			handle = NULL;
			gnome_vfs_unlink( uri );
			handle = NULL;
		}
	}

	return handle;
#endif
}



NeutrinoWAV *neutrino_wav_new( const gchar *uri )
{
	NeutrinoWAV *wav;
	
	wav = NEUTRINO_WAV( g_object_new( neutrino_wav_get_type(), NULL ) );

	wav->details->uri = g_strdup( uri );

	if( ! neutrino_wav_scan( wav ) ) {
		g_object_unref( G_OBJECT( wav ) );
		wav = NULL;
	}

	return wav;
}

gchar *neutrino_wav_get_length( NeutrinoWAV *wav )
{
	guint64 sec;
	FormatChunk *fmt;
	gchar *ret;

	fmt = wav->details->fmt;

	/* only support PCM */
	if( fmt->format_tag == 1 ) {
		sec = wav->details->data_size;
		sec /= fmt->channels;
		sec /= ( fmt->bits_per_sample / 8 );
		sec /= fmt->samples_per_sec;

		ret = seconds_to_mmss( sec );
	} else {
		ret = NULL;
	}

	return ret;
}



/* static stuff */
static gboolean neutrino_wav_scan( NeutrinoWAV *wav )
{
#ifndef HAVE_GNOME_VFS
	return FALSE;
#else
	GnomeVFSHandle *handle;
	GnomeVFSResult result;

	gboolean ret;
	gboolean iswav;
	gboolean opened;

	ret = iswav = opened = FALSE;

	result = gnome_vfs_open( &handle, wav->details->uri, 
				 GNOME_VFS_OPEN_READ | 
				 GNOME_VFS_OPEN_RANDOM );
	if( result == GNOME_VFS_OK ) {
		ChunkHeader head;
		GnomeVFSFileSize got;
		gchar buff[ 4 ];

		opened = TRUE;

		result = gnome_vfs_read( handle, &head, 
					 sizeof( ChunkHeader ), &got );
		if( result == GNOME_VFS_OK &&
		    ! strncmp( "RIFF", head.chunk_id, 4 ) ) {
			
			result = gnome_vfs_read( handle, buff, 4,
						 &got );
			iswav = ( result == GNOME_VFS_OK &&
				  ! strncmp( "WAVE", buff, 4 ) );
		}
	}
	if( iswav ) {
		FormatChunk *fmt;
		ChunkHeader *data;

		fmt = (FormatChunk*)neutrino_wav_get_chunk( handle,
							    GNOME_VFS_SEEK_CURRENT,
							    0,
							    "fmt ",
							    TRUE );
		if( fmt ) {
			GnomeVFSFileSize offset;
			GnomeVFSSeekPosition pos;

#if G_BYTE_ORDER == G_BIG_ENDIAN
			fmt->format_tag = GINT16_FROM_LE( fmt->format_tag );
			fmt->channels = GUINT16_FROM_LE( fmt->channels );
			fmt->samples_per_sec = GUINT32_FROM_LE( fmt->samples_per_sec );
			fmt->avg_bytes_per_sec = GUINT32_FROM_LE(fmt->avg_bytes_per_sec);
			fmt->block_align = GUINT16_FROM_LE( fmt->block_align );
			fmt->bits_per_sample = GUINT16_FROM_LE( fmt->bits_per_sample );
#endif
			wav->details->fmt = fmt;
			wav->details->data_size = 0;
			offset = 12;  /* RIFFxxxxWAVE */
			pos = GNOME_VFS_SEEK_START;
			ret = TRUE;
			do {
				data = neutrino_wav_get_chunk( handle,
							       pos,
							       offset,
							       "data",
							       FALSE );
				if( data ) {
					pos = GNOME_VFS_SEEK_CURRENT;
					offset = data->chunk_size;
					wav->details->data_size += data->chunk_size;
				}
				g_free( data );
			} while( data != NULL );

		}
	}
	if( opened ) {
		gnome_vfs_close( handle );
	}

	return ret;
#endif
}

static ChunkHeader *neutrino_wav_get_chunk( void *handle,
					    gsize pos,
					    gsize offset,
					    const gchar *id,
					    gboolean read_chunk )
{
	gboolean ok;
	gchar *chunk;
	ChunkHeader temp;
#ifndef HAVE_GNOME_VFS
	FILE *file;
	gsize got;

	chunk = NULL;

	file = (FILE*)handle;
	ok = ( fseek( file, offset, pos ) == 0 );
	while( ok ) {

		got = fread( &temp, 1, sizeof( ChunkHeader ), file );
		ok = ( got == sizeof( ChunkHeader ) );
		if( ok ) {
#if G_BYTE_ORDER == G_BIG_ENDIAN
			temp.chunk_size = GINT32_FROM_LE( temp.chunk_size );
#endif
			if( ! strncmp( id, temp.chunk_id, 4 ) ) {
				break;
			}
			ok = ( fseek( handle, 
					temp.chunk_size, SEEK_CUR ) == 0 );

		}
	}
#else
	GnomeVFSResult result;
	GnomeVFSFileSize got;

	chunk = NULL;

	result = gnome_vfs_seek( handle, pos, offset );
	ok = FALSE;
	while( result == GNOME_VFS_OK ) {

		result = gnome_vfs_read( handle, &temp,
					 sizeof( ChunkHeader ),
					 &got );
		if( result == GNOME_VFS_OK ) {
#if G_BYTE_ORDER == G_BIG_ENDIAN
			temp.chunk_size = GINT32_FROM_LE( temp.chunk_size );
#endif
			if( ! strncmp( id, temp.chunk_id, 4 ) ) {
				ok = TRUE;
				break;
			}
			result = gnome_vfs_seek( handle,
						 GNOME_VFS_SEEK_CURRENT,
						 temp.chunk_size );
		}
	}
#endif
	if( ok ) {
		if( read_chunk ) {
			chunk = (gchar*)g_new0( gchar, 
						      sizeof( ChunkHeader ) +
						      temp.chunk_size );
			memcpy( chunk, &temp, sizeof( ChunkHeader ) );
#ifndef HAVE_GNOME_VFS
			got = fread( chunk + sizeof( ChunkHeader ), 1,
					temp.chunk_size, file );
			ok = ( got == temp.chunk_size );
#else
			result = gnome_vfs_read( handle, 
						 chunk + sizeof( ChunkHeader ),
						 temp.chunk_size,
						 &got );
#endif
		} else {
			chunk = (gchar*)g_new( ChunkHeader, 1 );
			memcpy( chunk, &temp, sizeof( ChunkHeader ) );
		}
	}

	return (ChunkHeader*)chunk;
}

/* G Object stuff */

#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void
neutrino_wav_class_init( NeutrinoWAVClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS( klass );

	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = neutrino_wav_finalize;
	object_class->get_property = neutrino_wav_get_prop;
	object_class->set_property = neutrino_wav_set_prop;
}

static void
neutrino_wav_init( NeutrinoWAV *wav )
{
	wav->details = g_new0( NeutrinoWAVDetails, 1 );
}

static void
neutrino_wav_set_prop( GObject *object, guint prop_id, 
			 const GValue *value, GParamSpec *spec )
{
	NeutrinoWAV *wav;

	wav = NEUTRINO_WAV( object );

	switch( prop_id ) {
	default:
		break;
	}
}

static void
neutrino_wav_get_prop( GObject *object, guint prop_id, 
		       GValue *value, GParamSpec *spec )
{
	NeutrinoWAV *wav;

	wav = NEUTRINO_WAV( object );

	switch( prop_id ) {
	default:
		break;
	}
}

static void
neutrino_wav_finalize( GObject *object )
{
	NeutrinoWAV *wav;

	wav = NEUTRINO_WAV( object );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}

GType neutrino_wav_get_type()
{
	static GType type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( NeutrinoWAVClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)neutrino_wav_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( NeutrinoWAV ),
			0, /* n_preallocs */
			(GInstanceInitFunc)neutrino_wav_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "NeutrinoWAV",
					       &info, 0 );
	}

	return type;
}
