/*  Neutrino:  neutrino-window.c
 *
 *  Copyright (C) 2002 David A Knight <david@ritter.demon.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <time.h>

#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-exec.h>

#include <libgnomeui/gnome-entry.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <glib/grand.h>

#include <gtk/gtk.h>

#include <glade/glade.h>

#include <gconf/gconf-client.h>

#include "neutrino-window.h"
#include "neutrino-window-private.h"

#include "nomad-jukebox.h"

#include "neutrino-jukebox-view.h"

#include "neutrino-file-browser.h"

#include "neutrino-mp3.h"
#include "neutrino-wav.h"
#include "neutrino-wma.h"

#include "neutrino-genre.h"

#include "neutrino-util.h"
#include "nomad-util.h"

enum {
	ARG_0,
	ARG_SHELL
};

enum {
	TARGET_NONE = 0,
        TARGET_URI_LIST,
	TARGET_NEUTRINO_TRACK_LIST
};

static const GtkTargetEntry drag_types[] = {
        { "application/x-neutrino-track-list", GTK_TARGET_SAME_APP, TARGET_NEUTRINO_TRACK_LIST },
	{ "text/uri-list", 0, TARGET_URI_LIST }
};
static const int num_drag_types = sizeof(drag_types) / sizeof(drag_types[ 0 ]);
                                                                                
static const GtkTargetEntry drop_types[] = {
        { "text/uri-list", 0, TARGET_URI_LIST }
};
static const int num_drop_types = sizeof(drop_types) / sizeof(drop_types[ 0 ]);

static const GtkTargetEntry internal_drop_types[] = {
        { "application/x-neutrino-track-list", GTK_TARGET_SAME_APP, TARGET_NEUTRINO_TRACK_LIST },
	{ "GTK_TREE_MODEL_ROW", GTK_TARGET_SAME_APP, TARGET_NEUTRINO_TRACK_LIST }
};
static const int num_internal_drop_types = sizeof(internal_drop_types) / 
sizeof(internal_drop_types[ 0 ]);
                                                                                

void neutrino_window_fm_expanded( GtkTreeView *view,
				  GtkTreeIter *iter,
				  GtkTreePath *path );

void neutrino_window_fm_activated( GtkTreeView *view,
				   GtkTreePath *path,
				   GtkTreeViewColumn *column );

void neutrino_window_tracks_popup_key( GtkWidget *view, gpointer data );
gboolean neutrino_window_tracks_popup( GtkWidget *view,
				       GdkEventButton *event,
				       gpointer data );

static void neutrino_window_class_init( NeutrinoWindowClass *klass );
static void neutrino_window_init( NeutrinoWindow *window );
static void neutrino_window_finalize( GObject *object );
static void neutrino_window_set_prop( GObject *object, guint prop_id, 
				     const GValue *value, GParamSpec *spec );
static void neutrino_window_get_prop( GObject *object, guint prop_id, 
				     GValue *value, GParamSpec *spec );
static void neutrino_window_size_request( GtkWidget *widget, 
					  GtkRequisition *req );
static void neutrino_window_realize( GtkWidget *widget );
static void neutrino_window_show( GtkWidget *widget );


static void neutrino_has_selection( GtkTreeModel *model,
			     GtkTreePath *path,
			     GtkTreeIter *it,
			     gpointer data );

static void neutrino_util_get_selection( GtkTreeModel *model,
				  GtkTreePath *path,
				  GtkTreeIter *iter,
				  gpointer data );


static void neutrino_window_constructed( NeutrinoWindow *window );


static void neutrino_window_build_album_view( NeutrinoWindow *window );
static void neutrino_window_build_artist_view( NeutrinoWindow *window );
static void neutrino_window_build_tracks_view( NeutrinoWindow *window );

static void neutrino_window_build_playqueue_view( NeutrinoWindow *window );

static void neutrino_window_build_file_manager( NeutrinoWindow *window );

/* jukebox signals */
static void neutrino_window_usage_received( NomadJukebox *jukebox,
					    guint64 total, guint64 free, 
					    guint64 used, gpointer data );

static void neutrino_window_scanning_tracks( NomadJukebox *jukebox,
					     gpointer data );
static void neutrino_window_track_add( NomadJukebox *jukebox,
				       guint total_tracks,
				       NomadTrack *track,
				       gpointer data );
static void neutrino_window_scanned_tracks( NomadJukebox *jukebox,
					    guint total,
					    gpointer data );
static void neutrino_window_scanning_playlists( NomadJukebox *jukebox,
					     gpointer data );
static void neutrino_window_playlist_add( NomadJukebox *jukebox,
					  guint total_tracks,
					  const gchar *name,
					  gpointer data );
static void neutrino_window_playlist_remove( NomadJukebox *jukebox,
					     const gchar *name,
					     gpointer data );
static void neutrino_window_playlist_renamed( NomadJukebox *jukebox,
					      const gchar *orig,
					      const gchar *newname,
					      gpointer data );

static void neutrino_window_scanned_playlists( NomadJukebox *jukebox,
					       guint total,
					       gpointer data );
static void neutrino_window_transfer( NomadJukebox *jukebox,
				      const gchar *uri,
				      gboolean upload,
				      guint64 sent,
				      guint64 total,
				      gpointer data );
static void neutrino_window_downloaded_track( NomadJukebox *jukebox,
					      const gchar *uri,
					      NomadTrack *track,
					      gpointer data );
static void neutrino_window_playlist_track_erased( NomadJukebox *jukebox,
						   const gchar *playlist,
						   const gchar *artist,
						   const gchar *album,
						   const gchar *title,
						   gpointer data );
static void neutrino_window_playlist_tracks_erased( NomadJukebox *jukebox,
						    const gchar *playlist,
						    gpointer data );
static void neutrino_window_track_remove( NomadJukebox *jukebox,
					  NomadTrack *track,
					  gpointer data );

static void neutrino_window_set_time( NomadJukebox *jukebox,
				      const gchar *timestring,
				      gpointer data );

static void neutrino_window_eax( NomadJukebox *jukebox,
				 gpointer eax,
				 gpointer data );

static void neutrino_play_begin( NomadJukebox *jukebox,
				 NomadTrack *track, gpointer data );
static gboolean neutrino_window_play_status( NeutrinoWindow *window );
static void neutrino_play_finished( NomadJukebox *jukebox, gpointer data );
static void neutrino_play_progress( NomadJukebox *jukebox,
				    guint elapsed, gboolean changed,
				    NomadTrack *track, 
				    gpointer data );

static void neutrino_window_artist_select( GtkTreeSelection *selection,
					   NeutrinoWindow *window );
static void neutrino_window_album_select( GtkTreeSelection *selection,
					  NeutrinoWindow *window );
static void neutrino_window_track_select( GtkTreeSelection *selection,
					  NeutrinoWindow *window );
static void neutrino_window_playlist_select( const gchar *name,
					     NeutrinoWindow *window );
static void neutrino_window_library_select( GtkWidget *button,
					    NeutrinoWindow *window );
static void neutrino_window_local_select( GtkWidget *button,
					  NeutrinoWindow *window );

static void neutrino_window_library_sensitive( NeutrinoWindow *window );
static void neutrino_window_playlist_sensitive( NeutrinoWindow *window );


static void neutrino_window_playqueue_select( GtkTreeSelection *selection,
					      NeutrinoWindow *window );
static void neutrino_window_play_random( NeutrinoWindow *window );


static void neutrino_window_show_sidebar( GConfClient *client,
					  guint cnxn_id,
					  GConfEntry *entry,
					  gpointer data );
static void neutrino_window_show_browser( GConfClient *client,
					  guint cnxn_id,
					  GConfEntry *entry,
					  gpointer data );
static void neutrino_window_show_playback( GConfClient *client,
					   guint cnxn_id,
					   GConfEntry *entry,
					   gpointer data );
static void neutrino_window_show_toolbar( GConfClient *client,
					  guint cnxn_id,
					  GConfEntry *entry,
					  gpointer data );

static GList *neutrino_window_get_selected_track_ids( NeutrinoWindow *window );

static void neutrino_window_upload_from_list( NeutrinoWindow *window, 
					      GList *tmp );

static void neutrino_window_queue_track( NeutrinoWindow *window,
					 guint trackid );

static void neutrino_window_error( NomadJukebox *jukebox,
				   const gchar *errmsg,
				   NeutrinoWindow *window );

static void neutrino_window_transfer_complete( NomadJukebox *jukebox,
					       NeutrinoWindow *window );

NomadTrack *neutrino_window_edit_track( NeutrinoWindow *window,
					   NomadTrack *track );

static void neutrino_window_view_select( GtkTreeSelection *selection,
					 gpointer data );
static void neutrino_window_set_playlists( NeutrinoWindow *window );
static void neutrino_window_playlist_drop( GtkWidget *widget, 
				GdkDragContext *context,
				gint x, gint y, 
				GtkSelectionData *sel_data,
				guint info, guint time, 
				gpointer data );
static gboolean neutrino_window_playlist_motion( GtkWidget *widget,
		GdkDragContext *context, gint x, gint y,
		guint time,
		gpointer data );

static void neutrino_window_set_status( NeutrinoWindow *window, const gchar *txt );

void
neutrino_window_close( NeutrinoWindow *window )
{
	g_return_if_fail( NEUTRINO_IS_WINDOW( window ) );

	neutrino_window_set_jukebox( window, NULL );
	
	gtk_widget_destroy( GTK_WIDGET( window ) );
}

void neutrino_window_set_jukebox( NeutrinoWindow *window,
				  NomadJukebox *jukebox )
{
	EggActionGroup *group;
	EggAction *action;
	gint i;
	static const gchar* elements[] = {
		"Upload", "0",
		"Download", "0",
		"Cancel", "0",
		"Delete", "0",
		"Scan", "1",
		"EditMeta", "0",
		"NewPlaylist", "0",
		"AddToPlaylist", "0",
		"JukeboxInfo", "0",
		"EAX", "0",
		"playback", "0",
		"Queue", "0",
		"stop", "0",
		"prev", "0",
		"next", "0",
		"DeletePlaylist", "0",
		"RenamePlaylist", "0",
		NULL
	};
	GtkWidget *widget;
	gboolean sensitive;

	g_return_if_fail( NEUTRINO_IS_WINDOW( window ) );

	if( jukebox ) {
		g_return_if_fail( NOMAD_IS_JUKEBOX( jukebox ) );
		if( jukebox == window->jukebox ) {
			return;
		}
	}

	
	/* disconnect signals from previous jukebox */
	if( window->jukebox ) {
		nomad_jukebox_lock( window->jukebox );
		g_signal_handlers_disconnect_matched( G_OBJECT( window->jukebox ),
						      G_SIGNAL_MATCH_DATA,
						      0, 0, NULL,
						      NULL,
						      window );
		nomad_jukebox_unlock( window->jukebox );
	}

	neutrino_jukebox_view_set_jukebox( window->details->view,
					   jukebox );

	/* set the models for the tree views */
	window->jukebox = jukebox;
	if( jukebox ) {
		nomad_jukebox_lock( jukebox );

		window->details->total_tracks = nomad_jukebox_get_num_tracks( jukebox );
		window->details->total_playlists = nomad_jukebox_get_num_playlists( jukebox );
		window->details->total_datafiles = nomad_jukebox_get_num_data_files( jukebox );
		
		/* hookup to signals */
		g_signal_connect( G_OBJECT( jukebox ), "usage",
				  G_CALLBACK( neutrino_window_usage_received ),
				  window );
		g_signal_connect( G_OBJECT( jukebox ), "scanning_tracks",
				  G_CALLBACK( neutrino_window_scanning_tracks),
				  window );
		g_signal_connect( G_OBJECT( jukebox ), "track_add",
				  G_CALLBACK( neutrino_window_track_add ),
				  window );
		g_signal_connect( G_OBJECT( jukebox ), "scanned_tracks",
				  G_CALLBACK( neutrino_window_scanned_tracks ),
				  window );
		g_signal_connect(G_OBJECT( jukebox ), "scanning_playlists",
				 G_CALLBACK(neutrino_window_scanning_playlists),
				 window );

		/* connect to playlist add/remove signals so the
		   sidebar can be managed */
		g_signal_connect( G_OBJECT( jukebox ), "playlist_add",
				  G_CALLBACK(neutrino_window_playlist_add),
				  window );
		g_signal_connect( G_OBJECT( jukebox ), "playlist_remove",
				  G_CALLBACK(neutrino_window_playlist_remove),
				  window );
		g_signal_connect(G_OBJECT( jukebox ), "scanned_playlists",
				 G_CALLBACK(neutrino_window_scanned_playlists),
				 window );
		g_signal_connect( G_OBJECT( jukebox ), "playlist_renamed",
				  G_CALLBACK(neutrino_window_playlist_renamed),
				  window );
		g_signal_connect( G_OBJECT( jukebox ), "transfer",
				  G_CALLBACK(neutrino_window_transfer ),
				  window );
		g_signal_connect( G_OBJECT( jukebox ), "downloaded_track",
				  G_CALLBACK(neutrino_window_downloaded_track),
				  window );

		g_signal_connect(G_OBJECT( jukebox ), "playlist_track_erased",
				 G_CALLBACK(neutrino_window_playlist_track_erased),
				 window );
		g_signal_connect(G_OBJECT( jukebox ), "playlist_tracks_erased",
				 G_CALLBACK(neutrino_window_playlist_tracks_erased),
				 window );
		g_signal_connect( G_OBJECT( jukebox ), "track_remove",
				  G_CALLBACK(neutrino_window_track_remove ),
				  window );
		g_signal_connect( G_OBJECT( jukebox ), "eax",
				  G_CALLBACK( neutrino_window_eax ),
				  window );

		g_signal_connect( G_OBJECT( jukebox ),
				  "play_begin",
				  G_CALLBACK( neutrino_play_begin ),
				  window );	
		g_signal_connect( G_OBJECT( jukebox ),
				  "play_finished",
				  G_CALLBACK( neutrino_play_finished ),
				  window );
		g_signal_connect( G_OBJECT( jukebox ),
				  "play_progress",
				  G_CALLBACK( neutrino_play_progress ),
				  window );

		g_signal_connect( G_OBJECT( jukebox ),
				  "error",
				  G_CALLBACK( neutrino_window_error ),
				  window );

		g_signal_connect(G_OBJECT( jukebox ),
				 "download_complete",
				 G_CALLBACK(neutrino_window_transfer_complete),
				 window );

		neutrino_window_set_playlists( window );
		
		nomad_jukebox_unlock( jukebox );

		nomad_jukebox_getusage( jukebox );
	} else {
		neutrino_window_set_playlists( window );
	}

	/* enable/disable ui elements */
	gdk_threads_enter();

	group = window->details->group;
	
	for( i = 0; elements[ i ]; i += 2) {
		action = egg_action_group_get_action( group, 
							elements[ i ] );
		sensitive  = ( *elements[ i + 1 ] == '1' );
		g_object_set( G_OBJECT( action ), "sensitive",
				sensitive, NULL );

	}

	widget = glade_xml_get_widget( window->xml, "daptable" );
	gtk_widget_set_sensitive( widget, jukebox != NULL );

	gdk_threads_leave();
}

void neutrino_window_upload( NeutrinoWindow *window )
{
	GtkWidget *tree;
	NeutrinoFileBrowser *browser;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GList *list;
	GList *tmp;
	GConfClient *client;
	gboolean recurse;

	client = gconf_client_get_default();

	recurse = gconf_client_get_bool( client, 
					 "/apps/neutrino/upload/recurse",
					 NULL );

	g_object_unref( client );

	tree = window->filemanager;

	browser = window->details->browser;
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( tree ) );
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( tree ) );

	list = NULL;
       	gtk_tree_selection_selected_foreach( selection, 
					     neutrino_util_get_selection,
					     &list );

	/* list now contains a list of iterators */
	tmp = neutrino_file_browser_get_pathnames( browser, list, recurse );
	g_list_free( list );

	neutrino_window_upload_from_list( window, tmp );
	g_list_free( tmp );
}

void neutrino_window_download( NeutrinoWindow *window )
{
	GList *list;
	GConfClient *client;
	gchar *path;
	gchar *format;
	gchar *temp;
	EggAction *action;

	client = gconf_client_get_default();

	list = neutrino_window_get_selected_track_ids( window );

	/* list now contains a list of track ids suitable to be passed
	   onto the NomadJukebox */
	path = gconf_client_get_string( client, "/apps/neutrino/download/path",
					NULL );
	if( ! path || ! strcmp( "current", path ) || *path == '\0' ) {
		/* take path from file browser */
		g_free( path );
		path = g_strdup( gtk_tree_view_column_get_title( GTK_TREE_VIEW_COLUMN( window->col ) ) );
	}

	/* we can only support local pathnames due to libnjb, so
	   strip the scheme */
#ifdef HAVE_GNOME_VFS
	temp = gnome_vfs_get_local_path_from_uri( path );
	if( temp ) {
		g_free( path );
		path = temp;
	}
#endif
	if( path[ strlen( path ) - 1 ] != G_DIR_SEPARATOR ) {
		temp = path;
		path = g_strconcat( temp, G_DIR_SEPARATOR_S, NULL );
		g_free( temp );
	}
	
	format = gconf_client_get_string( client,
					  "/apps/neutrino/download/format",
					  NULL );
	if( ! format ) {
		format = g_strdup( "%a - %t" );
	}

	window->num_transfers ++;

	action = egg_action_group_get_action( window->details->group,
						"Cancel" );
	g_object_set( G_OBJECT( action ), "sensitive", TRUE, NULL );

     	nomad_jukebox_download( window->jukebox, path, format, list );

	g_free( path );

	g_free( format );

	g_list_free( list );

	g_object_unref( client );
}

void neutrino_window_delete( NeutrinoWindow *window )
{
	GList *list;
	GtkWidget *widget;
	guint view;
	GtkTreeSelection *sel;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeIter it;
	
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( window->tracks ) );
	
	list = neutrino_window_get_selected_track_ids( window );
		
	widget = glade_xml_get_widget( window->xml, "mode" );
	view = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( widget ), 
						    "view" ) );

	if( view == PLAYLIST_VIEW ) {
		const gchar *name;

		name = (const gchar*)g_object_get_data( G_OBJECT( widget ),
							"name" );
		/* name will be NULL if this is the play queue */
		if( name ) {
			nomad_jukebox_delete_tracks_from_playlist( window->jukebox,
								      list,
								      name );
		} else {
			/* remove from playqueue */
			GList *tmp;

			g_list_free( list );
			list = gtk_tree_selection_get_selected_rows( sel, &model );
			for( tmp = list; tmp; tmp = tmp->next ) {
				path = tmp->data;
				gtk_tree_model_get_iter( model, &it,
						path );
				tmp->data = gtk_tree_iter_copy( &it );
				gtk_tree_path_free( path );
			}
			for( tmp = list; tmp; tmp = tmp->next ) {
				gtk_list_store_remove( GTK_LIST_STORE( model ), tmp->data );
				gtk_tree_iter_free( tmp->data );
			}
		}
	} else {
		nomad_jukebox_delete_tracks( window->jukebox, list );
	}

	g_list_free( list );
}

void neutrino_window_edit_meta( NeutrinoWindow *window )
{
	GList *list;
     	GList *tmp;

	g_return_if_fail( NEUTRINO_IS_WINDOW( window ) );

	list = neutrino_window_get_selected_track_ids( window );


	for( tmp = list; tmp; tmp = tmp->next ) {
		NomadTrack *track;
		guint id;

		id = GPOINTER_TO_UINT( tmp->data );
		track = nomad_jukebox_request_track( window->jukebox, id );

		if( track ) {
			/* we have a track to edit */
			NomadTrack *ntrack;

			ntrack = neutrino_window_edit_track( window, track );
				
			if( ntrack ) {
				nomad_jukebox_set_metadata( window->jukebox,
							       ntrack );
				nomad_track_free( ntrack );
			}
			nomad_track_free( track );
		}
	}

	g_list_free( list );
}


void neutrino_window_new_playlist( NeutrinoWindow *window )
{
	GladeXML *xml;
	GtkWidget *widget;
	gint button;
	GList *tracks;

	widget = window->tracks;


	tracks = neutrino_window_get_selected_track_ids( window );

	xml = glade_xml_new( DATADIR"/neutrino/glade/neutrino.glade",
			     "new_playlist", NULL );
	glade_xml_signal_autoconnect( xml );

	widget = glade_xml_get_widget( xml, "playlist_icon" );
	gtk_image_set_from_stock( GTK_IMAGE( widget ),
				  "Neutrino_Playlist",
				  GTK_ICON_SIZE_DIALOG );

	widget = glade_xml_get_widget( xml, "new_playlist" );

	button = gtk_dialog_run( GTK_DIALOG( widget ) );
	if( button == GTK_RESPONSE_OK ) {
		const gchar *name;

		widget = glade_xml_get_widget( xml, "playlist_name" );
		name = gtk_entry_get_text( GTK_ENTRY( widget ) );

		nomad_jukebox_create_playlist( window->jukebox,
						  name );

		widget = glade_xml_get_widget( xml, "playlist_add" );
		if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) ) {
			nomad_jukebox_add_tracks_to_playlist( window->jukebox,
								 name,
								 tracks );
		}
	}

	widget = glade_xml_get_widget( xml, "new_playlist" );
	gtk_widget_destroy( widget );

	g_list_free( tracks );

	g_object_unref( G_OBJECT( xml ) );
}

void neutrino_window_add_to_playlist( NeutrinoWindow *window )
{
	GladeXML *xml;
	GtkWidget *widget;
	gint button;
	GSList *list;
	GSList *playlists;
	GList *tracks;

	GtkWidget *checkbox;
	GtkTreeModel *model;
	GtkTreeIter it;
	gint type;
	gchar *name;
	
	widget = window->tracks;

	tracks = neutrino_window_get_selected_track_ids( window );

	g_assert( tracks );

	xml = glade_xml_new( DATADIR"/neutrino/glade/neutrino.glade",
			     "add_to_playlist", NULL );
	glade_xml_signal_autoconnect( xml );

	widget = glade_xml_get_widget( xml, "playlists" );

	playlists = NULL;
	model = GTK_TREE_MODEL( window->details->store );
	if( gtk_tree_model_get_iter_first( model, &it ) ) {
		do {
			gtk_tree_model_get( model, &it,
					NEUTRINO_TYPE_COL, &type, 
					NEUTRINO_DATA_COL, &name,
					-1 );
			if( type == NEUTRINO_VIEW_PLAYLIST &&
				name ) {

				checkbox = gtk_check_button_new_with_label( name );
				playlists = g_slist_prepend( playlists, checkbox );
				gtk_widget_show( checkbox );

				g_object_set_data( G_OBJECT( checkbox ), "name", name );
				gtk_box_pack_start( GTK_BOX( widget ), checkbox, FALSE, FALSE, 0 );
			} else {
				g_free( name );
			}
		} while( gtk_tree_model_iter_next( model, &it ) );
	}
	
	widget = glade_xml_get_widget( xml, "add_to_playlist" );

	button = gtk_dialog_run( GTK_DIALOG( widget ) );
		
	list = NULL;
	if( button == GTK_RESPONSE_OK ) {
		GSList *lists;
		gboolean all;

		widget = glade_xml_get_widget( xml, "playlists" );

		all = ! GTK_WIDGET_SENSITIVE( widget );


		for( lists = playlists; lists; lists = lists->next ) {
			GtkToggleButton *checkbox;
		       
			checkbox = GTK_TOGGLE_BUTTON( lists->data );
			name = g_object_get_data( G_OBJECT( lists->data ),
						  "name" );
			if( all || gtk_toggle_button_get_active( checkbox ) ) {
				list = g_slist_prepend( list, name );
			} 		}
		g_slist_free( playlists );

	}

	widget = glade_xml_get_widget( xml, "add_to_playlist" );
	gtk_widget_destroy( widget );

	for( playlists = list; playlists; playlists = playlists->next ) {
		name = playlists->data;
		nomad_jukebox_add_tracks_to_playlist( window->jukebox,
							 name,
							 tracks );
	}
	if( list ) {
		g_slist_free( list );
	}
	g_list_free( tracks );
	
	g_object_unref( G_OBJECT( xml ) );
}

void neutrino_window_delete_playlist( NeutrinoWindow *window )
{
	GtkWidget *widget;
	const gchar *name;
	guint view;

	widget = glade_xml_get_widget( window->xml, "mode" );
	view = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( widget ), 
						    "view" ) );

	name = (const gchar*)g_object_get_data( G_OBJECT( widget ),
						"name" );
	g_assert( name );

	nomad_jukebox_delete_playlist( window->jukebox, name );
}

void neutrino_window_rename_playlist( NeutrinoWindow *window )
{
	GtkWidget *widget;
	const gchar *name;
	guint view;
	GladeXML *xml;
	gint button;

	widget = glade_xml_get_widget( window->xml, "mode" );
	view = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( widget ), 
						    "view" ) );

	name = (const gchar*)g_object_get_data( G_OBJECT( widget ),
						"name" );
	g_assert( name );

	/* get new name */
	xml = glade_xml_new( DATADIR"/neutrino/glade/neutrino.glade",
			     "rename_playlist", NULL );
	glade_xml_signal_autoconnect( xml );

	widget = glade_xml_get_widget( xml, "playlist_icon" );
	gtk_image_set_from_stock( GTK_IMAGE( widget ),
				  "Neutrino_Playlist",
				  GTK_ICON_SIZE_DIALOG );

	widget = glade_xml_get_widget( xml, "rename_playlist" );

	button = gtk_dialog_run( GTK_DIALOG( widget ) );
	if( button == GTK_RESPONSE_OK ) {
		const gchar *newname;

		widget = glade_xml_get_widget( xml, "playlist_name" );
		newname = gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		if( strcmp( name, newname ) ) {
			nomad_jukebox_rename_playlist( window->jukebox, 
							  name,
							  newname );
		}
	}
	widget = glade_xml_get_widget( xml, "rename_playlist" );
	gtk_widget_destroy( widget );
	
	g_object_unref( G_OBJECT( xml ) );
}

void neutrino_window_jukebox_info( NeutrinoWindow *window )
{
	GladeXML *xml;
	GtkWidget *widget;
	gint response;
	const gchar *str;
	gchar *totals;
	gulong sigid;

	NomadJukebox *jukebox;

	jukebox = window->jukebox;

	g_assert( jukebox );

	xml = glade_xml_new( DATADIR"/neutrino/glade/neutrino.glade",
			     "jukeboxinfo", NULL );

	widget = glade_xml_get_widget( xml, "firmware" );
	str = nomad_jukebox_get_firmware( jukebox );
	gtk_label_set_text( GTK_LABEL( widget ), str );

	widget = glade_xml_get_widget( xml, "product" );
	str = nomad_jukebox_get_prodname( jukebox );
	gtk_label_set_text( GTK_LABEL( widget ), str )	;

	if( nomad_jukebox_get_power( jukebox ) ) {
		widget = glade_xml_get_widget( xml, "poweryes" );
	} else {
		widget = glade_xml_get_widget( xml, "powerno" );
	}
	gtk_widget_show( widget );

	widget = glade_xml_get_widget( xml, "owner" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	str = nomad_jukebox_get_ownerstring( jukebox );
	gtk_entry_set_text( GTK_ENTRY( widget ), str );

	widget = glade_xml_get_widget( xml, "clock" );
	sigid = g_signal_connect( G_OBJECT( jukebox ), "time",
				  G_CALLBACK( neutrino_window_set_time ),
				  widget );
	nomad_jukebox_get_time( jukebox );

	widget = glade_xml_get_widget( xml, "tracks" );
	totals = g_strdup_printf( "%i", window->details->total_tracks );
	gtk_label_set_text( GTK_LABEL( widget ), totals );
	g_free( totals );

	widget = glade_xml_get_widget( xml, "playlists" );
	totals = g_strdup_printf( "%i", window->details->total_playlists );
	gtk_label_set_text( GTK_LABEL( widget ), totals );
	g_free( totals );

	widget = glade_xml_get_widget( xml, "datafiles" );
	totals = g_strdup_printf( "%i", window->details->total_datafiles );
	gtk_label_set_text( GTK_LABEL( widget ), totals );
	g_free( totals );

	widget = glade_xml_get_widget( xml, "info_usage" );
	if( window->details->total_space != 0 ) {
		gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( widget ),
					       window->details->total_used /
					       (gdouble)window->details->total_space );
	} else {
		/* should never really happen */
		gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( widget ),
					       0.0 );
	}

	widget = glade_xml_get_widget( xml, "jukeboxinfo" );

	gtk_window_set_transient_for( GTK_WINDOW( widget ),
				      GTK_WINDOW( window ) );
	response = gtk_dialog_run( GTK_DIALOG( widget ) );

	if( response == GTK_RESPONSE_APPLY ) {
		widget = glade_xml_get_widget( xml, "owner" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		str = gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		nomad_jukebox_set_ownerstring( jukebox, str );
	} else {
		widget = glade_xml_get_widget( xml, "jukeboxinfo" );
		g_signal_handler_disconnect( G_OBJECT( jukebox ), sigid );
		gtk_widget_destroy( widget );
	}

	g_object_unref( G_OBJECT( xml ) );
}

void neutrino_window_edit_eax( NeutrinoWindow *window )
{
	NomadJukebox *jukebox;

	jukebox = window->jukebox;

	nomad_jukebox_get_eax( jukebox );
}


void neutrino_window_playback( NeutrinoWindow *window )
{
	GtkWidget *tree;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GList *list;
	GList *tmp;

	GtkListStore *store;
	GtkTreeIter it;
	GtkTreePath *path;

	g_random_set_seed( time( NULL ) );

	tree = window->tracks;
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( tree ) );
	store = GTK_LIST_STORE( window->details->queue );

	list = gtk_tree_selection_get_selected_rows( selection, 
						&model );
	
	if( model != GTK_TREE_MODEL( store ) ) {
		gtk_list_store_clear( store );

		window->playqueue_size = 0;
		for( tmp = list; tmp; tmp = tmp->next ) {
			guint id;

			path = (GtkTreePath*)tmp->data;
			gtk_tree_model_get_iter( model, &it, path );
			
			gtk_tree_model_get( model, &it,
					    TRACK_TRACK_ID_COL, &id,
					    -1 );
	
			neutrino_window_queue_track( window, id );

			gtk_tree_path_free( path );
		}
		
		selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( window->playqueue ) );
		if( gtk_tree_model_get_iter_first( GTK_TREE_MODEL( store ),
						   &it ) ) {
			gtk_tree_selection_unselect_all( selection );
			gtk_tree_selection_select_iter( selection, &it );
		}
	} else {
		/* set item from the play queue playing */

		path = list->data;
		
		selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( window->playqueue ) );
		gtk_tree_selection_unselect_all( selection );
		model = gtk_tree_view_get_model( GTK_TREE_VIEW( window->playqueue ) );
		gtk_tree_selection_select_path( selection, path );

		g_list_foreach( list, (GFunc)gtk_tree_path_free, NULL );
	}

	g_list_free( list );
}

void neutrino_window_queue( NeutrinoWindow *window )
{
	GList *list;
	GList *tmp;

	g_return_if_fail( NEUTRINO_IS_WINDOW( window ) );

	list = neutrino_window_get_selected_track_ids( window );

	for( tmp = list; tmp; tmp = tmp->next ) {
		neutrino_window_queue_track( window,
					     GPOINTER_TO_UINT( tmp->data ) );
	}

	g_list_free( list );
}

void neutrino_window_cancel( NeutrinoWindow *window )
{
	EggAction *action;

	action = egg_action_group_get_action( window->details->group,
						"Cancel" );
	
	window->num_transfers --;
	if( window->num_transfers == 0 ) {
		g_object_set( G_OBJECT( action ), "sensitive", FALSE,
				NULL );
		nomad_jukebox_abort( window->jukebox );
	}
}

void neutrino_window_select_all( NeutrinoWindow *window )
{
	GtkTreeView *view;
	GtkTreeSelection *selection;

	view = GTK_TREE_VIEW( window->tracks );
	selection = gtk_tree_view_get_selection( view );

	gtk_tree_selection_select_all( selection );
}

/* connected signals */
void neutrino_window_fm_expanded( GtkTreeView *view,
				  GtkTreeIter *iter,
				  GtkTreePath *path )
{
	NeutrinoWindow *window;
	NeutrinoFileBrowser *browser;
	
	if( ! g_object_get_data( G_OBJECT( view ), "block" ) ) {
		window = NEUTRINO_WINDOW( g_object_get_data( G_OBJECT( view ), 
							     "window" ) );
		browser = window->details->browser;
		
		neutrino_file_browser_scan_iter( browser, iter, -1 );
		
		g_object_set_data( G_OBJECT( view ), "block", 
				   GUINT_TO_POINTER( 1 ) );
		gtk_tree_view_expand_row( view, path, FALSE );
		g_object_set_data( G_OBJECT( view ), "block", NULL );
	}
}

void neutrino_window_fm_activated( GtkTreeView *view,
				   GtkTreePath *path,
				   GtkTreeViewColumn *column )
{
	NeutrinoWindow *window;
	NeutrinoFileBrowser *browser;
	GtkTreeModel *model;
	GtkTreeIter it;

	gchar *dirname;
	const gchar *mime;
	GnomeVFSMimeApplication *app;
	GtkWidget *widget;
	
	window = NEUTRINO_WINDOW( g_object_get_data( G_OBJECT( view ), 
						     "window" ) );
	browser = window->details->browser;
	
	model = gtk_tree_view_get_model( view );
	if( gtk_tree_model_get_iter( model, &it, path ) ) {
		const gchar *current;
		current = neutrino_file_browser_scan_iter( browser, &it, -1 );
		if( current ) {
			gtk_tree_view_column_set_title( GTK_TREE_VIEW_COLUMN( window->col ),
						       current );
		} else {
#ifdef HAVE_GNOME_VFS
			/* not a directory, get a filename */
			GList *list;
			GList *ret;
			gchar *url;
			gchar *exstr;
			gchar *temp;
			list = g_list_append( NULL, &it );
			ret = neutrino_file_browser_get_pathnames( browser,
								   list, 
								   FALSE );
			g_list_free( list );
			if( ret ) {
				current = ret->data;
				mime = gnome_vfs_get_mime_type( current );
				app = gnome_vfs_mime_get_default_application( mime );
				if( app ) {
					dirname = g_path_get_dirname( current );
					url = NULL;
					if( ! app->expects_uris ) {
						url = gnome_vfs_get_local_path_from_uri( current );
					} 
					if( ! url ) {
						url = g_strdup( current );
					}
					current = strstr( app->command, "%s" );
					if( ! current ) {
						exstr = g_strconcat( app->command, " \"", url, "\"", NULL );
					} else {
						temp = g_strndup( app->command, current - app->command );
						exstr = g_strconcat( temp, url, current + strlen( "%s" ), NULL );
						g_free( temp );
					}
					if( app->requires_terminal ) {
						gnome_execute_terminal_shell( dirname, exstr );
					} else {
						gnome_execute_shell( dirname, exstr );
					}
					g_free( exstr );
					g_free( dirname );
				} else {
					widget = gtk_message_dialog_new( GTK_WINDOW( window ), GTK_DIALOG_MODAL, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK, "No application for for handling %s files", mime );
					gtk_dialog_run( GTK_DIALOG( widget ) );
					gtk_widget_destroy( widget );
				}
				g_list_free( ret );
			}
#endif
		}
	}
}


void neutrino_eax_environment_change( GtkOptionMenu *option_menu )
{
	NeutrinoWindow *window;
	NomadJukebox *jukebox;
	gint item;

	window = g_object_get_data( G_OBJECT( option_menu->menu_item ),
				    "window" );
	jukebox = window->jukebox;

	item = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( option_menu->menu_item ), "effect" ) );

	gdk_threads_leave();
	nomad_jukebox_set_effect( jukebox, item );
	gdk_threads_enter();
}

void neutrino_eax_headphone_change( GtkOptionMenu *option_menu )
{
	NeutrinoWindow *window;
	NomadJukebox *jukebox;
	gint item;

	window = g_object_get_data( G_OBJECT( option_menu->menu_item ),
				    "window" );
	jukebox = window->jukebox;

	item = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( option_menu->menu_item ), "headphone" ) );

	gdk_threads_leave();
	nomad_jukebox_set_headphone( jukebox, item );
	gdk_threads_enter();
}

void neutrino_eax_rear_change( GtkOptionMenu *option_menu )
{
	NeutrinoWindow *window;
	NomadJukebox *jukebox;
	gint item;

	window = g_object_get_data( G_OBJECT( option_menu->menu_item ),
				    "window" );
	jukebox = window->jukebox;

	item = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( option_menu->menu_item ), "rear" ) );
	
	gdk_threads_leave();
	nomad_jukebox_set_rearmode( jukebox, item );
	gdk_threads_enter();
}

void neutrino_eax_midfreq_change( GtkOptionMenu *option_menu )
{
	NeutrinoWindow *window;
	NomadJukebox *jukebox;
	gint item;

	window = g_object_get_data( G_OBJECT( option_menu->menu_item ),
				    "window" );
	jukebox = window->jukebox;

	item = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( option_menu->menu_item ), "midfreq" ) );

	gdk_threads_leave();
	nomad_jukebox_set_midfreq( jukebox, item );
	gdk_threads_enter();
}

void neutrino_eax_eq_active( GtkToggleButton *toggle )
{
	NeutrinoWindow *window;
	NomadJukebox *jukebox;

	window = g_object_get_data( G_OBJECT( toggle ), "window" );
	jukebox = window->jukebox;

	gdk_threads_leave();
	nomad_jukebox_set_eq_active( jukebox,
					gtk_toggle_button_get_active(toggle) );
	gdk_threads_enter();
}

void neutrino_eax_effect_amount( GtkRange *range )
{
	NeutrinoWindow *window;
	NomadJukebox *jukebox;

	window = g_object_get_data( G_OBJECT( range ), "window" );
	jukebox = window->jukebox;

	gdk_threads_leave();
	nomad_jukebox_set_effamt( jukebox,
				     gtk_range_get_value( range ) );
	gdk_threads_enter();
}

void neutrino_eax_set_vol( GtkRange *range )
{
	NeutrinoWindow *window;
	NomadJukebox *jukebox;

	window = g_object_get_data( G_OBJECT( range ), "window" );

	jukebox = window->jukebox;
	
	gdk_threads_leave();
	nomad_jukebox_set_volume( jukebox,
				     gtk_range_get_value( range ) * 5 );
	gdk_threads_enter();
}

void neutrino_eax_set_bass( GtkRange *range )
{
	NeutrinoWindow *window;
	NomadJukebox *jukebox;

	window = g_object_get_data( G_OBJECT( range ), "window" );

	jukebox = window->jukebox;

	gdk_threads_leave();
	nomad_jukebox_set_bass( jukebox,
				   gtk_range_get_value( range ) );
	gdk_threads_enter();
}

void neutrino_eax_set_mid( GtkRange *range )
{
	NeutrinoWindow *window;
	NomadJukebox *jukebox;

	window = g_object_get_data( G_OBJECT( range ), "window" );

	jukebox = window->jukebox;

	gdk_threads_leave();
	nomad_jukebox_set_midrange( jukebox,
				       gtk_range_get_value( range ) );
	gdk_threads_enter();
}

void neutrino_eax_set_treble( GtkRange *range )
{
	NeutrinoWindow *window;
	NomadJukebox *jukebox;

	window = g_object_get_data( G_OBJECT( range ), "window" );

	jukebox = window->jukebox;

	gdk_threads_leave();
	nomad_jukebox_set_treble( jukebox,
				     gtk_range_get_value( range ) );
	gdk_threads_enter();
}

void neutrino_window_play_close( NeutrinoWindow *window )
{
	GtkWidget *widget;

	widget = glade_xml_get_widget( window->xml, "play_icon" );
	gtk_image_set_from_stock( GTK_IMAGE( widget ), "Neutrino_Stop",
				  GTK_ICON_SIZE_SMALL_TOOLBAR );

	nomad_jukebox_play_stop( window->jukebox );
}

void neutrino_window_play_prev( NeutrinoWindow *window )
{
	GtkTreeView *view;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter it;

	view = GTK_TREE_VIEW( window->playqueue );
	selection = gtk_tree_view_get_selection( view );

	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {
		GtkTreeIter pit;
		
		if( g_object_get_data( G_OBJECT( window ), "shuffle" ) ) {
			/* select a random track */
			neutrino_window_play_random( window );
		} else if( gtk_tree_model_get_iter_first( model, &pit ) ) {
			GtkTreeIter *nit;
			GtkTreeIter *prev;
			gboolean set;

			nit = gtk_tree_iter_copy( &pit );
			prev = NULL;

			set = FALSE;

			do {
				set = ! memcmp( nit, &it,
						sizeof( GtkTreeIter ) );
				if( ! set ) {
					if( prev ) {
						gtk_tree_iter_free( prev );
					}
					prev = gtk_tree_iter_copy( nit );
				}
			} while( (! set ) &&
				 gtk_tree_model_iter_next( model, nit ) );

			gtk_tree_iter_free( nit );

			if( set && prev ) {
				gtk_tree_selection_select_iter( selection, 
								prev );
			} else {
				gtk_tree_selection_unselect_iter( selection,
								  &it );
				if( g_object_get_data( G_OBJECT( window ), 
						       "repeat" ) ) {
					/* select first iter */
					gtk_tree_model_get_iter_first( model,
								       &it );
					gtk_tree_selection_select_iter( selection,
									&it );
				}
			}
			if( prev ) {
				gtk_tree_iter_free( prev );
			}
		} else {
			nomad_jukebox_play_stop( window->jukebox );
		}
	} else {
		nomad_jukebox_play_stop( window->jukebox );
	}
}

gboolean neutrino_window_play_next( NeutrinoWindow *window )
{
	GtkTreeView *view;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter it;

	view = GTK_TREE_VIEW( window->playqueue );
	selection = gtk_tree_view_get_selection( view );

	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {
		GtkTreeIter *nit;

		if( g_object_get_data( G_OBJECT( window ), "shuffle" ) ) {
			/* select a random track */
			neutrino_window_play_random( window );
		} else {
			nit = gtk_tree_iter_copy( &it );
			if( gtk_tree_model_iter_next( model, nit ) ) {
				gtk_tree_selection_select_iter( selection, 
								nit );
			} else {
				gtk_tree_selection_unselect_iter( selection, 
								  &it );
				if( g_object_get_data( G_OBJECT( window ), 
						       "repeat" ) ) {
					/* select first iter */
					gtk_tree_model_get_iter_first( model,
								       &it );
					gtk_tree_selection_select_iter( selection,
									&it );
				}
			}
			gtk_tree_iter_free( nit );
		}
	} else {
		nomad_jukebox_play_stop( window->jukebox );
	}

	return FALSE;
}

void neutrino_window_search( GtkWidget *widget )
{
	GladeXML *xml;
	NeutrinoWindow *window;
	const gchar *search_string;
	NeutrinoJukeboxSearchMode mode;
	gchar *title;

	xml = glade_get_widget_tree( widget );

	widget = glade_xml_get_widget( xml, "tracks" );

	window = NEUTRINO_WINDOW( g_object_get_data( G_OBJECT( widget ),
						     "window") );

	widget = glade_xml_get_widget( xml, "search" );
	search_string = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "artists_albums" );
	if( ! GTK_WIDGET_VISIBLE( widget ) ) {
		neutrino_window_library_select(window->details->library,
					window );
		gtk_widget_hide( widget );
	}
	
	widget = glade_xml_get_widget( xml, "search_mode" );
	widget = GTK_OPTION_MENU( widget )->menu_item;
	mode = (NeutrinoJukeboxSearchMode)
		g_object_get_data( G_OBJECT( widget ), "search_mode" );
	if( neutrino_jukebox_view_search( window->details->view,
				      mode, search_string ) ) {
		widget = glade_xml_get_widget( window->xml,
					       "mode" );
		title = g_strdup_printf( "<span weight=\"bold\" size=\"large\">Search Results (%i)</span>",
					 window->details->view->num );
		gtk_label_set_markup( GTK_LABEL( widget ), title );
		g_object_set_data( G_OBJECT( widget ), "view",
				   GUINT_TO_POINTER( SEARCH_VIEW ) );
		g_free( title );

		title = g_strdup_printf( "Search Results (%i) - Neutrino",
					 window->details->view->num );
		gtk_window_set_title( GTK_WINDOW( window ), title );
		g_free( title );
	} else {
		widget = gtk_message_dialog_new( GTK_WINDOW( window ),
						GTK_DIALOG_MODAL,
						GTK_MESSAGE_ERROR,
						GTK_BUTTONS_OK,
						"Regular Expressions are used for searching, the entered expression was not valid" );
		gtk_dialog_run( GTK_DIALOG( widget ) );
		gtk_widget_destroy( widget );
	}
}

void neutrino_window_tracks_popup_key( GtkWidget *view, gpointer data )
{
	neutrino_window_tracks_popup( view, NULL, NULL );
	g_signal_stop_emission_by_name( G_OBJECT( view ),
					"popup_menu" );
}

gboolean neutrino_window_tracks_popup( GtkWidget *view,
				       GdkEventButton *event,
				       gpointer data )
{
	gboolean ret;
	GtkWidget *widget;
	NeutrinoWindow *window;

	ret = FALSE;
	if( ( ! event ) || ( event->button == 3 ) ) {
		ret = TRUE;

		window = NEUTRINO_WINDOW( g_object_get_data( G_OBJECT( view ), 
							     "window" ) );

		if( event && event->button == 3 ) {
			g_signal_stop_emission_by_name( G_OBJECT( view ),
							"button_press_event" );
		}

		widget = egg_menu_merge_get_widget( window->details->merge, "/popups/tracks" );

		gnome_popup_menu_do_popup_modal( widget, 0, 0,
						 event, 0, view );
	}

	return ret;
}

void neutrino_window_drop( GtkWidget *widget, GdkDragContext *context,
			   gint x, gint y, 
			   GtkSelectionData *sel_data,
			   guint info, guint time, 
			   gpointer data )
{
	NeutrinoWindow *window;
	gchar **uris;
	guint i;
	GList *list;
	GList *files;
	gboolean recurse;
	GConfClient *client;

	g_signal_stop_emission_by_name( G_OBJECT( widget ),
                                        "drag_data_received" );

	window = g_object_get_data( G_OBJECT( widget ), "window" );


	client = gconf_client_get_default();
	recurse = gconf_client_get_bool( client, 
					 "/apps/neutrino/upload/recurse",
					 NULL );
	g_object_unref( client );

	switch( info ) {
	case TARGET_URI_LIST:
		uris = g_strsplit( sel_data->data, "\r\n", 0 );
	
		for( list = NULL, i = 0; uris[ i ]; ++ i ) {
			if( recurse ) {
				files = neutrino_util_recurse_directory(uris[i],
									NULL );
				list = g_list_concat( list, files );
			} else if( strlen( uris[ i ] ) ) {
				list = g_list_prepend( list, uris[ i ] );
			}
		}

		neutrino_window_upload_from_list( window, list );

		g_strfreev( uris );
		g_list_free( list );

		break;
	default:
		g_assert( FALSE );
		break;
	}
      
	gtk_drag_finish( context, TRUE, FALSE, time );
}

void neutrino_window_tracks_set_dnd_data( GtkWidget *widget,
					  GdkDragContext *context,
					  GtkSelectionData *sel_data,
					  guint info, guint time,
					  gpointer data )
{
	NeutrinoWindow *window;
	GList *list;
	GList *tmp;
	GString *string;
	const gchar *idstring;

	window = NEUTRINO_WINDOW( g_object_get_data( G_OBJECT( widget ),
						     "window" ) );

	list = neutrino_window_get_selected_track_ids( window );

	string = g_string_new( "" );
	idstring = nomad_jukebox_get_idstring( window->jukebox );

	for( tmp = list; tmp; tmp = tmp->next ) {
		guint id;

		id = GPOINTER_TO_UINT( tmp->data );

		g_string_append_printf( string, "nomad://%s/tracks/%u\r\n",
					idstring, id );
	}

	gtk_selection_data_set( sel_data, sel_data->target,
				8, string->str, string->len );


	g_string_free( string, TRUE );
}

void neutrino_window_tracks_dnd_delete( GtkWidget *widget, 
					GdkDragContext *context,
					gpointer data )
{
        g_signal_stop_emission_by_name( G_OBJECT( widget ), 
					"drag_data_delete" );
}

void neutrino_window_playqueue_drop( GtkWidget *widget, GdkDragContext *context,
				     gint x, gint y, 
				     GtkSelectionData *sel_data,
				     guint info, guint time, 
				     gpointer data )
{
	NeutrinoWindow *window;
	gchar **uris;
	guint i;
	GtkTreeModel *model;
       	const gchar *idstring;

	g_signal_stop_emission_by_name( G_OBJECT( widget ),
                                        "drag_data_received" );

	window = g_object_get_data( G_OBJECT( widget ), "window" );
	if( ! window ) {
		window = NEUTRINO_WINDOW( data );
	}

	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	idstring = nomad_jukebox_get_idstring( window->jukebox );

	switch( info ) {
	case TARGET_NEUTRINO_TRACK_LIST:
		uris = g_strsplit( sel_data->data, "\r\n", 0 );
		
		for( i = 0; uris[ i ]; ++ i ) {
			gchar *uri;
			gchar *type;
			gchar *dropid;

			uri = uris[ i ];
			dropid = type = NULL;

			if( ! strncmp( "nomad://", uri,
				       strlen( "nomad://" ) ) ) {
				uri += strlen( "nomad://" );
				type = strchr( uri, '/' );
				
				g_assert( type != NULL );

				dropid = g_strndup( uri, type - uri );
			}
			if( dropid && ! strcmp( dropid, idstring ) ) {
				/* drop from the same jukebox, allow it */
				gchar *trackid;
				guint id;
			
				trackid = strchr( type + 1, '/' );
				
				g_assert( trackid != NULL );

				trackid ++;

				id = strtoul( trackid, NULL, 10 );
				neutrino_window_queue_track( window, id );
			}
		}
		
		g_strfreev( uris );

		break;
	default:
		g_assert( FALSE );
		break;
	}
      
	gtk_drag_finish( context, TRUE, FALSE, time );
}

void neutrino_window_scan_jukeboxes( NeutrinoWindow *window )
{
	NeutrinoWindowDetails *details;
	guint boxes;
	GdkPixbuf *pixbuf;
	guint i;
	NomadJukebox *jukebox;
	gchar *title;
	const gchar *id;
	GtkTreeIter tit;
	GtkTreeIter it;
	guint type;

	details = window->details;
	boxes = neutrino_shell_get_jukeboxes( window->shell );
	pixbuf = gtk_widget_render_icon( details->list,
			"Neutrino_Library", GTK_ICON_SIZE_MENU,
			NULL );

	gtk_tree_model_get_iter( GTK_TREE_MODEL( details->store ),
				&tit, details->jukeboxes_path );
	
	/* clear old */
	it = tit;
	i = 0;
	while( gtk_tree_model_iter_next( GTK_TREE_MODEL( details->store ),
				&it ) ) {
		gtk_tree_model_get( GTK_TREE_MODEL( details->store ),
				&it,
				NEUTRINO_TYPE_COL, &type,
				-1 );
		if( type == NEUTRINO_VIEW_NONE ) {
			break;
		}
		i ++;		
	}
	if( i != 0 ) {
		it = tit;
		gtk_tree_model_iter_next( GTK_TREE_MODEL( details->store ),
				&it );
		while( i > 0 ) {
			gtk_list_store_remove( details->store, &it );
			i --;
		}
	}

	/* insert new */
	
	for( i = 0; i < boxes; ++ i ) {
		gtk_list_store_insert_after( details->store, 
						&it, &tit );
		title = g_strdup_printf( "%s %i",
				_( "Jukebox" ), i + 1 );
		jukebox = neutrino_shell_get_nth_jukebox( window->shell,
							  i );
		id = nomad_jukebox_get_idstring( jukebox );
		gtk_list_store_set( details->store, &it,
			    NEUTRINO_DISPLAY_COL, title,
			    NEUTRINO_TYPE_COL, NEUTRINO_VIEW_JUKEBOX,
			    NEUTRINO_DATA_COL, id,
			    NEUTRINO_ICON_COL, pixbuf,
			    -1 );
		g_free( title );
	}
	if( pixbuf ) {
		g_object_unref( pixbuf );
	}

}

/* static stuff */
static void neutrino_has_selection( GtkTreeModel *model,
			     GtkTreePath *path,
			     GtkTreeIter *it,
			     gpointer data )
{
	gboolean *has;

	has = (gboolean*)data;

	*has = TRUE;
}

static void neutrino_util_get_selection( GtkTreeModel *model,
				  GtkTreePath *path,
				  GtkTreeIter *iter,
				  gpointer data )
{
	GList **list;

	list = (GList**)data;

	*list = g_list_append( *list, gtk_tree_iter_copy( iter ) );
}


	
static void
neutrino_window_constructed( NeutrinoWindow *window )
{
	NeutrinoWindowDetails *details;
	GtkWidget *content;
	GConfClient *client;
	EggAction *action;

	GtkWidget *contents;
	static const gchar *search_modes[] = {
		"Artists Matching",
		"Albums Matching",
		"Titles Matching",
		"Genres Matching",
		"Years Matching",
		NULL
	};
	gint i;
	GtkWidget *widget;
	GtkWidget *menu;

	GtkWidget *view;
	GtkWidget *sw;
	GtkCellRenderer *rend;
	GtkCellRenderer *prend;
	GtkTreeViewColumn *col;
	GtkTreeIter it;
	GtkTreeSelection *sel;
	GdkPixbuf *pixbuf;
	gchar *title;
	guint boxes;
	NomadJukebox *jukebox;
	const gchar *id;
	
	details = window->details;
	
	gtk_window_set_title( GTK_WINDOW( window ), "Neutrino" );
	gtk_window_set_wmclass( GTK_WINDOW( window ), "Neutrino",
				"mainwindow" );
	gtk_window_set_default_size( GTK_WINDOW( window ), 620, 400 );

	window->content_hbox = gtk_vbox_new( FALSE, 0 );
	gtk_widget_show( window->content_hbox );

	neutrino_window_initialise_menus( window );

	contents = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( window->content_hbox ), 
				contents, TRUE, TRUE, 0 );

	widget = gtk_statusbar_new();
	gtk_widget_show( widget );
	gtk_box_pack_start( GTK_BOX( window->content_hbox ), widget, FALSE, FALSE, 0 );
	details->statusbar = widget;

	/* create main content for window */
	window->xml = glade_xml_new( DATADIR"/neutrino/glade/neutrino.glade",
				     "window_content", NULL );

	content = glade_xml_get_widget( window->xml, "window_content" );

	gtk_drag_dest_set( GTK_WIDGET( content ),
			   GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP |
			   GTK_DEST_DEFAULT_MOTION,
			   drop_types, num_drop_types,
			   GDK_ACTION_COPY );
	g_object_set_data( G_OBJECT( content ), "window", window );

	view = details->list = gtk_tree_view_new();
	details->store = gtk_list_store_new( NEUTRINO_COLS, 
				G_TYPE_STRING, G_TYPE_INT, 
				G_TYPE_STRING, GDK_TYPE_PIXBUF );

        rend = gtk_cell_renderer_text_new();
	prend = gtk_cell_renderer_pixbuf_new();

	col = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start( col, prend, FALSE );
	gtk_tree_view_column_pack_start( col, rend, TRUE );
    
	gtk_tree_view_column_set_attributes( col, rend,
					"markup",
					 NEUTRINO_DISPLAY_COL,
					 NULL );
	gtk_tree_view_column_set_attributes( col, prend,
					 "pixbuf",
					 NEUTRINO_ICON_COL,
					 NULL);
        gtk_tree_view_append_column( GTK_TREE_VIEW( view ), col );

	gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( view ),
					   FALSE );
	gtk_tree_view_set_model( GTK_TREE_VIEW( view ),
				GTK_TREE_MODEL( details->store ) );
	/* use this so we get the default row highlighting */
	gtk_tree_view_enable_model_drag_dest( GTK_TREE_VIEW( view ),
			internal_drop_types, num_internal_drop_types,
			GDK_ACTION_LINK );
	g_signal_connect( G_OBJECT( view ), "drag_data_received",
			 G_CALLBACK( neutrino_window_playlist_drop ),
			 window );
	/* hookup to the motion signal, so we can prevent drops
	 * on non playlists */
	g_signal_connect( G_OBJECT( view ), "drag_motion",
			  G_CALLBACK( neutrino_window_playlist_motion ),
			  window );
	
	sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
					GTK_POLICY_NEVER,
					GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
			GTK_SHADOW_IN );
	gtk_container_add( GTK_CONTAINER( sw ), details->list );
	gtk_widget_show_all( sw );
	gtk_box_pack_start( GTK_BOX( contents ), 
			    sw, FALSE, FALSE, 6 );

	gtk_list_store_append( details->store, &it );
	title = g_strconcat( "<b>", _( "View" ), "</b>", NULL );
	gtk_list_store_set( details->store, &it,
			    NEUTRINO_DISPLAY_COL, title, 
			    NEUTRINO_TYPE_COL, NEUTRINO_VIEW_NONE,
			    -1 );
	g_free( title );
	pixbuf = gtk_widget_render_icon( details->list,
			"Neutrino_Local", GTK_ICON_SIZE_MENU,
			NULL );
	gtk_list_store_append( details->store, &it );
	gtk_list_store_set( details->store, &it,
			    NEUTRINO_DISPLAY_COL, _( "Local Files" ),
			    NEUTRINO_TYPE_COL, NEUTRINO_VIEW_LOCAL,
			    NEUTRINO_ICON_COL, pixbuf,
			    -1 );
	if( pixbuf ) {
		g_object_unref( pixbuf );
	}
	pixbuf = gtk_widget_render_icon( details->list,
			"Neutrino_Playlist", GTK_ICON_SIZE_MENU,
			NULL );
	gtk_list_store_append( details->store, &it );
	gtk_list_store_set( details->store, &it,
			NEUTRINO_DISPLAY_COL, _( "Play Queue" ),
			NEUTRINO_TYPE_COL, NEUTRINO_VIEW_PLAYLIST,
			NEUTRINO_DATA_COL, NULL,
			NEUTRINO_ICON_COL, pixbuf,
			-1 );
	if( pixbuf ) {
		g_object_unref( pixbuf );
	}
	
	gtk_list_store_append( details->store, &it );
	title = g_strconcat( "<b>", _( "Jukeboxes" ), "</b>", NULL );
	gtk_list_store_set( details->store, &it,
			    NEUTRINO_DISPLAY_COL, title, 
			    NEUTRINO_TYPE_COL, NEUTRINO_VIEW_NONE,
			    -1 );
	details->jukeboxes_path = gtk_tree_model_get_path( GTK_TREE_MODEL( details->store ), &it );
	g_free( title );

	neutrino_window_scan_jukeboxes( window );

	gtk_list_store_append( details->store, &it );
	title = g_strconcat( "<b>", _( "Playlists" ), "</b>", NULL );
	gtk_list_store_set( details->store, &it,
			    NEUTRINO_DISPLAY_COL, title, 
			    NEUTRINO_TYPE_COL, NEUTRINO_VIEW_NONE,
			    -1 );
	g_free( title );
	
	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );
	g_signal_connect( G_OBJECT( sel ), "changed",
			  G_CALLBACK( neutrino_window_view_select ),
			  window );

	details->sidebar = sw;

	gtk_box_pack_start( GTK_BOX( contents ), content, 
			    TRUE, TRUE, 6 );

	window->albums = glade_xml_get_widget( window->xml, "albums" );
	neutrino_window_build_album_view( window );

	window->tracks = glade_xml_get_widget( window->xml, "tracks" );
	neutrino_window_build_tracks_view( window );

	window->artists = glade_xml_get_widget( window->xml, "artists" );
	neutrino_window_build_artist_view( window );

	window->playqueue = glade_xml_get_widget( window->xml, "playqueue" );
	neutrino_window_build_playqueue_view( window );

	neutrino_window_build_file_manager( window );

	widget = glade_xml_get_widget( window->xml, "search_mode" );
	menu = gtk_menu_new();
	for( i = 0; search_modes[ i ]; ++ i ) {
		GtkWidget *item;

		item = gtk_menu_item_new_with_label( search_modes[ i ] );
		g_object_set_data( G_OBJECT( item ), "search_mode",
				   GINT_TO_POINTER( i ) );
		gtk_widget_show( item );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ), item );
	}
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );

	neutrino_window_set_jukebox( window, NULL );

	gtk_widget_show( window->content_hbox );

	gtk_container_add( GTK_CONTAINER( window ), 
			window->content_hbox );

	gtk_widget_show_all( contents );
	widget = glade_xml_get_widget( window->xml, "playqueuewindow" );
	gtk_widget_hide( widget );

	glade_xml_signal_autoconnect( window->xml );

	/* hook up to gconf for show/hide toggling */
	client = gconf_client_get_default();

	gconf_client_add_dir( client, "/apps/neutrino",
			      GCONF_CLIENT_PRELOAD_ONELEVEL, NULL );

	window->sidebar = 
		gconf_client_get_bool( client,
				       "/apps/neutrino/ui/show_sidebar",
				       NULL );
	window->show_sidebar =
		gconf_client_notify_add( client,
					 "/apps/neutrino/ui/show_sidebar",
					 neutrino_window_show_sidebar,
					 window, NULL, NULL );
	window->browser = 
		gconf_client_get_bool( client,
				       "/apps/neutrino/ui/show_browser",
				       NULL );
	window->show_browser =
		gconf_client_notify_add( client,
					 "/apps/neutrino/ui/show_browser",
					 neutrino_window_show_browser,
					 window, NULL, NULL );
	window->playback = 
		gconf_client_get_bool( client,
				       "/apps/neutrino/ui/show_playback",
				       NULL );
	window->show_playback =
		gconf_client_notify_add( client,
					 "/apps/neutrino/ui/show_playback",
					 neutrino_window_show_playback,
					 window, NULL, NULL );
	window->toolbar = 
		gconf_client_get_bool( client,
				       "/apps/neutrino/ui/show_toolbar",
				       NULL );
	window->show_toolbar =
		gconf_client_notify_add( client,
					 "/apps/neutrino/ui/show_toolbar",
					 neutrino_window_show_toolbar,
					 window, NULL, NULL );

	action = egg_action_group_get_action( details->group,
						"ViewSidebar" );
	egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ),
					window->sidebar );
	if( ! window->sidebar ) {
		gtk_widget_hide( details->sidebar );
	}

	action = egg_action_group_get_action( details->group,
						"ViewBrowser" );
	egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ),
					window->browser );
	if( ! window->browser ) {
		widget = glade_xml_get_widget( window->xml,
					       "artists_albums" );
		gtk_widget_hide( widget );
	}

	action = egg_action_group_get_action( details->group,
						"ViewPlayback" );
	egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ),
					window->playback );
	if( ! window->playback ) {
		widget = glade_xml_get_widget( window->xml,
					       "playback" );
		gtk_widget_hide( widget );
	}

	action = egg_action_group_get_action( details->group,
						"ViewToolbar" );
	egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ),
					window->toolbar );
	
	widget = egg_menu_merge_get_widget( details->merge,
					"/Main Toolbar" );
	if( widget && ! window->toolbar ) {
		gtk_widget_hide( widget );
	}
	
}

static void neutrino_window_build_album_view( NeutrinoWindow *window )
{
	GtkTreeView *view;
	GtkCellRenderer *rend;
	GtkTreeViewColumn *col;
	GtkTreeSelection *selection;

	view = GTK_TREE_VIEW( window->albums );

	g_object_set_data( G_OBJECT( view ), "window", window );

	selection = gtk_tree_view_get_selection( view );
	gtk_tree_selection_set_mode( selection, GTK_SELECTION_SINGLE );

	/* connect to change signal */
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( neutrino_window_album_select ),
			  window );

        rend = gtk_cell_renderer_text_new ();
        col = gtk_tree_view_column_new_with_attributes ( "Albums",
							 rend,
							 "markup",
							 ALBUM_NAME_COL,
							 NULL);
        gtk_tree_view_append_column( view, col );
	gtk_tree_view_column_set_sort_column_id( col, ALBUM_NAME_COL );

	gtk_tree_view_set_model( GTK_TREE_VIEW( view ), 
				 GTK_TREE_MODEL( window->details->view->albums ) );

	gtk_tree_view_set_headers_clickable( GTK_TREE_VIEW( view ), TRUE );
}

static void neutrino_window_build_artist_view( NeutrinoWindow *window )
{
	GtkTreeView *view;
	GtkCellRenderer *rend;
	GtkTreeViewColumn *col;

	GtkTreeSelection *selection;

	view = GTK_TREE_VIEW( window->artists );

	g_object_set_data( G_OBJECT( view ), "window", window );

	selection = gtk_tree_view_get_selection( view );
	gtk_tree_selection_set_mode( selection, GTK_SELECTION_SINGLE );

	/* connect to change signal */
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( neutrino_window_artist_select ),
			  window );

        rend = gtk_cell_renderer_text_new ();
        col = gtk_tree_view_column_new_with_attributes ( "Artists",
							 rend,
							 "markup", 
							 ARTIST_NAME_COL,
							 NULL);
        gtk_tree_view_append_column( view, col );
	gtk_tree_view_column_set_sort_column_id( col, ARTIST_NAME_COL );
	
	gtk_tree_view_set_model( GTK_TREE_VIEW( view ), 
				 GTK_TREE_MODEL( window->details->view->artists ) );

	gtk_tree_view_set_headers_clickable( GTK_TREE_VIEW( view ), TRUE );
}

static void neutrino_window_build_tracks_view( NeutrinoWindow *window )
{
	GtkTreeView *view;
	GtkCellRenderer *rend;
	GtkTreeViewColumn *col;

	GtkTreeSelection *selection;

	view = GTK_TREE_VIEW( window->tracks );

	g_object_set_data( G_OBJECT( view ), "window", window );

	selection = gtk_tree_view_get_selection( view );
	gtk_tree_selection_set_mode( selection, GTK_SELECTION_MULTIPLE );

	/* connect to change signal */
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( neutrino_window_track_select ),
			  window );

	rend = gtk_cell_renderer_text_new ();
        col = gtk_tree_view_column_new_with_attributes ( "Track",
							 rend,
							 "text", 
							 TRACK_TRACK_NUM_COL,
							 NULL);
	gtk_tree_view_column_set_sort_column_id( col, TRACK_TRACK_NUM_COL );
        gtk_tree_view_column_set_resizable( col, FALSE );
        gtk_tree_view_append_column( view, col );

        rend = gtk_cell_renderer_text_new ();
        col = gtk_tree_view_column_new_with_attributes ( "Title",
							 rend,
							 "text", 
							 TRACK_TITLE_COL,
							 NULL);
	gtk_tree_view_column_set_sort_column_id( col, TRACK_TITLE_COL );
        gtk_tree_view_column_set_resizable( col, TRUE );
        gtk_tree_view_append_column( view, col );

        rend = gtk_cell_renderer_text_new ();
        col = gtk_tree_view_column_new_with_attributes ( "Genre",
							 rend,
							 "text", 
							 TRACK_GENRE_COL,
							 NULL);
	gtk_tree_view_column_set_sort_column_id( col, TRACK_GENRE_COL );
        gtk_tree_view_column_set_resizable( col, FALSE );
        gtk_tree_view_append_column( view, col );

        rend = gtk_cell_renderer_text_new ();
        col = gtk_tree_view_column_new_with_attributes ( "Time",
							 rend,
							 "text", 
							 TRACK_LENGTH_COL,
							 NULL);
	gtk_tree_view_column_set_sort_column_id( col, TRACK_LENGTH_COL );
        gtk_tree_view_column_set_resizable( col, FALSE );
        gtk_tree_view_append_column( view, col );

        rend = gtk_cell_renderer_text_new ();
        col = gtk_tree_view_column_new_with_attributes ( "Year",
							 rend,
							 "text", 
							 TRACK_YEAR_COL,
							 NULL);
	gtk_tree_view_column_set_sort_column_id( col, TRACK_YEAR_COL );
        gtk_tree_view_column_set_resizable( col, FALSE );
        gtk_tree_view_append_column( view, col );

        rend = gtk_cell_renderer_text_new ();
        col = gtk_tree_view_column_new_with_attributes ( "Size",
							 rend,
							 "text", 
							 TRACK_SIZE_COL,
							 NULL);
	gtk_tree_view_column_set_sort_column_id( col, TRACK_SIZE_COL );
        gtk_tree_view_column_set_resizable( col, FALSE );
        gtk_tree_view_append_column( view, col );

        rend = gtk_cell_renderer_text_new ();
        col = gtk_tree_view_column_new_with_attributes ( "Codec",
							 rend,
							 "text", 
							 TRACK_CODEC_COL,
							 NULL);
	gtk_tree_view_column_set_sort_column_id( col, TRACK_CODEC_COL );
        gtk_tree_view_column_set_resizable( col, FALSE );
        gtk_tree_view_append_column( view, col );

	rend = gtk_cell_renderer_toggle_new ();
        col = gtk_tree_view_column_new_with_attributes ( "Play Only",
							 rend,
							 "active", 
							 TRACK_PLAY_ONLY_COL,
							 NULL);
        gtk_tree_view_column_set_resizable( col, FALSE );
        gtk_tree_view_append_column( view, col );

	gtk_tree_view_set_model( GTK_TREE_VIEW( view ), 
				 GTK_TREE_MODEL( window->details->view->tracks ) );

	gtk_tree_view_set_headers_clickable( GTK_TREE_VIEW( view ), TRUE );

	gtk_tree_view_columns_autosize( GTK_TREE_VIEW( view ) );

	gtk_drag_source_set( GTK_WIDGET( view ), GDK_BUTTON1_MASK,
			     drag_types, num_drag_types,
			     GDK_ACTION_COPY | GDK_ACTION_LINK );
}

static void neutrino_window_build_playqueue_view( NeutrinoWindow *window )
{
	GtkTreeView *view;
	GtkCellRenderer *rend;
	GtkTreeViewColumn *col;

	GtkTreeSelection *selection;

	view = GTK_TREE_VIEW( window->playqueue );

	g_object_set_data( G_OBJECT( view ), "window", window );

	window->details->queue = gtk_list_store_new( TRACK_MAX_COL,
					   G_TYPE_UINT,
					   G_TYPE_STRING,
					   G_TYPE_UINT,
					   G_TYPE_STRING,
					   G_TYPE_STRING,
					   G_TYPE_STRING,
					   G_TYPE_STRING,
					   G_TYPE_STRING,
					   G_TYPE_BOOLEAN );

	selection = gtk_tree_view_get_selection( view );
	gtk_tree_selection_set_mode( selection, GTK_SELECTION_SINGLE );

	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( neutrino_window_playqueue_select ),
			  window );

        rend = gtk_cell_renderer_text_new ();
        col = gtk_tree_view_column_new_with_attributes ( "Play Queue",
							 rend,
							 "text", 
							 1,
							 NULL);
        gtk_tree_view_append_column( view, col );

	gtk_tree_view_set_model( GTK_TREE_VIEW( view ), 
				 GTK_TREE_MODEL( window->details->queue ) );

	gtk_drag_dest_set( GTK_WIDGET( view ),
			   GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP |
			   GTK_DEST_DEFAULT_MOTION,
			   internal_drop_types, num_internal_drop_types,
			   GDK_ACTION_COPY );
}

static void neutrino_window_build_file_manager( NeutrinoWindow *window )
{
	NeutrinoFileBrowser *browser;
	GtkCellRenderer *rend;
	GtkCellRenderer *prend;
	GtkTreeViewColumn *col;
	GtkTreeSelection *selection;
	GConfClient *client;
	gchar *path;

	g_return_if_fail( NEUTRINO_IS_WINDOW( window ) );

	window->details->browser = browser = neutrino_file_browser_new();
	neutrino_file_browser_set_mode( browser, FILE_BROWSE_FLAT );

	window->filemanager = glade_xml_get_widget( window->xml, 
						    "filemanager" );
	g_object_set_data( G_OBJECT( window->filemanager ), "window",
			   window );

	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( window->filemanager ) );
	gtk_tree_selection_set_mode( selection, GTK_SELECTION_MULTIPLE );

	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( neutrino_window_track_select ),
			  window );

	prend = gtk_cell_renderer_pixbuf_new();
	rend = gtk_cell_renderer_text_new();
        window->col = col = gtk_tree_view_column_new();

        gtk_tree_view_column_pack_start( col, prend, FALSE );
        gtk_tree_view_column_pack_start( col, rend, TRUE );
        gtk_tree_view_append_column( GTK_TREE_VIEW( window->filemanager ),
				     col );
	gtk_tree_view_column_set_sort_column_id( GTK_TREE_VIEW_COLUMN( col ),
					FILE_BROWSER_NAME_COL );
	gtk_tree_view_column_set_sort_order( GTK_TREE_VIEW_COLUMN( col ),
			GTK_SORT_ASCENDING );
	
        gtk_tree_view_column_set_attributes( col, rend,
                                             "text", FILE_BROWSER_NAME_COL,
					     NULL );
        gtk_tree_view_column_set_attributes( col, prend,
                                             "pixbuf",
                                             FILE_BROWSER_ICON_COL,
                                             NULL );
	gtk_tree_view_column_set_resizable( col, TRUE );

	client = gconf_client_get_default();
	path = gconf_client_get_string( client,
					"/apps/neutrino/general/musicpath",
					NULL );
	if( ! path ) {
		path = g_strdup( "/" );
	}

	gtk_tree_view_column_set_title( col, path );
	neutrino_file_browser_scan_directory( browser, path, -1 );

	g_free( path );

	gtk_tree_view_set_model( GTK_TREE_VIEW( window->filemanager ),
				 neutrino_file_browser_get_model( browser ) );

	gtk_tree_view_columns_autosize(GTK_TREE_VIEW(window->filemanager));
}


static void neutrino_window_usage_received( NomadJukebox *jukebox,
					    guint64 total, guint64 free, 
					    guint64 used, gpointer data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}
		
	window->details->total_space = total;
	window->details->total_free = free;
	window->details->total_used = used;
}


static void neutrino_window_scanning_tracks( NomadJukebox *jukebox,
					     gpointer data )
{
	NeutrinoWindow *window;
	gchar *text;

	window = NEUTRINO_WINDOW( data );
	
	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	text = g_strdup( "Scanning Tracks..." );

	neutrino_window_set_status( window, text );

	g_free( text );

	window->details->total_tracks = 0;

	gdk_threads_leave();
}

static void neutrino_window_track_add( NomadJukebox *jukebox,
				       guint total_tracks,
				       NomadTrack *track,
				       gpointer data )
{

	NeutrinoWindow *window;
	gchar *text;

	gchar *artist;
	gchar *album;
	gchar *title;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	nomad_track_get( track,
			    &artist, &title, &album,
			    NULL, NULL, NULL, NULL, NULL, NULL,
			    NULL, NULL );

	text = g_strdup_printf( "Tracks: %i \t%s - %s - %s",
				total_tracks, 
				artist, album, title );
	g_free( artist );
	g_free( album );
	g_free( title );
	
	neutrino_window_set_status( window, text );

	g_free( text );
	
	window->details->total_tracks = total_tracks;
		
	gdk_threads_leave();
}

static void neutrino_window_scanned_tracks( NomadJukebox *jukebox,
					    guint total,
					    gpointer data )
{
	NeutrinoWindow *window;
	gchar *text;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	text = g_strdup_printf( "Total Tracks: %i", total );

	neutrino_window_set_status( window, text );

	g_free( text );
		
	window->details->total_tracks = total;

	gdk_threads_leave();
}

static void neutrino_window_scanning_playlists( NomadJukebox *jukebox,
						gpointer data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}
	window->details->total_playlists = 0;
}

static void neutrino_window_playlist_add( NomadJukebox *jukebox,
					  guint total_tracks,
					  const gchar *name,
					  gpointer data )
{
	NeutrinoWindow *window;
	GtkTreeIter it;
	GdkPixbuf *pixbuf;
	gchar *displayname;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	pixbuf = gtk_widget_render_icon( window->details->list,
			"Neutrino_Playlist", GTK_ICON_SIZE_MENU,
			NULL );

	displayname = g_markup_escape_text( name, strlen( name ) );
	
	gtk_list_store_append( window->details->store, &it );
	gtk_list_store_set( window->details->store, &it,
			NEUTRINO_DISPLAY_COL, displayname,
			NEUTRINO_TYPE_COL, NEUTRINO_VIEW_PLAYLIST,
			NEUTRINO_DATA_COL, name,
			NEUTRINO_ICON_COL, pixbuf,
			-1 );
	g_free( displayname );
	if( pixbuf ) {
		g_object_unref( pixbuf );
	}
	
	gdk_threads_leave();
}

static void neutrino_window_playlist_remove( NomadJukebox *jukebox,
					     const gchar *name,
					     gpointer data )
{
	NeutrinoWindow *window;
	GtkTreeModel *model;
	GtkTreeIter it;
	gint type;
	gchar *pname;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}
	
	model = GTK_TREE_MODEL( window->details->store );
	
	gdk_threads_enter();

	if( gtk_tree_model_get_iter_first( model, &it ) ) {
		pname = NULL;
		do {
			gtk_tree_model_get( model, &it,
					NEUTRINO_TYPE_COL, &type, -1 );
			if( type == NEUTRINO_VIEW_PLAYLIST ) {

				gtk_tree_model_get( model, &it,
						NEUTRINO_DATA_COL,
						&pname, -1 );
				if( pname &&
					! strcmp( name, pname ) ) {
					break;
				}
				g_free( pname );
				pname = NULL;
			}
		} while( gtk_tree_model_iter_next( model, &it ) );

		if( pname ) {
			g_free( pname );
			gtk_list_store_remove( window->details->store,
						&it );
		}
	}

	gdk_threads_leave();
}


static void neutrino_window_playlist_renamed( NomadJukebox *jukebox,
					      const gchar *orig,
					      const gchar *newname,
					      gpointer data )
{
	NeutrinoWindow *window;
	GtkTreeModel *model;
	GtkTreeIter it;
	gint type;
	gchar *pname;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}

	model = GTK_TREE_MODEL( window->details->store );
	
	gdk_threads_enter();

	if( gtk_tree_model_get_iter_first( model, &it ) ) {
		pname = NULL;
		do {
			gtk_tree_model_get( model, &it,
					NEUTRINO_TYPE_COL, &type, -1 );
			if( type == NEUTRINO_VIEW_PLAYLIST ) {

				gtk_tree_model_get( model, &it,
						NEUTRINO_DATA_COL,
						&pname, -1 );
				if( ! strcmp( orig, pname ) ) {
					break;
				}
				g_free( pname );
				pname = NULL;
			}
		} while( gtk_tree_model_iter_next( model, &it ) );

		if( pname ) {
			g_free( pname );
			gtk_list_store_set( window->details->store,
					&it,
					NEUTRINO_DISPLAY_COL, newname,
					NEUTRINO_DATA_COL, newname,
					-1 );
		}
	}

	gdk_threads_leave();
}

static void neutrino_window_scanned_playlists( NomadJukebox *jukebox,
					       guint total,
					       gpointer data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	window->details->total_playlists = total;

	neutrino_window_library_select( window->details->library,
					window );

	gdk_threads_leave();
}

static void neutrino_window_transfer( NomadJukebox *jukebox,
				      const gchar *uri,
				      gboolean upload,
				      guint64 sent,
				      guint64 total,
				      gpointer data )
{
	NeutrinoWindow *window;
	gchar *text;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	text = g_strdup_printf( "%s (%lld)", uri, sent );

	neutrino_window_set_status( window, text );

	g_free( text );

	gdk_threads_leave();
}

static void neutrino_window_downloaded_track( NomadJukebox *jukebox,
					      const gchar *uri,
					      NomadTrack *track,
					      gpointer data )
{
	NeutrinoWindow *window;
	gchar *text;
	GConfClient *client;
	gboolean tag;
	gboolean v1dot1;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	text = g_strdup_printf( "%s downloaded", uri );

	neutrino_window_set_status( window, text );

	g_free( text );

	/* do we want to write an ID3 Tag based on the track? */
	client = gconf_client_get_default();

	tag = gconf_client_get_bool( client,
				     "/apps/neutrino/download/metatoid3",
				     NULL );
	v1dot1 = gconf_client_get_bool( client,
					"/apps/neutrino/download/id3version1",
					NULL );

	if( tag ) {
		neutrino_mp3_tag_file( uri, track, v1dot1 );
	}

	g_object_unref( client );

	gdk_threads_leave();
}

static void neutrino_window_playlist_track_erased( NomadJukebox *jukebox,
						   const gchar *playlist,
						   const gchar *artist,
						   const gchar *album,
						   const gchar *title,
						   gpointer data )
{
	NeutrinoWindow *window;
	gchar *text;
	GtkWidget *widget;
	guint view;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}
	
	gdk_threads_enter();

	widget = glade_xml_get_widget( window->xml, "mode" );
	view = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( widget ), 
						    "view" ) );
	if( view == PLAYLIST_VIEW ) {
		const gchar *name;

		name = (const gchar*)g_object_get_data( G_OBJECT( widget ),
							"name" );
		g_assert( name );

		if( ! strcmp( name, playlist ) ) {
			/* deleted a track in the currently viewed
			   playlist */
			neutrino_jukebox_view_select_playlist( window->details->view, name );
		}
	}

	text = g_strdup_printf( "Erased %s from %s", title, playlist );

	neutrino_window_set_status( window, text );

	g_free( text );

	gdk_threads_leave();
}

static void neutrino_window_playlist_tracks_erased( NomadJukebox *jukebox,
						    const gchar *playlist,
						    gpointer data )
{
	NeutrinoWindow *window;
	GtkWidget *widget;
	guint view;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	widget = glade_xml_get_widget( window->xml, "mode" );
	view = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( widget ), 
						    "view" ) );
	if( view == PLAYLIST_VIEW ) {
		const gchar *name;

		name = (const gchar*)g_object_get_data( G_OBJECT( widget ),
							"name" );
		g_assert( name );

		if( ! strcmp( name, playlist ) ) {
			/* deleted a track in the currently viewed
			   playlist */
			neutrino_jukebox_view_select_playlist( window->details->view, name );
		}
	}

	gdk_threads_leave();
}

static void neutrino_window_track_remove( NomadJukebox *jukebox,
					  NomadTrack *track,
					  gpointer data )
{
	NeutrinoWindow *window;
	gchar *text;
	gchar *title;
	
	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	title = nomad_track_get_title( track );
	text = g_strdup_printf( "Erased %s", title );
	g_free( title );

	neutrino_window_set_status( window, text );
		
	g_free( text );	

	window->details->total_tracks --;

	gdk_threads_leave();
}

static void neutrino_window_set_time( NomadJukebox *jukebox,
				      const gchar *timestring,
				      gpointer data )
{
	gdk_threads_enter();

	gtk_label_set_text( GTK_LABEL( data ), timestring );

	gdk_threads_leave();
}

static void neutrino_window_eax( NomadJukebox *jukebox,
				 gpointer settings,
				 gpointer data )
{
	eax_t *eax;
	NeutrinoWindow *window;
	GladeXML *xml;
	GtkWidget *widget;

	GtkWidget *menu;
	gint i;

	GtkSizeGroup *group;
	static const gchar *labels[] = {
		"preset", "amount", "headphone", "rear",
		NULL
	};

	eax = (eax_t*)settings;
	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	xml = glade_xml_new( DATADIR"/neutrino/glade/neutrino.glade",
			     "eax", NULL );

	group = gtk_size_group_new( GTK_SIZE_GROUP_HORIZONTAL );
	for( i = 0; labels[ i ]; ++ i ) {
		widget = glade_xml_get_widget( xml, labels[ i ] );
		gtk_size_group_add_widget( group, widget );
	}

	widget = glade_xml_get_widget( xml, "eqactive" );
	g_object_set_data( G_OBJECT( widget ), "window", window );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
				      eax->eq_status );

	widget = glade_xml_get_widget( xml, "vol" );
	g_object_set_data( G_OBJECT( widget ), "window", window );
	if( eax->volume != 0 ) {
		gtk_range_set_value( GTK_RANGE( widget ),
				     (gdouble)( eax->volume / 5 ) );
	} else {
		gtk_range_set_value( GTK_RANGE( widget ), 0.0 );
	}

	widget = glade_xml_get_widget( xml, "bass" );
	gtk_widget_set_sensitive( widget, eax->eq_status );
	g_object_set_data( G_OBJECT( widget ), "window", window );
	gtk_range_set_range( GTK_RANGE( widget ),
			     (gdouble)eax->bassmin,
			     (gdouble)eax->bassmax );
	gtk_range_set_value( GTK_RANGE( widget ),
			     (gdouble)eax->bass );

	widget = glade_xml_get_widget( xml, "mid" );
	gtk_widget_set_sensitive( widget, eax->eq_status );
	g_object_set_data( G_OBJECT( widget ), "window", window );
	gtk_range_set_range( GTK_RANGE( widget ),
			     (gdouble)eax->midrangemin,
			     (gdouble)eax->midrangemax );
	gtk_range_set_value( GTK_RANGE( widget ),
			     (gdouble)eax->midrange );

	widget = glade_xml_get_widget( xml, "treble" );
	gtk_widget_set_sensitive( widget, eax->eq_status );
	g_object_set_data( G_OBJECT( widget ), "window", window );
	gtk_range_set_range( GTK_RANGE( widget ),
			     (gdouble)eax->treblemin,
			     (gdouble)eax->treblemax );
	gtk_range_set_value( GTK_RANGE( widget ),
			     (gdouble)eax->treble );

	widget = glade_xml_get_widget( xml, "midfreq" );
	gtk_widget_set_sensitive( widget, eax->eq_status );
	menu = gtk_menu_new();
	for( i = 0; i < eax->nfreq; ++ i ) {
		GtkWidget *item;
		gchar *label;
		
		label = g_strdup_printf( "%d Hz", eax->frequencies[ i ] );
		
		item = gtk_menu_item_new_with_label( label );
		g_object_set_data( G_OBJECT( item ), "midfreq",
				   GINT_TO_POINTER( i ) );
		g_object_set_data( G_OBJECT( item ), "window", window );
		
		g_free( label );

		gtk_widget_show( item );

		if( i == eax->freqset ) {
			gtk_menu_shell_prepend( GTK_MENU_SHELL( menu ),
						item );
		} else {
			gtk_menu_shell_append( GTK_MENU_SHELL( menu ),
					       item );
		}
	}
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );

	widget = glade_xml_get_widget( xml, "environment" );
	menu = gtk_menu_new();
	for( i = 0; i < eax->neffects; ++ i ) {
		GtkWidget *item;
				
		item = gtk_menu_item_new_with_label( eax->effects[ i ] );
		g_object_set_data( G_OBJECT( item ), "effect",
				   GINT_TO_POINTER( i ) );
		g_object_set_data( G_OBJECT( item ), "window", window );

		gtk_widget_show( item );

		if( i == eax->acteffect ) {
			gtk_menu_shell_prepend( GTK_MENU_SHELL( menu ),
						item );
		} else {
			gtk_menu_shell_append( GTK_MENU_SHELL( menu ),
					       item );
		}
	}
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );

	widget = glade_xml_get_widget( xml, "effect_percentage" );
	g_object_set_data( G_OBJECT( widget ), "window", window );
	gtk_range_set_value( GTK_RANGE( widget ),
			     (gdouble)eax->effamts[ eax->acteffect ] );

	widget = glade_xml_get_widget( xml, "headphones" );
	menu = gtk_menu_new();
	for( i = 0; i < eax->nphone; ++ i ) {
		GtkWidget *item;
				
		item = gtk_menu_item_new_with_label( eax->phonemodes[ i ] );
		g_object_set_data( G_OBJECT( item ), "headphone",
				   GINT_TO_POINTER( i ) );
		g_object_set_data( G_OBJECT( item ), "window", window );
		
		gtk_widget_show( item );

		if( i == eax->actphone ) {
			gtk_menu_shell_prepend( GTK_MENU_SHELL( menu ),
						item );
		} else {
			gtk_menu_shell_append( GTK_MENU_SHELL( menu ),
					       item );
		}
	}
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );

	widget = glade_xml_get_widget( xml, "rearspeaker" );
	menu = gtk_menu_new();
	for( i = 0; i < eax->nrear; ++ i ) {
		GtkWidget *item;
				
		item = gtk_menu_item_new_with_label( eax->rears[ i ] );
		g_object_set_data( G_OBJECT( item ), "rear",
				   GINT_TO_POINTER( i ) );
		g_object_set_data( G_OBJECT( item ), "window", window );

		gtk_widget_show( item );

		if( i == eax->actrear ) {
			gtk_menu_shell_prepend( GTK_MENU_SHELL( menu ),
						item );
		} else {
			gtk_menu_shell_append( GTK_MENU_SHELL( menu ),
					       item );
		}
	}
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );

	widget = glade_xml_get_widget( xml, "eax" );

	gtk_window_set_transient_for( GTK_WINDOW( widget ),
				      GTK_WINDOW( window ) );

	gtk_widget_show_all( widget );
	
	glade_xml_signal_autoconnect( xml );

	gdk_threads_leave();
}

static void neutrino_play_begin( NomadJukebox *jukebox,
				 NomadTrack *track, gpointer data )
{
	GtkWidget *widget;
	gchar *ttitle;
	gchar *title;
	gchar *artist;
	gchar *album;
	guint view;

	NeutrinoWindow *window;
		
	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	nomad_track_get( track,
			    &artist, &ttitle, &album,
			    NULL, NULL, NULL, NULL, NULL, NULL,
			    NULL, NULL );

	title = g_markup_escape_text( ttitle, strlen( ttitle ) );
	g_free( ttitle );
	ttitle = title;
	
	title = g_strconcat( "<span weight=\"bold\" size=\"large\">",
			     ttitle,
			     "</span>", NULL );
	g_free( ttitle );
	
	widget = glade_xml_get_widget( window->xml, "title" );
	gtk_label_set_markup( GTK_LABEL( widget ), title );
	
	widget = glade_xml_get_widget( window->xml, "artist" );
	gtk_label_set_text( GTK_LABEL( widget ), artist );
	g_free( artist );
	
	widget = glade_xml_get_widget( window->xml, "album" );
	gtk_label_set_text( GTK_LABEL( widget ), album );
	g_free( album );
	
	g_free( title );
	
	widget = glade_xml_get_widget( window->xml, "elapsed" );
	gtk_label_set_text( GTK_LABEL( widget ), "0:00" );
	
	widget = glade_xml_get_widget( window->xml, "play_progress" );
	gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( widget ), 0.0 );
	
	widget = glade_xml_get_widget( window->xml, "play_icon" );
	gtk_image_set_from_stock( GTK_IMAGE( widget ), "Neutrino_Play",
				  GTK_ICON_SIZE_SMALL_TOOLBAR );

	window->had_zero = FALSE;

	g_assert( window->play_progress == 0 );

	window->play_progress = 
		g_timeout_add( 1000, 
			       (GSourceFunc)neutrino_window_play_status,
			       window );

	widget = glade_xml_get_widget( window->xml, "mode" );
	view = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( widget ), 
						    "view" ) );
	switch( view ) {
	case JUKEBOX_VIEW:
	case SEARCH_VIEW:
		neutrino_window_library_sensitive( window );
		break;
	case PLAYLIST_VIEW:
		neutrino_window_playlist_sensitive( window );
		break;
	} 

	gdk_threads_leave();
}

static gboolean neutrino_window_play_status( NeutrinoWindow *window )
{
	nomad_jukebox_play_elapsed( window->jukebox );
	return TRUE;
}

static void neutrino_play_finished( NomadJukebox *jukebox, gpointer data )
{
	GtkWidget *widget;
	guint view;
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	if( window->play_progress != 0 ) {
		g_source_remove( window->play_progress );
		window->play_progress = 0;
	}

	widget = glade_xml_get_widget( window->xml, "elapsed" );
	gtk_label_set_text( GTK_LABEL( widget ), "0:00" );

	widget = glade_xml_get_widget( window->xml, "play_progress" );
	gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( widget ), 0.0 );

	widget = glade_xml_get_widget( window->xml, "play_icon" );
	gtk_image_set_from_stock( GTK_IMAGE( widget ), "Neutrino_Stop",
				 GTK_ICON_SIZE_SMALL_TOOLBAR );

	/* reselect current view */
	widget = glade_xml_get_widget( window->xml, "mode" );
	view = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( widget ), 
						    "view" ) );
	switch( view ) {
	case JUKEBOX_VIEW:
	case SEARCH_VIEW:
		neutrino_window_library_sensitive( window );
		break;
	case PLAYLIST_VIEW:
		neutrino_window_playlist_sensitive( window );
		break;
	} 

	gdk_threads_leave();
}

static void neutrino_play_progress( NomadJukebox *jukebox,
				    guint elapsed, gboolean changed,
				    NomadTrack *track, 
				    gpointer data )
{
	GtkWidget *widget;
	gdouble secs;
	gdouble percent;

	gchar *length;
	gchar *elapsetxt;

	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( data );

	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	if( elapsed > 0 ) {
		window->had_zero = TRUE;
	}

	if( track ) {
		elapsetxt = seconds_to_mmss( elapsed );
		length = nomad_track_get_length( track );
		secs = (gdouble) mmss_to_seconds( length );
		g_free( length );
		
		widget = glade_xml_get_widget( window->xml, "elapsed" );
		gtk_label_set_text( GTK_LABEL( widget ), elapsetxt );
		
		g_free( elapsetxt );
		
		if( elapsed > 0 ) {
			percent = ( elapsed / secs );
			/* incase the length field is wrong */
			if( percent > 1.0 ) {
				percent = 1.0;
			}
		} else {
			percent = 0.0;
		}
		
		/* main window display */
		widget = glade_xml_get_widget( window->xml, "play_progress" );
		gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( widget ), 
					       percent );
	}
	if( changed || ( window->had_zero && elapsed == 0 ) ) {
		window->had_zero = FALSE;
		/* use an idle source to avoid locking issues */
		g_idle_add( (GSourceFunc)neutrino_window_play_next,
				window );
	}

	gdk_threads_leave();
}

static void neutrino_window_artist_select( GtkTreeSelection *selection,
					   NeutrinoWindow *window )
{
	GtkTreeModel *model;
	GtkTreeIter it;

	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {
		GValue value = { 0 };
		const gchar *artist;
		GtkWidget *widget;

		widget = glade_xml_get_widget( window->xml, "search" );
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		gtk_tree_model_get_value( model, &it,
					  ARTIST_NAME_COL, &value );
		artist = g_value_get_string( &value );

		neutrino_jukebox_view_select_artist( window->details->view,
						     artist );

		g_value_unset( &value );
	} else {
		neutrino_jukebox_view_select_artist( window->details->view,
						     NULL );
	}
}

static void neutrino_window_album_select( GtkTreeSelection *selection,
					  NeutrinoWindow *window )
{
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkWidget *widget;
	gchar *title;

	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {
		GValue value = { 0 };
		const gchar *album;

		widget = glade_xml_get_widget( window->xml, "search" );
		gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		gtk_tree_model_get_value( model, &it,
					  ALBUM_NAME_COL, &value );
		album = g_value_get_string( &value );

		neutrino_jukebox_view_select_album( window->details->view,
						    album );

		g_value_unset( &value );
	
		title = g_strdup_printf( "Jukebox (%i) - Neutrino",
				 window->details->view->num );
		gtk_window_set_title( GTK_WINDOW( window ), title );
		g_free( title );

		title = g_strdup_printf( "<span weight=\"bold\" size=\"large\">Jukebox (%i)</span>", window->details->view->num );

	} else {
		neutrino_jukebox_view_select_album( window->details->view,
						    NULL );
		title = g_strdup( "Jukebox - Neutrino" );
		gtk_window_set_title( GTK_WINDOW( window ), title );
		g_free( title );
		
		title = g_strdup_printf( "<span weight=\"bold\" size=\"large\">Jukebox</span>" );

	}
	widget = glade_xml_get_widget( window->xml, "mode" );
	gtk_label_set_markup( GTK_LABEL( widget ), title );
	g_free( title );

}

static void neutrino_window_track_select( GtkTreeSelection *selection,
					  NeutrinoWindow *window )
{
	GtkWidget *widget;
	guint view;

	widget = glade_xml_get_widget( window->xml, "mode" );
	view = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( widget ), 
						    "view" ) );
	switch( view ) {
	case JUKEBOX_VIEW:
	case SEARCH_VIEW:
		neutrino_window_library_sensitive( window );
		break;
	case PLAYLIST_VIEW:
		neutrino_window_playlist_sensitive( window );
		break;
	} 
}

static void neutrino_window_playlist_select( const gchar *name,
					     NeutrinoWindow *window )
{
	GtkTreeModel *model;
	GtkWidget *widget;
	gchar *title;
	gchar *ename;
	
	widget = glade_xml_get_widget( window->xml, "artists_albums" );
	gtk_widget_hide( widget );

	widget = glade_xml_get_widget( window->xml, "search" );
	gtk_entry_set_text( GTK_ENTRY( widget ), "" );

	if( name ) {
		model = GTK_TREE_MODEL( window->details->view->tracks );
 
		neutrino_jukebox_view_select_playlist( window->details->view,
						       name );
		title = g_strdup_printf( "Playlist: %s (%i)- Neutrino",
				name, window->details->view->num );
		gtk_window_set_title( GTK_WINDOW( window ), title );
		g_free( title );

		ename = g_markup_escape_text( name, strlen( name ) );
		title = g_strdup_printf( "<span weight=\"bold\" size=\"large\">Playlist: %s (%i)</span>", ename, window->details->view->num );
		g_free( ename );
	} else {
		model = GTK_TREE_MODEL( window->details->queue );
	
		title = g_strdup_printf( "%s - Neutrino",
				_( "Play Queue" ) );
		gtk_window_set_title( GTK_WINDOW( window ), title );
		g_free( title );

		title = g_strdup_printf( "<span weight=\"bold\" size=\"large\">%s</span>", _( "Play Queue" ) );
	}

	gtk_tree_view_set_model( GTK_TREE_VIEW( window->tracks ),
				model );
	
	widget = glade_xml_get_widget( window->xml,
				       "mode" );
	
	gtk_label_set_markup( GTK_LABEL( widget ), title );
	g_free( title );
	g_object_set_data( G_OBJECT( widget ), "view",
			   GUINT_TO_POINTER( PLAYLIST_VIEW ) );
	if( name ) {
		g_object_set_data( G_OBJECT( widget ), "name",
				   g_strdup( name ) );
	} else {
		g_object_set_data( G_OBJECT( widget ), "name",
				   NULL );
	}
	
	neutrino_window_playlist_sensitive( window );
}

static void neutrino_window_library_select( GtkWidget *button,
					    NeutrinoWindow *window )
{
	GtkWidget *widget;
	GtkTreeSelection *selection;
	const gchar *search;

	gtk_tree_view_set_model( GTK_TREE_VIEW( window->tracks ),
				GTK_TREE_MODEL( window->details->view->tracks ) );
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( window->albums ) );
	
	widget = glade_xml_get_widget( window->xml, "viewbook" );
	gtk_notebook_set_current_page( GTK_NOTEBOOK( widget ), 0 );
	
	widget = glade_xml_get_widget( window->xml, "artists_albums" );
	if( window->browser ) {
			gtk_widget_show( widget );
	}
	
	widget = glade_xml_get_widget( window->xml, "search" );
	search = gtk_entry_get_text( GTK_ENTRY( widget ) );
	if( *search == '\0' ) {
		gchar *title;
		widget = glade_xml_get_widget( window->xml,
					       "mode" );
		g_object_set_data( G_OBJECT( widget ), "view",	
					   GUINT_TO_POINTER( JUKEBOX_VIEW ) );
		neutrino_window_album_select( selection, window );
		title = g_strdup_printf( "<span weight=\"bold\" size=\"large\">Jukebox (%i)</span>", window->details->view->num );
		gtk_label_set_markup( GTK_LABEL( widget ), title );
		g_free( title );	

		title = g_strdup_printf( "Jukebox (%i) - Neutrino",
				window->details->view->num );
		gtk_window_set_title( GTK_WINDOW( window ), title );
		g_free( title );
	}
	
	neutrino_window_library_sensitive( window );
}

static void neutrino_window_local_select( GtkWidget *button,
					  NeutrinoWindow *window )
{
	GtkWidget *widget;

	widget = glade_xml_get_widget( window->xml, "viewbook" );
	gtk_notebook_set_current_page( GTK_NOTEBOOK( widget ), 1 );
	
	widget = glade_xml_get_widget( window->xml, "artists_albums" );
	if( window->browser ) {
		gtk_widget_show( widget );
	}
	
	neutrino_window_library_sensitive( window );
}

static void neutrino_window_library_sensitive( NeutrinoWindow *window )
{
	gint i;
	static const gchar* elements[] = {
		"Scan", "1",
		"NewPlaylist", "1",
		"JukeboxInfo", "1",
		"EAX", "1",
		"DeletePlaylist", "0",
		"RenamePlaylist", "0",
		NULL
	};
	static const gchar *trackbased[] = {
		"Download",
		"Delete",
		"EditMeta",
		"AddToPlaylist",
		"playback",
		"Queue",
		NULL
	};
	static const gchar *playbased[] = {
		"stop",
		"prev",
		"next",
		NULL
	};
	static const gchar *overide[] = {
		"Download",
		"Upload",
		"Delete",
		NULL
	};
	GtkTreeSelection *selection;
	gboolean has;
	EggAction *action;

	for( i = 0; playbased[ i ];  i ++ ) {
		action = egg_action_group_get_action( window->details->group, playbased[ i ] );
		g_object_set( G_OBJECT( action ), "sensitive",
				window->jukebox &&
				window->play_progress != 0, NULL );
	}

	selection = 
		gtk_tree_view_get_selection( GTK_TREE_VIEW( window->tracks ) );

	has = FALSE;
  	gtk_tree_selection_selected_foreach( selection, 
					     neutrino_has_selection,
					     &has );
	has = ( has && window->jukebox );
	/* enable/disable track based */
	for( i = 0; trackbased[ i ];  i ++ ) {
		action = egg_action_group_get_action( window->details->group, trackbased[ i ] );
		g_object_set( G_OBJECT( action ), "sensitive",
				has, NULL );
	}

	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( window->filemanager ) );
	has = FALSE;
  	gtk_tree_selection_selected_foreach( selection, 
					     neutrino_has_selection,
					     &has );
	has = ( has && window->jukebox );
	action = egg_action_group_get_action( window->details->group,
						"Upload" );
	g_object_set( G_OBJECT( action ), "sensitive",
				has, NULL );

	for( i = 0; elements[ i ];  i += 2 ) {
		action = egg_action_group_get_action( window->details->group, elements[ i ] );
		has = ( ( window->jukebox ) &&
			( *elements[ i + 1 ] == '1' ) );
		g_object_set( G_OBJECT( action ), "sensitive", has, NULL );
	}

	/* overide settings if we are playing */
	for( i = 0; overide[ i ];  i ++ ) {
		action = egg_action_group_get_action( window->details->group, overide[ i ] );
		g_object_set( G_OBJECT( action ), "sensitive",
				window->jukebox &&
				window->play_progress == 0, NULL );
	}
}

static void neutrino_window_playlist_sensitive( NeutrinoWindow *window )
{
	gint i;
	static const gchar* elements[] = {
		"Scan", "1",
		"NewPlaylist", "1",
		"JukeboxInfo", "1",
		"EAX", "1",
		"DeletePlaylist", "1",
		"RenamePlaylist", "1",
		NULL
	};
	static const gchar *trackbased[] = {
		"Download",
		"Delete",
		"EditMeta",
		"AddToPlaylist",
		"playback",
		"Queue",
		NULL
	};
	static const gchar *playbased[] = {
		"stop",
		"prev",
		"next",
		NULL
	};
	static const gchar *overide[] = {
		"Download",
		"Upload",
		"Delete",
		NULL
	};
	GtkTreeSelection *selection;
	gboolean has;
	GtkTreeModel *model;
	EggAction *action;

	model = gtk_tree_view_get_model( GTK_TREE_VIEW( window->tracks ) );
	
	for( i = 0; playbased[ i ];  i ++ ) {
		action = egg_action_group_get_action( window->details->group, playbased[ i ] );
		g_object_set( G_OBJECT( action ), "sensitive",
				window->jukebox &&
			window->play_progress != 0, NULL );
	}

	selection = 
		gtk_tree_view_get_selection( GTK_TREE_VIEW( window->tracks ) );

	has = FALSE;
  	gtk_tree_selection_selected_foreach( selection, 
					     neutrino_has_selection,
					     &has );
	has = ( has && window->jukebox );
	for( i = 0; trackbased[ i ];  i ++ ) {
		action = egg_action_group_get_action( window->details->group, trackbased[ i ] );
		g_object_set( G_OBJECT( action ), "sensitive",
				has, NULL );
	}

	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( window->filemanager ) );
	has = FALSE;
  	gtk_tree_selection_selected_foreach( selection, 
					     neutrino_has_selection,
					     &has );
	has = ( has && window->jukebox );

	action = egg_action_group_get_action( window->details->group,
						"Upload" );
	g_object_set( G_OBJECT( action ), "sensitive",
				has, NULL );

	for( i = 0; elements[ i ];  i += 2 ) {
		action = egg_action_group_get_action( window->details->group, elements[ i ] );
		has = ( ( window->jukebox ) &&
			( *elements[ i + 1 ] == '1' ) );
		g_object_set( G_OBJECT( action ), "sensitive",
				has, NULL );
	}
	
	/* overide settings if we are playing */
	for( i = 0; overide[ i ];  i ++ ) {
		action = egg_action_group_get_action( window->details->group, overide[ i ] );
		g_object_set( G_OBJECT( action ), "sensitive",
			window->jukebox &&
			window->play_progress == 0, NULL );
	}

	/* override queue and delete sensitivity if the play queue
	 * is being display */
	if( model == GTK_TREE_MODEL( window->details->queue ) ) {
		action = egg_action_group_get_action( window->details->group, "Queue" );
		g_object_set( G_OBJECT( action ), "sensitive",
				FALSE, NULL );
		action = egg_action_group_get_action( window->details->group, "Delete" );
		g_object_set( G_OBJECT( action ), "sensitive",
				TRUE, NULL );
	}
}


static void neutrino_window_playqueue_select( GtkTreeSelection *selection,
					      NeutrinoWindow *window )
{

	GtkTreeModel *model;
	GtkTreeIter it;

	nomad_jukebox_play_stop( window->jukebox );

	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {
		guint id;

		gtk_tree_model_get( model, &it,
				  TRACK_TRACK_ID_COL, &id,
				  -1 );

		nomad_jukebox_play_track( window->jukebox, id );
	}
}

static void neutrino_window_play_random( NeutrinoWindow *window )
{
	GtkTreeSelection *selection;
	GtkTreeView *view;
	GtkTreeModel *model;
	GtkTreeIter it;
	guint tracknum;

	model = GTK_TREE_MODEL( window->details->queue );
	view = GTK_TREE_VIEW( window->playqueue );
	selection = gtk_tree_view_get_selection( view );

	/* yes this sucks and isn't very random */
	tracknum = g_random_int_range( 0, window->playqueue_size );

	if( gtk_tree_model_get_iter_first( model, &it ) ) {
		while( tracknum > 0 ) {
			gtk_tree_model_iter_next( model, &it );
			tracknum --;
		}
		gtk_tree_selection_select_iter( selection, &it );
	}
}

static void neutrino_window_show_sidebar( GConfClient *client,
					  guint cnxn_id,
					  GConfEntry *entry,
					  gpointer data )
{
	NeutrinoWindow *window;
	gboolean state;
	EggAction *action;

	window = NEUTRINO_WINDOW( data );

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		GtkWidget *widget;

		widget = window->details->sidebar;

		state = gconf_value_get_bool( entry->value );

		if( state != window->sidebar ) {
			window->sidebar = state;
			gdk_threads_enter();
			if( state ) {
				gtk_widget_show( widget );
			} else {
				gtk_widget_hide( widget );
			}
			action = egg_action_group_get_action( window->details->group, "ViewSidebar" );
			egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ), state );

			gdk_threads_leave();
		}	
	}
}

static void neutrino_window_show_browser( GConfClient *client,
					  guint cnxn_id,
					  GConfEntry *entry,
					  gpointer data )
{
	NeutrinoWindow *window;
	gboolean state;
	EggAction *action;
	
	window = NEUTRINO_WINDOW( data );

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		GtkWidget *widget;
	
		widget = glade_xml_get_widget( window->xml, "artists_albums" );

		state = gconf_value_get_bool( entry->value );
		if( state != window->browser ) {
			gdk_threads_enter();
			window->browser = state;
			if( state ) {
				gtk_widget_show( widget );
			} else {
				gtk_widget_hide( widget );
			}
			action = egg_action_group_get_action( window->details->group, "ViewBrowser" );
			egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ), state );

			gdk_threads_leave();
		}	
	}
}

static void neutrino_window_show_playback( GConfClient *client,
					   guint cnxn_id,
					   GConfEntry *entry,
					   gpointer data )
{
	NeutrinoWindow *window;
	gboolean state;
	EggAction *action;
	
	window = NEUTRINO_WINDOW( data );

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		GtkWidget *widget;
	
		widget = glade_xml_get_widget( window->xml, "playback" );

		state = gconf_value_get_bool( entry->value );
		if( state != window->playback ) {
			gdk_threads_enter();
			window->playback = state;
			if( state ) {
				gtk_widget_show( widget );
			} else {
				gtk_widget_hide( widget );
			}
			action = egg_action_group_get_action( window->details->group, "ViewPlayback" );
			egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ), state );
			gdk_threads_leave();
		}	
	}
}

static void neutrino_window_show_toolbar( GConfClient *client,
					  guint cnxn_id,
					  GConfEntry *entry,
					  gpointer data )
{
	NeutrinoWindow *window;
	gboolean state;
	EggAction *action;
	GtkWidget *widget;
	
	window = NEUTRINO_WINDOW( data );

	if( entry->value && entry->value->type == GCONF_VALUE_BOOL ) {
		state = gconf_value_get_bool( entry->value );
		if( state != window->toolbar ) {
			gdk_threads_enter();
			window->toolbar = state;

			action = egg_action_group_get_action( window->details->group, "ViewToolbar" );
			egg_toggle_action_set_active( EGG_TOGGLE_ACTION( action ), state );
			widget = egg_menu_merge_get_widget( window->details->merge, "/Main Toolbar" );
			if( state ) {
				gtk_widget_show( widget );
			} else {
				gtk_widget_hide( widget );
			}
			gdk_threads_leave();
		}	
	}
}


static GList *neutrino_window_get_selected_track_ids( NeutrinoWindow *window )
{
	GList *ret;
	GList *tmp;
	GtkTreeView *tree;
	GtkTreeSelection *selection;
	GtkTreeModel *model;

	ret = NULL;
	
	tree = GTK_TREE_VIEW( window->tracks );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( tree ) );
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( tree ) );

 	gtk_tree_selection_selected_foreach( selection, 
					     neutrino_util_get_selection,
					     &ret );
	/* ret now contains a list of iterators, we do this
	   as the get selection callback doesn't know the column
	   we want to pull the track from */
	for( tmp = ret; tmp; tmp = tmp->next ) {
		GValue value = { 0 };
		GtkTreeIter *it;
		guint id;

		it = (GtkTreeIter*)tmp->data;

		gtk_tree_model_get_value( model, it, TRACK_TRACK_ID_COL,
					  &value );
		id = g_value_get_uint( &value );

		tmp->data = GUINT_TO_POINTER( id );

		g_value_unset( &value );
		gtk_tree_iter_free( it );

	}

	return ret;
}

static void neutrino_window_upload_from_list( NeutrinoWindow *window, 
					      GList *tmp )
{
	GConfClient *client;
	GList *list;
	EggAction *action;
	
	client = gconf_client_get_default();

	/* list now contains a list of filenames */
	for( list = tmp, tmp = NULL; list; list = list->next ) {
		NomadTrack *track;
		const gchar *uri;
		const gchar *mime_type;
		GnomeVFSURI *vfsuri;
		gboolean local;

		uri = (const gchar *)list->data;
#ifndef HAVE_GNOME_VFS
		local = TRUE;

		if( ! strncmp( "file:", uri, strlen( "file:" ) ) ) {
			uri += strlen( "file:" );
		}
		
		mime_type = strrchr( uri, '.' );
		if( mime_type ) {
			if( ! strcmp( ".mp3", mime_type ) ) {
				mime_type = "audio/mp3";
			} else if( ! strcmp( ".wav", mime_type ) ) {
				mime_type = "audio/x-wav";
			} else if( ! strcmp( ".wma", mime_type ) ) {
				mime_type = "audio/x-ms-wma";
			} else if( ! strcmp( ".mpu", mime_type ) ) {
				mime_type = "audio/x-mpegurl";
			} else {
				mime_type = NULL;
			}
		}
#else
		vfsuri = gnome_vfs_uri_new( uri );
		if( vfsuri ) {
			local = gnome_vfs_uri_is_local( vfsuri );
		} else {
			local = TRUE;
		}
		if( ! local ) {
			/* FIXME: we need to make a temp copy of uri */

		}
		
		mime_type = gnome_vfs_get_mime_type( uri );
#endif
		if( mime_type ) {
			if( ! strcmp( "audio/mpeg", mime_type ) ||
			    ! strcmp( "audio/x-mp3", mime_type ) ||
			    ! strcmp( "audio/mp3", mime_type ) ) {
				/* MP3 audio */
				track = neutrino_mp3_track( uri );
			} else if( ! strcmp( "audio/x-wav", mime_type ) ) {
				/* WAV file */
				track = neutrino_wav_track( uri );
			} else if( ! strcmp( "audio/x-ms-wma", mime_type ) ) {
				/* WMA file */
				track = neutrino_wma_track( uri );
			} else if( ! strcmp( "audio/x-mpegurl", mime_type ) ) {
				/* mpu file */
				neutrino_mp3_read_mpu( uri, list );
				track = NULL;
			} else {
				/* unsupported audio format - we don't support
				   data yet */
				track = NULL;
			}
		} else {
			track = NULL;
		}
		
		if( track ) {
			/* we want to have at least we have artist,
			   album, and title */
			gchar *title;
			gchar *artist;
			gchar *album;
			gboolean show;

			title = nomad_track_get_title( track );
			artist = nomad_track_get_artist( track );
			album = nomad_track_get_album( track );
			
			show = gconf_client_get_bool( client,
						      "/apps/neutrino/upload/prompt",
						      NULL );
			if( show &&
			    ( ! strcmp( "<Unknown>", title ) ||
			      ! strcmp( "<Unknown>", artist ) ||
			      ! strcmp( "<Unknown>", album ) ) ) {
				/* show meta info dialog */
				NomadTrack *ntrack;

				ntrack = neutrino_window_edit_track( window,
								     track );
				if( ntrack ) {
					nomad_track_free( track );
					track = ntrack;
				}
			}

			if( track ) {
				tmp = g_list_append( tmp, track );
			}

			g_free( title );
			g_free( artist );
			g_free( album );
		}
	}

	if( tmp ) {
		window->num_transfers ++;

		action = egg_action_group_get_action( window->details->group, "Cancel" );
		g_object_set( G_OBJECT( action ), "sensitive", TRUE,
				NULL );
		nomad_jukebox_upload( window->jukebox, tmp );

		g_list_free( tmp );
	}

	g_object_unref( client );
}

static void neutrino_window_queue_track( NeutrinoWindow *window,
					 guint trackid )
{
	GtkListStore *store;
	GtkTreeIter it;
	NomadTrack *track;
	gchar *title;
	gchar *trck;
	gchar *genre;
	gchar *length;
	gchar *year;
	gchar *size;
	gchar *codec;

	store = GTK_LIST_STORE( window->details->queue );
	track = nomad_jukebox_request_track( window->jukebox, trackid );
	g_assert( track );

	title = nomad_track_get_title( track );

	nomad_track_get( track, NULL, &title, NULL,
			    &genre, &length, &year, &size, &codec,
			    &trck, NULL, NULL );

	gtk_list_store_append( store, &it );
	gtk_list_store_set( store, &it,
			    TRACK_TITLE_COL, title,
			    TRACK_TRACK_NUM_COL,
			    atoi( trck ),
			    TRACK_GENRE_COL, genre,
			    TRACK_LENGTH_COL, length,
			    TRACK_YEAR_COL, year,
			    TRACK_SIZE_COL, size,
			    TRACK_CODEC_COL, codec,
			    TRACK_PLAY_ONLY_COL,
			    nomad_track_get_play_only( track ),
			    TRACK_TRACK_ID_COL, trackid, 
			    -1 );

	g_free( title );
	g_free( trck );
	g_free( genre );
	g_free( length );
	g_free( year );
	g_free( size );
	g_free( codec );

	nomad_track_free( track );

	window->playqueue_size ++;
}

static void neutrino_window_error( NomadJukebox *jukebox,
				   const gchar *errmsg,
				   NeutrinoWindow *window )
{
	if( jukebox != window->jukebox ) {
		return;
	}

	gdk_threads_enter();

	neutrino_window_set_status( window, errmsg );

	gdk_threads_leave();
}

static void neutrino_window_transfer_complete( NomadJukebox *jukebox,
					       NeutrinoWindow *window )
{
	EggAction *action;
	
	if( jukebox != window->jukebox ) {
		return;
	}

	window->num_transfers --;
	if( window->num_transfers == 0 ) {
		action = egg_action_group_get_action( window->details->group, "Cancel" );
		g_object_set( G_OBJECT( action ), "sensitive", FALSE,
				NULL );
	}
}

NomadTrack *neutrino_window_edit_track( NeutrinoWindow *window,
					   NomadTrack *track )
{
	GladeXML *xml;
	GtkWidget *widget;

	guint tracknum = 0;
	guint id;
	gchar *artist;
	gchar *title;
	gchar *album;
	gchar *genre;
	gchar *length;
	gchar *year;
	gchar *size;
	gchar *codec;
	gchar *trackno;
	gchar *name;
	gint response;
	gint i;
	GtkWidget *menu;
	GtkWidget *item;

	NomadTrack *ntrack;

	xml = glade_xml_new( DATADIR"/neutrino/glade/neutrino.glade",
			     "edit_metadata", NULL );

	nomad_track_get( track,
			    &artist, &title, &album,
			    &genre, &length, &year,
			    &size, &codec, &trackno,
			    &name, &id );
	
	widget = glade_xml_get_widget( xml, "artist" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
	gtk_entry_set_text( GTK_ENTRY( widget ), artist );
	g_free( artist );

	widget = glade_xml_get_widget( xml, "title" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
	gtk_entry_set_text( GTK_ENTRY( widget ), title );
	g_free( title );
	
	widget = glade_xml_get_widget( xml, "album" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
	gtk_entry_set_text( GTK_ENTRY( widget ), album );
	g_free( album );

	widget = glade_xml_get_widget( xml, "length" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
	gtk_entry_set_text( GTK_ENTRY( widget ), length );
	g_free( length );
	
	widget = glade_xml_get_widget( xml, "year" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
	gtk_entry_set_text( GTK_ENTRY( widget ), year );
	g_free( year );
	
	widget = glade_xml_get_widget( xml, "size" );
	gtk_label_set_text( GTK_LABEL( widget ), size );
	
	widget = glade_xml_get_widget( xml, "codec" );
	gtk_label_set_text( GTK_LABEL( widget ), codec );
	
	widget = glade_xml_get_widget( xml, "track" );
	tracknum = atoi( trackno );

	gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ),
				   (gdouble)tracknum );
	
	widget = glade_xml_get_widget( xml, "name" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
	gtk_entry_set_text( GTK_ENTRY( widget ), name );
	g_free( name );
	
	widget = glade_xml_get_widget( xml, "genre" );
	menu = gtk_menu_new();
	item = gtk_menu_item_new_with_label( genre );
	gtk_widget_set_name( item, genre );
	gtk_widget_show( item );
	gtk_menu_shell_append( GTK_MENU_SHELL( menu ), item );
	item = gtk_menu_item_new_with_label( "(None)" );
	gtk_widget_set_name( item, "(None)" );
	gtk_widget_show( item );
	gtk_menu_shell_append( GTK_MENU_SHELL( menu ), item );
	for( i = 0; i < genre_count; ++ i ) {
		const gchar *genre_name;
		
		genre_name = neutrino_id3_genre_table[ i ];

		if( *genre_name && 
		    strcmp( genre, genre_name ) ) {
			item = gtk_menu_item_new_with_label( genre_name );
			gtk_widget_set_name( item,
					     genre_name );
			gtk_widget_show( item );
			gtk_menu_shell_append( GTK_MENU_SHELL( menu ),
					       item );
		}
	}
	g_free( genre );
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ),
				  menu );
	
	widget = glade_xml_get_widget( xml, "edit_metadata" );
	
	gtk_window_set_transient_for( GTK_WINDOW( widget ),
				      GTK_WINDOW( window ) );
	response = gtk_dialog_run( GTK_DIALOG( widget ) );
	
	if( response == GTK_RESPONSE_OK ) {
		widget = glade_xml_get_widget( xml, "artist" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
		artist = (gchar*)gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		widget = glade_xml_get_widget( xml, "title" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
		title = (gchar*)gtk_entry_get_text( GTK_ENTRY(widget));
		
		widget = glade_xml_get_widget( xml, "album" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
		album = (gchar*)gtk_entry_get_text( GTK_ENTRY(widget));
		
		widget = glade_xml_get_widget( xml, "length" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
		length = (gchar*)gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		widget = glade_xml_get_widget( xml, "year" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
		year = (gchar*)gtk_entry_get_text( GTK_ENTRY(widget) );
		
		widget = glade_xml_get_widget( xml, "track" );
		tracknum=gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widget));
		trackno = g_strdup_printf( "%d", tracknum );
		
		widget = glade_xml_get_widget( xml, "name" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget) );
		name = (gchar*)gtk_entry_get_text( GTK_ENTRY(widget) );
		
		widget = glade_xml_get_widget( xml, "genre" );
		item = GTK_OPTION_MENU( widget )->menu_item;
		genre = (gchar*)gtk_widget_get_name( item );
		
		ntrack = nomad_track_new_with_id( id,
						     artist, 
						     title,
						     album, 
						     genre,
						     length,
						     year,
						     size,
						     codec,
						     trackno,
						     name );
		g_free( trackno );
	} else {
		ntrack = NULL;
	}
	
	g_free( size );
	g_free( codec );

	widget = glade_xml_get_widget( xml, "edit_metadata" );
	gtk_widget_destroy( widget );
	g_object_unref( G_OBJECT( xml ) );

	return ntrack;
}

static void neutrino_window_view_select( GtkTreeSelection *selection,
					 gpointer data )
{
	NeutrinoWindow *window;
	NomadJukebox *jukebox;
	GtkTreeModel *model;
	GtkTreeIter it;
	gint type;
	gchar *cdata;
	
	window = NEUTRINO_WINDOW( data );
	

	if( gtk_tree_selection_get_selected( selection, &model,
						 &it ) ) {
		gtk_tree_model_get( model, &it,
				NEUTRINO_TYPE_COL, &type,
				NEUTRINO_DATA_COL, &cdata,
				-1 );

		switch( type ) {
			case NEUTRINO_VIEW_JUKEBOX:
				jukebox = neutrino_shell_get_jukebox_by_id( window->shell, cdata );
				gdk_threads_leave();
				neutrino_window_set_jukebox( window, jukebox );
				gdk_threads_enter();
				neutrino_window_library_select( window->details->library, window );
				break;
			case NEUTRINO_VIEW_LOCAL:
				neutrino_window_local_select( NULL, window );
				break;
			case NEUTRINO_VIEW_PLAYLIST:
				neutrino_window_playlist_select( cdata, window );
			break;
		}

		g_free( cdata );
	}

}

static void neutrino_window_set_initial_playlists( gpointer key,
						      gpointer value,
						      gpointer data )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( data );

	neutrino_window_playlist_add( window->jukebox,
				window->details->total_playlists,
				(const gchar*)key,
				window );
}

static void neutrino_window_set_playlists( NeutrinoWindow *window )
{
	GtkTreeModel *model;
	GtkTreeIter it;
	GSList *list;
	GSList *tmp;
	GHashTable *table;
	gint type;
	gchar *name;
	
	model = GTK_TREE_MODEL( window->details->store );
	
	if( gtk_tree_model_get_iter_first( model, &it ) ) {
		list = NULL;
		do {
			gtk_tree_model_get( model, &it,
					NEUTRINO_TYPE_COL, &type,
					NEUTRINO_DATA_COL, &name,
					-1 );
			if( type == NEUTRINO_VIEW_PLAYLIST && name ) {
				list = g_slist_prepend( list,
						gtk_tree_iter_copy( &it ) );
			}
			g_free( name );
		} while( gtk_tree_model_iter_next( model, &it ) );

		for( tmp = list; tmp; tmp = tmp->next ) {
			gtk_list_store_remove( window->details->store,
					tmp->data );
			gtk_tree_iter_free( tmp->data );
		}
		g_slist_free( list );
	}

	if( window->jukebox ) {
		table = nomad_jukebox_request_playlists( window->jukebox );
		g_hash_table_foreach( table,
				      (GHFunc)neutrino_window_set_initial_playlists,
				      window );

	}
}

static void neutrino_window_playlist_drop( GtkWidget *widget, 
				GdkDragContext *context,
				gint x, gint y, 
				GtkSelectionData *sel_data,
				guint info, guint time, 
				gpointer data )
{
	NeutrinoWindow *window;
	NomadJukebox *jukebox;
	gchar **uris;
	guint i;
	GtkTreeModel *model;
       	const gchar *idstring;
	GList *list;
	GtkTreePath *path;
	GtkTreeIter it;
	gint type;
	gchar *name;
	GtkTreeViewDropPosition dpos;
	
	g_signal_stop_emission_by_name( G_OBJECT( widget ),
                                        "drag_data_received" );

	window = NEUTRINO_WINDOW( data );

	jukebox = NOMAD_JUKEBOX( window->jukebox );
	
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	gtk_tree_view_get_dest_row_at_pos( GTK_TREE_VIEW( widget ),
			x, y, &path, &dpos );

	if( ! path ) {
		gtk_drag_finish( context, FALSE, FALSE, time );
		return;
	}
	
	if( ! gtk_tree_model_get_iter( model, &it, path ) ) {
		gtk_tree_path_free( path );
		gtk_drag_finish( context, FALSE, FALSE, time );
		return;
	}
	gtk_tree_path_free( path );

	/* only allow drops on playlists */
	gtk_tree_model_get( model, &it,
			NEUTRINO_TYPE_COL, &type,
			NEUTRINO_DATA_COL, &name,
			-1 );
	if( type != NEUTRINO_VIEW_PLAYLIST ) {
		g_free( name );
		gtk_drag_finish( context, FALSE, FALSE, time );
		return;
	}

	if( info == 0 ) {
		if( sel_data->type == gdk_atom_intern( "application/x-neutrino-track-list", TRUE ) ) {

			info = TARGET_NEUTRINO_TRACK_LIST;
		}
	}

	/* we have a NULL name for the play queue */
	if( ! name ) {
		neutrino_window_playqueue_drop( widget, 
				context, x, y, sel_data,
				info, time, data );
		g_free( name );
		return;
	}
	
	idstring = nomad_jukebox_get_idstring( jukebox );

	g_print( "DROP TO: %s\n", name );
	
	switch( info ) {
	case TARGET_NEUTRINO_TRACK_LIST:
		uris = g_strsplit( sel_data->data, "\r\n", 0 );

		list = NULL;
		for( i = 0; uris[ i ]; ++ i ) {
			gchar *uri;
			gchar *type;
			gchar *dropid;

			uri = uris[ i ];
			dropid = type = NULL;

			if( ! strncmp( "nomad://", uri,
				       strlen( "nomad://" ) ) ) {
				uri += strlen( "nomad://" );
				type = strchr( uri, '/' );
				
				g_assert( type != NULL );

				dropid = g_strndup( uri, type - uri );
			}
			if( dropid && ! strcmp( dropid, idstring ) ) {
				/* drop from the same jukebox, allow it */
				gchar *trackid;
				guint id;
			
				trackid = strchr( type + 1, '/' );
				
				g_assert( trackid != NULL );

				trackid ++;
				id = strtoul( trackid, NULL, 10 );

				list = g_list_prepend( list, GUINT_TO_POINTER( id ) );
			}
		}
		if( list ) {
			nomad_jukebox_add_tracks_to_playlist( jukebox, name, list );
		}
		g_list_free( list );
		
		g_strfreev( uris );

		break;
	default:
		g_assert( FALSE );
		break;
	}
      
	g_free( name );
	
	gtk_drag_finish( context, TRUE, FALSE, time );
}

static gboolean neutrino_window_playlist_motion( GtkWidget *widget,
		GdkDragContext *context, gint x, gint y,
		guint time,
		gpointer data )
{
	gboolean ret;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeIter it;
	GtkTreeViewDropPosition dpos;
	gint type;
	
	ret = FALSE;

	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	gtk_tree_view_get_dest_row_at_pos( GTK_TREE_VIEW( widget ),
			x, y, &path, &dpos );

	if( ! path ) { 
		ret = FALSE;
	} else if( ! gtk_tree_model_get_iter( model, &it, path ) ) {
		gtk_tree_path_free( path );
		ret = FALSE;
	} else {
		gtk_tree_path_free( path );

		/* only allow drops on playlists */
		gtk_tree_model_get( model, &it,
				NEUTRINO_TYPE_COL, &type,
				-1 );
		if( type == NEUTRINO_VIEW_PLAYLIST ) {
			ret = TRUE;
		}
	}
	gdk_drag_status( context, 0, time );

	return ! ret;
}

static gboolean neutrino_window_clear_status( NeutrinoWindow *window )
{
	gtk_statusbar_pop( GTK_STATUSBAR( window->details->statusbar ), 0 );

	window->details->statusclear = 0;
	
	return FALSE;
}

static void neutrino_window_set_status( NeutrinoWindow *window, const gchar *txt )
{
	if( window->details->statusclear ) {
		gtk_statusbar_pop( GTK_STATUSBAR( window->details->statusbar ), 0 );
		g_source_remove( window->details->statusclear );
	}
	
	gtk_statusbar_push( GTK_STATUSBAR( window->details->statusbar ), 0, txt );

	window->details->statusclear = g_timeout_add( 2000,
						(GSourceFunc)neutrino_window_clear_status,
						window );
}


/* G Object stuff */

#define PARENT_TYPE GTK_TYPE_WINDOW

static gpointer parent_class;

static void
neutrino_window_class_init( NeutrinoWindowClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS( klass );
	GtkWidgetClass *widget_class;

	widget_class = (GtkWidgetClass *)klass;
	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = neutrino_window_finalize;
	object_class->get_property = neutrino_window_get_prop;
	object_class->set_property = neutrino_window_set_prop;

	widget_class->show = neutrino_window_show;
	

	g_object_class_install_property(object_class,
					ARG_SHELL,
					g_param_spec_object("shell",
							    "Shell",
							    "The Shell",
							    G_TYPE_OBJECT,
							    G_PARAM_READWRITE |
							    G_PARAM_CONSTRUCT)
					);

	widget_class->realize = neutrino_window_realize;
	widget_class->size_request = neutrino_window_size_request;
}

static void
neutrino_window_init( NeutrinoWindow *window )
{
	window->details = g_new0( NeutrinoWindowDetails, 1 );

	window->details->view = neutrino_jukebox_view_new();
}

static void
neutrino_window_set_prop( GObject *object, guint prop_id, 
			 const GValue *value, GParamSpec *spec )
{
	const gchar *name;
	gchar *old_name;
	GObject *obj;
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( object );

	switch( prop_id ) {
	case ARG_SHELL:
		obj = g_value_get_object( value );
		window->shell = NEUTRINO_SHELL( obj );
		neutrino_window_constructed( window );
		break;
	}
}

static void
neutrino_window_get_prop( GObject *object, guint prop_id, 
			 GValue *value, GParamSpec *spec )
{
	GObject *obj;
	gchar *name;

	switch( prop_id ) {
	case ARG_SHELL:
		obj = G_OBJECT( NEUTRINO_WINDOW( object )->shell );
		g_value_set_object( value, obj );
		break;
	}
}

static void
neutrino_window_finalize( GObject *object )
{
	NeutrinoWindow *window;
	NeutrinoWindowDetails *details;
	
	window = NEUTRINO_WINDOW( object );
	details = window->details;
	
	neutrino_jukebox_view_free( details->view );
	g_free( details->view );
	g_object_unref( G_OBJECT( details->browser ) );	
	g_free( details );

	g_object_unref( G_OBJECT( window->xml ) );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static void
neutrino_window_realize( GtkWidget *widget )
{

	/* Create GdkWindow */
	GTK_WIDGET_CLASS( parent_class )->realize( widget );
}

static void
neutrino_window_size_request( GtkWidget *widget, GtkRequisition *req )
{
	g_return_if_fail( NEUTRINO_IS_WINDOW( widget ) );
	g_return_if_fail( req != NULL );

	GTK_WIDGET_CLASS( parent_class )->size_request( widget, req );
}

static void
neutrino_window_show( GtkWidget *widget )
{
	NeutrinoWindow *window;

	window = NEUTRINO_WINDOW( widget );

	GTK_WIDGET_CLASS( parent_class )->show( widget );
}

GType neutrino_window_get_type()
{
	static GType type = 0;

	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( NeutrinoWindowClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)neutrino_window_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( NeutrinoWindow ),
			0, /* n_preallocs */
			(GInstanceInitFunc)neutrino_window_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "NeutrinoWindow",
					       &info, 0 );
	}

	return type;
}
