/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// DlgManageEntities.cpp: implementation of the DlgManageEntities class.
//
//////////////////////////////////////////////////////////////////////

#ifdef _WIN32
#pragma warning(disable:4786)
#endif

#include "DlgMessage.h"
#include "DlgSetRepsOrder.h"
#include "DlgManageEntities.h"
#include "dlgs_wdr.h"
#include "clintl.h"
#include <math.h>



BEGIN_DECLARE_EVENT_TYPES()
    DECLARE_EVENT_TYPE(wxEVT_DISPLAY_ENTITY_INFO, 7777)
END_DECLARE_EVENT_TYPES()

DEFINE_EVENT_TYPE(wxEVT_DISPLAY_ENTITY_INFO)

// it may also be convenient to define an event table macro for this event type
#define EVT_DISPLAY_ENTITY_INFO(fn) \
    DECLARE_EVENT_TABLE_ENTRY( \
        wxEVT_DISPLAY_ENTITY_INFO, wxID_ANY, wxID_ANY, \
        (wxObjectEventFunction)(wxEventFunction)(void (wxEvtHandler::*)(wxEntityInfoEvent&))&fn, \
        (wxObject *) NULL \
    ),



//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNAMIC_CLASS(myEntitiesCanvas, wxScrolledWindow)

BEGIN_EVENT_TABLE(myEntitiesCanvas, wxScrolledWindow)
    EVT_PAINT  (myEntitiesCanvas::OnPaint)
    EVT_MOTION (myEntitiesCanvas::OnMouseMove)
    EVT_LEFT_DOWN (myEntitiesCanvas::OnMouseDownL)
    EVT_RIGHT_DOWN (myEntitiesCanvas::OnMouseDownR)
    EVT_LEFT_UP (myEntitiesCanvas::OnMouseUp)
	EVT_MENU(IDM_ADD_LINK, myEntitiesCanvas::OnAddLink)
	EVT_MENU(IDM_DEL_LINK, myEntitiesCanvas::OnDelLink)
	EVT_MENU(IDM_CONFIGURE_REP_ORDER, myEntitiesCanvas::OnSetRepsOrder)
	EVT_MENU(IDM_SRC_CAN_ACCESS, myEntitiesCanvas::OnSrcCanAccess)
	EVT_MENU(IDM_DST_CAN_ACCESS, myEntitiesCanvas::OnDstCanAccess)
END_EVENT_TABLE()

myEntitiesCanvas::myEntitiesCanvas()
{
	m_links = NULL;
	SelectedItem = NULL;
	MouseoverItem = NULL;
}

myEntitiesCanvas::myEntitiesCanvas(wxWindow *parent,
                         wxWindowID id,
                         const wxPoint& pos,
                         const wxSize& size,
                         long style,
                         const wxString& name):wxScrolledWindow(parent, id, pos, size, style |= wxNO_FULL_REPAINT_ON_RESIZE)
{
	m_links = NULL;
	m_owner = NULL;
	SelectedItem = NULL;
	MouseoverItem = NULL;
	WasDragging = false;

	LogoPkiBig = Entities_GetBitmap(IDB_PKI_BIG);
	LogoCaBig = Entities_GetBitmap(IDB_CA_BIG);
	LogoRaBig = Entities_GetBitmap(IDB_RA_BIG);
	LogoRepBig = Entities_GetBitmap(IDB_REP_BIG);
	LogoPubBig = Entities_GetBitmap(IDB_PUB_BIG);
	LogoKeyBig = Entities_GetBitmap(IDB_KEY_BIG);
	LogoEEBig = Entities_GetBitmap(IDB_EE_BIG);
	hPicBg = Backgrounds_GetBitmap(0);

	last_max_x = 0;
	last_max_y = 0;
}
     
myEntitiesCanvas::~myEntitiesCanvas()
{

}


bool myEntitiesCanvas::lineHitTest( int x1, int y1, int x2, int y2, int x, int y )
{
	float steep;
	float origine;
	float yTest;

	if( x1 > x2 ) 
	{
		steep = x1;
		x1 = x2;
		x2 = (int)steep;

		steep = y1;
		y1 = y2;
		y2 = (int)steep;
	}

	if( x < x1 - (LINK_WIDTH/2) - 2 || x > x2 + (LINK_WIDTH/2) + 2)
		return false;
	
	if(y1 < y2)
	{
		if( y < y1 - (LINK_WIDTH/2) - 2 || y > y2 + (LINK_WIDTH/2) + 2)
			return false;
	}
	else
	{
		if( y < y2 - (LINK_WIDTH/2) - 2|| y > y1 + (LINK_WIDTH/2) + 2)
			return false;
	}

	if( x1 == x2 && ((y>=y1 && y<=y2) || (y>=y2 && y<=y1)) )
		return true;

	steep = (float)(y2-y1) / ((float)x2-x1);
	origine = y2 - (x2 * steep);
	yTest = x * steep + origine;

	if( y > (int)yTest+(LINK_WIDTH/2.0)+2 || y < (int)yTest-(LINK_WIDTH/2.0)-2 )
		return false;

	return true;
}


int myEntitiesCanvas::GetMaxWidth()
{
	return last_max_x;
}

int myEntitiesCanvas::GetMaxHeight()
{
	return last_max_y;
}

bool myEntitiesCanvas::SetSavedValue(char * OptionName, const char * Value)
{
#ifdef _WIN32
		HKEY hk; 

		if (RegCreateKey(HKEY_CURRENT_USER, NEWPKI_PATH, &hk) != ERROR_SUCCESS)
		{
			return false;
		}

		
		if (RegSetValueEx(hk, OptionName, 0, REG_SZ, (PUCHAR)Value, strlen(Value)) != ERROR_SUCCESS)
		{
			RegCloseKey(hk);
			return false;
		}
		RegCloseKey(hk);
		return true;
#else
		return true;
#endif
}

char * myEntitiesCanvas::GetSavedValue(char * OptionName)
{
#ifdef _WIN32
	long lRet;
	HKEY KeyRet;
	
	DWORD KeyType;

	static unsigned char OutDatas[300];
	DWORD OutDatasLen;

	*OutDatas=0;
	
	

	lRet = RegOpenKey(HKEY_CURRENT_USER, NEWPKI_PATH, &KeyRet);
	if(lRet!=ERROR_SUCCESS)
	{
		return "";
	}

	OutDatasLen = sizeof(OutDatas);
	lRet = RegQueryValueEx(KeyRet,OptionName,0,&KeyType,OutDatas,&OutDatasLen);
	if(lRet!=ERROR_SUCCESS)
	{
		RegCloseKey(KeyRet);
		return "";
	}
	
	RegCloseKey(KeyRet);

	return (char*)OutDatas;
#else
	return "";
#endif
}

void myEntitiesCanvas::SaveObjectsPos()
{
	wxString ObjectName;
	std::map<mString, OBJECT_POSITION>::const_iterator i;

#ifdef _WIN32
	wxString Value;

	for(i=objects_position.begin(); i != objects_position.end(); i++)
	{
		ObjectName = i->first.c_str();
		ObjectName += "_x";
		Value.sprintf("%d", i->second.x);
		SetSavedValue((char*)ObjectName.GetData(), (char*)Value.GetData());

		ObjectName = i->first.c_str();
		ObjectName += "_y";
		Value.sprintf("%d", i->second.y);
		SetSavedValue((char*)ObjectName.GetData(), (char*)Value.GetData());
	}
#else
	char * path = getenv("HOME");
	if(!path) return;
	ObjectName.sprintf("%s%s%s", path, ENV_OBJECTS_FILE, m_PkiName);

	FILE * fd = fopen(ObjectName.GetData(), "wb");
	if(!fd) return;

	for(i=objects_position.begin(); i != objects_position.end(); i++)
	{
		ObjectName = i->first.c_str();
		ObjectName += "_x";
		fprintf(fd, "%s=%d\n", ObjectName.GetData(), i->second.x);

		ObjectName = i->first.c_str();
		ObjectName += "_y";
		fprintf(fd, "%s=%d\n", ObjectName.GetData(), i->second.y);
	}
	fclose(fd);

#endif
}


void myEntitiesCanvas::LoadObjectsPos()
{
	wxString ObjectName;
	std::map<mString, OBJECT_POSITION>::iterator i;

#ifdef _WIN32
	char * value;

	for(i=objects_position.begin(); i != objects_position.end(); i++)
	{
		ObjectName = i->first.c_str();
		ObjectName += "_x";

		value = GetSavedValue((char*)ObjectName.GetData());
		if(value && *value)
			sscanf(value, "%d", &i->second.x);

		ObjectName = i->first.c_str();
		ObjectName += "_y";

		value = GetSavedValue((char*)ObjectName.GetData());
		if(value && *value)
			sscanf(value, "%d", &i->second.y);
	}
#else
	char buffer[500];
	char * Value;
	char * path = getenv("HOME");
	if(!path) return;
	ObjectName.sprintf("%s%s%s", path, ENV_OBJECTS_FILE, m_PkiName);

	FILE * fd = fopen(ObjectName.GetData(), "r");
	if(!fd) return;

	while(!feof(fd))
	{
		if(!fgets(buffer, sizeof(buffer), fd)) continue;
		buffer[strlen(buffer)-1]=0;
		
		Value=strstr(buffer, "=");
		if(!Value) continue;
		
		*Value=0;
		Value++;

		for(i=objects_position.begin(); i != objects_position.end(); i++)
		{
			ObjectName = i->first.c_str();
			ObjectName += "_x";

			if(strcmp(buffer, ObjectName.GetData()) == 0)
			{
				sscanf(Value, "%d", &i->second.x);
			}

			ObjectName = i->first.c_str();
			ObjectName += "_y";
			if(strcmp(buffer, ObjectName.GetData()) == 0)
			{
				sscanf(Value, "%d", &i->second.y);
			}
		}
	}
	fclose(fd);

#endif
}

void myEntitiesCanvas::OnSetRepsOrder(wxCommandEvent &event)
{
	if(!SelectedItem)
		return;

	DlgSetRepsOrder Dlg(m_owner, SelectedItem->get_dsts());
}



void myEntitiesCanvas::OnAddLink(wxCommandEvent &event)
{
	size_t i;
	DlgGetLink Dlg(m_owner, (*m_links), list_unpresent);
	if(!Dlg.IsOK())
		return;
	const vector<size_t> & Selection = Dlg.GetSelection();
	if(!Selection.size()) return;

	for(i=0; i<Selection.size(); i++)
	{
		add_LinkToEntity((*m_links), *SelectedItem, (*m_links)[Selection[i]].get_src().get_name());
	}
	Refresh();
}


void myEntitiesCanvas::OnDelLink(wxCommandEvent &event)
{
	size_t i;
	DlgGetLink Dlg(m_owner, (*m_links), list_present);
	if(!Dlg.IsOK())
		return;
	const vector<size_t> & Selection = Dlg.GetSelection();
	if(!Selection.size()) return;

	for(i=0; i<Selection.size(); i++)
	{
		del_LinkToEntity((*m_links), *SelectedItem, (*m_links)[Selection[i]].get_src().get_name());
	}
	Refresh();
}

void myEntitiesCanvas::OnMouseMove(wxMouseEvent &event)
{
	std::map<mString, OBJECT_POSITION>::const_iterator i;
	int x, y;
	wxEntityInfoEvent eventCustom;
	CalcUnscrolledPosition(event.GetX(), event.GetY(), &x, &y);

	WasDragging = event.Dragging();
	eventCustom.SetEventType(wxEVT_DISPLAY_ENTITY_INFO);


	for(i=objects_position.begin(); i != objects_position.end(); i++)
	{
		if(	x > (i->second.x) && 
			x < ((i->second.x + ICON_WIDTH)) &&
			y > (i->second.y) && 
			y < ((i->second.y + ICON_HEIGHT)))
		{
			if(!MouseoverItem || ! (i->first == MouseoverItem->get_src().get_name()) )
			{
				MouseoverItem = get_EntityLink((*m_links), i->first.c_str());
				if(*MouseoverItem)
				{
					SetToolTip(i->first.c_str());
					eventCustom.SetEntityInfo(*MouseoverItem);
					wxPostEvent(m_owner, eventCustom);
				}
			}
			return;

		}
	}

	if(MouseoverItem)
	{
		wxPostEvent(m_owner, eventCustom);
		SetToolTip("");
		MouseoverItem = NULL;
	}
}

void myEntitiesCanvas::OnMouseUp(wxMouseEvent &event)
{
	int x, y;

	CalcUnscrolledPosition(event.GetX(), event.GetY(), &x, &y);

	if(WasDragging && SelectedItem)
	{
		objects_position[SelectedItem->get_src().get_name()].x = (x + m_drag_x);
		objects_position[SelectedItem->get_src().get_name()].y = (y + m_drag_y);
		WasDragging = false;
		Refresh();
		return;
	}

	WasDragging = false;
}


void myEntitiesCanvas::OnMouseDownL(wxMouseEvent &event)
{
	std::map<mString, OBJECT_POSITION>::const_iterator i;
	mString SelectedName;
	int x, y;
	CalcUnscrolledPosition(event.GetX(), event.GetY(), &x, &y);

	SelectedName = SelectedItem?SelectedItem->get_src().get_name():"";
	SelectedItem = NULL;
	for(i=objects_position.begin(); i != objects_position.end(); i++)
	{
		if(	x > (i->second.x) && 
			x < ((i->second.x + ICON_WIDTH)) &&
			y > (i->second.y) && 
			y < ((i->second.y + ICON_HEIGHT)))
		{
			m_drag_x = i->second.x - x;
			m_drag_y = i->second.y - y;
			SelectedItem = get_EntityLink((*m_links), i->first.c_str());
			if(! (SelectedName == SelectedItem->get_src().get_name()) )
				Refresh();
			return;
		}
	}

	if (SelectedName.size() && !SelectedItem)
	{
		Refresh();
	}
}


void myEntitiesCanvas::OnMouseDownR(wxMouseEvent &event)
{
	wxMenu * menu;
	size_t i;
	size_t j;
	int src_x;
	int src_y;
	int dst_x;
	int dst_y;
	int x, y;
	int x_entry, y_entry;
	int x_link, y_link;
	mString MenuItemLabel;
	
	OnMouseDownL(event);
	wxMenuBar * menu_b = myEntitiesCanvas_GetPopup();
	if(!menu_b)
		return;

	if(SelectedItem)		//Clicked an entity
	{
		menu = menu_b->GetMenu(0);
		if(menu)
		{
			list_unpresent.clear();
			list_present.clear();

			GetLinksItems(*SelectedItem);

			PopupMenu(menu, event.GetX(), event.GetY());
		}
	}
	else
	{
		CalcUnscrolledPosition(event.GetX(), event.GetY(), &x, &y);

		// Did user click on a repository/repository link?
		for(i=0; i<(*m_links).size(); i++)
		{
			// We only want a repository
			if((*m_links)[i].get_src().get_type() != ENTITY_TYPE_REPOSITORY)
				continue;

			x_entry = objects_position[(*m_links)[i].get_src().get_name()].x;
			y_entry = objects_position[(*m_links)[i].get_src().get_name()].y;

			//Enumerate its links
			for(j=0; j<(*m_links)[i].get_dsts().size(); j++)
			{
				// We only want a repository
				if((*m_links)[i].get_dsts()[j].get_type() != ENTITY_TYPE_REPOSITORY)
					continue;

				x_link = objects_position[(*m_links)[i].get_dsts()[j].get_name()].x;
				y_link = objects_position[(*m_links)[i].get_dsts()[j].get_name()].y;

				//Calculating the right corner for the src
				if(x_link > x_entry)
				{
					src_x = x_entry + ICON_WIDTH;
					dst_x = x_link;
				}
				else
				{
					src_x = x_entry;
					dst_x = x_link + ICON_WIDTH;
				}
				if(y_link > y_entry)
				{
					src_y = y_entry + ICON_HEIGHT;
					dst_y = y_link;
				}
				else
				{
					src_y = y_entry;
					dst_y = y_link + ICON_HEIGHT;
				}

				// Did the user right click on a link line
				if(lineHitTest(src_x, src_y, dst_x, dst_y, x, y))
				{
					menu = menu_b->GetMenu(1);
					if(menu)
					{
						// RepSource is always the repository that can connect

						if(ASN1_BIT_STRING_get_bit((*m_links)[i].get_dsts()[j].get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO))
						{
							RepSource = (*m_links)[i].get_src().get_name();
							RepDest = (*m_links)[i].get_dsts()[j].get_name();
							menu->Check(IDM_SRC_CAN_ACCESS, true);
							RepSourceIsChecked = true;
							menu->Enable(IDM_SRC_CAN_ACCESS, false);
							menu->Check(IDM_DST_CAN_ACCESS, false);
							RepDestIsChecked = false;
						}
						else if(ASN1_BIT_STRING_get_bit((*m_links)[i].get_dsts()[j].get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED))
						{
							RepSource = (*m_links)[i].get_dsts()[j].get_name();
							RepDest = (*m_links)[i].get_src().get_name();
							menu->Check(IDM_SRC_CAN_ACCESS, true);
							RepSourceIsChecked = true;
							menu->Enable(IDM_SRC_CAN_ACCESS, false);
							menu->Check(IDM_DST_CAN_ACCESS, false);
							RepDestIsChecked = false;
						}
						else
						{
							RepSource = (*m_links)[i].get_src().get_name();
							RepDest = (*m_links)[i].get_dsts()[j].get_name();
							menu->Check(IDM_SRC_CAN_ACCESS, true);
							menu->Check(IDM_DST_CAN_ACCESS, true);
							RepSourceIsChecked = true;
							RepDestIsChecked = true;
						}

						MenuItemLabel.sprintf(_("%s can connect to %s"), RepSource.c_str(), RepDest.c_str());
						menu->SetLabel(IDM_SRC_CAN_ACCESS, MenuItemLabel.c_str());

						MenuItemLabel.sprintf(_("%s can connect to %s"), RepDest.c_str(), RepSource.c_str());
						menu->SetLabel(IDM_DST_CAN_ACCESS, MenuItemLabel.c_str());

						PopupMenu(menu, event.GetX(), event.GetY());
					}
					break;
				}
			}
		}
	}

	delete menu_b;
}

void myEntitiesCanvas::OnSrcCanAccess(wxCommandEvent &event)
{
	if(RepSourceIsChecked)
	{
		// RepSource can't connect to RepDst
		EntityLinks * currEntry = get_EntityLink((*m_links), RepSource);
		if(!currEntry)
			return;
		EntityLinkInfo * currLink = get_LinkToEntity(*currEntry, RepDest);
		if(!currLink)
			return;
		ASN1_BIT_STRING_set_bit(currLink->get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED, 1);
		ASN1_BIT_STRING_set_bit(currLink->get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO, 0);


		currEntry = get_EntityLink((*m_links), RepDest);
		if(!currEntry)
			return;
		currLink = get_LinkToEntity(*currEntry, RepSource);
		if(!currLink)
			return;
		ASN1_BIT_STRING_set_bit(currLink->get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED, 0);
		ASN1_BIT_STRING_set_bit(currLink->get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO, 1);

		Refresh();
	}
}


void myEntitiesCanvas::OnDstCanAccess(wxCommandEvent&event)
{
	EntityLinks * currEntry;
	EntityLinkInfo * currLink;

	if(RepDestIsChecked)
	{
		// RepDst can't connect to RepSource
		currEntry = get_EntityLink((*m_links), RepSource);
		if(!currEntry)
			return;
		currLink = get_LinkToEntity(*currEntry, RepDest);
		if(!currLink)
			return;
		ASN1_BIT_STRING_set_bit(currLink->get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED, 0);
		ASN1_BIT_STRING_set_bit(currLink->get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO, 1);

		currEntry = get_EntityLink((*m_links), RepDest);
		if(!currEntry)
			return;
		currLink = get_LinkToEntity(*currEntry, RepSource);
		if(!currLink)
			return;
		ASN1_BIT_STRING_set_bit(currLink->get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED, 1);
		ASN1_BIT_STRING_set_bit(currLink->get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO, 0);
	}
	else
	{
		// Both repositories can connect to each others
		currEntry = get_EntityLink((*m_links), RepSource);
		if(!currEntry)
			return;
		currLink = get_LinkToEntity(*currEntry, RepDest);
		if(!currLink)
			return;
		ASN1_BIT_STRING_set_bit(currLink->get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED, 0);
		ASN1_BIT_STRING_set_bit(currLink->get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO, 0);

		currEntry = get_EntityLink((*m_links), RepDest.c_str());
		if(!currEntry)
			return;
		currLink = get_LinkToEntity(*currEntry, RepSource.c_str());
		if(!currLink)
			return;
		ASN1_BIT_STRING_set_bit(currLink->get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED, 0);
		ASN1_BIT_STRING_set_bit(currLink->get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO, 0);
	}
	Refresh();
}


void myEntitiesCanvas::GetLinksItems(EntityLinks & SrcEntry)
{
	size_t i;
	int Type;
	size_t eeRaCount = 0;

	Type = SrcEntry.get_src().get_type();

	for(i=0; i<(*m_links).size(); i++)
	{
		//We ignore ourself
		if(SrcEntry.get_src().get_name() == (*m_links)[i].get_src().get_name())
			continue;
		if( (SrcEntry.get_src().get_type() == ENTITY_TYPE_RA &&
			(*m_links)[i].get_src().get_type() == ENTITY_TYPE_EE) ||
			(SrcEntry.get_src().get_type() == ENTITY_TYPE_RA &&
			(*m_links)[i].get_src().get_type() == ENTITY_TYPE_EE))
		{
			if(get_LinkToEntity(SrcEntry, (*m_links)[i].get_src().get_name()))
				eeRaCount++;
		}
	}

	//For each object
	for(i=0; i<(*m_links).size(); i++)
	{
		//We ignore ourself
		if(SrcEntry.get_src().get_name() == (*m_links)[i].get_src().get_name())
			continue;

		//Is it allowed?
		if(!IsLinkAllowed(Type, (*m_links)[i].get_src().get_type()))
			continue;
		
		//Search if it is presnt
		if(!get_LinkToEntity(SrcEntry, (*m_links)[i].get_src().get_name()))
		{
			if( !eeRaCount ||
				!((SrcEntry.get_src().get_type() == ENTITY_TYPE_RA &&
				(*m_links)[i].get_src().get_type() == ENTITY_TYPE_EE) ||
				(SrcEntry.get_src().get_type() == ENTITY_TYPE_RA &&
				(*m_links)[i].get_src().get_type() == ENTITY_TYPE_EE)))
			{
				list_unpresent.push_back(i);
			}
		}
		else
		{
			list_present.push_back(i);
		}
	}
}

void myEntitiesCanvas::SetOwner(wxDialog *owner)
{
	m_owner = owner;
}

void myEntitiesCanvas::OnPaint(wxPaintEvent &event)
{
	wxPaintDC dc(this);
	PrepareDC(dc);

	if(m_owner)
		m_owner->PrepareDC(dc);
	

	dc.BeginDrawing();
    dc.SetBackgroundMode( wxSOLID );
    dc.SetBackground( *wxWHITE_BRUSH );
	dc.Clear();

	PaintPicture(dc);
	dc.EndDrawing();
}

void myEntitiesCanvas::PaintPicture(wxDC & dc)
{
	size_t i;
	int max_x;
	int max_y;
	int x_unit;
	int y_unit;


	max_x = 0;
	max_y = 0;
	DrawLinks(dc);
	for(i=0; i<(*m_links).size(); i++)
	{
		DrawObject(dc, (*m_links)[i], (SelectedItem && SelectedItem->get_src().get_name() == (*m_links)[i].get_src().get_name()), max_x, max_y);
	}
	// Do we need to resize the scroll bars ?
	if(max_x != last_max_x || max_y != last_max_y)
	{
		last_max_x = max_x;
		last_max_y = max_y;
		GetScrollPixelsPerUnit(&x_unit, &y_unit);
		SetScrollbars(x_unit, y_unit, (max_x/x_unit)+3, (max_y/y_unit)+2, (max_x/x_unit)+3, (max_y/y_unit)+2);
	}
}

void myEntitiesCanvas::SetLinks(const char * PkiName, mVector<EntityLinks> * links)
{
	m_links = links;
	m_PkiName = PkiName;

	size_t i;
	OBJECT_POSITION position;
	wxRect mySize = wxWindow::GetRect();

	position.x = 10;
	position.y = 10;
	for(i=0; i<(*m_links).size(); i++)
	{
		if(position.x > (mySize.GetWidth() - 50))
		{
			position.x = 10;
			position.y += ICON_HEIGHT + 25;
		}
		else
		{
			position.x += ICON_WIDTH + 10;
		}
		objects_position[(*m_links)[i].get_src().get_name()] =  position;
	}
	LoadObjectsPos();
}

void DrawArrow(wxDC &dc, int src_x, int src_y, int dst_x, int dst_y, const wxColour & DrawClr, bool InvertDirection)
{
	wxPoint listPoints[3];
	double slopy , cosy , siny;
	double Par = 12.0;  //length of Arrow (>)

	slopy = atan2( ( src_y - dst_y ), ( src_x - dst_x ) );

	cosy = cos( slopy );
	siny = sin( slopy );

	listPoints[0].x = src_x + (int)( - Par * cosy - ( Par / 2.0 * siny ));
	listPoints[0].y = src_y + (int)( - Par * siny + ( Par / 2.0 * cosy ));

	listPoints[1].x = src_x + (int)( - Par * cosy + ( Par / 2.0 * siny ) );
	listPoints[1].y = src_y - (int)( Par / 2.0 * cosy + Par * siny );

	listPoints[2].x = src_x;
	listPoints[2].y = src_y;

/*
	pdc->MoveTo( m_Two );
	pdc->LineTo( dst_x + int( Par * cosy - ( Par / 2.0 * siny ) ),
	dst_y + int( Par * siny + ( Par / 2.0 * cosy ) ) );
	pdc->LineTo( dst_x + int( Par * cosy + Par / 2.0 * siny ),
	dst_y - int( Par / 2.0 * cosy - Par * siny ) );
	pdc->LineTo( m_Two );
*/
	
	dc.DrawPolygon(3, listPoints);
}

void myEntitiesCanvas::DrawLinks(wxDC &dc)
{
	size_t i;
	size_t j;
	int src_x;
	int src_y;
	int dst_x;
	int dst_y;
	bool begin_ar;
	bool end_ar;
	wxColour DrawClr;
	bool SrcRepIsFirewalled;
	bool DstRepIsFirewalled;
	map<mString, int> DrawnLinks;
	mString ObjectName;


	for(i=0; i<(*m_links).size(); i++)
	{
		//Enumerate its links
		for(j=0; j<(*m_links)[i].get_dsts().size(); j++)
		{
			// We don't draw 2 times all double
			// direction link (Rep1 -> Rep2 = Rep2 -> Rep1)
			ObjectName = (*m_links)[i].get_dsts()[j].get_name();
			ObjectName += (*m_links)[i].get_src().get_name();
			if(DrawnLinks.find(ObjectName) != DrawnLinks.end())
				continue;

			ObjectName = (*m_links)[i].get_src().get_name();
			ObjectName += (*m_links)[i].get_dsts()[j].get_name();
			if(DrawnLinks.find(ObjectName) != DrawnLinks.end())
				continue;

			DrawnLinks[ObjectName] = 1;

			begin_ar = false;
			end_ar = false;
			SrcRepIsFirewalled = false;
			DstRepIsFirewalled = false;

			switch((*m_links)[i].get_dsts()[j].get_type())
			{
				case ENTITY_TYPE_PKI:
					DrawClr.Set(PKI_COLOR);
					break;
				case ENTITY_TYPE_RA:
					switch((*m_links)[i].get_src().get_type())
					{
						case ENTITY_TYPE_CA:
							DrawClr.Set(CA_COLOR);
							begin_ar = true;
							break;
						case ENTITY_TYPE_KEY_STORE:
							DrawClr.Set(KEY_COLOR);
							begin_ar = true;
							break;
						case ENTITY_TYPE_EE:
							DrawClr.Set(EE_COLOR);
							begin_ar = true;
							break;
					}
				case ENTITY_TYPE_CA:
					switch((*m_links)[i].get_src().get_type())
					{
						case ENTITY_TYPE_RA:
							DrawClr.Set(CA_COLOR);
							end_ar = true;
							break;
						case ENTITY_TYPE_PUBLICATION:
							DrawClr.Set(PUB_COLOR);
							begin_ar = true;
							break;
					}
					break;
				case ENTITY_TYPE_REPOSITORY:
					switch((*m_links)[i].get_src().get_type())
					{
						case ENTITY_TYPE_RA:
						case ENTITY_TYPE_CA:
						case ENTITY_TYPE_PUBLICATION:
						case ENTITY_TYPE_KEY_STORE:
						case ENTITY_TYPE_EE:
						case ENTITY_TYPE_PKI:
							end_ar = true;
							break;
						case ENTITY_TYPE_REPOSITORY:
							begin_ar = true;
							end_ar = true;
							if(ASN1_BIT_STRING_get_bit((*m_links)[i].get_dsts()[j].get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO))
							{
								DstRepIsFirewalled = false;
								SrcRepIsFirewalled = true;
							}
							else if(ASN1_BIT_STRING_get_bit((*m_links)[i].get_dsts()[j].get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED))
							{
								DstRepIsFirewalled = true;
								SrcRepIsFirewalled = false;
							}
							break;
					}
					DrawClr.Set(REP_COLOR);
					break;
				case ENTITY_TYPE_PUBLICATION:
					DrawClr.Set(PUB_COLOR);
					end_ar = true;
					break;
				case ENTITY_TYPE_KEY_STORE:
					DrawClr.Set(KEY_COLOR);
					end_ar = true;
					break;
				case ENTITY_TYPE_EE:
					DrawClr.Set(EE_COLOR);
					end_ar = true;
					break;
			}

			//Calculating the right corner for the src
			if(objects_position[(*m_links)[i].get_dsts()[j].get_name()].x > objects_position[(*m_links)[i].get_src().get_name()].x)
			{
				src_x = objects_position[(*m_links)[i].get_src().get_name()].x + ICON_WIDTH;
				dst_x = objects_position[(*m_links)[i].get_dsts()[j].get_name()].x;
			}
			else
			{
				src_x = objects_position[(*m_links)[i].get_src().get_name()].x;
				dst_x = objects_position[(*m_links)[i].get_dsts()[j].get_name()].x + ICON_WIDTH;
			}
			if(objects_position[(*m_links)[i].get_dsts()[j].get_name()].y > objects_position[(*m_links)[i].get_src().get_name()].y)
			{
				src_y = objects_position[(*m_links)[i].get_src().get_name()].y + ICON_HEIGHT;
				dst_y = objects_position[(*m_links)[i].get_dsts()[j].get_name()].y;
			}
			else
			{
				src_y = objects_position[(*m_links)[i].get_src().get_name()].y;
				dst_y = objects_position[(*m_links)[i].get_dsts()[j].get_name()].y + ICON_HEIGHT;
			}

			if(DstRepIsFirewalled)
			{
				dc.SetPen( wxPen(*wxBLACK, LINK_WIDTH, wxDOT));
				dc.DrawLine(src_x + ((dst_x - src_x)/2), src_y + ((dst_y - src_y)/2), dst_x, dst_y);
				dc.SetPen( wxPen(DrawClr, LINK_WIDTH, wxSOLID));
				dc.DrawLine(src_x, src_y, dst_x - ((dst_x - src_x)/2), dst_y - ((dst_y - src_y)/2));
			}
			else if(SrcRepIsFirewalled)
			{
				dc.SetPen( wxPen(*wxBLACK, LINK_WIDTH, wxDOT));
				dc.DrawLine(src_x, src_y, dst_x - ((dst_x - src_x)/2), dst_y - ((dst_y - src_y)/2));
				dc.SetPen( wxPen(DrawClr, LINK_WIDTH, wxSOLID));
				dc.DrawLine(src_x + ((dst_x - src_x)/2), src_y + ((dst_y - src_y)/2), dst_x, dst_y);
			}
			else
			{
				dc.SetPen( wxPen(DrawClr, LINK_WIDTH, wxSOLID));
				dc.DrawLine(src_x, src_y, dst_x, dst_y);
				dc.DrawLine(src_x, src_y, dst_x - ((dst_x - src_x)/2), dst_y - ((dst_y - src_y)/2));
			}
			wxBrush savedBrush = dc.GetBrush();
			
			if(begin_ar)
			{
				if(SrcRepIsFirewalled)
				{
					dc.SetBrush(wxBrush(*wxBLACK, wxSOLID));
					dc.SetPen( wxPen(*wxBLACK, LINK_WIDTH, wxSOLID));
				}
				else
				{
					dc.SetBrush(wxBrush(DrawClr, wxSOLID));
					dc.SetPen( wxPen(DrawClr, LINK_WIDTH, wxSOLID));
				}

				dc.DrawCircle(src_x, src_y, ANCHOR_RADIUS);
			}
			if(end_ar)
			{
				if(DstRepIsFirewalled)
				{
					dc.SetBrush(wxBrush(*wxBLACK, wxSOLID));
					dc.SetPen( wxPen(*wxBLACK, LINK_WIDTH, wxSOLID));
				}
				else
				{
					dc.SetBrush(wxBrush(DrawClr, wxSOLID));
					dc.SetPen( wxPen(DrawClr, LINK_WIDTH, wxSOLID));
				}

				dc.DrawCircle(dst_x, dst_y, ANCHOR_RADIUS);
			}

			if(begin_ar)
			{
				if(!SrcRepIsFirewalled)
					DrawArrow(dc, src_x + ((dst_x - src_x)/2), src_y + ((dst_y - src_y)/2), dst_x, dst_y, DrawClr, false);
			}
			if(end_ar)
			{
				if(!DstRepIsFirewalled)
					DrawArrow(dc, dst_x - ((dst_x - src_x)/2), dst_y - ((dst_y - src_y)/2), src_x, src_y, DrawClr, false);
			}			
			dc.SetBrush(savedBrush);
		}
	}
}

void myEntitiesCanvas::DrawObject(wxDC & dc, EntityLinks & Object, bool Selected, int & max_x, int & max_y)
{
	wxBitmap * logo;
	int pos_x;
	int pos_y;
	int str_len;
	int flag;
	int pos;
	OBJECT_POSITION Position = objects_position[Object.get_src().get_name()];
	
	switch(Object.get_src().get_type())
	{
		case ENTITY_TYPE_PKI:
			logo = &LogoPkiBig;
			break;
		case ENTITY_TYPE_CA:
			logo = &LogoCaBig;
			break;
		case ENTITY_TYPE_RA:
			logo = &LogoRaBig;
			break;
		case ENTITY_TYPE_REPOSITORY:
			logo = &LogoRepBig;
			break;
		case ENTITY_TYPE_PUBLICATION:
			logo = &LogoPubBig;
			break;
		case ENTITY_TYPE_KEY_STORE:
			logo = &LogoKeyBig;
			break;
		case ENTITY_TYPE_EE:
			logo = &LogoEEBig;
			break;
		default:
			return;
	}


	wxMemoryDC memDC;
	memDC.SelectObject(*logo);
	if(Selected)
	{
		flag = wxCOPY;
		pos = 3;
	}
	else
	{
		flag = wxCOPY;
		pos = 2;
	}
	dc.SetPen( wxPen(*wxWHITE, 1, wxSOLID));
	wxBrush savedBrush = dc.GetBrush();
	dc.SetBrush(wxBrush(*wxWHITE, wxSOLID));
	dc.DrawRectangle(Position.x, Position.y, ICON_WIDTH, ICON_HEIGHT);
	dc.SetBrush(savedBrush);
	dc.Blit(Position.x+pos, Position.y+pos, ICON_WIDTH-5, ICON_HEIGHT-5, &memDC, 0, 0, flag, TRUE);
	memDC.SelectObject( wxNullBitmap );


	if(Selected)
	{
		dc.SetPen( wxPen(wxColour(212, 208, 200), 1, wxSOLID));
		dc.DrawLine(Position.x+1, Position.y+ICON_HEIGHT-1, Position.x+ICON_WIDTH-1, Position.y+ICON_HEIGHT-1);
		dc.DrawLine(Position.x+ICON_WIDTH-1, Position.y, Position.x+ICON_WIDTH-1, Position.y+ICON_HEIGHT-1);

		dc.SetPen( wxPen(wxColour(64, 64, 64), 1, wxSOLID));
		dc.DrawLine(Position.x, Position.y, Position.x+ICON_WIDTH, Position.y);
		dc.DrawLine(Position.x, Position.y, Position.x, Position.y+ICON_HEIGHT);

		dc.SetPen( wxPen(wxColour(128, 128, 128), 1, wxSOLID));
		dc.DrawLine(Position.x+1, Position.y+1, Position.x+ICON_WIDTH-1, Position.y+1);
		dc.DrawLine(Position.x+1, Position.y+1, Position.x+1, Position.y+ICON_HEIGHT-1);
	}
	else
	{
		dc.SetPen( wxPen(wxColour(64, 64, 64), 1, wxSOLID));
		dc.DrawLine(Position.x, Position.y+ICON_HEIGHT-1, Position.x+ICON_WIDTH-1, Position.y+ICON_HEIGHT-1);
		dc.DrawLine(Position.x+ICON_WIDTH-1, Position.y, Position.x+ICON_WIDTH-1, Position.y+ICON_HEIGHT);

		dc.SetPen( wxPen(wxColour(128, 128, 128), 1, wxSOLID));
		dc.DrawLine(Position.x+1, Position.y+ICON_HEIGHT-2, Position.x+ICON_WIDTH-2, Position.y+ICON_HEIGHT-2);
		dc.DrawLine(Position.x+ICON_WIDTH-2, Position.y+1, Position.x+ICON_WIDTH-2, Position.y+ICON_HEIGHT-1);

		dc.SetPen( wxPen(wxColour(212, 208, 200), 1, wxSOLID));
		dc.DrawLine(Position.x, Position.y, Position.x+ICON_WIDTH-1, Position.y);
		dc.DrawLine(Position.x, Position.y, Position.x, Position.y+ICON_HEIGHT-1);
	}



	str_len = (int)(((float)(Object.get_src().get_name().size() * dc.GetCharWidth())) * 1.1);
	pos_x = (Position.x + (ICON_WIDTH/2)) - (str_len/2);
	pos_y = Position.y + ICON_HEIGHT + 3;
	dc.SetBrush(wxBrush(*wxWHITE, wxSOLID));

	dc.DrawText(Object.get_src().get_name().c_str(), pos_x, pos_y);


	if( (pos_x + str_len) > max_x)
	{
		max_x = pos_x + str_len + 1;
	}
	if( (pos_y + dc.GetCharHeight()) > max_y)
	{
		max_y = pos_y + dc.GetCharHeight() + 1;
	}
}


BEGIN_EVENT_TABLE(DlgManageEntities, wxDialog)
    EVT_BUTTON(IDC_OK, DlgManageEntities::OnOk)
    EVT_BUTTON(IDC_CANCEL, DlgManageEntities::OnCancel)
    EVT_BUTTON(IDC_SAVE_SCREEN, DlgManageEntities::OnSaveScreen)
	EVT_DISPLAY_ENTITY_INFO(DlgManageEntities::OnDisplayInfo)
END_EVENT_TABLE()

DlgManageEntities::DlgManageEntities(wxWindow * wParent, const char * PkiName, PkiClient * ClientPKI):wxDialog(wParent, (wxWindowID)-1, _("Manage entities links"), wxDefaultPosition, wxDefaultSize, wxSYSTEM_MENU | wxMINIMIZE_BOX | wxCAPTION | wxMAXIMIZE_BOX | wxRESIZE_BORDER)
{
	m_ClientPKI = ClientPKI;

	DlgManageEntities_SetWindow(this);

	drawing = (myEntitiesCanvas *)FindWindow(IDC_DRAWING);
	display_info = (wxTextCtrl *)FindWindow(IDC_RESUME);
	drawing->SetOwner(this);

	display_info->SetBackgroundColour(GetBackgroundColour());

	DlgMessage dlg(this);
	dlg.wShow(_("Getting Entities Links..."));
	if(!m_ClientPKI->GetEntitiesLinks(m_links))
	{
		dlg.wHide();
		HandleError(m_ClientPKI->GetError(), this);
		Close(TRUE);
		return;
	}
	dlg.wHide();

	drawing->SetLinks(PkiName, &m_links);
	drawing->Scroll(0, 0);


	CenterOnScreen();
	ShowModal();
	
}

DlgManageEntities::~DlgManageEntities()
{
}

void DlgManageEntities::OnOk(wxCommandEvent& event)
{
	drawing->SaveObjectsPos();

	if(DisplayMessage(this, _("Are you sure to want to modify the entities links ?"), wxYES_NO) == wxNO) return;

	DlgMessage dlg(this);
	dlg.wShow(_("Setting Entities Links..."));
	if(!m_ClientPKI->SetEntitiesLinks(m_links))
	{
		dlg.wHide();
		HandleError(m_ClientPKI->GetError(), this);
		return;
	}
	dlg.wHide();

	Close(TRUE);
}

void DlgManageEntities::OnSaveScreen(wxCommandEvent& event)
{
	int x_unit;
	int y_unit;
	wxFileDialog dialog(this, _("Save picture"), "", "", _("BMP File (*.bmp)|*.bmp"), wxSAVE|wxOVERWRITE_PROMPT);
	wxMemoryDC dc;
	int w;
	int h;
	wxString tt;
	wxFont savedFont;

	// Calculate the pciture's size, including the legend
	// with is futur font
	savedFont = dc.GetFont();
	dc.SetFont(*wxNORMAL_FONT);
	w = drawing->GetMaxWidth();
	h = drawing->GetMaxHeight() + dc.GetCharHeight() + 15;
	dc.SetFont(savedFont);
	
	wxBitmap bmp(w, h);
	wxBitmap hPicBg = Backgrounds_GetBitmap(0);

	dc.SelectObject(bmp);
    dc.SetBackgroundMode( wxSOLID );
    dc.SetBackground( *wxWHITE_BRUSH );
	dc.Clear();

	if(hPicBg.GetWidth() < w && 
		hPicBg.GetHeight() < h)
	{
		x_unit = (bmp.GetWidth() - hPicBg.GetWidth()) / 2;
		y_unit = (bmp.GetHeight() - hPicBg.GetHeight()) / 2;
		dc.DrawBitmap(hPicBg, x_unit, y_unit);
	}


	drawing->PaintPicture(dc);
	tt.sprintf("Generated by NewPKI %s", NEWPKI_VERSION);
	dc.SetTextForeground(*wxBLUE);
	dc.SetFont(*wxNORMAL_FONT);
	dc.DrawText(tt.GetData(), 5, h - dc.GetCharHeight() - 5);

	dc.SelectObject( wxNullBitmap );
	if (dialog.ShowModal() != wxID_OK)
	{
		return;
	}
	bmp.SaveFile(dialog.GetPath(), wxBITMAP_TYPE_BMP);
}

void DlgManageEntities::OnCancel(wxCommandEvent& event)
{
	Close(TRUE);
}


void DlgManageEntities::OnDisplayInfo(wxEntityInfoEvent &event)
{
	const char * EntityType;

	const EntityLinks & info = event.GetEntityInfo();
	if(!info.get_src().get_name().size())
	{
		display_info->SetValue("");
		return;
	}

	wxString DisplayInfo;
	const char * RepHeader = NULL;
	const char * CaHeader = NULL;
	const char * RaHeader = NULL;
	const char * PubHeader = NULL;
	const char * KeyHeader = NULL;
	wxString RepList;
	wxString CaList;
	wxString RaList;
	wxString PubList;
	wxString KeyList;
	wxString EEList;

	switch(info.get_src().get_type())
	{
		case ENTITY_TYPE_REPOSITORY:
			EntityType = _("Repository");
			break;
		case ENTITY_TYPE_PKI:
			EntityType = _("PKI");
			break;
		case ENTITY_TYPE_CA:
			EntityType = _("CA");
			break;
		case ENTITY_TYPE_RA:
			EntityType = _("RA");
			break;
		case ENTITY_TYPE_PUBLICATION:
			EntityType = _("Publication");
			break;
		case ENTITY_TYPE_KEY_STORE:
			EntityType = _("Key Store");
			break;
		case ENTITY_TYPE_EE:
			EntityType = _("EE");
			break;
		default:
			EntityType = _("Unknown");
			break;
	}

	for(size_t i=0; i < info.get_dsts().size(); i++)
	{
		switch(info.get_src().get_type())
		{
			case ENTITY_TYPE_REPOSITORY:
				switch(info.get_dsts()[i].get_type())
				{
					case ENTITY_TYPE_REPOSITORY:
						if(ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)info.get_dsts()[i].get_flags(), REP_ENTRY_INFO_FLAG_FIREWALLED))
						{
							RepHeader = _("Get configuration from:\n");
							RepList += info.get_dsts()[i].get_name().c_str();
							RepList += "\n";
						}
						else if(ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)info.get_dsts()[i].get_flags(), REP_ENTRY_INFO_FLAG_SHOULD_FULL_SYNCHRO))
						{
							RepHeader = _("Can push configuration on:\n");
							RepList += info.get_dsts()[i].get_name().c_str();
							RepList += "\n";
						}
						else
						{
							RepHeader = _("Can push configuration on:\n");
							RepList += info.get_dsts()[i].get_name().c_str();
							RepList += "\n";
						}
						break;
				}
				break;
			case ENTITY_TYPE_PKI:
				switch(info.get_dsts()[i].get_type())
				{
					case ENTITY_TYPE_REPOSITORY:
						RepHeader = _("Can push configuration on:\n");
						RepList += info.get_dsts()[i].get_name().c_str();
						RepList += "\n";
						break;
				}
				break;
			case ENTITY_TYPE_CA:
				switch(info.get_dsts()[i].get_type())
				{
					case ENTITY_TYPE_REPOSITORY:
						RepHeader = _("Get personnal configuration from:\n");
						RepList += info.get_dsts()[i].get_name().c_str();
						RepList += "\n";
						break;
					case ENTITY_TYPE_RA:
						RaHeader = _("Can receive certificate requests from:\n");
						RaList += info.get_dsts()[i].get_name().c_str();
						RaList += "\n";
						break;
					case ENTITY_TYPE_PUBLICATION:
						PubHeader = _("Can publish certificates and CRL to:\n");
						PubList += info.get_dsts()[i].get_name().c_str();
						PubList += "\n";
						break;
				}
				break;
			case ENTITY_TYPE_RA:
				switch(info.get_dsts()[i].get_type())
				{
					case ENTITY_TYPE_REPOSITORY:
						RepHeader = _("Get personnal configuration from:\n");
						RepList += info.get_dsts()[i].get_name().c_str();
						RepList += "\n";
						break;
					case ENTITY_TYPE_CA:
						CaHeader = _("Can request certificates from:\n");
						CaList += info.get_dsts()[i].get_name().c_str();
						CaList += "\n";
						break;
					case ENTITY_TYPE_KEY_STORE:
						KeyHeader = _("Can store encipherment key in:\n");
						KeyList += info.get_dsts()[i].get_name().c_str();
						KeyList += "\n";
						break;
					case ENTITY_TYPE_EE:
						KeyHeader = _("Can get profiles and requests from:\n");
						EEList += info.get_dsts()[i].get_name().c_str();
						EEList += "\n";
						break;
				}
				break;
			case ENTITY_TYPE_PUBLICATION:
				switch(info.get_dsts()[i].get_type())
				{
					case ENTITY_TYPE_REPOSITORY:
						RepHeader = _("Get personnal configuration from:\n");
						RepList += info.get_dsts()[i].get_name().c_str();
						RepList += "\n";
						break;
					case ENTITY_TYPE_CA:
						CaHeader = _("Get publication requests from:\n");
						CaList += info.get_dsts()[i].get_name().c_str();
						CaList += "\n";
						break;
				}
				break;
			case ENTITY_TYPE_KEY_STORE:
				switch(info.get_dsts()[i].get_type())
				{
					case ENTITY_TYPE_REPOSITORY:
						RepHeader = _("Get personnal configuration from:\n");
						RepList += info.get_dsts()[i].get_name().c_str();
						RepList += "\n";
						break;
					case ENTITY_TYPE_RA:
						CaHeader = _("Get key storing requests from:\n");
						CaList += info.get_dsts()[i].get_name().c_str();
						CaList += "\n";
						break;
				}
				break;
			case ENTITY_TYPE_EE:
				switch(info.get_dsts()[i].get_type())
				{
					case ENTITY_TYPE_REPOSITORY:
						RepHeader = _("Get personnal configuration from:\n");
						RepList += info.get_dsts()[i].get_name().c_str();
						RepList += "\n";
						break;
					case ENTITY_TYPE_RA:
						CaHeader = _("Can send profiles and requests to:\n");
						CaList += info.get_dsts()[i].get_name().c_str();
						CaList += "\n";
						break;
				}
				break;
		}
	}


	DisplayInfo.sprintf(_("Entity Name: %s\nEntity Type: %s\n\n"), info.get_src().get_name().c_str(), EntityType);

	if(RepHeader)
	{
		DisplayInfo += RepHeader;
		DisplayInfo += RepList;
		DisplayInfo += "\n";
	}
	if(CaHeader)
	{
		DisplayInfo += CaHeader;
		DisplayInfo += CaList;
		DisplayInfo += "\n";
	}
	if(RaHeader)
	{
		DisplayInfo += RaHeader;
		DisplayInfo += RaList;
		DisplayInfo += "\n";
	}
	if(PubHeader)
	{
		DisplayInfo += PubHeader;
		DisplayInfo += PubList;
		DisplayInfo += "\n";
	}
	if(KeyHeader)
	{
		DisplayInfo += KeyHeader;
		DisplayInfo += KeyList;
		DisplayInfo += "\n";
	}
	display_info->SetValue(DisplayInfo);
}

