/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#include "XmlImport.h"
#include "clintl.h"
#include <PKI_CSR.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

XmlRequest::XmlRequest(const EntityConf & cEntityConf):m_EntityConf(cEntityConf)
{
	m_keylen = 0;
	m_error = false;
	m_isImported = false;
}

XmlRequest::~XmlRequest()
{
}

bool XmlRequest::IsImported() const
{
	return m_isImported;
}

void XmlRequest::set_Imported(bool imported)
{
	m_isImported = imported;
}

bool XmlRequest::Load(xmlNodePtr RequestNode)
{
	ERR_clear_error();

	if(!PrivLoad(RequestNode))
	{
		ERR_to_mstring(m_strerror, true);
		m_error = true;
		return false;
	}
	m_error = false;
	m_request.set_isOK();
	return true;
}

bool XmlRequest::Dump(xmlNodePtr ParentNode) const
{
	xmlNodePtr NewNode;
	xmlNodePtr ReqNode;
	mString value;

	if(m_isImported)
	{
		return true;
	}

	
	NewNode = xmlNewNode(NULL, (const xmlChar*)"REQUEST");
	if(!NewNode)
	{
		return false;
	}
	xmlAddChild(ParentNode, NewNode);

	// Set properties
	if(!xmlSetProp(NewNode, (const xmlChar*)"ca", (const xmlChar*)m_request.get_caName().c_str()))
	{
		return false;
	}

	value.sprintf("%ld", m_request.get_validity());
	if(!xmlSetProp(NewNode, (const xmlChar*)"validity", (const xmlChar*)value.c_str()))
	{
		return false;
	}

	value.sprintf("%d", 
				ASN1_BIT_STRING_get_bit(
							(ASN1_BIT_STRING*)m_request.get_flags(), 
							REQUEST_CERT_FLAG_SEND_CERT_TO_USER));
	if(!xmlSetProp(NewNode, (const xmlChar*)"send_cert", (const xmlChar*)value.c_str()))
	{
		return false;
	}

	switch(m_request.get_type())
	{
		case NEWPKI_PROFILE_CERT_TYPE_PKCS12:
			ReqNode = xmlNewTextChild(NewNode, NULL, (const xmlChar*)"REQ_P12", (const xmlChar*)m_request.get_request().get_p12Privdatas().get_passwd().c_str());
			if(!ReqNode)
			{
				return false;
			}

			value.sprintf("%d", 
						ASN1_BIT_STRING_get_bit(
									(ASN1_BIT_STRING*)m_request.get_flags(), 
									REQUEST_CERT_FLAG_SEND_P12_PASSWD_TO_USER));
			if(!xmlSetProp(ReqNode, (const xmlChar*)"send_pwd", (const xmlChar*)value.c_str()))
			{
				return false;
			}

			value.sprintf("%d", 
						ASN1_BIT_STRING_get_bit(
									(ASN1_BIT_STRING*)m_request.get_flags(), 
									REQUEST_CERT_FLAG_RA_GENERATE_P12_PASSWD));
			if(!xmlSetProp(ReqNode, (const xmlChar*)"gen_pwd", (const xmlChar*)value.c_str()))
			{
				return false;
			}
			
			value.sprintf("%ld", m_keylen);
			if(!xmlSetProp(ReqNode, (const xmlChar*)"keylen", (const xmlChar*)value.c_str()))
			{
				return false;
			}
			break;
		case NEWPKI_PROFILE_CERT_TYPE_PKCS10:
			ReqNode = xmlNewTextChild(NewNode, NULL, (const xmlChar*)"REQ_P10", (const xmlChar*)m_request.get_request().get_pkcs10().GetPemCSR().c_str());
			if(!ReqNode)
			{
				return false;
			}
			break;
		case NEWPKI_PROFILE_CERT_TYPE_SC:
			ReqNode = xmlNewTextChild(NewNode, NULL, (const xmlChar*)"REQ_SC", (const xmlChar*)m_csp.c_str());
			if(!ReqNode)
			{
				return false;
			}
			value.sprintf("%ld", m_keylen);
			if(!xmlSetProp(ReqNode, (const xmlChar*)"keylen", (const xmlChar*)value.c_str()))
			{
				return false;
			}
			break;
	}
	return true;
}

bool XmlRequest::PrivLoad(xmlNodePtr RequestNode)
{
	xmlNodePtr currRequest;
	long lngValue;
	xmlChar * strValue;
	PKI_CSR csr;
	mString strCvrt;
	size_t i;
	bool retval = true;


	// Getting the ca attribute
	strValue = xmlGetProp(RequestNode, (xmlChar*)"ca");
	if(!strValue || !*strValue)
	{
		NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
		ERR_add_error_data(1, _("Expecting a \"ca\" attribute for the request"));
		retval = false;
		m_caname = _("Unknown");
		m_request.set_caName(m_caname);
	}
	else
	{
		// Making sure the CA exists
		for(i=0; i<m_EntityConf.get_cas().get_list().size(); i++)
		{
			if(m_EntityConf.get_cas().get_list()[i].get_name() == strValue)
				break;
		}
		if(i == m_EntityConf.get_cas().get_list().size())
		{
			NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
			ERR_add_error_data(2, _("Unknown CA:"), strValue);
			retval = false;
		}
	}
	if(!retval)
		m_caname = _("Unknown");
	else
		m_caname = strValue;
	m_request.set_caName(m_caname);
	if(strValue)
		xmlFree(strValue);


	// Getting the validity length attribute
	strValue = xmlGetProp(RequestNode, (xmlChar*)"validity");
	if(strValue && *strValue)
	{
		sscanf((char*)strValue, "%ld", &lngValue);
		if(lngValue)
		{
			m_request.set_validity(lngValue);
		}
		else
		{
			m_request.set_validity(m_EntityConf.get_body().get_raConf().RA_CONF_PTR.get_defaultValidity());
		}
	}
	else
	{
		m_request.set_validity(m_EntityConf.get_body().get_raConf().RA_CONF_PTR.get_defaultValidity());
	}
	if(strValue)
		xmlFree(strValue);
	
	strValue = xmlGetProp(RequestNode, (xmlChar*)"send_cert");
	if(strValue && *strValue && strValue[0] != '0')
	{
		ASN1_BIT_STRING_set_bit((ASN1_BIT_STRING*)m_request.get_flags(), REQUEST_CERT_FLAG_SEND_CERT_TO_USER, 1);
	}
	if(strValue)
		xmlFree(strValue);


	for(currRequest=RequestNode->children; currRequest; currRequest=currRequest->next)
	{
		// This is a PKCS#12 request
		if(xmlStrcmp(currRequest->name, (xmlChar*)"REQ_P12") == 0)
		{
			m_request.set_type(NEWPKI_PROFILE_CERT_TYPE_PKCS12);
			if(!m_request.get_request().set_type(REQUEST_CERT_BODY_PKCS12))
			{
				NEWPKIerr(CLIENT_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			// Getting the keylen attribute
			strValue = xmlGetProp(currRequest, (xmlChar*)"keylen");
			if(!strValue || !*strValue)
			{
				NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
				ERR_add_error_data(1, _("Expecting a \"keylen\" attribute for the PKCS#12 request"));
				retval = false;
			}
			else
			{
				sscanf((char*)strValue, "%ld", &m_keylen);
				if(!m_keylen)
				{
					NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
					ERR_add_error_data(1, _("Expecting a \"keylen\" attribute for the PKCS#12 request"));
					retval = false;
				}
				m_request.get_request().get_p12Privdatas().set_keylen(m_keylen);

				if(m_keylen < m_EntityConf.get_body().get_raConf().RA_CONF_PTR.get_minkeylen())
				{
					NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
					strCvrt.sprintf("%ld", m_EntityConf.get_body().get_raConf().RA_CONF_PTR.get_minkeylen());
					ERR_add_error_data(2, _("\"keylen\" is too short, minimum length is "), strCvrt.c_str());
					retval = false;
				}
			}
			if(strValue)
				xmlFree(strValue);

			// Getting the send_pwd attribute
			strValue = xmlGetProp(currRequest, (xmlChar*)"send_pwd");
			if(strValue && *strValue && strValue[0] != '0')
			{
				ASN1_BIT_STRING_set_bit((ASN1_BIT_STRING*)m_request.get_flags(), REQUEST_CERT_FLAG_SEND_P12_PASSWD_TO_USER, 1);
			}
			if(strValue)
				xmlFree(strValue);

			// Getting the gen_pwd attribute
			strValue = xmlGetProp(currRequest, (xmlChar*)"gen_pwd");
			if(strValue && *strValue && strValue[0] != '0')
			{
				ASN1_BIT_STRING_set_bit((ASN1_BIT_STRING*)m_request.get_flags(), REQUEST_CERT_FLAG_RA_GENERATE_P12_PASSWD, 1);
				xmlFree(strValue);
			}
			else
			{
				if(strValue)
					xmlFree(strValue);
				// Getting the password value
				strValue = xmlNodeGetContent(currRequest);
				if(!strValue || !*strValue)
				{
					NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
					ERR_add_error_data(1, _("Expecting a password content for the PKCS#12 request"));
					retval = false;
				}
				else
				{
					m_request.get_request().get_p12Privdatas().set_passwd((char*)strValue);

					if(m_request.get_request().get_p12Privdatas().get_passwd().size() < m_EntityConf.get_body().get_raConf().RA_CONF_PTR.get_minpwdlen())
					{
						NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
						strCvrt.sprintf("%ld", m_EntityConf.get_body().get_raConf().RA_CONF_PTR.get_minpwdlen());
						ERR_add_error_data(2, _("password is too short, minimum length is "), strCvrt.c_str());
						retval = false;
					}
				}
				if(strValue)
					xmlFree(strValue);
			}
			break;
		}
		// This is a PKCS#10 request
		else if(xmlStrcmp(currRequest->name, (xmlChar*)"REQ_P10") == 0)
		{
			m_request.set_type(NEWPKI_PROFILE_CERT_TYPE_PKCS10);
			if(!m_request.get_request().set_type(REQUEST_CERT_BODY_PKCS10))
			{
				NEWPKIerr(CLIENT_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			// The PKCS#10
			strValue = xmlNodeGetContent(currRequest);
			if(!strValue || !*strValue)
			{
				NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
				ERR_add_error_data(1, _("You must provide a PEM encoded PKCS#10"));
				retval = false;
			}
			else
			{
				if(!m_request.get_request().get_pkcs10().SetCSR((char*)strValue))
				{
					NEWPKIerr(CLIENT_ERROR_TXT, ERROR_ABORT);
					retval = false;
				}
				m_keylen = (EVP_PKEY_size((EVP_PKEY*)m_request.get_request().get_pkcs10().GetPublicKey())*8);
				if( m_keylen < m_EntityConf.get_body().get_raConf().RA_CONF_PTR.get_minkeylen())
				{
					NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
					strCvrt.sprintf("%ld", m_EntityConf.get_body().get_raConf().RA_CONF_PTR.get_minkeylen());
					ERR_add_error_data(2, _("\"keylen\" is too short, minimum length is "), strCvrt.c_str());
					retval = false;
				}
			}
			if(strValue)
				xmlFree(strValue);
			break;
		}
		// This is a SC request
		else if(xmlStrcmp(currRequest->name, (xmlChar*)"REQ_SC") == 0)
		{
			m_request.set_type(NEWPKI_PROFILE_CERT_TYPE_SC);
			if(!m_request.get_request().set_type(REQUEST_CERT_BODY_PKCS10))
			{
				NEWPKIerr(CLIENT_ERROR_TXT, ERROR_MALLOC);
				return false;
			}

			// Getting the keylen attribute
			strValue = xmlGetProp(currRequest, (xmlChar*)"keylen");
			if(!strValue || !*strValue)
			{
				NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
				ERR_add_error_data(1, _("Expecting a \"keylen\" attribute for the SC request"));
				retval = false;
			}
			else
			{
				sscanf((char*)strValue, "%ld", &m_keylen);
				if(!m_keylen)
				{
					NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
					ERR_add_error_data(1, _("Expecting a \"keylen\" attribute for the SC request"));
					return false;
				}

				if(m_keylen < m_EntityConf.get_body().get_raConf().RA_CONF_PTR.get_minkeylen())
				{
					NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
					strCvrt.sprintf("%ld", m_EntityConf.get_body().get_raConf().RA_CONF_PTR.get_minkeylen());
					ERR_add_error_data(2, _("\"keylen\" is too short, minimum length is "), strCvrt.c_str());
					return false;
				}
			}
			if(strValue)
				xmlFree(strValue);

			// The CSP
			strValue = xmlNodeGetContent(currRequest);
			if(strValue && *strValue)
			{
				m_csp = strValue;
			}
			else
			{
				NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
				ERR_add_error_data(1, _("The content of a \"REQ_SC\" node must contain the name of the CSP"));
				retval = false;
			}
			if(strValue)
				xmlFree(strValue);
			break;
		}
	}

	return retval;
}

NewpkiRequestCert & XmlRequest::get_Request()
{
	return m_request;
}

void XmlRequest::set_Request(const NewpkiRequestCert & request)
{
	m_request = request;
	m_request.set_isOK();
}


const mString & XmlRequest::get_CSP() const
{
	return m_csp;
}

long XmlRequest::get_KeyLen() const
{
	return m_keylen;
}

void XmlRequest::set_CSP(const mString & csp)
{
	m_csp = csp;
}

void XmlRequest::set_CaName(const mString & caname)
{
	m_caname = caname;
}

void XmlRequest::set_KeyLen(long keylen)
{
	m_keylen = keylen;
}

bool XmlRequest::IsError() const
{
	return m_error;
}

const mString & XmlRequest::get_Error() const
{
	return m_strerror;
}

const mString & XmlRequest::get_CaName() const
{
	return m_caname;
}

void XmlRequest::set_Error(const mString & error)
{
	m_error = true;
	m_strerror = error;
}














XmlProfile::XmlProfile(const EntityConf & cEntityConf):m_EntityConf(cEntityConf)
{
	dn_name = NULL;
	m_error = false;
	m_Id = 0;
	m_isImported = false;
}

XmlProfile::~XmlProfile()
{
	Clear();
}

void XmlProfile::Clear()
{
	size_t i;
	m_profile.Clear();
	for(i=0; i<m_requests.size(); i++)
	{
		if(m_requests[i])
			delete m_requests[i];
	}
	m_requests.clear();
	if(dn_name)
		free(dn_name);
	m_Id = 0;
	m_isImported = false;
	m_error = false;
}

bool XmlProfile::Load(xmlNodePtr ProfileNode)
{
	ERR_clear_error();
	m_error = false;
	if(!PrivLoad(ProfileNode))
	{
		ERR_to_mstring(m_strerror, true);
		return false;
	}
	m_profile.set_isOK();

	return true;
}

bool XmlProfile::Dump(FILE * fp, xmlDocPtr xDoc) const
{
	xmlNodePtr NewNode;
	xmlNodePtr DnNode;
	xmlNodePtr DnEntry;
	xmlNodePtr DnVal;
	mString value;
	size_t i;
	HashTable_Dn dn;
	const char * dnName;
	const char * dnValue;

	if(m_isImported)
	{
		// We only add the imported node, if there is
		// at least one not generated request
		for(i=0; i<m_requests.size(); i++)
		{
			if(!m_requests[i]->IsImported())
				break;
		}
		if(i == m_requests.size())
			return true;

		NewNode = xmlNewNode(NULL, (const xmlChar*)"IMPORTED_PROFILE");
		if(!NewNode)
		{
			return false;
		}
		value.sprintf("%ld", m_Id);
		if(!xmlSetProp(NewNode, (const xmlChar*)"id", (const xmlChar*)value.c_str()))
		{
			xmlFreeNode(NewNode);
			return false;
		}
	}
	else
	{
		NewNode = xmlNewNode(NULL, (const xmlChar*)"PROFILE");
		if(!NewNode)
		{
			return false;
		}
		// Set properties
		if(!xmlSetProp(NewNode, (const xmlChar*)"owner_group", (const xmlChar*)m_ownerGroup.c_str()))
		{
			xmlFreeNode(NewNode);
			return false;
		}

		if(!xmlSetProp(NewNode, (const xmlChar*)"ldap_uid", (const xmlChar*)m_profile.get_ldapUid().c_str()))
		{
			xmlFreeNode(NewNode);
			return false;
		}
	}

	// Dump DN
	DnNode = xmlNewNode(NULL, (const xmlChar*)"DN");
	if(!DnNode)
	{
		xmlFreeNode(NewNode);
		return false;
	}
	xmlAddChild(NewNode, DnNode);

	if(!dn.From_X509_NAME(m_profile.get_dn()))
	{
		xmlFreeNode(NewNode);
		return false;
	}
	for(i=0; i < dn.EntriesCount(); i++)
	{
		dnName = dn.GetName(i);
		dnValue = dn.Get(i);
		if(!dnName || !dnValue)
			continue;

		DnEntry = xmlNewNode(NULL, (const xmlChar*)"DN_ENTRY");
		if(!DnEntry)
		{
			xmlFreeNode(NewNode);
			return false;
		}
		xmlAddChild(DnNode, DnEntry);
		
		DnVal = xmlNewTextChild(DnEntry, NULL, (const xmlChar*)"NAME", (const xmlChar*)dnName);
		if(!DnVal)
		{
			xmlFreeNode(NewNode);
			return false;
		}
		DnVal = xmlNewTextChild(DnEntry, NULL, (const xmlChar*)"VALUE", (const xmlChar*)dnValue);
		if(!DnVal)
		{
			xmlFreeNode(NewNode);
			return false;
		}			
	}

	DnVal = xmlNewNode(NULL, (const xmlChar*)"REQUESTS");
	if(!DnVal)
	{
		xmlFreeNode(NewNode);
		return false;
	}
	xmlAddChild(NewNode, DnVal);

	// Dump the requests
	for(i=0; i<m_requests.size(); i++)
	{
		if(m_requests[i]->IsImported())
			continue;
		if(!m_requests[i]->Dump(DnVal))
		{
			xmlFreeNode(NewNode);
			return false;
		}
	}
	xmlElemDump(fp, xDoc, NewNode);
	xmlFreeNode(NewNode);
	return true;
}

bool XmlProfile::PrivLoad(xmlNodePtr ProfileNode)
{
	xmlNodePtr currChild;
	xmlNodePtr currRequest;
	xmlChar * strValue;
	XmlRequest * newRequest;
	size_t i;

	ERR_clear_error();
	Clear();

	// A profile that was already imported
	if(xmlStrcmp(ProfileNode->name, (xmlChar*)"IMPORTED_PROFILE") == 0)
	{
		strValue = xmlGetProp(ProfileNode, (xmlChar*)"id");
		if(!strValue || !*strValue)
		{
			NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
			ERR_add_error_data(1, _("Expecting an id attribute in node IMPORTED_PROFILE"));
			if(strValue)
				xmlFree(strValue);
			return false;
		}
		sscanf((char*)strValue, "%ld", &m_Id);
		if(!m_Id)
		{
			NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
			ERR_add_error_data(1, _("Expecting an id attribute in node IMPORTED_PROFILE"));
			if(strValue)
				xmlFree(strValue);
			return false;
		}
		if(strValue)
			xmlFree(strValue);
		m_isImported = true;
	}
	else
	{
		// A PROFILE Node
		if(xmlStrcmp(ProfileNode->name, (xmlChar*)"PROFILE") != 0)
		{
			NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
			ERR_add_error_data(1, _("Expecting node type PROFILE or IMPORTED_PROFILE"));
			return false;
		}
		m_profile.set_eeValidation(0);

		strValue = xmlGetProp(ProfileNode, (xmlChar*)"ldap_uid");		
		if(strValue && *strValue)
		{
			m_profile.set_ldapUid((char*)strValue);
		}
		else
		{
			m_profile.set_ldapUid("");
		}
		if(strValue)
			xmlFree(strValue);

		strValue = xmlGetProp(ProfileNode, (xmlChar*)"owner_group");
		if(strValue && *strValue)
		{
			for(i=0; i<m_EntityConf.get_groups().size(); i++)
			{
				if(m_EntityConf.get_groups()[i].get_name() == strValue)
				{
					break;
				}
			}
			if(i == m_EntityConf.get_groups().size())
			{
				NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
				ERR_add_error_data(2, _("Unknown group:"), strValue);
				ERR_to_mstring(m_strerror, true);
				m_error = true;
			}
			else
			{
				m_profile.set_ownerGroupSerial(m_EntityConf.get_groups()[i].get_serial());
				m_ownerGroup = strValue;
			}
		}
		else
		{
			m_profile.set_ownerGroupSerial(0);
			m_ownerGroup = "";
		}
		if(strValue)
			xmlFree(strValue);
	}


	for(currChild=ProfileNode->children; currChild; currChild=currChild->next)
	{
		if(xmlStrcmp(currChild->name, (xmlChar*)"DN") == 0)
		{
			ERR_clear_error();
			if(!LoadDn(currChild))
			{
				ERR_to_mstring(m_strerror, true);
				m_error = true;
			}
		}
		else if(xmlStrcmp(currChild->name, (xmlChar*)"REQUESTS") == 0)
		{
			for(currRequest=currChild->children; currRequest; currRequest=currRequest->next)
			{
				if(xmlStrcmp(currRequest->name, (xmlChar*)"REQUEST") != 0)
					continue;

				newRequest = new XmlRequest(m_EntityConf);
				if(!newRequest)
				{
					NEWPKIerr(CLIENT_ERROR_TXT, ERROR_MALLOC);
					return false;
				}
				if(!newRequest->Load(currRequest) || newRequest->IsError())
				{
					m_strerror += _("Request(s) error\n");
					m_error = true;
				}
				m_requests.push_back(newRequest);
			}
		}
	}

	return true;
}

bool XmlProfile::LoadDn(xmlNodePtr DnNode)
{
	xmlNodePtr currDnEntry;
	xmlNodePtr currEntry;
	xmlChar * Name;
	xmlChar * Value;
	int nid;

	for(currDnEntry=DnNode->children; currDnEntry; currDnEntry=currDnEntry->next)
	{
		Name = NULL;
		Value = NULL;
		for(currEntry=currDnEntry->children; currEntry; currEntry=currEntry->next)
		{
			if(xmlStrcmp(currEntry->name, (xmlChar*)"NAME") == 0)
			{
				Name = xmlNodeGetContent(currEntry);
			}
			else if(xmlStrcmp(currEntry->name, (xmlChar*)"VALUE") == 0)
			{
				Value = xmlNodeGetContent(currEntry);
			}
		}
		if(!Name || !*Name)
		{
			NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
			ERR_add_error_data(1, _("Expecting node type NAME"));
			if(Name)
				xmlFree(Name);
			return false;
		}
		if(!Value || !*Value)
		{
			NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
			ERR_add_error_data(1, _("Expecting node type VALUE"));
			if(Name)
				xmlFree(Name);
			if(Value)
				xmlFree(Value);
			return false;
		}

		nid = OBJ_txt2nid((char*)Name);
		if(nid == NID_undef)
		{
			NEWPKIerr(CLIENT_ERROR_TXT, ERROR_BAD_DATAS);
			ERR_add_error_data(2, _("Unknown DN field"), Name);
			return false;
		}

		if(!X509_NAME_add_entry_by_NID(m_profile.get_dn(), nid, MBSTRING_ASC, Value, -1,-1,0))
		{
			NEWPKIerr(CLIENT_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(Name)
			xmlFree(Name);
		if(Value)
			xmlFree(Value);
	}

	dn_name = X509_NAME_oneline(m_profile.get_dn(), NULL, 0);
	if(!dn_name)
	{
		NEWPKIerr(CLIENT_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

const char * XmlProfile::GetName()
{
	if(dn_name)
		return dn_name;
	else
		return _("Unknown");
}

bool XmlProfile::IsError() const
{
	return m_error;
}

const mString & XmlProfile::get_Error() const
{
	return m_strerror;
}

vector<XmlRequest*> & XmlProfile::get_Requests()
{
	return m_requests;
}

const NewpkiProfile & XmlProfile::get_Profile() const
{
	return m_profile;
}

bool XmlProfile::IsImported() const
{
	return m_isImported;
}

void XmlProfile::set_Imported(bool imported)
{
	m_isImported = imported;
}

unsigned long XmlProfile::get_Id() const
{
	return m_Id;
}

void XmlProfile::set_Id(unsigned long id)
{
	m_Id = id;
}

void XmlProfile::set_Error(const mString & error)
{
	m_error = true;
	m_strerror = error;
}

void XmlProfile::set_ItemId(int id)
{
	m_ItemId = id;
}

int XmlProfile::get_ItemId()
{
	return m_ItemId;
}