/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PlugCA.h: interface for the PlugCA class.
//
//////////////////////////////////////////////////////////////////////

#ifndef PLUGCA_H
#define PLUGCA_H

#include <PKI_PLUG_CA.h>
#include <openssl/dso.h>
#include <ExceptionNewPKI.h>
#include <mString.h>


/*!
	This class represents a CA plug module
*/
class PlugCA
{
public:
	/*! \brief This is the constructor.
	 *  \param LibraryPath [IN] The path to the library.
	 *  \exception ExceptionNewPKI An error occured.
	 */
	PlugCA(const mString & LibraryPath);

	/*! \brief This is the destructor.
	 */
	virtual ~PlugCA();

	/*! \brief This function is called to validate the plugin options.
	 *  \param Options [IN] The options.
	 *  \return true on success, false on failure.
	 */
	bool ValidateOptions(HashTable_String & Options);

	/*! \brief This function is called when a new certification request arrives to the CA.
	 *  \param Options [IN] The options.
	 *  \param CaCert [IN] The CA cert.
	 *  \param Mailer [IN] The SMTP client.
	 *  \param Csr [IN] The CSR.
	 *  \return true on success, false on failure.
	 */
	bool OnNewRequest(HashTable_String & Options, const PKI_CERT & CaCert, const SmtpClient & Mailer, const PKI_CSR & Csr);

	/*! \brief This function is called when a new certificate has been generated.
	 *  \param Options [IN] The options.
	 *  \param CaCert [IN] The CA cert.
	 *  \param Mailer [IN] The SMTP client.
	 *  \param Cert [IN] The certificate.
	 *  \return true on success, false on failure.
	 */
	bool OnNewCertificate(HashTable_String & Options, const PKI_CERT & CaCert, const SmtpClient & Mailer, const PKI_CERT & Cert);

	/*! \brief This function is called when a new revocation request arrives to the CA.
	 *  \param Options [IN] The options.
	 *  \param CaCert [IN] The CA cert.
	 *  \param Mailer [IN] The SMTP client.
	 *  \param Serial [IN] The serial to revoke.
	 *  \return true on success, false on failure.
	 */
	bool OnNewRevocation(HashTable_String & Options, const PKI_CERT & CaCert, const SmtpClient & Mailer, unsigned long Serial);

	/*! \brief This function is called when a new CRL has been generated.
	 *  \param Options [IN] The options.
	 *  \param CaCert [IN] The CA cert.
	 *  \param Mailer [IN] The SMTP client.
	 *  \param Crl [IN] The CRL.
	 *  \return true on success, false on failure.
	 */
	bool OnNewCRL(HashTable_String & Options, const PKI_CERT & CaCert, const SmtpClient & Mailer, const PKI_CRL & Crl);
	
	/*! \brief This function returns the library path.
	 *  \return The library path.
	 */
	const mString & GetLibraryPath();

private:
	DSO * hLib;
	PKI_PLUG_CA * clPlugCa;
	GET_PKI_PLUG_CA GetPlugCa;
	FREE_PKI_PLUG_CA FreePlugCa;
	mString m_LibraryPath;
};

#endif // PlugCA_H
