/**************************************************************************/
/*                                N I C E                                 */
/*             A high-level object-oriented research language             */
/*                        (c) Daniel Bonniot 2002                         */
/*                                                                        */
/*  This program is free software; you can redistribute it and/or modify  */
/*  it under the terms of the GNU General Public License as published by  */
/*  the Free Software Foundation; either version 2 of the License, or     */
/*  (at your option) any later version.                                   */
/*                                                                        */
/**************************************************************************/

package bossa.link;

import bossa.util.*;

import mlsub.typing.*;

import bossa.syntax.MethodDeclaration;
import bossa.syntax.Pattern;
import bossa.syntax.LocatedString;
import bossa.syntax.Node;
import bossa.syntax.ConstantExp;

import gnu.bytecode.*;
import gnu.expr.*;

import nice.tools.code.Gen;

import java.util.*;

/**
 * Represents a method alternative in the link.
 *
 * It can be build either from information in a Bossa source,
 * or read from a compiled bytecode file.
 *
 * @author bonniot
 */

public abstract class Alternative implements Located
{
  /**
     The full name uniquely identifies the method by including the 
     complete type.
  */
  public Alternative(String methodName, String fullName, Pattern[] patterns)
  {
    this.methodName = methodName;
    this.fullName = fullName;
    this.patterns = patterns;
    add();
  }

  /****************************************************************
   * Graph traversal
   ****************************************************************/

  static final int 
    UNVISITED = 1,
    VISITING  = 2,
    VISITED   = 3;
  
  /** Marks the state of this node in the graph traversal. */
  int mark = UNVISITED;
  
  /****************************************************************
   * Order on alternatives
   ****************************************************************/

  /**
   * Returns true iff 'a' is more precise than 'b'.
   */
  public static boolean leq(Alternative a, Alternative b)
  {
    for(int i = 0; i<a.patterns.length; i++)
      if (!a.patterns[i].leq(b.patterns[i]))
	return false;
    return true;
  }

  /**
   * Returns true iff 'a' is strictly more precise than 'b'.
   */
  public static boolean less(Alternative a, Alternative b)
  {
    boolean strictly = false;

    for (int i = 0; i < a.patterns.length; i++)
      if (!a.patterns[i].leq(b.patterns[i]))
	return false;
      else if (!b.patterns[i].leq(a.patterns[i]))
	strictly = true;

    return strictly;
  }

  /**
   * Tests the matching of tags against a method alternative.
   */
  boolean matches(TypeConstructor[] tags)
  {
    for(int i = 0; i < patterns.length; i++)
      if (!patterns[i].matches(tags[i]))
	return false;

    return true;
  }

  boolean matchesTypePart(TypeConstructor[] tags, boolean[] isValue)
  {
    for(int i = 0; i < patterns.length; i++)
      if (!isValue[i] && !patterns[i].matches(tags[i]))
	return false;

    return true;
  }

  boolean matchesValuePart(ConstantExp[] values, boolean[] isValue)
  {
    for(int i = 0; i < patterns.length; i++)
      if (isValue[i] && !patterns[i].matchesValue(values[i]))
	return false;

    return true;
  }

  boolean containsTypeMatchingValue()
  {
    for(int i = 0; i < patterns.length; i++)
      if (patterns[i].atTypeMatchingValue())
	return true;

    return false;
  }

  /****************************************************************
   * Code generation
   ****************************************************************/

  /**
   * @return the expression that represents the method body.
   */
  public abstract Expression methodExp();

  /**
     @return the expression that tests if this alternative matches
     the tuple <code>parameters</code>.
   */
  Expression matchTest(Expression[] parameters, boolean skipFirst)
  {
    if (parameters.length != patterns.length)
      Internal.error("Incorrect parameters "+
		     Util.map("",", ","",parameters)+
		     " for " + this);

    Expression result = QuoteExp.trueExp;

    for (int index = 0; index < parameters.length; index++)
      result = Gen.shortCircuitAnd
        (result,
         patterns[index].matchTest(parameters[index], index == 0 && skipFirst));

    return result;
  }

  public String toString()
  {
    return methodName + Util.map("(", ", ", ")", patterns);
  }

  String printLocated()
  {
    return toString();
  }

  String methodName;
  String fullName;
  Pattern[] patterns;

  public Pattern[] getPatterns() { return patterns; }

  /****************************************************************
   * Regrouping alternatives per method
   ****************************************************************/

  private static HashMap alternatives;
  public static void reset()
  {
    alternatives = new HashMap();
  }
  
  private void add()
  {
    List l = (List) alternatives.get(fullName);
    if (l == null)
      {
	// XXX change to LinkedList
	l = new ArrayList();
	alternatives.put(fullName,l);
      }
    // Dispatch.sort(final List alternatives) assumes that new alternatives
    // are added at the end
    l.add(this);
  }
  
  public static Stack sortedAlternatives(MethodDeclaration m)
  {
    List list = (List) alternatives.get(m.getFullName());
    
    if (list == null)
      // It's not an error for a method to have no alternative
      // as long as its domain is empty.
      // this will be checked later
      return new Stack();
    
    return sort(list);
  }

  /****************************************************************
   * Sorting alternatives by generality
   ****************************************************************/

  /**
     Computes a topological sorting of the list of alternatives.
     
     Uses a postfix travsersal of the graph.
  */
  private static Stack sort(final List alternatives)
  {
    Stack sortedAlternatives = new Stack();
    
    if (alternatives.size() == 0)
      return sortedAlternatives;

    // Test if another sort has been done before.
    // In that case reset the marks.
    if (((Alternative) alternatives.get(0)).mark != Alternative.UNVISITED)
      for(Iterator i = alternatives.iterator(); i.hasNext();)
	((Alternative) i.next()).mark = Alternative.UNVISITED;
	
    for(Iterator i = alternatives.iterator(); i.hasNext();)
      {
	Alternative a = (Alternative) i.next();
	if (a.mark == Alternative.UNVISITED)
	  visit(a, alternatives, sortedAlternatives);
      }

    return sortedAlternatives;
  }
  
  private final static void visit
    (final Alternative a, 
     final List alternatives,
     final Stack sortedAlternatives
     )
  {
    // Cycles are possible
    //   * if two alternatives are identical (same patterns)
    //   * if there is a cyclic subclass relation
    //   * that's all, folks ! :-)
    switch(a.mark)
      {
      case Alternative.VISITING : User.error("Cycle in alternatives: "+a);
      case Alternative.UNVISITED: a.mark = Alternative.VISITING; break;
      case Alternative.VISITED  : return; //should not happen
      }
    
    for(Iterator i = alternatives.iterator();
	i.hasNext();)
      {
	Alternative daughter = (Alternative) i.next();
	if(daughter != a 
	   && daughter.mark == Alternative.UNVISITED 
	   && Alternative.leq(daughter,a))
	  visit(daughter,alternatives,sortedAlternatives);
      }
    a.mark = Alternative.VISITED;
    sortedAlternatives.push(a);
  }
}
